<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        try {
            $indexedTablesPath = base_path('includes/table_indexes.json');

            // Check if the index configuration file exists
            if (!File::exists($indexedTablesPath)) {
                Log::info('Table indexes configuration file not found at: ' . $indexedTablesPath);
                return;
            }

            // Decode JSON file content
            $indexedTables = json_decode(file_get_contents($indexedTablesPath), true);

            if (empty($indexedTables)) {
                Log::info('No indexed tables found in table_indexes file.');
                return;
            }

            foreach ($indexedTables as $tableName => $tableData) {
                $alterRecordExists = DB::table('alter_tables')
                    ->where('name', 'indexed_' . $tableData['table_name'])
                    ->exists();

                if (!$alterRecordExists) {
                    $jsonTable = $tableData['table_name'];
                    $indexQueries = [];

                    // Start building the ALTER TABLE query
                    foreach ($tableData['indexes'] as $index) {
                        if (!isFieldIndexed($jsonTable, $index) && Schema::hasColumn($jsonTable, $index)) {
                            $columnType = getColumnType($jsonTable, $index);

                            // Prepare index query based on column type
                            $indexQuery = "ADD INDEX `$index` (`$index`";
                            if (in_array($columnType, ['blob', 'text', 'mediumblob', 'mediumtext', 'longblob', 'longtext'])) {
                                $indexQuery .= "(255)";
                            }
                            $indexQuery .= ")";
                            $indexQueries[] = $indexQuery;
                        }
                    }

                    // If there are any queries to execute, build and store them
                    if (!empty($indexQueries)) {
                        $alterQuery = "ALTER TABLE `$jsonTable` " . implode(', ', $indexQueries);

                        $data = [
                            'name' => 'indexed_' . $jsonTable,
                            'type' => 'query',
                            'query' => $alterQuery,
                            'status' => 0,
                        ];

                        DB::table('alter_tables')->insert($data);
                    }
                }
            }

            // Call the artisan command to apply the changes
            Artisan::call('alter:tables');

        } catch (\Exception $e) {
            Log::error('Error while processing table indexing: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);
        }

    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        //
    }
};
