
(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				// CommonJS environments without a window global must pass a
				// root. This will give an error otherwise
				root = window;
			}

			if ( ! $ ) {
				$ = typeof window !== 'undefined' ? // jQuery's factory checks for a global window
					require('jquery') :
					require('jquery')( root );
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}
(function( $, window, document, undefined ) {
	"use strict";

	
	var DataTable = function ( options )
	{
		
		this.$ = function ( sSelector, oOpts )
		{
			return this.api(true).$( sSelector, oOpts );
		};
		
		
		this._ = function ( sSelector, oOpts )
		{
			return this.api(true).rows( sSelector, oOpts ).data();
		};
		
		
		this.api = function ( traditional )
		{
			return traditional ?
				new _Api(
					_fnSettingsFromNode( this[ _ext.iApiIndex ] )
				) :
				new _Api( this );
		};
		
		
		this.fnAddData = function( data, redraw )
		{
			var api = this.api( true );
		
			/* Check if we want to add multiple rows or not */
			var rows = $.isArray(data) && ( $.isArray(data[0]) || $.isPlainObject(data[0]) ) ?
				api.rows.add( data ) :
				api.row.add( data );
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return rows.flatten().toArray();
		};
		
		this.fnAdjustColumnSizing = function ( bRedraw )
		{
			var api = this.api( true ).columns.adjust();
			var settings = api.settings()[0];
			var scroll = settings.oScroll;
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw( false );
			}
			else if ( scroll.sX !== "" || scroll.sY !== "" ) {
				/* If not redrawing, but scrolling, we want to apply the new column sizes anyway */
				_fnScrollDraw( settings );
			}
		};
		
		this.fnClearTable = function( bRedraw )
		{
			var api = this.api( true ).clear();
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
		};
		
		this.fnClose = function( nTr )
		{
			this.api( true ).row( nTr ).child.hide();
		};
		
		this.fnDeleteRow = function( target, callback, redraw )
		{
			var api = this.api( true );
			var rows = api.rows( target );
			var settings = rows.settings()[0];
			var data = settings.aoData[ rows[0][0] ];
		
			rows.remove();
		
			if ( callback ) {
				callback.call( this, settings, data );
			}
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return data;
		};
		
		this.fnDestroy = function ( remove )
		{
			this.api( true ).destroy( remove );
		};
		
		this.fnDraw = function( complete )
		{
			// Note that this isn't an exact match to the old call to _fnDraw - it takes
			// into account the new data, but can hold position.
			this.api( true ).draw( complete );
		};
		
		this.fnFilter = function( sInput, iColumn, bRegex, bSmart, bShowGlobal, bCaseInsensitive )
		{
			var api = this.api( true );
		
			if ( iColumn === null || iColumn === undefined ) {
				api.search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
			else {
				api.column( iColumn ).search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
		
			api.draw();
		};
		
		this.fnGetData = function( src, col )
		{
			var api = this.api( true );
		
			if ( src !== undefined ) {
				var type = src.nodeName ? src.nodeName.toLowerCase() : '';
		
				return col !== undefined || type == 'td' || type == 'th' ?
					api.cell( src, col ).data() :
					api.row( src ).data() || null;
			}
		
			return api.data().toArray();
		};
		
		this.fnGetNodes = function( iRow )
		{
			var api = this.api( true );
		
			return iRow !== undefined ?
				api.row( iRow ).node() :
				api.rows().nodes().flatten().toArray();
		};
		
		this.fnGetPosition = function( node )
		{
			var api = this.api( true );
			var nodeName = node.nodeName.toUpperCase();
		
			if ( nodeName == 'TR' ) {
				return api.row( node ).index();
			}
			else if ( nodeName == 'TD' || nodeName == 'TH' ) {
				var cell = api.cell( node ).index();
		
				return [
					cell.row,
					cell.columnVisible,
					cell.column
				];
			}
			return null;
		};
		
		this.fnIsOpen = function( nTr )
		{
			return this.api( true ).row( nTr ).child.isShown();
		};
		
		this.fnOpen = function( nTr, mHtml, sClass )
		{
			return this.api( true )
				.row( nTr )
				.child( mHtml, sClass )
				.show()
				.child()[0];
		};
		
		this.fnPageChange = function ( mAction, bRedraw )
		{
			var api = this.api( true ).page( mAction );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw(false);
			}
		};
		
		this.fnSetColumnVis = function ( iCol, bShow, bRedraw )
		{
			var api = this.api( true ).column( iCol ).visible( bShow );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.columns.adjust().draw();
			}
		};
		
		this.fnSettings = function()
		{
			return _fnSettingsFromNode( this[_ext.iApiIndex] );
		};
		
		this.fnSort = function( aaSort )
		{
			this.api( true ).order( aaSort ).draw();
		};
		
		this.fnSortListener = function( nNode, iColumn, fnCallback )
		{
			this.api( true ).order.listener( nNode, iColumn, fnCallback );
		};
		
		this.fnUpdate = function( mData, mRow, iColumn, bRedraw, bAction )
		{
			var api = this.api( true );
		
			if ( iColumn === undefined || iColumn === null ) {
				api.row( mRow ).data( mData );
			}
			else {
				api.cell( mRow, iColumn ).data( mData );
			}
		
			if ( bAction === undefined || bAction ) {
				api.columns.adjust();
			}
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
			return 0;
		};
		
		this.fnVersionCheck = _ext.fnVersionCheck;
		

		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;

		if ( emptyInit ) {
			options = {};
		}

		this.oApi = this.internal = _ext.internal;

		// Extend with old style plug-in API methods
		for ( var fn in DataTable.ext.internal ) {
			if ( fn ) {
				this[fn] = _fnExternApiFunc(fn);
			}
		}

		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;

			/*global oInit,_that,emptyInit*/
			var i=0, iLen, j, jLen, k, kLen;
			var sId = this.getAttribute( 'id' );
			var bInitHandedOff = false;
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			
			/* Sanity check */
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, $this.data() ) );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						s.oInstance.fnDestroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId
			} );
			oSettings.nTable = this;
			oSettings.oApi   = _that.internal;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			_fnLanguageCompat( oInit.oLanguage );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = $.isArray( oInit.aLengthMenu[0] ) ?
					oInit.aLengthMenu[0][0] : oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"asStripeClasses",
				"ajax",
				"fnServerData",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"sAjaxSource",
				"sAjaxDataProp",
				"iStateDuration",
				"sDom",
				"bSortCellsTop",
				"iTabIndex",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback,      'user' );
			_fnCallbackReg( oSettings, 'aoServerParams',       oInit.fnServerParams,      'user' );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow,        'user' );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete,      'user' );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback,   'user' );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.sTable );
			
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			if ( oInit.iDeferLoading !== null )
			{
				oSettings.bDeferLoading = true;
				var tmp = $.isArray( oInit.iDeferLoading );
				oSettings._iRecordsDisplay = tmp ? oInit.iDeferLoading[0] : oInit.iDeferLoading;
				oSettings._iRecordsTotal = tmp ? oInit.iDeferLoading[1] : oInit.iDeferLoading;
			}
			
			/* Language definitions */
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl )
			{
				
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnLanguageCompat( json );
						_fnCamelToHungarian( defaults.oLanguage, json );
						$.extend( true, oLanguage, json );
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file, continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
				bInitHandedOff = true;
			}
			
			/*
			 * Stripes
			 */
			if ( oInit.asStripeClasses === null )
			{
				oSettings.asStripeClasses =[
					oClasses.sStripeOdd,
					oClasses.sStripeEven
				];
			}
			
			/* Remove row stripe classes if they are already on the table row */
			var stripeClasses = oSettings.asStripeClasses;
			var rowOne = $this.children('tbody').find('tr').eq(0);
			if ( $.inArray( true, $.map( stripeClasses, function(el, i) {
				return rowOne.hasClass(el);
			} ) ) !== -1 ) {
				$('tbody tr', this).removeClass( stripeClasses.join(' ') );
				oSettings.asDestroyStripes = stripeClasses.slice();
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var anThs = [];
			var aoColumnsInit;
			var nThead = this.getElementsByTagName('thead');
			if ( nThead.length !== 0 )
			{
				_fnDetectHeader( oSettings.aoHeader, nThead[0] );
				anThs = _fnGetUniqueThs( oSettings );
			}
			
			/* If not given a column array, generate one with nulls */
			if ( oInit.aoColumns === null )
			{
				aoColumnsInit = [];
				for ( i=0, iLen=anThs.length ; i<iLen ; i++ )
				{
					aoColumnsInit.push( null );
				}
			}
			else
			{
				aoColumnsInit = oInit.aoColumns;
			}
			
			/* Add the columns */
			for ( i=0, iLen=aoColumnsInit.length ; i<iLen ; i++ )
			{
				_fnAddColumn( oSettings, anThs ? anThs[i] : null );
			}
			
			/* Apply the column definitions */
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, aoColumnsInit, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			var features = oSettings.oFeatures;
			var loadedInit = function () {
				/*
				 * Sorting
				 * @todo For modularisation (1.11) this needs to do into a sort start up handler
				 */
			
				// If aaSorting is not defined, then we use the first indicator in asSorting
				// in case that has been altered, so the default sort reflects that option
				if ( oInit.aaSorting === undefined ) {
					var sorting = oSettings.aaSorting;
					for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
						sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
					}
				}
			
				/* Do a first pass on the sorting classes (allows any size changes to be taken into
				 * account, and also will apply sorting disabled classes if disabled
				 */
				_fnSortingClasses( oSettings );
			
				if ( features.bSort ) {
					_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
						if ( oSettings.bSorted ) {
							var aSort = _fnSortFlatten( oSettings );
							var sortedColumns = {};
			
							$.each( aSort, function (i, val) {
								sortedColumns[ val.src ] = val.dir;
							} );
			
							_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort, sortedColumns] );
							_fnSortAria( oSettings );
						}
					} );
				}
			
				_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
					if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
						_fnSortingClasses( oSettings );
					}
				}, 'sc' );
			
			
				/*
				 * Final init
				 * Cache the header, body and footer as required, creating them if needed
				 */
			
				// Work around for Webkit bug 83867 - store the caption-side before removing from doc
				var captions = $this.children('caption').each( function () {
					this._captionSide = $(this).css('caption-side');
				} );
			
				var thead = $this.children('thead');
				if ( thead.length === 0 ) {
					thead = $('<thead/>').appendTo($this);
				}
				oSettings.nTHead = thead[0];
			
				var tbody = $this.children('tbody');
				if ( tbody.length === 0 ) {
					tbody = $('<tbody/>').appendTo($this);
				}
				oSettings.nTBody = tbody[0];
			
				var tfoot = $this.children('tfoot');
				if ( tfoot.length === 0 && captions.length > 0 && (oSettings.oScroll.sX !== "" || oSettings.oScroll.sY !== "") ) {
					// If we are a scrolling table, and no footer has been given, then we need to create
					// a tfoot element for the caption element to be appended to
					tfoot = $('<tfoot/>').appendTo($this);
				}
			
				if ( tfoot.length === 0 || tfoot.children().length === 0 ) {
					$this.addClass( oClasses.sNoFooter );
				}
				else if ( tfoot.length > 0 ) {
					oSettings.nTFoot = tfoot[0];
					_fnDetectHeader( oSettings.aoFooter, oSettings.nTFoot );
				}
			
				/* Check if there is data passing into the constructor */
				if ( oInit.aaData ) {
					for ( i=0 ; i<oInit.aaData.length ; i++ ) {
						_fnAddData( oSettings, oInit.aaData[ i ] );
					}
				}
				else if ( oSettings.bDeferLoading || _fnDataSource( oSettings ) == 'dom' ) {
					/* Grab the data from the page - only do this when deferred loading or no Ajax
					 * source since there is no point in reading the DOM data if we are then going
					 * to replace it with Ajax data
					 */
					_fnAddTr( oSettings, $(oSettings.nTBody).children('tr') );
				}
			
				/* Copy the data index array */
				oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
				/* Initialisation complete - table can be drawn */
				oSettings.bInitialised = true;
			
				/* Check if we need to initialise the table (it might not have been handed off to the
				 * language processor)
				 */
				if ( bInitHandedOff === false ) {
					_fnInitialise( oSettings );
				}
			};
			
			/* Must be done after everything which can be overridden by the state saving! */
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
				_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState, 'state_save' );
				_fnLoadState( oSettings, oInit, loadedInit );
			}
			else {
				loadedInit();
			}
			
		} );
		_that = null;
		return this;
	};
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n]/g;
	var _re_html = /<.*?>/g;
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[\.\/\-]\d{1,2}[\.\/\-]\d{1,2}([T ]{1}\d{1,2}[:\.]\d{2}([\.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	
	var _re_formatted_numeric = /[',$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted ) {
		var strType = typeof d === 'string';
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	
	var _htmlNumeric = function ( d, decimalPoint, formatted ) {
		if ( _empty( d ) ) {
			return true;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				out.push( a[ order[i] ][ prop ] );
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	
	var _stripHtml = function ( d ) {
		return d.replace( _re_html, '' );
	};
	
	
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.com/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	DataTable.util = {
		
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
	
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		}
	};
	
	
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key, val) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key, val) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	function _fnLanguageCompat( lang )
	{
		// Note the use of the Hungarian notation for the parameters in this method as
		// this is called after the mapping of camelCase to Hungarian
		var defaults = DataTable.defaults.oLanguage;
	
		// Default mapping
		var defaultDecimal = defaults.sDecimal;
		if ( defaultDecimal ) {
			_addNumericSort( defaultDecimal );
		}
	
		if ( lang ) {
			var zeroRecords = lang.sZeroRecords;
	
			// Backwards compatibility - if there is no sEmptyTable given, then use the same as
			// sZeroRecords - assuming that is given.
			if ( ! lang.sEmptyTable && zeroRecords &&
				defaults.sEmptyTable === "No data available in table" )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sEmptyTable' );
			}
	
			// Likewise with loading records
			if ( ! lang.sLoadingRecords && zeroRecords &&
				defaults.sLoadingRecords === "Loading..." )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sLoadingRecords' );
			}
	
			// Old parameter name of the thousands separator mapped onto the new
			if ( lang.sInfoThousands ) {
				lang.sThousands = lang.sInfoThousands;
			}
	
			var decimal = lang.sDecimal;
			if ( decimal && defaultDecimal !== decimal ) {
				_addNumericSort( decimal );
			}
		}
	}
	
	
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	}
	
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! $.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	function _fnBrowserDetect( settings )
	{
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: $(window).scrollLeft()*-1, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();

			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			browser.bScrollOversize = inner[0].offsetWidth === 100 && outer[0].clientWidth !== 100;
	
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			// IE8- don't provide height and width for getBoundingClientRect
			browser.bBounding = n[0].getBoundingClientRect().width ? true : false;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	
	function _fnReduce ( that, fn, init, start, end, inc )
	{
		var
			i = start,
			value,
			isSet = false;
	
		if ( init !== undefined ) {
			value = init;
			isSet = true;
		}
	
		while ( i !== end ) {
			if ( ! that.hasOwnProperty(i) ) {
				continue;
			}
	
			value = isSet ?
				fn( value, that[i], i, that ) :
				that[i];
	
			isSet = true;
			i += inc;
		}
	
		return value;
	}

	function _fnAddColumn( oSettings, nTh )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"nTh": nTh ? nTh : document.createElement('th'),
			"sTitle":    oDefaults.sTitle    ? oDefaults.sTitle    : nTh ? nTh.innerHTML : '',
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol
		} );
		oSettings.aoColumns.push( oCol );
	
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	
		// Use the default column options function to initialise classes etc
		_fnColumnOptions( oSettings, iCol, $(nTh).data() );
	}
	
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
		var oClasses = oSettings.oClasses;
		var th = $(oCol.nTh);

		if ( ! oCol.sWidthOrig ) {
			// Width attribute
			oCol.sWidthOrig = th.attr('width') || null;
	
			// Style attribute
			var t = (th.attr('style') || '').match(/width:\s*(\d+[pxem%]+)/);
			if ( t ) {
				oCol.sWidthOrig = t[1];
			}
		}
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
	
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
			if ( oOptions.sClass ) {
				th.addClass( oOptions.sClass );
			}
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
		var mRender = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return mRender && type ?
				mRender( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
			th.addClass( oClasses.sSortableNone ); // Have to add class here as order event isn't called
		}
	
		/* Check that the class assignment is correct for sorting */
		var bAsc = $.inArray('asc', oCol.asSorting) !== -1;
		var bDesc = $.inArray('desc', oCol.asSorting) !== -1;
		if ( !oCol.bSortable || (!bAsc && !bDesc) )
		{
			oCol.sSortingClass = oClasses.sSortableNone;
			oCol.sSortingClassJUI = "";
		}
		else if ( bAsc && !bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableAsc;
			oCol.sSortingClassJUI = oClasses.sSortJUIAscAllowed;
		}
		else if ( !bAsc && bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableDesc;
			oCol.sSortingClassJUI = oClasses.sSortJUIDescAllowed;
		}
		else
		{
			oCol.sSortingClass = oClasses.sSortable;
			oCol.sSortingClassJUI = oClasses.sSortJUI;
		}
	}
	
	function _fnAdjustColumnSizing ( settings )
	{
		/* Not interested in doing column width calculation if auto-width is disabled */
		if ( settings.oFeatures.bAutoWidth !== false )
		{
			var columns = settings.aoColumns;
	
			_fnCalculateColumnWidths( settings );
			for ( var i=0 , iLen=columns.length ; i<iLen ; i++ )
			{
				columns[i].nTh.style.width = columns[i].sWidth;
			}
		}
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '')
		{
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Covert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = $.inArray( iMatch, aiVis );
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	function _fnVisbleColumns( oSettings )
	{
		var vis = 0;
	
		// No reduce in IE8, use a loop for now
		$.each( oSettings.aoColumns, function ( i, col ) {
			if ( col.bVisible && $(col.nTh).css('display') !== 'none' ) {
				vis++;
			}
		} );
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		$.map( oSettings.aoColumns, function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, cell, detectedType, cache;
	
		// For each column, spin over the 
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						detectedType = types[j]( cache[k], settings );
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-1 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string
						if ( detectedType === 'html' ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( detectedType ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
		}
	}
	
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.targets !== undefined ?
					def.targets :
					def.aTargets;
	
				if ( ! $.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					if ( typeof aTargets[j] === 'number' && aTargets[j] >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= aTargets[j] )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'number' && aTargets[j] < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'string' )
					{
						/* Class name matching on TH element */
						for ( k=0, kLen=columns.length ; k<kLen ; k++ )
						{
							if ( aTargets[j] == "_all" ||
							     $(columns[k].nTh).hasClass( aTargets[j] ) )
							{
								fn( k, def );
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols )
		{
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ )
			{
				fn( i, aoCols[i] );
			}
		}
	}
	
	function _fnAddData ( oSettings, aDataIn, nTr, anTds )
	{
		/* Create the object for storing information about this new row */
		var iRow = oSettings.aoData.length;
		var oData = $.extend( true, {}, DataTable.models.oRow, {
			src: nTr ? 'dom' : 'data',
			idx: iRow
		} );
	
		oData._aData = aDataIn;
		oSettings.aoData.push( oData );
	
		/* Create the cells */
		var nTd, sThisType;
		var columns = oSettings.aoColumns;
	
		// Invalidate the column types as the new data needs to be revalidated
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		oSettings.aiDisplayMaster.push( iRow );
	
		var id = oSettings.rowIdFn( aDataIn );
		if ( id !== undefined ) {
			oSettings.aIds[ id ] = oData;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( nTr || ! oSettings.oFeatures.bDeferRender )
		{
			_fnCreateTr( oSettings, iRow, nTr, anTds );
		}
	
		return iRow;
	}
	
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}

	function _fnNodeToDataIndex( oSettings, n )
	{
		return (n._DT_RowIndex!==undefined) ? n._DT_RowIndex : null;
	}
	
	function _fnNodeToColumnIndex( oSettings, iRow, n )
	{
		return $.inArray( n, oSettings.aoData[ iRow ].anCells );
	}
	
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = settings.aoData[rowIdx]._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type == 'display' ) {
			return '';
		}
		return cellData;
	}
	
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	
	function _fnSplitObjNotation( str )
	{
		return $.map( str.match(/(\\.|[^\.])+/g) || [''], function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	function _fnGetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			/* Build an object of get functions, and wrap them in a single call */
			var o = {};
			$.each( mSource, function (key, val) {
				if ( val ) {
					o[key] = _fnGetObjectDataFn( val );
				}
			} );
	
			return function (data, type, row, meta) {
				var t = o[type] || o._;
				return t !== undefined ?
					t(data, type, row, meta) :
					data;
			};
		}
		else if ( mSource === null )
		{
			/* Give an empty string for rendering / sorting etc */
			return function (data) { // type, row and meta also passed, but not used
				return data;
			};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, type, row, meta) {
				return mSource( data, type, row, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			
			var fetchData = function (data, type, src) {
				var arrayNotation, funcNotation, out, innerSrc;
	
				if ( src !== "" )
				{
					var a = _fnSplitObjNotation( src );
	
					for ( var i=0, iLen=a.length ; i<iLen ; i++ )
					{
						// Check if we are dealing with special notation
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
	
						if ( arrayNotation )
						{
							// Array notation
							a[i] = a[i].replace(__reArray, '');
	
							// Condition allows simply [] to be passed in
							if ( a[i] !== "" ) {
								data = data[ a[i] ];
							}
							out = [];
	
							// Get the remainder of the nested object to get
							a.splice( 0, i+1 );
							innerSrc = a.join('.');
	
							// Traverse each entry in the array getting the properties requested
							if ( $.isArray( data ) ) {
								for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
									out.push( fetchData( data[j], type, innerSrc ) );
								}
							}
	
							// If a string is given in between the array notation indicators, that
							// is used to join the strings together, otherwise an array is returned
							var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
							data = (join==="") ? out : out.join(join);
	
							// The inner call to fetchData has already traversed through the remainder
							// of the source requested, so we exit from the loop
							break;
						}
						else if ( funcNotation )
						{
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]();
							continue;
						}
	
						if ( data === null || data[ a[i] ] === undefined )
						{
							return undefined;
						}
						data = data[ a[i] ];
					}
				}
	
				return data;
			};
	
			return function (data, type) { // row and meta also passed, but not used
				return fetchData( data, type, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, type) { // row and meta also passed, but not used
				return data[mSource];
			};
		}
	}
	
	
	function _fnSetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			
			return _fnSetObjectDataFn( mSource._ );
		}
		else if ( mSource === null )
		{
			/* Nothing to do when the data source is null */
			return function () {};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, val, meta) {
				mSource( data, 'set', val, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			/* Like the get, we need to get data from a nested object */
			var setData = function (data, val, src) {
				var a = _fnSplitObjNotation( src ), b;
				var aLast = a[a.length-1];
				var arrayNotation, funcNotation, o, innerSrc;
	
				for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ )
				{
					// Check if we are dealing with an array notation request
					arrayNotation = a[i].match(__reArray);
					funcNotation = a[i].match(__reFn);
	
					if ( arrayNotation )
					{
						a[i] = a[i].replace(__reArray, '');
						data[ a[i] ] = [];
	
						// Get the remainder of the nested object to set so we can recurse
						b = a.slice();
						b.splice( 0, i+1 );
						innerSrc = b.join('.');
	
						// Traverse each entry in the array setting the properties requested
						if ( $.isArray( val ) )
						{
							for ( var j=0, jLen=val.length ; j<jLen ; j++ )
							{
								o = {};
								setData( o, val[j], innerSrc );
								data[ a[i] ].push( o );
							}
						}
						else
						{
							// We've been asked to save data to an array, but it
							// isn't array data to be saved. Best that can be done
							// is to just save the value.
							data[ a[i] ] = val;
						}
	
						// The inner call to setData has already traversed through the remainder
						// of the source and has set the data, thus we can exit here
						return;
					}
					else if ( funcNotation )
					{
						// Function call
						a[i] = a[i].replace(__reFn, '');
						data = data[ a[i] ]( val );
					}
	
					// If the nested object doesn't currently exist - since we are
					// trying to set the value - create it
					if ( data[ a[i] ] === null || data[ a[i] ] === undefined )
					{
						data[ a[i] ] = {};
					}
					data = data[ a[i] ];
				}
	
				// Last item in the input - i.e, the actual set
				if ( aLast.match(__reFn ) )
				{
					// Function call
					data = data[ aLast.replace(__reFn, '') ]( val );
				}
				else
				{
					// If array notation is used, we just want to strip it and use the property name
					// and assign the value. If it isn't used, then we get the result we want anyway
					data[ aLast.replace(__reArray, '') ] = val;
				}
			};
	
			return function (data, val) { // meta is also passed in, but not used
				return setData( data, val, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, val) { // meta is also passed in, but not used
				data[mSource] = val;
			};
		}
	}
	
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	function _fnDeleteIndex( a, iTarget, splice )
	{
		var iTargetIndex = -1;
	
		for ( var i=0, iLen=a.length ; i<iLen ; i++ )
		{
			if ( a[i] == iTarget )
			{
				iTargetIndex = i;
			}
			else if ( a[i] > iTarget )
			{
				a[i]--;
			}
		}
	
		if ( iTargetIndex != -1 && splice === undefined )
		{
			a.splice( iTargetIndex, 1 );
		}
	}
	
	
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
		var cellWrite = function ( cell, col ) {
			// This is very frustrating, but in IE if you just write directly
			// to innerHTML, and elements that are overwritten are GC'ed,
			// even if there is a reference to them elsewhere
			while ( cell.childNodes.length ) {
				cell.removeChild( cell.firstChild );
			}
	
			cell.innerHTML = _fnGetCellData( settings, rowIdx, col, 'display' );
		};
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					cellWrite( cells[colIdx], colIdx );
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						cellWrite( cells[i], i );
					}
				}
			}
		}
	
		// For both row and cell invalidation, the cached data for sorting and
		// filtering is nulled out
		row._aSortData = null;
		row._aFilterData = null;
	
		// Invalidate the type for a specific column (if given) or all columns since
		// the data might have changed
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			cols[ colIdx ].sType = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, o, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = $.trim(cell.innerHTML);
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
	
				nTd = nTrIn ? anTds[i] : document.createElement( oCol.sCellType );
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
	
				// Need to create the HTML if new, or if a rendering function is defined
				if ( (!nTrIn || oCol.mRender || oCol.mData !== i) &&
					 (!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
				) {
					nTd.innerHTML = _fnGetCellData( oSettings, iRow, i, 'display' );
				}
	
				/* Add user defined class */
				if ( oCol.sClass )
				{
					nTd.className += ' '+oCol.sClass;
				}
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && ! nTrIn )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && nTrIn )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', null, [nTr, rowData, iRow, cells] );
		}
	
		// Remove once webkit bug 131819 and Chromium bug 365619 have been resolved
		// and deployed
		row.nTr.setAttribute( 'role', 'row' );
	}
	
	
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	
	function _fnBuildHead( oSettings )
	{
		var i, ien, cell, row, column;
		var thead = oSettings.nTHead;
		var tfoot = oSettings.nTFoot;
		var createHeader = $('th, td', thead).length === 0;
		var classes = oSettings.oClasses;
		var columns = oSettings.aoColumns;
	
		if ( createHeader ) {
			row = $('<tr/>').appendTo( thead );
		}
	
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			column = columns[i];
			cell = $( column.nTh ).addClass( column.sClass );
	
			if ( createHeader ) {
				cell.appendTo( row );
			}
	
			// 1.11 move into sorting
			if ( oSettings.oFeatures.bSort ) {
				cell.addClass( column.sSortingClass );
	
				if ( column.bSortable !== false ) {
					cell
						.attr( 'tabindex', oSettings.iTabIndex )
						.attr( 'aria-controls', oSettings.sTableId );
	
					_fnSortAttachListener( oSettings, column.nTh, i );
				}
			}
	
			if ( column.sTitle != cell[0].innerHTML ) {
				cell.html( column.sTitle );
			}
	
			_fnRenderer( oSettings, 'header' )(
				oSettings, cell, column, classes
			);
		}
	
		if ( createHeader ) {
			_fnDetectHeader( oSettings.aoHeader, thead );
		}
		
		/* ARIA role for the rows */
	 	$(thead).find('>tr').attr('role', 'row');
	
		/* Deal with the footer - add classes if required */
		$(thead).find('>tr>th, >tr>td').addClass( classes.sHeaderTH );
		$(tfoot).find('>tr>th, >tr>td').addClass( classes.sFooterTH );
	
		// Cache the footer cells. Note that we only take the cells from the first
		// row in the footer. If there is more than one row the user wants to
		// interact with, they need to use the table().foot() method. Note also this
		// allows cells to be used for multiple columns using colspan
		if ( tfoot !== null ) {
			var cells = oSettings.aoFooter[0];
	
			for ( i=0, ien=cells.length ; i<ien ; i++ ) {
				column = columns[i];
				column.nTf = cells[i].cell;
	
				if ( column.sClass ) {
					$(column.nTf).addClass( column.sClass );
				}
			}
		}
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states. The
	 * methodology here is to use the layout array from _fnDetectHeader, modified for
	 * the instantaneous column visibility, to construct the new layout. The grid is
	 * traversed over cell at a time in a rows x columns grid fashion, although each
	 * cell insert can cover multiple elements in the grid - which is tracks using the
	 * aApplied array. Cell inserts in the grid will only occur where there isn't
	 * already a cell in that position.
	 *  @param {object} oSettings dataTables settings object
	 *  @param array {objects} aoSource Layout array from _fnDetectHeader
	 *  @param {boolean} [bIncludeHidden=false] If true then include the hidden columns in the calc,
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( oSettings, aoSource, bIncludeHidden )
	{
		var i, iLen, j, jLen, k, kLen, n, nLocalTr;
		var aoLocal = [];
		var aApplied = [];
		var iColumns = oSettings.aoColumns.length;
		var iRowspan, iColspan;
	
		if ( ! aoSource )
		{
			return;
		}
	
		if (  bIncludeHidden === undefined )
		{
			bIncludeHidden = false;
		}
	
		/* Make a copy of the master layout array, but without the visible columns in it */
		for ( i=0, iLen=aoSource.length ; i<iLen ; i++ )
		{
			aoLocal[i] = aoSource[i].slice();
			aoLocal[i].nTr = aoSource[i].nTr;
	
			/* Remove any columns which are currently hidden */
			for ( j=iColumns-1 ; j>=0 ; j-- )
			{
				if ( !oSettings.aoColumns[j].bVisible && !bIncludeHidden )
				{
					aoLocal[i].splice( j, 1 );
				}
			}
	
			/* Prep the applied array - it needs an element for each row */
			aApplied.push( [] );
		}
	
		for ( i=0, iLen=aoLocal.length ; i<iLen ; i++ )
		{
			nLocalTr = aoLocal[i].nTr;
	
			/* All cells are going to be replaced, so empty out the row */
			if ( nLocalTr )
			{
				while( (n = nLocalTr.firstChild) )
				{
					nLocalTr.removeChild( n );
				}
			}
	
			for ( j=0, jLen=aoLocal[i].length ; j<jLen ; j++ )
			{
				iRowspan = 1;
				iColspan = 1;
	
				/* Check to see if there is already a cell (row/colspan) covering our target
				 * insert point. If there is, then there is nothing to do.
				 */
				if ( aApplied[i][j] === undefined )
				{
					nLocalTr.appendChild( aoLocal[i][j].cell );
					aApplied[i][j] = 1;
	
					/* Expand the cell to cover as many rows as needed */
					while ( aoLocal[i+iRowspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i+iRowspan][j].cell )
					{
						aApplied[i+iRowspan][j] = 1;
						iRowspan++;
					}
	
					/* Expand the cell to cover as many columns as needed */
					while ( aoLocal[i][j+iColspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i][j+iColspan].cell )
					{
						/* Must update the applied array over the rows for the columns */
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aApplied[i+k][j+iColspan] = 1;
						}
						iColspan++;
					}
	
					/* Do the actual expansion in the DOM */
					$(aoLocal[i][j].cell)
						.attr('rowspan', iRowspan)
						.attr('colspan', iColspan);
				}
			}
		}
	}
	
	
	function _fnDraw( oSettings )
	{
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( $.inArray( false, aPreDraw ) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var i, iLen, n;
		var anRows = [];
		var iRowCount = 0;
		var asStripeClasses = oSettings.asStripeClasses;
		var iStripes = asStripeClasses.length;
		var iOpenRows = oSettings.aoOpenRows.length;
		var oLang = oSettings.oLanguage;
		var iInitDisplayStart = oSettings.iInitDisplayStart;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
	
		oSettings.bDrawing = true;
	
		/* Check and see if we have an initial draw position from state saving */
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
	
		/* Server-side processing draw intercept */
		if ( oSettings.bDeferLoading )
		{
			oSettings.bDeferLoading = false;
			oSettings.iDraw++;
			_fnProcessingDisplay( oSettings, false );
		}
		else if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !_fnAjaxUpdate( oSettings ) )
		{
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				/* Remove the old striping classes and then add the new one */
				if ( iStripes !== 0 )
				{
					var sStripe = asStripeClasses[ iRowCount % iStripes ];
					if ( aoData._sRowStripe != sStripe )
					{
						$(nRow).removeClass( aoData._sRowStripe ).addClass( sStripe );
						aoData._sRowStripe = sStripe;
					}
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			/* Table is empty - create a row with an empty message in it */
			var sZero = oLang.sZeroRecords;
			if ( oSettings.iDraw == 1 &&  _fnDataSource( oSettings ) == 'ajax' )
			{
				sZero = oLang.sLoadingRecords;
			}
			else if ( oLang.sEmptyTable && oSettings.fnRecordsTotal() === 0 )
			{
				sZero = oLang.sEmptyTable;
			}
	
			anRows[ 0 ] = $( '<tr/>', { 'class': iStripes ? asStripeClasses[0] : '' } )
				.append( $('<td />', {
					'valign':  'top',
					'colSpan': _fnVisbleColumns( oSettings ),
					'class':   oSettings.oClasses.sRowEmpty
				} ).html( sZero ) )[0];
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		var body = $(oSettings.nTBody);
	
		body.children().detach();
		body.append( $(anRows) );
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings] );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	

	function _fnReDraw( settings, holdPosition )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if ( sort ) {
			_fnSort( settings );
		}
	
		if ( filter ) {
			_fnFilterComplete( settings, settings.oPreviousSearch );
		}
		else {
			// No filtering, so we want to just use the display master
			settings.aiDisplay = settings.aiDisplayMaster.slice();
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings._drawHold = false;
	}
	
	
	function _fnAddOptionsHtml ( oSettings )
	{
		var classes = oSettings.oClasses;
		var table = $(oSettings.nTable);
		var holding = $('<div/>').insertBefore( table ); // Holding element for speed
		var features = oSettings.oFeatures;
	
		// All DataTables are wrapped in a div
		var insert = $('<div/>', {
			id:      oSettings.sTableId+'_wrapper',
			'class': classes.sWrapper + (oSettings.nTFoot ? '' : ' '+classes.sNoFooter)
		} );
	
		oSettings.nHolding = holding[0];
		oSettings.nTableWrapper = insert[0];
		oSettings.nTableReinsertBefore = oSettings.nTable.nextSibling;
	
		/* Loop over the user set positioning and place the elements as needed */
		var aDom = oSettings.sDom.split('');
		var featureNode, cOption, nNewNode, cNext, sAttr, j;
		for ( var i=0 ; i<aDom.length ; i++ )
		{
			featureNode = null;
			cOption = aDom[i];
	
			if ( cOption == '<' )
			{
				/* New container div */
				nNewNode = $('<div/>')[0];
	
				/* Check to see if we should append an id and/or a class name to the container */
				cNext = aDom[i+1];
				if ( cNext == "'" || cNext == '"' )
				{
					sAttr = "";
					j = 2;
					while ( aDom[i+j] != cNext )
					{
						sAttr += aDom[i+j];
						j++;
					}
	
					/* Replace jQuery UI constants @todo depreciated */
					if ( sAttr == "H" )
					{
						sAttr = classes.sJUIHeader;
					}
					else if ( sAttr == "F" )
					{
						sAttr = classes.sJUIFooter;
					}
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( sAttr.indexOf('.') != -1 )
					{
						var aSplit = sAttr.split('.');
						nNewNode.id = aSplit[0].substr(1, aSplit[0].length-1);
						nNewNode.className = aSplit[1];
					}
					else if ( sAttr.charAt(0) == "#" )
					{
						nNewNode.id = sAttr.substr(1, sAttr.length-1);
					}
					else
					{
						nNewNode.className = sAttr;
					}
	
					i += j; /* Move along the position array */
				}
	
				insert.append( nNewNode );
				insert = $(nNewNode);
			}
			else if ( cOption == '>' )
			{
				/* End container div */
				insert = insert.parent();
			}
			// @todo Move options into their own plugins?
			else if ( cOption == 'l' && features.bPaginate && features.bLengthChange )
			{
				/* Length */
				featureNode = _fnFeatureHtmlLength( oSettings );
			}
			else if ( cOption == 'f' && features.bFilter )
			{
				/* Filter */
				featureNode = _fnFeatureHtmlFilter( oSettings );
			}
			else if ( cOption == 'r' && features.bProcessing )
			{
				/* pRocessing */
				featureNode = _fnFeatureHtmlProcessing( oSettings );
			}
			else if ( cOption == 't' )
			{
				/* Table */
				featureNode = _fnFeatureHtmlTable( oSettings );
			}
			else if ( cOption ==  'i' && features.bInfo )
			{
				/* Info */
				featureNode = _fnFeatureHtmlInfo( oSettings );
			}
			else if ( cOption == 'p' && features.bPaginate )
			{
				/* Pagination */
				featureNode = _fnFeatureHtmlPaginate( oSettings );
			}
			else if ( DataTable.ext.feature.length !== 0 )
			{
				/* Plug-in features */
				var aoFeatures = DataTable.ext.feature;
				for ( var k=0, kLen=aoFeatures.length ; k<kLen ; k++ )
				{
					if ( cOption == aoFeatures[k].cFeature )
					{
						featureNode = aoFeatures[k].fnInit( oSettings );
						break;
					}
				}
			}
	
			/* Add to the 2D features array */
			if ( featureNode )
			{
				var aanFeatures = oSettings.aanFeatures;
	
				if ( ! aanFeatures[cOption] )
				{
					aanFeatures[cOption] = [];
				}
	
				aanFeatures[cOption].push( featureNode );
				insert.append( featureNode );
			}
		}
	
		/* Built our DOM structure - replace the holding div with what we want */
		holding.replaceWith( insert );
		oSettings.nHolding = null;
	}
	
	
	function _fnDetectHeader ( aLayout, nThead )
	{
		var nTrs = $(nThead).children('tr');
		var nTr, nCell;
		var i, k, l, iLen, jLen, iColShifted, iColumn, iColspan, iRowspan;
		var bUnique;
		var fnShiftCol = function ( a, i, j ) {
			var k = a[i];
	                while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		aLayout.splice( 0, aLayout.length );
	
		/* We know how many rows there are in the layout - so prep it */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			aLayout.push( [] );
		}
	
		/* Calculate a layout array */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			nTr = nTrs[i];
			iColumn = 0;
	
			/* For every cell in the row... */
			nCell = nTr.firstChild;
			while ( nCell ) {
				if ( nCell.nodeName.toUpperCase() == "TD" ||
				     nCell.nodeName.toUpperCase() == "TH" )
				{
					/* Get the col and rowspan attributes from the DOM and sanitise them */
					iColspan = nCell.getAttribute('colspan') * 1;
					iRowspan = nCell.getAttribute('rowspan') * 1;
					iColspan = (!iColspan || iColspan===0 || iColspan===1) ? 1 : iColspan;
					iRowspan = (!iRowspan || iRowspan===0 || iRowspan===1) ? 1 : iRowspan;
	
					/* There might be colspan cells already in this row, so shift our target
					 * accordingly
					 */
					iColShifted = fnShiftCol( aLayout, i, iColumn );
	
					/* Cache calculation for unique columns */
					bUnique = iColspan === 1 ? true : false;
	
					/* If there is col / rowspan, copy the information into the layout grid */
					for ( l=0 ; l<iColspan ; l++ )
					{
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aLayout[i+k][iColShifted+l] = {
								"cell": nCell,
								"unique": bUnique
							};
							aLayout[i+k].nTr = nTr;
						}
					}
				}
				nCell = nCell.nextSibling;
			}
		}
	}
	
	
	function _fnGetUniqueThs ( oSettings, nHeader, aLayout )
	{
		var aReturn = [];
		if ( !aLayout )
		{
			aLayout = oSettings.aoHeader;
			if ( nHeader )
			{
				aLayout = [];
				_fnDetectHeader( aLayout, nHeader );
			}
		}
	
		for ( var i=0, iLen=aLayout.length ; i<iLen ; i++ )
		{
			for ( var j=0, jLen=aLayout[i].length ; j<jLen ; j++ )
			{
				if ( aLayout[i][j].unique &&
					 (!aReturn[j] || !oSettings.bSortCellsTop) )
				{
					aReturn[j] = aLayout[i][j].cell;
				}
			}
		}
	
		return aReturn;
	}
	
	function _fnBuildAjax( oSettings, data, fn )
	{
		// Compatibility with 1.9-, allow fnServerData and event to manipulate
		_fnCallbackFire( oSettings, 'aoServerParams', 'serverParams', [data] );
	
		// Convert to object based for 1.10+ if using the old array scheme which can
		// come from server-side processing or serverParams
		if ( data && $.isArray(data) ) {
			var tmp = {};
			var rbracket = /(.*?)\[\]$/;
	
			$.each( data, function (key, val) {
				var match = val.name.match(rbracket);
	
				if ( match ) {
					// Support for arrays
					var name = match[0];
	
					if ( ! tmp[ name ] ) {
						tmp[ name ] = [];
					}
					tmp[ name ].push( val.value );
				}
				else {
					tmp[val.name] = val.value;
				}
			} );
			data = tmp;
		}
	
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR] );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"data": data,
			"success": function (json) {
				var error = json.error || json.sError;
				if ( error ) {
					_fnLog( oSettings, 0, error );
				}
	
				oSettings.json = json;
				callback( json );
			},
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error, thrown) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR] );
	
				if ( $.inArray( true, ret ) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data] );
	
		if ( oSettings.fnServerData )
		{
			// DataTables 1.9- compatibility
			oSettings.fnServerData.call( instance,
				oSettings.sAjaxSource,
				$.map( data, function (val, key) { // Need to convert back to 1.9 trad format
					return { name: key, value: val };
				} ),
				callback,
				oSettings
			);
		}
		else if ( oSettings.sAjaxSource || typeof ajax === 'string' )
		{
			// DataTables 1.9- compatibility
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, {
				url: ajax || oSettings.sAjaxSource
			} ) );
		}
		else if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else
		{
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, ajax ) );
	
			// Restore for next time around
			ajax.data = ajaxData;
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		if ( settings.bAjaxDataGet ) {
			settings.iDraw++;
			_fnProcessingDisplay( settings, true );
	
			_fnBuildAjax(
				settings,
				_fnAjaxParameters( settings ),
				function(json) {
					_fnAjaxUpdateDraw( settings, json );
				}
			);
	
			return false;
		}
		return true;
	}
	
	
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			columnCount = columns.length,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			i, data = [], dataProp, column, columnSearch,
			sort = _fnSortFlatten( settings ),
			displayStart = settings._iDisplayStart,
			displayLength = features.bPaginate !== false ?
				settings._iDisplayLength :
				-1;
	
		var param = function ( name, value ) {
			data.push( { 'name': name, 'value': value } );
		};
	
		// DataTables 1.9- compatible method
		param( 'sEcho',          settings.iDraw );
		param( 'iColumns',       columnCount );
		param( 'sColumns',       _pluck( columns, 'sName' ).join(',') );
		param( 'iDisplayStart',  displayStart );
		param( 'iDisplayLength', displayLength );
	
		// DataTables 1.10+ method
		var d = {
			draw:    settings.iDraw,
			columns: [],
			order:   [],
			start:   displayStart,
			length:  displayLength,
			search:  {
				value: preSearch.sSearch,
				regex: preSearch.bRegex
			}
		};
	
		for ( i=0 ; i<columnCount ; i++ ) {
			column = columns[i];
			columnSearch = preColSearch[i];
			dataProp = typeof column.mData=="function" ? 'function' : column.mData ;
	
			d.columns.push( {
				data:       dataProp,
				name:       column.sName,
				searchable: column.bSearchable,
				orderable:  column.bSortable,
				search:     {
					value: columnSearch.sSearch,
					regex: columnSearch.bRegex
				}
			} );
	
			param( "mDataProp_"+i, dataProp );
	
			if ( features.bFilter ) {
				param( 'sSearch_'+i,     columnSearch.sSearch );
				param( 'bRegex_'+i,      columnSearch.bRegex );
				param( 'bSearchable_'+i, column.bSearchable );
			}
	
			if ( features.bSort ) {
				param( 'bSortable_'+i, column.bSortable );
			}
		}
	
		if ( features.bFilter ) {
			param( 'sSearch', preSearch.sSearch );
			param( 'bRegex', preSearch.bRegex );
		}
	
		if ( features.bSort ) {
			$.each( sort, function ( i, val ) {
				d.order.push( { column: val.col, dir: val.dir } );
	
				param( 'iSortCol_'+i, val.col );
				param( 'sSortDir_'+i, val.dir );
			} );
	
			param( 'iSortingCols', sort.length );
		}
	
		// If the legacy.ajax parameter is null, then we automatically decide which
		// form to use, based on sAjaxSource
		var legacy = DataTable.ext.legacy.ajax;
		if ( legacy === null ) {
			return settings.sAjaxSource ? data : d;
		}
	
		// Otherwise, if legacy has been specified then we use that to decide on the
		// form
		return legacy ? data : d;
	}
	
	function _fnAjaxUpdateDraw ( settings, json )
	{
		// v1.10 uses camelCase variables, while 1.9 uses Hungarian notation.
		// Support both
		var compat = function ( old, modern ) {
			return json[old] !== undefined ? json[old] : json[modern];
		};
	
		var data = _fnAjaxDataSrc( settings, json );
		var draw            = compat( 'sEcho',                'draw' );
		var recordsTotal    = compat( 'iTotalRecords',        'recordsTotal' );
		var recordsFiltered = compat( 'iTotalDisplayRecords', 'recordsFiltered' );
	
		if ( draw ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		settings.bAjaxDataGet = false;
		_fnDraw( settings );
	
		if ( ! settings._bInitComplete ) {
			_fnInitComplete( settings, json );
		}
	
		settings.bAjaxDataGet = true;
		_fnProcessingDisplay( settings, false );
	}
	
	function _fnAjaxDataSrc ( oSettings, json )
	{
		var dataSrc = $.isPlainObject( oSettings.ajax ) && oSettings.ajax.dataSrc !== undefined ?
			oSettings.ajax.dataSrc :
			oSettings.sAjaxDataProp; // Compatibility with 1.9-.
	
		// Compatibility with 1.9-. In order to read from aaData, check if the
		// default has been changed, if not, check for aaData
		if ( dataSrc === 'data' ) {
			return json.aaData || json[dataSrc];
		}
	
		return dataSrc !== "" ?
			_fnGetObjectDataFn( dataSrc )( json ) :
			json;
	}
	
	
	function _fnFeatureHtmlFilter ( settings )
	{
		var classes = settings.oClasses;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var features = settings.aanFeatures;
		var input = '<input type="search" class="'+classes.sFilterInput+'"/>';
	
		var str = language.sSearch;
		str = str.match(/_INPUT_/) ?
			str.replace('_INPUT_', input) :
			str+input;
	
		var filter = $('<div/>', {
				'id': ! features.f ? tableId+'_filter' : null,
				'class': classes.sFilter
			} )
			.append( $('<label/>' ).append( str ) );
	
		var searchFn = function() {
			/* Update all other filter input elements for the new display */
			var n = features.f;
			var val = !this.value ? "" : this.value; // mental IE8 fix :-(
	
			/* Now do the filter */
			if ( val != previousSearch.sSearch ) {
				_fnFilterComplete( settings, {
					"sSearch": val,
					"bRegex": previousSearch.bRegex,
					"bSmart": previousSearch.bSmart ,
					"bCaseInsensitive": previousSearch.bCaseInsensitive
				} );
	
				// Need to redraw, without resorting
				settings._iDisplayStart = 0;
				_fnDraw( settings );
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			_fnDataSource( settings ) === 'ssp' ?
				400 :
				0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.sSearch )
			.attr( 'placeholder', language.sSearchPlaceholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					_fnThrottle( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s ) {
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame...
				try {
					if ( jqFilter[0] !== document.activeElement ) {
						jqFilter.val( previousSearch.sSearch );
					}
				}
				catch ( e ) {}
			}
		} );
	
		return filter[0];
	}
	
	
	function _fnFilterComplete ( oSettings, oInput, iForce )
	{
		var oPrevSearch = oSettings.oPreviousSearch;
		var aoPrevSearch = oSettings.aoPreSearchCols;
		var fnSaveFilter = function ( oFilter ) {
			/* Save the filtering values */
			oPrevSearch.sSearch = oFilter.sSearch;
			oPrevSearch.bRegex = oFilter.bRegex;
			oPrevSearch.bSmart = oFilter.bSmart;
			oPrevSearch.bCaseInsensitive = oFilter.bCaseInsensitive;
		};
		var fnRegex = function ( o ) {
			// Backwards compatibility with the bEscapeRegex option
			return o.bEscapeRegex !== undefined ? !o.bEscapeRegex : o.bRegex;
		};
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo As per sort - can this be moved into an event handler?
		_fnColumnTypes( oSettings );
	
		/* In server-side processing all filtering is done by the server, so no point hanging around here */
		if ( _fnDataSource( oSettings ) != 'ssp' )
		{
			/* Global filter */
			_fnFilter( oSettings, oInput.sSearch, iForce, fnRegex(oInput), oInput.bSmart, oInput.bCaseInsensitive );
			fnSaveFilter( oInput );
	
			/* Now do the individual column filter */
			for ( var i=0 ; i<aoPrevSearch.length ; i++ )
			{
				_fnFilterColumn( oSettings, aoPrevSearch[i].sSearch, i, fnRegex(aoPrevSearch[i]),
					aoPrevSearch[i].bSmart, aoPrevSearch[i].bCaseInsensitive );
			}
	
			/* Custom filtering */
			_fnFilterCustom( oSettings );
		}
		else
		{
			fnSaveFilter( oInput );
		}
	
		/* Tell the draw function we have been filtering */
		oSettings.bFiltered = true;
		_fnCallbackFire( oSettings, null, 'search', [oSettings] );
	}
	
	
	
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			$.merge( displayRows, rows );
		}
	}
	
	
	
	function _fnFilterColumn ( settings, searchStr, colIdx, regex, smart, caseInsensitive )
	{
		if ( searchStr === '' ) {
			return;
		}
	
		var data;
		var out = [];
		var display = settings.aiDisplay;
		var rpSearch = _fnFilterCreateSearch( searchStr, regex, smart, caseInsensitive );
	
		for ( var i=0 ; i<display.length ; i++ ) {
			data = settings.aoData[ display[i] ]._aFilterData[ colIdx ];
	
			if ( rpSearch.test( data ) ) {
				out.push( display[i] );
			}
		}
	
		settings.aiDisplay = out;
	}
	
	
	function _fnFilter( settings, input, force, regex, smart, caseInsensitive )
	{
		var rpSearch = _fnFilterCreateSearch( input, regex, smart, caseInsensitive );
		var prevSearch = settings.oPreviousSearch.sSearch;
		var displayMaster = settings.aiDisplayMaster;
		var display, invalidated, i;
		var filtered = [];
	
		// Need to take account of custom filtering functions - always filter
		if ( DataTable.ext.search.length !== 0 ) {
			force = true;
		}
	
		// Check if any of the rows were invalidated
		invalidated = _fnFilterData( settings );
	
		// If the input is blank - we just want the full data set
		if ( input.length <= 0 ) {
			settings.aiDisplay = displayMaster.slice();
		}
		else {
			// New search - start from the master array
			if ( invalidated ||
				 force ||
				 prevSearch.length > input.length ||
				 input.indexOf(prevSearch) !== 0 ||
				 settings.bSorted // On resort, the display master needs to be
				                  // re-filtered since indexes will have changed
			) {
				settings.aiDisplay = displayMaster.slice();
			}
	
			// Search the display array
			display = settings.aiDisplay;
	
			for ( i=0 ; i<display.length ; i++ ) {
				if ( rpSearch.test( settings.aoData[ display[i] ]._sFilterRow ) ) {
					filtered.push( display[i] );
				}
			}
	
			settings.aiDisplay = filtered;
		}
	}
	
	function _fnFilterCreateSearch( search, regex, smart, caseInsensitive )
	{
		search = regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. So this is what we want to
			 * generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var a = $.map( search.match( /"[^"]+"|[^ ]+/g ) || [''], function ( word ) {
				if ( word.charAt(0) === '"' ) {
					var m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
	
				return word.replace('"', '');
			} );
	
			search = '^(?=.*?'+a.join( ')(?=.*?' )+').*$';
		}
	
		return new RegExp( search, caseInsensitive ? 'i' : '' );
	}
	
	
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var column;
		var i, j, ien, jen, filterData, cellData, row;
		var fomatters = DataTable.ext.type.search;
		var wasInvalidated = false;
	
		for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, i, j, 'filter' );
	
						if ( fomatters[ column.sType ] ) {
							cellData = fomatters[ column.sType ]( cellData );
						}
	
						// Search in DataTables 1.10 is string based. In 1.11 this
						// should be altered to also allow strict type checking.
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	function _fnSearchToCamel ( obj )
	{
		return {
			search:          obj.sSearch,
			smart:           obj.bSmart,
			regex:           obj.bRegex,
			caseInsensitive: obj.bCaseInsensitive
		};
	}
	
	
	function _fnSearchToHung ( obj )
	{
		return {
			sSearch:          obj.search,
			bSmart:           obj.smart,
			bRegex:           obj.regex,
			bCaseInsensitive: obj.caseInsensitive
		};
	}
	
	function _fnFeatureHtmlInfo ( settings )
	{
		var
			tid = settings.sTableId,
			nodes = settings.aanFeatures.i,
			n = $('<div/>', {
				'class': settings.oClasses.sInfo,
				'id': ! nodes ? tid+'_info' : null
			} );
	
		if ( ! nodes ) {
			// Update display on each draw
			settings.aoDrawCallback.push( {
				"fn": _fnUpdateInfo,
				"sName": "information"
			} );
	
			n
				.attr( 'role', 'status' )
				.attr( 'aria-live', 'polite' );
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
		}
	
		return n[0];
	}
	
	function _fnUpdateInfo ( settings )
	{
		/* Show information about the table */
		var nodes = settings.aanFeatures.i;
		if ( nodes.length === 0 ) {
			return;
		}
	
		var
			lang  = settings.oLanguage,
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total ?
				lang.sInfo :
				lang.sInfoEmpty;
	
		if ( total !== max ) {
			/* Record set after filtering */
			out += ' ' + lang.sInfoFiltered;
		}
	
		// Convert the macros
		out += lang.sInfoPostFix;
		out = _fnInfoMacros( settings, out );
	
		var callback = lang.fnInfoCallback;
		if ( callback !== null ) {
			out = callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		$(nodes).html( out );
	}
	
	
	function _fnInfoMacros ( settings, str )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is used only
		// internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, settings.fnRecordsTotal() ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) );
	}
	
	
	function _fnInitialise ( settings )
	{
		var i, iLen, iAjaxStart=settings.iInitDisplayStart;
		var columns = settings.aoColumns, column;
		var features = settings.oFeatures;
		var deferLoading = settings.bDeferLoading; // value modified by the draw
	
		/* Ensure that the table data is fully initialised */
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		/* Show the display HTML options */
		_fnAddOptionsHtml( settings );
	
		/* Build and draw the header / footer for the table */
		_fnBuildHead( settings );
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		/* Okay to show that something is going on now */
		_fnProcessingDisplay( settings, true );
	
		/* Calculate sizes for columns */
		if ( features.bAutoWidth ) {
			_fnCalculateColumnWidths( settings );
		}
	
		for ( i=0, iLen=columns.length ; i<iLen ; i++ ) {
			column = columns[i];
	
			if ( column.sWidth ) {
				column.nTh.style.width = _fnStringToCss( column.sWidth );
			}
		}
	
		_fnCallbackFire( settings, null, 'preInit', [settings] );
	
		_fnReDraw( settings );
	
		// Server-side processing init complete is done by _fnAjaxUpdateDraw
		var dataSrc = _fnDataSource( settings );
		if ( dataSrc != 'ssp' || deferLoading ) {
			// if there is an ajax source load the data
			if ( dataSrc == 'ajax' ) {
				_fnBuildAjax( settings, [], function(json) {
					var aData = _fnAjaxDataSrc( settings, json );
	
					// Got the data - add it to the table
					for ( i=0 ; i<aData.length ; i++ ) {
						_fnAddData( settings, aData[i] );
					}
	
					settings.iInitDisplayStart = iAjaxStart;
	
					_fnReDraw( settings );
	
					_fnProcessingDisplay( settings, false );
					_fnInitComplete( settings, json );
				}, settings );
			}
			else {
				_fnProcessingDisplay( settings, false );
				_fnInitComplete( settings );
			}
		}
	}
	

	function _fnInitComplete ( settings, json )
	{
		settings._bInitComplete = true;
	
		// When data was added after the initialisation (data or Ajax) we need to
		// calculate the column sizing
		if ( json || settings.oInit.aaData ) {
			_fnAdjustColumnSizing( settings );
		}
	
		_fnCallbackFire( settings, null, 'plugin-init', [settings, json] );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', [settings, json] );
	}
	
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}

	function _fnFeatureHtmlLength ( settings )
	{
		var
			classes  = settings.oClasses,
			tableId  = settings.sTableId,
			menu     = settings.aLengthMenu,
			d2       = $.isArray( menu[0] ),
			lengths  = d2 ? menu[0] : menu,
			language = d2 ? menu[1] : menu;
	
		var select = $('<select/>', {
			'name':          tableId+'_length',
			'aria-controls': tableId,
			'class':         classes.sLengthSelect
		} );
	
		for ( var i=0, ien=lengths.length ; i<ien ; i++ ) {
			select[0][ i ] = new Option(
				typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i],
				lengths[i]
			);
		}
	
		var div = $('<div><label/></div>').addClass( classes.sLength );
		if ( ! settings.aanFeatures.l ) {
			div[0].id = tableId+'_length';
		}
	
		div.children().append(
			settings.oLanguage.sLengthMenu.replace( '_MENU_', select[0].outerHTML )
		);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function(e) {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
			}
		} );
	
		return div[0];
	}
	
	function _fnFeatureHtmlPaginate ( settings )
	{
		var
			type   = settings.sPaginationType,
			plugin = DataTable.ext.pager[ type ],
			modern = typeof plugin === 'function',
			redraw = function( settings ) {
				_fnDraw( settings );
			},
			node = $('<div/>').addClass( settings.oClasses.sPaging + type )[0],
			features = settings.aanFeatures;
	
		if ( ! modern ) {
			plugin.fnInit( settings, node, redraw );
		}
	
		/* Add a draw callback for the pagination on first instance, to update the paging display */
		if ( ! features.p )
		{
			node.id = settings.sTableId+'_paginate';
	
			settings.aoDrawCallback.push( {
				"fn": function( settings ) {
					if ( modern ) {
						var
							start      = settings._iDisplayStart,
							len        = settings._iDisplayLength,
							visRecords = settings.fnRecordsDisplay(),
							all        = len === -1,
							page = all ? 0 : Math.ceil( start / len ),
							pages = all ? 1 : Math.ceil( visRecords / len ),
							buttons = plugin(page, pages),
							i, ien;
	
						for ( i=0, ien=features.p.length ; i<ien ; i++ ) {
							_fnRenderer( settings, 'pageButton' )(
								settings, features.p[i], i, buttons, page, pages
							);
						}
					}
					else {
						plugin.fnUpdate( settings, redraw );
					}
				},
				"sName": "pagination"
			} );
		}
	
		return node;
	}
	
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
			  start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		if ( changed ) {
			_fnCallbackFire( settings, null, 'page', [settings] );
	
			if ( redraw ) {
				_fnDraw( settings );
			}
		}
	
		return changed;
	}
	
	
	function _fnFeatureHtmlProcessing ( settings )
	{
		return $('<div/>', {
				'id': ! settings.aanFeatures.r ? settings.sTableId+'_processing' : null,
				'class': settings.oClasses.sProcessing
			} )
			.html( settings.oLanguage.sProcessing )
			.insertBefore( settings.nTable )[0];
	}
	
	
	function _fnProcessingDisplay ( settings, show )
	{
		if ( settings.oFeatures.bProcessing ) {
			$(settings.aanFeatures.r).css( 'display', show ? 'block' : 'none' );
		}
	
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Add the ARIA grid role to the table
		table.attr( 'role', 'grid' );
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses;
		var caption = table.children('caption');
		var captionSide = caption.length ? caption[0]._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		var scroller = $( _div, { 'class': classes.sScrollWrapper } )
			.append(
				$(_div, { 'class': classes.sScrollHead } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollHeadInner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.sScrollBody } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.sScrollFoot } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollFootInner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		if ( scrollX ) {
			$(scrollBody).on( 'scroll.DT', function (e) {
				var scrollLeft = this.scrollLeft;
	
				scrollHead.scrollLeft = scrollLeft;
	
				if ( footer ) {
					scrollFoot.scrollLeft = scrollLeft;
				}
			} );
		}
	
		$(scrollBody).css(
			scrollY && scroll.bCollapse ? 'max-height' : 'height', 
			scrollY
		);
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push( {
			"fn": _fnScrollDraw,
			"sName": "scrolling"
		} );
	
		return scroller[0];
	}
	
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			scrollX        = scroll.sX,
			scrollXInner   = scroll.sXInner,
			scrollY        = scroll.sY,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderStyle = divHeader[0].style,
			divHeaderInner = divHeader.children('div'),
			divHeaderInnerStyle = divHeaderInner[0].style,
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divBodyStyle   = divBodyEl.style,
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			tableEl        = table[0],
			tableStyle     = tableEl.style,
			footer         = settings.nTFoot ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			ie67           = browser.bScrollOversize,
			dtHeaderCells  = _pluck( settings.aoColumns, 'nTh' ),
			headerTrgEls, footerTrgEls,
			headerSrcEls, footerSrcEls,
			headerCopy, footerCopy,
			headerWidths=[], footerWidths=[],
			headerContent=[], footerContent=[],
			idx, correction, sanityWidth,
			zeroOut = function(nSizer) {
				var style = nSizer.style;
				style.paddingTop = "0";
				style.paddingBottom = "0";
				style.borderTopWidth = "0";
				style.borderBottomWidth = "0";
				style.height = 0;
			};
	
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		table.children('thead, tfoot').remove();
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerTrgEls = footer.find('tr'); // the original tfoot is in its own table and must be sized
			footerSrcEls = footerCopy.find('tr');
		}
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerTrgEls = header.find('tr'); // original header is in its own table
		headerSrcEls = headerCopy.find('tr');
		headerCopy.find('th, td').removeAttr('tabindex');
	
	
		if ( ! scrollX )
		{
			divBodyStyle.width = '100%';
			divHeader[0].style.width = '100%';
		}
	
		$.each( _fnGetUniqueThs( settings, headerCopy ), function ( i, el ) {
			idx = _fnVisibleToColumnIndex( settings, i );
			el.style.width = settings.aoColumns[idx].sWidth;
		} );
	
		if ( footer ) {
			_fnApplyToChildren( function(n) {
				n.style.width = "";
			}, footerSrcEls );
		}
	
		// Size the table as a whole
		sanityWidth = table.outerWidth();
		if ( scrollX === "" ) {
			// No x scrolling
			tableStyle.width = "100%";
	
			if ( ie67 && (table.find('tbody').height() > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( table.outerWidth() - barWidth);
			}

			sanityWidth = table.outerWidth();
		}
		else if ( scrollXInner !== "" ) {
			// legacy x scroll inner has been given - use it
			tableStyle.width = _fnStringToCss(scrollXInner);
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}

		_fnApplyToChildren( zeroOut, headerSrcEls );
	
		// Read all widths in next pass
		_fnApplyToChildren( function(nSizer) {
			headerContent.push( nSizer.innerHTML );
			headerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
		}, headerSrcEls );
	
		// Apply all widths in final pass
		_fnApplyToChildren( function(nToSize, i) {
			// Only apply widths to the DataTables detected header cells - this
			// prevents complex headers from having contradictory sizes applied
			if ( $.inArray( nToSize, dtHeaderCells ) !== -1 ) {
				nToSize.style.width = headerWidths[i];
			}
		}, headerTrgEls );
	
		$(headerSrcEls).height(0);
	
		/* Same again with the footer if we have one */
		if ( footer )
		{
			_fnApplyToChildren( zeroOut, footerSrcEls );
	
			_fnApplyToChildren( function(nSizer) {
				footerContent.push( nSizer.innerHTML );
				footerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
			}, footerSrcEls );
	
			_fnApplyToChildren( function(nToSize, i) {
				nToSize.style.width = footerWidths[i];
			}, footerTrgEls );
	
			$(footerSrcEls).height(0);
		}
	
		_fnApplyToChildren( function(nSizer, i) {
			nSizer.innerHTML = '<div class="dataTables_sizing">'+headerContent[i]+'</div>';
			nSizer.childNodes[0].style.height = "0";
			nSizer.childNodes[0].style.overflow = "hidden";
			nSizer.style.width = headerWidths[i];
		}, headerSrcEls );
	
		if ( footer )
		{
			_fnApplyToChildren( function(nSizer, i) {
				nSizer.innerHTML = '<div class="dataTables_sizing">'+footerContent[i]+'</div>';
				nSizer.childNodes[0].style.height = "0";
				nSizer.childNodes[0].style.overflow = "hidden";
				nSizer.style.width = footerWidths[i];
			}, footerSrcEls );
		}
	
		// Sanity check that the table is of a sensible width. If not then we are going to get
		// misalignment - try to prevent this by not allowing the table to shrink below its min width
		if ( table.outerWidth() < sanityWidth )
		{
			// The min width depends upon if we have a vertical scrollbar visible or not */
			correction = ((divBodyEl.scrollHeight > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")) ?
					sanityWidth+barWidth :
					sanityWidth;
	
			// IE6/7 are a law unto themselves...
			if ( ie67 && (divBodyEl.scrollHeight >
				divBodyEl.offsetHeight || divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( correction-barWidth );
			}
	
			// And give the user a warning that we've stopped the table getting too small
			if ( scrollX === "" || scrollXInner !== "" ) {
				_fnLog( settings, 1, 'Possible column misalignment', 6 );
			}
		}
		else
		{
			correction = '100%';
		}
	
		// Apply to the container elements
		divBodyStyle.width = _fnStringToCss( correction );
		divHeaderStyle.width = _fnStringToCss( correction );
	
		if ( footer ) {
			settings.nScrollFoot.style.width = _fnStringToCss( correction );
		}

		if ( ! scrollY ) {

			if ( ie67 ) {
				divBodyStyle.height = _fnStringToCss( tableEl.offsetHeight+barWidth );
			}
		}
	
		/* Finally set the width's of the header and footer tables */
		var iOuterWidth = table.outerWidth();
		divHeaderTable[0].style.width = _fnStringToCss( iOuterWidth );
		divHeaderInnerStyle.width = _fnStringToCss( iOuterWidth );
	
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var bScrolling = table.height() > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var padding = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
		divHeaderInnerStyle[ padding ] = bScrolling ? barWidth+"px" : "0px";
	
		if ( footer ) {
			divFooterTable[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style[padding] = bScrolling ? barWidth+"px" : "0px";
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').insertBefore( table.children('thead') );
	
		/* Adjust the position of the header in case we loose the y-scrollbar */
		divBody.scroll();

		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	
	function _fnApplyToChildren( fn, an1, an2 )
	{
		var index=0, i=0, iLen=an1.length;
		var nNode1, nNode2;
	
		while ( i < iLen ) {
			nNode1 = an1[i].firstChild;
			nNode2 = an2 ? an2[i].firstChild : null;
	
			while ( nNode1 ) {
				if ( nNode1.nodeType === 1 ) {
					if ( an2 ) {
						fn( nNode1, nNode2, index );
					}
					else {
						fn( nNode1, index );
					}
	
					index++;
				}
	
				nNode1 = nNode1.nextSibling;
				nNode2 = an2 ? nNode2.nextSibling : null;
			}
	
			i++;
		}
	}
	
	
	
	var __re_html_remove = /<.*?>/g;

	function _fnCalculateColumnWidths ( oSettings )
	{
		var
			table = oSettings.nTable,
			columns = oSettings.aoColumns,
			scroll = oSettings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			columnCount = columns.length,
			visibleColumns = _fnGetColumns( oSettings, 'bVisible' ),
			headerCells = $('th', oSettings.nTHead),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			userInputs = false,
			i, column, columnIdx, width, outerWidth,
			browser = oSettings.oBrowser,
			ie67 = browser.bScrollOversize;
	
		var styleWidth = table.style.width;
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		/* Convert any user input sizes into pixel sizes */
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			column = columns[ visibleColumns[i] ];
	
			if ( column.sWidth !== null ) {
				column.sWidth = _fnConvertToWidth( column.sWidthOrig, tableContainer );
	
				userInputs = true;
			}
		}
	
		if ( ie67 || ! userInputs && ! scrollX && ! scrollY &&
		     columnCount == _fnVisbleColumns( oSettings ) &&
		     columnCount == headerCells.length
		) {
			for ( i=0 ; i<columnCount ; i++ ) {
				var colIdx = _fnVisibleToColumnIndex( oSettings, i );
	
				if ( colIdx !== null ) {
					columns[ colIdx ].sWidth = _fnStringToCss( headerCells.eq(i).width() );
				}
			}
		}
		else
		{

			var tmpTable = $(table).clone() // don't use cloneNode - IE8 will remove events on the main table
				.css( 'visibility', 'hidden' )
				.removeAttr( 'id' );
	
			// Clean up the table body
			tmpTable.find('tbody tr').remove();
			var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );

			tmpTable.find('thead, tfoot').remove();
			tmpTable
				.append( $(oSettings.nTHead).clone() )
				.append( $(oSettings.nTFoot).clone() );
	
			// Remove any assigned widths from the footer (from scrolling)
			tmpTable.find('tfoot th, tfoot td').css('width', '');
	
			// Apply custom sizing to the cloned header
			headerCells = _fnGetUniqueThs( oSettings, tmpTable.find('thead')[0] );
	
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				column = columns[ visibleColumns[i] ];
	
				headerCells[i].style.width = column.sWidthOrig !== null && column.sWidthOrig !== '' ?
					_fnStringToCss( column.sWidthOrig ) :
					'';

				if ( column.sWidthOrig && scrollX ) {
					$( headerCells[i] ).append( $('<div/>').css( {
						width: column.sWidthOrig,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
	
			// Find the widest cell for each column and put it into the table
			if ( oSettings.aoData.length ) {
				for ( i=0 ; i<visibleColumns.length ; i++ ) {
					columnIdx = visibleColumns[i];
					column = columns[ columnIdx ];
	
					$( _fnGetWidestNode( oSettings, columnIdx ) )
						.clone( false )
						.append( column.sContentPadding )
						.appendTo( tr );
				}
			}
	
			// Tidy the temporary table - remove name attributes so there aren't
			// duplicated in the dom (radio elements for example)
			$('[name]', tmpTable).removeAttr('name');

			var holder = $('<div/>').css( scrollX || scrollY ?
					{
						position: 'absolute',
						top: 0,
						left: 0,
						height: 1,
						right: 0,
						overflow: 'hidden'
					} :
					{}
				)
				.append( tmpTable )
				.appendTo( tableContainer );
	
			if ( scrollX && scrollXInner ) {
				tmpTable.width( scrollXInner );
			}
			else if ( scrollX ) {
				tmpTable.css( 'width', 'auto' );
				tmpTable.removeAttr('width');
	
				// If there is no width attribute or style, then allow the table to
				// collapse
				if ( tmpTable.width() < tableContainer.clientWidth && tableWidthAttr ) {
					tmpTable.width( tableContainer.clientWidth );
				}
			}
			else if ( scrollY ) {
				tmpTable.width( tableContainer.clientWidth );
			}
			else if ( tableWidthAttr ) {
				tmpTable.width( tableWidthAttr );
			}

			var total = 0;
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				var cell = $(headerCells[i]);
				var border = cell.outerWidth() - cell.width();
	
				// Use getBounding... where possible (not IE8-) because it can give
				// sub-pixel accuracy, which we then want to round up!
				var bounding = browser.bBounding ?
					Math.ceil( headerCells[i].getBoundingClientRect().width ) :
					cell.outerWidth();
	
				// Total is tracked to remove any sub-pixel errors as the outerWidth
				// of the table might not equal the total given here (IE!).
				total += bounding;
	
				// Width for each column to use
				columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding - border );
			}
	
			table.style.width = _fnStringToCss( total );
	
			// Finished with the table - ditch it
			holder.remove();
		}
	
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! oSettings._reszEvt ) {
			var bindResize = function () {
				$(window).on('resize.DT-'+oSettings.sInstance, _fnThrottle( function () {
					_fnAdjustColumnSizing( oSettings );
				} ) );
			};
	
			// IE6/7 will crash if we bind a resize event handler on page load.
			// To be removed in 1.11 which drops IE6/7 support
			if ( ie67 ) {
				setTimeout( bindResize, 1000 );
			}
			else {
				bindResize();
			}
	
			oSettings._reszEvt = true;
		}
	}

	var _fnThrottle = DataTable.util.throttle;

	function _fnConvertToWidth ( width, parent )
	{
		if ( ! width ) {
			return 0;
		}
	
		var n = $('<div/>')
			.css( 'width', _fnStringToCss( width ) )
			.appendTo( parent || document.body );
	
		var val = n[0].offsetWidth;
		n.remove();
	
		return val;
	}
	
	function _fnGetWidestNode( settings, colIdx )
	{
		var idx = _fnGetMaxLenString( settings, colIdx );
		if ( idx < 0 ) {
			return null;
		}
	
		var data = settings.aoData[ idx ];
		return ! data.nTr ? // Might not have been created when deferred rendering
			$('<td/>').html( _fnGetCellData( settings, idx, colIdx, 'display' ) )[0] :
			data.anCells[ colIdx ];
	}

	function _fnGetMaxLenString( settings, colIdx )
	{
		var s, max=-1, maxIdx = -1;
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			s = _fnGetCellData( settings, i, colIdx, 'display' )+'';
			s = s.replace( __re_html_remove, '' );
			s = s.replace( /&nbsp;/g, ' ' );
	
			if ( s.length > max ) {
				max = s.length;
				maxIdx = i;
			}
		}
	
		return maxIdx;
	}
	
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, iLen, k, kLen,
			aSort = [],
			aiOrig = [],
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [],
			add = function ( a ) {
				if ( a.length && ! $.isArray( a[0] ) ) {
					// 1D array
					nestedSort.push( a );
				}
				else {
					// 2D array
					$.merge( nestedSort, a );
				}
			};
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( $.isArray( fixed ) ) {
			add( fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			add( fixed.pre );
		}
	
		add( settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			add( fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
			aDataSort = aoColumns[ srcCol ].aDataSort;
	
			for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
			{
				iCol = aDataSort[k];
				sType = aoColumns[ iCol ].sType || 'string';
	
				if ( nestedSort[i]._idx === undefined ) {
					nestedSort[i]._idx = $.inArray( nestedSort[i][1], aoColumns[iCol].asSorting );
				}
	
				aSort.push( {
					src:       srcCol,
					col:       iCol,
					dir:       nestedSort[i][1],
					index:     nestedSort[i]._idx,
					type:      sType,
					formatter: DataTable.ext.type.order[ sType+"-pre" ]
				} );
			}
		}
	
		return aSort;
	}
	
	function _fnSort ( oSettings )
	{
		var
			i, ien, iLen, j, jLen, k, kLen,
			sDataType, nTh,
			aiOrig = [],
			oExtSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			aoColumns = oSettings.aoColumns,
			aDataSort, data, iCol, sType, oSort,
			formatters = 0,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;

		_fnColumnTypes( oSettings );
	
		aSort = _fnSortFlatten( oSettings );
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Track if we can use the fast sort algorithm
			if ( sortCol.formatter ) {
				formatters++;
			}
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Create a value - key array of the current row positions such that we can use their
			// current position during the sort, if values match, in order to perform stable sorting
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ displayMaster[i] ] = i;
			}

			if ( formatters === aSort.length ) {
				// All sort types have formatting functions
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, test, sort,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						test = x<y ? -1 : x>y ? 1 : 0;
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
			else {

				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, l, test, sort, fn,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						fn = oExtSort[ sort.type+"-"+sort.dir ] || oExtSort[ "string-"+sort.dir ];
						test = fn( x, y );
						if ( test !== 0 ) {
							return test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
		}
	
		/* Tell the draw function that we have sorted the data */
		oSettings.bSorted = true;
	}
	
	
	function _fnSortAria ( settings )
	{
		var label;
		var nextSort;
		var columns = settings.aoColumns;
		var aSort = _fnSortFlatten( settings );
		var oAria = settings.oLanguage.oAria;
	
		// ARIA attributes - need to loop all columns, to update all (removing old
		// attributes as needed)
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			var col = columns[i];
			var asSorting = col.asSorting;
			var sTitle = col.sTitle.replace( /<.*?>/g, "" );
			var th = col.nTh;
	
			th.removeAttribute('aria-sort');
	
			/* In ARIA only the first sorting column can be marked as sorting - no multi-sort option */
			if ( col.bSortable ) {
				if ( aSort.length > 0 && aSort[0].col == i ) {
					th.setAttribute('aria-sort', aSort[0].dir=="asc" ? "ascending" : "descending" );
					nextSort = asSorting[ aSort[0].index+1 ] || asSorting[0];
				}
				else {
					nextSort = asSorting[0];
				}
	
				label = sTitle + ( nextSort === "asc" ?
					oAria.sSortAscending :
					oAria.sSortDescending
				);
			}
			else {
				label = sTitle;
			}
	
			th.setAttribute('aria-label', label);
		}
	}

	function _fnSortListener ( settings, colIdx, append, callback )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = $.inArray( a[1], asSorting );
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( append && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = $.inArray( colIdx, _pluck(sorting, '0') );
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else {
				// No sort on this column yet
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	
		// Run the sort by calling a full redraw
		_fnReDraw( settings );
	
		// callback used for async user interaction
		if ( typeof callback == 'function' ) {
			callback( settings );
		}
	}
	
	function _fnSortAttachListener ( settings, attachTo, colIdx, callback )
	{
		var col = settings.aoColumns[ colIdx ];
	
		_fnBindAction( attachTo, {}, function (e) {
			/* If the column is not sortable - don't to anything */
			if ( col.bSortable === false ) {
				return;
			}
	
			// If processing is enabled use a timeout to allow the processing
			// display to be shown - otherwise to it synchronously
			if ( settings.oFeatures.bProcessing ) {
				_fnProcessingDisplay( settings, true );
	
				setTimeout( function() {
					_fnSortListener( settings, colIdx, e.shiftKey, callback );
	
					// In server-side processing, the draw callback will remove the
					// processing display
					if ( _fnDataSource( settings ) !== 'ssp' ) {
						_fnProcessingDisplay( settings, false );
					}
				}, 0 );
			}
			else {
				_fnSortListener( settings, colIdx, e.shiftKey, callback );
			}
		} );
	}
	
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.sSortColumn;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	function _fnSortData( settings, idx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ idx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, idx,
				_fnColumnIndexToVisible( settings, idx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[idx] || customSort ) {
				cellData = customSort ?
					customData[i] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, i, idx, 'sort' );
	
				row._aSortData[ idx ] = formatter ?
					formatter( cellData ) :
					cellData;
			}
		}
	}

	function _fnSaveState ( settings )
	{
		if ( !settings.oFeatures.bStateSave || settings.bDestroying )
		{
			return;
		}
	
		/* Store the interesting variables */
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   $.extend( true, [], settings.aaSorting ),
			search:  _fnSearchToCamel( settings.oPreviousSearch ),
			columns: $.map( settings.aoColumns, function ( col, i ) {
				return {
					visible: col.bVisible,
					search: _fnSearchToCamel( settings.aoPreSearchCols[i] )
				};
			} )
		};
	
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
	
		settings.oSavedState = state;
		settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
	}

	function _fnLoadState ( settings, oInit, callback )
	{
		var i, ien;
		var columns = settings.aoColumns;
		var loaded = function ( s ) {
			if ( ! s || ! s.time ) {
				callback();
				return;
			}
	
			// Allow custom and plug-in manipulation functions to alter the saved data set and
			// cancelling of loading by returning false
			var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
			if ( $.inArray( false, abStateLoad ) !== -1 ) {
				callback();
				return;
			}
	
			// Reject old data
			var duration = settings.iStateDuration;
			if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
				callback();
				return;
			}
	
			// Number of columns have changed - all bets are off, no restore of settings
			if ( s.columns && columns.length !== s.columns.length ) {
				callback();
				return;
			}
	
			// Store the saved state so it might be accessed at any time
			settings.oLoadedState = $.extend( true, {}, s );
	
			// Restore key features - todo - for 1.11 this needs to be done by
			// subscribed events
			if ( s.start !== undefined ) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			if ( s.length !== undefined ) {
				settings._iDisplayLength   = s.length;
			}
	
			// Order
			if ( s.order !== undefined ) {
				settings.aaSorting = [];
				$.each( s.order, function ( i, col ) {
					settings.aaSorting.push( col[0] >= columns.length ?
						[ 0, col[1] ] :
						col
					);
				} );
			}
	
			// Search
			if ( s.search !== undefined ) {
				$.extend( settings.oPreviousSearch, _fnSearchToHung( s.search ) );
			}
	
			// Columns
			//
			if ( s.columns ) {
				for ( i=0, ien=s.columns.length ; i<ien ; i++ ) {
					var col = s.columns[i];
	
					// Visibility
					if ( col.visible !== undefined ) {
						columns[i].bVisible = col.visible;
					}
	
					// Search
					if ( col.search !== undefined ) {
						$.extend( settings.aoPreSearchCols[i], _fnSearchToHung( col.search ) );
					}
				}
			}
	
			_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
			callback();
		}
	
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			loaded( state );
		}
		// otherwise, wait for the loaded callback to be executed
	}

	function _fnSettingsFromNode ( table )
	{
		var settings = DataTable.settings;
		var idx = $.inArray( table, _pluck( settings, 'nTable' ) );
	
		return idx !== -1 ?
			settings[ idx ] :
			null;
	}
	
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'http://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'error', [ settings, tn, msg ] );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	function _fnMap( ret, src, name, mappedName )
	{
		if ( $.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( $.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}

	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( extender.hasOwnProperty(prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && $.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	function _fnBindAction( n, oData, fn )
	{
		$(n)
			.on( 'click.DT', oData, function (e) {
					$(n).blur(); // Remove focus outline for mouse users
					fn(e);
				} )
			.on( 'keypress.DT', oData, function (e){
					if ( e.which === 13 ) {
						e.preventDefault();
						fn(e);
					}
				} )
			.on( 'selectstart.DT', function () {
					/* Take the brutal approach to cancelling text selection */
					return false;
				} );
	}
	

	function _fnCallbackReg( oSettings, sStore, fn, sName )
	{
		if ( fn )
		{
			oSettings[sStore].push( {
				"fn": fn,
				"sName": sName
			} );
		}
	}
	

	function _fnCallbackFire( settings, callbackArr, eventName, args )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = $.map( settings[callbackArr].slice().reverse(), function (val, i) {
				return val.fn.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null ) {
			var e = $.Event( eventName+'.dt' );
	
			$(settings.nTable).trigger( e, args );
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}

	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax || settings.sAjaxSource ) {
			return 'ajax';
		}
		return 'dom';
	}
	
	var __apiStruct = [];

	var __arrayProto = Array.prototype;

	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = $.map( settings, function (el, i) {
			return el.nTable;
		} );
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oApi ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = $.inArray( mixed, tables );
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed);
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed;
		}
	
		if ( jq ) {
			return jq.map( function(i) {
				idx = $.inArray( this, tables );
				return idx !== -1 ? settings[idx] : null;
			} ).toArray();
		}
	};
	
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings = settings.concat( a );
			}
		};
	
		if ( $.isArray( context ) ) {
			for ( var i=0, ien=context.length ; i<ien ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = _unique( settings );
	
		// Initial data
		if ( data ) {
			$.merge( this, data );
		}
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
	
		concat:  __arrayProto.concat,
	
	
		context: [], // array of table settings objects
	
	
		count: function ()
		{
			return this.flatten().length;
		},
	
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
	
		filter: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.filter ) {
				a = __arrayProto.filter.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					if ( fn.call( this, this[i], i, this ) ) {
						a.push( this[i] );
					}
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		flatten: function ()
		{
			var a = [];
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
	
		join:    __arrayProto.join,
	
	
		indexOf: __arrayProto.indexOf || function (obj, start)
		{
			for ( var i=(start || 0), ien=this.length ; i<ien ; i++ ) {
				if ( this[i] === obj ) {
					return i;
				}
			}
			return -1;
		},
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
	
		lastIndexOf: __arrayProto.lastIndexOf || function (obj, start)
		{
			// Bit cheeky...
			return this.indexOf.apply( this.toArray.reverse(), arguments );
		},
	
	
		length:  0,
	
	
		map: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.map ) {
				a = __arrayProto.map.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					a.push( fn.call( this, this[i], i ) );
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		pluck: function ( prop )
		{
			return this.map( function ( el ) {
				return el[ prop ];
			} );
		},
	
		pop:     __arrayProto.pop,
	
	
		push:    __arrayProto.push,
	
	
		// Does not return an API instance
		reduce: __arrayProto.reduce || function ( fn, init )
		{
			return _fnReduce( this, fn, init, 0, this.length, 1 );
		},
	
	
		reduceRight: __arrayProto.reduceRight || function ( fn, init )
		{
			return _fnReduce( this, fn, init, this.length-1, -1, -1 );
		},
	
	
		reverse: __arrayProto.reverse,
	
	
		// Object with rows, columns and opts
		selector: null,
	
	
		shift:   __arrayProto.shift,
	
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
	
		sort:    __arrayProto.sort, // ? name - order?
	
	
		splice:  __arrayProto.splice,
	
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
	
		to$: function ()
		{
			return $( this );
		},
	
	
		toJQuery: function ()
		{
			return $( this );
		},
	
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this) );
		},
	
	
		unshift: __arrayProto.unshift
	} );
	
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			j, jen,
			struct, inner,
			methodScoping = function ( scope, fn, struc ) {
				return function () {
					var ret = fn.apply( scope, arguments );
	
					// Method extension
					_Api.extend( ret, ret, struc.methodExt );
					return ret;
				};
			};
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			// Value
			obj[ struct.name ] = typeof struct.val === 'function' ?
				methodScoping( scope, struct.val, struct ) :
				$.isPlainObject( struct.val ) ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( $.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		var find = function ( src, name ) {
			for ( var i=0, ien=src.length ; i<ien ; i++ ) {
				if ( src[i].name === name ) {
					return src[i];
				}
			}
			return null;
		};
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   []
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					$.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	var __table_selector = function ( selector, a )
	{
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = $.map( a, function (el, i) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function (i) {
				// Need to translate back from the table node to the settings
				var idx = $.inArray( this, nodes );
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	
	_api_registerPlural( 'tables().nodes()', 'table().node()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTable;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().body()', 'table().body()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTBody;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().header()', 'table().header()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTHead;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().footer()', 'table().footer()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTFoot;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	_api_register( 'page.info()', function ( action ) {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, [], function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return ctx.ajax ?
				$.isPlainObject( ctx.ajax ) ?
					ctx.ajax.url :
					ctx.ajax :
				ctx.sAjaxSource;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
			// No need to consider sAjaxSource here since DataTables gives priority
			// to `ajax` over `sAjaxSource`. So setting `ajax` here, renders any
			// value of `sAjaxSource` redundant.
		} );
	} );
	
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			a, i, ien, j, jen,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			// Only split on simple strings - complex expressions will be jQuery selectors
			a = selector[i] && selector[i].split && ! selector[i].match(/[\[\(:]/) ?
				selector[i].split(',') :
				[ selector[i] ];
	
			for ( j=0, jen=a.length ; j<jen ; j++ ) {
				res = selectFn( typeof a[j] === 'string' ? $.trim(a[j]) : a[j] );
	
				if ( res && res.length ) {
					out = out.concat( res );
				}
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	var _selector_first = function ( inst )
	{
		// Reduce the API instance to the first item found
		for ( var i=0, ien=inst.length ; i<ien ; i++ ) {
			if ( inst[i].length > 0 ) {
				// Assign the first element to the first item in the instance
				// and truncate the instance and context
				inst[0] = inst[i];
				inst[0].length = 1;
				inst.length = 1;
				inst.context = [ inst.context[i] ];
	
				return inst;
			}
		}
	
		// Not found - return an empty instance
		inst.length = 0;
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
		else if ( page == 'current' ) {
			// Current page implies that order=current and fitler=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( var i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				a = $.map( displayMaster, function (el) {
					return ! displayFilteredMap.hasOwnProperty(el) ?
						el :
						null;
				} );
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = $.inArray( i, displayFiltered );
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
	
		return a;
	};
	
	
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var i, ien;
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && $.inArray( selInt, rows ) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return $.map( rows, function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);

			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		return _selector_run( 'row', selector, run, settings, opts );
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		var that = this;
	
		this.iterator( 'row', function ( settings, row, thatIdx ) {
			var data = settings.aoData;
			var rowData = data[ row ];
			var i, ien, j, jen;
			var loopRow, loopCells;
	
			data.splice( row, 1 );
	
			// Update the cached indexes
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				loopRow = data[i];
				loopCells = loopRow.anCells;
	
				// Rows
				if ( loopRow.nTr !== null ) {
					loopRow.nTr._DT_RowIndex = i;
				}
	
				// Cells
				if ( loopCells !== null ) {
					for ( j=0, jen=loopCells.length ; j<jen ; j++ ) {
						loopCells[j]._DT_CellIndex.row = i;
					}
				}
			}
	
			// Delete from the display arrays
			_fnDeleteIndex( settings.aiDisplayMaster, row );
			_fnDeleteIndex( settings.aiDisplay, row );
			_fnDeleteIndex( that[ thatIdx ], row, false ); // maintain local indexes
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
		} );
	
		this.iterator( 'table', function ( settings ) {
			for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				settings.aoData[i].idx = i;
			}
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		$.merge( modRows, newRows );
	
		return modRows;
	} );
	
	
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( $.isArray( data ) && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		return ctx.length && this.length ?
			ctx[0].aoData[ this[0] ].nTr || null :
			null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( $.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td/></tr>').addClass( k );
				$('td', created)
					.addClass( k )
					.html( r )
					[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
				}
				else {
					row._details.detach();
				}
	
				__details_events( ctx[0] );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-visibility'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx, idx, vis ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row._details ) {
						row._details.children('td[colspan]').attr('colspan', visible );
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._details :
				undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function ( show ) {   // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	var __re_column_selector = /^([^:]+):(name|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column ) );
		}
		return a;
	};
	
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names = _pluck( columns, 'sName' ),
			nodes = _pluck( columns, 'nTh' );
	
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return $.map( columns, function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							nodes[ idx ]
						) ? idx : null;
				} );
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						var idx = parseInt( match[1], 10 );
						// Visible index given, convert to column index
						if ( idx < 0 ) {
							// Counting from the right
							var visColumns = $.map( columns, function (col,i) {
								return col.bVisible ? i : null;
							} );
							return [ visColumns[ visColumns.length + idx ] ];
						}
						// Counting from the left
						return [ _fnVisibleToColumnIndex( settings, idx ) ];
	
					case 'name':
						// match by name. `names` is column index complete and in order
						return $.map( names, function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return $.inArray( this, nodes ); // `nodes` is column index complete and in order
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		return _selector_run( 'column', selector, run, settings, opts );
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			row, cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = $.inArray( true, _pluck(cols, 'bVisible'), column+1 );
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				tr = data[i].nTr;
				cells = data[i].anCells;
	
				if ( tr ) {
					// insertBefore can act like appendChild if 2nd arg is null
					tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		// Update colspan for no records display. Child rows and extensions will use their own
		// listeners to do this - only need to update the empty table item here
		if ( ! settings.aiDisplay.length ) {
			$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
		}
	
		_fnSaveState( settings );
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTh;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTf;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			__setColumnVis( settings, column, vis );
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			// Second loop once the first is done for events
			this.iterator( 'column', function ( settings, column ) {
				_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
			} );
	
			if ( calc === undefined || calc ) {
				this.columns.adjust();
			}
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $( [].concat.apply([], cells) );
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && $.inArray( s.row, rows ) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
	 				};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// Row + column selector
		var columns = this.columns( columnSelector );
		var rows = this.rows( rowSelector );
		var a, i, ien, j, jen;
	
		this.iterator( 'table', function ( settings, idx ) {
			a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
		}, 1 );
	
	    // Now pass through the cell selector for options
	    var cells = this.cells( a, opts );
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( order.length && ! $.isArray( order[0] ) ) {
			// Arguments passed in (list of 1D arrays)
			order = Array.prototype.slice.call( arguments );
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSorting = order.slice();
		} );
	} );
	
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener( settings, node, column, callback );
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return $.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		return this.iterator( 'table', function ( settings, i ) {
			var sort = [];
	
			$.each( that[i], function (j, col) {
				sort.push( [ col, dir ] );
			} );
	
			settings.aaSorting = sort;
		} );
	} );
	
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.sSearch :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			_fnFilterComplete( settings, $.extend( {}, settings.oPreviousSearch, {
				"sSearch": input+"",
				"bRegex":  regex === null ? false : regex,
				"bSmart":  smart === null ? true  : smart,
				"bCaseInsensitive": caseInsen === null ? true : caseInsen
			} ), 1 );
		} );
	} );
	
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].sSearch;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				$.extend( preSearch[ column ], {
					"sSearch": input+"",
					"bRegex":  regex === null ? false : regex,
					"bSmart":  smart === null ? true  : smart,
					"bCaseInsensitive": caseInsen === null ? true : caseInsen
				} );
	
				_fnFilterComplete( settings, settings.oPreviousSearch, 1 );
			} );
		}
	);
	
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function () {
		return this.context.length ?
			this.context[0].oSavedState :
			null;
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	DataTable.versionCheck = DataTable.fnVersionCheck = function( version )
	{
		var aThis = DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	DataTable.isDataTable = DataTable.fnIsDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	DataTable.tables = DataTable.fnTables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = $.map( DataTable.settings, function (o) {
			if ( !visible || (visible && $(o.nTable).is(':visible')) ) {
				return o.nTable;
			}
		} );
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = $.map( args[0].split( /\s/ ), function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var orig      = settings.nTableWrapper.parentNode;
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = $.map( settings.aoData, function (r) { return r.nTr; } );
			var i, ien;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings] );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$( rows ).removeClass( settings.asStripeClasses.join(' ') );
	
			$('th, td', thead).removeClass( classes.sSortable+' '+
				classes.sSortableAsc+' '+classes.sSortableDesc+' '+classes.sSortableNone
			);
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, settings.nTableReinsertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.sTable );
	
				// If the were originally stripe classes - then we add them back here.
				// Note this is not fool proof (for example if not all rows had stripe
				// classes - but it's a good effort without getting carried away
				ien = settings.asDestroyStripes.length;
	
				if ( ien ) {
					jqTbody.children().each( function (i) {
						$(this).addClass( settings.asDestroyStripes[i % ien] );
					} );
				}
			}
	
			/* Remove the settings object from the settings array */
			var idx = $.inArray( settings, DataTable.settings );
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
	
			return this.iterator( type, function ( settings, arg1, arg2, arg3, arg4 ) {
				
				fn.call(
					api[ type ](
						arg1,
						type==='cell' ? arg2 : opts,
						type==='cell' ? opts : undefined
					),
					arg1, arg2, arg3, arg4
				);
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( plural !== undefined && $.isPlainObject( resolved ) ) {
			resolved = resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return resolved.replace( '%d', plural ); // nb: plural might be undefined,
	} );

	DataTable.version = "1.10.19";

	DataTable.settings = [];

	DataTable.models = {};

	DataTable.models.oSearch = {

		"bCaseInsensitive": true,

		"sSearch": "",

		"bRegex": false,

		"bSmart": true
	};
	
	
	DataTable.models.oRow = {

		"nTr": null,

		"anCells": null,
	
		"_aData": [],
	
		"_aSortData": null,
	
		"_aFilterData": null,

		"_sFilterRow": null,
	
		"_sRowStripe": "",

		"src": null,

		"idx": -1
	};

	DataTable.models.oColumn = {

		"idx": null,

		"aDataSort": null,

		"asSorting": null,

		"bSearchable": null,

		"bSortable": null,

		"bVisible": null,
	
		"_sManualType": null,
	
		"_bAttrSrc": false,
	
		"fnCreatedCell": null,
	
		"fnGetData": null,
	
		"fnSetData": null,
	
		"mData": null,
	
		"mRender": null,
	
		"nTh": null,
	
		"nTf": null,
	
		"sClass": null,
	
		"sContentPadding": null,
	
		"sDefaultContent": null,
	
		"sName": null,
	
		"sSortDataType": 'std',
	
		"sSortingClass": null,
	
		"sSortingClassJUI": null,

		"sTitle": null,

		"sType": null,

		"sWidth": null,

		"sWidthOrig": null
	};

	DataTable.defaults = {
		
		"aaData": null,
	
		"aaSorting": [[0,'asc']],
	
		"aaSortingFixed": [],
	
		"ajax": null,
	
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
		"aoColumns": null,
	
		"aoColumnDefs": null,
	
		"aoSearchCols": [],
	
		"asStripeClasses": null,
	
		"bAutoWidth": true,
	
		"bDeferRender": false,
	
		"bDestroy": false,
	
		"bFilter": true,
	
		"bInfo": true,
	
		"bLengthChange": true,
	
		"bPaginate": true,
	
		"bProcessing": false,
	
		"bRetrieve": false,
	
		"bScrollCollapse": false,
	
		"bServerSide": false,
	
		"bSort": true,

		"bSortMulti": true,
	
		"bSortCellsTop": false,
	
		"bSortClasses": true,
	
		"bStateSave": false,
	
		"fnCreatedRow": null,
	
		"fnDrawCallback": null,
	
		"fnFooterCallback": null,
	
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
		"fnHeaderCallback": null,
	
		"fnInfoCallback": null,
	
		"fnInitComplete": null,

		"fnPreDrawCallback": null,

		"fnRowCallback": null,

		"fnServerData": null,

		"fnServerParams": null,

		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {}
		},
	
		"fnStateLoadParams": null,

		"fnStateLoaded": null,

		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {}
		},

		"fnStateSaveParams": null,

		"iStateDuration": 7200,

		"iDeferLoading": null,

		"iDisplayLength": 10,

		"iDisplayStart": 0,

		"iTabIndex": 0,
	
		"oClasses": {},
	
		"oLanguage": {
			
			"oAria": {
				
				"sSortAscending": ": activate to sort column ascending",
	
				"sSortDescending": ": activate to sort column descending"
			},
	

			"oPaginate": {

				"sFirst": "First",

				"sLast": "Last",


				"sPrevious": "Previous"
			},
	
			"sEmptyTable": "No data available in table",

			"sInfo": "Showing _START_ to _END_ of _TOTAL_ entries",

			"sInfoEmpty": "Showing 0 to 0 of 0 entries",

			"sInfoFiltered": "(filtered from _MAX_ total entries)",

			"sInfoPostFix": "",

			"sDecimal": "",

			"sThousands": ",",

			"sLengthMenu": "Show _MENU_ entries",

			"sLoadingRecords": "Loading...",

			"sProcessing": "Processing...",

			"sSearch": "Search:",

			"sSearchPlaceholder": "",

			"sUrl": "",

			"sZeroRecords": "No matching records found"
		},
	

		"oSearch": $.extend( {}, DataTable.models.oSearch ),

		"sAjaxDataProp": "data",

		"sAjaxSource": null,

		"sDom": "lfrtip",

		"searchDelay": null,

		"sPaginationType": "simple_numbers",

		"sScrollX": "",

		"sScrollXInner": "",

		"sScrollY": "",
	
		"sServerMethod": "GET",

		"renderer": null,

		"rowId": "DT_RowId"
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	DataTable.defaults.column = {
		
		"aDataSort": null,
		"iDataSort": -1,
	
		"asSorting": [ 'asc', 'desc' ],
	
		"bSearchable": true,
	
		"bSortable": true,

		"bVisible": true,

		"fnCreatedCell": null,

		"mData": null,
	
		"mRender": null,
	
		"sCellType": "td",
	
		"sClass": "",
	
		"sContentPadding": "",
	
		"sDefaultContent": null,
	
		"sName": "",
	
		"sSortDataType": "std",
	
		"sTitle": null,
	
		"sType": null,
	
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	DataTable.models.oSettings = {

		"oFeatures": {

			"bAutoWidth": null,

			"bDeferRender": null,

			"bFilter": null,
	
			"bInfo": null,

			"bLengthChange": null,

			"bPaginate": null,

			"bProcessing": null,

			"bServerSide": null,

			"bSort": null,

			"bSortMulti": null,

			"bSortClasses": null,

			"bStateSave": null
		},
	
		"oScroll": {

			"bCollapse": null,

			"iBarWidth": 0,

			"sX": null,

			"sXInner": null,
	
			"sY": null
		},
	
		"oLanguage": {

			"fnInfoCallback": null
		},
	
		"oBrowser": {

			"bScrollOversize": false,

			"bScrollbarLeft": false,
	
			"bBounding": false,

			"barWidth": 0
		},
	
	
		"ajax": null,

		"aanFeatures": [],

		"aoData": [],

		"aiDisplay": [],

		"aiDisplayMaster": [],

		"aIds": {},

		"aoColumns": [],

		"aoHeader": [],

		"aoFooter": [],

		"oPreviousSearch": {},

		"aoPreSearchCols": [],

		"aaSorting": null,

		"aaSortingFixed": [],

		"asStripeClasses": null,

		"asDestroyStripes": [],

		"sDestroyWidth": 0,

		"aoRowCallback": [],

		"aoHeaderCallback": [],

		"aoFooterCallback": [],

		"aoDrawCallback": [],

		"aoRowCreatedCallback": [],

		"aoPreDrawCallback": [],

		"aoInitComplete": [],

		"aoStateSaveParams": [],

		"aoStateLoadParams": [],

		"aoStateLoaded": [],

		"sTableId": "",

		"nTable": null,

		"nTHead": null,

		"nTFoot": null,
	
		"nTBody": null,

		"nTableWrapper": null,

		"bDeferLoading": false,

		"bInitialised": false,

		"aoOpenRows": [],

		"sDom": null,

		"searchDelay": null,
	
		"sPaginationType": "two_button",

		"iStateDuration": 0,

		"aoStateSave": [],

		"aoStateLoad": [],

		"oSavedState": null,

		"oLoadedState": null,
	
		"sAjaxSource": null,

		"sAjaxDataProp": null,

		"bAjaxDataGet": true,

		"jqXHR": null,
	
		"json": undefined,

		"oAjaxData": undefined,

		"fnServerData": null,

		"aoServerParams": [],

		"sServerMethod": null,

		"fnFormatNumber": null,

		"aLengthMenu": null,

		"iDraw": 0,

		"bDrawing": false,

		"iDrawError": -1,

		"_iDisplayLength": 10,

		"_iDisplayStart": 0,

		"_iRecordsTotal": 0,
	
		"_iRecordsDisplay": 0,

		"oClasses": {},

		"bFiltered": false,

		"bSorted": false,

		"bSortCellsTop": null,

		"oInit": null,
	
		"aoDestroyCallback": [],

		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 *  @type function
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 *  @type function
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		"oInstance": null,
	
		"sInstance": null,

		"iTabIndex": 0,

		"nScrollHead": null,

		"nScrollFoot": null,

		"aLastSort": [],

		"oPlugins": {},

		"rowIdFn": null,

		"rowId": null
	};

	DataTable.ext = _ext = {

		buttons: {},

		classes: {},
	
		builder: "-source-",
	
		errMode: "alert",
	
		feature: [],
	
		search: [],
	
		selector: {
			cell: [],
			column: [],
			row: []
		},

		internal: {},
	
		legacy: {

			ajax: null
		},
	
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
		order: {},

		type: {
	
			detect: [],
	
			search: {},
	
			order: {}
		},

		_unique: 0,

		fnVersionCheck: DataTable.fnVersionCheck,

		iApiIndex: 0,

		oJUIClasses: {},

		sVersion: DataTable.version
	};
	

	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oApi:         _ext.internal,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		"sTable": "dataTable",
		"sNoFooter": "no-footer",
	
		/* Paging buttons */
		"sPageButton": "paginate_button",
		"sPageButtonActive": "current",
		"sPageButtonDisabled": "disabled",
	
		/* Striping classes */
		"sStripeOdd": "odd",
		"sStripeEven": "even",
	
		/* Empty row */
		"sRowEmpty": "dataTables_empty",
	
		/* Features */
		"sWrapper": "dataTables_wrapper",
		"sFilter": "dataTables_filter",
		"sInfo": "dataTables_info",
		"sPaging": "dataTables_paginate paging_", /* Note that the type is postfixed */
		"sLength": "dataTables_length",
		"sProcessing": "dataTables_processing",
	
		/* Sorting */
		"sSortAsc": "sorting_asc",
		"sSortDesc": "sorting_desc",
		"sSortable": "sorting", /* Sortable in both directions */
		"sSortableAsc": "sorting_asc_disabled",
		"sSortableDesc": "sorting_desc_disabled",
		"sSortableNone": "sorting_disabled",
		"sSortColumn": "sorting_", /* Note that an int is postfixed for the sorting order */
	
		/* Filtering */
		"sFilterInput": "",
	
		/* Page length */
		"sLengthSelect": "",
	
		/* Scrolling */
		"sScrollWrapper": "dataTables_scroll",
		"sScrollHead": "dataTables_scrollHead",
		"sScrollHeadInner": "dataTables_scrollHeadInner",
		"sScrollBody": "dataTables_scrollBody",
		"sScrollFoot": "dataTables_scrollFoot",
		"sScrollFootInner": "dataTables_scrollFootInner",
	
		/* Misc */
		"sHeaderTH": "",
		"sFooterTH": "",
	
		// Deprecated
		"sSortJUIAsc": "",
		"sSortJUIDesc": "",
		"sSortJUI": "",
		"sSortJUIAscAllowed": "",
		"sSortJUIDescAllowed": "",
		"sSortJUIWrapper": "",
		"sSortIcon": "",
		"sJUIHeader": "",
		"sJUIFooter": ""
	} );
	
	
	var extPagination = DataTable.ext.pager;
	
	function _numbers ( page, pages ) {
		var
			numbers = [],
			buttons = extPagination.numbers_length,
			half = Math.floor( buttons / 2 ),
			i = 1;
	
		if ( pages <= buttons ) {
			numbers = _range( 0, pages );
		}
		else if ( page <= half ) {
			numbers = _range( 0, buttons-2 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range( pages-(buttons-2), pages );
			numbers.splice( 0, 0, 'ellipsis' ); // no unshift in ie6
			numbers.splice( 0, 0, 0 );
		}
		else {
			numbers = _range( page-half+2, page+half-1 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
			numbers.splice( 0, 0, 'ellipsis' );
			numbers.splice( 0, 0, 0 );
		}
	
		numbers.DT_el = 'span';
		return numbers;
	}
	
	
	$.extend( extPagination, {
		simple: function ( page, pages ) {
			return [ 'previous', 'next' ];
		},
	
		full: function ( page, pages ) {
			return [  'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function ( page, pages ) {
			return [ _numbers(page, pages) ];
		},
	
		simple_numbers: function ( page, pages ) {
			return [ 'previous', _numbers(page, pages), 'next' ];
		},
	
		full_numbers: function ( page, pages ) {
			return [ 'first', 'previous', _numbers(page, pages), 'next', 'last' ];
		},
		
		first_last_numbers: function (page, pages) {
	 		return ['first', _numbers(page, pages), 'last'];
	 	},
	
		// For testing and plug-ins to use
		_numbers: _numbers,
	
		// Number of number buttons (including ellipsis) to show. _Must be odd!_
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pageButton: {
			_: function ( settings, host, idx, buttons, page, pages ) {
				var classes = settings.oClasses;
				var lang = settings.oLanguage.oPaginate;
				var aria = settings.oLanguage.oAria.paginate || {};
				var btnDisplay, btnClass, counter=0;
	
				var attach = function( container, buttons ) {
					var i, ien, node, button;
					var clickHandler = function ( e ) {
						_fnPageChange( settings, e.data.action, true );
					};
	
					for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
						button = buttons[i];
	
						if ( $.isArray( button ) ) {
							var inner = $( '<'+(button.DT_el || 'div')+'/>' )
								.appendTo( container );
							attach( inner, button );
						}
						else {
							btnDisplay = null;
							btnClass = '';
	
							switch ( button ) {
								case 'ellipsis':
									container.append('<span class="ellipsis">&#x2026;</span>');
									break;
	
								case 'first':
									btnDisplay = lang.sFirst;
									btnClass = button + (page > 0 ?
										'' : ' '+classes.sPageButtonDisabled);
									break;
	
								case 'previous':
									btnDisplay = lang.sPrevious;
									btnClass = button + (page > 0 ?
										'' : ' '+classes.sPageButtonDisabled);
									break;
	
								case 'next':
									btnDisplay = lang.sNext;
									btnClass = button + (page < pages-1 ?
										'' : ' '+classes.sPageButtonDisabled);
									break;
	
								case 'last':
									btnDisplay = lang.sLast;
									btnClass = button + (page < pages-1 ?
										'' : ' '+classes.sPageButtonDisabled);
									break;
	
								default:
									btnDisplay = button + 1;
									btnClass = page === button ?
										classes.sPageButtonActive : '';
									break;
							}
	
							if ( btnDisplay !== null ) {
								node = $('<a>', {
										'class': classes.sPageButton+' '+btnClass,
										'aria-controls': settings.sTableId,
										'aria-label': aria[ button ],
										'data-dt-idx': counter,
										'tabindex': settings.iTabIndex,
										'id': idx === 0 && typeof button === 'string' ?
											settings.sTableId +'_'+ button :
											null
									} )
									.html( btnDisplay )
									.appendTo( container );
	
								_fnBindAction(
									node, {action: button}, clickHandler
								);
	
								counter++;
							}
						}
					}
				};
	
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame. Try / catch the error. Not good for
				// accessibility, but neither are frames.
				var activeEl;
	
				try {
					// Because this approach is destroying and recreating the paging
					// elements, focus is lost on the select button which is bad for
					// accessibility. So we want to restore focus once the draw has
					// completed
					activeEl = $(host).find(document.activeElement).data('dt-idx');
				}
				catch (e) {}
	
				attach( $(host).empty(), buttons );
	
				if ( activeEl !== undefined ) {
					$(host).find( '[data-dt-idx='+activeEl+']' ).focus();
				}
			}
		}
	} );
	
	
	
	// Built in type detection. See model.ext.aTypes for information about
	// what is required from this methods.
	$.extend( DataTable.ext.type.detect, [
		// Plain numbers - first since V8 detects some plain numbers as dates
		// e.g. Date.parse('55') (but not all, e.g. Date.parse('22')...).
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal ) ? 'num'+decimal : null;
		},
	
		// Dates (only those recognised by the browser's Date.parse)
		function ( d, settings )
		{
			// V8 tries _very_ hard to make a string passed into `Date.parse()`
			// valid, so we need to use a regex to restrict date formats. Use a
			// plug-in for anything other than ISO8601 style strings
			if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
				return null;
			}
			var parsed = Date.parse(d);
			return (parsed !== null && !isNaN(parsed)) || _empty(d) ? 'date' : null;
		},
	
		// Formatted numbers
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal, true ) ? 'num-fmt'+decimal : null;
		},
	
		// HTML numeric
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal ) ? 'html-num'+decimal : null;
		},
	
		// HTML numeric, formatted
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal, true ) ? 'html-num-fmt'+decimal : null;
		},
	
		// HTML (this is strict checking - there must be html)
		function ( d, settings )
		{
			return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1) ?
				'html' : null;
		}
	] );
	
	
	
	// Filter formatting functions. See model.ext.ofnSearch for information about
	// what is required from these methods.
	// 
	// Note that additional search methods are added for the html numbers and
	// html formatted numbers by `_addNumericSort()` when we know what the decimal
	// place is
	
	
	$.extend( DataTable.ext.type.search, {
		html: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data
						.replace( _re_new_lines, " " )
						.replace( _re_html, "" ) :
					'';
		},
	
		string: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data.replace( _re_new_lines, " " ) :
					data;
		}
	} );
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	// Add the numeric 'deformatting' functions for sorting and search. This is done
	// in a function to provide an easy ability for the language options to add
	// additional methods if a non-period decimal place is used.
	function _addNumericSort ( decimalPlace ) {
		$.each(
			{
				// Plain numbers
				"num": function ( d ) {
					return __numericReplace( d, decimalPlace );
				},
	
				// Formatted numbers
				"num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_formatted_numeric );
				},
	
				// HTML numeric
				"html-num": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html );
				},
	
				// HTML numeric, formatted
				"html-num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html, _re_formatted_numeric );
				}
			},
			function ( key, fn ) {
				// Add the ordering method
				_ext.type.order[ key+decimalPlace+'-pre' ] = fn;
	
				// For HTML types add a search formatter that will strip the HTML
				if ( key.match(/^html\-/) ) {
					_ext.type.search[ key+decimalPlace ] = _ext.type.search.html;
				}
			}
		);
	}
	
	
	// Default sort methods
	$.extend( _ext.type.order, {
		// Dates
		"date-pre": function ( d ) {
			var ts = Date.parse( d );
			return isNaN(ts) ? -Infinity : ts;
		},
	
		// html
		"html-pre": function ( a ) {
			return _empty(a) ?
				'' :
				a.replace ?
					a.replace( /<.*?>/g, "" ).toLowerCase() :
					a+'';
		},
	
		// string
		"string-pre": function ( a ) {
			// This is a little complex, but faster than always calling toString,
			// http://jsperf.com/tostring-v-check
			return _empty(a) ?
				'' :
				typeof a === 'string' ?
					a.toLowerCase() :
					! a.toString ?
						'' :
						a.toString();
		},
	
		// string-asc and -desc are retained only for compatibility with the old
		// sort methods
		"string-asc": function ( x, y ) {
			return ((x < y) ? -1 : ((x > y) ? 1 : 0));
		},
	
		"string-desc": function ( x, y ) {
			return ((x < y) ? 1 : ((x > y) ? -1 : 0));
		}
	} );
	
	
	// Numeric sorting types - order doesn't matter here
	_addNumericSort( '' );
	
	
	$.extend( true, DataTable.ext.renderer, {
		header: {
			_: function ( settings, cell, column, classes ) {
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass(
							column.sSortingClass +' '+
							classes.sSortAsc +' '+
							classes.sSortDesc
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
				} );
			},
	
			jqueryui: function ( settings, cell, column, classes ) {
				$('<div/>')
					.addClass( classes.sSortJUIWrapper )
					.append( cell.contents() )
					.append( $('<span/>')
						.addClass( classes.sSortIcon+' '+column.sSortingClassJUI )
					)
					.appendTo( cell );
	
				// Attach a sort listener to update on sort
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) {
						return;
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass( classes.sSortAsc +" "+classes.sSortDesc )
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
	
					cell
						.find( 'span.'+classes.sSortIcon )
						.removeClass(
							classes.sSortJUIAsc +" "+
							classes.sSortJUIDesc +" "+
							classes.sSortJUI +" "+
							classes.sSortJUIAscAllowed +" "+
							classes.sSortJUIDescAllowed
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortJUIAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortJUIDesc :
								column.sSortingClassJUI
						);
				} );
			}
		}
	} );
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	var __htmlEscapeEntities = function ( d ) {
		return typeof d === 'string' ?
			d.replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;') :
			d;
	};

	DataTable.render = {
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return __htmlEscapeEntities( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: __htmlEscapeEntities,
				filter: __htmlEscapeEntities
			};
		}
	};
	
	function _fnExternApiFunc (fn)
	{
		return function() {
			var args = [_fnSettingsFromNode( this[DataTable.ext.iApiIndex] )].concat(
				Array.prototype.slice.call(arguments)
			);
			return DataTable.ext.internal[fn].apply( this, args );
		};
	}

	$.extend( DataTable.ext.internal, {
		_fnExternApiFunc: _fnExternApiFunc,
		_fnBuildAjax: _fnBuildAjax,
		_fnAjaxUpdate: _fnAjaxUpdate,
		_fnAjaxParameters: _fnAjaxParameters,
		_fnAjaxUpdateDraw: _fnAjaxUpdateDraw,
		_fnAjaxDataSrc: _fnAjaxDataSrc,
		_fnAddColumn: _fnAddColumn,
		_fnColumnOptions: _fnColumnOptions,
		_fnAdjustColumnSizing: _fnAdjustColumnSizing,
		_fnVisibleToColumnIndex: _fnVisibleToColumnIndex,
		_fnColumnIndexToVisible: _fnColumnIndexToVisible,
		_fnVisbleColumns: _fnVisbleColumns,
		_fnGetColumns: _fnGetColumns,
		_fnColumnTypes: _fnColumnTypes,
		_fnApplyColumnDefs: _fnApplyColumnDefs,
		_fnHungarianMap: _fnHungarianMap,
		_fnCamelToHungarian: _fnCamelToHungarian,
		_fnLanguageCompat: _fnLanguageCompat,
		_fnBrowserDetect: _fnBrowserDetect,
		_fnAddData: _fnAddData,
		_fnAddTr: _fnAddTr,
		_fnNodeToDataIndex: _fnNodeToDataIndex,
		_fnNodeToColumnIndex: _fnNodeToColumnIndex,
		_fnGetCellData: _fnGetCellData,
		_fnSetCellData: _fnSetCellData,
		_fnSplitObjNotation: _fnSplitObjNotation,
		_fnGetObjectDataFn: _fnGetObjectDataFn,
		_fnSetObjectDataFn: _fnSetObjectDataFn,
		_fnGetDataMaster: _fnGetDataMaster,
		_fnClearTable: _fnClearTable,
		_fnDeleteIndex: _fnDeleteIndex,
		_fnInvalidate: _fnInvalidate,
		_fnGetRowElements: _fnGetRowElements,
		_fnCreateTr: _fnCreateTr,
		_fnBuildHead: _fnBuildHead,
		_fnDrawHead: _fnDrawHead,
		_fnDraw: _fnDraw,
		_fnReDraw: _fnReDraw,
		_fnAddOptionsHtml: _fnAddOptionsHtml,
		_fnDetectHeader: _fnDetectHeader,
		_fnGetUniqueThs: _fnGetUniqueThs,
		_fnFeatureHtmlFilter: _fnFeatureHtmlFilter,
		_fnFilterComplete: _fnFilterComplete,
		_fnFilterCustom: _fnFilterCustom,
		_fnFilterColumn: _fnFilterColumn,
		_fnFilter: _fnFilter,
		_fnFilterCreateSearch: _fnFilterCreateSearch,
		_fnEscapeRegex: _fnEscapeRegex,
		_fnFilterData: _fnFilterData,
		_fnFeatureHtmlInfo: _fnFeatureHtmlInfo,
		_fnUpdateInfo: _fnUpdateInfo,
		_fnInfoMacros: _fnInfoMacros,
		_fnInitialise: _fnInitialise,
		_fnInitComplete: _fnInitComplete,
		_fnLengthChange: _fnLengthChange,
		_fnFeatureHtmlLength: _fnFeatureHtmlLength,
		_fnFeatureHtmlPaginate: _fnFeatureHtmlPaginate,
		_fnPageChange: _fnPageChange,
		_fnFeatureHtmlProcessing: _fnFeatureHtmlProcessing,
		_fnProcessingDisplay: _fnProcessingDisplay,
		_fnFeatureHtmlTable: _fnFeatureHtmlTable,
		_fnScrollDraw: _fnScrollDraw,
		_fnApplyToChildren: _fnApplyToChildren,
		_fnCalculateColumnWidths: _fnCalculateColumnWidths,
		_fnThrottle: _fnThrottle,
		_fnConvertToWidth: _fnConvertToWidth,
		_fnGetWidestNode: _fnGetWidestNode,
		_fnGetMaxLenString: _fnGetMaxLenString,
		_fnStringToCss: _fnStringToCss,
		_fnSortFlatten: _fnSortFlatten,
		_fnSort: _fnSort,
		_fnSortAria: _fnSortAria,
		_fnSortListener: _fnSortListener,
		_fnSortAttachListener: _fnSortAttachListener,
		_fnSortingClasses: _fnSortingClasses,
		_fnSortData: _fnSortData,
		_fnSaveState: _fnSaveState,
		_fnLoadState: _fnLoadState,
		_fnSettingsFromNode: _fnSettingsFromNode,
		_fnLog: _fnLog,
		_fnMap: _fnMap,
		_fnBindAction: _fnBindAction,
		_fnCallbackReg: _fnCallbackReg,
		_fnCallbackFire: _fnCallbackFire,
		_fnLengthOverflow: _fnLengthOverflow,
		_fnRenderer: _fnRenderer,
		_fnDataSource: _fnDataSource,
		_fnRowAttributes: _fnRowAttributes,
		_fnExtend: _fnExtend,
		_fnCalculateEnd: function () {} 
	} );
	

	// jQuery access
	$.fn.dataTable = DataTable;

	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;

	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;

	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};

	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );

	return $.fn.dataTable;
}));


(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	dom:
		"<'row'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
		"<'row'<'col-sm-12'tr>>" +
		"<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>",
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm",
	sLengthSelect: "custom-select custom-select-sm form-control form-control-sm",
	sProcessing:   "dataTables_processing card",
	sPageButton:   "paginate_button page-item"
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( $.isArray( button ) ) {
				attach( container, button );
			}
			else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame. 
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).focus();
	}
};


return DataTable;
}));

"use strict";
var defaults = {
	"language": {
		"paginate": {
			"first": '<i class="la la-angle-double-left"></i>',
			"last": '<i class="la la-angle-double-right"></i>',
			"next": '<i class="la la-angle-right"></i>',
			"previous": '<i class="la la-angle-left"></i>'
		}
	}
};

if (KTUtil.isRTL()) {
	defaults = {
		"language": {
			"paginate": {
				"first": '<i class="la la-angle-double-right"></i>',
				"last": '<i class="la la-angle-double-left"></i>',
				"next": '<i class="la la-angle-left"></i>',
				"previous": '<i class="la la-angle-right"></i>'
			}
		}
	}
}

$.extend(true, $.fn.dataTable.defaults, defaults);

// fix dropdown overflow inside datatable
//KTApp.initAbsoluteDropdown('.dataTables_wrapper');
/*!
 AutoFill 2.3.3
 ©2008-2018 SpryMedia Ltd - datatables.net/license
*/
(function(e){"function"===typeof define&&define.amd?define(["jquery","datatables.net"],function(h){return e(h,window,document)}):"object"===typeof exports?module.exports=function(h,i){h||(h=window);if(!i||!i.fn.dataTable)i=require("datatables.net")(h,i).$;return e(i,h,h.document)}:e(jQuery,window,document)})(function(e,h,i,r){var l=e.fn.dataTable,s=0,k=function(b,c){if(!l.versionCheck||!l.versionCheck("1.10.8"))throw"Warning: AutoFill requires DataTables 1.10.8 or greater";this.c=e.extend(!0,{},l.defaults.autoFill,
k.defaults,c);this.s={dt:new l.Api(b),namespace:".autoFill"+s++,scroll:{},scrollInterval:null,handle:{height:0,width:0},enabled:!1};this.dom={handle:e('<div class="dt-autofill-handle"/>'),select:{top:e('<div class="dt-autofill-select top"/>'),right:e('<div class="dt-autofill-select right"/>'),bottom:e('<div class="dt-autofill-select bottom"/>'),left:e('<div class="dt-autofill-select left"/>')},background:e('<div class="dt-autofill-background"/>'),list:e('<div class="dt-autofill-list">'+this.s.dt.i18n("autoFill.info",
"")+"<ul/></div>"),dtScroll:null,offsetParent:null};this._constructor()};e.extend(k.prototype,{enabled:function(){return this.s.enabled},enable:function(b){var c=this;if(!1===b)return this.disable();this.s.enabled=!0;this._focusListener();this.dom.handle.on("mousedown",function(a){c._mousedown(a);return!1});return this},disable:function(){this.s.enabled=!1;this._focusListenerRemove();return this},_constructor:function(){var b=this,c=this.s.dt,a=e("div.dataTables_scrollBody",this.s.dt.table().container());
c.settings()[0].autoFill=this;a.length&&(this.dom.dtScroll=a,"static"===a.css("position")&&a.css("position","relative"));!1!==this.c.enable&&this.enable();c.on("destroy.autoFill",function(){b._focusListenerRemove()})},_attach:function(b){var c=this.s.dt,a=c.cell(b).index(),d=this.dom.handle,f=this.s.handle;if(!a||-1===c.columns(this.c.columns).indexes().indexOf(a.column))this._detach();else{this.dom.offsetParent||(this.dom.offsetParent=e(c.table().node()).offsetParent());if(!f.height||!f.width)d.appendTo("body"),
f.height=d.outerHeight(),f.width=d.outerWidth();c=this._getPosition(b,this.dom.offsetParent);this.dom.attachedTo=b;d.css({top:c.top+b.offsetHeight-f.height,left:c.left+b.offsetWidth-f.width}).appendTo(this.dom.offsetParent)}},_actionSelector:function(b){var c=this,a=this.s.dt,d=k.actions,f=[];e.each(d,function(c,d){d.available(a,b)&&f.push(c)});if(1===f.length&&!1===this.c.alwaysAsk){var j=d[f[0]].execute(a,b);this._update(j,b)}else{var g=this.dom.list.children("ul").empty();f.push("cancel");e.each(f,
function(f,j){g.append(e("<li/>").append('<div class="dt-autofill-question">'+d[j].option(a,b)+"<div>").append(e('<div class="dt-autofill-button">').append(e('<button class="'+k.classes.btn+'">'+a.i18n("autoFill.button","&gt;")+"</button>").on("click",function(){var f=d[j].execute(a,b,e(this).closest("li"));c._update(f,b);c.dom.background.remove();c.dom.list.remove()}))))});this.dom.background.appendTo("body");this.dom.list.appendTo("body");this.dom.list.css("margin-top",-1*(this.dom.list.outerHeight()/
2))}},_detach:function(){this.dom.attachedTo=null;this.dom.handle.detach()},_drawSelection:function(b){var c=this.s.dt,a=this.s.start,d=e(this.dom.start),f={row:this.c.vertical?c.rows({page:"current"}).nodes().indexOf(b.parentNode):a.row,column:this.c.horizontal?e(b).index():a.column},b=c.column.index("toData",f.column),j=c.row(":eq("+f.row+")",{page:"current"}),j=e(c.cell(j.index(),b).node());if(c.cell(j).any()&&-1!==c.columns(this.c.columns).indexes().indexOf(b)){this.s.end=f;var g,c=a.row<f.row?
d:j;g=a.row<f.row?j:d;b=a.column<f.column?d:j;d=a.column<f.column?j:d;c=this._getPosition(c.get(0)).top;b=this._getPosition(b.get(0)).left;a=this._getPosition(g.get(0)).top+g.outerHeight()-c;d=this._getPosition(d.get(0)).left+d.outerWidth()-b;f=this.dom.select;f.top.css({top:c,left:b,width:d});f.left.css({top:c,left:b,height:a});f.bottom.css({top:c+a,left:b,width:d});f.right.css({top:c,left:b+d,height:a})}},_editor:function(b){var c=this.s.dt,a=this.c.editor;if(a){for(var d={},f=[],e=a.fields(),g=
0,i=b.length;g<i;g++)for(var q=0,k=b[g].length;q<k;q++){var m=b[g][q],o=c.settings()[0].aoColumns[m.index.column],p=o.editField;if(p===r)for(var o=o.mData,h=0,l=e.length;h<l;h++){var n=a.field(e[h]);if(n.dataSrc()===o){p=n.name();break}}if(!p)throw"Could not automatically determine field data. Please see https://datatables.net/tn/11";d[p]||(d[p]={});o=c.row(m.index.row).id();d[p][o]=m.set;f.push(m.index)}a.bubble(f,!1).multiSet(d).submit()}},_emitEvent:function(b,c){this.s.dt.iterator("table",function(a){e(a.nTable).triggerHandler(b+
".dt",c)})},_focusListener:function(){var b=this,c=this.s.dt,a=this.s.namespace,d=null!==this.c.focus?this.c.focus:c.init().keys||c.settings()[0].keytable?"focus":"hover";if("focus"===d)c.on("key-focus.autoFill",function(a,c,d){b._attach(d.node())}).on("key-blur.autoFill",function(){b._detach()});else if("click"===d)e(c.table().body()).on("click"+a,"td, th",function(){b._attach(this)}),e(i.body).on("click"+a,function(a){e(a.target).parents().filter(c.table().body()).length||b._detach()});else e(c.table().body()).on("mouseenter"+
a,"td, th",function(){b._attach(this)}).on("mouseleave"+a,function(a){e(a.relatedTarget).hasClass("dt-autofill-handle")||b._detach()})},_focusListenerRemove:function(){var b=this.s.dt;b.off(".autoFill");e(b.table().body()).off(this.s.namespace);e(i.body).off(this.s.namespace)},_getPosition:function(b,c){var a=b,d,f=0,j=0;c||(c=e(e(this.s.dt.table().node())[0].offsetParent));do{var g=a.offsetTop,i=a.offsetLeft;d=e(a.offsetParent);f+=g+d.scrollTop();j+=i+d.scrollLeft();f+=1*parseInt(d.css("margin-top"));
f+=1*parseInt(d.css("border-top-width"));if("body"===a.nodeName.toLowerCase())break;a=d.get(0)}while(d.get(0)!==c.get(0));return{top:f,left:j}},_mousedown:function(b){var c=this,a=this.s.dt;this.dom.start=this.dom.attachedTo;this.s.start={row:a.rows({page:"current"}).nodes().indexOf(e(this.dom.start).parent()[0]),column:e(this.dom.start).index()};e(i.body).on("mousemove.autoFill",function(a){c._mousemove(a)}).on("mouseup.autoFill",function(a){c._mouseup(a)});var d=this.dom.select,a=e(a.table().node()).offsetParent();
d.top.appendTo(a);d.left.appendTo(a);d.right.appendTo(a);d.bottom.appendTo(a);this._drawSelection(this.dom.start,b);this.dom.handle.css("display","none");b=this.dom.dtScroll;this.s.scroll={windowHeight:e(h).height(),windowWidth:e(h).width(),dtTop:b?b.offset().top:null,dtLeft:b?b.offset().left:null,dtHeight:b?b.outerHeight():null,dtWidth:b?b.outerWidth():null}},_mousemove:function(b){var c=b.target.nodeName.toLowerCase();"td"!==c&&"th"!==c||(this._drawSelection(b.target,b),this._shiftScroll(b))},_mouseup:function(b){e(i.body).off(".autoFill");
var c=this,a=this.s.dt,d=this.dom.select;d.top.remove();d.left.remove();d.right.remove();d.bottom.remove();this.dom.handle.css("display","block");var d=this.s.start,f=this.s.end;if(!(d.row===f.row&&d.column===f.column)){var j=a.cell(":eq("+d.row+")",d.column+":visible",{page:"current"});if(e("div.DTE",j.node()).length){var g=a.editor();g.on("submitSuccess.dtaf close.dtaf",function(){g.off(".dtaf");setTimeout(function(){c._mouseup(b)},100)}).on("submitComplete.dtaf preSubmitCancelled.dtaf close.dtaf",
function(){g.off(".dtaf")});g.submit()}else{for(var h=this._range(d.row,f.row),d=this._range(d.column,f.column),f=[],k=a.settings()[0],l=k.aoColumns,m=0;m<h.length;m++)f.push(e.map(d,function(b){var c=a.row(":eq("+h[m]+")",{page:"current"}),b=a.cell(c.index(),b+":visible"),c=b.data(),d=b.index(),f=l[d.column].editField;f!==r&&(c=k.oApi._fnGetObjectDataFn(f)(a.row(d.row).data()));return{cell:b,data:c,label:b.data(),index:d}}));this._actionSelector(f);clearInterval(this.s.scrollInterval);this.s.scrollInterval=
null}}},_range:function(b,c){var a=[],d;if(b<=c)for(d=b;d<=c;d++)a.push(d);else for(d=b;d>=c;d--)a.push(d);return a},_shiftScroll:function(b){var c=this,a=this.s.scroll,d=!1,f=b.pageY-i.body.scrollTop,e=b.pageX-i.body.scrollLeft,g,h,k,l;65>f?g=-5:f>a.windowHeight-65&&(g=5);65>e?h=-5:e>a.windowWidth-65&&(h=5);null!==a.dtTop&&b.pageY<a.dtTop+65?k=-5:null!==a.dtTop&&b.pageY>a.dtTop+a.dtHeight-65&&(k=5);null!==a.dtLeft&&b.pageX<a.dtLeft+65?l=-5:null!==a.dtLeft&&b.pageX>a.dtLeft+a.dtWidth-65&&(l=5);g||
h||k||l?(a.windowVert=g,a.windowHoriz=h,a.dtVert=k,a.dtHoriz=l,d=!0):this.s.scrollInterval&&(clearInterval(this.s.scrollInterval),this.s.scrollInterval=null);!this.s.scrollInterval&&d&&(this.s.scrollInterval=setInterval(function(){if(a.windowVert)i.body.scrollTop=i.body.scrollTop+a.windowVert;if(a.windowHoriz)i.body.scrollLeft=i.body.scrollLeft+a.windowHoriz;if(a.dtVert||a.dtHoriz){var b=c.dom.dtScroll[0];if(a.dtVert)b.scrollTop=b.scrollTop+a.dtVert;if(a.dtHoriz)b.scrollLeft=b.scrollLeft+a.dtHoriz}},
20))},_update:function(b,c){if(!1!==b){var a=this.s.dt,d,f=a.columns(this.c.columns).indexes();this._emitEvent("preAutoFill",[a,c]);this._editor(c);if(null!==this.c.update?this.c.update:!this.c.editor){for(var e=0,g=c.length;e<g;e++)for(var h=0,i=c[e].length;h<i;h++)d=c[e][h],-1!==f.indexOf(d.index.column)&&d.cell.data(d.set);a.draw(!1)}this._emitEvent("autoFill",[a,c])}}});k.actions={increment:{available:function(b,c){var a=c[0][0].label;return!isNaN(a-parseFloat(a))},option:function(b){return b.i18n("autoFill.increment",
'Increment / decrement each cell by: <input type="number" value="1">')},execute:function(b,c,a){for(var b=1*c[0][0].data,a=1*e("input",a).val(),d=0,f=c.length;d<f;d++)for(var j=0,g=c[d].length;j<g;j++)c[d][j].set=b,b+=a}},fill:{available:function(){return!0},option:function(b,c){return b.i18n("autoFill.fill","Fill all cells with <i>"+c[0][0].label+"</i>")},execute:function(b,c){for(var a=c[0][0].data,d=0,f=c.length;d<f;d++)for(var e=0,g=c[d].length;e<g;e++)c[d][e].set=a}},fillHorizontal:{available:function(b,
c){return 1<c.length&&1<c[0].length},option:function(b){return b.i18n("autoFill.fillHorizontal","Fill cells horizontally")},execute:function(b,c){for(var a=0,d=c.length;a<d;a++)for(var f=0,e=c[a].length;f<e;f++)c[a][f].set=c[a][0].data}},fillVertical:{available:function(b,c){return 1<c.length&&1<c[0].length},option:function(b){return b.i18n("autoFill.fillVertical","Fill cells vertically")},execute:function(b,c){for(var a=0,d=c.length;a<d;a++)for(var f=0,e=c[a].length;f<e;f++)c[a][f].set=c[0][f].data}},
cancel:{available:function(){return!1},option:function(b){return b.i18n("autoFill.cancel","Cancel")},execute:function(){return!1}}};k.version="2.3.3";k.defaults={alwaysAsk:!1,focus:null,columns:"",enable:!0,update:null,editor:null,vertical:!0,horizontal:!0};k.classes={btn:"btn"};var n=e.fn.dataTable.Api;n.register("autoFill()",function(){return this});n.register("autoFill().enabled()",function(){var b=this.context[0];return b.autoFill?b.autoFill.enabled():!1});n.register("autoFill().enable()",function(b){return this.iterator("table",
function(c){c.autoFill&&c.autoFill.enable(b)})});n.register("autoFill().disable()",function(){return this.iterator("table",function(b){b.autoFill&&b.autoFill.disable()})});e(i).on("preInit.dt.autofill",function(b,c){if("dt"===b.namespace){var a=c.oInit.autoFill,d=l.defaults.autoFill;if(a||d)d=e.extend({},a,d),!1!==a&&new k(c,d)}});l.AutoFill=k;return l.AutoFill=k});

/*!
 Bootstrap integration for DataTables' AutoFill
 ©2015 SpryMedia Ltd - datatables.net/license
*/
(function(a){"function"===typeof define&&define.amd?define(["jquery","datatables.net-bs4","datatables.net-autofill"],function(b){return a(b,window,document)}):"object"===typeof exports?module.exports=function(b,c){b||(b=window);if(!c||!c.fn.dataTable)c=require("datatables.net-bs4")(b,c).$;c.fn.dataTable.AutoFill||require("datatables.net-autofill")(b,c);return a(c,b,b.document)}:a(jQuery,window,document)})(function(a){a=a.fn.dataTable;a.AutoFill.classes.btn="btn btn-primary";return a});

/*!

JSZip v3.2.1 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/master/LICENSE
*/

!function(t){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=t();else if("function"==typeof define&&define.amd)define([],t);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=t()}}(function(){return function s(a,o,h){function u(r,t){if(!o[r]){if(!a[r]){var e="function"==typeof require&&require;if(!t&&e)return e(r,!0);if(l)return l(r,!0);var i=new Error("Cannot find module '"+r+"'");throw i.code="MODULE_NOT_FOUND",i}var n=o[r]={exports:{}};a[r][0].call(n.exports,function(t){var e=a[r][1][t];return u(e||t)},n,n.exports,s,a,o,h)}return o[r].exports}for(var l="function"==typeof require&&require,t=0;t<h.length;t++)u(h[t]);return u}({1:[function(t,e,r){"use strict";var c=t("./utils"),d=t("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(t){for(var e,r,i,n,s,a,o,h=[],u=0,l=t.length,f=l,d="string"!==c.getTypeOf(t);u<t.length;)f=l-u,i=d?(e=t[u++],r=u<l?t[u++]:0,u<l?t[u++]:0):(e=t.charCodeAt(u++),r=u<l?t.charCodeAt(u++):0,u<l?t.charCodeAt(u++):0),n=e>>2,s=(3&e)<<4|r>>4,a=1<f?(15&r)<<2|i>>6:64,o=2<f?63&i:64,h.push(p.charAt(n)+p.charAt(s)+p.charAt(a)+p.charAt(o));return h.join("")},r.decode=function(t){var e,r,i,n,s,a,o=0,h=0,u="data:";if(t.substr(0,u.length)===u)throw new Error("Invalid base64 input, it looks like a data url.");var l,f=3*(t=t.replace(/[^A-Za-z0-9\+\/\=]/g,"")).length/4;if(t.charAt(t.length-1)===p.charAt(64)&&f--,t.charAt(t.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(l=d.uint8array?new Uint8Array(0|f):new Array(0|f);o<t.length;)e=p.indexOf(t.charAt(o++))<<2|(n=p.indexOf(t.charAt(o++)))>>4,r=(15&n)<<4|(s=p.indexOf(t.charAt(o++)))>>2,i=(3&s)<<6|(a=p.indexOf(t.charAt(o++))),l[h++]=e,64!==s&&(l[h++]=r),64!==a&&(l[h++]=i);return l}},{"./support":30,"./utils":32}],2:[function(t,e,r){"use strict";var i=t("./external"),n=t("./stream/DataWorker"),s=t("./stream/DataLengthProbe"),a=t("./stream/Crc32Probe");s=t("./stream/DataLengthProbe");function o(t,e,r,i,n){this.compressedSize=t,this.uncompressedSize=e,this.crc32=r,this.compression=i,this.compressedContent=n}o.prototype={getContentWorker:function(){var t=new n(i.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new s("data_length")),e=this;return t.on("end",function(){if(this.streamInfo.data_length!==e.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),t},getCompressedWorker:function(){return new n(i.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(t,e,r){return t.pipe(new a).pipe(new s("uncompressedSize")).pipe(e.compressWorker(r)).pipe(new s("compressedSize")).withStreamInfo("compression",e)},e.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(t,e,r){"use strict";var i=t("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(t){return new i("STORE compression")},uncompressWorker:function(){return new i("STORE decompression")}},r.DEFLATE=t("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(t,e,r){"use strict";var i=t("./utils");var o=function(){for(var t,e=[],r=0;r<256;r++){t=r;for(var i=0;i<8;i++)t=1&t?3988292384^t>>>1:t>>>1;e[r]=t}return e}();e.exports=function(t,e){return void 0!==t&&t.length?"string"!==i.getTypeOf(t)?function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e[a])];return-1^t}(0|e,t,t.length,0):function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e.charCodeAt(a))];return-1^t}(0|e,t,t.length,0):0}},{"./utils":32}],5:[function(t,e,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(t,e,r){"use strict";var i=null;i="undefined"!=typeof Promise?Promise:t("lie"),e.exports={Promise:i}},{lie:37}],7:[function(t,e,r){"use strict";var i="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,n=t("pako"),s=t("./utils"),a=t("./stream/GenericWorker"),o=i?"uint8array":"array";function h(t,e){a.call(this,"FlateWorker/"+t),this._pako=null,this._pakoAction=t,this._pakoOptions=e,this.meta={}}r.magic="\b\0",s.inherits(h,a),h.prototype.processChunk=function(t){this.meta=t.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,t.data),!1)},h.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},h.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},h.prototype._createPako=function(){this._pako=new n[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var e=this;this._pako.onData=function(t){e.push({data:t,meta:e.meta})}},r.compressWorker=function(t){return new h("Deflate",t)},r.uncompressWorker=function(){return new h("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(t,e,r){"use strict";function A(t,e){var r,i="";for(r=0;r<e;r++)i+=String.fromCharCode(255&t),t>>>=8;return i}function i(t,e,r,i,n,s){var a,o,h=t.file,u=t.compression,l=s!==O.utf8encode,f=I.transformTo("string",s(h.name)),d=I.transformTo("string",O.utf8encode(h.name)),c=h.comment,p=I.transformTo("string",s(c)),m=I.transformTo("string",O.utf8encode(c)),_=d.length!==h.name.length,g=m.length!==c.length,b="",v="",y="",w=h.dir,k=h.date,x={crc32:0,compressedSize:0,uncompressedSize:0};e&&!r||(x.crc32=t.crc32,x.compressedSize=t.compressedSize,x.uncompressedSize=t.uncompressedSize);var S=0;e&&(S|=8),l||!_&&!g||(S|=2048);var z=0,C=0;w&&(z|=16),"UNIX"===n?(C=798,z|=function(t,e){var r=t;return t||(r=e?16893:33204),(65535&r)<<16}(h.unixPermissions,w)):(C=20,z|=function(t){return 63&(t||0)}(h.dosPermissions)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v=A(1,1)+A(B(f),4)+d,b+="up"+A(v.length,2)+v),g&&(y=A(1,1)+A(B(p),4)+m,b+="uc"+A(y.length,2)+y);var E="";return E+="\n\0",E+=A(S,2),E+=u.magic,E+=A(a,2),E+=A(o,2),E+=A(x.crc32,4),E+=A(x.compressedSize,4),E+=A(x.uncompressedSize,4),E+=A(f.length,2),E+=A(b.length,2),{fileRecord:R.LOCAL_FILE_HEADER+E+f+b,dirRecord:R.CENTRAL_FILE_HEADER+A(C,2)+E+A(p.length,2)+"\0\0\0\0"+A(z,4)+A(i,4)+f+b+p}}var I=t("../utils"),n=t("../stream/GenericWorker"),O=t("../utf8"),B=t("../crc32"),R=t("../signature");function s(t,e,r,i){n.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=e,this.zipPlatform=r,this.encodeFileName=i,this.streamFiles=t,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}I.inherits(s,n),s.prototype.push=function(t){var e=t.meta.percent||0,r=this.entriesCount,i=this._sources.length;this.accumulate?this.contentBuffer.push(t):(this.bytesWritten+=t.data.length,n.prototype.push.call(this,{data:t.data,meta:{currentFile:this.currentFile,percent:r?(e+100*(r-i-1))/r:100}}))},s.prototype.openedSource=function(t){this.currentSourceOffset=this.bytesWritten,this.currentFile=t.file.name;var e=this.streamFiles&&!t.file.dir;if(e){var r=i(t,e,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},s.prototype.closedSource=function(t){this.accumulate=!1;var e=this.streamFiles&&!t.file.dir,r=i(t,e,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(r.dirRecord),e)this.push({data:function(t){return R.DATA_DESCRIPTOR+A(t.crc32,4)+A(t.compressedSize,4)+A(t.uncompressedSize,4)}(t),meta:{percent:100}});else for(this.push({data:r.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},s.prototype.flush=function(){for(var t=this.bytesWritten,e=0;e<this.dirRecords.length;e++)this.push({data:this.dirRecords[e],meta:{percent:100}});var r=this.bytesWritten-t,i=function(t,e,r,i,n){var s=I.transformTo("string",n(i));return R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+A(t,2)+A(t,2)+A(e,4)+A(r,4)+A(s.length,2)+s}(this.dirRecords.length,r,t,this.zipComment,this.encodeFileName);this.push({data:i,meta:{percent:100}})},s.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},s.prototype.registerPrevious=function(t){this._sources.push(t);var e=this;return t.on("data",function(t){e.processChunk(t)}),t.on("end",function(){e.closedSource(e.previous.streamInfo),e._sources.length?e.prepareNextSource():e.end()}),t.on("error",function(t){e.error(t)}),this},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},s.prototype.error=function(t){var e=this._sources;if(!n.prototype.error.call(this,t))return!1;for(var r=0;r<e.length;r++)try{e[r].error(t)}catch(t){}return!0},s.prototype.lock=function(){n.prototype.lock.call(this);for(var t=this._sources,e=0;e<t.length;e++)t[e].lock()},e.exports=s},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(t,e,r){"use strict";var u=t("../compressions"),i=t("./ZipFileWorker");r.generateWorker=function(t,a,e){var o=new i(a.streamFiles,e,a.platform,a.encodeFileName),h=0;try{t.forEach(function(t,e){h++;var r=function(t,e){var r=t||e,i=u[r];if(!i)throw new Error(r+" is not a valid compression method !");return i}(e.options.compression,a.compression),i=e.options.compressionOptions||a.compressionOptions||{},n=e.dir,s=e.date;e._compressWorker(r,i).withStreamInfo("file",{name:t,dir:n,date:s,comment:e.comment||"",unixPermissions:e.unixPermissions,dosPermissions:e.dosPermissions}).pipe(o)}),o.entriesCount=h}catch(t){o.error(t)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(t,e,r){"use strict";function i(){if(!(this instanceof i))return new i;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files={},this.comment=null,this.root="",this.clone=function(){var t=new i;for(var e in this)"function"!=typeof this[e]&&(t[e]=this[e]);return t}}(i.prototype=t("./object")).loadAsync=t("./load"),i.support=t("./support"),i.defaults=t("./defaults"),i.version="3.2.0",i.loadAsync=function(t,e){return(new i).loadAsync(t,e)},i.external=t("./external"),e.exports=i},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(t,e,r){"use strict";var i=t("./utils"),n=t("./external"),o=t("./utf8"),h=(i=t("./utils"),t("./zipEntries")),s=t("./stream/Crc32Probe"),u=t("./nodejsUtils");function l(i){return new n.Promise(function(t,e){var r=i.decompressed.getContentWorker().pipe(new s);r.on("error",function(t){e(t)}).on("end",function(){r.streamInfo.crc32!==i.decompressed.crc32?e(new Error("Corrupted zip : CRC32 mismatch")):t()}).resume()})}e.exports=function(t,s){var a=this;return s=i.extend(s||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:o.utf8decode}),u.isNode&&u.isStream(t)?n.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):i.prepareContent("the loaded zip file",t,!0,s.optimizedBinaryString,s.base64).then(function(t){var e=new h(s);return e.load(t),e}).then(function(t){var e=[n.Promise.resolve(t)],r=t.files;if(s.checkCRC32)for(var i=0;i<r.length;i++)e.push(l(r[i]));return n.Promise.all(e)}).then(function(t){for(var e=t.shift(),r=e.files,i=0;i<r.length;i++){var n=r[i];a.file(n.fileNameStr,n.decompressed,{binary:!0,optimizedBinaryString:!0,date:n.date,dir:n.dir,comment:n.fileCommentStr.length?n.fileCommentStr:null,unixPermissions:n.unixPermissions,dosPermissions:n.dosPermissions,createFolders:s.createFolders})}return e.zipComment.length&&(a.comment=e.zipComment),a})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(t,e,r){"use strict";var i=t("../utils"),n=t("../stream/GenericWorker");function s(t,e){n.call(this,"Nodejs stream input adapter for "+t),this._upstreamEnded=!1,this._bindStream(e)}i.inherits(s,n),s.prototype._bindStream=function(t){var e=this;(this._stream=t).pause(),t.on("data",function(t){e.push({data:t,meta:{percent:0}})}).on("error",function(t){e.isPaused?this.generatedError=t:e.error(t)}).on("end",function(){e.isPaused?e._upstreamEnded=!0:e.end()})},s.prototype.pause=function(){return!!n.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},e.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(t,e,r){"use strict";var n=t("readable-stream").Readable;function i(t,e,r){n.call(this,e),this._helper=t;var i=this;t.on("data",function(t,e){i.push(t)||i._helper.pause(),r&&r(e)}).on("error",function(t){i.emit("error",t)}).on("end",function(){i.push(null)})}t("../utils").inherits(i,n),i.prototype._read=function(){this._helper.resume()},e.exports=i},{"../utils":32,"readable-stream":16}],14:[function(t,e,r){"use strict";e.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(t,e){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(t,e);if("number"==typeof t)throw new Error('The "data" argument must not be a number');return new Buffer(t,e)},allocBuffer:function(t){if(Buffer.alloc)return Buffer.alloc(t);var e=new Buffer(t);return e.fill(0),e},isBuffer:function(t){return Buffer.isBuffer(t)},isStream:function(t){return t&&"function"==typeof t.on&&"function"==typeof t.pause&&"function"==typeof t.resume}}},{}],15:[function(t,e,r){"use strict";function s(t,e,r){var i,n=u.getTypeOf(e),s=u.extend(r||{},f);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(t=g(t)),s.createFolders&&(i=_(t))&&b.call(this,i,!0);var a="string"===n&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!a),(e instanceof d&&0===e.uncompressedSize||s.dir||!e||0===e.length)&&(s.base64=!1,s.binary=!0,e="",s.compression="STORE",n="string");var o=null;o=e instanceof d||e instanceof l?e:p.isNode&&p.isStream(e)?new m(t,e):u.prepareContent(t,e,s.binary,s.optimizedBinaryString,s.base64);var h=new c(t,o,s);this.files[t]=h}var n=t("./utf8"),u=t("./utils"),l=t("./stream/GenericWorker"),a=t("./stream/StreamHelper"),f=t("./defaults"),d=t("./compressedObject"),c=t("./zipObject"),o=t("./generate"),p=t("./nodejsUtils"),m=t("./nodejs/NodejsStreamInputAdapter"),_=function(t){"/"===t.slice(-1)&&(t=t.substring(0,t.length-1));var e=t.lastIndexOf("/");return 0<e?t.substring(0,e):""},g=function(t){return"/"!==t.slice(-1)&&(t+="/"),t},b=function(t,e){return e=void 0!==e?e:f.createFolders,t=g(t),this.files[t]||s.call(this,t,null,{dir:!0,createFolders:e}),this.files[t]};function h(t){return"[object RegExp]"===Object.prototype.toString.call(t)}var i={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(t){var e,r,i;for(e in this.files)this.files.hasOwnProperty(e)&&(i=this.files[e],(r=e.slice(this.root.length,e.length))&&e.slice(0,this.root.length)===this.root&&t(r,i))},filter:function(r){var i=[];return this.forEach(function(t,e){r(t,e)&&i.push(e)}),i},file:function(t,e,r){if(1!==arguments.length)return t=this.root+t,s.call(this,t,e,r),this;if(h(t)){var i=t;return this.filter(function(t,e){return!e.dir&&i.test(t)})}var n=this.files[this.root+t];return n&&!n.dir?n:null},folder:function(r){if(!r)return this;if(h(r))return this.filter(function(t,e){return e.dir&&r.test(t)});var t=this.root+r,e=b.call(this,t),i=this.clone();return i.root=e.name,i},remove:function(r){r=this.root+r;var t=this.files[r];if(t||("/"!==r.slice(-1)&&(r+="/"),t=this.files[r]),t&&!t.dir)delete this.files[r];else for(var e=this.filter(function(t,e){return e.name.slice(0,r.length)===r}),i=0;i<e.length;i++)delete this.files[e[i].name];return this},generate:function(t){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(t){var e,r={};try{if((r=u.extend(t||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:n.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");u.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var i=r.comment||this.comment||"";e=o.generateWorker(this,r,i)}catch(t){(e=new l("error")).error(t)}return new a(e,r.type||"string",r.mimeType)},generateAsync:function(t,e){return this.generateInternalStream(t).accumulate(e)},generateNodeStream:function(t,e){return(t=t||{}).type||(t.type="nodebuffer"),this.generateInternalStream(t).toNodejsStream(e)}};e.exports=i},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(t,e,r){e.exports=t("stream")},{stream:void 0}],17:[function(t,e,r){"use strict";var i=t("./DataReader");function n(t){i.call(this,t);for(var e=0;e<this.data.length;e++)t[e]=255&t[e]}t("../utils").inherits(n,i),n.prototype.byteAt=function(t){return this.data[this.zero+t]},n.prototype.lastIndexOfSignature=function(t){for(var e=t.charCodeAt(0),r=t.charCodeAt(1),i=t.charCodeAt(2),n=t.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===e&&this.data[s+1]===r&&this.data[s+2]===i&&this.data[s+3]===n)return s-this.zero;return-1},n.prototype.readAndCheckSignature=function(t){var e=t.charCodeAt(0),r=t.charCodeAt(1),i=t.charCodeAt(2),n=t.charCodeAt(3),s=this.readData(4);return e===s[0]&&r===s[1]&&i===s[2]&&n===s[3]},n.prototype.readData=function(t){if(this.checkOffset(t),0===t)return[];var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./DataReader":18}],18:[function(t,e,r){"use strict";var i=t("../utils");function n(t){this.data=t,this.length=t.length,this.index=0,this.zero=0}n.prototype={checkOffset:function(t){this.checkIndex(this.index+t)},checkIndex:function(t){if(this.length<this.zero+t||t<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+t+"). Corrupted zip ?")},setIndex:function(t){this.checkIndex(t),this.index=t},skip:function(t){this.setIndex(this.index+t)},byteAt:function(t){},readInt:function(t){var e,r=0;for(this.checkOffset(t),e=this.index+t-1;e>=this.index;e--)r=(r<<8)+this.byteAt(e);return this.index+=t,r},readString:function(t){return i.transformTo("string",this.readData(t))},readData:function(t){},lastIndexOfSignature:function(t){},readAndCheckSignature:function(t){},readDate:function(){var t=this.readInt(4);return new Date(Date.UTC(1980+(t>>25&127),(t>>21&15)-1,t>>16&31,t>>11&31,t>>5&63,(31&t)<<1))}},e.exports=n},{"../utils":32}],19:[function(t,e,r){"use strict";var i=t("./Uint8ArrayReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.readData=function(t){this.checkOffset(t);var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(t,e,r){"use strict";var i=t("./DataReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.byteAt=function(t){return this.data.charCodeAt(this.zero+t)},n.prototype.lastIndexOfSignature=function(t){return this.data.lastIndexOf(t)-this.zero},n.prototype.readAndCheckSignature=function(t){return t===this.readData(4)},n.prototype.readData=function(t){this.checkOffset(t);var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./DataReader":18}],21:[function(t,e,r){"use strict";var i=t("./ArrayReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.readData=function(t){if(this.checkOffset(t),0===t)return new Uint8Array(0);var e=this.data.subarray(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./ArrayReader":17}],22:[function(t,e,r){"use strict";var i=t("../utils"),n=t("../support"),s=t("./ArrayReader"),a=t("./StringReader"),o=t("./NodeBufferReader"),h=t("./Uint8ArrayReader");e.exports=function(t){var e=i.getTypeOf(t);return i.checkSupport(e),"string"!==e||n.uint8array?"nodebuffer"===e?new o(t):n.uint8array?new h(i.transformTo("uint8array",t)):new s(i.transformTo("array",t)):new a(t)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(t,e,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(t,e,r){"use strict";var i=t("./GenericWorker"),n=t("../utils");function s(t){i.call(this,"ConvertWorker to "+t),this.destType=t}n.inherits(s,i),s.prototype.processChunk=function(t){this.push({data:n.transformTo(this.destType,t.data),meta:t.meta})},e.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(t,e,r){"use strict";var i=t("./GenericWorker"),n=t("../crc32");function s(){i.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}t("../utils").inherits(s,i),s.prototype.processChunk=function(t){this.streamInfo.crc32=n(t.data,this.streamInfo.crc32||0),this.push(t)},e.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(t,e,r){"use strict";var i=t("../utils"),n=t("./GenericWorker");function s(t){n.call(this,"DataLengthProbe for "+t),this.propName=t,this.withStreamInfo(t,0)}i.inherits(s,n),s.prototype.processChunk=function(t){if(t){var e=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=e+t.data.length}n.prototype.processChunk.call(this,t)},e.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(t,e,r){"use strict";var i=t("../utils"),n=t("./GenericWorker");function s(t){n.call(this,"DataWorker");var e=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,t.then(function(t){e.dataIsReady=!0,e.data=t,e.max=t&&t.length||0,e.type=i.getTypeOf(t),e.isPaused||e._tickAndRepeat()},function(t){e.error(t)})}i.inherits(s,n),s.prototype.cleanUp=function(){n.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,i.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(i.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var t=null,e=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":t=this.data.substring(this.index,e);break;case"uint8array":t=this.data.subarray(this.index,e);break;case"array":case"nodebuffer":t=this.data.slice(this.index,e)}return this.index=e,this.push({data:t,meta:{percent:this.max?this.index/this.max*100:0}})},e.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(t,e,r){"use strict";function i(t){this.name=t||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}i.prototype={push:function(t){this.emit("data",t)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(t){this.emit("error",t)}return!0},error:function(t){return!this.isFinished&&(this.isPaused?this.generatedError=t:(this.isFinished=!0,this.emit("error",t),this.previous&&this.previous.error(t),this.cleanUp()),!0)},on:function(t,e){return this._listeners[t].push(e),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(t,e){if(this._listeners[t])for(var r=0;r<this._listeners[t].length;r++)this._listeners[t][r].call(this,e)},pipe:function(t){return t.registerPrevious(this)},registerPrevious:function(t){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=t.streamInfo,this.mergeStreamInfo(),this.previous=t;var e=this;return t.on("data",function(t){e.processChunk(t)}),t.on("end",function(){e.end()}),t.on("error",function(t){e.error(t)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var t=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),t=!0),this.previous&&this.previous.resume(),!t},flush:function(){},processChunk:function(t){this.push(t)},withStreamInfo:function(t,e){return this.extraStreamInfo[t]=e,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var t in this.extraStreamInfo)this.extraStreamInfo.hasOwnProperty(t)&&(this.streamInfo[t]=this.extraStreamInfo[t])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var t="Worker "+this.name;return this.previous?this.previous+" -> "+t:t}},e.exports=i},{}],29:[function(t,e,r){"use strict";var h=t("../utils"),n=t("./ConvertWorker"),s=t("./GenericWorker"),u=t("../base64"),i=t("../support"),a=t("../external"),o=null;if(i.nodestream)try{o=t("../nodejs/NodejsStreamOutputAdapter")}catch(t){}function l(t,o){return new a.Promise(function(e,r){var i=[],n=t._internalType,s=t._outputType,a=t._mimeType;t.on("data",function(t,e){i.push(t),o&&o(e)}).on("error",function(t){i=[],r(t)}).on("end",function(){try{var t=function(t,e,r){switch(t){case"blob":return h.newBlob(h.transformTo("arraybuffer",e),r);case"base64":return u.encode(e);default:return h.transformTo(t,e)}}(s,function(t,e){var r,i=0,n=null,s=0;for(r=0;r<e.length;r++)s+=e[r].length;switch(t){case"string":return e.join("");case"array":return Array.prototype.concat.apply([],e);case"uint8array":for(n=new Uint8Array(s),r=0;r<e.length;r++)n.set(e[r],i),i+=e[r].length;return n;case"nodebuffer":return Buffer.concat(e);default:throw new Error("concat : unsupported type '"+t+"'")}}(n,i),a);e(t)}catch(t){r(t)}i=[]}).resume()})}function f(t,e,r){var i=e;switch(e){case"blob":case"arraybuffer":i="uint8array";break;case"base64":i="string"}try{this._internalType=i,this._outputType=e,this._mimeType=r,h.checkSupport(i),this._worker=t.pipe(new n(i)),t.lock()}catch(t){this._worker=new s("error"),this._worker.error(t)}}f.prototype={accumulate:function(t){return l(this,t)},on:function(t,e){var r=this;return"data"===t?this._worker.on(t,function(t){e.call(r,t.data,t.meta)}):this._worker.on(t,function(){h.delay(e,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(t){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},t)}},e.exports=f},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(t,e,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var i=new ArrayBuffer(0);try{r.blob=0===new Blob([i],{type:"application/zip"}).size}catch(t){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);n.append(i),r.blob=0===n.getBlob("application/zip").size}catch(t){r.blob=!1}}}try{r.nodestream=!!t("readable-stream").Readable}catch(t){r.nodestream=!1}},{"readable-stream":16}],31:[function(t,e,s){"use strict";for(var o=t("./utils"),h=t("./support"),r=t("./nodejsUtils"),i=t("./stream/GenericWorker"),u=new Array(256),n=0;n<256;n++)u[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;u[254]=u[254]=1;function a(){i.call(this,"utf-8 decode"),this.leftOver=null}function l(){i.call(this,"utf-8 encode")}s.utf8encode=function(t){return h.nodebuffer?r.newBufferFrom(t,"utf-8"):function(t){var e,r,i,n,s,a=t.length,o=0;for(n=0;n<a;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),o+=r<128?1:r<2048?2:r<65536?3:4;for(e=h.uint8array?new Uint8Array(o):new Array(o),n=s=0;s<o;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),r<128?e[s++]=r:(r<2048?e[s++]=192|r>>>6:(r<65536?e[s++]=224|r>>>12:(e[s++]=240|r>>>18,e[s++]=128|r>>>12&63),e[s++]=128|r>>>6&63),e[s++]=128|63&r);return e}(t)},s.utf8decode=function(t){return h.nodebuffer?o.transformTo("nodebuffer",t).toString("utf-8"):function(t){var e,r,i,n,s=t.length,a=new Array(2*s);for(e=r=0;e<s;)if((i=t[e++])<128)a[r++]=i;else if(4<(n=u[i]))a[r++]=65533,e+=n-1;else{for(i&=2===n?31:3===n?15:7;1<n&&e<s;)i=i<<6|63&t[e++],n--;1<n?a[r++]=65533:i<65536?a[r++]=i:(i-=65536,a[r++]=55296|i>>10&1023,a[r++]=56320|1023&i)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(t=o.transformTo(h.uint8array?"uint8array":"array",t))},o.inherits(a,i),a.prototype.processChunk=function(t){var e=o.transformTo(h.uint8array?"uint8array":"array",t.data);if(this.leftOver&&this.leftOver.length){if(h.uint8array){var r=e;(e=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),e.set(r,this.leftOver.length)}else e=this.leftOver.concat(e);this.leftOver=null}var i=function(t,e){var r;for((e=e||t.length)>t.length&&(e=t.length),r=e-1;0<=r&&128==(192&t[r]);)r--;return r<0?e:0===r?e:r+u[t[r]]>e?r:e}(e),n=e;i!==e.length&&(h.uint8array?(n=e.subarray(0,i),this.leftOver=e.subarray(i,e.length)):(n=e.slice(0,i),this.leftOver=e.slice(i,e.length))),this.push({data:s.utf8decode(n),meta:t.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(l,i),l.prototype.processChunk=function(t){this.push({data:s.utf8encode(t.data),meta:t.meta})},s.Utf8EncodeWorker=l},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(t,e,a){"use strict";var o=t("./support"),h=t("./base64"),r=t("./nodejsUtils"),i=t("set-immediate-shim"),u=t("./external");function n(t){return t}function l(t,e){for(var r=0;r<t.length;++r)e[r]=255&t.charCodeAt(r);return e}a.newBlob=function(e,r){a.checkSupport("blob");try{return new Blob([e],{type:r})}catch(t){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return i.append(e),i.getBlob(r)}catch(t){throw new Error("Bug : can't construct the Blob.")}}};var s={stringifyByChunk:function(t,e,r){var i=[],n=0,s=t.length;if(s<=r)return String.fromCharCode.apply(null,t);for(;n<s;)"array"===e||"nodebuffer"===e?i.push(String.fromCharCode.apply(null,t.slice(n,Math.min(n+r,s)))):i.push(String.fromCharCode.apply(null,t.subarray(n,Math.min(n+r,s)))),n+=r;return i.join("")},stringifyByChar:function(t){for(var e="",r=0;r<t.length;r++)e+=String.fromCharCode(t[r]);return e},applyCanBeUsed:{uint8array:function(){try{return o.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(t){return!1}}(),nodebuffer:function(){try{return o.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(t){return!1}}()}};function f(t){var e=65536,r=a.getTypeOf(t),i=!0;if("uint8array"===r?i=s.applyCanBeUsed.uint8array:"nodebuffer"===r&&(i=s.applyCanBeUsed.nodebuffer),i)for(;1<e;)try{return s.stringifyByChunk(t,r,e)}catch(t){e=Math.floor(e/2)}return s.stringifyByChar(t)}function d(t,e){for(var r=0;r<t.length;r++)e[r]=t[r];return e}a.applyFromCharCode=f;var c={};c.string={string:n,array:function(t){return l(t,new Array(t.length))},arraybuffer:function(t){return c.string.uint8array(t).buffer},uint8array:function(t){return l(t,new Uint8Array(t.length))},nodebuffer:function(t){return l(t,r.allocBuffer(t.length))}},c.array={string:f,array:n,arraybuffer:function(t){return new Uint8Array(t).buffer},uint8array:function(t){return new Uint8Array(t)},nodebuffer:function(t){return r.newBufferFrom(t)}},c.arraybuffer={string:function(t){return f(new Uint8Array(t))},array:function(t){return d(new Uint8Array(t),new Array(t.byteLength))},arraybuffer:n,uint8array:function(t){return new Uint8Array(t)},nodebuffer:function(t){return r.newBufferFrom(new Uint8Array(t))}},c.uint8array={string:f,array:function(t){return d(t,new Array(t.length))},arraybuffer:function(t){return t.buffer},uint8array:n,nodebuffer:function(t){return r.newBufferFrom(t)}},c.nodebuffer={string:f,array:function(t){return d(t,new Array(t.length))},arraybuffer:function(t){return c.nodebuffer.uint8array(t).buffer},uint8array:function(t){return d(t,new Uint8Array(t.length))},nodebuffer:n},a.transformTo=function(t,e){if(e=e||"",!t)return e;a.checkSupport(t);var r=a.getTypeOf(e);return c[r][t](e)},a.getTypeOf=function(t){return"string"==typeof t?"string":"[object Array]"===Object.prototype.toString.call(t)?"array":o.nodebuffer&&r.isBuffer(t)?"nodebuffer":o.uint8array&&t instanceof Uint8Array?"uint8array":o.arraybuffer&&t instanceof ArrayBuffer?"arraybuffer":void 0},a.checkSupport=function(t){if(!o[t.toLowerCase()])throw new Error(t+" is not supported by this platform")},a.MAX_VALUE_16BITS=65535,a.MAX_VALUE_32BITS=-1,a.pretty=function(t){var e,r,i="";for(r=0;r<(t||"").length;r++)i+="\\x"+((e=t.charCodeAt(r))<16?"0":"")+e.toString(16).toUpperCase();return i},a.delay=function(t,e,r){i(function(){t.apply(r||null,e||[])})},a.inherits=function(t,e){function r(){}r.prototype=e.prototype,t.prototype=new r},a.extend=function(){var t,e,r={};for(t=0;t<arguments.length;t++)for(e in arguments[t])arguments[t].hasOwnProperty(e)&&void 0===r[e]&&(r[e]=arguments[t][e]);return r},a.prepareContent=function(r,t,i,n,s){return u.Promise.resolve(t).then(function(i){return o.blob&&(i instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(i)))&&"undefined"!=typeof FileReader?new u.Promise(function(e,r){var t=new FileReader;t.onload=function(t){e(t.target.result)},t.onerror=function(t){r(t.target.error)},t.readAsArrayBuffer(i)}):i}).then(function(t){var e=a.getTypeOf(t);return e?("arraybuffer"===e?t=a.transformTo("uint8array",t):"string"===e&&(s?t=h.decode(t):i&&!0!==n&&(t=function(t){return l(t,o.uint8array?new Uint8Array(t.length):new Array(t.length))}(t))),t):u.Promise.reject(new Error("Can't read the data of '"+r+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,"set-immediate-shim":54}],33:[function(t,e,r){"use strict";var i=t("./reader/readerFor"),n=t("./utils"),s=t("./signature"),a=t("./zipEntry"),o=(t("./utf8"),t("./support"));function h(t){this.files=[],this.loadOptions=t}h.prototype={checkSignature:function(t){if(!this.reader.readAndCheckSignature(t)){this.reader.index-=4;var e=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+n.pretty(e)+", expected "+n.pretty(t)+")")}},isSignature:function(t,e){var r=this.reader.index;this.reader.setIndex(t);var i=this.reader.readString(4)===e;return this.reader.setIndex(r),i},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var t=this.reader.readData(this.zipCommentLength),e=o.uint8array?"uint8array":"array",r=n.transformTo(e,t);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var t,e,r,i=this.zip64EndOfCentralSize-44;0<i;)t=this.reader.readInt(2),e=this.reader.readInt(4),r=this.reader.readData(e),this.zip64ExtensibleData[t]={id:t,length:e,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var t,e;for(t=0;t<this.files.length;t++)e=this.files[t],this.reader.setIndex(e.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),e.readLocalPart(this.reader),e.handleUTF8(),e.processAttributes()},readCentralDir:function(){var t;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(t=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(t);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var t=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(t<0)throw!this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html"):new Error("Corrupted zip: can't find end of central directory");this.reader.setIndex(t);var e=t;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===n.MAX_VALUE_16BITS||this.diskWithCentralDirStart===n.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===n.MAX_VALUE_16BITS||this.centralDirRecords===n.MAX_VALUE_16BITS||this.centralDirSize===n.MAX_VALUE_32BITS||this.centralDirOffset===n.MAX_VALUE_32BITS){if(this.zip64=!0,(t=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(t),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var i=e-r;if(0<i)this.isSignature(e,s.CENTRAL_FILE_HEADER)||(this.reader.zero=i);else if(i<0)throw new Error("Corrupted zip: missing "+Math.abs(i)+" bytes.")},prepareReader:function(t){this.reader=i(t)},load:function(t){this.prepareReader(t),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},e.exports=h},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utf8":31,"./utils":32,"./zipEntry":34}],34:[function(t,e,r){"use strict";var i=t("./reader/readerFor"),s=t("./utils"),n=t("./compressedObject"),a=t("./crc32"),o=t("./utf8"),h=t("./compressions"),u=t("./support");function l(t,e){this.options=t,this.loadOptions=e}l.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(t){var e,r;if(t.skip(22),this.fileNameLength=t.readInt(2),r=t.readInt(2),this.fileName=t.readData(this.fileNameLength),t.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough informations from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(e=function(t){for(var e in h)if(h.hasOwnProperty(e)&&h[e].magic===t)return h[e];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new n(this.compressedSize,this.uncompressedSize,this.crc32,e,t.readData(this.compressedSize))},readCentralPart:function(t){this.versionMadeBy=t.readInt(2),t.skip(2),this.bitFlag=t.readInt(2),this.compressionMethod=t.readString(2),this.date=t.readDate(),this.crc32=t.readInt(4),this.compressedSize=t.readInt(4),this.uncompressedSize=t.readInt(4);var e=t.readInt(2);if(this.extraFieldsLength=t.readInt(2),this.fileCommentLength=t.readInt(2),this.diskNumberStart=t.readInt(2),this.internalFileAttributes=t.readInt(2),this.externalFileAttributes=t.readInt(4),this.localHeaderOffset=t.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");t.skip(e),this.readExtraFields(t),this.parseZIP64ExtraField(t),this.fileComment=t.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var t=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==t&&(this.dosPermissions=63&this.externalFileAttributes),3==t&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(t){if(this.extraFields[1]){var e=i(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=e.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=e.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=e.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=e.readInt(4))}},readExtraFields:function(t){var e,r,i,n=t.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});t.index<n;)e=t.readInt(2),r=t.readInt(2),i=t.readData(r),this.extraFields[e]={id:e,length:r,value:i}},handleUTF8:function(){var t=u.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var e=this.findExtraFieldUnicodePath();if(null!==e)this.fileNameStr=e;else{var r=s.transformTo(t,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var i=this.findExtraFieldUnicodeComment();if(null!==i)this.fileCommentStr=i;else{var n=s.transformTo(t,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(n)}}},findExtraFieldUnicodePath:function(){var t=this.extraFields[28789];if(t){var e=i(t.value);return 1!==e.readInt(1)?null:a(this.fileName)!==e.readInt(4)?null:o.utf8decode(e.readData(t.length-5))}return null},findExtraFieldUnicodeComment:function(){var t=this.extraFields[25461];if(t){var e=i(t.value);return 1!==e.readInt(1)?null:a(this.fileComment)!==e.readInt(4)?null:o.utf8decode(e.readData(t.length-5))}return null}},e.exports=l},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(t,e,r){"use strict";function i(t,e,r){this.name=t,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=e,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=t("./stream/StreamHelper"),n=t("./stream/DataWorker"),a=t("./utf8"),o=t("./compressedObject"),h=t("./stream/GenericWorker");i.prototype={internalStream:function(t){var e=null,r="string";try{if(!t)throw new Error("No output type specified.");var i="string"===(r=t.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),e=this._decompressWorker();var n=!this._dataBinary;n&&!i&&(e=e.pipe(new a.Utf8EncodeWorker)),!n&&i&&(e=e.pipe(new a.Utf8DecodeWorker))}catch(t){(e=new h("error")).error(t)}return new s(e,r,"")},async:function(t,e){return this.internalStream(t).accumulate(e)},nodeStream:function(t,e){return this.internalStream(t||"nodebuffer").toNodejsStream(e)},_compressWorker:function(t,e){if(this._data instanceof o&&this._data.compression.magic===t.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,t,e)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof h?this._data:new n(this._data)}};for(var u=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],l=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},f=0;f<u.length;f++)i.prototype[u[f]]=l;e.exports=i},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(t,l,e){(function(e){"use strict";var r,i,t=e.MutationObserver||e.WebKitMutationObserver;if(t){var n=0,s=new t(u),a=e.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=n=++n%2}}else if(e.setImmediate||void 0===e.MessageChannel)r="document"in e&&"onreadystatechange"in e.document.createElement("script")?function(){var t=e.document.createElement("script");t.onreadystatechange=function(){u(),t.onreadystatechange=null,t.parentNode.removeChild(t),t=null},e.document.documentElement.appendChild(t)}:function(){setTimeout(u,0)};else{var o=new e.MessageChannel;o.port1.onmessage=u,r=function(){o.port2.postMessage(0)}}var h=[];function u(){var t,e;i=!0;for(var r=h.length;r;){for(e=h,h=[],t=-1;++t<r;)e[t]();r=h.length}i=!1}l.exports=function(t){1!==h.push(t)||i||r()}}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(t,e,r){"use strict";var n=t("immediate");function u(){}var l={},s=["REJECTED"],a=["FULFILLED"],i=["PENDING"];function o(t){if("function"!=typeof t)throw new TypeError("resolver must be a function");this.state=i,this.queue=[],this.outcome=void 0,t!==u&&c(this,t)}function h(t,e,r){this.promise=t,"function"==typeof e&&(this.onFulfilled=e,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function f(e,r,i){n(function(){var t;try{t=r(i)}catch(t){return l.reject(e,t)}t===e?l.reject(e,new TypeError("Cannot resolve promise with itself")):l.resolve(e,t)})}function d(t){var e=t&&t.then;if(t&&("object"==typeof t||"function"==typeof t)&&"function"==typeof e)return function(){e.apply(t,arguments)}}function c(e,t){var r=!1;function i(t){r||(r=!0,l.reject(e,t))}function n(t){r||(r=!0,l.resolve(e,t))}var s=p(function(){t(n,i)});"error"===s.status&&i(s.value)}function p(t,e){var r={};try{r.value=t(e),r.status="success"}catch(t){r.status="error",r.value=t}return r}(e.exports=o).prototype.finally=function(e){if("function"!=typeof e)return this;var r=this.constructor;return this.then(function(t){return r.resolve(e()).then(function(){return t})},function(t){return r.resolve(e()).then(function(){throw t})})},o.prototype.catch=function(t){return this.then(null,t)},o.prototype.then=function(t,e){if("function"!=typeof t&&this.state===a||"function"!=typeof e&&this.state===s)return this;var r=new this.constructor(u);this.state!==i?f(r,this.state===a?t:e,this.outcome):this.queue.push(new h(r,t,e));return r},h.prototype.callFulfilled=function(t){l.resolve(this.promise,t)},h.prototype.otherCallFulfilled=function(t){f(this.promise,this.onFulfilled,t)},h.prototype.callRejected=function(t){l.reject(this.promise,t)},h.prototype.otherCallRejected=function(t){f(this.promise,this.onRejected,t)},l.resolve=function(t,e){var r=p(d,e);if("error"===r.status)return l.reject(t,r.value);var i=r.value;if(i)c(t,i);else{t.state=a,t.outcome=e;for(var n=-1,s=t.queue.length;++n<s;)t.queue[n].callFulfilled(e)}return t},l.reject=function(t,e){t.state=s,t.outcome=e;for(var r=-1,i=t.queue.length;++r<i;)t.queue[r].callRejected(e);return t},o.resolve=function(t){if(t instanceof this)return t;return l.resolve(new this(u),t)},o.reject=function(t){var e=new this(u);return l.reject(e,t)},o.all=function(t){var r=this;if("[object Array]"!==Object.prototype.toString.call(t))return this.reject(new TypeError("must be an array"));var i=t.length,n=!1;if(!i)return this.resolve([]);var s=new Array(i),a=0,e=-1,o=new this(u);for(;++e<i;)h(t[e],e);return o;function h(t,e){r.resolve(t).then(function(t){s[e]=t,++a!==i||n||(n=!0,l.resolve(o,s))},function(t){n||(n=!0,l.reject(o,t))})}},o.race=function(t){var e=this;if("[object Array]"!==Object.prototype.toString.call(t))return this.reject(new TypeError("must be an array"));var r=t.length,i=!1;if(!r)return this.resolve([]);var n=-1,s=new this(u);for(;++n<r;)a=t[n],e.resolve(a).then(function(t){i||(i=!0,l.resolve(s,t))},function(t){i||(i=!0,l.reject(s,t))});var a;return s}},{immediate:36}],38:[function(t,e,r){"use strict";var i={};(0,t("./lib/utils/common").assign)(i,t("./lib/deflate"),t("./lib/inflate"),t("./lib/zlib/constants")),e.exports=i},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(t,e,r){"use strict";var a=t("./zlib/deflate"),o=t("./utils/common"),h=t("./utils/strings"),n=t("./zlib/messages"),s=t("./zlib/zstream"),u=Object.prototype.toString,l=0,f=-1,d=0,c=8;function p(t){if(!(this instanceof p))return new p(t);this.options=o.assign({level:f,method:c,chunkSize:16384,windowBits:15,memLevel:8,strategy:d,to:""},t||{});var e=this.options;e.raw&&0<e.windowBits?e.windowBits=-e.windowBits:e.gzip&&0<e.windowBits&&e.windowBits<16&&(e.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,e.level,e.method,e.windowBits,e.memLevel,e.strategy);if(r!==l)throw new Error(n[r]);if(e.header&&a.deflateSetHeader(this.strm,e.header),e.dictionary){var i;if(i="string"==typeof e.dictionary?h.string2buf(e.dictionary):"[object ArrayBuffer]"===u.call(e.dictionary)?new Uint8Array(e.dictionary):e.dictionary,(r=a.deflateSetDictionary(this.strm,i))!==l)throw new Error(n[r]);this._dict_set=!0}}function i(t,e){var r=new p(e);if(r.push(t,!0),r.err)throw r.msg||n[r.err];return r.result}p.prototype.push=function(t,e){var r,i,n=this.strm,s=this.options.chunkSize;if(this.ended)return!1;i=e===~~e?e:!0===e?4:0,"string"==typeof t?n.input=h.string2buf(t):"[object ArrayBuffer]"===u.call(t)?n.input=new Uint8Array(t):n.input=t,n.next_in=0,n.avail_in=n.input.length;do{if(0===n.avail_out&&(n.output=new o.Buf8(s),n.next_out=0,n.avail_out=s),1!==(r=a.deflate(n,i))&&r!==l)return this.onEnd(r),!(this.ended=!0);0!==n.avail_out&&(0!==n.avail_in||4!==i&&2!==i)||("string"===this.options.to?this.onData(h.buf2binstring(o.shrinkBuf(n.output,n.next_out))):this.onData(o.shrinkBuf(n.output,n.next_out)))}while((0<n.avail_in||0===n.avail_out)&&1!==r);return 4===i?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===l):2!==i||(this.onEnd(l),!(n.avail_out=0))},p.prototype.onData=function(t){this.chunks.push(t)},p.prototype.onEnd=function(t){t===l&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=t,this.msg=this.strm.msg},r.Deflate=p,r.deflate=i,r.deflateRaw=function(t,e){return(e=e||{}).raw=!0,i(t,e)},r.gzip=function(t,e){return(e=e||{}).gzip=!0,i(t,e)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(t,e,r){"use strict";var d=t("./zlib/inflate"),c=t("./utils/common"),p=t("./utils/strings"),m=t("./zlib/constants"),i=t("./zlib/messages"),n=t("./zlib/zstream"),s=t("./zlib/gzheader"),_=Object.prototype.toString;function a(t){if(!(this instanceof a))return new a(t);this.options=c.assign({chunkSize:16384,windowBits:0,to:""},t||{});var e=this.options;e.raw&&0<=e.windowBits&&e.windowBits<16&&(e.windowBits=-e.windowBits,0===e.windowBits&&(e.windowBits=-15)),!(0<=e.windowBits&&e.windowBits<16)||t&&t.windowBits||(e.windowBits+=32),15<e.windowBits&&e.windowBits<48&&0==(15&e.windowBits)&&(e.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new n,this.strm.avail_out=0;var r=d.inflateInit2(this.strm,e.windowBits);if(r!==m.Z_OK)throw new Error(i[r]);this.header=new s,d.inflateGetHeader(this.strm,this.header)}function o(t,e){var r=new a(e);if(r.push(t,!0),r.err)throw r.msg||i[r.err];return r.result}a.prototype.push=function(t,e){var r,i,n,s,a,o,h=this.strm,u=this.options.chunkSize,l=this.options.dictionary,f=!1;if(this.ended)return!1;i=e===~~e?e:!0===e?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof t?h.input=p.binstring2buf(t):"[object ArrayBuffer]"===_.call(t)?h.input=new Uint8Array(t):h.input=t,h.next_in=0,h.avail_in=h.input.length;do{if(0===h.avail_out&&(h.output=new c.Buf8(u),h.next_out=0,h.avail_out=u),(r=d.inflate(h,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&l&&(o="string"==typeof l?p.string2buf(l):"[object ArrayBuffer]"===_.call(l)?new Uint8Array(l):l,r=d.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===f&&(r=m.Z_OK,f=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);h.next_out&&(0!==h.avail_out&&r!==m.Z_STREAM_END&&(0!==h.avail_in||i!==m.Z_FINISH&&i!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(n=p.utf8border(h.output,h.next_out),s=h.next_out-n,a=p.buf2string(h.output,n),h.next_out=s,h.avail_out=u-s,s&&c.arraySet(h.output,h.output,n,s,0),this.onData(a)):this.onData(c.shrinkBuf(h.output,h.next_out)))),0===h.avail_in&&0===h.avail_out&&(f=!0)}while((0<h.avail_in||0===h.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(i=m.Z_FINISH),i===m.Z_FINISH?(r=d.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):i!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(h.avail_out=0))},a.prototype.onData=function(t){this.chunks.push(t)},a.prototype.onEnd=function(t){t===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=c.flattenChunks(this.chunks)),this.chunks=[],this.err=t,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(t,e){return(e=e||{}).raw=!0,o(t,e)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(t,e,r){"use strict";var i="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(t){for(var e=Array.prototype.slice.call(arguments,1);e.length;){var r=e.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var i in r)r.hasOwnProperty(i)&&(t[i]=r[i])}}return t},r.shrinkBuf=function(t,e){return t.length===e?t:t.subarray?t.subarray(0,e):(t.length=e,t)};var n={arraySet:function(t,e,r,i,n){if(e.subarray&&t.subarray)t.set(e.subarray(r,r+i),n);else for(var s=0;s<i;s++)t[n+s]=e[r+s]},flattenChunks:function(t){var e,r,i,n,s,a;for(e=i=0,r=t.length;e<r;e++)i+=t[e].length;for(a=new Uint8Array(i),e=n=0,r=t.length;e<r;e++)s=t[e],a.set(s,n),n+=s.length;return a}},s={arraySet:function(t,e,r,i,n){for(var s=0;s<i;s++)t[n+s]=e[r+s]},flattenChunks:function(t){return[].concat.apply([],t)}};r.setTyped=function(t){t?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,n)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(i)},{}],42:[function(t,e,r){"use strict";var h=t("./common"),n=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(t){n=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(t){s=!1}for(var u=new h.Buf8(256),i=0;i<256;i++)u[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;function l(t,e){if(e<65537&&(t.subarray&&s||!t.subarray&&n))return String.fromCharCode.apply(null,h.shrinkBuf(t,e));for(var r="",i=0;i<e;i++)r+=String.fromCharCode(t[i]);return r}u[254]=u[254]=1,r.string2buf=function(t){var e,r,i,n,s,a=t.length,o=0;for(n=0;n<a;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),o+=r<128?1:r<2048?2:r<65536?3:4;for(e=new h.Buf8(o),n=s=0;s<o;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),r<128?e[s++]=r:(r<2048?e[s++]=192|r>>>6:(r<65536?e[s++]=224|r>>>12:(e[s++]=240|r>>>18,e[s++]=128|r>>>12&63),e[s++]=128|r>>>6&63),e[s++]=128|63&r);return e},r.buf2binstring=function(t){return l(t,t.length)},r.binstring2buf=function(t){for(var e=new h.Buf8(t.length),r=0,i=e.length;r<i;r++)e[r]=t.charCodeAt(r);return e},r.buf2string=function(t,e){var r,i,n,s,a=e||t.length,o=new Array(2*a);for(r=i=0;r<a;)if((n=t[r++])<128)o[i++]=n;else if(4<(s=u[n]))o[i++]=65533,r+=s-1;else{for(n&=2===s?31:3===s?15:7;1<s&&r<a;)n=n<<6|63&t[r++],s--;1<s?o[i++]=65533:n<65536?o[i++]=n:(n-=65536,o[i++]=55296|n>>10&1023,o[i++]=56320|1023&n)}return l(o,i)},r.utf8border=function(t,e){var r;for((e=e||t.length)>t.length&&(e=t.length),r=e-1;0<=r&&128==(192&t[r]);)r--;return r<0?e:0===r?e:r+u[t[r]]>e?r:e}},{"./common":41}],43:[function(t,e,r){"use strict";e.exports=function(t,e,r,i){for(var n=65535&t|0,s=t>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(n=n+e[i++]|0)|0,--a;);n%=65521,s%=65521}return n|s<<16|0}},{}],44:[function(t,e,r){"use strict";e.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(t,e,r){"use strict";var o=function(){for(var t,e=[],r=0;r<256;r++){t=r;for(var i=0;i<8;i++)t=1&t?3988292384^t>>>1:t>>>1;e[r]=t}return e}();e.exports=function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e[a])];return-1^t}},{}],46:[function(t,e,r){"use strict";var h,d=t("../utils/common"),u=t("./trees"),c=t("./adler32"),p=t("./crc32"),i=t("./messages"),l=0,f=4,m=0,_=-2,g=-1,b=4,n=2,v=8,y=9,s=286,a=30,o=19,w=2*s+1,k=15,x=3,S=258,z=S+x+1,C=42,E=113,A=1,I=2,O=3,B=4;function R(t,e){return t.msg=i[e],e}function T(t){return(t<<1)-(4<t?9:0)}function D(t){for(var e=t.length;0<=--e;)t[e]=0}function F(t){var e=t.state,r=e.pending;r>t.avail_out&&(r=t.avail_out),0!==r&&(d.arraySet(t.output,e.pending_buf,e.pending_out,r,t.next_out),t.next_out+=r,e.pending_out+=r,t.total_out+=r,t.avail_out-=r,e.pending-=r,0===e.pending&&(e.pending_out=0))}function N(t,e){u._tr_flush_block(t,0<=t.block_start?t.block_start:-1,t.strstart-t.block_start,e),t.block_start=t.strstart,F(t.strm)}function U(t,e){t.pending_buf[t.pending++]=e}function P(t,e){t.pending_buf[t.pending++]=e>>>8&255,t.pending_buf[t.pending++]=255&e}function L(t,e){var r,i,n=t.max_chain_length,s=t.strstart,a=t.prev_length,o=t.nice_match,h=t.strstart>t.w_size-z?t.strstart-(t.w_size-z):0,u=t.window,l=t.w_mask,f=t.prev,d=t.strstart+S,c=u[s+a-1],p=u[s+a];t.prev_length>=t.good_match&&(n>>=2),o>t.lookahead&&(o=t.lookahead);do{if(u[(r=e)+a]===p&&u[r+a-1]===c&&u[r]===u[s]&&u[++r]===u[s+1]){s+=2,r++;do{}while(u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&s<d);if(i=S-(d-s),s=d-S,a<i){if(t.match_start=e,o<=(a=i))break;c=u[s+a-1],p=u[s+a]}}}while((e=f[e&l])>h&&0!=--n);return a<=t.lookahead?a:t.lookahead}function j(t){var e,r,i,n,s,a,o,h,u,l,f=t.w_size;do{if(n=t.window_size-t.lookahead-t.strstart,t.strstart>=f+(f-z)){for(d.arraySet(t.window,t.window,f,f,0),t.match_start-=f,t.strstart-=f,t.block_start-=f,e=r=t.hash_size;i=t.head[--e],t.head[e]=f<=i?i-f:0,--r;);for(e=r=f;i=t.prev[--e],t.prev[e]=f<=i?i-f:0,--r;);n+=f}if(0===t.strm.avail_in)break;if(a=t.strm,o=t.window,h=t.strstart+t.lookahead,u=n,l=void 0,l=a.avail_in,u<l&&(l=u),r=0===l?0:(a.avail_in-=l,d.arraySet(o,a.input,a.next_in,l,h),1===a.state.wrap?a.adler=c(a.adler,o,l,h):2===a.state.wrap&&(a.adler=p(a.adler,o,l,h)),a.next_in+=l,a.total_in+=l,l),t.lookahead+=r,t.lookahead+t.insert>=x)for(s=t.strstart-t.insert,t.ins_h=t.window[s],t.ins_h=(t.ins_h<<t.hash_shift^t.window[s+1])&t.hash_mask;t.insert&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[s+x-1])&t.hash_mask,t.prev[s&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=s,s++,t.insert--,!(t.lookahead+t.insert<x)););}while(t.lookahead<z&&0!==t.strm.avail_in)}function Z(t,e){for(var r,i;;){if(t.lookahead<z){if(j(t),t.lookahead<z&&e===l)return A;if(0===t.lookahead)break}if(r=0,t.lookahead>=x&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!==r&&t.strstart-r<=t.w_size-z&&(t.match_length=L(t,r)),t.match_length>=x)if(i=u._tr_tally(t,t.strstart-t.match_start,t.match_length-x),t.lookahead-=t.match_length,t.match_length<=t.max_lazy_match&&t.lookahead>=x){for(t.match_length--;t.strstart++,t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart,0!=--t.match_length;);t.strstart++}else t.strstart+=t.match_length,t.match_length=0,t.ins_h=t.window[t.strstart],t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+1])&t.hash_mask;else i=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++;if(i&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=t.strstart<x-1?t.strstart:x-1,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}function W(t,e){for(var r,i,n;;){if(t.lookahead<z){if(j(t),t.lookahead<z&&e===l)return A;if(0===t.lookahead)break}if(r=0,t.lookahead>=x&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),t.prev_length=t.match_length,t.prev_match=t.match_start,t.match_length=x-1,0!==r&&t.prev_length<t.max_lazy_match&&t.strstart-r<=t.w_size-z&&(t.match_length=L(t,r),t.match_length<=5&&(1===t.strategy||t.match_length===x&&4096<t.strstart-t.match_start)&&(t.match_length=x-1)),t.prev_length>=x&&t.match_length<=t.prev_length){for(n=t.strstart+t.lookahead-x,i=u._tr_tally(t,t.strstart-1-t.prev_match,t.prev_length-x),t.lookahead-=t.prev_length-1,t.prev_length-=2;++t.strstart<=n&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!=--t.prev_length;);if(t.match_available=0,t.match_length=x-1,t.strstart++,i&&(N(t,!1),0===t.strm.avail_out))return A}else if(t.match_available){if((i=u._tr_tally(t,0,t.window[t.strstart-1]))&&N(t,!1),t.strstart++,t.lookahead--,0===t.strm.avail_out)return A}else t.match_available=1,t.strstart++,t.lookahead--}return t.match_available&&(i=u._tr_tally(t,0,t.window[t.strstart-1]),t.match_available=0),t.insert=t.strstart<x-1?t.strstart:x-1,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}function M(t,e,r,i,n){this.good_length=t,this.max_lazy=e,this.nice_length=r,this.max_chain=i,this.func=n}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=v,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new d.Buf16(2*w),this.dyn_dtree=new d.Buf16(2*(2*a+1)),this.bl_tree=new d.Buf16(2*(2*o+1)),D(this.dyn_ltree),D(this.dyn_dtree),D(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new d.Buf16(k+1),this.heap=new d.Buf16(2*s+1),D(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new d.Buf16(2*s+1),D(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function G(t){var e;return t&&t.state?(t.total_in=t.total_out=0,t.data_type=n,(e=t.state).pending=0,e.pending_out=0,e.wrap<0&&(e.wrap=-e.wrap),e.status=e.wrap?C:E,t.adler=2===e.wrap?0:1,e.last_flush=l,u._tr_init(e),m):R(t,_)}function K(t){var e=G(t);return e===m&&function(t){t.window_size=2*t.w_size,D(t.head),t.max_lazy_match=h[t.level].max_lazy,t.good_match=h[t.level].good_length,t.nice_match=h[t.level].nice_length,t.max_chain_length=h[t.level].max_chain,t.strstart=0,t.block_start=0,t.lookahead=0,t.insert=0,t.match_length=t.prev_length=x-1,t.match_available=0,t.ins_h=0}(t.state),e}function Y(t,e,r,i,n,s){if(!t)return _;var a=1;if(e===g&&(e=6),i<0?(a=0,i=-i):15<i&&(a=2,i-=16),n<1||y<n||r!==v||i<8||15<i||e<0||9<e||s<0||b<s)return R(t,_);8===i&&(i=9);var o=new H;return(t.state=o).strm=t,o.wrap=a,o.gzhead=null,o.w_bits=i,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=n+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+x-1)/x),o.window=new d.Buf8(2*o.w_size),o.head=new d.Buf16(o.hash_size),o.prev=new d.Buf16(o.w_size),o.lit_bufsize=1<<n+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new d.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=e,o.strategy=s,o.method=r,K(t)}h=[new M(0,0,0,0,function(t,e){var r=65535;for(r>t.pending_buf_size-5&&(r=t.pending_buf_size-5);;){if(t.lookahead<=1){if(j(t),0===t.lookahead&&e===l)return A;if(0===t.lookahead)break}t.strstart+=t.lookahead,t.lookahead=0;var i=t.block_start+r;if((0===t.strstart||t.strstart>=i)&&(t.lookahead=t.strstart-i,t.strstart=i,N(t,!1),0===t.strm.avail_out))return A;if(t.strstart-t.block_start>=t.w_size-z&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):(t.strstart>t.block_start&&(N(t,!1),t.strm.avail_out),A)}),new M(4,4,8,4,Z),new M(4,5,16,8,Z),new M(4,6,32,32,Z),new M(4,4,16,16,W),new M(8,16,32,32,W),new M(8,16,128,128,W),new M(8,32,128,256,W),new M(32,128,258,1024,W),new M(32,258,258,4096,W)],r.deflateInit=function(t,e){return Y(t,e,v,15,8,0)},r.deflateInit2=Y,r.deflateReset=K,r.deflateResetKeep=G,r.deflateSetHeader=function(t,e){return t&&t.state?2!==t.state.wrap?_:(t.state.gzhead=e,m):_},r.deflate=function(t,e){var r,i,n,s;if(!t||!t.state||5<e||e<0)return t?R(t,_):_;if(i=t.state,!t.output||!t.input&&0!==t.avail_in||666===i.status&&e!==f)return R(t,0===t.avail_out?-5:_);if(i.strm=t,r=i.last_flush,i.last_flush=e,i.status===C)if(2===i.wrap)t.adler=0,U(i,31),U(i,139),U(i,8),i.gzhead?(U(i,(i.gzhead.text?1:0)+(i.gzhead.hcrc?2:0)+(i.gzhead.extra?4:0)+(i.gzhead.name?8:0)+(i.gzhead.comment?16:0)),U(i,255&i.gzhead.time),U(i,i.gzhead.time>>8&255),U(i,i.gzhead.time>>16&255),U(i,i.gzhead.time>>24&255),U(i,9===i.level?2:2<=i.strategy||i.level<2?4:0),U(i,255&i.gzhead.os),i.gzhead.extra&&i.gzhead.extra.length&&(U(i,255&i.gzhead.extra.length),U(i,i.gzhead.extra.length>>8&255)),i.gzhead.hcrc&&(t.adler=p(t.adler,i.pending_buf,i.pending,0)),i.gzindex=0,i.status=69):(U(i,0),U(i,0),U(i,0),U(i,0),U(i,0),U(i,9===i.level?2:2<=i.strategy||i.level<2?4:0),U(i,3),i.status=E);else{var a=v+(i.w_bits-8<<4)<<8;a|=(2<=i.strategy||i.level<2?0:i.level<6?1:6===i.level?2:3)<<6,0!==i.strstart&&(a|=32),a+=31-a%31,i.status=E,P(i,a),0!==i.strstart&&(P(i,t.adler>>>16),P(i,65535&t.adler)),t.adler=1}if(69===i.status)if(i.gzhead.extra){for(n=i.pending;i.gzindex<(65535&i.gzhead.extra.length)&&(i.pending!==i.pending_buf_size||(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending!==i.pending_buf_size));)U(i,255&i.gzhead.extra[i.gzindex]),i.gzindex++;i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),i.gzindex===i.gzhead.extra.length&&(i.gzindex=0,i.status=73)}else i.status=73;if(73===i.status)if(i.gzhead.name){n=i.pending;do{if(i.pending===i.pending_buf_size&&(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending===i.pending_buf_size)){s=1;break}s=i.gzindex<i.gzhead.name.length?255&i.gzhead.name.charCodeAt(i.gzindex++):0,U(i,s)}while(0!==s);i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),0===s&&(i.gzindex=0,i.status=91)}else i.status=91;if(91===i.status)if(i.gzhead.comment){n=i.pending;do{if(i.pending===i.pending_buf_size&&(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending===i.pending_buf_size)){s=1;break}s=i.gzindex<i.gzhead.comment.length?255&i.gzhead.comment.charCodeAt(i.gzindex++):0,U(i,s)}while(0!==s);i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),0===s&&(i.status=103)}else i.status=103;if(103===i.status&&(i.gzhead.hcrc?(i.pending+2>i.pending_buf_size&&F(t),i.pending+2<=i.pending_buf_size&&(U(i,255&t.adler),U(i,t.adler>>8&255),t.adler=0,i.status=E)):i.status=E),0!==i.pending){if(F(t),0===t.avail_out)return i.last_flush=-1,m}else if(0===t.avail_in&&T(e)<=T(r)&&e!==f)return R(t,-5);if(666===i.status&&0!==t.avail_in)return R(t,-5);if(0!==t.avail_in||0!==i.lookahead||e!==l&&666!==i.status){var o=2===i.strategy?function(t,e){for(var r;;){if(0===t.lookahead&&(j(t),0===t.lookahead)){if(e===l)return A;break}if(t.match_length=0,r=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++,r&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}(i,e):3===i.strategy?function(t,e){for(var r,i,n,s,a=t.window;;){if(t.lookahead<=S){if(j(t),t.lookahead<=S&&e===l)return A;if(0===t.lookahead)break}if(t.match_length=0,t.lookahead>=x&&0<t.strstart&&(i=a[n=t.strstart-1])===a[++n]&&i===a[++n]&&i===a[++n]){s=t.strstart+S;do{}while(i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&n<s);t.match_length=S-(s-n),t.match_length>t.lookahead&&(t.match_length=t.lookahead)}if(t.match_length>=x?(r=u._tr_tally(t,1,t.match_length-x),t.lookahead-=t.match_length,t.strstart+=t.match_length,t.match_length=0):(r=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++),r&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}(i,e):h[i.level].func(i,e);if(o!==O&&o!==B||(i.status=666),o===A||o===O)return 0===t.avail_out&&(i.last_flush=-1),m;if(o===I&&(1===e?u._tr_align(i):5!==e&&(u._tr_stored_block(i,0,0,!1),3===e&&(D(i.head),0===i.lookahead&&(i.strstart=0,i.block_start=0,i.insert=0))),F(t),0===t.avail_out))return i.last_flush=-1,m}return e!==f?m:i.wrap<=0?1:(2===i.wrap?(U(i,255&t.adler),U(i,t.adler>>8&255),U(i,t.adler>>16&255),U(i,t.adler>>24&255),U(i,255&t.total_in),U(i,t.total_in>>8&255),U(i,t.total_in>>16&255),U(i,t.total_in>>24&255)):(P(i,t.adler>>>16),P(i,65535&t.adler)),F(t),0<i.wrap&&(i.wrap=-i.wrap),0!==i.pending?m:1)},r.deflateEnd=function(t){var e;return t&&t.state?(e=t.state.status)!==C&&69!==e&&73!==e&&91!==e&&103!==e&&e!==E&&666!==e?R(t,_):(t.state=null,e===E?R(t,-3):m):_},r.deflateSetDictionary=function(t,e){var r,i,n,s,a,o,h,u,l=e.length;if(!t||!t.state)return _;if(2===(s=(r=t.state).wrap)||1===s&&r.status!==C||r.lookahead)return _;for(1===s&&(t.adler=c(t.adler,e,l,0)),r.wrap=0,l>=r.w_size&&(0===s&&(D(r.head),r.strstart=0,r.block_start=0,r.insert=0),u=new d.Buf8(r.w_size),d.arraySet(u,e,l-r.w_size,r.w_size,0),e=u,l=r.w_size),a=t.avail_in,o=t.next_in,h=t.input,t.avail_in=l,t.next_in=0,t.input=e,j(r);r.lookahead>=x;){for(i=r.strstart,n=r.lookahead-(x-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[i+x-1])&r.hash_mask,r.prev[i&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=i,i++,--n;);r.strstart=i,r.lookahead=x-1,j(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=x-1,r.match_available=0,t.next_in=o,t.input=h,t.avail_in=a,r.wrap=s,m},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(t,e,r){"use strict";e.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(t,e,r){"use strict";e.exports=function(t,e){var r,i,n,s,a,o,h,u,l,f,d,c,p,m,_,g,b,v,y,w,k,x,S,z,C;r=t.state,i=t.next_in,z=t.input,n=i+(t.avail_in-5),s=t.next_out,C=t.output,a=s-(e-t.avail_out),o=s+(t.avail_out-257),h=r.dmax,u=r.wsize,l=r.whave,f=r.wnext,d=r.window,c=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,b=(1<<r.distbits)-1;t:do{p<15&&(c+=z[i++]<<p,p+=8,c+=z[i++]<<p,p+=8),v=m[c&g];e:for(;;){if(c>>>=y=v>>>24,p-=y,0===(y=v>>>16&255))C[s++]=65535&v;else{if(!(16&y)){if(0==(64&y)){v=m[(65535&v)+(c&(1<<y)-1)];continue e}if(32&y){r.mode=12;break t}t.msg="invalid literal/length code",r.mode=30;break t}w=65535&v,(y&=15)&&(p<y&&(c+=z[i++]<<p,p+=8),w+=c&(1<<y)-1,c>>>=y,p-=y),p<15&&(c+=z[i++]<<p,p+=8,c+=z[i++]<<p,p+=8),v=_[c&b];r:for(;;){if(c>>>=y=v>>>24,p-=y,!(16&(y=v>>>16&255))){if(0==(64&y)){v=_[(65535&v)+(c&(1<<y)-1)];continue r}t.msg="invalid distance code",r.mode=30;break t}if(k=65535&v,p<(y&=15)&&(c+=z[i++]<<p,(p+=8)<y&&(c+=z[i++]<<p,p+=8)),h<(k+=c&(1<<y)-1)){t.msg="invalid distance too far back",r.mode=30;break t}if(c>>>=y,p-=y,(y=s-a)<k){if(l<(y=k-y)&&r.sane){t.msg="invalid distance too far back",r.mode=30;break t}if(S=d,(x=0)===f){if(x+=u-y,y<w){for(w-=y;C[s++]=d[x++],--y;);x=s-k,S=C}}else if(f<y){if(x+=u+f-y,(y-=f)<w){for(w-=y;C[s++]=d[x++],--y;);if(x=0,f<w){for(w-=y=f;C[s++]=d[x++],--y;);x=s-k,S=C}}}else if(x+=f-y,y<w){for(w-=y;C[s++]=d[x++],--y;);x=s-k,S=C}for(;2<w;)C[s++]=S[x++],C[s++]=S[x++],C[s++]=S[x++],w-=3;w&&(C[s++]=S[x++],1<w&&(C[s++]=S[x++]))}else{for(x=s-k;C[s++]=C[x++],C[s++]=C[x++],C[s++]=C[x++],2<(w-=3););w&&(C[s++]=C[x++],1<w&&(C[s++]=C[x++]))}break}}break}}while(i<n&&s<o);i-=w=p>>3,c&=(1<<(p-=w<<3))-1,t.next_in=i,t.next_out=s,t.avail_in=i<n?n-i+5:5-(i-n),t.avail_out=s<o?o-s+257:257-(s-o),r.hold=c,r.bits=p}},{}],49:[function(t,e,r){"use strict";var I=t("../utils/common"),O=t("./adler32"),B=t("./crc32"),R=t("./inffast"),T=t("./inftrees"),D=1,F=2,N=0,U=-2,P=1,i=852,n=592;function L(t){return(t>>>24&255)+(t>>>8&65280)+((65280&t)<<8)+((255&t)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(t){var e;return t&&t.state?(e=t.state,t.total_in=t.total_out=e.total=0,t.msg="",e.wrap&&(t.adler=1&e.wrap),e.mode=P,e.last=0,e.havedict=0,e.dmax=32768,e.head=null,e.hold=0,e.bits=0,e.lencode=e.lendyn=new I.Buf32(i),e.distcode=e.distdyn=new I.Buf32(n),e.sane=1,e.back=-1,N):U}function o(t){var e;return t&&t.state?((e=t.state).wsize=0,e.whave=0,e.wnext=0,a(t)):U}function h(t,e){var r,i;return t&&t.state?(i=t.state,e<0?(r=0,e=-e):(r=1+(e>>4),e<48&&(e&=15)),e&&(e<8||15<e)?U:(null!==i.window&&i.wbits!==e&&(i.window=null),i.wrap=r,i.wbits=e,o(t))):U}function u(t,e){var r,i;return t?(i=new s,(t.state=i).window=null,(r=h(t,e))!==N&&(t.state=null),r):U}var l,f,d=!0;function j(t){if(d){var e;for(l=new I.Buf32(512),f=new I.Buf32(32),e=0;e<144;)t.lens[e++]=8;for(;e<256;)t.lens[e++]=9;for(;e<280;)t.lens[e++]=7;for(;e<288;)t.lens[e++]=8;for(T(D,t.lens,0,288,l,0,t.work,{bits:9}),e=0;e<32;)t.lens[e++]=5;T(F,t.lens,0,32,f,0,t.work,{bits:5}),d=!1}t.lencode=l,t.lenbits=9,t.distcode=f,t.distbits=5}function Z(t,e,r,i){var n,s=t.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),i>=s.wsize?(I.arraySet(s.window,e,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(i<(n=s.wsize-s.wnext)&&(n=i),I.arraySet(s.window,e,r-i,n,s.wnext),(i-=n)?(I.arraySet(s.window,e,r-i,i,0),s.wnext=i,s.whave=s.wsize):(s.wnext+=n,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=n))),0}r.inflateReset=o,r.inflateReset2=h,r.inflateResetKeep=a,r.inflateInit=function(t){return u(t,15)},r.inflateInit2=u,r.inflate=function(t,e){var r,i,n,s,a,o,h,u,l,f,d,c,p,m,_,g,b,v,y,w,k,x,S,z,C=0,E=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!t||!t.state||!t.output||!t.input&&0!==t.avail_in)return U;12===(r=t.state).mode&&(r.mode=13),a=t.next_out,n=t.output,h=t.avail_out,s=t.next_in,i=t.input,o=t.avail_in,u=r.hold,l=r.bits,f=o,d=h,x=N;t:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(2&r.wrap&&35615===u){E[r.check=0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0),l=u=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&u)<<8)+(u>>8))%31){t.msg="incorrect header check",r.mode=30;break}if(8!=(15&u)){t.msg="unknown compression method",r.mode=30;break}if(l-=4,k=8+(15&(u>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){t.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,t.adler=r.check=1,r.mode=512&u?10:12,l=u=0;break;case 2:for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(r.flags=u,8!=(255&r.flags)){t.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){t.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=u>>8&1),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=3;case 3:for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.head&&(r.head.time=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,E[2]=u>>>16&255,E[3]=u>>>24&255,r.check=B(r.check,E,4,0)),l=u=0,r.mode=4;case 4:for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.head&&(r.head.xflags=255&u,r.head.os=u>>8),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=5;case 5:if(1024&r.flags){for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.length=u,r.head&&(r.head.extra_len=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(c=r.length)&&(c=o),c&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,i,s,c,k)),512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,r.length-=c),r.length))break t;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break t;for(c=0;k=i[s+c++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,k)break t}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break t;for(c=0;k=i[s+c++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,k)break t}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u!==(65535&r.check)){t.msg="header crc mismatch",r.mode=30;break}l=u=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),t.adler=r.check=0,r.mode=12;break;case 10:for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}t.adler=r.check=L(u),l=u=0,r.mode=11;case 11:if(0===r.havedict)return t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,2;t.adler=r.check=1,r.mode=12;case 12:if(5===e||6===e)break t;case 13:if(r.last){u>>>=7&l,l-=7&l,r.mode=27;break}for(;l<3;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}switch(r.last=1&u,l-=1,3&(u>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==e)break;u>>>=2,l-=2;break t;case 2:r.mode=17;break;case 3:t.msg="invalid block type",r.mode=30}u>>>=2,l-=2;break;case 14:for(u>>>=7&l,l-=7&l;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if((65535&u)!=(u>>>16^65535)){t.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&u,l=u=0,r.mode=15,6===e)break t;case 15:r.mode=16;case 16:if(c=r.length){if(o<c&&(c=o),h<c&&(c=h),0===c)break t;I.arraySet(n,i,s,c,a),o-=c,s+=c,h-=c,a+=c,r.length-=c;break}r.mode=12;break;case 17:for(;l<14;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(r.nlen=257+(31&u),u>>>=5,l-=5,r.ndist=1+(31&u),u>>>=5,l-=5,r.ncode=4+(15&u),u>>>=4,l-=4,286<r.nlen||30<r.ndist){t.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;l<3;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.lens[A[r.have++]]=7&u,u>>>=3,l-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=T(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){t.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(b<16)u>>>=_,l-=_,r.lens[r.have++]=b;else{if(16===b){for(z=_+2;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u>>>=_,l-=_,0===r.have){t.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],c=3+(3&u),u>>>=2,l-=2}else if(17===b){for(z=_+3;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}l-=_,k=0,c=3+(7&(u>>>=_)),u>>>=3,l-=3}else{for(z=_+7;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}l-=_,k=0,c=11+(127&(u>>>=_)),u>>>=7,l-=7}if(r.have+c>r.nlen+r.ndist){t.msg="invalid bit length repeat",r.mode=30;break}for(;c--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){t.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=T(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){t.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=T(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){t.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===e)break t;case 20:r.mode=21;case 21:if(6<=o&&258<=h){t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,R(t,d),a=t.next_out,n=t.output,h=t.avail_out,s=t.next_in,i=t.input,o=t.avail_in,u=r.hold,l=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(g&&0==(240&g)){for(v=_,y=g,w=b;g=(C=r.lencode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,r.length=b,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){t.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.length+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(C=r.distcode[u&(1<<r.distbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(0==(240&g)){for(v=_,y=g,w=b;g=(C=r.distcode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,64&g){t.msg="invalid distance code",r.mode=30;break}r.offset=b,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.offset+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){t.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===h)break t;if(c=d-h,r.offset>c){if((c=r.offset-c)>r.whave&&r.sane){t.msg="invalid distance too far back",r.mode=30;break}p=c>r.wnext?(c-=r.wnext,r.wsize-c):r.wnext-c,c>r.length&&(c=r.length),m=r.window}else m=n,p=a-r.offset,c=r.length;for(h<c&&(c=h),h-=c,r.length-=c;n[a++]=m[p++],--c;);0===r.length&&(r.mode=21);break;case 26:if(0===h)break t;n[a++]=r.length,h--,r.mode=21;break;case 27:if(r.wrap){for(;l<32;){if(0===o)break t;o--,u|=i[s++]<<l,l+=8}if(d-=h,t.total_out+=d,r.total+=d,d&&(t.adler=r.check=r.flags?B(r.check,n,d,a-d):O(r.check,n,d,a-d)),d=h,(r.flags?u:L(u))!==r.check){t.msg="incorrect data check",r.mode=30;break}l=u=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u!==(4294967295&r.total)){t.msg="incorrect length check",r.mode=30;break}l=u=0}r.mode=29;case 29:x=1;break t;case 30:x=-3;break t;case 31:return-4;case 32:default:return U}return t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,(r.wsize||d!==t.avail_out&&r.mode<30&&(r.mode<27||4!==e))&&Z(t,t.output,t.next_out,d-t.avail_out)?(r.mode=31,-4):(f-=t.avail_in,d-=t.avail_out,t.total_in+=f,t.total_out+=d,r.total+=d,r.wrap&&d&&(t.adler=r.check=r.flags?B(r.check,n,d,t.next_out-d):O(r.check,n,d,t.next_out-d)),t.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==f&&0===d||4===e)&&x===N&&(x=-5),x)},r.inflateEnd=function(t){if(!t||!t.state)return U;var e=t.state;return e.window&&(e.window=null),t.state=null,N},r.inflateGetHeader=function(t,e){var r;return t&&t.state?0==(2&(r=t.state).wrap)?U:((r.head=e).done=!1,N):U},r.inflateSetDictionary=function(t,e){var r,i=e.length;return t&&t.state?0!==(r=t.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,e,i,0)!==r.check?-3:Z(t,e,i,i)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(t,e,r){"use strict";var D=t("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];e.exports=function(t,e,r,i,n,s,a,o){var h,u,l,f,d,c,p,m,_,g=o.bits,b=0,v=0,y=0,w=0,k=0,x=0,S=0,z=0,C=0,E=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),R=null,T=0;for(b=0;b<=15;b++)O[b]=0;for(v=0;v<i;v++)O[e[r+v]]++;for(k=g,w=15;1<=w&&0===O[w];w--);if(w<k&&(k=w),0===w)return n[s++]=20971520,n[s++]=20971520,o.bits=1,0;for(y=1;y<w&&0===O[y];y++);for(k<y&&(k=y),b=z=1;b<=15;b++)if(z<<=1,(z-=O[b])<0)return-1;if(0<z&&(0===t||1!==w))return-1;for(B[1]=0,b=1;b<15;b++)B[b+1]=B[b]+O[b];for(v=0;v<i;v++)0!==e[r+v]&&(a[B[e[r+v]]++]=v);if(c=0===t?(A=R=a,19):1===t?(A=F,I-=257,R=N,T-=257,256):(A=U,R=P,-1),b=y,d=s,S=v=E=0,l=-1,f=(C=1<<(x=k))-1,1===t&&852<C||2===t&&592<C)return 1;for(;;){for(p=b-S,_=a[v]<c?(m=0,a[v]):a[v]>c?(m=R[T+a[v]],A[I+a[v]]):(m=96,0),h=1<<b-S,y=u=1<<x;n[d+(E>>S)+(u-=h)]=p<<24|m<<16|_|0,0!==u;);for(h=1<<b-1;E&h;)h>>=1;if(0!==h?(E&=h-1,E+=h):E=0,v++,0==--O[b]){if(b===w)break;b=e[r+a[v]]}if(k<b&&(E&f)!==l){for(0===S&&(S=k),d+=y,z=1<<(x=b-S);x+S<w&&!((z-=O[x+S])<=0);)x++,z<<=1;if(C+=1<<x,1===t&&852<C||2===t&&592<C)return 1;n[l=E&f]=k<<24|x<<16|d-s|0}}return 0!==E&&(n[d+E]=b-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(t,e,r){"use strict";e.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(t,e,r){"use strict";var n=t("../utils/common"),o=0,h=1;function i(t){for(var e=t.length;0<=--e;)t[e]=0}var s=0,a=29,u=256,l=u+1+a,f=30,d=19,_=2*l+1,g=15,c=16,p=7,m=256,b=16,v=17,y=18,w=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],k=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],x=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],z=new Array(2*(l+2));i(z);var C=new Array(2*f);i(C);var E=new Array(512);i(E);var A=new Array(256);i(A);var I=new Array(a);i(I);var O,B,R,T=new Array(f);function D(t,e,r,i,n){this.static_tree=t,this.extra_bits=e,this.extra_base=r,this.elems=i,this.max_length=n,this.has_stree=t&&t.length}function F(t,e){this.dyn_tree=t,this.max_code=0,this.stat_desc=e}function N(t){return t<256?E[t]:E[256+(t>>>7)]}function U(t,e){t.pending_buf[t.pending++]=255&e,t.pending_buf[t.pending++]=e>>>8&255}function P(t,e,r){t.bi_valid>c-r?(t.bi_buf|=e<<t.bi_valid&65535,U(t,t.bi_buf),t.bi_buf=e>>c-t.bi_valid,t.bi_valid+=r-c):(t.bi_buf|=e<<t.bi_valid&65535,t.bi_valid+=r)}function L(t,e,r){P(t,r[2*e],r[2*e+1])}function j(t,e){for(var r=0;r|=1&t,t>>>=1,r<<=1,0<--e;);return r>>>1}function Z(t,e,r){var i,n,s=new Array(g+1),a=0;for(i=1;i<=g;i++)s[i]=a=a+r[i-1]<<1;for(n=0;n<=e;n++){var o=t[2*n+1];0!==o&&(t[2*n]=j(s[o]++,o))}}function W(t){var e;for(e=0;e<l;e++)t.dyn_ltree[2*e]=0;for(e=0;e<f;e++)t.dyn_dtree[2*e]=0;for(e=0;e<d;e++)t.bl_tree[2*e]=0;t.dyn_ltree[2*m]=1,t.opt_len=t.static_len=0,t.last_lit=t.matches=0}function M(t){8<t.bi_valid?U(t,t.bi_buf):0<t.bi_valid&&(t.pending_buf[t.pending++]=t.bi_buf),t.bi_buf=0,t.bi_valid=0}function H(t,e,r,i){var n=2*e,s=2*r;return t[n]<t[s]||t[n]===t[s]&&i[e]<=i[r]}function G(t,e,r){for(var i=t.heap[r],n=r<<1;n<=t.heap_len&&(n<t.heap_len&&H(e,t.heap[n+1],t.heap[n],t.depth)&&n++,!H(e,i,t.heap[n],t.depth));)t.heap[r]=t.heap[n],r=n,n<<=1;t.heap[r]=i}function K(t,e,r){var i,n,s,a,o=0;if(0!==t.last_lit)for(;i=t.pending_buf[t.d_buf+2*o]<<8|t.pending_buf[t.d_buf+2*o+1],n=t.pending_buf[t.l_buf+o],o++,0===i?L(t,n,e):(L(t,(s=A[n])+u+1,e),0!==(a=w[s])&&P(t,n-=I[s],a),L(t,s=N(--i),r),0!==(a=k[s])&&P(t,i-=T[s],a)),o<t.last_lit;);L(t,m,e)}function Y(t,e){var r,i,n,s=e.dyn_tree,a=e.stat_desc.static_tree,o=e.stat_desc.has_stree,h=e.stat_desc.elems,u=-1;for(t.heap_len=0,t.heap_max=_,r=0;r<h;r++)0!==s[2*r]?(t.heap[++t.heap_len]=u=r,t.depth[r]=0):s[2*r+1]=0;for(;t.heap_len<2;)s[2*(n=t.heap[++t.heap_len]=u<2?++u:0)]=1,t.depth[n]=0,t.opt_len--,o&&(t.static_len-=a[2*n+1]);for(e.max_code=u,r=t.heap_len>>1;1<=r;r--)G(t,s,r);for(n=h;r=t.heap[1],t.heap[1]=t.heap[t.heap_len--],G(t,s,1),i=t.heap[1],t.heap[--t.heap_max]=r,t.heap[--t.heap_max]=i,s[2*n]=s[2*r]+s[2*i],t.depth[n]=(t.depth[r]>=t.depth[i]?t.depth[r]:t.depth[i])+1,s[2*r+1]=s[2*i+1]=n,t.heap[1]=n++,G(t,s,1),2<=t.heap_len;);t.heap[--t.heap_max]=t.heap[1],function(t,e){var r,i,n,s,a,o,h=e.dyn_tree,u=e.max_code,l=e.stat_desc.static_tree,f=e.stat_desc.has_stree,d=e.stat_desc.extra_bits,c=e.stat_desc.extra_base,p=e.stat_desc.max_length,m=0;for(s=0;s<=g;s++)t.bl_count[s]=0;for(h[2*t.heap[t.heap_max]+1]=0,r=t.heap_max+1;r<_;r++)p<(s=h[2*h[2*(i=t.heap[r])+1]+1]+1)&&(s=p,m++),h[2*i+1]=s,u<i||(t.bl_count[s]++,a=0,c<=i&&(a=d[i-c]),o=h[2*i],t.opt_len+=o*(s+a),f&&(t.static_len+=o*(l[2*i+1]+a)));if(0!==m){do{for(s=p-1;0===t.bl_count[s];)s--;t.bl_count[s]--,t.bl_count[s+1]+=2,t.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(i=t.bl_count[s];0!==i;)u<(n=t.heap[--r])||(h[2*n+1]!==s&&(t.opt_len+=(s-h[2*n+1])*h[2*n],h[2*n+1]=s),i--)}}(t,e),Z(s,u,t.bl_count)}function X(t,e,r){var i,n,s=-1,a=e[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),e[2*(r+1)+1]=65535,i=0;i<=r;i++)n=a,a=e[2*(i+1)+1],++o<h&&n===a||(o<u?t.bl_tree[2*n]+=o:0!==n?(n!==s&&t.bl_tree[2*n]++,t.bl_tree[2*b]++):o<=10?t.bl_tree[2*v]++:t.bl_tree[2*y]++,s=n,u=(o=0)===a?(h=138,3):n===a?(h=6,3):(h=7,4))}function V(t,e,r){var i,n,s=-1,a=e[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),i=0;i<=r;i++)if(n=a,a=e[2*(i+1)+1],!(++o<h&&n===a)){if(o<u)for(;L(t,n,t.bl_tree),0!=--o;);else 0!==n?(n!==s&&(L(t,n,t.bl_tree),o--),L(t,b,t.bl_tree),P(t,o-3,2)):o<=10?(L(t,v,t.bl_tree),P(t,o-3,3)):(L(t,y,t.bl_tree),P(t,o-11,7));s=n,u=(o=0)===a?(h=138,3):n===a?(h=6,3):(h=7,4)}}i(T);var q=!1;function J(t,e,r,i){P(t,(s<<1)+(i?1:0),3),function(t,e,r,i){M(t),i&&(U(t,r),U(t,~r)),n.arraySet(t.pending_buf,t.window,e,r,t.pending),t.pending+=r}(t,e,r,!0)}r._tr_init=function(t){q||(function(){var t,e,r,i,n,s=new Array(g+1);for(i=r=0;i<a-1;i++)for(I[i]=r,t=0;t<1<<w[i];t++)A[r++]=i;for(A[r-1]=i,i=n=0;i<16;i++)for(T[i]=n,t=0;t<1<<k[i];t++)E[n++]=i;for(n>>=7;i<f;i++)for(T[i]=n<<7,t=0;t<1<<k[i]-7;t++)E[256+n++]=i;for(e=0;e<=g;e++)s[e]=0;for(t=0;t<=143;)z[2*t+1]=8,t++,s[8]++;for(;t<=255;)z[2*t+1]=9,t++,s[9]++;for(;t<=279;)z[2*t+1]=7,t++,s[7]++;for(;t<=287;)z[2*t+1]=8,t++,s[8]++;for(Z(z,l+1,s),t=0;t<f;t++)C[2*t+1]=5,C[2*t]=j(t,5);O=new D(z,w,u+1,l,g),B=new D(C,k,0,f,g),R=new D(new Array(0),x,0,d,p)}(),q=!0),t.l_desc=new F(t.dyn_ltree,O),t.d_desc=new F(t.dyn_dtree,B),t.bl_desc=new F(t.bl_tree,R),t.bi_buf=0,t.bi_valid=0,W(t)},r._tr_stored_block=J,r._tr_flush_block=function(t,e,r,i){var n,s,a=0;0<t.level?(2===t.strm.data_type&&(t.strm.data_type=function(t){var e,r=4093624447;for(e=0;e<=31;e++,r>>>=1)if(1&r&&0!==t.dyn_ltree[2*e])return o;if(0!==t.dyn_ltree[18]||0!==t.dyn_ltree[20]||0!==t.dyn_ltree[26])return h;for(e=32;e<u;e++)if(0!==t.dyn_ltree[2*e])return h;return o}(t)),Y(t,t.l_desc),Y(t,t.d_desc),a=function(t){var e;for(X(t,t.dyn_ltree,t.l_desc.max_code),X(t,t.dyn_dtree,t.d_desc.max_code),Y(t,t.bl_desc),e=d-1;3<=e&&0===t.bl_tree[2*S[e]+1];e--);return t.opt_len+=3*(e+1)+5+5+4,e}(t),n=t.opt_len+3+7>>>3,(s=t.static_len+3+7>>>3)<=n&&(n=s)):n=s=r+5,r+4<=n&&-1!==e?J(t,e,r,i):4===t.strategy||s===n?(P(t,2+(i?1:0),3),K(t,z,C)):(P(t,4+(i?1:0),3),function(t,e,r,i){var n;for(P(t,e-257,5),P(t,r-1,5),P(t,i-4,4),n=0;n<i;n++)P(t,t.bl_tree[2*S[n]+1],3);V(t,t.dyn_ltree,e-1),V(t,t.dyn_dtree,r-1)}(t,t.l_desc.max_code+1,t.d_desc.max_code+1,a+1),K(t,t.dyn_ltree,t.dyn_dtree)),W(t),i&&M(t)},r._tr_tally=function(t,e,r){return t.pending_buf[t.d_buf+2*t.last_lit]=e>>>8&255,t.pending_buf[t.d_buf+2*t.last_lit+1]=255&e,t.pending_buf[t.l_buf+t.last_lit]=255&r,t.last_lit++,0===e?t.dyn_ltree[2*r]++:(t.matches++,e--,t.dyn_ltree[2*(A[r]+u+1)]++,t.dyn_dtree[2*N(e)]++),t.last_lit===t.lit_bufsize-1},r._tr_align=function(t){P(t,2,3),L(t,m,z),function(t){16===t.bi_valid?(U(t,t.bi_buf),t.bi_buf=0,t.bi_valid=0):8<=t.bi_valid&&(t.pending_buf[t.pending++]=255&t.bi_buf,t.bi_buf>>=8,t.bi_valid-=8)}(t)}},{"../utils/common":41}],53:[function(t,e,r){"use strict";e.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(t,e,r){"use strict";e.exports="function"==typeof setImmediate?setImmediate:function(){var t=[].slice.apply(arguments);t.splice(1,0,0),setTimeout.apply(null,t)}},{}]},{},[10])(10)});
/*! pdfmake v0.1.57, @license MIT, @link http://pdfmake.org */
!function(t,e){if("object"==typeof exports&&"object"==typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define([],e);else{var n=e();for(var r in n)("object"==typeof exports?exports:t)[r]=n[r]}}("undefined"!=typeof self?self:this,function(){return r={},i.m=n=[function(t,e,n){"use strict";function o(t){return Array.isArray(t)}t.exports={isString:function(t){return"string"==typeof t||t instanceof String},isNumber:function(t){return"number"==typeof t||t instanceof Number},isBoolean:function(t){return"boolean"==typeof t},isArray:o,isFunction:function(t){return"function"==typeof t},isObject:function(t){return null!==t&&"object"==typeof t},isNull:function(t){return null===t},isUndefined:function(t){return void 0===t},pack:function(){for(var t={},e=0,n=arguments.length;e<n;e++){var r=arguments[e];if(r)for(var i in r)r.hasOwnProperty(i)&&(t[i]=r[i])}return t},fontStringify:function(t,e){return"font"===t?"font":e},offsetVector:function(t,e,n){switch(t.type){case"ellipse":case"rect":t.x+=e,t.y+=n;break;case"line":t.x1+=e,t.x2+=e,t.y1+=n,t.y2+=n;break;case"polyline":for(var r=0,i=t.points.length;r<i;r++)t.points[r].x+=e,t.points[r].y+=n}},getNodeId:function t(e){if(e.id)return e.id;if(o(e.text))for(var n=0,r=e.text.length;n<r;n++){var i=t(e.text[n]);if(i)return i}return null}}},function(t,e,n){var r;t.exports=(r=r||function(f){var n=Object.create||function(){function n(){}return function(t){var e;return n.prototype=t,e=new n,n.prototype=null,e}}(),t={},e=t.lib={},r=e.Base={extend:function(t){var e=n(this);return t&&e.mixIn(t),e.hasOwnProperty("init")&&this.init!==e.init||(e.init=function(){e.$super.init.apply(this,arguments)}),(e.init.prototype=e).$super=this,e},create:function(){var t=this.extend();return t.init.apply(t,arguments),t},init:function(){},mixIn:function(t){for(var e in t)t.hasOwnProperty(e)&&(this[e]=t[e]);t.hasOwnProperty("toString")&&(this.toString=t.toString)},clone:function(){return this.init.prototype.extend(this)}},h=e.WordArray=r.extend({init:function(t,e){t=this.words=t||[],this.sigBytes=null!=e?e:4*t.length},toString:function(t){return(t||o).stringify(this)},concat:function(t){var e=this.words,n=t.words,r=this.sigBytes,i=t.sigBytes;if(this.clamp(),r%4)for(var o=0;o<i;o++){var a=n[o>>>2]>>>24-o%4*8&255;e[r+o>>>2]|=a<<24-(r+o)%4*8}else for(var o=0;o<i;o+=4)e[r+o>>>2]=n[o>>>2];return this.sigBytes+=i,this},clamp:function(){var t=this.words,e=this.sigBytes;t[e>>>2]&=4294967295<<32-e%4*8,t.length=f.ceil(e/4)},clone:function(){var t=r.clone.call(this);return t.words=this.words.slice(0),t},random:function(t){for(var e,n=[],r=function(e){var e=e,n=987654321,r=4294967295;return function(){var t=((n=36969*(65535&n)+(n>>16)&r)<<16)+(e=18e3*(65535&e)+(e>>16)&r)&r;return t/=4294967296,(t+=.5)*(.5<f.random()?1:-1)}},i=0;i<t;i+=4){var o=r(4294967296*(e||f.random()));e=987654071*o(),n.push(4294967296*o()|0)}return new h.init(n,t)}}),i=t.enc={},o=i.Hex={stringify:function(t){for(var e=t.words,n=t.sigBytes,r=[],i=0;i<n;i++){var o=e[i>>>2]>>>24-i%4*8&255;r.push((o>>>4).toString(16)),r.push((15&o).toString(16))}return r.join("")},parse:function(t){for(var e=t.length,n=[],r=0;r<e;r+=2)n[r>>>3]|=parseInt(t.substr(r,2),16)<<24-r%8*4;return new h.init(n,e/2)}},a=i.Latin1={stringify:function(t){for(var e=t.words,n=t.sigBytes,r=[],i=0;i<n;i++){var o=e[i>>>2]>>>24-i%4*8&255;r.push(String.fromCharCode(o))}return r.join("")},parse:function(t){for(var e=t.length,n=[],r=0;r<e;r++)n[r>>>2]|=(255&t.charCodeAt(r))<<24-r%4*8;return new h.init(n,e)}},s=i.Utf8={stringify:function(t){try{return decodeURIComponent(escape(a.stringify(t)))}catch(t){throw new Error("Malformed UTF-8 data")}},parse:function(t){return a.parse(unescape(encodeURIComponent(t)))}},l=e.BufferedBlockAlgorithm=r.extend({reset:function(){this._data=new h.init,this._nDataBytes=0},_append:function(t){"string"==typeof t&&(t=s.parse(t)),this._data.concat(t),this._nDataBytes+=t.sigBytes},_process:function(t){var e=this._data,n=e.words,r=e.sigBytes,i=this.blockSize,o=4*i,a=r/o,s=(a=t?f.ceil(a):f.max((0|a)-this._minBufferSize,0))*i,l=f.min(4*s,r);if(s){for(var u=0;u<s;u+=i)this._doProcessBlock(n,u);var c=n.splice(0,s);e.sigBytes-=l}return new h.init(c,l)},clone:function(){var t=r.clone.call(this);return t._data=this._data.clone(),t},_minBufferSize:0}),u=(e.Hasher=l.extend({cfg:r.extend(),init:function(t){this.cfg=this.cfg.extend(t),this.reset()},reset:function(){l.reset.call(this),this._doReset()},update:function(t){return this._append(t),this._process(),this},finalize:function(t){t&&this._append(t);var e=this._doFinalize();return e},blockSize:16,_createHelper:function(n){return function(t,e){return new n.init(e).finalize(t)}},_createHmacHelper:function(n){return function(t,e){return new u.HMAC.init(n,e).finalize(t)}}}),t.algo={});return t}(Math),r)},function(t,e){var n=t.exports={version:"2.6.9"};"number"==typeof __e&&(__e=n)},function(t,e,n){var r=n(62)("wks"),i=n(29),o=n(8).Symbol,a="function"==typeof o;(t.exports=function(t){return r[t]||(r[t]=a&&o[t]||(a?o:i)("Symbol."+t))}).store=r},function(t,D,z){"use strict";(function(t){
/*!
 * The buffer module from node.js, for the browser.
 *
 * @author   Feross Aboukhadijeh <feross@feross.org> <http://feross.org>
 * @license  MIT
 */
var r=z(198),o=z(199),a=z(129);function n(){return f.TYPED_ARRAY_SUPPORT?2147483647:1073741823}function s(t,e){if(n()<e)throw new RangeError("Invalid typed array length");return f.TYPED_ARRAY_SUPPORT?(t=new Uint8Array(e)).__proto__=f.prototype:(null===t&&(t=new f(e)),t.length=e),t}function f(t,e,n){if(!(f.TYPED_ARRAY_SUPPORT||this instanceof f))return new f(t,e,n);if("number"!=typeof t)return i(this,t,e,n);if("string"==typeof e)throw new Error("If encoding is specified then the first argument must be a string");return u(this,t)}function i(t,e,n,r){if("number"==typeof e)throw new TypeError('"value" argument must not be a number');return"undefined"!=typeof ArrayBuffer&&e instanceof ArrayBuffer?function(t,e,n,r){if(e.byteLength,n<0||e.byteLength<n)throw new RangeError("'offset' is out of bounds");if(e.byteLength<n+(r||0))throw new RangeError("'length' is out of bounds");e=void 0===n&&void 0===r?new Uint8Array(e):void 0===r?new Uint8Array(e,n):new Uint8Array(e,n,r);f.TYPED_ARRAY_SUPPORT?(t=e).__proto__=f.prototype:t=c(t,e);return t}(t,e,n,r):"string"==typeof e?function(t,e,n){"string"==typeof n&&""!==n||(n="utf8");if(!f.isEncoding(n))throw new TypeError('"encoding" must be a valid string encoding');var r=0|d(e,n),i=(t=s(t,r)).write(e,n);i!==r&&(t=t.slice(0,i));return t}(t,e,n):function(t,e){if(f.isBuffer(e)){var n=0|h(e.length);return 0===(t=s(t,n)).length||e.copy(t,0,0,n),t}if(e){if("undefined"!=typeof ArrayBuffer&&e.buffer instanceof ArrayBuffer||"length"in e)return"number"!=typeof e.length||function(t){return t!=t}(e.length)?s(t,0):c(t,e);if("Buffer"===e.type&&a(e.data))return c(t,e.data)}throw new TypeError("First argument must be a string, Buffer, ArrayBuffer, Array, or array-like object.")}(t,e)}function l(t){if("number"!=typeof t)throw new TypeError('"size" argument must be a number');if(t<0)throw new RangeError('"size" argument must not be negative')}function u(t,e){if(l(e),t=s(t,e<0?0:0|h(e)),!f.TYPED_ARRAY_SUPPORT)for(var n=0;n<e;++n)t[n]=0;return t}function c(t,e){var n=e.length<0?0:0|h(e.length);t=s(t,n);for(var r=0;r<n;r+=1)t[r]=255&e[r];return t}function h(t){if(t>=n())throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+n().toString(16)+" bytes");return 0|t}function d(t,e){if(f.isBuffer(t))return t.length;if("undefined"!=typeof ArrayBuffer&&"function"==typeof ArrayBuffer.isView&&(ArrayBuffer.isView(t)||t instanceof ArrayBuffer))return t.byteLength;"string"!=typeof t&&(t=""+t);var n=t.length;if(0===n)return 0;for(var r=!1;;)switch(e){case"ascii":case"latin1":case"binary":return n;case"utf8":case"utf-8":case void 0:return R(t).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*n;case"hex":return n>>>1;case"base64":return M(t).length;default:if(r)return R(t).length;e=(""+e).toLowerCase(),r=!0}}function p(t,e,n){var r=t[e];t[e]=t[n],t[n]=r}function g(t,e,n,r,i){if(0===t.length)return-1;if("string"==typeof n?(r=n,n=0):2147483647<n?n=2147483647:n<-2147483648&&(n=-2147483648),n=+n,isNaN(n)&&(n=i?0:t.length-1),n<0&&(n=t.length+n),n>=t.length){if(i)return-1;n=t.length-1}else if(n<0){if(!i)return-1;n=0}if("string"==typeof e&&(e=f.from(e,r)),f.isBuffer(e))return 0===e.length?-1:v(t,e,n,r,i);if("number"==typeof e)return e&=255,f.TYPED_ARRAY_SUPPORT&&"function"==typeof Uint8Array.prototype.indexOf?i?Uint8Array.prototype.indexOf.call(t,e,n):Uint8Array.prototype.lastIndexOf.call(t,e,n):v(t,[e],n,r,i);throw new TypeError("val must be string, number or Buffer")}function v(t,e,n,r,i){var o,a=1,s=t.length,l=e.length;if(void 0!==r&&("ucs2"===(r=String(r).toLowerCase())||"ucs-2"===r||"utf16le"===r||"utf-16le"===r)){if(t.length<2||e.length<2)return-1;s/=a=2,l/=2,n/=2}function u(t,e){return 1===a?t[e]:t.readUInt16BE(e*a)}if(i){var c=-1;for(o=n;o<s;o++)if(u(t,o)===u(e,-1===c?0:o-c)){if(-1===c&&(c=o),o-c+1===l)return c*a}else-1!==c&&(o-=o-c),c=-1}else for(s<n+l&&(n=s-l),o=n;0<=o;o--){for(var f=!0,h=0;h<l;h++)if(u(t,o+h)!==u(e,h)){f=!1;break}if(f)return o}return-1}function y(t,e,n,r){n=Number(n)||0;var i=t.length-n;r?i<(r=Number(r))&&(r=i):r=i;var o=e.length;if(o%2!=0)throw new TypeError("Invalid hex string");o/2<r&&(r=o/2);for(var a=0;a<r;++a){var s=parseInt(e.substr(2*a,2),16);if(isNaN(s))return a;t[n+a]=s}return a}function b(t,e,n,r){return F(function(t){for(var e=[],n=0;n<t.length;++n)e.push(255&t.charCodeAt(n));return e}(e),t,n,r)}function m(t,e,n){return 0===e&&n===t.length?r.fromByteArray(t):r.fromByteArray(t.slice(e,n))}function w(t,e,n){n=Math.min(t.length,n);for(var r=[],i=e;i<n;){var o,a,s,l,u=t[i],c=null,f=239<u?4:223<u?3:191<u?2:1;if(i+f<=n)switch(f){case 1:u<128&&(c=u);break;case 2:128==(192&(o=t[i+1]))&&127<(l=(31&u)<<6|63&o)&&(c=l);break;case 3:o=t[i+1],a=t[i+2],128==(192&o)&&128==(192&a)&&2047<(l=(15&u)<<12|(63&o)<<6|63&a)&&(l<55296||57343<l)&&(c=l);break;case 4:o=t[i+1],a=t[i+2],s=t[i+3],128==(192&o)&&128==(192&a)&&128==(192&s)&&65535<(l=(15&u)<<18|(63&o)<<12|(63&a)<<6|63&s)&&l<1114112&&(c=l)}null===c?(c=65533,f=1):65535<c&&(c-=65536,r.push(c>>>10&1023|55296),c=56320|1023&c),r.push(c),i+=f}return function(t){var e=t.length;if(e<=x)return String.fromCharCode.apply(String,t);var n="",r=0;for(;r<e;)n+=String.fromCharCode.apply(String,t.slice(r,r+=x));return n}(r)}D.Buffer=f,D.SlowBuffer=function(t){+t!=t&&(t=0);return f.alloc(+t)},D.INSPECT_MAX_BYTES=50,f.TYPED_ARRAY_SUPPORT=void 0!==t.TYPED_ARRAY_SUPPORT?t.TYPED_ARRAY_SUPPORT:function(){try{var t=new Uint8Array(1);return t.__proto__={__proto__:Uint8Array.prototype,foo:function(){return 42}},42===t.foo()&&"function"==typeof t.subarray&&0===t.subarray(1,1).byteLength}catch(t){return!1}}(),D.kMaxLength=n(),f.poolSize=8192,f._augment=function(t){return t.__proto__=f.prototype,t},f.from=function(t,e,n){return i(null,t,e,n)},f.TYPED_ARRAY_SUPPORT&&(f.prototype.__proto__=Uint8Array.prototype,f.__proto__=Uint8Array,"undefined"!=typeof Symbol&&Symbol.species&&f[Symbol.species]===f&&Object.defineProperty(f,Symbol.species,{value:null,configurable:!0})),f.alloc=function(t,e,n){return function(t,e,n,r){return l(e),e<=0?s(t,e):void 0!==n?"string"==typeof r?s(t,e).fill(n,r):s(t,e).fill(n):s(t,e)}(null,t,e,n)},f.allocUnsafe=function(t){return u(null,t)},f.allocUnsafeSlow=function(t){return u(null,t)},f.isBuffer=function(t){return!(null==t||!t._isBuffer)},f.compare=function(t,e){if(!f.isBuffer(t)||!f.isBuffer(e))throw new TypeError("Arguments must be Buffers");if(t===e)return 0;for(var n=t.length,r=e.length,i=0,o=Math.min(n,r);i<o;++i)if(t[i]!==e[i]){n=t[i],r=e[i];break}return n<r?-1:r<n?1:0},f.isEncoding=function(t){switch(String(t).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},f.concat=function(t,e){if(!a(t))throw new TypeError('"list" argument must be an Array of Buffers');if(0===t.length)return f.alloc(0);var n;if(void 0===e)for(n=e=0;n<t.length;++n)e+=t[n].length;var r=f.allocUnsafe(e),i=0;for(n=0;n<t.length;++n){var o=t[n];if(!f.isBuffer(o))throw new TypeError('"list" argument must be an Array of Buffers');o.copy(r,i),i+=o.length}return r},f.byteLength=d,f.prototype._isBuffer=!0,f.prototype.swap16=function(){var t=this.length;if(t%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var e=0;e<t;e+=2)p(this,e,e+1);return this},f.prototype.swap32=function(){var t=this.length;if(t%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var e=0;e<t;e+=4)p(this,e,e+3),p(this,e+1,e+2);return this},f.prototype.swap64=function(){var t=this.length;if(t%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var e=0;e<t;e+=8)p(this,e,e+7),p(this,e+1,e+6),p(this,e+2,e+5),p(this,e+3,e+4);return this},f.prototype.toString=function(){var t=0|this.length;return 0==t?"":0===arguments.length?w(this,0,t):function(t,e,n){var r=!1;if((void 0===e||e<0)&&(e=0),e>this.length)return"";if((void 0===n||n>this.length)&&(n=this.length),n<=0)return"";if((n>>>=0)<=(e>>>=0))return"";for(t=t||"utf8";;)switch(t){case"hex":return S(this,e,n);case"utf8":case"utf-8":return w(this,e,n);case"ascii":return _(this,e,n);case"latin1":case"binary":return k(this,e,n);case"base64":return m(this,e,n);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return C(this,e,n);default:if(r)throw new TypeError("Unknown encoding: "+t);t=(t+"").toLowerCase(),r=!0}}.apply(this,arguments)},f.prototype.equals=function(t){if(!f.isBuffer(t))throw new TypeError("Argument must be a Buffer");return this===t||0===f.compare(this,t)},f.prototype.inspect=function(){var t="",e=D.INSPECT_MAX_BYTES;return 0<this.length&&(t=this.toString("hex",0,e).match(/.{2}/g).join(" "),this.length>e&&(t+=" ... ")),"<Buffer "+t+">"},f.prototype.compare=function(t,e,n,r,i){if(!f.isBuffer(t))throw new TypeError("Argument must be a Buffer");if(void 0===e&&(e=0),void 0===n&&(n=t?t.length:0),void 0===r&&(r=0),void 0===i&&(i=this.length),e<0||n>t.length||r<0||i>this.length)throw new RangeError("out of range index");if(i<=r&&n<=e)return 0;if(i<=r)return-1;if(n<=e)return 1;if(this===t)return 0;for(var o=(i>>>=0)-(r>>>=0),a=(n>>>=0)-(e>>>=0),s=Math.min(o,a),l=this.slice(r,i),u=t.slice(e,n),c=0;c<s;++c)if(l[c]!==u[c]){o=l[c],a=u[c];break}return o<a?-1:a<o?1:0},f.prototype.includes=function(t,e,n){return-1!==this.indexOf(t,e,n)},f.prototype.indexOf=function(t,e,n){return g(this,t,e,n,!0)},f.prototype.lastIndexOf=function(t,e,n){return g(this,t,e,n,!1)},f.prototype.write=function(t,e,n,r){if(void 0===e)r="utf8",n=this.length,e=0;else if(void 0===n&&"string"==typeof e)r=e,n=this.length,e=0;else{if(!isFinite(e))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");e|=0,isFinite(n)?(n|=0,void 0===r&&(r="utf8")):(r=n,n=void 0)}var i=this.length-e;if((void 0===n||i<n)&&(n=i),0<t.length&&(n<0||e<0)||e>this.length)throw new RangeError("Attempt to write outside buffer bounds");r=r||"utf8";for(var o,a,s,l,u,c,f,h,d,p=!1;;)switch(r){case"hex":return y(this,t,e,n);case"utf8":case"utf-8":return h=e,d=n,F(R(t,(f=this).length-h),f,h,d);case"ascii":return b(this,t,e,n);case"latin1":case"binary":return b(this,t,e,n);case"base64":return l=this,u=e,c=n,F(M(t),l,u,c);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return a=e,s=n,F(function(t,e){for(var n,r,i,o=[],a=0;a<t.length&&!((e-=2)<0);++a)n=t.charCodeAt(a),r=n>>8,i=n%256,o.push(i),o.push(r);return o}(t,(o=this).length-a),o,a,s);default:if(p)throw new TypeError("Unknown encoding: "+r);r=(""+r).toLowerCase(),p=!0}},f.prototype.toJSON=function(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};var x=4096;function _(t,e,n){var r="";n=Math.min(t.length,n);for(var i=e;i<n;++i)r+=String.fromCharCode(127&t[i]);return r}function k(t,e,n){var r="";n=Math.min(t.length,n);for(var i=e;i<n;++i)r+=String.fromCharCode(t[i]);return r}function S(t,e,n){var r=t.length;(!e||e<0)&&(e=0),(!n||n<0||r<n)&&(n=r);for(var i="",o=e;o<n;++o)i+=L(t[o]);return i}function C(t,e,n){for(var r=t.slice(e,n),i="",o=0;o<r.length;o+=2)i+=String.fromCharCode(r[o]+256*r[o+1]);return i}function A(t,e,n){if(t%1!=0||t<0)throw new RangeError("offset is not uint");if(n<t+e)throw new RangeError("Trying to access beyond buffer length")}function P(t,e,n,r,i,o){if(!f.isBuffer(t))throw new TypeError('"buffer" argument must be a Buffer instance');if(i<e||e<o)throw new RangeError('"value" argument is out of bounds');if(n+r>t.length)throw new RangeError("Index out of range")}function E(t,e,n,r){e<0&&(e=65535+e+1);for(var i=0,o=Math.min(t.length-n,2);i<o;++i)t[n+i]=(e&255<<8*(r?i:1-i))>>>8*(r?i:1-i)}function O(t,e,n,r){e<0&&(e=4294967295+e+1);for(var i=0,o=Math.min(t.length-n,4);i<o;++i)t[n+i]=e>>>8*(r?i:3-i)&255}function T(t,e,n,r){if(n+r>t.length)throw new RangeError("Index out of range");if(n<0)throw new RangeError("Index out of range")}function I(t,e,n,r,i){return i||T(t,0,n,4),o.write(t,e,n,r,23,4),n+4}function B(t,e,n,r,i){return i||T(t,0,n,8),o.write(t,e,n,r,52,8),n+8}f.prototype.slice=function(t,e){var n,r=this.length;if((t=~~t)<0?(t+=r)<0&&(t=0):r<t&&(t=r),(e=void 0===e?r:~~e)<0?(e+=r)<0&&(e=0):r<e&&(e=r),e<t&&(e=t),f.TYPED_ARRAY_SUPPORT)(n=this.subarray(t,e)).__proto__=f.prototype;else{var i=e-t;n=new f(i,void 0);for(var o=0;o<i;++o)n[o]=this[o+t]}return n},f.prototype.readUIntLE=function(t,e,n){t|=0,e|=0,n||A(t,e,this.length);for(var r=this[t],i=1,o=0;++o<e&&(i*=256);)r+=this[t+o]*i;return r},f.prototype.readUIntBE=function(t,e,n){t|=0,e|=0,n||A(t,e,this.length);for(var r=this[t+--e],i=1;0<e&&(i*=256);)r+=this[t+--e]*i;return r},f.prototype.readUInt8=function(t,e){return e||A(t,1,this.length),this[t]},f.prototype.readUInt16LE=function(t,e){return e||A(t,2,this.length),this[t]|this[t+1]<<8},f.prototype.readUInt16BE=function(t,e){return e||A(t,2,this.length),this[t]<<8|this[t+1]},f.prototype.readUInt32LE=function(t,e){return e||A(t,4,this.length),(this[t]|this[t+1]<<8|this[t+2]<<16)+16777216*this[t+3]},f.prototype.readUInt32BE=function(t,e){return e||A(t,4,this.length),16777216*this[t]+(this[t+1]<<16|this[t+2]<<8|this[t+3])},f.prototype.readIntLE=function(t,e,n){t|=0,e|=0,n||A(t,e,this.length);for(var r=this[t],i=1,o=0;++o<e&&(i*=256);)r+=this[t+o]*i;return(i*=128)<=r&&(r-=Math.pow(2,8*e)),r},f.prototype.readIntBE=function(t,e,n){t|=0,e|=0,n||A(t,e,this.length);for(var r=e,i=1,o=this[t+--r];0<r&&(i*=256);)o+=this[t+--r]*i;return(i*=128)<=o&&(o-=Math.pow(2,8*e)),o},f.prototype.readInt8=function(t,e){return e||A(t,1,this.length),128&this[t]?-1*(255-this[t]+1):this[t]},f.prototype.readInt16LE=function(t,e){e||A(t,2,this.length);var n=this[t]|this[t+1]<<8;return 32768&n?4294901760|n:n},f.prototype.readInt16BE=function(t,e){e||A(t,2,this.length);var n=this[t+1]|this[t]<<8;return 32768&n?4294901760|n:n},f.prototype.readInt32LE=function(t,e){return e||A(t,4,this.length),this[t]|this[t+1]<<8|this[t+2]<<16|this[t+3]<<24},f.prototype.readInt32BE=function(t,e){return e||A(t,4,this.length),this[t]<<24|this[t+1]<<16|this[t+2]<<8|this[t+3]},f.prototype.readFloatLE=function(t,e){return e||A(t,4,this.length),o.read(this,t,!0,23,4)},f.prototype.readFloatBE=function(t,e){return e||A(t,4,this.length),o.read(this,t,!1,23,4)},f.prototype.readDoubleLE=function(t,e){return e||A(t,8,this.length),o.read(this,t,!0,52,8)},f.prototype.readDoubleBE=function(t,e){return e||A(t,8,this.length),o.read(this,t,!1,52,8)},f.prototype.writeUIntLE=function(t,e,n,r){t=+t,e|=0,n|=0,r||P(this,t,e,n,Math.pow(2,8*n)-1,0);var i=1,o=0;for(this[e]=255&t;++o<n&&(i*=256);)this[e+o]=t/i&255;return e+n},f.prototype.writeUIntBE=function(t,e,n,r){t=+t,e|=0,n|=0,r||P(this,t,e,n,Math.pow(2,8*n)-1,0);var i=n-1,o=1;for(this[e+i]=255&t;0<=--i&&(o*=256);)this[e+i]=t/o&255;return e+n},f.prototype.writeUInt8=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,1,255,0),f.TYPED_ARRAY_SUPPORT||(t=Math.floor(t)),this[e]=255&t,e+1},f.prototype.writeUInt16LE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,2,65535,0),f.TYPED_ARRAY_SUPPORT?(this[e]=255&t,this[e+1]=t>>>8):E(this,t,e,!0),e+2},f.prototype.writeUInt16BE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,2,65535,0),f.TYPED_ARRAY_SUPPORT?(this[e]=t>>>8,this[e+1]=255&t):E(this,t,e,!1),e+2},f.prototype.writeUInt32LE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,4,4294967295,0),f.TYPED_ARRAY_SUPPORT?(this[e+3]=t>>>24,this[e+2]=t>>>16,this[e+1]=t>>>8,this[e]=255&t):O(this,t,e,!0),e+4},f.prototype.writeUInt32BE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,4,4294967295,0),f.TYPED_ARRAY_SUPPORT?(this[e]=t>>>24,this[e+1]=t>>>16,this[e+2]=t>>>8,this[e+3]=255&t):O(this,t,e,!1),e+4},f.prototype.writeIntLE=function(t,e,n,r){if(t=+t,e|=0,!r){var i=Math.pow(2,8*n-1);P(this,t,e,n,i-1,-i)}var o=0,a=1,s=0;for(this[e]=255&t;++o<n&&(a*=256);)t<0&&0===s&&0!==this[e+o-1]&&(s=1),this[e+o]=(t/a>>0)-s&255;return e+n},f.prototype.writeIntBE=function(t,e,n,r){if(t=+t,e|=0,!r){var i=Math.pow(2,8*n-1);P(this,t,e,n,i-1,-i)}var o=n-1,a=1,s=0;for(this[e+o]=255&t;0<=--o&&(a*=256);)t<0&&0===s&&0!==this[e+o+1]&&(s=1),this[e+o]=(t/a>>0)-s&255;return e+n},f.prototype.writeInt8=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,1,127,-128),f.TYPED_ARRAY_SUPPORT||(t=Math.floor(t)),t<0&&(t=255+t+1),this[e]=255&t,e+1},f.prototype.writeInt16LE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,2,32767,-32768),f.TYPED_ARRAY_SUPPORT?(this[e]=255&t,this[e+1]=t>>>8):E(this,t,e,!0),e+2},f.prototype.writeInt16BE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,2,32767,-32768),f.TYPED_ARRAY_SUPPORT?(this[e]=t>>>8,this[e+1]=255&t):E(this,t,e,!1),e+2},f.prototype.writeInt32LE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,4,2147483647,-2147483648),f.TYPED_ARRAY_SUPPORT?(this[e]=255&t,this[e+1]=t>>>8,this[e+2]=t>>>16,this[e+3]=t>>>24):O(this,t,e,!0),e+4},f.prototype.writeInt32BE=function(t,e,n){return t=+t,e|=0,n||P(this,t,e,4,2147483647,-2147483648),t<0&&(t=4294967295+t+1),f.TYPED_ARRAY_SUPPORT?(this[e]=t>>>24,this[e+1]=t>>>16,this[e+2]=t>>>8,this[e+3]=255&t):O(this,t,e,!1),e+4},f.prototype.writeFloatLE=function(t,e,n){return I(this,t,e,!0,n)},f.prototype.writeFloatBE=function(t,e,n){return I(this,t,e,!1,n)},f.prototype.writeDoubleLE=function(t,e,n){return B(this,t,e,!0,n)},f.prototype.writeDoubleBE=function(t,e,n){return B(this,t,e,!1,n)},f.prototype.copy=function(t,e,n,r){if(n=n||0,r||0===r||(r=this.length),e>=t.length&&(e=t.length),e=e||0,0<r&&r<n&&(r=n),r===n)return 0;if(0===t.length||0===this.length)return 0;if(e<0)throw new RangeError("targetStart out of bounds");if(n<0||n>=this.length)throw new RangeError("sourceStart out of bounds");if(r<0)throw new RangeError("sourceEnd out of bounds");r>this.length&&(r=this.length),t.length-e<r-n&&(r=t.length-e+n);var i,o=r-n;if(this===t&&n<e&&e<r)for(i=o-1;0<=i;--i)t[i+e]=this[i+n];else if(o<1e3||!f.TYPED_ARRAY_SUPPORT)for(i=0;i<o;++i)t[i+e]=this[i+n];else Uint8Array.prototype.set.call(t,this.subarray(n,n+o),e);return o},f.prototype.fill=function(t,e,n,r){if("string"==typeof t){if("string"==typeof e?(r=e,e=0,n=this.length):"string"==typeof n&&(r=n,n=this.length),1===t.length){var i=t.charCodeAt(0);i<256&&(t=i)}if(void 0!==r&&"string"!=typeof r)throw new TypeError("encoding must be a string");if("string"==typeof r&&!f.isEncoding(r))throw new TypeError("Unknown encoding: "+r)}else"number"==typeof t&&(t&=255);if(e<0||this.length<e||this.length<n)throw new RangeError("Out of range index");if(n<=e)return this;var o;if(e>>>=0,n=void 0===n?this.length:n>>>0,"number"==typeof(t=t||0))for(o=e;o<n;++o)this[o]=t;else{var a=f.isBuffer(t)?t:R(new f(t,r).toString()),s=a.length;for(o=0;o<n-e;++o)this[o+e]=a[o%s]}return this};var e=/[^+\/0-9A-Za-z-_]/g;function L(t){return t<16?"0"+t.toString(16):t.toString(16)}function R(t,e){var n;e=e||1/0;for(var r=t.length,i=null,o=[],a=0;a<r;++a){if(55295<(n=t.charCodeAt(a))&&n<57344){if(!i){if(56319<n){-1<(e-=3)&&o.push(239,191,189);continue}if(a+1===r){-1<(e-=3)&&o.push(239,191,189);continue}i=n;continue}if(n<56320){-1<(e-=3)&&o.push(239,191,189),i=n;continue}n=65536+(i-55296<<10|n-56320)}else i&&-1<(e-=3)&&o.push(239,191,189);if(i=null,n<128){if((e-=1)<0)break;o.push(n)}else if(n<2048){if((e-=2)<0)break;o.push(n>>6|192,63&n|128)}else if(n<65536){if((e-=3)<0)break;o.push(n>>12|224,n>>6&63|128,63&n|128)}else{if(!(n<1114112))throw new Error("Invalid code point");if((e-=4)<0)break;o.push(n>>18|240,n>>12&63|128,n>>6&63|128,63&n|128)}}return o}function M(t){return r.toByteArray(function(t){if((t=function(t){return t.trim?t.trim():t.replace(/^\s+|\s+$/g,"")}(t).replace(e,"")).length<2)return"";for(;t.length%4!=0;)t+="=";return t}(t))}function F(t,e,n,r){for(var i=0;i<r&&!(i+n>=e.length||i>=t.length);++i)e[i+n]=t[i];return i}}).call(this,z(25))},function(t,e,n){var _;t.exports=(_=n(1),n(34),void(_.lib.Cipher||function(){var t=_,e=t.lib,n=e.Base,l=e.WordArray,r=e.BufferedBlockAlgorithm,i=t.enc,o=(i.Utf8,i.Base64),a=t.algo,s=a.EvpKDF,u=e.Cipher=r.extend({cfg:n.extend(),createEncryptor:function(t,e){return this.create(this._ENC_XFORM_MODE,t,e)},createDecryptor:function(t,e){return this.create(this._DEC_XFORM_MODE,t,e)},init:function(t,e,n){this.cfg=this.cfg.extend(n),this._xformMode=t,this._key=e,this.reset()},reset:function(){r.reset.call(this),this._doReset()},process:function(t){return this._append(t),this._process()},finalize:function(t){t&&this._append(t);var e=this._doFinalize();return e},keySize:4,ivSize:4,_ENC_XFORM_MODE:1,_DEC_XFORM_MODE:2,_createHelper:function(){function i(t){return"string"==typeof t?x:b}return function(r){return{encrypt:function(t,e,n){return i(e).encrypt(r,t,e,n)},decrypt:function(t,e,n){return i(e).decrypt(r,t,e,n)}}}}()}),c=(e.StreamCipher=u.extend({_doFinalize:function(){var t=this._process(!0);return t},blockSize:1}),t.mode={}),f=e.BlockCipherMode=n.extend({createEncryptor:function(t,e){return this.Encryptor.create(t,e)},createDecryptor:function(t,e){return this.Decryptor.create(t,e)},init:function(t,e){this._cipher=t,this._iv=e}}),h=c.CBC=function(){var t=f.extend();function o(t,e,n){var r=this._iv;if(r){var i=r;this._iv=void 0}else var i=this._prevBlock;for(var o=0;o<n;o++)t[e+o]^=i[o]}return t.Encryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize;o.call(this,t,e,r),n.encryptBlock(t,e),this._prevBlock=t.slice(e,e+r)}}),t.Decryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize,i=t.slice(e,e+r);n.decryptBlock(t,e),o.call(this,t,e,r),this._prevBlock=i}}),t}(),d=t.pad={},p=d.Pkcs7={pad:function(t,e){for(var n=4*e,r=n-t.sigBytes%n,i=r<<24|r<<16|r<<8|r,o=[],a=0;a<r;a+=4)o.push(i);var s=l.create(o,r);t.concat(s)},unpad:function(t){var e=255&t.words[t.sigBytes-1>>>2];t.sigBytes-=e}},g=(e.BlockCipher=u.extend({cfg:u.cfg.extend({mode:h,padding:p}),reset:function(){u.reset.call(this);var t=this.cfg,e=t.iv,n=t.mode;if(this._xformMode==this._ENC_XFORM_MODE)var r=n.createEncryptor;else{var r=n.createDecryptor;this._minBufferSize=1}this._mode&&this._mode.__creator==r?this._mode.init(this,e&&e.words):(this._mode=r.call(n,this,e&&e.words),this._mode.__creator=r)},_doProcessBlock:function(t,e){this._mode.processBlock(t,e)},_doFinalize:function(){var t=this.cfg.padding;if(this._xformMode==this._ENC_XFORM_MODE){t.pad(this._data,this.blockSize);var e=this._process(!0)}else{var e=this._process(!0);t.unpad(e)}return e},blockSize:4}),e.CipherParams=n.extend({init:function(t){this.mixIn(t)},toString:function(t){return(t||this.formatter).stringify(this)}})),v=t.format={},y=v.OpenSSL={stringify:function(t){var e=t.ciphertext,n=t.salt;if(n)var r=l.create([1398893684,1701076831]).concat(n).concat(e);else var r=e;return r.toString(o)},parse:function(t){var e=o.parse(t),n=e.words;if(1398893684==n[0]&&1701076831==n[1]){var r=l.create(n.slice(2,4));n.splice(0,4),e.sigBytes-=16}return g.create({ciphertext:e,salt:r})}},b=e.SerializableCipher=n.extend({cfg:n.extend({format:y}),encrypt:function(t,e,n,r){r=this.cfg.extend(r);var i=t.createEncryptor(n,r),o=i.finalize(e),a=i.cfg;return g.create({ciphertext:o,key:n,iv:a.iv,algorithm:t,mode:a.mode,padding:a.padding,blockSize:t.blockSize,formatter:r.format})},decrypt:function(t,e,n,r){r=this.cfg.extend(r),e=this._parse(e,r.format);var i=t.createDecryptor(n,r).finalize(e.ciphertext);return i},_parse:function(t,e){return"string"==typeof t?e.parse(t,this):t}}),m=t.kdf={},w=m.OpenSSL={execute:function(t,e,n,r){r=r||l.random(8);var i=s.create({keySize:e+n}).compute(t,r),o=l.create(i.words.slice(e),4*n);return i.sigBytes=4*e,g.create({key:i,iv:o,salt:r})}},x=e.PasswordBasedCipher=b.extend({cfg:b.cfg.extend({kdf:w}),encrypt:function(t,e,n,r){var i=(r=this.cfg.extend(r)).kdf.execute(n,t.keySize,t.ivSize);r.iv=i.iv;var o=b.encrypt.call(this,t,e,i.key,r);return o.mixIn(i),o},decrypt:function(t,e,n,r){r=this.cfg.extend(r),e=this._parse(e,r.format);var i=r.kdf.execute(n,t.keySize,t.ivSize,e.salt);r.iv=i.iv;var o=b.decrypt.call(this,t,e,i.key,r);return o}})}()))},function(t,e,n){var g=n(8),v=n(39),y=n(15),b=n(22),m=n(54),w="prototype",x=function(t,e,n){var r,i,o,a,s=t&x.F,l=t&x.G,u=t&x.S,c=t&x.P,f=t&x.B,h=l?g:u?g[e]||(g[e]={}):(g[e]||{})[w],d=l?v:v[e]||(v[e]={}),p=d[w]||(d[w]={});for(r in l&&(n=e),n)o=((i=!s&&h&&void 0!==h[r])?h:n)[r],a=f&&i?m(o,g):c&&"function"==typeof o?m(Function.call,o):o,h&&b(h,r,o,t&x.U),d[r]!=o&&y(d,r,a),c&&p[r]!=o&&(p[r]=o)};g.core=v,x.F=1,x.G=2,x.S=4,x.P=8,x.B=16,x.W=32,x.U=64,x.R=128,t.exports=x},function(t,e,n){var g=n(21),v=n(2),y=n(38),b=n(27),m=n(36),w="prototype",x=function(t,e,n){var r,i,o,a=t&x.F,s=t&x.G,l=t&x.S,u=t&x.P,c=t&x.B,f=t&x.W,h=s?v:v[e]||(v[e]={}),d=h[w],p=s?g:l?g[e]:(g[e]||{})[w];for(r in s&&(n=e),n)(i=!a&&p&&void 0!==p[r])&&m(h,r)||(o=i?p[r]:n[r],h[r]=s&&"function"!=typeof p[r]?n[r]:c&&i?y(o,g):f&&p[r]==o?function(r){function t(t,e,n){if(this instanceof r){switch(arguments.length){case 0:return new r;case 1:return new r(t);case 2:return new r(t,e)}return new r(t,e,n)}return r.apply(this,arguments)}return t[w]=r[w],t}(o):u&&"function"==typeof o?y(Function.call,o):o,u&&((h.virtual||(h.virtual={}))[r]=o,t&x.R&&d&&!d[r]&&b(d,r,o)))};x.F=1,x.G=2,x.S=4,x.P=8,x.B=16,x.W=32,x.U=64,x.R=128,t.exports=x},function(t,e){var n=t.exports="undefined"!=typeof window&&window.Math==Math?window:"undefined"!=typeof self&&self.Math==Math?self:Function("return this")();"number"==typeof __g&&(__g=n)},function(t,e,n){t.exports=!n(10)(function(){return 7!=Object.defineProperty({},"a",{get:function(){return 7}}).a})},function(t,e){t.exports=function(t){try{return!!t()}catch(t){return!0}}},function(t,e,n){var r=n(13),i=n(131),o=n(53),a=Object.defineProperty;e.f=n(9)?Object.defineProperty:function(t,e,n){if(r(t),e=o(e,!0),r(n),i)try{return a(t,e,n)}catch(t){}if("get"in n||"set"in n)throw TypeError("Accessors not supported!");return"value"in n&&(t[e]=n.value),t}},function(t,e,n){var r=n(115)("wks"),i=n(77),o=n(21).Symbol,a="function"==typeof o;(t.exports=function(t){return r[t]||(r[t]=a&&o[t]||(a?o:i)("Symbol."+t))}).store=r},function(t,e,n){var r=n(18);t.exports=function(t){if(!r(t))throw TypeError(t+" is not an object!");return t}},function(t,e,n){t.exports=!n(37)(function(){return 7!=Object.defineProperty({},"a",{get:function(){return 7}}).a})},function(t,e,n){var r=n(11),i=n(40);t.exports=n(9)?function(t,e,n){return r.f(t,e,i(1,n))}:function(t,e,n){return t[e]=n,t}},function(t,e,n){var r=n(31),i=Math.min;t.exports=function(t){return 0<t?i(r(t),9007199254740991):0}},function(t,e,n){var r=n(28),i=n(164),o=n(109),a=Object.defineProperty;e.f=n(14)?Object.defineProperty:function(t,e,n){if(r(t),e=o(e,!0),r(n),i)try{return a(t,e,n)}catch(t){}if("get"in n||"set"in n)throw TypeError("Accessors not supported!");return"value"in n&&(t[e]=n.value),t}},function(t,e){t.exports=function(t){return"object"==typeof t?null!==t:"function"==typeof t}},function(t,e,n){var r=n(30);t.exports=function(t){return Object(r(t))}},function(t,e){t.exports=function(t){return"object"==typeof t?null!==t:"function"==typeof t}},function(t,e){var n=t.exports="undefined"!=typeof window&&window.Math==Math?window:"undefined"!=typeof self&&self.Math==Math?self:Function("return this")();"number"==typeof __g&&(__g=n)},function(t,e,n){var o=n(8),a=n(15),s=n(23),l=n(29)("src"),r=n(205),i="toString",u=(""+r).split(i);n(39).inspectSource=function(t){return r.call(t)},(t.exports=function(t,e,n,r){var i="function"==typeof n;i&&(s(n,"name")||a(n,"name",e)),t[e]!==n&&(i&&(s(n,l)||a(n,l,t[e]?""+t[e]:u.join(String(e)))),t===o?t[e]=n:r?t[e]?t[e]=n:a(t,e,n):(delete t[e],a(t,e,n)))})(Function.prototype,i,function(){return"function"==typeof this&&this[l]||r.call(this)})},function(t,e){var n={}.hasOwnProperty;t.exports=function(t,e){return n.call(t,e)}},function(t,e){var n,r,i=t.exports={};function o(){throw new Error("setTimeout has not been defined")}function a(){throw new Error("clearTimeout has not been defined")}function s(e){if(n===setTimeout)return setTimeout(e,0);if((n===o||!n)&&setTimeout)return n=setTimeout,setTimeout(e,0);try{return n(e,0)}catch(t){try{return n.call(null,e,0)}catch(t){return n.call(this,e,0)}}}!function(){try{n="function"==typeof setTimeout?setTimeout:o}catch(t){n=o}try{r="function"==typeof clearTimeout?clearTimeout:a}catch(t){r=a}}();var l,u=[],c=!1,f=-1;function h(){c&&l&&(c=!1,l.length?u=l.concat(u):f=-1,u.length&&d())}function d(){if(!c){var t=s(h);c=!0;for(var e=u.length;e;){for(l=u,u=[];++f<e;)l&&l[f].run();f=-1,e=u.length}l=null,c=!1,function(e){if(r===clearTimeout)return clearTimeout(e);if((r===a||!r)&&clearTimeout)return r=clearTimeout,clearTimeout(e);try{r(e)}catch(t){try{return r.call(null,e)}catch(t){return r.call(this,e)}}}(t)}}function p(t,e){this.fun=t,this.array=e}function g(){}i.nextTick=function(t){var e=new Array(arguments.length-1);if(1<arguments.length)for(var n=1;n<arguments.length;n++)e[n-1]=arguments[n];u.push(new p(t,e)),1!==u.length||c||s(d)},p.prototype.run=function(){this.fun.apply(null,this.array)},i.title="browser",i.browser=!0,i.env={},i.argv=[],i.version="",i.versions={},i.on=g,i.addListener=g,i.once=g,i.off=g,i.removeListener=g,i.removeAllListeners=g,i.emit=g,i.prependListener=g,i.prependOnceListener=g,i.listeners=function(t){return[]},i.binding=function(t){throw new Error("process.binding is not supported")},i.cwd=function(){return"/"},i.chdir=function(t){throw new Error("process.chdir is not supported")},i.umask=function(){return 0}},function(t,e){var n;n=function(){return this}();try{n=n||new Function("return this")()}catch(t){"object"==typeof window&&(n=window)}t.exports=n},function(t,e,n){(function(){var i,t;i=n(48).Number,e.resolveLength=function(t,e,n){var r;if("number"==typeof t?r=t:"function"==typeof t?r=t.call(n,n):n&&"string"==typeof t?r=n[t]:e&&t instanceof i&&(r=t.decode(e)),isNaN(r))throw new Error("Not a fixed size");return r},t=function(t){var e,n;for(e in null==t&&(t={}),this.enumerable=!0,this.configurable=!0,t)n=t[e],this[e]=n},e.PropertyDescriptor=t}).call(this)},function(t,e,n){var r=n(17),i=n(59);t.exports=n(14)?function(t,e,n){return r.f(t,e,i(1,n))}:function(t,e,n){return t[e]=n,t}},function(t,e,n){var r=n(20);t.exports=function(t){if(!r(t))throw TypeError(t+" is not an object!");return t}},function(t,e){var n=0,r=Math.random();t.exports=function(t){return"Symbol(".concat(void 0===t?"":t,")_",(++n+r).toString(36))}},function(t,e){t.exports=function(t){if(null==t)throw TypeError("Can't call method on  "+t);return t}},function(t,e){var n=Math.ceil,r=Math.floor;t.exports=function(t){return isNaN(t=+t)?0:(0<t?r:n)(t)}},function(t,e){"function"==typeof Object.create?t.exports=function(t,e){t.super_=e,t.prototype=Object.create(e.prototype,{constructor:{value:t,enumerable:!1,writable:!0,configurable:!0}})}:t.exports=function(t,e){t.super_=e;function n(){}n.prototype=e.prototype,t.prototype=new n,t.prototype.constructor=t}},function(t,e,n){"use strict";var r=n(69),i=Object.keys||function(t){var e=[];for(var n in t)e.push(n);return e};t.exports=f;var o=n(58);o.inherits=n(32);var a=n(150),s=n(97);o.inherits(f,a);for(var l=i(s.prototype),u=0;u<l.length;u++){var c=l[u];f.prototype[c]||(f.prototype[c]=s.prototype[c])}function f(t){if(!(this instanceof f))return new f(t);a.call(this,t),s.call(this,t),t&&!1===t.readable&&(this.readable=!1),t&&!1===t.writable&&(this.writable=!1),this.allowHalfOpen=!0,t&&!1===t.allowHalfOpen&&(this.allowHalfOpen=!1),this.once("end",h)}function h(){this.allowHalfOpen||this._writableState.ended||r.nextTick(d,this)}function d(t){t.end()}Object.defineProperty(f.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Object.defineProperty(f.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed&&this._writableState.destroyed)},set:function(t){void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed=t,this._writableState.destroyed=t)}}),f.prototype._destroy=function(t,e){this.push(null),this.end(),r.nextTick(e,t)}},function(t,e,n){var a;t.exports=(a=n(1),n(100),n(101),function(){var t=a,e=t.lib,n=e.Base,c=e.WordArray,r=t.algo,i=r.MD5,o=r.EvpKDF=n.extend({cfg:n.extend({keySize:4,hasher:i,iterations:1}),init:function(t){this.cfg=this.cfg.extend(t)},compute:function(t,e){for(var n=this.cfg,r=n.hasher.create(),i=c.create(),o=i.words,a=n.keySize,s=n.iterations;o.length<a;){l&&r.update(l);var l=r.update(t).finalize(e);r.reset();for(var u=1;u<s;u++)l=r.finalize(l),r.reset();i.concat(l)}return i.sigBytes=4*a,i}});t.EvpKDF=function(t,e,n){return o.create(n).compute(t,e)}}(),a.EvpKDF)},function(t,e,n){var r=n(105),i=n(107);t.exports=function(t){return r(i(t))}},function(t,e){var n={}.hasOwnProperty;t.exports=function(t,e){return n.call(t,e)}},function(t,e){t.exports=function(t){try{return!!t()}catch(t){return!0}}},function(t,e,n){var o=n(166);t.exports=function(r,i,t){if(o(r),void 0===i)return r;switch(t){case 1:return function(t){return r.call(i,t)};case 2:return function(t,e){return r.call(i,t,e)};case 3:return function(t,e,n){return r.call(i,t,e,n)}}return function(){return r.apply(i,arguments)}}},function(t,e){var n=t.exports={version:"2.6.9"};"number"==typeof __e&&(__e=n)},function(t,e){t.exports=function(t,e){return{enumerable:!(1&t),configurable:!(2&t),writable:!(4&t),value:e}}},function(t,e){t.exports=!1},function(t,e,n){var r=n(134),i=n(84);t.exports=Object.keys||function(t){return r(t,i)}},function(t,e,n){var r=n(81),i=n(30);t.exports=function(t){return r(i(t))}},function(t,e){t.exports={}},function(t,e,n){var r;t.exports=(r=n(1),function(){var t=r,l=t.lib.WordArray;t.enc.Base64={stringify:function(t){var e=t.words,n=t.sigBytes,r=this._map;t.clamp();for(var i=[],o=0;o<n;o+=3)for(var a=(e[o>>>2]>>>24-o%4*8&255)<<16|(e[o+1>>>2]>>>24-(o+1)%4*8&255)<<8|e[o+2>>>2]>>>24-(o+2)%4*8&255,s=0;s<4&&o+.75*s<n;s++)i.push(r.charAt(a>>>6*(3-s)&63));var l=r.charAt(64);if(l)for(;i.length%4;)i.push(l);return i.join("")},parse:function(t){var e=t.length,n=this._map,r=this._reverseMap;if(!r){r=this._reverseMap=[];for(var i=0;i<n.length;i++)r[n.charCodeAt(i)]=i}var o=n.charAt(64);if(o){var a=t.indexOf(o);-1!==a&&(e=a)}return function(t,e,n){for(var r=[],i=0,o=0;o<e;o++)if(o%4){var a=n[t.charCodeAt(o-1)]<<o%4*2,s=n[t.charCodeAt(o)]>>>6-o%4*2;r[i>>>2]|=(a|s)<<24-i%4*8,i++}return l.create(r,i)}(t,e,r)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="}}(),r.enc.Base64)},function(t,e,n){var a;t.exports=(a=n(1),function(c){var t=a,e=t.lib,n=e.WordArray,r=e.Hasher,i=t.algo,A=[];!function(){for(var t=0;t<64;t++)A[t]=4294967296*c.abs(c.sin(t+1))|0}();var o=i.MD5=r.extend({_doReset:function(){this._hash=new n.init([1732584193,4023233417,2562383102,271733878])},_doProcessBlock:function(t,e){for(var n=0;n<16;n++){var r=e+n,i=t[r];t[r]=16711935&(i<<8|i>>>24)|4278255360&(i<<24|i>>>8)}var o=this._hash.words,a=t[e+0],s=t[e+1],l=t[e+2],u=t[e+3],c=t[e+4],f=t[e+5],h=t[e+6],d=t[e+7],p=t[e+8],g=t[e+9],v=t[e+10],y=t[e+11],b=t[e+12],m=t[e+13],w=t[e+14],x=t[e+15],_=o[0],k=o[1],S=o[2],C=o[3];k=T(k=T(k=T(k=T(k=O(k=O(k=O(k=O(k=E(k=E(k=E(k=E(k=P(k=P(k=P(k=P(k,S=P(S,C=P(C,_=P(_,k,S,C,a,7,A[0]),k,S,s,12,A[1]),_,k,l,17,A[2]),C,_,u,22,A[3]),S=P(S,C=P(C,_=P(_,k,S,C,c,7,A[4]),k,S,f,12,A[5]),_,k,h,17,A[6]),C,_,d,22,A[7]),S=P(S,C=P(C,_=P(_,k,S,C,p,7,A[8]),k,S,g,12,A[9]),_,k,v,17,A[10]),C,_,y,22,A[11]),S=P(S,C=P(C,_=P(_,k,S,C,b,7,A[12]),k,S,m,12,A[13]),_,k,w,17,A[14]),C,_,x,22,A[15]),S=E(S,C=E(C,_=E(_,k,S,C,s,5,A[16]),k,S,h,9,A[17]),_,k,y,14,A[18]),C,_,a,20,A[19]),S=E(S,C=E(C,_=E(_,k,S,C,f,5,A[20]),k,S,v,9,A[21]),_,k,x,14,A[22]),C,_,c,20,A[23]),S=E(S,C=E(C,_=E(_,k,S,C,g,5,A[24]),k,S,w,9,A[25]),_,k,u,14,A[26]),C,_,p,20,A[27]),S=E(S,C=E(C,_=E(_,k,S,C,m,5,A[28]),k,S,l,9,A[29]),_,k,d,14,A[30]),C,_,b,20,A[31]),S=O(S,C=O(C,_=O(_,k,S,C,f,4,A[32]),k,S,p,11,A[33]),_,k,y,16,A[34]),C,_,w,23,A[35]),S=O(S,C=O(C,_=O(_,k,S,C,s,4,A[36]),k,S,c,11,A[37]),_,k,d,16,A[38]),C,_,v,23,A[39]),S=O(S,C=O(C,_=O(_,k,S,C,m,4,A[40]),k,S,a,11,A[41]),_,k,u,16,A[42]),C,_,h,23,A[43]),S=O(S,C=O(C,_=O(_,k,S,C,g,4,A[44]),k,S,b,11,A[45]),_,k,x,16,A[46]),C,_,l,23,A[47]),S=T(S,C=T(C,_=T(_,k,S,C,a,6,A[48]),k,S,d,10,A[49]),_,k,w,15,A[50]),C,_,f,21,A[51]),S=T(S,C=T(C,_=T(_,k,S,C,b,6,A[52]),k,S,u,10,A[53]),_,k,v,15,A[54]),C,_,s,21,A[55]),S=T(S,C=T(C,_=T(_,k,S,C,p,6,A[56]),k,S,x,10,A[57]),_,k,h,15,A[58]),C,_,m,21,A[59]),S=T(S,C=T(C,_=T(_,k,S,C,c,6,A[60]),k,S,y,10,A[61]),_,k,l,15,A[62]),C,_,g,21,A[63]),o[0]=o[0]+_|0,o[1]=o[1]+k|0,o[2]=o[2]+S|0,o[3]=o[3]+C|0},_doFinalize:function(){var t=this._data,e=t.words,n=8*this._nDataBytes,r=8*t.sigBytes;e[r>>>5]|=128<<24-r%32;var i=c.floor(n/4294967296),o=n;e[15+(64+r>>>9<<4)]=16711935&(i<<8|i>>>24)|4278255360&(i<<24|i>>>8),e[14+(64+r>>>9<<4)]=16711935&(o<<8|o>>>24)|4278255360&(o<<24|o>>>8),t.sigBytes=4*(e.length+1),this._process();for(var a=this._hash,s=a.words,l=0;l<4;l++){var u=s[l];s[l]=16711935&(u<<8|u>>>24)|4278255360&(u<<24|u>>>8)}return a},clone:function(){var t=r.clone.call(this);return t._hash=this._hash.clone(),t}});function P(t,e,n,r,i,o,a){var s=t+(e&n|~e&r)+i+a;return(s<<o|s>>>32-o)+e}function E(t,e,n,r,i,o,a){var s=t+(e&r|n&~r)+i+a;return(s<<o|s>>>32-o)+e}function O(t,e,n,r,i,o,a){var s=t+(e^n^r)+i+a;return(s<<o|s>>>32-o)+e}function T(t,e,n,r,i,o,a){var s=t+(n^(e|~r))+i+a;return(s<<o|s>>>32-o)+e}t.MD5=r._createHelper(o),t.HmacMD5=r._createHmacHelper(o)}(Math),a.MD5)},function(a,t,s){"use strict";(function(t){var e,n=s(4),i=n.Buffer,r={};for(e in n)n.hasOwnProperty(e)&&"SlowBuffer"!==e&&"Buffer"!==e&&(r[e]=n[e]);var o=r.Buffer={};for(e in i)i.hasOwnProperty(e)&&"allocUnsafe"!==e&&"allocUnsafeSlow"!==e&&(o[e]=i[e]);if(r.Buffer.prototype=i.prototype,o.from&&o.from!==Uint8Array.from||(o.from=function(t,e,n){if("number"==typeof t)throw new TypeError('The "value" argument must not be of type number. Received type '+typeof t);if(t&&void 0===t.length)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof t);return i(t,e,n)}),o.alloc||(o.alloc=function(t,e,n){if("number"!=typeof t)throw new TypeError('The "size" argument must be of type number. Received type '+typeof t);if(t<0||2*(1<<30)<=t)throw new RangeError('The value "'+t+'" is invalid for option "size"');var r=i(t);return e&&0!==e.length?"string"==typeof n?r.fill(e,n):r.fill(e):r.fill(0),r}),!r.kStringMaxLength)try{r.kStringMaxLength=t.binding("buffer").kStringMaxLength}catch(t){}r.constants||(r.constants={MAX_LENGTH:r.kMaxLength},r.kStringMaxLength&&(r.constants.MAX_STRING_LENGTH=r.kStringMaxLength)),a.exports=r}).call(this,s(24))},function(t,a,s){(function(){var t,e,n,i={}.hasOwnProperty;function r(t,e){this.type=t,this.endian=null!=e?e:"BE",this.fn=this.type,"8"!==this.type[this.type.length-1]&&(this.fn+=this.endian)}function o(t,e,n){null==n&&(n=t>>1),o.__super__.constructor.call(this,"Int"+t,e),this._point=1<<n}t=s(102),r.prototype.size=function(){return t.TYPES[this.type]},r.prototype.decode=function(t){return t["read"+this.fn]()},r.prototype.encode=function(t,e){return t["write"+this.fn](e)},n=r,a.Number=n,a.uint8=new n("UInt8"),a.uint16be=a.uint16=new n("UInt16","BE"),a.uint16le=new n("UInt16","LE"),a.uint24be=a.uint24=new n("UInt24","BE"),a.uint24le=new n("UInt24","LE"),a.uint32be=a.uint32=new n("UInt32","BE"),a.uint32le=new n("UInt32","LE"),a.int8=new n("Int8"),a.int16be=a.int16=new n("Int16","BE"),a.int16le=new n("Int16","LE"),a.int24be=a.int24=new n("Int24","BE"),a.int24le=new n("Int24","LE"),a.int32be=a.int32=new n("Int32","BE"),a.int32le=new n("Int32","LE"),a.floatbe=a.float=new n("Float","BE"),a.floatle=new n("Float","LE"),a.doublebe=a.double=new n("Double","BE"),a.doublele=new n("Double","LE"),function(t,e){for(var n in e)i.call(e,n)&&(t[n]=e[n]);function r(){this.constructor=t}r.prototype=e.prototype,t.prototype=new r,t.__super__=e.prototype}(o,n),o.prototype.decode=function(t){return o.__super__.decode.call(this,t)/this._point},o.prototype.encode=function(t,e){return o.__super__.encode.call(this,t,e*this._point|0)},e=o,a.Fixed=e,a.fixed16be=a.fixed16=new e(16,"BE"),a.fixed16le=new e(16,"LE"),a.fixed32be=a.fixed32=new e(32,"BE"),a.fixed32le=new e(32,"LE")}).call(this)},function(t,e){t.exports={}},function(t,e,n){var r=n(107);t.exports=function(t){return Object(r(t))}},function(t,e,n){"use strict";var r=n(338)(!0);n(112)(String,"String",function(t){this._t=String(t),this._i=0},function(){var t,e=this._t,n=this._i;return n>=e.length?{value:void 0,done:!0}:(t=r(e,n),this._i+=t.length,{value:t,done:!1})})},function(n,t,e){"use strict";(function(i,e){function t(){this.fileSystem={},this.dataSystem={}}function o(t){return 0===t.indexOf(e)&&(t=t.substring(e.length)),0===t.indexOf("/")&&(t=t.substring(1)),t}t.prototype.readFileSync=function(t,e){t=o(t);var n=this.dataSystem[t];if("string"==typeof n&&"utf8"===e)return n;if(n)return new i(n,"string"==typeof n?"base64":void 0);var r=this.fileSystem[t];if(r)return r;throw"File '"+t+"' not found in virtual file system"},t.prototype.writeFileSync=function(t,e){this.fileSystem[o(t)]=e},t.prototype.bindFS=function(t){this.dataSystem=t||{}},n.exports=new t}).call(this,e(4).Buffer,"/")},function(t,e,n){var i=n(18);t.exports=function(t,e){if(!i(t))return t;var n,r;if(e&&"function"==typeof(n=t.toString)&&!i(r=n.call(t)))return r;if("function"==typeof(n=t.valueOf)&&!i(r=n.call(t)))return r;if(!e&&"function"==typeof(n=t.toString)&&!i(r=n.call(t)))return r;throw TypeError("Can't convert object to primitive value")}},function(t,e,n){var o=n(133);t.exports=function(r,i,t){if(o(r),void 0===i)return r;switch(t){case 1:return function(t){return r.call(i,t)};case 2:return function(t,e){return r.call(i,t,e)};case 3:return function(t,e,n){return r.call(i,t,e,n)}}return function(){return r.apply(i,arguments)}}},function(t,e){var n={}.toString;t.exports=function(t){return n.call(t).slice(8,-1)}},function(t,e,n){var r=n(31),i=Math.max,o=Math.min;t.exports=function(t,e){return(t=r(t))<0?i(t+e,0):o(t,e)}},function(t,e,n){var r=n(134),i=n(84).concat("length","prototype");e.f=Object.getOwnPropertyNames||function(t){return r(t,i)}},function(t,n,e){(function(t){function e(t){return Object.prototype.toString.call(t)}n.isArray=function(t){return Array.isArray?Array.isArray(t):"[object Array]"===e(t)},n.isBoolean=function(t){return"boolean"==typeof t},n.isNull=function(t){return null===t},n.isNullOrUndefined=function(t){return null==t},n.isNumber=function(t){return"number"==typeof t},n.isString=function(t){return"string"==typeof t},n.isSymbol=function(t){return"symbol"==typeof t},n.isUndefined=function(t){return void 0===t},n.isRegExp=function(t){return"[object RegExp]"===e(t)},n.isObject=function(t){return"object"==typeof t&&null!==t},n.isDate=function(t){return"[object Date]"===e(t)},n.isError=function(t){return"[object Error]"===e(t)||t instanceof Error},n.isFunction=function(t){return"function"==typeof t},n.isPrimitive=function(t){return null===t||"boolean"==typeof t||"number"==typeof t||"string"==typeof t||"symbol"==typeof t||void 0===t},n.isBuffer=t.isBuffer}).call(this,e(4).Buffer)},function(t,e){t.exports=function(t,e){return{enumerable:!(1&t),configurable:!(2&t),writable:!(4&t),value:e}}},function(t,e,n){n(332);for(var r=n(21),i=n(27),o=n(49),a=n(12)("toStringTag"),s="CSSRuleList,CSSStyleDeclaration,CSSValueList,ClientRectList,DOMRectList,DOMStringList,DOMTokenList,DataTransferItemList,FileList,HTMLAllCollection,HTMLCollection,HTMLFormElement,HTMLSelectElement,MediaList,MimeTypeArray,NamedNodeMap,NodeList,PaintRequestList,Plugin,PluginArray,SVGLengthList,SVGNumberList,SVGPathSegList,SVGPointList,SVGStringList,SVGTransformList,SourceBufferList,StyleSheetList,TextTrackCueList,TextTrackList,TouchList".split(","),l=0;l<s.length;l++){var u=s[l],c=r[u],f=c&&c.prototype;f&&!f[a]&&i(f,a,u),o[u]=o.Array}},function(t,e,n){var r=n(170),i=n(116);t.exports=Object.keys||function(t){return r(t,i)}},function(t,e,n){var r=n(39),i=n(8),o="__core-js_shared__",a=i[o]||(i[o]={});(t.exports=function(t,e){return a[t]||(a[t]=void 0!==e?e:{})})("versions",[]).push({version:r.version,mode:n(41)?"pure":"global",copyright:"© 2019 Denis Pushkarev (zloirock.ru)"})},function(t,e){e.f={}.propertyIsEnumerable},function(t,e,n){var r=n(11).f,i=n(23),o=n(3)("toStringTag");t.exports=function(t,e,n){t&&!i(t=n?t:t.prototype,o)&&r(t,o,{configurable:!0,value:e})}},function(t,e,n){var i=n(55),o=n(3)("toStringTag"),a="Arguments"==i(function(){return arguments}());t.exports=function(t){var e,n,r;return void 0===t?"Undefined":null===t?"Null":"string"==typeof(n=function(t,e){try{return t[e]}catch(t){}}(e=Object(t),o))?n:a?i(e):"Object"==(r=i(e))&&"function"==typeof e.callee?"Arguments":r}},function(t,e,r){function i(){}var o=r(13),a=r(213),s=r(84),l=r(83)("IE_PROTO"),u="prototype",c=function(){var t,e=r(132)("iframe"),n=s.length;for(e.style.display="none",r(214).appendChild(e),e.src="javascript:",(t=e.contentWindow.document).open(),t.write("<script>document.F=Object<\/script>"),t.close(),c=t.F;n--;)delete c[u][s[n]];return c()};t.exports=Object.create||function(t,e){var n;return null!==t?(i[u]=o(t),n=new i,i[u]=null,n[l]=t):n=c(),void 0===e?n:a(n,e)}},function(t,e,n){var r=n(63),i=n(40),o=n(43),a=n(53),s=n(23),l=n(131),u=Object.getOwnPropertyDescriptor;e.f=n(9)?u:function(t,e){if(t=o(t),e=a(e,!0),l)try{return u(t,e)}catch(t){}if(s(t,e))return i(!r.f.call(t,e),t[e])}},function(t,e,n){"use strict";var r,i="object"==typeof Reflect?Reflect:null,c=i&&"function"==typeof i.apply?i.apply:function(t,e,n){return Function.prototype.apply.call(t,e,n)};r=i&&"function"==typeof i.ownKeys?i.ownKeys:Object.getOwnPropertySymbols?function(t){return Object.getOwnPropertyNames(t).concat(Object.getOwnPropertySymbols(t))}:function(t){return Object.getOwnPropertyNames(t)};var o=Number.isNaN||function(t){return t!=t};function a(){a.init.call(this)}((t.exports=a).EventEmitter=a).prototype._events=void 0,a.prototype._eventsCount=0,a.prototype._maxListeners=void 0;var s=10;function l(t){return void 0===t._maxListeners?a.defaultMaxListeners:t._maxListeners}function u(t,e,n,r){var i,o,a;if("function"!=typeof n)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof n);if(void 0===(o=t._events)?(o=t._events=Object.create(null),t._eventsCount=0):(void 0!==o.newListener&&(t.emit("newListener",e,n.listener?n.listener:n),o=t._events),a=o[e]),void 0===a)a=o[e]=n,++t._eventsCount;else if("function"==typeof a?a=o[e]=r?[n,a]:[a,n]:r?a.unshift(n):a.push(n),0<(i=l(t))&&a.length>i&&!a.warned){a.warned=!0;var s=new Error("Possible EventEmitter memory leak detected. "+a.length+" "+String(e)+" listeners added. Use emitter.setMaxListeners() to increase limit");s.name="MaxListenersExceededWarning",s.emitter=t,s.type=e,s.count=a.length,console&&console.warn}return t}function f(t,e,n){var r={fired:!1,wrapFn:void 0,target:t,type:e,listener:n},i=function(){for(var t=[],e=0;e<arguments.length;e++)t.push(arguments[e]);this.fired||(this.target.removeListener(this.type,this.wrapFn),this.fired=!0,c(this.listener,this.target,t))}.bind(r);return i.listener=n,r.wrapFn=i}function h(t,e,n){var r=t._events;if(void 0===r)return[];var i=r[e];return void 0===i?[]:"function"==typeof i?n?[i.listener||i]:[i]:n?function(t){for(var e=new Array(t.length),n=0;n<e.length;++n)e[n]=t[n].listener||t[n];return e}(i):p(i,i.length)}function d(t){var e=this._events;if(void 0!==e){var n=e[t];if("function"==typeof n)return 1;if(void 0!==n)return n.length}return 0}function p(t,e){for(var n=new Array(e),r=0;r<e;++r)n[r]=t[r];return n}Object.defineProperty(a,"defaultMaxListeners",{enumerable:!0,get:function(){return s},set:function(t){if("number"!=typeof t||t<0||o(t))throw new RangeError('The value of "defaultMaxListeners" is out of range. It must be a non-negative number. Received '+t+".");s=t}}),a.init=function(){void 0!==this._events&&this._events!==Object.getPrototypeOf(this)._events||(this._events=Object.create(null),this._eventsCount=0),this._maxListeners=this._maxListeners||void 0},a.prototype.setMaxListeners=function(t){if("number"!=typeof t||t<0||o(t))throw new RangeError('The value of "n" is out of range. It must be a non-negative number. Received '+t+".");return this._maxListeners=t,this},a.prototype.getMaxListeners=function(){return l(this)},a.prototype.emit=function(t){for(var e=[],n=1;n<arguments.length;n++)e.push(arguments[n]);var r="error"===t,i=this._events;if(void 0!==i)r=r&&void 0===i.error;else if(!r)return!1;if(r){var o;if(0<e.length&&(o=e[0]),o instanceof Error)throw o;var a=new Error("Unhandled error."+(o?" ("+o.message+")":""));throw a.context=o,a}var s=i[t];if(void 0===s)return!1;if("function"==typeof s)c(s,this,e);else{var l=s.length,u=p(s,l);for(n=0;n<l;++n)c(u[n],this,e)}return!0},a.prototype.on=a.prototype.addListener=function(t,e){return u(this,t,e,!1)},a.prototype.prependListener=function(t,e){return u(this,t,e,!0)},a.prototype.once=function(t,e){if("function"!=typeof e)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof e);return this.on(t,f(this,t,e)),this},a.prototype.prependOnceListener=function(t,e){if("function"!=typeof e)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof e);return this.prependListener(t,f(this,t,e)),this},a.prototype.off=a.prototype.removeListener=function(t,e){var n,r,i,o,a;if("function"!=typeof e)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof e);if(void 0===(r=this._events))return this;if(void 0===(n=r[t]))return this;if(n===e||n.listener===e)0==--this._eventsCount?this._events=Object.create(null):(delete r[t],r.removeListener&&this.emit("removeListener",t,n.listener||e));else if("function"!=typeof n){for(i=-1,o=n.length-1;0<=o;o--)if(n[o]===e||n[o].listener===e){a=n[o].listener,i=o;break}if(i<0)return this;0===i?n.shift():function(t,e){for(;e+1<t.length;e++)t[e]=t[e+1];t.pop()}(n,i),1===n.length&&(r[t]=n[0]),void 0!==r.removeListener&&this.emit("removeListener",t,a||e)}return this},a.prototype.removeAllListeners=function(t){var e,n,r;if(void 0===(n=this._events))return this;if(void 0===n.removeListener)return 0===arguments.length?(this._events=Object.create(null),this._eventsCount=0):void 0!==n[t]&&(0==--this._eventsCount?this._events=Object.create(null):delete n[t]),this;if(0===arguments.length){var i,o=Object.keys(n);for(r=0;r<o.length;++r)"removeListener"!==(i=o[r])&&this.removeAllListeners(i);return this.removeAllListeners("removeListener"),this._events=Object.create(null),this._eventsCount=0,this}if("function"==typeof(e=n[t]))this.removeListener(t,e);else if(void 0!==e)for(r=e.length-1;0<=r;r--)this.removeListener(t,e[r]);return this},a.prototype.listeners=function(t){return h(this,t,!0)},a.prototype.rawListeners=function(t){return h(this,t,!1)},a.listenerCount=function(t,e){return"function"==typeof t.listenerCount?t.listenerCount(e):d.call(t,e)},a.prototype.listenerCount=d,a.prototype.eventNames=function(){return 0<this._eventsCount?r(this._events):[]}},function(t,e,n){"use strict";(function(s){!s.version||0===s.version.indexOf("v0.")||0===s.version.indexOf("v1.")&&0!==s.version.indexOf("v1.8.")?t.exports={nextTick:function(t,e,n,r){if("function"!=typeof t)throw new TypeError('"callback" argument must be a function');var i,o,a=arguments.length;switch(a){case 0:case 1:return s.nextTick(t);case 2:return s.nextTick(function(){t.call(null,e)});case 3:return s.nextTick(function(){t.call(null,e,n)});case 4:return s.nextTick(function(){t.call(null,e,n,r)});default:for(i=new Array(a-1),o=0;o<i.length;)i[o++]=arguments[o];return s.nextTick(function(){t.apply(null,i)})}}}:t.exports=s}).call(this,n(24))},function(t,e,n){var r=n(4),i=r.Buffer;function o(t,e){for(var n in t)e[n]=t[n]}function a(t,e,n){return i(t,e,n)}i.from&&i.alloc&&i.allocUnsafe&&i.allocUnsafeSlow?t.exports=r:(o(r,e),e.Buffer=a),o(i,a),a.from=function(t,e,n){if("number"==typeof t)throw new TypeError("Argument must not be a number");return i(t,e,n)},a.alloc=function(t,e,n){if("number"!=typeof t)throw new TypeError("Argument must be a number");var r=i(t);return void 0!==e?"string"==typeof n?r.fill(e,n):r.fill(e):r.fill(0),r},a.allocUnsafe=function(t){if("number"!=typeof t)throw new TypeError("Argument must be a number");return i(t)},a.allocUnsafeSlow=function(t){if("number"!=typeof t)throw new TypeError("Argument must be a number");return r.SlowBuffer(t)}},function(t,e,n){"use strict";var r="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;e.assign=function(t){for(var e,n,r=Array.prototype.slice.call(arguments,1);r.length;){var i=r.shift();if(i){if("object"!=typeof i)throw new TypeError(i+"must be non-object");for(var o in i)e=i,n=o,Object.prototype.hasOwnProperty.call(e,n)&&(t[o]=i[o])}}return t},e.shrinkBuf=function(t,e){return t.length===e?t:t.subarray?t.subarray(0,e):(t.length=e,t)};var i={arraySet:function(t,e,n,r,i){if(e.subarray&&t.subarray)t.set(e.subarray(n,n+r),i);else for(var o=0;o<r;o++)t[i+o]=e[n+o]},flattenChunks:function(t){var e,n,r,i,o,a;for(e=r=0,n=t.length;e<n;e++)r+=t[e].length;for(a=new Uint8Array(r),e=i=0,n=t.length;e<n;e++)o=t[e],a.set(o,i),i+=o.length;return a}},o={arraySet:function(t,e,n,r,i){for(var o=0;o<r;o++)t[i+o]=e[n+o]},flattenChunks:function(t){return[].concat.apply([],t)}};e.setTyped=function(t){t?(e.Buf8=Uint8Array,e.Buf16=Uint16Array,e.Buf32=Int32Array,e.assign(e,i)):(e.Buf8=Array,e.Buf16=Array,e.Buf32=Array,e.assign(e,o))},e.setTyped(r)},function(t,e,n){var r;t.exports=(r=n(1),function(){var t=r,e=t.lib,i=e.Base,o=e.WordArray,n=t.x64={};n.Word=i.extend({init:function(t,e){this.high=t,this.low=e}}),n.WordArray=i.extend({init:function(t,e){t=this.words=t||[],this.sigBytes=null!=e?e:8*t.length},toX32:function(){for(var t=this.words,e=t.length,n=[],r=0;r<e;r++){var i=t[r];n.push(i.high),n.push(i.low)}return o.create(n,this.sigBytes)},clone:function(){for(var t=i.clone.call(this),e=t.words=this.words.slice(0),n=e.length,r=0;r<n;r++)e[r]=e[r].clone();return t}})}(),r)},function(t,e){e.f={}.propertyIsEnumerable},function(t,e){t.exports=!0},function(t,e,r){function i(){}var o=r(28),a=r(169),s=r(116),l=r(114)("IE_PROTO"),u="prototype",c=function(){var t,e=r(165)("iframe"),n=s.length;for(e.style.display="none",r(336).appendChild(e),e.src="javascript:",(t=e.contentWindow.document).open(),t.write("<script>document.F=Object<\/script>"),t.close(),c=t.F;n--;)delete c[u][s[n]];return c()};t.exports=Object.create||function(t,e){var n;return null!==t?(i[u]=o(t),n=new i,i[u]=null,n[l]=t):n=c(),void 0===e?n:a(n,e)}},function(t,e,n){var r=n(113),i=Math.min;t.exports=function(t){return 0<t?i(r(t),9007199254740991):0}},function(t,e){var n=0,r=Math.random();t.exports=function(t){return"Symbol(".concat(void 0===t?"":t,")_",(++n+r).toString(36))}},function(t,e,n){var r=n(17).f,i=n(36),o=n(12)("toStringTag");t.exports=function(t,e,n){t&&!i(t=n?t:t.prototype,o)&&r(t,o,{configurable:!0,value:e})}},function(t,e,n){function r(t){s(t,i,{value:{i:"O"+ ++l,w:{}}})}var i=n(77)("meta"),o=n(20),a=n(36),s=n(17).f,l=0,u=Object.isExtensible||function(){return!0},c=!n(37)(function(){return u(Object.preventExtensions({}))}),f=t.exports={KEY:i,NEED:!1,fastKey:function(t,e){if(!o(t))return"symbol"==typeof t?t:("string"==typeof t?"S":"P")+t;if(!a(t,i)){if(!u(t))return"F";if(!e)return"E";r(t)}return t[i].i},getWeak:function(t,e){if(!a(t,i)){if(!u(t))return!0;if(!e)return!1;r(t)}return t[i].w},onFreeze:function(t){return c&&f.NEED&&u(t)&&!a(t,i)&&r(t),t}}},function(t,e,n){var h=n(38),d=n(180),p=n(181),g=n(28),v=n(76),y=n(117),b={},m={};(e=t.exports=function(t,e,n,r,i){var o,a,s,l,u=i?function(){return t}:y(t),c=h(n,r,e?2:1),f=0;if("function"!=typeof u)throw TypeError(t+" is not iterable!");if(p(u)){for(o=v(t.length);f<o;f++)if((l=e?c(g(a=t[f])[0],a[1]):c(t[f]))===b||l===m)return l}else for(s=u.call(t);!(a=s.next()).done;)if((l=d(s,c,a.value,e))===b||l===m)return l}).BREAK=b,e.RETURN=m},function(t,e,n){var r=n(55);t.exports=Object("z").propertyIsEnumerable(0)?Object:function(t){return"String"==r(t)?t.split(""):Object(t)}},function(t,e,n){var l=n(43),u=n(16),c=n(56);t.exports=function(s){return function(t,e,n){var r,i=l(t),o=u(i.length),a=c(n,o);if(s&&e!=e){for(;a<o;)if((r=i[a++])!=r)return!0}else for(;a<o;a++)if((s||a in i)&&i[a]===e)return s||a||0;return!s&&-1}}},function(t,e,n){var r=n(62)("keys"),i=n(29);t.exports=function(t){return r[t]||(r[t]=i(t))}},function(t,e){t.exports="constructor,hasOwnProperty,isPrototypeOf,propertyIsEnumerable,toLocaleString,toString,valueOf".split(",")},function(t,e){e.f=Object.getOwnPropertySymbols},function(t,e,n){"use strict";var u=n(19),c=n(56),f=n(16);t.exports=function(t,e,n){for(var r=u(this),i=f(r.length),o=arguments.length,a=c(1<o?e:void 0,i),s=2<o?n:void 0,l=void 0===s?i:c(s,i);a<l;)r[a++]=t;return r}},function(t,e,n){var r=n(3)("unscopables"),i=Array.prototype;null==i[r]&&n(15)(i,r,{}),t.exports=function(t){i[r][t]=!0}},function(t,e,n){"use strict";var r=n(87),i=n(218),o=n(44),a=n(43);t.exports=n(144)(Array,"Array",function(t,e){this._t=a(t),this._i=0,this._k=e},function(){var t=this._t,e=this._k,n=this._i++;return!t||n>=t.length?(this._t=void 0,i(1)):i(0,"keys"==e?n:"values"==e?t[n]:[n,t[n]])},"values"),o.Arguments=o.Array,r("keys"),r("values"),r("entries")},function(t,e,n){"use strict";var r=n(90)(!0);t.exports=function(t,e,n){return e+(n?r(t,e).length:1)}},function(t,e,n){var l=n(31),u=n(30);t.exports=function(s){return function(t,e){var n,r,i=String(u(t)),o=l(e),a=i.length;return o<0||a<=o?s?"":void 0:(n=i.charCodeAt(o))<55296||56319<n||o+1===a||(r=i.charCodeAt(o+1))<56320||57343<r?s?i.charAt(o):n:s?i.slice(o,o+2):r-56320+(n-55296<<10)+65536}}},function(t,e,n){"use strict";var i=n(65),o=RegExp.prototype.exec;t.exports=function(t,e){var n=t.exec;if("function"==typeof n){var r=n.call(t,e);if("object"!=typeof r)throw new TypeError("RegExp exec method returned something other than an Object or null");return r}if("RegExp"!==i(t))throw new TypeError("RegExp#exec called on incompatible receiver");return o.call(t,e)}},function(t,e,n){"use strict";n(223);var c=n(22),f=n(15),h=n(10),d=n(30),p=n(3),g=n(93),v=p("species"),y=!h(function(){var t=/./;return t.exec=function(){var t=[];return t.groups={a:"7"},t},"7"!=="".replace(t,"$<a>")}),b=function(){var t=/(?:)/,e=t.exec;t.exec=function(){return e.apply(this,arguments)};var n="ab".split(t);return 2===n.length&&"a"===n[0]&&"b"===n[1]}();t.exports=function(n,t,e){var r=p(n),o=!h(function(){var t={};return t[r]=function(){return 7},7!=""[n](t)}),i=o?!h(function(){var t=!1,e=/a/;return e.exec=function(){return t=!0,null},"split"===n&&(e.constructor={},e.constructor[v]=function(){return e}),e[r](""),!t}):void 0;if(!o||!i||"replace"===n&&!y||"split"===n&&!b){var a=/./[r],s=e(d,r,""[n],function(t,e,n,r,i){return e.exec===g?o&&!i?{done:!0,value:a.call(e,n,r)}:{done:!0,value:t.call(n,e,r)}:{done:!1}}),l=s[0],u=s[1];c(String.prototype,n,l),f(RegExp.prototype,r,2==t?function(t,e){return u.call(t,this,e)}:function(t){return u.call(t,this)})}}},function(t,e,n){"use strict";var r,i,a=n(94),s=RegExp.prototype.exec,l=String.prototype.replace,o=s,u="lastIndex",c=(r=/a/,i=/b*/g,s.call(r,"a"),s.call(i,"a"),0!==r[u]||0!==i[u]),f=void 0!==/()??/.exec("")[1];(c||f)&&(o=function(t){var e,n,r,i,o=this;return f&&(n=new RegExp("^"+o.source+"$(?!\\s)",a.call(o))),c&&(e=o[u]),r=s.call(o,t),c&&r&&(o[u]=o.global?r.index+r[0].length:e),f&&r&&1<r.length&&l.call(r[0],n,function(){for(i=1;i<arguments.length-2;i++)void 0===arguments[i]&&(r[i]=void 0)}),r}),t.exports=o},function(t,e,n){"use strict";var r=n(13);t.exports=function(){var t=r(this),e="";return t.global&&(e+="g"),t.ignoreCase&&(e+="i"),t.multiline&&(e+="m"),t.unicode&&(e+="u"),t.sticky&&(e+="y"),e}},function(t,e,n){t.exports=r;var c=n(68).EventEmitter;function r(){c.call(this)}n(32)(r,c),r.Readable=n(96),r.Writable=n(259),r.Duplex=n(260),r.Transform=n(261),r.PassThrough=n(262),(r.Stream=r).prototype.pipe=function(e,t){var n=this;function r(t){e.writable&&!1===e.write(t)&&n.pause&&n.pause()}function i(){n.readable&&n.resume&&n.resume()}n.on("data",r),e.on("drain",i),e._isStdio||t&&!1===t.end||(n.on("end",a),n.on("close",s));var o=!1;function a(){o||(o=!0,e.end())}function s(){o||(o=!0,"function"==typeof e.destroy&&e.destroy())}function l(t){if(u(),0===c.listenerCount(this,"error"))throw t}function u(){n.removeListener("data",r),e.removeListener("drain",i),n.removeListener("end",a),n.removeListener("close",s),n.removeListener("error",l),e.removeListener("error",l),n.removeListener("end",u),n.removeListener("close",u),e.removeListener("close",u)}return n.on("error",l),e.on("error",l),n.on("end",u),n.on("close",u),e.on("close",u),e.emit("pipe",n),e}},function(t,e,n){(((e=t.exports=n(150)).Stream=e).Readable=e).Writable=n(97),e.Duplex=n(33),e.Transform=n(153),e.PassThrough=n(258)},function(_,t,k){"use strict";(function(t,e){var s=k(69);function f(t){var e=this;this.next=null,this.entry=null,this.finish=function(){!function(t,e,n){var r=t.entry;t.entry=null;for(;r;){var i=r.callback;e.pendingcb--,i(n),r=r.next}e.corkedRequestsFree?e.corkedRequestsFree.next=t:e.corkedRequestsFree=t}(e,t)}}_.exports=g;var l,u=!t.browser&&-1<["v0.10","v0.9."].indexOf(t.version.slice(0,5))?setImmediate:s.nextTick;g.WritableState=p;var n=k(58);n.inherits=k(32);var r={deprecate:k(257)},i=k(151),c=k(70).Buffer,a=e.Uint8Array||function(){};var o,h=k(152);function d(){}function p(t,e){l=l||k(33),t=t||{};var n=e instanceof l;this.objectMode=!!t.objectMode,n&&(this.objectMode=this.objectMode||!!t.writableObjectMode);var r=t.highWaterMark,i=t.writableHighWaterMark,o=this.objectMode?16:16384;this.highWaterMark=r||0===r?r:n&&(i||0===i)?i:o,this.highWaterMark=Math.floor(this.highWaterMark),this.finalCalled=!1,this.needDrain=!1,this.ending=!1,this.ended=!1,this.finished=!1;var a=(this.destroyed=!1)===t.decodeStrings;this.decodeStrings=!a,this.defaultEncoding=t.defaultEncoding||"utf8",this.length=0,this.writing=!1,this.corked=0,this.sync=!0,this.bufferProcessing=!1,this.onwrite=function(t){!function(t,e){var n=t._writableState,r=n.sync,i=n.writecb;if(function(t){t.writing=!1,t.writecb=null,t.length-=t.writelen,t.writelen=0}(n),e)!function(t,e,n,r,i){--e.pendingcb,n?(s.nextTick(i,r),s.nextTick(x,t,e),t._writableState.errorEmitted=!0,t.emit("error",r)):(i(r),t._writableState.errorEmitted=!0,t.emit("error",r),x(t,e))}(t,n,r,e,i);else{var o=m(n);o||n.corked||n.bufferProcessing||!n.bufferedRequest||b(t,n),r?u(y,t,n,o,i):y(t,n,o,i)}}(e,t)},this.writecb=null,this.writelen=0,this.bufferedRequest=null,this.lastBufferedRequest=null,this.pendingcb=0,this.prefinished=!1,this.errorEmitted=!1,this.bufferedRequestCount=0,this.corkedRequestsFree=new f(this)}function g(t){if(l=l||k(33),!(o.call(g,this)||this instanceof l))return new g(t);this._writableState=new p(t,this),this.writable=!0,t&&("function"==typeof t.write&&(this._write=t.write),"function"==typeof t.writev&&(this._writev=t.writev),"function"==typeof t.destroy&&(this._destroy=t.destroy),"function"==typeof t.final&&(this._final=t.final)),i.call(this)}function v(t,e,n,r,i,o,a){e.writelen=r,e.writecb=a,e.writing=!0,e.sync=!0,n?t._writev(i,e.onwrite):t._write(i,o,e.onwrite),e.sync=!1}function y(t,e,n,r){n||function(t,e){0===e.length&&e.needDrain&&(e.needDrain=!1,t.emit("drain"))}(t,e),e.pendingcb--,r(),x(t,e)}function b(t,e){e.bufferProcessing=!0;var n=e.bufferedRequest;if(t._writev&&n&&n.next){var r=e.bufferedRequestCount,i=new Array(r),o=e.corkedRequestsFree;o.entry=n;for(var a=0,s=!0;n;)(i[a]=n).isBuf||(s=!1),n=n.next,a+=1;i.allBuffers=s,v(t,e,!0,e.length,i,"",o.finish),e.pendingcb++,e.lastBufferedRequest=null,o.next?(e.corkedRequestsFree=o.next,o.next=null):e.corkedRequestsFree=new f(e),e.bufferedRequestCount=0}else{for(;n;){var l=n.chunk,u=n.encoding,c=n.callback;if(v(t,e,!1,e.objectMode?1:l.length,l,u,c),n=n.next,e.bufferedRequestCount--,e.writing)break}null===n&&(e.lastBufferedRequest=null)}e.bufferedRequest=n,e.bufferProcessing=!1}function m(t){return t.ending&&0===t.length&&null===t.bufferedRequest&&!t.finished&&!t.writing}function w(e,n){e._final(function(t){n.pendingcb--,t&&e.emit("error",t),n.prefinished=!0,e.emit("prefinish"),x(e,n)})}function x(t,e){var n=m(e);return n&&(function(t,e){e.prefinished||e.finalCalled||("function"==typeof t._final?(e.pendingcb++,e.finalCalled=!0,s.nextTick(w,t,e)):(e.prefinished=!0,t.emit("prefinish")))}(t,e),0===e.pendingcb&&(e.finished=!0,t.emit("finish"))),n}n.inherits(g,i),p.prototype.getBuffer=function(){for(var t=this.bufferedRequest,e=[];t;)e.push(t),t=t.next;return e},function(){try{Object.defineProperty(p.prototype,"buffer",{get:r.deprecate(function(){return this.getBuffer()},"_writableState.buffer is deprecated. Use _writableState.getBuffer instead.","DEP0003")})}catch(t){}}(),"function"==typeof Symbol&&Symbol.hasInstance&&"function"==typeof Function.prototype[Symbol.hasInstance]?(o=Function.prototype[Symbol.hasInstance],Object.defineProperty(g,Symbol.hasInstance,{value:function(t){return!!o.call(this,t)||this===g&&(t&&t._writableState instanceof p)}})):o=function(t){return t instanceof this},g.prototype.pipe=function(){this.emit("error",new Error("Cannot pipe, not readable"))},g.prototype.write=function(t,e,n){var r=this._writableState,i=!1,o=!r.objectMode&&function(t){return c.isBuffer(t)||t instanceof a}(t);return o&&!c.isBuffer(t)&&(t=function(t){return c.from(t)}(t)),"function"==typeof e&&(n=e,e=null),e=o?"buffer":e||r.defaultEncoding,"function"!=typeof n&&(n=d),r.ended?function(t,e){var n=new Error("write after end");t.emit("error",n),s.nextTick(e,n)}(this,n):(o||function(t,e,n,r){var i=!0,o=!1;return null===n?o=new TypeError("May not write null values to stream"):"string"==typeof n||void 0===n||e.objectMode||(o=new TypeError("Invalid non-string/buffer chunk")),o&&(t.emit("error",o),s.nextTick(r,o),i=!1),i}(this,r,t,n))&&(r.pendingcb++,i=function(t,e,n,r,i,o){if(!n){var a=function(t,e,n){t.objectMode||!1===t.decodeStrings||"string"!=typeof e||(e=c.from(e,n));return e}(e,r,i);r!==a&&(n=!0,i="buffer",r=a)}var s=e.objectMode?1:r.length;e.length+=s;var l=e.length<e.highWaterMark;l||(e.needDrain=!0);if(e.writing||e.corked){var u=e.lastBufferedRequest;e.lastBufferedRequest={chunk:r,encoding:i,isBuf:n,callback:o,next:null},u?u.next=e.lastBufferedRequest:e.bufferedRequest=e.lastBufferedRequest,e.bufferedRequestCount+=1}else v(t,e,!1,s,r,i,o);return l}(this,r,o,t,e,n)),i},g.prototype.cork=function(){this._writableState.corked++},g.prototype.uncork=function(){var t=this._writableState;t.corked&&(t.corked--,t.writing||t.corked||t.finished||t.bufferProcessing||!t.bufferedRequest||b(this,t))},g.prototype.setDefaultEncoding=function(t){if("string"==typeof t&&(t=t.toLowerCase()),!(-1<["hex","utf8","utf-8","ascii","binary","base64","ucs2","ucs-2","utf16le","utf-16le","raw"].indexOf((t+"").toLowerCase())))throw new TypeError("Unknown encoding: "+t);return this._writableState.defaultEncoding=t,this},Object.defineProperty(g.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),g.prototype._write=function(t,e,n){n(new Error("_write() is not implemented"))},g.prototype._writev=null,g.prototype.end=function(t,e,n){var r=this._writableState;"function"==typeof t?(n=t,e=t=null):"function"==typeof e&&(n=e,e=null),null!=t&&this.write(t,e),r.corked&&(r.corked=1,this.uncork()),r.ending||r.finished||function(t,e,n){e.ending=!0,x(t,e),n&&(e.finished?s.nextTick(n):t.once("finish",n));e.ended=!0,t.writable=!1}(this,r,n)},Object.defineProperty(g.prototype,"destroyed",{get:function(){return void 0!==this._writableState&&this._writableState.destroyed},set:function(t){this._writableState&&(this._writableState.destroyed=t)}}),g.prototype.destroy=h.destroy,g.prototype._undestroy=h.undestroy,g.prototype._destroy=function(t,e){this.end(),e(t)}}).call(this,k(24),k(25))},function(t,e,n){"use strict";var r=n(70).Buffer,i=r.isEncoding||function(t){switch((t=""+t)&&t.toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":case"raw":return!0;default:return!1}};function o(t){var e;switch(this.encoding=function(t){var e=function(t){if(!t)return"utf8";for(var e;;)switch(t){case"utf8":case"utf-8":return"utf8";case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return"utf16le";case"latin1":case"binary":return"latin1";case"base64":case"ascii":case"hex":return t;default:if(e)return;t=(""+t).toLowerCase(),e=!0}}(t);if("string"!=typeof e&&(r.isEncoding===i||!i(t)))throw new Error("Unknown encoding: "+t);return e||t}(t),this.encoding){case"utf16le":this.text=l,this.end=u,e=4;break;case"utf8":this.fillLast=s,e=4;break;case"base64":this.text=c,this.end=f,e=3;break;default:return this.write=h,void(this.end=d)}this.lastNeed=0,this.lastTotal=0,this.lastChar=r.allocUnsafe(e)}function a(t){return t<=127?0:t>>5==6?2:t>>4==14?3:t>>3==30?4:t>>6==2?-1:-2}function s(t){var e=this.lastTotal-this.lastNeed,n=function(t,e){if(128!=(192&e[0]))return t.lastNeed=0,"�";if(1<t.lastNeed&&1<e.length){if(128!=(192&e[1]))return t.lastNeed=1,"�";if(2<t.lastNeed&&2<e.length&&128!=(192&e[2]))return t.lastNeed=2,"�"}}(this,t);return void 0!==n?n:this.lastNeed<=t.length?(t.copy(this.lastChar,e,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal)):(t.copy(this.lastChar,e,0,t.length),void(this.lastNeed-=t.length))}function l(t,e){if((t.length-e)%2!=0)return this.lastNeed=1,this.lastTotal=2,this.lastChar[0]=t[t.length-1],t.toString("utf16le",e,t.length-1);var n=t.toString("utf16le",e);if(n){var r=n.charCodeAt(n.length-1);if(55296<=r&&r<=56319)return this.lastNeed=2,this.lastTotal=4,this.lastChar[0]=t[t.length-2],this.lastChar[1]=t[t.length-1],n.slice(0,-1)}return n}function u(t){var e=t&&t.length?this.write(t):"";if(this.lastNeed){var n=this.lastTotal-this.lastNeed;return e+this.lastChar.toString("utf16le",0,n)}return e}function c(t,e){var n=(t.length-e)%3;return 0==n?t.toString("base64",e):(this.lastNeed=3-n,this.lastTotal=3,1==n?this.lastChar[0]=t[t.length-1]:(this.lastChar[0]=t[t.length-2],this.lastChar[1]=t[t.length-1]),t.toString("base64",e,t.length-n))}function f(t){var e=t&&t.length?this.write(t):"";return this.lastNeed?e+this.lastChar.toString("base64",0,3-this.lastNeed):e}function h(t){return t.toString(this.encoding)}function d(t){return t&&t.length?this.write(t):""}(e.StringDecoder=o).prototype.write=function(t){if(0===t.length)return"";var e,n;if(this.lastNeed){if(void 0===(e=this.fillLast(t)))return"";n=this.lastNeed,this.lastNeed=0}else n=0;return n<t.length?e?e+this.text(t,n):this.text(t,n):e||""},o.prototype.end=function(t){var e=t&&t.length?this.write(t):"";return this.lastNeed?e+"�":e},o.prototype.text=function(t,e){var n=function(t,e,n){var r=e.length-1;if(r<n)return 0;var i=a(e[r]);if(0<=i)return 0<i&&(t.lastNeed=i-1),i;if(--r<n||-2===i)return 0;if(0<=(i=a(e[r])))return 0<i&&(t.lastNeed=i-2),i;if(--r<n||-2===i)return 0;if(0<=(i=a(e[r])))return 0<i&&(2===i?i=0:t.lastNeed=i-3),i;return 0}(this,t,e);if(!this.lastNeed)return t.toString("utf8",e);this.lastTotal=n;var r=t.length-(n-this.lastNeed);return t.copy(this.lastChar,0,r),t.toString("utf8",e,r)},o.prototype.fillLast=function(t){if(this.lastNeed<=t.length)return t.copy(this.lastChar,this.lastTotal-this.lastNeed,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal);t.copy(this.lastChar,this.lastTotal-this.lastNeed,0,t.length),this.lastNeed-=t.length}},function(t,P,E){(function(a){var e=Object.getOwnPropertyDescriptors||function(t){for(var e=Object.keys(t),n={},r=0;r<e.length;r++)n[e[r]]=Object.getOwnPropertyDescriptor(t,e[r]);return n},s=/%[sdj%]/g;P.format=function(t){if(!m(t)){for(var e=[],n=0;n<arguments.length;n++)e.push(l(arguments[n]));return e.join(" ")}n=1;for(var r=arguments,i=r.length,o=String(t).replace(s,function(t){if("%%"===t)return"%";if(i<=n)return t;switch(t){case"%s":return String(r[n++]);case"%d":return Number(r[n++]);case"%j":try{return JSON.stringify(r[n++])}catch(t){return"[Circular]"}default:return t}}),a=r[n];n<i;a=r[++n])y(a)||!u(a)?o+=" "+a:o+=" "+l(a);return o},P.deprecate=function(t,e){if(void 0!==a&&!0===a.noDeprecation)return t;if(void 0===a)return function(){return P.deprecate(t,e).apply(this,arguments)};var n=!1;return function(){if(!n){if(a.throwDeprecation)throw new Error(e);a.traceDeprecation,n=!0}return t.apply(this,arguments)}};var n,r={};function l(t,e){var n={seen:[],stylize:o};return 3<=arguments.length&&(n.depth=arguments[2]),4<=arguments.length&&(n.colors=arguments[3]),v(e)?n.showHidden=e:e&&P._extend(n,e),w(n.showHidden)&&(n.showHidden=!1),w(n.depth)&&(n.depth=2),w(n.colors)&&(n.colors=!1),w(n.customInspect)&&(n.customInspect=!0),n.colors&&(n.stylize=i),h(n,t,n.depth)}function i(t,e){var n=l.styles[e];return n?"["+l.colors[n][0]+"m"+t+"["+l.colors[n][1]+"m":t}function o(t,e){return t}function h(e,n,r){if(e.customInspect&&n&&S(n.inspect)&&n.inspect!==P.inspect&&(!n.constructor||n.constructor.prototype!==n)){var t=n.inspect(r,e);return m(t)||(t=h(e,t,r)),t}var i=function(t,e){if(w(e))return t.stylize("undefined","undefined");if(m(e)){var n="'"+JSON.stringify(e).replace(/^"|"$/g,"").replace(/'/g,"\\'").replace(/\\"/g,'"')+"'";return t.stylize(n,"string")}if(b(e))return t.stylize(""+e,"number");if(v(e))return t.stylize(""+e,"boolean");if(y(e))return t.stylize("null","null")}(e,n);if(i)return i;var o=Object.keys(n),a=function(t){var n={};return t.forEach(function(t,e){n[t]=!0}),n}(o);if(e.showHidden&&(o=Object.getOwnPropertyNames(n)),k(n)&&(0<=o.indexOf("message")||0<=o.indexOf("description")))return d(n);if(0===o.length){if(S(n)){var s=n.name?": "+n.name:"";return e.stylize("[Function"+s+"]","special")}if(x(n))return e.stylize(RegExp.prototype.toString.call(n),"regexp");if(_(n))return e.stylize(Date.prototype.toString.call(n),"date");if(k(n))return d(n)}var l,u="",c=!1,f=["{","}"];g(n)&&(c=!0,f=["[","]"]),S(n)&&(u=" [Function"+(n.name?": "+n.name:"")+"]");return x(n)&&(u=" "+RegExp.prototype.toString.call(n)),_(n)&&(u=" "+Date.prototype.toUTCString.call(n)),k(n)&&(u=" "+d(n)),0!==o.length||c&&0!=n.length?r<0?x(n)?e.stylize(RegExp.prototype.toString.call(n),"regexp"):e.stylize("[Object]","special"):(e.seen.push(n),l=c?function(e,n,r,i,t){for(var o=[],a=0,s=n.length;a<s;++a)C(n,String(a))?o.push(p(e,n,r,i,String(a),!0)):o.push("");return t.forEach(function(t){t.match(/^\d+$/)||o.push(p(e,n,r,i,t,!0))}),o}(e,n,r,a,o):o.map(function(t){return p(e,n,r,a,t,c)}),e.seen.pop(),function(t,e,n){if(60<t.reduce(function(t,e){return 0,0<=e.indexOf("\n")&&0,t+e.replace(/\u001b\[\d\d?m/g,"").length+1},0))return n[0]+(""===e?"":e+"\n ")+" "+t.join(",\n  ")+" "+n[1];return n[0]+e+" "+t.join(", ")+" "+n[1]}(l,u,f)):f[0]+u+f[1]}function d(t){return"["+Error.prototype.toString.call(t)+"]"}function p(t,e,n,r,i,o){var a,s,l;if((l=Object.getOwnPropertyDescriptor(e,i)||{value:e[i]}).get?s=l.set?t.stylize("[Getter/Setter]","special"):t.stylize("[Getter]","special"):l.set&&(s=t.stylize("[Setter]","special")),C(r,i)||(a="["+i+"]"),s||(t.seen.indexOf(l.value)<0?-1<(s=y(n)?h(t,l.value,null):h(t,l.value,n-1)).indexOf("\n")&&(s=o?s.split("\n").map(function(t){return"  "+t}).join("\n").substr(2):"\n"+s.split("\n").map(function(t){return"   "+t}).join("\n")):s=t.stylize("[Circular]","special")),w(a)){if(o&&i.match(/^\d+$/))return s;a=(a=JSON.stringify(""+i)).match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)?(a=a.substr(1,a.length-2),t.stylize(a,"name")):(a=a.replace(/'/g,"\\'").replace(/\\"/g,'"').replace(/(^"|"$)/g,"'"),t.stylize(a,"string"))}return a+": "+s}function g(t){return Array.isArray(t)}function v(t){return"boolean"==typeof t}function y(t){return null===t}function b(t){return"number"==typeof t}function m(t){return"string"==typeof t}function w(t){return void 0===t}function x(t){return u(t)&&"[object RegExp]"===c(t)}function u(t){return"object"==typeof t&&null!==t}function _(t){return u(t)&&"[object Date]"===c(t)}function k(t){return u(t)&&("[object Error]"===c(t)||t instanceof Error)}function S(t){return"function"==typeof t}function c(t){return Object.prototype.toString.call(t)}P.debuglog=function(t){if(w(n)&&(n=a.env.NODE_DEBUG||""),t=t.toUpperCase(),!r[t])if(new RegExp("\\b"+t+"\\b","i").test(n)){a.pid;r[t]=function(){P.format.apply(P,arguments)}}else r[t]=function(){};return r[t]},(P.inspect=l).colors={bold:[1,22],italic:[3,23],underline:[4,24],inverse:[7,27],white:[37,39],grey:[90,39],black:[30,39],blue:[34,39],cyan:[36,39],green:[32,39],magenta:[35,39],red:[31,39],yellow:[33,39]},l.styles={special:"cyan",number:"yellow",boolean:"yellow",undefined:"grey",null:"bold",string:"green",date:"magenta",regexp:"red"},P.isArray=g,P.isBoolean=v,P.isNull=y,P.isNullOrUndefined=function(t){return null==t},P.isNumber=b,P.isString=m,P.isSymbol=function(t){return"symbol"==typeof t},P.isUndefined=w,P.isRegExp=x,P.isObject=u,P.isDate=_,P.isError=k,P.isFunction=S,P.isPrimitive=function(t){return null===t||"boolean"==typeof t||"number"==typeof t||"string"==typeof t||"symbol"==typeof t||void 0===t},P.isBuffer=E(265);function C(t,e){return Object.prototype.hasOwnProperty.call(t,e)}P.log=function(){},P.inherits=E(32),P._extend=function(t,e){if(!e||!u(e))return t;for(var n=Object.keys(e),r=n.length;r--;)t[n[r]]=e[n[r]];return t};var f="undefined"!=typeof Symbol?Symbol("util.promisify.custom"):void 0;function A(t,e){if(!t){var n=new Error("Promise was rejected with a falsy value");n.reason=t,t=n}return e(t)}P.promisify=function(o){if("function"!=typeof o)throw new TypeError('The "original" argument must be of type Function');if(f&&o[f]){var t;if("function"!=typeof(t=o[f]))throw new TypeError('The "util.promisify.custom" argument must be of type Function');return Object.defineProperty(t,f,{value:t,enumerable:!1,writable:!1,configurable:!0}),t}function t(){for(var n,r,t=new Promise(function(t,e){n=t,r=e}),e=[],i=0;i<arguments.length;i++)e.push(arguments[i]);e.push(function(t,e){t?r(t):n(e)});try{o.apply(this,e)}catch(t){r(t)}return t}return Object.setPrototypeOf(t,Object.getPrototypeOf(o)),f&&Object.defineProperty(t,f,{value:t,enumerable:!1,writable:!1,configurable:!0}),Object.defineProperties(t,e(o))},P.promisify.custom=f,P.callbackify=function(o){if("function"!=typeof o)throw new TypeError('The "original" argument must be of type Function');function t(){for(var t=[],e=0;e<arguments.length;e++)t.push(arguments[e]);var n=t.pop();if("function"!=typeof n)throw new TypeError("The last argument must be of type Function");function r(){return n.apply(i,arguments)}var i=this;o.apply(this,t).then(function(t){a.nextTick(r,null,t)},function(t){a.nextTick(A,t,r)})}return Object.setPrototypeOf(t,Object.getPrototypeOf(o)),Object.defineProperties(t,e(o)),t}}).call(this,E(24))},function(t,e,n){var a;t.exports=(a=n(1),function(){var t=a,e=t.lib,n=e.WordArray,r=e.Hasher,i=t.algo,f=[],o=i.SHA1=r.extend({_doReset:function(){this._hash=new n.init([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function(t,e){for(var n=this._hash.words,r=n[0],i=n[1],o=n[2],a=n[3],s=n[4],l=0;l<80;l++){if(l<16)f[l]=0|t[e+l];else{var u=f[l-3]^f[l-8]^f[l-14]^f[l-16];f[l]=u<<1|u>>>31}var c=(r<<5|r>>>27)+s+f[l];c+=l<20?1518500249+(i&o|~i&a):l<40?1859775393+(i^o^a):l<60?(i&o|i&a|o&a)-1894007588:(i^o^a)-899497514,s=a,a=o,o=i<<30|i>>>2,i=r,r=c}n[0]=n[0]+r|0,n[1]=n[1]+i|0,n[2]=n[2]+o|0,n[3]=n[3]+a|0,n[4]=n[4]+s|0},_doFinalize:function(){var t=this._data,e=t.words,n=8*this._nDataBytes,r=8*t.sigBytes;return e[r>>>5]|=128<<24-r%32,e[14+(64+r>>>9<<4)]=Math.floor(n/4294967296),e[15+(64+r>>>9<<4)]=n,t.sigBytes=4*e.length,this._process(),this._hash},clone:function(){var t=r.clone.call(this);return t._hash=this._hash.clone(),t}});t.SHA1=r._createHelper(o),t.HmacSHA1=r._createHmacHelper(o)}(),a.SHA1)},function(t,e,n){var o;t.exports=(o=n(1),void function(){var t=o,e=t.lib,n=e.Base,r=t.enc,u=r.Utf8,i=t.algo;i.HMAC=n.extend({init:function(t,e){t=this._hasher=new t.init,"string"==typeof e&&(e=u.parse(e));var n=t.blockSize,r=4*n;e.sigBytes>r&&(e=t.finalize(e)),e.clamp();for(var i=this._oKey=e.clone(),o=this._iKey=e.clone(),a=i.words,s=o.words,l=0;l<n;l++)a[l]^=1549556828,s[l]^=909522486;i.sigBytes=o.sigBytes=r,this.reset()},reset:function(){var t=this._hasher;t.reset(),t.update(this._iKey)},update:function(t){return this._hasher.update(t),this},finalize:function(t){var e=this._hasher,n=e.finalize(t);e.reset();var r=e.finalize(this._oKey.clone().concat(n));return r}})}())},function(e,t,n){(function(l){(function(){var t,s;try{s=n(103)}catch(t){}t=function(){var t;function r(t){this.buffer=t,this.pos=0,this.length=this.buffer.length}for(t in r.TYPES={UInt8:1,UInt16:2,UInt24:3,UInt32:4,Int8:1,Int16:2,Int24:3,Int32:4,Float:4,Double:8},l.prototype)"read"===t.slice(0,4)&&function(e){var n;n=r.TYPES[e.replace(/read|[BL]E/g,"")],r.prototype[e]=function(){var t;return t=this.buffer[e](this.pos),this.pos+=n,t}}(t);return r.prototype.readString=function(t,e){var n,r,i,o,a;switch(null==e&&(e="ascii"),e){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.buffer.toString(e,this.pos,this.pos+=t);case"utf16be":for(i=o=0,a=(n=new l(this.readBuffer(t))).length-1;o<a;i=o+=2)r=n[i],n[i]=n[i+1],n[i+1]=r;return n.toString("utf16le");default:if(n=this.readBuffer(t),s)try{return s.decode(n,e)}catch(t){}return n}},r.prototype.readBuffer=function(t){return this.buffer.slice(this.pos,this.pos+=t)},r.prototype.readUInt24BE=function(){return(this.readUInt16BE()<<8)+this.readUInt8()},r.prototype.readUInt24LE=function(){return this.readUInt16LE()+(this.readUInt8()<<16)},r.prototype.readInt24BE=function(){return(this.readInt16BE()<<8)+this.readUInt8()},r.prototype.readInt24LE=function(){return this.readUInt16LE()+(this.readInt8()<<16)},r}(),e.exports=t}).call(this)}).call(this,n(4).Buffer)},function(r,t,l){"use strict";(function(t){var a=l(47).Buffer,i=l(301),s=r.exports;s.encodings=null,s.defaultCharUnicode="�",s.defaultCharSingleByte="?",s.encode=function(t,e,n){t=""+(t||"");var r=s.getEncoder(e,n),i=r.write(t),o=r.end();return o&&0<o.length?a.concat([i,o]):i},s.decode=function(t,e,n){"string"==typeof t&&(s.skipDecodeWarning||(s.skipDecodeWarning=!0),t=a.from(""+(t||""),"binary"));var r=s.getDecoder(e,n),i=r.write(t),o=r.end();return o?i+o:i},s.encodingExists=function(t){try{return s.getCodec(t),!0}catch(t){return!1}},s.toEncoding=s.encode,s.fromEncoding=s.decode,s._codecDataCache={},s.getCodec=function(t){s.encodings||(s.encodings=l(302));for(var e=s._canonicalizeEncoding(t),n={};;){var r=s._codecDataCache[e];if(r)return r;var i=s.encodings[e];switch(typeof i){case"string":e=i;break;case"object":for(var o in i)n[o]=i[o];n.encodingName||(n.encodingName=e),e=i.type;break;case"function":return n.encodingName||(n.encodingName=e),r=new i(n,s),s._codecDataCache[n.encodingName]=r;default:throw new Error("Encoding not recognized: '"+t+"' (searched as: '"+e+"')")}}},s._canonicalizeEncoding=function(t){return(""+t).toLowerCase().replace(/:\d{4}$|[^0-9a-z]/g,"")},s.getEncoder=function(t,e){var n=s.getCodec(t),r=new n.encoder(e,n);return n.bomAware&&e&&e.addBOM&&(r=new i.PrependBOM(r,e)),r},s.getDecoder=function(t,e){var n=s.getCodec(t),r=new n.decoder(e,n);return!n.bomAware||e&&!1===e.stripBOM||(r=new i.StripBOM(r,e)),r};var e=void 0!==t&&t.versions&&t.versions.node;if(e){var n=e.split(".").map(Number);(0<n[0]||10<=n[1])&&l(316)(s),l(317)(s)}}).call(this,l(24))},function(t){t.exports=[["0","\0",127,"€"],["8140","丂丄丅丆丏丒丗丟丠両丣並丩丮丯丱丳丵丷丼乀乁乂乄乆乊乑乕乗乚乛乢乣乤乥乧乨乪",5,"乲乴",9,"乿",6,"亇亊"],["8180","亐亖亗亙亜亝亞亣亪亯亰亱亴亶亷亸亹亼亽亾仈仌仏仐仒仚仛仜仠仢仦仧仩仭仮仯仱仴仸仹仺仼仾伀伂",6,"伋伌伒",4,"伜伝伡伣伨伩伬伭伮伱伳伵伷伹伻伾",4,"佄佅佇",5,"佒佔佖佡佢佦佨佪佫佭佮佱佲併佷佸佹佺佽侀侁侂侅來侇侊侌侎侐侒侓侕侖侘侙侚侜侞侟価侢"],["8240","侤侫侭侰",4,"侶",8,"俀俁係俆俇俈俉俋俌俍俒",4,"俙俛俠俢俤俥俧俫俬俰俲俴俵俶俷俹俻俼俽俿",11],["8280","個倎倐們倓倕倖倗倛倝倞倠倢倣値倧倫倯",10,"倻倽倿偀偁偂偄偅偆偉偊偋偍偐",4,"偖偗偘偙偛偝",7,"偦",5,"偭",8,"偸偹偺偼偽傁傂傃傄傆傇傉傊傋傌傎",20,"傤傦傪傫傭",4,"傳",6,"傼"],["8340","傽",17,"僐",5,"僗僘僙僛",10,"僨僩僪僫僯僰僱僲僴僶",4,"僼",9,"儈"],["8380","儉儊儌",5,"儓",13,"儢",28,"兂兇兊兌兎兏児兒兓兗兘兙兛兝",4,"兣兤兦內兩兪兯兲兺兾兿冃冄円冇冊冋冎冏冐冑冓冔冘冚冝冞冟冡冣冦",4,"冭冮冴冸冹冺冾冿凁凂凃凅凈凊凍凎凐凒",5],["8440","凘凙凚凜凞凟凢凣凥",5,"凬凮凱凲凴凷凾刄刅刉刋刌刏刐刓刔刕刜刞刟刡刢刣別刦刧刪刬刯刱刲刴刵刼刾剄",5,"剋剎剏剒剓剕剗剘"],["8480","剙剚剛剝剟剠剢剣剤剦剨剫剬剭剮剰剱剳",9,"剾劀劃",4,"劉",6,"劑劒劔",6,"劜劤劥劦劧劮劯劰労",9,"勀勁勂勄勅勆勈勊勌勍勎勏勑勓勔動勗務",5,"勠勡勢勣勥",10,"勱",7,"勻勼勽匁匂匃匄匇匉匊匋匌匎"],["8540","匑匒匓匔匘匛匜匞匟匢匤匥匧匨匩匫匬匭匯",9,"匼匽區卂卄卆卋卌卍卐協単卙卛卝卥卨卪卬卭卲卶卹卻卼卽卾厀厁厃厇厈厊厎厏"],["8580","厐",4,"厖厗厙厛厜厞厠厡厤厧厪厫厬厭厯",6,"厷厸厹厺厼厽厾叀參",4,"収叏叐叒叓叕叚叜叝叞叡叢叧叴叺叾叿吀吂吅吇吋吔吘吙吚吜吢吤吥吪吰吳吶吷吺吽吿呁呂呄呅呇呉呌呍呎呏呑呚呝",4,"呣呥呧呩",7,"呴呹呺呾呿咁咃咅咇咈咉咊咍咑咓咗咘咜咞咟咠咡"],["8640","咢咥咮咰咲咵咶咷咹咺咼咾哃哅哊哋哖哘哛哠",4,"哫哬哯哰哱哴",5,"哻哾唀唂唃唄唅唈唊",4,"唒唓唕",5,"唜唝唞唟唡唥唦"],["8680","唨唩唫唭唲唴唵唶唸唹唺唻唽啀啂啅啇啈啋",4,"啑啒啓啔啗",4,"啝啞啟啠啢啣啨啩啫啯",5,"啹啺啽啿喅喆喌喍喎喐喒喓喕喖喗喚喛喞喠",6,"喨",8,"喲喴営喸喺喼喿",4,"嗆嗇嗈嗊嗋嗎嗏嗐嗕嗗",4,"嗞嗠嗢嗧嗩嗭嗮嗰嗱嗴嗶嗸",4,"嗿嘂嘃嘄嘅"],["8740","嘆嘇嘊嘋嘍嘐",7,"嘙嘚嘜嘝嘠嘡嘢嘥嘦嘨嘩嘪嘫嘮嘯嘰嘳嘵嘷嘸嘺嘼嘽嘾噀",11,"噏",4,"噕噖噚噛噝",4],["8780","噣噥噦噧噭噮噯噰噲噳噴噵噷噸噹噺噽",7,"嚇",6,"嚐嚑嚒嚔",14,"嚤",10,"嚰",6,"嚸嚹嚺嚻嚽",12,"囋",8,"囕囖囘囙囜団囥",5,"囬囮囯囲図囶囷囸囻囼圀圁圂圅圇國",6],["8840","園",9,"圝圞圠圡圢圤圥圦圧圫圱圲圴",4,"圼圽圿坁坃坄坅坆坈坉坋坒",4,"坘坙坢坣坥坧坬坮坰坱坲坴坵坸坹坺坽坾坿垀"],["8880","垁垇垈垉垊垍",4,"垔",6,"垜垝垞垟垥垨垪垬垯垰垱垳垵垶垷垹",8,"埄",6,"埌埍埐埑埓埖埗埛埜埞埡埢埣埥",7,"埮埰埱埲埳埵埶執埻埼埾埿堁堃堄堅堈堉堊堌堎堏堐堒堓堔堖堗堘堚堛堜堝堟堢堣堥",4,"堫",4,"報堲堳場堶",7],["8940","堾",5,"塅",6,"塎塏塐塒塓塕塖塗塙",4,"塟",5,"塦",4,"塭",16,"塿墂墄墆墇墈墊墋墌"],["8980","墍",4,"墔",4,"墛墜墝墠",7,"墪",17,"墽墾墿壀壂壃壄壆",10,"壒壓壔壖",13,"壥",5,"壭壯壱売壴壵壷壸壺",7,"夃夅夆夈",4,"夎夐夑夒夓夗夘夛夝夞夠夡夢夣夦夨夬夰夲夳夵夶夻"],["8a40","夽夾夿奀奃奅奆奊奌奍奐奒奓奙奛",4,"奡奣奤奦",12,"奵奷奺奻奼奾奿妀妅妉妋妌妎妏妐妑妔妕妘妚妛妜妝妟妠妡妢妦"],["8a80","妧妬妭妰妱妳",5,"妺妼妽妿",6,"姇姈姉姌姍姎姏姕姖姙姛姞",4,"姤姦姧姩姪姫姭",11,"姺姼姽姾娀娂娊娋娍娎娏娐娒娔娕娖娗娙娚娛娝娞娡娢娤娦娧娨娪",6,"娳娵娷",4,"娽娾娿婁",4,"婇婈婋",9,"婖婗婘婙婛",5],["8b40","婡婣婤婥婦婨婩婫",8,"婸婹婻婼婽婾媀",17,"媓",6,"媜",13,"媫媬"],["8b80","媭",4,"媴媶媷媹",4,"媿嫀嫃",5,"嫊嫋嫍",4,"嫓嫕嫗嫙嫚嫛嫝嫞嫟嫢嫤嫥嫧嫨嫪嫬",4,"嫲",22,"嬊",11,"嬘",25,"嬳嬵嬶嬸",7,"孁",6],["8c40","孈",7,"孒孖孞孠孡孧孨孫孭孮孯孲孴孶孷學孹孻孼孾孿宂宆宊宍宎宐宑宒宔宖実宧宨宩宬宭宮宯宱宲宷宺宻宼寀寁寃寈寉寊寋寍寎寏"],["8c80","寑寔",8,"寠寢寣實寧審",4,"寯寱",6,"寽対尀専尃尅將專尋尌對導尐尒尓尗尙尛尞尟尠尡尣尦尨尩尪尫尭尮尯尰尲尳尵尶尷屃屄屆屇屌屍屒屓屔屖屗屘屚屛屜屝屟屢層屧",6,"屰屲",6,"屻屼屽屾岀岃",4,"岉岊岋岎岏岒岓岕岝",4,"岤",4],["8d40","岪岮岯岰岲岴岶岹岺岻岼岾峀峂峃峅",5,"峌",5,"峓",5,"峚",6,"峢峣峧峩峫峬峮峯峱",9,"峼",4],["8d80","崁崄崅崈",5,"崏",4,"崕崗崘崙崚崜崝崟",4,"崥崨崪崫崬崯",4,"崵",7,"崿",7,"嵈嵉嵍",10,"嵙嵚嵜嵞",10,"嵪嵭嵮嵰嵱嵲嵳嵵",12,"嶃",21,"嶚嶛嶜嶞嶟嶠"],["8e40","嶡",21,"嶸",12,"巆",6,"巎",12,"巜巟巠巣巤巪巬巭"],["8e80","巰巵巶巸",4,"巿帀帄帇帉帊帋帍帎帒帓帗帞",7,"帨",4,"帯帰帲",4,"帹帺帾帿幀幁幃幆",5,"幍",6,"幖",4,"幜幝幟幠幣",14,"幵幷幹幾庁庂広庅庈庉庌庍庎庒庘庛庝庡庢庣庤庨",4,"庮",4,"庴庺庻庼庽庿",6],["8f40","廆廇廈廋",5,"廔廕廗廘廙廚廜",11,"廩廫",8,"廵廸廹廻廼廽弅弆弇弉弌弍弎弐弒弔弖弙弚弜弝弞弡弢弣弤"],["8f80","弨弫弬弮弰弲",6,"弻弽弾弿彁",14,"彑彔彙彚彛彜彞彟彠彣彥彧彨彫彮彯彲彴彵彶彸彺彽彾彿徃徆徍徎徏徑従徔徖徚徛徝從徟徠徢",5,"復徫徬徯",5,"徶徸徹徺徻徾",4,"忇忈忊忋忎忓忔忕忚忛応忞忟忢忣忥忦忨忩忬忯忰忲忳忴忶忷忹忺忼怇"],["9040","怈怉怋怌怐怑怓怗怘怚怞怟怢怣怤怬怭怮怰",4,"怶",4,"怽怾恀恄",6,"恌恎恏恑恓恔恖恗恘恛恜恞恟恠恡恥恦恮恱恲恴恵恷恾悀"],["9080","悁悂悅悆悇悈悊悋悎悏悐悑悓悕悗悘悙悜悞悡悢悤悥悧悩悪悮悰悳悵悶悷悹悺悽",7,"惇惈惉惌",4,"惒惓惔惖惗惙惛惞惡",4,"惪惱惲惵惷惸惻",4,"愂愃愄愅愇愊愋愌愐",4,"愖愗愘愙愛愜愝愞愡愢愥愨愩愪愬",18,"慀",6],["9140","慇慉態慍慏慐慒慓慔慖",6,"慞慟慠慡慣慤慥慦慩",6,"慱慲慳慴慶慸",18,"憌憍憏",4,"憕"],["9180","憖",6,"憞",8,"憪憫憭",9,"憸",5,"憿懀懁懃",4,"應懌",4,"懓懕",16,"懧",13,"懶",8,"戀",5,"戇戉戓戔戙戜戝戞戠戣戦戧戨戩戫戭戯戰戱戲戵戶戸",4,"扂扄扅扆扊"],["9240","扏扐払扖扗扙扚扜",6,"扤扥扨扱扲扴扵扷扸扺扻扽抁抂抃抅抆抇抈抋",5,"抔抙抜抝択抣抦抧抩抪抭抮抯抰抲抳抴抶抷抸抺抾拀拁"],["9280","拃拋拏拑拕拝拞拠拡拤拪拫拰拲拵拸拹拺拻挀挃挄挅挆挊挋挌挍挏挐挒挓挔挕挗挘挙挜挦挧挩挬挭挮挰挱挳",5,"挻挼挾挿捀捁捄捇捈捊捑捒捓捔捖",7,"捠捤捥捦捨捪捫捬捯捰捲捳捴捵捸捹捼捽捾捿掁掃掄掅掆掋掍掑掓掔掕掗掙",6,"採掤掦掫掯掱掲掵掶掹掻掽掿揀"],["9340","揁揂揃揅揇揈揊揋揌揑揓揔揕揗",6,"揟揢揤",4,"揫揬揮揯揰揱揳揵揷揹揺揻揼揾搃搄搆",4,"損搎搑搒搕",5,"搝搟搢搣搤"],["9380","搥搧搨搩搫搮",5,"搵",4,"搻搼搾摀摂摃摉摋",6,"摓摕摖摗摙",4,"摟",7,"摨摪摫摬摮",9,"摻",6,"撃撆撈",8,"撓撔撗撘撚撛撜撝撟",4,"撥撦撧撨撪撫撯撱撲撳撴撶撹撻撽撾撿擁擃擄擆",6,"擏擑擓擔擕擖擙據"],["9440","擛擜擝擟擠擡擣擥擧",24,"攁",7,"攊",7,"攓",4,"攙",8],["9480","攢攣攤攦",4,"攬攭攰攱攲攳攷攺攼攽敀",4,"敆敇敊敋敍敎敐敒敓敔敗敘敚敜敟敠敡敤敥敧敨敩敪敭敮敯敱敳敵敶數",14,"斈斉斊斍斎斏斒斔斕斖斘斚斝斞斠斢斣斦斨斪斬斮斱",7,"斺斻斾斿旀旂旇旈旉旊旍旐旑旓旔旕旘",7,"旡旣旤旪旫"],["9540","旲旳旴旵旸旹旻",4,"昁昄昅昇昈昉昋昍昐昑昒昖昗昘昚昛昜昞昡昢昣昤昦昩昪昫昬昮昰昲昳昷",4,"昽昿晀時晄",6,"晍晎晐晑晘"],["9580","晙晛晜晝晞晠晢晣晥晧晩",4,"晱晲晳晵晸晹晻晼晽晿暀暁暃暅暆暈暉暊暋暍暎暏暐暒暓暔暕暘",4,"暞",8,"暩",4,"暯",4,"暵暶暷暸暺暻暼暽暿",25,"曚曞",7,"曧曨曪",5,"曱曵曶書曺曻曽朁朂會"],["9640","朄朅朆朇朌朎朏朑朒朓朖朘朙朚朜朞朠",5,"朧朩朮朰朲朳朶朷朸朹朻朼朾朿杁杄杅杇杊杋杍杒杔杕杗",4,"杝杢杣杤杦杧杫杬杮東杴杶"],["9680","杸杹杺杻杽枀枂枃枅枆枈枊枌枍枎枏枑枒枓枔枖枙枛枟枠枡枤枦枩枬枮枱枲枴枹",7,"柂柅",9,"柕柖柗柛柟柡柣柤柦柧柨柪柫柭柮柲柵",7,"柾栁栂栃栄栆栍栐栒栔栕栘",4,"栞栟栠栢",6,"栫",6,"栴栵栶栺栻栿桇桋桍桏桒桖",5],["9740","桜桝桞桟桪桬",7,"桵桸",8,"梂梄梇",7,"梐梑梒梔梕梖梘",9,"梣梤梥梩梪梫梬梮梱梲梴梶梷梸"],["9780","梹",6,"棁棃",5,"棊棌棎棏棐棑棓棔棖棗棙棛",4,"棡棢棤",9,"棯棲棳棴棶棷棸棻棽棾棿椀椂椃椄椆",4,"椌椏椑椓",11,"椡椢椣椥",7,"椮椯椱椲椳椵椶椷椸椺椻椼椾楀楁楃",16,"楕楖楘楙楛楜楟"],["9840","楡楢楤楥楧楨楩楪楬業楯楰楲",4,"楺楻楽楾楿榁榃榅榊榋榌榎",5,"榖榗榙榚榝",9,"榩榪榬榮榯榰榲榳榵榶榸榹榺榼榽"],["9880","榾榿槀槂",7,"構槍槏槑槒槓槕",5,"槜槝槞槡",11,"槮槯槰槱槳",9,"槾樀",9,"樋",11,"標",5,"樠樢",5,"権樫樬樭樮樰樲樳樴樶",6,"樿",4,"橅橆橈",7,"橑",6,"橚"],["9940","橜",4,"橢橣橤橦",10,"橲",6,"橺橻橽橾橿檁檂檃檅",8,"檏檒",4,"檘",7,"檡",5],["9980","檧檨檪檭",114,"欥欦欨",6],["9a40","欯欰欱欳欴欵欶欸欻欼欽欿歀歁歂歄歅歈歊歋歍",11,"歚",7,"歨歩歫",13,"歺歽歾歿殀殅殈"],["9a80","殌殎殏殐殑殔殕殗殘殙殜",4,"殢",7,"殫",7,"殶殸",6,"毀毃毄毆",4,"毌毎毐毑毘毚毜",4,"毢",7,"毬毭毮毰毱毲毴毶毷毸毺毻毼毾",6,"氈",4,"氎氒気氜氝氞氠氣氥氫氬氭氱氳氶氷氹氺氻氼氾氿汃汄汅汈汋",4,"汑汒汓汖汘"],["9b40","汙汚汢汣汥汦汧汫",4,"汱汳汵汷汸決汻汼汿沀沄沇沊沋沍沎沑沒沕沖沗沘沚沜沝沞沠沢沨沬沯沰沴沵沶沷沺泀況泂泃泆泇泈泋泍泎泏泑泒泘"],["9b80","泙泚泜泝泟泤泦泧泩泬泭泲泴泹泿洀洂洃洅洆洈洉洊洍洏洐洑洓洔洕洖洘洜洝洟",5,"洦洨洩洬洭洯洰洴洶洷洸洺洿浀浂浄浉浌浐浕浖浗浘浛浝浟浡浢浤浥浧浨浫浬浭浰浱浲浳浵浶浹浺浻浽",4,"涃涄涆涇涊涋涍涏涐涒涖",4,"涜涢涥涬涭涰涱涳涴涶涷涹",5,"淁淂淃淈淉淊"],["9c40","淍淎淏淐淒淓淔淕淗淚淛淜淟淢淣淥淧淨淩淪淭淯淰淲淴淵淶淸淺淽",7,"渆渇済渉渋渏渒渓渕渘渙減渜渞渟渢渦渧渨渪測渮渰渱渳渵"],["9c80","渶渷渹渻",7,"湅",7,"湏湐湑湒湕湗湙湚湜湝湞湠",10,"湬湭湯",14,"満溁溂溄溇溈溊",4,"溑",6,"溙溚溛溝溞溠溡溣溤溦溨溩溫溬溭溮溰溳溵溸溹溼溾溿滀滃滄滅滆滈滉滊滌滍滎滐滒滖滘滙滛滜滝滣滧滪",5],["9d40","滰滱滲滳滵滶滷滸滺",7,"漃漄漅漇漈漊",4,"漐漑漒漖",9,"漡漢漣漥漦漧漨漬漮漰漲漴漵漷",6,"漿潀潁潂"],["9d80","潃潄潅潈潉潊潌潎",9,"潙潚潛潝潟潠潡潣潤潥潧",5,"潯潰潱潳潵潶潷潹潻潽",6,"澅澆澇澊澋澏",12,"澝澞澟澠澢",4,"澨",10,"澴澵澷澸澺",5,"濁濃",5,"濊",6,"濓",10,"濟濢濣濤濥"],["9e40","濦",7,"濰",32,"瀒",7,"瀜",6,"瀤",6],["9e80","瀫",9,"瀶瀷瀸瀺",17,"灍灎灐",13,"灟",11,"灮灱灲灳灴灷灹灺灻災炁炂炃炄炆炇炈炋炌炍炏炐炑炓炗炘炚炛炞",12,"炰炲炴炵炶為炾炿烄烅烆烇烉烋",12,"烚"],["9f40","烜烝烞烠烡烢烣烥烪烮烰",6,"烸烺烻烼烾",10,"焋",4,"焑焒焔焗焛",10,"焧",7,"焲焳焴"],["9f80","焵焷",13,"煆煇煈煉煋煍煏",12,"煝煟",4,"煥煩",4,"煯煰煱煴煵煶煷煹煻煼煾",5,"熅",4,"熋熌熍熎熐熑熒熓熕熖熗熚",4,"熡",6,"熩熪熫熭",5,"熴熶熷熸熺",8,"燄",9,"燏",4],["a040","燖",9,"燡燢燣燤燦燨",5,"燯",9,"燺",11,"爇",19],["a080","爛爜爞",9,"爩爫爭爮爯爲爳爴爺爼爾牀",6,"牉牊牋牎牏牐牑牓牔牕牗牘牚牜牞牠牣牤牥牨牪牫牬牭牰牱牳牴牶牷牸牻牼牽犂犃犅",4,"犌犎犐犑犓",11,"犠",11,"犮犱犲犳犵犺",6,"狅狆狇狉狊狋狌狏狑狓狔狕狖狘狚狛"],["a1a1","　、。·ˉˇ¨〃々—～‖…‘’“”〔〕〈",7,"〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓"],["a2a1","ⅰ",9],["a2b1","⒈",19,"⑴",19,"①",9],["a2e5","㈠",9],["a2f1","Ⅰ",11],["a3a1","！＂＃￥％",88,"￣"],["a4a1","ぁ",82],["a5a1","ァ",85],["a6a1","Α",16,"Σ",6],["a6c1","α",16,"σ",6],["a6e0","︵︶︹︺︿﹀︽︾﹁﹂﹃﹄"],["a6ee","︻︼︷︸︱"],["a6f4","︳︴"],["a7a1","А",5,"ЁЖ",25],["a7d1","а",5,"ёж",25],["a840","ˊˋ˙–―‥‵℅℉↖↗↘↙∕∟∣≒≦≧⊿═",35,"▁",6],["a880","█",7,"▓▔▕▼▽◢◣◤◥☉⊕〒〝〞"],["a8a1","āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüêɑ"],["a8bd","ńň"],["a8c0","ɡ"],["a8c5","ㄅ",36],["a940","〡",8,"㊣㎎㎏㎜㎝㎞㎡㏄㏎㏑㏒㏕︰￢￤"],["a959","℡㈱"],["a95c","‐"],["a960","ー゛゜ヽヾ〆ゝゞ﹉",9,"﹔﹕﹖﹗﹙",8],["a980","﹢",4,"﹨﹩﹪﹫"],["a996","〇"],["a9a4","─",75],["aa40","狜狝狟狢",5,"狪狫狵狶狹狽狾狿猀猂猄",5,"猋猌猍猏猐猑猒猔猘猙猚猟猠猣猤猦猧猨猭猯猰猲猳猵猶猺猻猼猽獀",8],["aa80","獉獊獋獌獎獏獑獓獔獕獖獘",7,"獡",10,"獮獰獱"],["ab40","獲",11,"獿",4,"玅玆玈玊玌玍玏玐玒玓玔玕玗玘玙玚玜玝玞玠玡玣",5,"玪玬玭玱玴玵玶玸玹玼玽玾玿珁珃",4],["ab80","珋珌珎珒",6,"珚珛珜珝珟珡珢珣珤珦珨珪珫珬珮珯珰珱珳",4],["ac40","珸",10,"琄琇琈琋琌琍琎琑",8,"琜",5,"琣琤琧琩琫琭琯琱琲琷",4,"琽琾琿瑀瑂",11],["ac80","瑎",6,"瑖瑘瑝瑠",12,"瑮瑯瑱",4,"瑸瑹瑺"],["ad40","瑻瑼瑽瑿璂璄璅璆璈璉璊璌璍璏璑",10,"璝璟",7,"璪",15,"璻",12],["ad80","瓈",9,"瓓",8,"瓝瓟瓡瓥瓧",6,"瓰瓱瓲"],["ae40","瓳瓵瓸",6,"甀甁甂甃甅",7,"甎甐甒甔甕甖甗甛甝甞甠",4,"甦甧甪甮甴甶甹甼甽甿畁畂畃畄畆畇畉畊畍畐畑畒畓畕畖畗畘"],["ae80","畝",7,"畧畨畩畫",6,"畳畵當畷畺",4,"疀疁疂疄疅疇"],["af40","疈疉疊疌疍疎疐疓疕疘疛疜疞疢疦",4,"疭疶疷疺疻疿痀痁痆痋痌痎痏痐痑痓痗痙痚痜痝痟痠痡痥痩痬痭痮痯痲痳痵痶痷痸痺痻痽痾瘂瘄瘆瘇"],["af80","瘈瘉瘋瘍瘎瘏瘑瘒瘓瘔瘖瘚瘜瘝瘞瘡瘣瘧瘨瘬瘮瘯瘱瘲瘶瘷瘹瘺瘻瘽癁療癄"],["b040","癅",6,"癎",5,"癕癗",4,"癝癟癠癡癢癤",6,"癬癭癮癰",7,"癹発發癿皀皁皃皅皉皊皌皍皏皐皒皔皕皗皘皚皛"],["b080","皜",7,"皥",8,"皯皰皳皵",9,"盀盁盃啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥"],["b140","盄盇盉盋盌盓盕盙盚盜盝盞盠",4,"盦",7,"盰盳盵盶盷盺盻盽盿眀眂眃眅眆眊県眎",10,"眛眜眝眞眡眣眤眥眧眪眫"],["b180","眬眮眰",4,"眹眻眽眾眿睂睄睅睆睈",7,"睒",7,"睜薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳"],["b240","睝睞睟睠睤睧睩睪睭",11,"睺睻睼瞁瞂瞃瞆",5,"瞏瞐瞓",11,"瞡瞣瞤瞦瞨瞫瞭瞮瞯瞱瞲瞴瞶",4],["b280","瞼瞾矀",12,"矎",8,"矘矙矚矝",4,"矤病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖"],["b340","矦矨矪矯矰矱矲矴矵矷矹矺矻矼砃",5,"砊砋砎砏砐砓砕砙砛砞砠砡砢砤砨砪砫砮砯砱砲砳砵砶砽砿硁硂硃硄硆硈硉硊硋硍硏硑硓硔硘硙硚"],["b380","硛硜硞",11,"硯",7,"硸硹硺硻硽",6,"场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚"],["b440","碄碅碆碈碊碋碏碐碒碔碕碖碙碝碞碠碢碤碦碨",7,"碵碶碷碸確碻碼碽碿磀磂磃磄磆磇磈磌磍磎磏磑磒磓磖磗磘磚",9],["b480","磤磥磦磧磩磪磫磭",4,"磳磵磶磸磹磻",5,"礂礃礄礆",6,"础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮"],["b540","礍",5,"礔",9,"礟",4,"礥",14,"礵",4,"礽礿祂祃祄祅祇祊",8,"祔祕祘祙祡祣"],["b580","祤祦祩祪祫祬祮祰",6,"祹祻",4,"禂禃禆禇禈禉禋禌禍禎禐禑禒怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠"],["b640","禓",6,"禛",11,"禨",10,"禴",4,"禼禿秂秄秅秇秈秊秌秎秏秐秓秔秖秗秙",5,"秠秡秢秥秨秪"],["b680","秬秮秱",6,"秹秺秼秾秿稁稄稅稇稈稉稊稌稏",4,"稕稖稘稙稛稜丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二"],["b740","稝稟稡稢稤",14,"稴稵稶稸稺稾穀",5,"穇",9,"穒",4,"穘",16],["b780","穩",6,"穱穲穳穵穻穼穽穾窂窅窇窉窊窋窌窎窏窐窓窔窙窚窛窞窡窢贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服"],["b840","窣窤窧窩窪窫窮",4,"窴",10,"竀",10,"竌",9,"竗竘竚竛竜竝竡竢竤竧",5,"竮竰竱竲竳"],["b880","竴",4,"竻竼竾笀笁笂笅笇笉笌笍笎笐笒笓笖笗笘笚笜笝笟笡笢笣笧笩笭浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹"],["b940","笯笰笲笴笵笶笷笹笻笽笿",5,"筆筈筊筍筎筓筕筗筙筜筞筟筡筣",10,"筯筰筳筴筶筸筺筼筽筿箁箂箃箄箆",6,"箎箏"],["b980","箑箒箓箖箘箙箚箛箞箟箠箣箤箥箮箯箰箲箳箵箶箷箹",7,"篂篃範埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈"],["ba40","篅篈築篊篋篍篎篏篐篒篔",4,"篛篜篞篟篠篢篣篤篧篨篩篫篬篭篯篰篲",4,"篸篹篺篻篽篿",7,"簈簉簊簍簎簐",5,"簗簘簙"],["ba80","簚",4,"簠",5,"簨簩簫",12,"簹",5,"籂骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖"],["bb40","籃",9,"籎",36,"籵",5,"籾",9],["bb80","粈粊",6,"粓粔粖粙粚粛粠粡粣粦粧粨粩粫粬粭粯粰粴",4,"粺粻弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕"],["bc40","粿糀糂糃糄糆糉糋糎",6,"糘糚糛糝糞糡",6,"糩",5,"糰",7,"糹糺糼",13,"紋",5],["bc80","紑",14,"紡紣紤紥紦紨紩紪紬紭紮細",6,"肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件"],["bd40","紷",54,"絯",7],["bd80","絸",32,"健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸"],["be40","継",12,"綧",6,"綯",42],["be80","線",32,"尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻"],["bf40","緻",62],["bf80","縺縼",4,"繂",4,"繈",21,"俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀"],["c040","繞",35,"纃",23,"纜纝纞"],["c080","纮纴纻纼绖绤绬绹缊缐缞缷缹缻",6,"罃罆",9,"罒罓馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐"],["c140","罖罙罛罜罝罞罠罣",4,"罫罬罭罯罰罳罵罶罷罸罺罻罼罽罿羀羂",7,"羋羍羏",4,"羕",4,"羛羜羠羢羣羥羦羨",6,"羱"],["c180","羳",4,"羺羻羾翀翂翃翄翆翇翈翉翋翍翏",4,"翖翗翙",5,"翢翣痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿"],["c240","翤翧翨翪翫翬翭翯翲翴",6,"翽翾翿耂耇耈耉耊耎耏耑耓耚耛耝耞耟耡耣耤耫",5,"耲耴耹耺耼耾聀聁聄聅聇聈聉聎聏聐聑聓聕聖聗"],["c280","聙聛",13,"聫",5,"聲",11,"隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫"],["c340","聾肁肂肅肈肊肍",5,"肔肕肗肙肞肣肦肧肨肬肰肳肵肶肸肹肻胅胇",4,"胏",6,"胘胟胠胢胣胦胮胵胷胹胻胾胿脀脁脃脄脅脇脈脋"],["c380","脌脕脗脙脛脜脝脟",12,"脭脮脰脳脴脵脷脹",4,"脿谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸"],["c440","腀",5,"腇腉腍腎腏腒腖腗腘腛",4,"腡腢腣腤腦腨腪腫腬腯腲腳腵腶腷腸膁膃",4,"膉膋膌膍膎膐膒",5,"膙膚膞",4,"膤膥"],["c480","膧膩膫",7,"膴",5,"膼膽膾膿臄臅臇臈臉臋臍",6,"摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁"],["c540","臔",14,"臤臥臦臨臩臫臮",4,"臵",5,"臽臿舃與",4,"舎舏舑舓舕",5,"舝舠舤舥舦舧舩舮舲舺舼舽舿"],["c580","艀艁艂艃艅艆艈艊艌艍艎艐",7,"艙艛艜艝艞艠",7,"艩拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗"],["c640","艪艫艬艭艱艵艶艷艸艻艼芀芁芃芅芆芇芉芌芐芓芔芕芖芚芛芞芠芢芣芧芲芵芶芺芻芼芿苀苂苃苅苆苉苐苖苙苚苝苢苧苨苩苪苬苭苮苰苲苳苵苶苸"],["c680","苺苼",4,"茊茋茍茐茒茓茖茘茙茝",9,"茩茪茮茰茲茷茻茽啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐"],["c740","茾茿荁荂荄荅荈荊",4,"荓荕",4,"荝荢荰",6,"荹荺荾",6,"莇莈莊莋莌莍莏莐莑莔莕莖莗莙莚莝莟莡",6,"莬莭莮"],["c780","莯莵莻莾莿菂菃菄菆菈菉菋菍菎菐菑菒菓菕菗菙菚菛菞菢菣菤菦菧菨菫菬菭恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠"],["c840","菮華菳",4,"菺菻菼菾菿萀萂萅萇萈萉萊萐萒",5,"萙萚萛萞",5,"萩",7,"萲",5,"萹萺萻萾",7,"葇葈葉"],["c880","葊",6,"葒",4,"葘葝葞葟葠葢葤",4,"葪葮葯葰葲葴葷葹葻葼取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁"],["c940","葽",4,"蒃蒄蒅蒆蒊蒍蒏",7,"蒘蒚蒛蒝蒞蒟蒠蒢",12,"蒰蒱蒳蒵蒶蒷蒻蒼蒾蓀蓂蓃蓅蓆蓇蓈蓋蓌蓎蓏蓒蓔蓕蓗"],["c980","蓘",4,"蓞蓡蓢蓤蓧",4,"蓭蓮蓯蓱",10,"蓽蓾蔀蔁蔂伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳"],["ca40","蔃",8,"蔍蔎蔏蔐蔒蔔蔕蔖蔘蔙蔛蔜蔝蔞蔠蔢",8,"蔭",9,"蔾",4,"蕄蕅蕆蕇蕋",10],["ca80","蕗蕘蕚蕛蕜蕝蕟",4,"蕥蕦蕧蕩",8,"蕳蕵蕶蕷蕸蕼蕽蕿薀薁省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱"],["cb40","薂薃薆薈",6,"薐",10,"薝",6,"薥薦薧薩薫薬薭薱",5,"薸薺",6,"藂",6,"藊",4,"藑藒"],["cb80","藔藖",5,"藝",6,"藥藦藧藨藪",14,"恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔"],["cc40","藹藺藼藽藾蘀",4,"蘆",10,"蘒蘓蘔蘕蘗",15,"蘨蘪",13,"蘹蘺蘻蘽蘾蘿虀"],["cc80","虁",11,"虒虓處",4,"虛虜虝號虠虡虣",7,"獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃"],["cd40","虭虯虰虲",6,"蚃",6,"蚎",4,"蚔蚖",5,"蚞",4,"蚥蚦蚫蚭蚮蚲蚳蚷蚸蚹蚻",4,"蛁蛂蛃蛅蛈蛌蛍蛒蛓蛕蛖蛗蛚蛜"],["cd80","蛝蛠蛡蛢蛣蛥蛦蛧蛨蛪蛫蛬蛯蛵蛶蛷蛺蛻蛼蛽蛿蜁蜄蜅蜆蜋蜌蜎蜏蜐蜑蜔蜖汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威"],["ce40","蜙蜛蜝蜟蜠蜤蜦蜧蜨蜪蜫蜬蜭蜯蜰蜲蜳蜵蜶蜸蜹蜺蜼蜽蝀",6,"蝊蝋蝍蝏蝐蝑蝒蝔蝕蝖蝘蝚",5,"蝡蝢蝦",7,"蝯蝱蝲蝳蝵"],["ce80","蝷蝸蝹蝺蝿螀螁螄螆螇螉螊螌螎",4,"螔螕螖螘",6,"螠",4,"巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺"],["cf40","螥螦螧螩螪螮螰螱螲螴螶螷螸螹螻螼螾螿蟁",4,"蟇蟈蟉蟌",4,"蟔",6,"蟜蟝蟞蟟蟡蟢蟣蟤蟦蟧蟨蟩蟫蟬蟭蟯",9],["cf80","蟺蟻蟼蟽蟿蠀蠁蠂蠄",5,"蠋",7,"蠔蠗蠘蠙蠚蠜",4,"蠣稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓"],["d040","蠤",13,"蠳",5,"蠺蠻蠽蠾蠿衁衂衃衆",5,"衎",5,"衕衖衘衚",6,"衦衧衪衭衯衱衳衴衵衶衸衹衺"],["d080","衻衼袀袃袆袇袉袊袌袎袏袐袑袓袔袕袗",4,"袝",4,"袣袥",5,"小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄"],["d140","袬袮袯袰袲",4,"袸袹袺袻袽袾袿裀裃裄裇裈裊裋裌裍裏裐裑裓裖裗裚",4,"裠裡裦裧裩",6,"裲裵裶裷裺裻製裿褀褁褃",5],["d180","褉褋",4,"褑褔",4,"褜",4,"褢褣褤褦褧褨褩褬褭褮褯褱褲褳褵褷选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶"],["d240","褸",8,"襂襃襅",24,"襠",5,"襧",19,"襼"],["d280","襽襾覀覂覄覅覇",26,"摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐"],["d340","覢",30,"觃觍觓觔觕觗觘觙觛觝觟觠觡觢觤觧觨觩觪觬觭觮觰觱觲觴",6],["d380","觻",4,"訁",5,"計",21,"印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉"],["d440","訞",31,"訿",8,"詉",21],["d480","詟",25,"詺",6,"浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧"],["d540","誁",7,"誋",7,"誔",46],["d580","諃",32,"铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政"],["d640","諤",34,"謈",27],["d680","謤謥謧",30,"帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑"],["d740","譆",31,"譧",4,"譭",25],["d780","讇",24,"讬讱讻诇诐诪谉谞住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座"],["d840","谸",8,"豂豃豄豅豈豊豋豍",7,"豖豗豘豙豛",5,"豣",6,"豬",6,"豴豵豶豷豻",6,"貃貄貆貇"],["d880","貈貋貍",6,"貕貖貗貙",20,"亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝"],["d940","貮",62],["d980","賭",32,"佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼"],["da40","贎",14,"贠赑赒赗赟赥赨赩赪赬赮赯赱赲赸",8,"趂趃趆趇趈趉趌",4,"趒趓趕",9,"趠趡"],["da80","趢趤",12,"趲趶趷趹趻趽跀跁跂跅跇跈跉跊跍跐跒跓跔凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺"],["db40","跕跘跙跜跠跡跢跥跦跧跩跭跮跰跱跲跴跶跼跾",6,"踆踇踈踋踍踎踐踑踒踓踕",7,"踠踡踤",4,"踫踭踰踲踳踴踶踷踸踻踼踾"],["db80","踿蹃蹅蹆蹌",4,"蹓",5,"蹚",11,"蹧蹨蹪蹫蹮蹱邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝"],["dc40","蹳蹵蹷",4,"蹽蹾躀躂躃躄躆躈",6,"躑躒躓躕",6,"躝躟",11,"躭躮躰躱躳",6,"躻",7],["dc80","軃",10,"軏",21,"堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥"],["dd40","軥",62],["dd80","輤",32,"荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺"],["de40","轅",32,"轪辀辌辒辝辠辡辢辤辥辦辧辪辬辭辮辯農辳辴辵辷辸辺辻込辿迀迃迆"],["de80","迉",4,"迏迒迖迗迚迠迡迣迧迬迯迱迲迴迵迶迺迻迼迾迿逇逈逌逎逓逕逘蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖"],["df40","這逜連逤逥逧",5,"逰",4,"逷逹逺逽逿遀遃遅遆遈",4,"過達違遖遙遚遜",5,"遤遦遧適遪遫遬遯",4,"遶",6,"遾邁"],["df80","還邅邆邇邉邊邌",4,"邒邔邖邘邚邜邞邟邠邤邥邧邨邩邫邭邲邷邼邽邿郀摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼"],["e040","郂郃郆郈郉郋郌郍郒郔郕郖郘郙郚郞郟郠郣郤郥郩郪郬郮郰郱郲郳郵郶郷郹郺郻郼郿鄀鄁鄃鄅",19,"鄚鄛鄜"],["e080","鄝鄟鄠鄡鄤",10,"鄰鄲",6,"鄺",8,"酄唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼"],["e140","酅酇酈酑酓酔酕酖酘酙酛酜酟酠酦酧酨酫酭酳酺酻酼醀",4,"醆醈醊醎醏醓",6,"醜",5,"醤",5,"醫醬醰醱醲醳醶醷醸醹醻"],["e180","醼",10,"釈釋釐釒",9,"針",8,"帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺"],["e240","釦",62],["e280","鈥",32,"狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧",5,"饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂"],["e340","鉆",45,"鉵",16],["e380","銆",7,"銏",24,"恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾"],["e440","銨",5,"銯",24,"鋉",31],["e480","鋩",32,"洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑"],["e540","錊",51,"錿",10],["e580","鍊",31,"鍫濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣"],["e640","鍬",34,"鎐",27],["e680","鎬",29,"鏋鏌鏍妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩"],["e740","鏎",7,"鏗",54],["e780","鐎",32,"纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡",6,"缪缫缬缭缯",4,"缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬"],["e840","鐯",14,"鐿",43,"鑬鑭鑮鑯"],["e880","鑰",20,"钑钖钘铇铏铓铔铚铦铻锜锠琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹"],["e940","锧锳锽镃镈镋镕镚镠镮镴镵長",7,"門",42],["e980","閫",32,"椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋"],["ea40","闌",27,"闬闿阇阓阘阛阞阠阣",6,"阫阬阭阯阰阷阸阹阺阾陁陃陊陎陏陑陒陓陖陗"],["ea80","陘陙陚陜陝陞陠陣陥陦陫陭",4,"陳陸",12,"隇隉隊辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰"],["eb40","隌階隑隒隓隕隖隚際隝",9,"隨",7,"隱隲隴隵隷隸隺隻隿雂雃雈雊雋雐雑雓雔雖",9,"雡",6,"雫"],["eb80","雬雭雮雰雱雲雴雵雸雺電雼雽雿霂霃霅霊霋霌霐霑霒霔霕霗",4,"霝霟霠搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻"],["ec40","霡",8,"霫霬霮霯霱霳",4,"霺霻霼霽霿",18,"靔靕靗靘靚靜靝靟靣靤靦靧靨靪",7],["ec80","靲靵靷",4,"靽",7,"鞆",4,"鞌鞎鞏鞐鞓鞕鞖鞗鞙",4,"臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐"],["ed40","鞞鞟鞡鞢鞤",6,"鞬鞮鞰鞱鞳鞵",46],["ed80","韤韥韨韮",4,"韴韷",23,"怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨"],["ee40","頏",62],["ee80","顎",32,"睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶",4,"钼钽钿铄铈",6,"铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪"],["ef40","顯",5,"颋颎颒颕颙颣風",37,"飏飐飔飖飗飛飜飝飠",4],["ef80","飥飦飩",30,"铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒",4,"锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤",8,"镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔"],["f040","餈",4,"餎餏餑",28,"餯",26],["f080","饊",9,"饖",12,"饤饦饳饸饹饻饾馂馃馉稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨",4,"鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦",6,"鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙"],["f140","馌馎馚",10,"馦馧馩",47],["f180","駙",32,"瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃"],["f240","駺",62],["f280","騹",32,"颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒"],["f340","驚",17,"驲骃骉骍骎骔骕骙骦骩",6,"骲骳骴骵骹骻骽骾骿髃髄髆",4,"髍髎髏髐髒體髕髖髗髙髚髛髜"],["f380","髝髞髠髢髣髤髥髧髨髩髪髬髮髰",8,"髺髼",6,"鬄鬅鬆蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋"],["f440","鬇鬉",5,"鬐鬑鬒鬔",10,"鬠鬡鬢鬤",10,"鬰鬱鬳",7,"鬽鬾鬿魀魆魊魋魌魎魐魒魓魕",5],["f480","魛",32,"簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤"],["f540","魼",62],["f580","鮻",32,"酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜"],["f640","鯜",62],["f680","鰛",32,"觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅",5,"龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞",5,"鲥",4,"鲫鲭鲮鲰",7,"鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋"],["f740","鰼",62],["f780","鱻鱽鱾鲀鲃鲄鲉鲊鲌鲏鲓鲖鲗鲘鲙鲝鲪鲬鲯鲹鲾",4,"鳈鳉鳑鳒鳚鳛鳠鳡鳌",4,"鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄"],["f840","鳣",62],["f880","鴢",32],["f940","鵃",62],["f980","鶂",32],["fa40","鶣",62],["fa80","鷢",32],["fb40","鸃",27,"鸤鸧鸮鸰鸴鸻鸼鹀鹍鹐鹒鹓鹔鹖鹙鹝鹟鹠鹡鹢鹥鹮鹯鹲鹴",9,"麀"],["fb80","麁麃麄麅麆麉麊麌",5,"麔",8,"麞麠",5,"麧麨麩麪"],["fc40","麫",8,"麵麶麷麹麺麼麿",4,"黅黆黇黈黊黋黌黐黒黓黕黖黗黙黚點黡黣黤黦黨黫黬黭黮黰",8,"黺黽黿",6],["fc80","鼆",4,"鼌鼏鼑鼒鼔鼕鼖鼘鼚",5,"鼡鼣",8,"鼭鼮鼰鼱"],["fd40","鼲",4,"鼸鼺鼼鼿",4,"齅",10,"齒",38],["fd80","齹",5,"龁龂龍",11,"龜龝龞龡",4,"郎凉秊裏隣"],["fe40","兀嗀﨎﨏﨑﨓﨔礼﨟蘒﨡﨣﨤﨧﨨﨩"]]},function(t,e,n){var r=n(106);t.exports=Object("z").propertyIsEnumerable(0)?Object:function(t){return"String"==r(t)?t.split(""):Object(t)}},function(t,e){var n={}.toString;t.exports=function(t){return n.call(t).slice(8,-1)}},function(t,e){t.exports=function(t){if(null==t)throw TypeError("Can't call method on  "+t);return t}},function(t,e,n){var r=n(73),i=n(59),o=n(35),a=n(109),s=n(36),l=n(164),u=Object.getOwnPropertyDescriptor;e.f=n(14)?u:function(t,e){if(t=o(t),e=a(e,!0),l)try{return u(t,e)}catch(t){}if(s(t,e))return i(!r.f.call(t,e),t[e])}},function(t,e,n){var i=n(20);t.exports=function(t,e){if(!i(t))return t;var n,r;if(e&&"function"==typeof(n=t.toString)&&!i(r=n.call(t)))return r;if("function"==typeof(n=t.valueOf)&&!i(r=n.call(t)))return r;if(!e&&"function"==typeof(n=t.toString)&&!i(r=n.call(t)))return r;throw TypeError("Can't convert object to primitive value")}},function(t,e,n){var i=n(7),o=n(2),a=n(37);t.exports=function(t,e){var n=(o.Object||{})[t]||Object[t],r={};r[t]=e(n),i(i.S+i.F*a(function(){n(1)}),"Object",r)}},function(t,e,n){t.exports={default:n(331),__esModule:!0}},function(t,e,n){"use strict";function m(){return this}var w=n(74),x=n(7),_=n(168),k=n(27),S=n(49),C=n(334),A=n(78),P=n(337),E=n(12)("iterator"),O=!([].keys&&"next"in[].keys()),T="values";t.exports=function(t,e,n,r,i,o,a){C(n,e,r);function s(t){if(!O&&t in p)return p[t];switch(t){case"keys":case T:return function(){return new n(this,t)}}return function(){return new n(this,t)}}var l,u,c,f=e+" Iterator",h=i==T,d=!1,p=t.prototype,g=p[E]||p["@@iterator"]||i&&p[i],v=g||s(i),y=i?h?s("entries"):v:void 0,b="Array"==e&&p.entries||g;if(b&&(c=P(b.call(new t)))!==Object.prototype&&c.next&&(A(c,f,!0),w||"function"==typeof c[E]||k(c,E,m)),h&&g&&g.name!==T&&(d=!0,v=function(){return g.call(this)}),w&&!a||!O&&!d&&p[E]||k(p,E,v),S[e]=v,S[f]=m,i)if(l={values:h?v:s(T),keys:o?v:s("keys"),entries:y},a)for(u in l)u in p||_(p,u,l[u]);else x(x.P+x.F*(O||d),e,l);return l}},function(t,e){var n=Math.ceil,r=Math.floor;t.exports=function(t){return isNaN(t=+t)?0:(0<t?r:n)(t)}},function(t,e,n){var r=n(115)("keys"),i=n(77);t.exports=function(t){return r[t]||(r[t]=i(t))}},function(t,e,n){var r=n(2),i=n(21),o="__core-js_shared__",a=i[o]||(i[o]={});(t.exports=function(t,e){return a[t]||(a[t]=void 0!==e?e:{})})("versions",[]).push({version:r.version,mode:n(74)?"pure":"global",copyright:"© 2019 Denis Pushkarev (zloirock.ru)"})},function(t,e){t.exports="constructor,hasOwnProperty,isPrototypeOf,propertyIsEnumerable,toLocaleString,toString,valueOf".split(",")},function(t,e,n){var r=n(118),i=n(12)("iterator"),o=n(49);t.exports=n(2).getIteratorMethod=function(t){if(null!=t)return t[i]||t["@@iterator"]||o[r(t)]}},function(t,e,n){var i=n(106),o=n(12)("toStringTag"),a="Arguments"==i(function(){return arguments}());t.exports=function(t){var e,n,r;return void 0===t?"Undefined":null===t?"Null":"string"==typeof(n=function(t,e){try{return t[e]}catch(t){}}(e=Object(t),o))?n:a?i(e):"Object"==(r=i(e))&&"function"==typeof e.callee?"Arguments":r}},function(t,e,n){"use strict";e.__esModule=!0;var r=a(n(172)),i=a(n(344)),o="function"==typeof i.default&&"symbol"==typeof r.default?function(t){return typeof t}:function(t){return t&&"function"==typeof i.default&&t.constructor===i.default&&t!==i.default.prototype?"symbol":typeof t};function a(t){return t&&t.__esModule?t:{default:t}}e.default="function"==typeof i.default&&"symbol"===o(r.default)?function(t){return void 0===t?"undefined":o(t)}:function(t){return t&&"function"==typeof i.default&&t.constructor===i.default&&t!==i.default.prototype?"symbol":void 0===t?"undefined":o(t)}},function(t,e,n){e.f=n(12)},function(t,e,n){var r=n(21),i=n(2),o=n(74),a=n(120),s=n(17).f;t.exports=function(t){var e=i.Symbol||(i.Symbol=o?{}:r.Symbol||{});"_"==t.charAt(0)||t in e||s(e,t,{value:a.f(t)})}},function(t,e){e.f=Object.getOwnPropertySymbols},function(t,e){},function(t,e,n){t.exports={default:n(354),__esModule:!0}},function(t,e,n){var r=n(20);t.exports=function(t,e){if(!r(t)||t._t!==e)throw TypeError("Incompatible receiver, "+e+" required!");return t}},function(t,e,n){var r,i;function o(t){var e,n,r;(e="function"==typeof t.readUInt32BE&&"function"==typeof t.slice)||t instanceof Uint8Array?(t=e?(this.highStart=t.readUInt32BE(0),this.errorValue=t.readUInt32BE(4),n=t.readUInt32BE(8),t.slice(12)):(r=new DataView(t.buffer),this.highStart=r.getUint32(0),this.errorValue=r.getUint32(4),n=r.getUint32(8),t.subarray(12)),t=i(t,new Uint8Array(n)),t=i(t,new Uint8Array(n)),this.data=new Uint32Array(t.buffer)):(this.data=t.data,this.highStart=t.highStart,this.errorValue=t.errorValue)}i=n(186),o.prototype.get=function(t){var e;return t<0||1114111<t?this.errorValue:t<55296||56319<t&&t<=65535?(e=(this.data[t>>5]<<2)+(31&t),this.data[e]):t<=65535?(e=(this.data[2048+(t-55296>>5)]<<2)+(31&t),this.data[e]):t<this.highStart?(e=this.data[2080+(t>>11)],e=((e=this.data[e+(t>>5&63)])<<2)+(31&t),this.data[e]):this.data[this.data.length-4]},r=o,t.exports=r},function(t,e,n){"use strict";var r=n(0).isString,i=n(0).isNumber,g=n(0).isObject,v=n(0).isArray,y=n(0).isUndefined,s=n(191),C=/^(\s)+/g,A=/(\s)+$/g;function o(t){this.fontProvider=t}function b(t,e){var n=[];if(t=t.replace(/\t/g,"    "),e)return n.push({text:t}),n;for(var r,i=new s(t),o=0;r=i.nextBreak();){var a=t.slice(o,r.position);r.required||a.match(/\r?\n$|\r$/)?(a=a.replace(/\r?\n$|\r$/,""),n.push({text:a,lineEnd:!0})):n.push({text:a}),o=r.position}return n}function m(t,e){for(var n in e=e||{},t=t||{})"text"!=n&&t.hasOwnProperty(n)&&(e[n]=t[n]);return e}function w(t){return null==t?"":i(t)?t.toString():r(t)?t:t.toString()}function P(t,e,n,r){var i;return void 0!==t[n]&&null!==t[n]?t[n]:e?(e.auto(t,function(){i=e.getProperty(n)}),null!=i?i:r):r}function E(t,e,n,r,i){return e.widthOfString(t,n,i)+(r||0)*(t.length-1)}o.prototype.buildInlines=function(t,e){var n,r=function(k,t,S){var e=function(t,e){function n(t,e,n){if(y(e[t]))return null;if(e[t].lineEnd)return null;var r=e[t].text;if(n){var i=b(w(r),!1);if(y(i[i.length-1]))return null;r=i[i.length-1].text}return r}var r=[];v(t)||(t=[t]);t=function i(t){return t.reduce(function(t,e){var n=v(e.text)?i(e.text):e,r=[].concat(n).some(Array.isArray);return t.concat(r?i(n):n)},[])}(t);for(var i=null,o=0,a=t.length;o<a;o++){var s,l=t[o],u=null,c=P(l||{},e,"noWrap",!1);if(g(l)?(l._textRef&&l._textRef._textNodeRef.text&&(l.text=l._textRef._textNodeRef.text),s=b(w(l.text),c),u=m(l)):s=b(w(l),c),i&&s.length){var f=n(0,s,c);1===b(w(i+f),!1).length&&(r[r.length-1].noNewLine=!0)}for(var h=0,d=s.length;h<d;h++){var p={text:s[h].text};s[h].lineEnd&&(p.lineEnd=!0),m(u,p),r.push(p)}i=null,o+1<a&&(i=n(s.length-1,s,c))}return r}(t,S);if(e.length){var n=P(e[0],S,"leadingIndent",0);n&&(e[0].leadingCut=-n,e[0].leadingIndent=n)}return e.forEach(function(t){var e,n,r=P(t,S,"font","Roboto"),i=P(t,S,"fontSize",12),o=P(t,S,"fontFeatures",null),a=P(t,S,"bold",!1),s=P(t,S,"italics",!1),l=P(t,S,"color","black"),u=P(t,S,"decoration",null),c=P(t,S,"decorationColor",null),f=P(t,S,"decorationStyle",null),h=P(t,S,"background",null),d=P(t,S,"lineHeight",1),p=P(t,S,"characterSpacing",0),g=P(t,S,"link",null),v=P(t,S,"linkToPage",null),y=P(t,S,"linkToDestination",null),b=P(t,S,"noWrap",null),m=P(t,S,"preserveLeadingSpaces",!1),w=P(t,S,"preserveTrailingSpaces",!1),x=P(t,S,"opacity",1),_=k.provideFont(r,a,s);t.width=E(t.text,_,i,p,o),t.height=_.lineHeight(i)*d,t.leadingCut||(t.leadingCut=0),!m&&(e=t.text.match(C))&&(t.leadingCut+=E(e[0],_,i,p,o)),!w&&(n=t.text.match(A))?t.trailingCut=E(n[0],_,i,p,o):t.trailingCut=0,t.alignment=P(t,S,"alignment","left"),t.font=_,t.fontSize=i,t.fontFeatures=o,t.characterSpacing=p,t.color=l,t.decoration=u,t.decorationColor=c,t.decorationStyle=f,t.background=h,t.link=g,t.linkToPage=v,t.linkToDestination=y,t.noWrap=b,t.opacity=x}),e}(this.fontProvider,t,e),i=0,o=0;return r.forEach(function(t){i=Math.max(i,t.width-t.leadingCut-t.trailingCut),(n=n||{width:0,leadingCut:t.leadingCut,trailingCut:0}).width+=t.width,n.trailingCut=t.trailingCut,o=Math.max(o,function(t){return Math.max(0,t.width-t.leadingCut-t.trailingCut)}(n)),t.lineEnd&&(n=null)}),P({},e,"noWrap",!1)&&(i=o),{items:r,minWidth:i,maxWidth:o}},o.prototype.sizeOfString=function(t,e){t=t?t.toString().replace(/\t/g,"    "):"";var n=P({},e,"font","Roboto"),r=P({},e,"fontSize",12),i=P({},e,"fontFeatures",null),o=P({},e,"bold",!1),a=P({},e,"italics",!1),s=P({},e,"lineHeight",1),l=P({},e,"characterSpacing",0),u=this.fontProvider.provideFont(n,o,a);return{width:E(t,u,r,l,i),height:u.lineHeight(r)*s,fontSize:r,lineHeight:s,ascender:u.ascender/1e3*r,descender:u.descender/1e3*r}},o.prototype.widthOfString=function(t,e,n,r,i){return E(t,e,n,r,i)},t.exports=o},function(t,e,n){"use strict";var g=n(0).isString;function v(t){return"auto"===t.width}function y(t){return null===t.width||void 0===t.width||"*"===t.width||"star"===t.width}t.exports={buildColumnWidths:function(t,n){var e=[],r=0,i=0,o=[],a=0,s=0,l=[],u=n;t.forEach(function(t){v(t)?(e.push(t),r+=t._minWidth,i+=t._maxWidth):y(t)?(o.push(t),a=Math.max(a,t._minWidth),s=Math.max(s,t._maxWidth)):l.push(t)}),l.forEach(function(t){g(t.width)&&/\d+%/.test(t.width)&&(t.width=parseFloat(t.width)*u/100),t.width<t._minWidth&&t.elasticWidth?t._calcWidth=t._minWidth:t._calcWidth=t.width,n-=t._calcWidth});var c=r+a*o.length,f=i+s*o.length;if(n<=c)e.forEach(function(t){t._calcWidth=t._minWidth}),o.forEach(function(t){t._calcWidth=a});else{if(f<n)e.forEach(function(t){t._calcWidth=t._maxWidth,n-=t._calcWidth});else{var h=n-c,d=f-c;e.forEach(function(t){var e=t._maxWidth-t._minWidth;t._calcWidth=t._minWidth+e*h/d,n-=t._calcWidth})}if(0<o.length){var p=n/o.length;o.forEach(function(t){t._calcWidth=p})}}},measureMinMax:function(t){for(var e={min:0,max:0},n={min:0,max:0},r=0,i=0,o=t.length;i<o;i++){var a=t[i];y(a)?(n.min=Math.max(n.min,a._minWidth),n.max=Math.max(n.max,a._maxWidth),r++):v(a)?(e.min+=a._minWidth,e.max+=a._maxWidth):(e.min+=void 0!==a.width&&a.width||a._minWidth,e.max+=void 0!==a.width&&a.width||a._maxWidth)}return r&&(e.min+=r*n.min,e.max+=r*n.max),e},isAutoColumn:v,isStarColumn:y}},function(t,e){var n={}.toString;t.exports=Array.isArray||function(t){return"[object Array]"==n.call(t)}},function(t,e,n){function r(t,e,n,r){var i=String(a(t)),o="<"+e;return""!==n&&(o+=" "+n+'="'+String(r).replace(s,"&quot;")+'"'),o+">"+i+"</"+e+">"}var i=n(6),o=n(10),a=n(30),s=/"/g;t.exports=function(e,t){var n={};n[e]=t(r),i(i.P+i.F*o(function(){var t=""[e]('"');return t!==t.toLowerCase()||3<t.split('"').length}),"String",n)}},function(t,e,n){t.exports=!n(9)&&!n(10)(function(){return 7!=Object.defineProperty(n(132)("div"),"a",{get:function(){return 7}}).a})},function(t,e,n){var r=n(18),i=n(8).document,o=r(i)&&r(i.createElement);t.exports=function(t){return o?i.createElement(t):{}}},function(t,e){t.exports=function(t){if("function"!=typeof t)throw TypeError(t+" is not a function!");return t}},function(t,e,n){var a=n(23),s=n(43),l=n(82)(!1),u=n(83)("IE_PROTO");t.exports=function(t,e){var n,r=s(t),i=0,o=[];for(n in r)n!=u&&a(r,n)&&o.push(n);for(;e.length>i;)a(r,n=e[i++])&&(~l(o,n)||o.push(n));return o}},function(t,e,n){for(var r,i=n(8),o=n(15),a=n(29),s=a("typed_array"),l=a("view"),u=!(!i.ArrayBuffer||!i.DataView),c=u,f=0,h="Int8Array,Uint8Array,Uint8ClampedArray,Int16Array,Uint16Array,Int32Array,Uint32Array,Float32Array,Float64Array".split(",");f<9;)(r=i[h[f++]])?(o(r.prototype,s,!0),o(r.prototype,l,!0)):c=!1;t.exports={ABV:u,CONSTR:c,TYPED:s,VIEW:l}},function(t,e,n){var i=n(22);t.exports=function(t,e,n){for(var r in e)i(t,r,e[r],n);return t}},function(t,e){t.exports=function(t,e,n,r){if(!(t instanceof e)||void 0!==r&&r in t)throw TypeError(n+": incorrect invocation!");return t}},function(t,e,n){var r=n(31),i=n(16);t.exports=function(t){if(void 0===t)return 0;var e=r(t),n=i(e);if(e!==n)throw RangeError("Wrong length!");return n}},function(t,e,n){var r=n(44),i=n(3)("iterator"),o=Array.prototype;t.exports=function(t){return void 0!==t&&(r.Array===t||o[i]===t)}},function(t,e,n){var r=n(23),i=n(19),o=n(83)("IE_PROTO"),a=Object.prototype;t.exports=Object.getPrototypeOf||function(t){return t=i(t),r(t,o)?t[o]:"function"==typeof t.constructor&&t instanceof t.constructor?t.constructor.prototype:t instanceof Object?a:null}},function(t,e,n){var r=n(65),i=n(3)("iterator"),o=n(44);t.exports=n(39).getIteratorMethod=function(t){if(null!=t)return t[i]||t["@@iterator"]||o[r(t)]}},function(t,e,n){var r=n(55);t.exports=Array.isArray||function(t){return"Array"==r(t)}},function(t,e,n){var i=n(13),o=n(133),a=n(3)("species");t.exports=function(t,e){var n,r=i(t).constructor;return void 0===r||null==(n=i(r)[a])?e:o(n)}},function(t,e,n){"use strict";function m(){return this}var w=n(41),x=n(6),_=n(22),k=n(15),S=n(44),C=n(219),A=n(64),P=n(140),E=n(3)("iterator"),O=!([].keys&&"next"in[].keys()),T="values";t.exports=function(t,e,n,r,i,o,a){C(n,e,r);function s(t){if(!O&&t in p)return p[t];switch(t){case"keys":case T:return function(){return new n(this,t)}}return function(){return new n(this,t)}}var l,u,c,f=e+" Iterator",h=i==T,d=!1,p=t.prototype,g=p[E]||p["@@iterator"]||i&&p[i],v=g||s(i),y=i?h?s("entries"):v:void 0,b="Array"==e&&p.entries||g;if(b&&(c=P(b.call(new t)))!==Object.prototype&&c.next&&(A(c,f,!0),w||"function"==typeof c[E]||k(c,E,m)),h&&g&&g.name!==T&&(d=!0,v=function(){return g.call(this)}),w&&!a||!O&&!d&&p[E]||k(p,E,v),S[e]=v,S[f]=m,i)if(l={values:h?v:s(T),keys:o?v:s("keys"),entries:y},a)for(u in l)u in p||_(p,u,l[u]);else x(x.P+x.F*(O||d),e,l);return l}},function(t,e,n){var o=n(3)("iterator"),a=!1;try{var r=[7][o]();r.return=function(){a=!0},Array.from(r,function(){throw 2})}catch(t){}t.exports=function(t,e){if(!e&&!a)return!1;var n=!1;try{var r=[7],i=r[o]();i.next=function(){return{done:n=!0}},r[o]=function(){return i},t(r)}catch(t){}return n}},function(t,e,n){var r=n(18),i=n(55),o=n(3)("match");t.exports=function(t){var e;return r(t)&&(void 0!==(e=t[o])?!!e:"RegExp"==i(t))}},function(t,e,i){function o(t,e){if(r(t),!n(e)&&null!==e)throw TypeError(e+": can't set as prototype!")}var n=i(18),r=i(13);t.exports={set:Object.setPrototypeOf||("__proto__"in{}?function(t,n,r){try{(r=i(54)(Function.call,i(67).f(Object.prototype,"__proto__").set,2))(t,[]),n=!(t instanceof Array)}catch(t){n=!0}return function(t,e){return o(t,e),n?t.__proto__=e:r(t,e),t}}({},!1):void 0),check:o}},function(t,e,n){var r=n(8),i=n(39),o=n(41),a=n(149),s=n(11).f;t.exports=function(t){var e=i.Symbol||(i.Symbol=o?{}:r.Symbol||{});"_"==t.charAt(0)||t in e||s(e,t,{value:a.f(t)})}},function(t,e,n){e.f=n(3)},function(L,t,R){"use strict";(function(t,g){var v=R(69);L.exports=h;var a,y=R(129);h.ReadableState=o;function b(t,e){return t.listeners(e).length}R(68).EventEmitter;var i=R(151),s=R(70).Buffer,l=t.Uint8Array||function(){};var e=R(58);e.inherits=R(32);var n=R(254),m=void 0;m=n&&n.debuglog?n.debuglog("stream"):function(){};var u,c=R(255),r=R(152);e.inherits(h,i);var f=["error","close","destroy","pause","resume"];function o(t,e){t=t||{};var n=e instanceof(a=a||R(33));this.objectMode=!!t.objectMode,n&&(this.objectMode=this.objectMode||!!t.readableObjectMode);var r=t.highWaterMark,i=t.readableHighWaterMark,o=this.objectMode?16:16384;this.highWaterMark=r||0===r?r:n&&(i||0===i)?i:o,this.highWaterMark=Math.floor(this.highWaterMark),this.buffer=new c,this.length=0,this.pipes=null,this.pipesCount=0,this.flowing=null,this.ended=!1,this.endEmitted=!1,this.reading=!1,this.sync=!0,this.needReadable=!1,this.emittedReadable=!1,this.readableListening=!1,this.resumeScheduled=!1,this.destroyed=!1,this.defaultEncoding=t.defaultEncoding||"utf8",this.awaitDrain=0,this.readingMore=!1,this.decoder=null,this.encoding=null,t.encoding&&(u=u||R(98).StringDecoder,this.decoder=new u(t.encoding),this.encoding=t.encoding)}function h(t){if(a=a||R(33),!(this instanceof h))return new h(t);this._readableState=new o(t,this),this.readable=!0,t&&("function"==typeof t.read&&(this._read=t.read),"function"==typeof t.destroy&&(this._destroy=t.destroy)),i.call(this)}function d(t,e,n,r,i){var o,a=t._readableState;null===e?(a.reading=!1,function(t,e){if(e.ended)return;if(e.decoder){var n=e.decoder.end();n&&n.length&&(e.buffer.push(n),e.length+=e.objectMode?1:n.length)}e.ended=!0,_(t)}(t,a)):(i||(o=function(t,e){var n;(function(t){return s.isBuffer(t)||t instanceof l})(e)||"string"==typeof e||void 0===e||t.objectMode||(n=new TypeError("Invalid non-string/buffer chunk"));return n}(a,e)),o?t.emit("error",o):a.objectMode||e&&0<e.length?("string"==typeof e||a.objectMode||Object.getPrototypeOf(e)===s.prototype||(e=function(t){return s.from(t)}(e)),r?a.endEmitted?t.emit("error",new Error("stream.unshift() after end event")):p(t,a,e,!0):a.ended?t.emit("error",new Error("stream.push() after EOF")):(a.reading=!1,a.decoder&&!n?(e=a.decoder.write(e),a.objectMode||0!==e.length?p(t,a,e,!1):S(t,a)):p(t,a,e,!1))):r||(a.reading=!1));return function(t){return!t.ended&&(t.needReadable||t.length<t.highWaterMark||0===t.length)}(a)}function p(t,e,n,r){e.flowing&&0===e.length&&!e.sync?(t.emit("data",n),t.read(0)):(e.length+=e.objectMode?1:n.length,r?e.buffer.unshift(n):e.buffer.push(n),e.needReadable&&_(t)),S(t,e)}Object.defineProperty(h.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&this._readableState.destroyed},set:function(t){this._readableState&&(this._readableState.destroyed=t)}}),h.prototype.destroy=r.destroy,h.prototype._undestroy=r.undestroy,h.prototype._destroy=function(t,e){this.push(null),e(t)},h.prototype.push=function(t,e){var n,r=this._readableState;return r.objectMode?n=!0:"string"==typeof t&&((e=e||r.defaultEncoding)!==r.encoding&&(t=s.from(t,e),e=""),n=!0),d(this,t,e,!1,n)},h.prototype.unshift=function(t){return d(this,t,null,!0,!1)},h.prototype.isPaused=function(){return!1===this._readableState.flowing},h.prototype.setEncoding=function(t){return u=u||R(98).StringDecoder,this._readableState.decoder=new u(t),this._readableState.encoding=t,this};var w=8388608;function x(t,e){return t<=0||0===e.length&&e.ended?0:e.objectMode?1:t!=t?e.flowing&&e.length?e.buffer.head.data.length:e.length:(t>e.highWaterMark&&(e.highWaterMark=function(t){return w<=t?t=w:(t--,t|=t>>>1,t|=t>>>2,t|=t>>>4,t|=t>>>8,t|=t>>>16,t++),t}(t)),t<=e.length?t:e.ended?e.length:(e.needReadable=!0,0))}function _(t){var e=t._readableState;e.needReadable=!1,e.emittedReadable||(m("emitReadable",e.flowing),e.emittedReadable=!0,e.sync?v.nextTick(k,t):k(t))}function k(t){m("emit readable"),t.emit("readable"),E(t)}function S(t,e){e.readingMore||(e.readingMore=!0,v.nextTick(C,t,e))}function C(t,e){for(var n=e.length;!e.reading&&!e.flowing&&!e.ended&&e.length<e.highWaterMark&&(m("maybeReadMore read 0"),t.read(0),n!==e.length);)n=e.length;e.readingMore=!1}function A(t){m("readable nexttick read 0"),t.read(0)}function P(t,e){e.reading||(m("resume read 0"),t.read(0)),e.resumeScheduled=!1,e.awaitDrain=0,t.emit("resume"),E(t),e.flowing&&!e.reading&&t.read(0)}function E(t){var e=t._readableState;for(m("flow",e.flowing);e.flowing&&null!==t.read(););}function O(t,e){return 0===e.length?null:(e.objectMode?n=e.buffer.shift():!t||t>=e.length?(n=e.decoder?e.buffer.join(""):1===e.buffer.length?e.buffer.head.data:e.buffer.concat(e.length),e.buffer.clear()):n=function(t,e,n){var r;t<e.head.data.length?(r=e.head.data.slice(0,t),e.head.data=e.head.data.slice(t)):r=t===e.head.data.length?e.shift():n?function(t,e){var n=e.head,r=1,i=n.data;t-=i.length;for(;n=n.next;){var o=n.data,a=t>o.length?o.length:t;if(a===o.length?i+=o:i+=o.slice(0,t),0===(t-=a)){a===o.length?(++r,n.next?e.head=n.next:e.head=e.tail=null):(e.head=n).data=o.slice(a);break}++r}return e.length-=r,i}(t,e):function(t,e){var n=s.allocUnsafe(t),r=e.head,i=1;r.data.copy(n),t-=r.data.length;for(;r=r.next;){var o=r.data,a=t>o.length?o.length:t;if(o.copy(n,n.length-t,0,a),0===(t-=a)){a===o.length?(++i,r.next?e.head=r.next:e.head=e.tail=null):(e.head=r).data=o.slice(a);break}++i}return e.length-=i,n}(t,e);return r}(t,e.buffer,e.decoder),n);var n}function T(t){var e=t._readableState;if(0<e.length)throw new Error('"endReadable()" called on non-empty stream');e.endEmitted||(e.ended=!0,v.nextTick(I,e,t))}function I(t,e){t.endEmitted||0!==t.length||(t.endEmitted=!0,e.readable=!1,e.emit("end"))}function B(t,e){for(var n=0,r=t.length;n<r;n++)if(t[n]===e)return n;return-1}h.prototype.read=function(t){m("read",t),t=parseInt(t,10);var e=this._readableState,n=t;if(0!==t&&(e.emittedReadable=!1),0===t&&e.needReadable&&(e.length>=e.highWaterMark||e.ended))return m("read: emitReadable",e.length,e.ended),0===e.length&&e.ended?T(this):_(this),null;if(0===(t=x(t,e))&&e.ended)return 0===e.length&&T(this),null;var r,i=e.needReadable;return m("need readable",i),(0===e.length||e.length-t<e.highWaterMark)&&m("length less than watermark",i=!0),e.ended||e.reading?m("reading or ended",i=!1):i&&(m("do read"),e.reading=!0,e.sync=!0,0===e.length&&(e.needReadable=!0),this._read(e.highWaterMark),e.sync=!1,e.reading||(t=x(n,e))),null===(r=0<t?O(t,e):null)?(e.needReadable=!0,t=0):e.length-=t,0===e.length&&(e.ended||(e.needReadable=!0),n!==t&&e.ended&&T(this)),null!==r&&this.emit("data",r),r},h.prototype._read=function(t){this.emit("error",new Error("_read() is not implemented"))},h.prototype.pipe=function(n,t){var r=this,i=this._readableState;switch(i.pipesCount){case 0:i.pipes=n;break;case 1:i.pipes=[i.pipes,n];break;default:i.pipes.push(n)}i.pipesCount+=1,m("pipe count=%d opts=%j",i.pipesCount,t);var e=(!t||!1!==t.end)&&n!==g.stdout&&n!==g.stderr?a:p;function o(t,e){m("onunpipe"),t===r&&e&&!1===e.hasUnpiped&&(e.hasUnpiped=!0,m("cleanup"),n.removeListener("close",h),n.removeListener("finish",d),n.removeListener("drain",s),n.removeListener("error",f),n.removeListener("unpipe",o),r.removeListener("end",a),r.removeListener("end",p),r.removeListener("data",c),l=!0,!i.awaitDrain||n._writableState&&!n._writableState.needDrain||s())}function a(){m("onend"),n.end()}i.endEmitted?v.nextTick(e):r.once("end",e),n.on("unpipe",o);var s=function(e){return function(){var t=e._readableState;m("pipeOnDrain",t.awaitDrain),t.awaitDrain&&t.awaitDrain--,0===t.awaitDrain&&b(e,"data")&&(t.flowing=!0,E(e))}}(r);n.on("drain",s);var l=!1;var u=!1;function c(t){m("ondata"),(u=!1)!==n.write(t)||u||((1===i.pipesCount&&i.pipes===n||1<i.pipesCount&&-1!==B(i.pipes,n))&&!l&&(m("false write response, pause",r._readableState.awaitDrain),r._readableState.awaitDrain++,u=!0),r.pause())}function f(t){m("onerror",t),p(),n.removeListener("error",f),0===b(n,"error")&&n.emit("error",t)}function h(){n.removeListener("finish",d),p()}function d(){m("onfinish"),n.removeListener("close",h),p()}function p(){m("unpipe"),r.unpipe(n)}return r.on("data",c),function(t,e,n){if("function"==typeof t.prependListener)return t.prependListener(e,n);t._events&&t._events[e]?y(t._events[e])?t._events[e].unshift(n):t._events[e]=[n,t._events[e]]:t.on(e,n)}(n,"error",f),n.once("close",h),n.once("finish",d),n.emit("pipe",r),i.flowing||(m("pipe resume"),r.resume()),n},h.prototype.unpipe=function(t){var e=this._readableState,n={hasUnpiped:!1};if(0===e.pipesCount)return this;if(1===e.pipesCount)return t&&t!==e.pipes||(t=t||e.pipes,e.pipes=null,e.pipesCount=0,e.flowing=!1,t&&t.emit("unpipe",this,n)),this;if(!t){var r=e.pipes,i=e.pipesCount;e.pipes=null,e.pipesCount=0,e.flowing=!1;for(var o=0;o<i;o++)r[o].emit("unpipe",this,n);return this}var a=B(e.pipes,t);return-1===a||(e.pipes.splice(a,1),e.pipesCount-=1,1===e.pipesCount&&(e.pipes=e.pipes[0]),t.emit("unpipe",this,n)),this},h.prototype.addListener=h.prototype.on=function(t,e){var n=i.prototype.on.call(this,t,e);if("data"===t)!1!==this._readableState.flowing&&this.resume();else if("readable"===t){var r=this._readableState;r.endEmitted||r.readableListening||(r.readableListening=r.needReadable=!0,r.emittedReadable=!1,r.reading?r.length&&_(this):v.nextTick(A,this))}return n},h.prototype.resume=function(){var t=this._readableState;return t.flowing||(m("resume"),t.flowing=!0,function(t,e){e.resumeScheduled||(e.resumeScheduled=!0,v.nextTick(P,t,e))}(this,t)),this},h.prototype.pause=function(){return m("call pause flowing=%j",this._readableState.flowing),!1!==this._readableState.flowing&&(m("pause"),this._readableState.flowing=!1,this.emit("pause")),this},h.prototype.wrap=function(e){var n=this,r=this._readableState,i=!1;for(var t in e.on("end",function(){if(m("wrapped end"),r.decoder&&!r.ended){var t=r.decoder.end();t&&t.length&&n.push(t)}n.push(null)}),e.on("data",function(t){m("wrapped data"),r.decoder&&(t=r.decoder.write(t)),r.objectMode&&null==t||(r.objectMode||t&&t.length)&&(n.push(t)||(i=!0,e.pause()))}),e)void 0===this[t]&&"function"==typeof e[t]&&(this[t]=function(t){return function(){return e[t].apply(e,arguments)}}(t));for(var o=0;o<f.length;o++)e.on(f[o],this.emit.bind(this,f[o]));return this._read=function(t){m("wrapped _read",t),i&&(i=!1,e.resume())},this},Object.defineProperty(h.prototype,"readableHighWaterMark",{enumerable:!1,get:function(){return this._readableState.highWaterMark}}),h._fromList=O}).call(this,R(25),R(24))},function(t,e,n){t.exports=n(68).EventEmitter},function(t,e,n){"use strict";var o=n(69);function a(t,e){t.emit("error",e)}t.exports={destroy:function(t,e){var n=this,r=this._readableState&&this._readableState.destroyed,i=this._writableState&&this._writableState.destroyed;return r||i?e?e(t):!t||this._writableState&&this._writableState.errorEmitted||o.nextTick(a,this,t):(this._readableState&&(this._readableState.destroyed=!0),this._writableState&&(this._writableState.destroyed=!0),this._destroy(t||null,function(t){!e&&t?(o.nextTick(a,n,t),n._writableState&&(n._writableState.errorEmitted=!0)):e&&e(t)})),this},undestroy:function(){this._readableState&&(this._readableState.destroyed=!1,this._readableState.reading=!1,this._readableState.ended=!1,this._readableState.endEmitted=!1),this._writableState&&(this._writableState.destroyed=!1,this._writableState.ended=!1,this._writableState.ending=!1,this._writableState.finished=!1,this._writableState.errorEmitted=!1)}}},function(t,e,n){"use strict";t.exports=o;var r=n(33),i=n(58);function o(t){if(!(this instanceof o))return new o(t);r.call(this,t),this._transformState={afterTransform:function(t,e){var n=this._transformState;n.transforming=!1;var r=n.writecb;if(!r)return this.emit("error",new Error("write callback called multiple times"));n.writechunk=null,(n.writecb=null)!=e&&this.push(e),r(t);var i=this._readableState;i.reading=!1,(i.needReadable||i.length<i.highWaterMark)&&this._read(i.highWaterMark)}.bind(this),needTransform:!1,transforming:!1,writecb:null,writechunk:null,writeencoding:null},this._readableState.needReadable=!0,this._readableState.sync=!1,t&&("function"==typeof t.transform&&(this._transform=t.transform),"function"==typeof t.flush&&(this._flush=t.flush)),this.on("prefinish",a)}function a(){var n=this;"function"==typeof this._flush?this._flush(function(t,e){s(n,t,e)}):s(this,null,null)}function s(t,e,n){if(e)return t.emit("error",e);if(null!=n&&t.push(n),t._writableState.length)throw new Error("Calling transform done when ws.length != 0");if(t._transformState.transforming)throw new Error("Calling transform done when still transforming");return t.push(null)}i.inherits=n(32),i.inherits(o,r),o.prototype.push=function(t,e){return this._transformState.needTransform=!1,r.prototype.push.call(this,t,e)},o.prototype._transform=function(t,e,n){throw new Error("_transform() is not implemented")},o.prototype._write=function(t,e,n){var r=this._transformState;if(r.writecb=n,r.writechunk=t,r.writeencoding=e,!r.transforming){var i=this._readableState;(r.needTransform||i.needReadable||i.length<i.highWaterMark)&&this._read(i.highWaterMark)}},o.prototype._read=function(t){var e=this._transformState;null!==e.writechunk&&e.writecb&&!e.transforming?(e.transforming=!0,this._transform(e.writechunk,e.writeencoding,e.afterTransform)):e.needTransform=!0},o.prototype._destroy=function(t,e){var n=this;r.prototype._destroy.call(this,t,function(t){e(t),n.emit("close")})}},function(t,E,O){"use strict";(function(i){var v=O(4).Buffer,a=O(95).Transform,s=O(263),t=O(99),y=O(155).ok,b=O(4).kMaxLength,m="Cannot create final Buffer. It would be larger than 0x"+b.toString(16)+" bytes";s.Z_MIN_WINDOWBITS=8,s.Z_MAX_WINDOWBITS=15,s.Z_DEFAULT_WINDOWBITS=15,s.Z_MIN_CHUNK=64,s.Z_MAX_CHUNK=1/0,s.Z_DEFAULT_CHUNK=16384,s.Z_MIN_MEMLEVEL=1,s.Z_MAX_MEMLEVEL=9,s.Z_DEFAULT_MEMLEVEL=8,s.Z_MIN_LEVEL=-1,s.Z_MAX_LEVEL=9,s.Z_DEFAULT_LEVEL=s.Z_DEFAULT_COMPRESSION;for(var e=Object.keys(s),n=0;n<e.length;n++){var r=e[n];r.match(/^Z/)&&Object.defineProperty(E,r,{enumerable:!0,value:s[r],writable:!1})}for(var o={Z_OK:s.Z_OK,Z_STREAM_END:s.Z_STREAM_END,Z_NEED_DICT:s.Z_NEED_DICT,Z_ERRNO:s.Z_ERRNO,Z_STREAM_ERROR:s.Z_STREAM_ERROR,Z_DATA_ERROR:s.Z_DATA_ERROR,Z_MEM_ERROR:s.Z_MEM_ERROR,Z_BUF_ERROR:s.Z_BUF_ERROR,Z_VERSION_ERROR:s.Z_VERSION_ERROR},l=Object.keys(o),u=0;u<l.length;u++){var c=l[u];o[o[c]]=c}function f(n,t,r){var i=[],o=0;function e(){for(var t;null!==(t=n.read());)i.push(t),o+=t.length;n.once("readable",e)}function a(){var t,e=null;b<=o?e=new RangeError(m):t=v.concat(i,o),i=[],n.close(),r(e,t)}n.on("error",function(t){n.removeListener("end",a),n.removeListener("readable",e),r(t)}),n.on("end",a),n.end(t),e()}function h(t,e){if("string"==typeof e&&(e=v.from(e)),!v.isBuffer(e))throw new TypeError("Not a string or buffer");var n=t._finishFlushFlag;return t._processChunk(e,n)}function d(t){if(!(this instanceof d))return new d(t);C.call(this,t,s.DEFLATE)}function p(t){if(!(this instanceof p))return new p(t);C.call(this,t,s.INFLATE)}function g(t){if(!(this instanceof g))return new g(t);C.call(this,t,s.GZIP)}function w(t){if(!(this instanceof w))return new w(t);C.call(this,t,s.GUNZIP)}function x(t){if(!(this instanceof x))return new x(t);C.call(this,t,s.DEFLATERAW)}function _(t){if(!(this instanceof _))return new _(t);C.call(this,t,s.INFLATERAW)}function k(t){if(!(this instanceof k))return new k(t);C.call(this,t,s.UNZIP)}function S(t){return t===s.Z_NO_FLUSH||t===s.Z_PARTIAL_FLUSH||t===s.Z_SYNC_FLUSH||t===s.Z_FULL_FLUSH||t===s.Z_FINISH||t===s.Z_BLOCK}function C(t,e){var n=this;if(this._opts=t=t||{},this._chunkSize=t.chunkSize||E.Z_DEFAULT_CHUNK,a.call(this,t),t.flush&&!S(t.flush))throw new Error("Invalid flush flag: "+t.flush);if(t.finishFlush&&!S(t.finishFlush))throw new Error("Invalid flush flag: "+t.finishFlush);if(this._flushFlag=t.flush||s.Z_NO_FLUSH,this._finishFlushFlag=void 0!==t.finishFlush?t.finishFlush:s.Z_FINISH,t.chunkSize&&(t.chunkSize<E.Z_MIN_CHUNK||t.chunkSize>E.Z_MAX_CHUNK))throw new Error("Invalid chunk size: "+t.chunkSize);if(t.windowBits&&(t.windowBits<E.Z_MIN_WINDOWBITS||t.windowBits>E.Z_MAX_WINDOWBITS))throw new Error("Invalid windowBits: "+t.windowBits);if(t.level&&(t.level<E.Z_MIN_LEVEL||t.level>E.Z_MAX_LEVEL))throw new Error("Invalid compression level: "+t.level);if(t.memLevel&&(t.memLevel<E.Z_MIN_MEMLEVEL||t.memLevel>E.Z_MAX_MEMLEVEL))throw new Error("Invalid memLevel: "+t.memLevel);if(t.strategy&&t.strategy!=E.Z_FILTERED&&t.strategy!=E.Z_HUFFMAN_ONLY&&t.strategy!=E.Z_RLE&&t.strategy!=E.Z_FIXED&&t.strategy!=E.Z_DEFAULT_STRATEGY)throw new Error("Invalid strategy: "+t.strategy);if(t.dictionary&&!v.isBuffer(t.dictionary))throw new Error("Invalid dictionary: it should be a Buffer instance");this._handle=new s.Zlib(e);var r=this;this._hadError=!1,this._handle.onerror=function(t,e){A(r),r._hadError=!0;var n=new Error(t);n.errno=e,n.code=E.codes[e],r.emit("error",n)};var i=E.Z_DEFAULT_COMPRESSION;"number"==typeof t.level&&(i=t.level);var o=E.Z_DEFAULT_STRATEGY;"number"==typeof t.strategy&&(o=t.strategy),this._handle.init(t.windowBits||E.Z_DEFAULT_WINDOWBITS,i,t.memLevel||E.Z_DEFAULT_MEMLEVEL,o,t.dictionary),this._buffer=v.allocUnsafe(this._chunkSize),this._offset=0,this._level=i,this._strategy=o,this.once("end",this.close),Object.defineProperty(this,"_closed",{get:function(){return!n._handle},configurable:!0,enumerable:!0})}function A(t,e){e&&i.nextTick(e),t._handle&&(t._handle.close(),t._handle=null)}function P(t){t.emit("close")}Object.defineProperty(E,"codes",{enumerable:!0,value:Object.freeze(o),writable:!1}),E.Deflate=d,E.Inflate=p,E.Gzip=g,E.Gunzip=w,E.DeflateRaw=x,E.InflateRaw=_,E.Unzip=k,E.createDeflate=function(t){return new d(t)},E.createInflate=function(t){return new p(t)},E.createDeflateRaw=function(t){return new x(t)},E.createInflateRaw=function(t){return new _(t)},E.createGzip=function(t){return new g(t)},E.createGunzip=function(t){return new w(t)},E.createUnzip=function(t){return new k(t)},E.deflate=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new d(e),t,n)},E.deflateSync=function(t,e){return h(new d(e),t)},E.gzip=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new g(e),t,n)},E.gzipSync=function(t,e){return h(new g(e),t)},E.deflateRaw=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new x(e),t,n)},E.deflateRawSync=function(t,e){return h(new x(e),t)},E.unzip=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new k(e),t,n)},E.unzipSync=function(t,e){return h(new k(e),t)},E.inflate=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new p(e),t,n)},E.inflateSync=function(t,e){return h(new p(e),t)},E.gunzip=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new w(e),t,n)},E.gunzipSync=function(t,e){return h(new w(e),t)},E.inflateRaw=function(t,e,n){return"function"==typeof e&&(n=e,e={}),f(new _(e),t,n)},E.inflateRawSync=function(t,e){return h(new _(e),t)},t.inherits(C,a),C.prototype.params=function(t,e,n){if(t<E.Z_MIN_LEVEL||t>E.Z_MAX_LEVEL)throw new RangeError("Invalid compression level: "+t);if(e!=E.Z_FILTERED&&e!=E.Z_HUFFMAN_ONLY&&e!=E.Z_RLE&&e!=E.Z_FIXED&&e!=E.Z_DEFAULT_STRATEGY)throw new TypeError("Invalid strategy: "+e);if(this._level!==t||this._strategy!==e){var r=this;this.flush(s.Z_SYNC_FLUSH,function(){y(r._handle,"zlib binding closed"),r._handle.params(t,e),r._hadError||(r._level=t,r._strategy=e,n&&n())})}else i.nextTick(n)},C.prototype.reset=function(){return y(this._handle,"zlib binding closed"),this._handle.reset()},C.prototype._flush=function(t){this._transform(v.alloc(0),"",t)},C.prototype.flush=function(t,e){var n=this,r=this._writableState;"function"!=typeof t&&(void 0!==t||e)||(e=t,t=s.Z_FULL_FLUSH),r.ended?e&&i.nextTick(e):r.ending?e&&this.once("end",e):r.needDrain?e&&this.once("drain",function(){return n.flush(t,e)}):(this._flushFlag=t,this.write(v.alloc(0),"",e))},C.prototype.close=function(t){A(this,t),i.nextTick(P,this)},C.prototype._transform=function(t,e,n){var r,i=this._writableState,o=(i.ending||i.ended)&&(!t||i.length===t.length);return null===t||v.isBuffer(t)?this._handle?(o?r=this._finishFlushFlag:(r=this._flushFlag,t.length>=i.length&&(this._flushFlag=this._opts.flush||s.Z_NO_FLUSH)),void this._processChunk(t,r,n)):n(new Error("zlib binding closed")):n(new Error("invalid input"))},C.prototype._processChunk=function(o,a,s){var l=o&&o.length,u=this._chunkSize-this._offset,c=0,f=this,h="function"==typeof s;if(!h){var e,d=[],p=0;this.on("error",function(t){e=t}),y(this._handle,"zlib binding closed");do{var t=this._handle.writeSync(a,o,c,l,this._buffer,this._offset,u)}while(!this._hadError&&g(t[0],t[1]));if(this._hadError)throw e;if(b<=p)throw A(this),new RangeError(m);var n=v.concat(d,p);return A(this),n}y(this._handle,"zlib binding closed");var r=this._handle.write(a,o,c,l,this._buffer,this._offset,u);function g(t,e){if(this&&(this.buffer=null,this.callback=null),!f._hadError){var n=u-e;if(y(0<=n,"have should not go down"),0<n){var r=f._buffer.slice(f._offset,f._offset+n);f._offset+=n,h?f.push(r):(d.push(r),p+=r.length)}if((0===e||f._offset>=f._chunkSize)&&(u=f._chunkSize,f._offset=0,f._buffer=v.allocUnsafe(f._chunkSize)),0===e){if(c+=l-t,l=t,!h)return!0;var i=f._handle.write(a,o,c,l,f._buffer,f._offset,f._chunkSize);return i.callback=g,void(i.buffer=o)}if(!h)return!1;s()}}r.buffer=o,r.callback=g},t.inherits(d,C),t.inherits(p,C),t.inherits(g,C),t.inherits(w,C),t.inherits(x,C),t.inherits(_,C),t.inherits(k,C)}).call(this,O(24))},function(_,t,k){"use strict";(function(e){var t=k(264);
/*!
 * The buffer module from node.js, for the browser.
 *
 * @author   Feross Aboukhadijeh <feross@feross.org> <http://feross.org>
 * @license  MIT
 */function o(t,e){if(t===e)return 0;for(var n=t.length,r=e.length,i=0,o=Math.min(n,r);i<o;++i)if(t[i]!==e[i]){n=t[i],r=e[i];break}return n<r?-1:r<n?1:0}function a(t){return e.Buffer&&"function"==typeof e.Buffer.isBuffer?e.Buffer.isBuffer(t):!(null==t||!t._isBuffer)}var c=k(99),r=Object.prototype.hasOwnProperty,f=Array.prototype.slice,n="foo"===function(){}.name;function s(t){return Object.prototype.toString.call(t)}function l(t){return!a(t)&&("function"==typeof e.ArrayBuffer&&("function"==typeof ArrayBuffer.isView?ArrayBuffer.isView(t):!!t&&(t instanceof DataView||!!(t.buffer&&t.buffer instanceof ArrayBuffer))))}var u=_.exports=v,i=/\s*function\s+([^\(\s]*)\s*/;function h(t){if(c.isFunction(t)){if(n)return t.name;var e=t.toString().match(i);return e&&e[1]}}function d(t,e){return"string"==typeof t?t.length<e?t:t.slice(0,e):t}function p(t){if(n||!c.isFunction(t))return c.inspect(t);var e=h(t);return"[Function"+(e?": "+e:"")+"]"}function g(t,e,n,r,i){throw new u.AssertionError({message:n,actual:t,expected:e,operator:r,stackStartFunction:i})}function v(t,e){t||g(t,!0,e,"==",u.ok)}function y(t,e,n,r){if(t===e)return!0;if(a(t)&&a(e))return 0===o(t,e);if(c.isDate(t)&&c.isDate(e))return t.getTime()===e.getTime();if(c.isRegExp(t)&&c.isRegExp(e))return t.source===e.source&&t.global===e.global&&t.multiline===e.multiline&&t.lastIndex===e.lastIndex&&t.ignoreCase===e.ignoreCase;if(null!==t&&"object"==typeof t||null!==e&&"object"==typeof e){if(l(t)&&l(e)&&s(t)===s(e)&&!(t instanceof Float32Array||t instanceof Float64Array))return 0===o(new Uint8Array(t.buffer),new Uint8Array(e.buffer));if(a(t)!==a(e))return!1;var i=(r=r||{actual:[],expected:[]}).actual.indexOf(t);return-1!==i&&i===r.expected.indexOf(e)||(r.actual.push(t),r.expected.push(e),function(t,e,n,r){if(null==t||null==e)return!1;if(c.isPrimitive(t)||c.isPrimitive(e))return t===e;if(n&&Object.getPrototypeOf(t)!==Object.getPrototypeOf(e))return!1;var i=b(t),o=b(e);if(i&&!o||!i&&o)return!1;if(i)return t=f.call(t),e=f.call(e),y(t,e,n);var a,s,l=x(t),u=x(e);if(l.length!==u.length)return!1;for(l.sort(),u.sort(),s=l.length-1;0<=s;s--)if(l[s]!==u[s])return!1;for(s=l.length-1;0<=s;s--)if(a=l[s],!y(t[a],e[a],n,r))return!1;return!0}(t,e,n,r))}return n?t===e:t==e}function b(t){return"[object Arguments]"==Object.prototype.toString.call(t)}function m(t,e){if(!t||!e)return!1;if("[object RegExp]"==Object.prototype.toString.call(e))return e.test(t);try{if(t instanceof e)return!0}catch(t){}return!Error.isPrototypeOf(e)&&!0===e.call({},t)}function w(t,e,n,r){var i;if("function"!=typeof e)throw new TypeError('"block" argument must be a function');"string"==typeof n&&(r=n,n=null),i=function(t){var e;try{t()}catch(t){e=t}return e}(e),r=(n&&n.name?" ("+n.name+").":".")+(r?" "+r:"."),t&&!i&&g(i,n,"Missing expected exception"+r);var o="string"==typeof r,a=!t&&i&&!n;if((!t&&c.isError(i)&&o&&m(i,n)||a)&&g(i,n,"Got unwanted exception"+r),t&&i&&n&&!m(i,n)||!t&&i)throw i}u.AssertionError=function(t){this.name="AssertionError",this.actual=t.actual,this.expected=t.expected,this.operator=t.operator,t.message?(this.message=t.message,this.generatedMessage=!1):(this.message=function(t){return d(p(t.actual),128)+" "+t.operator+" "+d(p(t.expected),128)}(this),this.generatedMessage=!0);var e=t.stackStartFunction||g;if(Error.captureStackTrace)Error.captureStackTrace(this,e);else{var n=new Error;if(n.stack){var r=n.stack,i=h(e),o=r.indexOf("\n"+i);if(0<=o){var a=r.indexOf("\n",o+1);r=r.substring(a+1)}this.stack=r}}},c.inherits(u.AssertionError,Error),u.fail=g,u.ok=v,u.equal=function(t,e,n){t!=e&&g(t,e,n,"==",u.equal)},u.notEqual=function(t,e,n){t==e&&g(t,e,n,"!=",u.notEqual)},u.deepEqual=function(t,e,n){y(t,e,!1)||g(t,e,n,"deepEqual",u.deepEqual)},u.deepStrictEqual=function(t,e,n){y(t,e,!0)||g(t,e,n,"deepStrictEqual",u.deepStrictEqual)},u.notDeepEqual=function(t,e,n){y(t,e,!1)&&g(t,e,n,"notDeepEqual",u.notDeepEqual)},u.notDeepStrictEqual=function t(e,n,r){y(e,n,!0)&&g(e,n,r,"notDeepStrictEqual",t)},u.strictEqual=function(t,e,n){t!==e&&g(t,e,n,"===",u.strictEqual)},u.notStrictEqual=function(t,e,n){t===e&&g(t,e,n,"!==",u.notStrictEqual)},u.throws=function(t,e,n){w(!0,t,e,n)},u.doesNotThrow=function(t,e,n){w(!1,t,e,n)},u.ifError=function(t){if(t)throw t},u.strict=t(function t(e,n){e||g(e,!0,n,"==",t)},u,{equal:u.strictEqual,deepEqual:u.deepStrictEqual,notEqual:u.notStrictEqual,notDeepEqual:u.notDeepStrictEqual}),u.strict.strict=u.strict;var x=Object.keys||function(t){var e=[];for(var n in t)r.call(t,n)&&e.push(n);return e}}).call(this,k(25))},function(t,e,n){"use strict";t.exports=function(t,e,n,r){for(var i=65535&t|0,o=t>>>16&65535|0,a=0;0!==n;){for(n-=a=2e3<n?2e3:n;o=o+(i=i+e[r++]|0)|0,--a;);i%=65521,o%=65521}return i|o<<16|0}},function(t,e,n){"use strict";var s=function(){for(var t,e=[],n=0;n<256;n++){t=n;for(var r=0;r<8;r++)t=1&t?3988292384^t>>>1:t>>>1;e[n]=t}return e}();t.exports=function(t,e,n,r){var i=s,o=r+n;t^=-1;for(var a=r;a<o;a++)t=t>>>8^i[255&(t^e[a])];return-1^t}},function(t,e,n){var l;t.exports=(l=n(1),function(i){var t=l,e=t.lib,n=e.WordArray,r=e.Hasher,o=t.algo,a=[],m=[];!function(){function t(t){for(var e=i.sqrt(t),n=2;n<=e;n++)if(!(t%n))return!1;return!0}function e(t){return 4294967296*(t-(0|t))|0}for(var n=2,r=0;r<64;)t(n)&&(r<8&&(a[r]=e(i.pow(n,.5))),m[r]=e(i.pow(n,1/3)),r++),n++}();var w=[],s=o.SHA256=r.extend({_doReset:function(){this._hash=new n.init(a.slice(0))},_doProcessBlock:function(t,e){for(var n=this._hash.words,r=n[0],i=n[1],o=n[2],a=n[3],s=n[4],l=n[5],u=n[6],c=n[7],f=0;f<64;f++){if(f<16)w[f]=0|t[e+f];else{var h=w[f-15],d=(h<<25|h>>>7)^(h<<14|h>>>18)^h>>>3,p=w[f-2],g=(p<<15|p>>>17)^(p<<13|p>>>19)^p>>>10;w[f]=d+w[f-7]+g+w[f-16]}var v=r&i^r&o^i&o,y=(r<<30|r>>>2)^(r<<19|r>>>13)^(r<<10|r>>>22),b=c+((s<<26|s>>>6)^(s<<21|s>>>11)^(s<<7|s>>>25))+(s&l^~s&u)+m[f]+w[f];c=u,u=l,l=s,s=a+b|0,a=o,o=i,i=r,r=b+(y+v)|0}n[0]=n[0]+r|0,n[1]=n[1]+i|0,n[2]=n[2]+o|0,n[3]=n[3]+a|0,n[4]=n[4]+s|0,n[5]=n[5]+l|0,n[6]=n[6]+u|0,n[7]=n[7]+c|0},_doFinalize:function(){var t=this._data,e=t.words,n=8*this._nDataBytes,r=8*t.sigBytes;return e[r>>>5]|=128<<24-r%32,e[14+(64+r>>>9<<4)]=i.floor(n/4294967296),e[15+(64+r>>>9<<4)]=n,t.sigBytes=4*e.length,this._process(),this._hash},clone:function(){var t=r.clone.call(this);return t._hash=this._hash.clone(),t}});t.SHA256=r._createHelper(s),t.HmacSHA256=r._createHmacHelper(s)}(Math),l.SHA256)},function(t,e,n){var l;t.exports=(l=n(1),n(72),function(){var t=l,e=t.lib.Hasher,n=t.x64,r=n.Word,i=n.WordArray,o=t.algo;function a(){return r.create.apply(r,arguments)}var kt=[a(1116352408,3609767458),a(1899447441,602891725),a(3049323471,3964484399),a(3921009573,2173295548),a(961987163,4081628472),a(1508970993,3053834265),a(2453635748,2937671579),a(2870763221,3664609560),a(3624381080,2734883394),a(310598401,1164996542),a(607225278,1323610764),a(1426881987,3590304994),a(1925078388,4068182383),a(2162078206,991336113),a(2614888103,633803317),a(3248222580,3479774868),a(3835390401,2666613458),a(4022224774,944711139),a(264347078,2341262773),a(604807628,2007800933),a(770255983,1495990901),a(1249150122,1856431235),a(1555081692,3175218132),a(1996064986,2198950837),a(2554220882,3999719339),a(2821834349,766784016),a(2952996808,2566594879),a(3210313671,3203337956),a(3336571891,1034457026),a(3584528711,2466948901),a(113926993,3758326383),a(338241895,168717936),a(666307205,1188179964),a(773529912,1546045734),a(1294757372,1522805485),a(1396182291,2643833823),a(1695183700,2343527390),a(1986661051,1014477480),a(2177026350,1206759142),a(2456956037,344077627),a(2730485921,1290863460),a(2820302411,3158454273),a(3259730800,3505952657),a(3345764771,106217008),a(3516065817,3606008344),a(3600352804,1432725776),a(4094571909,1467031594),a(275423344,851169720),a(430227734,3100823752),a(506948616,1363258195),a(659060556,3750685593),a(883997877,3785050280),a(958139571,3318307427),a(1322822218,3812723403),a(1537002063,2003034995),a(1747873779,3602036899),a(1955562222,1575990012),a(2024104815,1125592928),a(2227730452,2716904306),a(2361852424,442776044),a(2428436474,593698344),a(2756734187,3733110249),a(3204031479,2999351573),a(3329325298,3815920427),a(3391569614,3928383900),a(3515267271,566280711),a(3940187606,3454069534),a(4118630271,4000239992),a(116418474,1914138554),a(174292421,2731055270),a(289380356,3203993006),a(460393269,320620315),a(685471733,587496836),a(852142971,1086792851),a(1017036298,365543100),a(1126000580,2618297676),a(1288033470,3409855158),a(1501505948,4234509866),a(1607167915,987167468),a(1816402316,1246189591)],St=[];!function(){for(var t=0;t<80;t++)St[t]=a()}();var s=o.SHA512=e.extend({_doReset:function(){this._hash=new i.init([new r.init(1779033703,4089235720),new r.init(3144134277,2227873595),new r.init(1013904242,4271175723),new r.init(2773480762,1595750129),new r.init(1359893119,2917565137),new r.init(2600822924,725511199),new r.init(528734635,4215389547),new r.init(1541459225,327033209)])},_doProcessBlock:function(t,e){for(var n=this._hash.words,r=n[0],i=n[1],o=n[2],a=n[3],s=n[4],l=n[5],u=n[6],c=n[7],f=r.high,h=r.low,d=i.high,p=i.low,g=o.high,v=o.low,y=a.high,b=a.low,m=s.high,w=s.low,x=l.high,_=l.low,k=u.high,S=u.low,C=c.high,A=c.low,P=f,E=h,O=d,T=p,I=g,B=v,L=y,R=b,M=m,F=w,D=x,z=_,N=k,U=S,W=C,j=A,G=0;G<80;G++){var H=St[G];if(G<16)var V=H.high=0|t[e+2*G],q=H.low=0|t[e+2*G+1];else{var Z=St[G-15],Y=Z.high,X=Z.low,K=(Y>>>1|X<<31)^(Y>>>8|X<<24)^Y>>>7,J=(X>>>1|Y<<31)^(X>>>8|Y<<24)^(X>>>7|Y<<25),Q=St[G-2],$=Q.high,tt=Q.low,et=($>>>19|tt<<13)^($<<3|tt>>>29)^$>>>6,nt=(tt>>>19|$<<13)^(tt<<3|$>>>29)^(tt>>>6|$<<26),rt=St[G-7],it=rt.high,ot=rt.low,at=St[G-16],st=at.high,lt=at.low;V=(V=(V=K+it+((q=J+ot)>>>0<J>>>0?1:0))+et+((q+=nt)>>>0<nt>>>0?1:0))+st+((q+=lt)>>>0<lt>>>0?1:0),H.high=V,H.low=q}var ut,ct=M&D^~M&N,ft=F&z^~F&U,ht=P&O^P&I^O&I,dt=E&T^E&B^T&B,pt=(P>>>28|E<<4)^(P<<30|E>>>2)^(P<<25|E>>>7),gt=(E>>>28|P<<4)^(E<<30|P>>>2)^(E<<25|P>>>7),vt=(M>>>14|F<<18)^(M>>>18|F<<14)^(M<<23|F>>>9),yt=(F>>>14|M<<18)^(F>>>18|M<<14)^(F<<23|M>>>9),bt=kt[G],mt=bt.high,wt=bt.low,xt=W+vt+((ut=j+yt)>>>0<j>>>0?1:0),_t=gt+dt;W=N,j=U,N=D,U=z,D=M,z=F,M=L+(xt=(xt=(xt=xt+ct+((ut+=ft)>>>0<ft>>>0?1:0))+mt+((ut+=wt)>>>0<wt>>>0?1:0))+V+((ut+=q)>>>0<q>>>0?1:0))+((F=R+ut|0)>>>0<R>>>0?1:0)|0,L=I,R=B,I=O,B=T,O=P,T=E,P=xt+(pt+ht+(_t>>>0<gt>>>0?1:0))+((E=ut+_t|0)>>>0<ut>>>0?1:0)|0}h=r.low=h+E,r.high=f+P+(h>>>0<E>>>0?1:0),p=i.low=p+T,i.high=d+O+(p>>>0<T>>>0?1:0),v=o.low=v+B,o.high=g+I+(v>>>0<B>>>0?1:0),b=a.low=b+R,a.high=y+L+(b>>>0<R>>>0?1:0),w=s.low=w+F,s.high=m+M+(w>>>0<F>>>0?1:0),_=l.low=_+z,l.high=x+D+(_>>>0<z>>>0?1:0),S=u.low=S+U,u.high=k+N+(S>>>0<U>>>0?1:0),A=c.low=A+j,c.high=C+W+(A>>>0<j>>>0?1:0)},_doFinalize:function(){var t=this._data,e=t.words,n=8*this._nDataBytes,r=8*t.sigBytes;return e[r>>>5]|=128<<24-r%32,e[30+(128+r>>>10<<5)]=Math.floor(n/4294967296),e[31+(128+r>>>10<<5)]=n,t.sigBytes=4*e.length,this._process(),this._hash.toX32()},clone:function(){var t=e.clone.call(this);return t._hash=this._hash.clone(),t},blockSize:32});t.SHA512=e._createHelper(s),t.HmacSHA512=e._createHmacHelper(s)}(),l.SHA512)},function(t){t.exports=[["a140","",62],["a180","",32],["a240","",62],["a280","",32],["a2ab","",5],["a2e3","€"],["a2ef",""],["a2fd",""],["a340","",62],["a380","",31,"　"],["a440","",62],["a480","",32],["a4f4","",10],["a540","",62],["a580","",32],["a5f7","",7],["a640","",62],["a680","",32],["a6b9","",7],["a6d9","",6],["a6ec",""],["a6f3",""],["a6f6","",8],["a740","",62],["a780","",32],["a7c2","",14],["a7f2","",12],["a896","",10],["a8bc",""],["a8bf","ǹ"],["a8c1",""],["a8ea","",20],["a958",""],["a95b",""],["a95d",""],["a989","〾⿰",11],["a997","",12],["a9f0","",14],["aaa1","",93],["aba1","",93],["aca1","",93],["ada1","",93],["aea1","",93],["afa1","",93],["d7fa","",4],["f8a1","",93],["f9a1","",93],["faa1","",93],["fba1","",93],["fca1","",93],["fda1","",93],["fe50","⺁⺄㑳㑇⺈⺋㖞㘚㘎⺌⺗㥮㤘㧏㧟㩳㧐㭎㱮㳠⺧⺪䁖䅟⺮䌷⺳⺶⺷䎱䎬⺻䏝䓖䙡䙌"],["fe80","䜣䜩䝼䞍⻊䥇䥺䥽䦂䦃䦅䦆䦟䦛䦷䦶䲣䲟䲠䲡䱷䲢䴓",6,"䶮",93]]},function(t){t.exports=[["0","\0",127],["a140","　，、。．‧；：？！︰…‥﹐﹑﹒·﹔﹕﹖﹗｜–︱—︳╴︴﹏（）︵︶｛｝︷︸〔〕︹︺【】︻︼《》︽︾〈〉︿﹀「」﹁﹂『』﹃﹄﹙﹚"],["a1a1","﹛﹜﹝﹞‘’“”〝〞‵′＃＆＊※§〃○●△▲◎☆★◇◆□■▽▼㊣℅¯￣＿ˍ﹉﹊﹍﹎﹋﹌﹟﹠﹡＋－×÷±√＜＞＝≦≧≠∞≒≡﹢",4,"～∩∪⊥∠∟⊿㏒㏑∫∮∵∴♀♂⊕⊙↑↓←→↖↗↙↘∥∣／"],["a240","＼∕﹨＄￥〒￠￡％＠℃℉﹩﹪﹫㏕㎜㎝㎞㏎㎡㎎㎏㏄°兙兛兞兝兡兣嗧瓩糎▁",7,"▏▎▍▌▋▊▉┼┴┬┤├▔─│▕┌┐└┘╭"],["a2a1","╮╰╯═╞╪╡◢◣◥◤╱╲╳０",9,"Ⅰ",9,"〡",8,"十卄卅Ａ",25,"ａ",21],["a340","ｗｘｙｚΑ",16,"Σ",6,"α",16,"σ",6,"ㄅ",10],["a3a1","ㄐ",25,"˙ˉˊˇˋ"],["a3e1","€"],["a440","一乙丁七乃九了二人儿入八几刀刁力匕十卜又三下丈上丫丸凡久么也乞于亡兀刃勺千叉口土士夕大女子孑孓寸小尢尸山川工己已巳巾干廾弋弓才"],["a4a1","丑丐不中丰丹之尹予云井互五亢仁什仃仆仇仍今介仄元允內六兮公冗凶分切刈勻勾勿化匹午升卅卞厄友及反壬天夫太夭孔少尤尺屯巴幻廿弔引心戈戶手扎支文斗斤方日曰月木欠止歹毋比毛氏水火爪父爻片牙牛犬王丙"],["a540","世丕且丘主乍乏乎以付仔仕他仗代令仙仞充兄冉冊冬凹出凸刊加功包匆北匝仟半卉卡占卯卮去可古右召叮叩叨叼司叵叫另只史叱台句叭叻四囚外"],["a5a1","央失奴奶孕它尼巨巧左市布平幼弁弘弗必戊打扔扒扑斥旦朮本未末札正母民氐永汁汀氾犯玄玉瓜瓦甘生用甩田由甲申疋白皮皿目矛矢石示禾穴立丞丟乒乓乩亙交亦亥仿伉伙伊伕伍伐休伏仲件任仰仳份企伋光兇兆先全"],["a640","共再冰列刑划刎刖劣匈匡匠印危吉吏同吊吐吁吋各向名合吃后吆吒因回囝圳地在圭圬圯圩夙多夷夸妄奸妃好她如妁字存宇守宅安寺尖屹州帆并年"],["a6a1","式弛忙忖戎戌戍成扣扛托收早旨旬旭曲曳有朽朴朱朵次此死氖汝汗汙江池汐汕污汛汍汎灰牟牝百竹米糸缶羊羽老考而耒耳聿肉肋肌臣自至臼舌舛舟艮色艾虫血行衣西阡串亨位住佇佗佞伴佛何估佐佑伽伺伸佃佔似但佣"],["a740","作你伯低伶余佝佈佚兌克免兵冶冷別判利刪刨劫助努劬匣即卵吝吭吞吾否呎吧呆呃吳呈呂君吩告吹吻吸吮吵吶吠吼呀吱含吟听囪困囤囫坊坑址坍"],["a7a1","均坎圾坐坏圻壯夾妝妒妨妞妣妙妖妍妤妓妊妥孝孜孚孛完宋宏尬局屁尿尾岐岑岔岌巫希序庇床廷弄弟彤形彷役忘忌志忍忱快忸忪戒我抄抗抖技扶抉扭把扼找批扳抒扯折扮投抓抑抆改攻攸旱更束李杏材村杜杖杞杉杆杠"],["a840","杓杗步每求汞沙沁沈沉沅沛汪決沐汰沌汨沖沒汽沃汲汾汴沆汶沍沔沘沂灶灼災灸牢牡牠狄狂玖甬甫男甸皂盯矣私秀禿究系罕肖肓肝肘肛肚育良芒"],["a8a1","芋芍見角言谷豆豕貝赤走足身車辛辰迂迆迅迄巡邑邢邪邦那酉釆里防阮阱阪阬並乖乳事些亞享京佯依侍佳使佬供例來侃佰併侈佩佻侖佾侏侑佺兔兒兕兩具其典冽函刻券刷刺到刮制剁劾劻卒協卓卑卦卷卸卹取叔受味呵"],["a940","咖呸咕咀呻呷咄咒咆呼咐呱呶和咚呢周咋命咎固垃坷坪坩坡坦坤坼夜奉奇奈奄奔妾妻委妹妮姑姆姐姍始姓姊妯妳姒姅孟孤季宗定官宜宙宛尚屈居"],["a9a1","屆岷岡岸岩岫岱岳帘帚帖帕帛帑幸庚店府底庖延弦弧弩往征彿彼忝忠忽念忿怏怔怯怵怖怪怕怡性怩怫怛或戕房戾所承拉拌拄抿拂抹拒招披拓拔拋拈抨抽押拐拙拇拍抵拚抱拘拖拗拆抬拎放斧於旺昔易昌昆昂明昀昏昕昊"],["aa40","昇服朋杭枋枕東果杳杷枇枝林杯杰板枉松析杵枚枓杼杪杲欣武歧歿氓氛泣注泳沱泌泥河沽沾沼波沫法泓沸泄油況沮泗泅泱沿治泡泛泊沬泯泜泖泠"],["aaa1","炕炎炒炊炙爬爭爸版牧物狀狎狙狗狐玩玨玟玫玥甽疝疙疚的盂盲直知矽社祀祁秉秈空穹竺糾罔羌羋者肺肥肢肱股肫肩肴肪肯臥臾舍芳芝芙芭芽芟芹花芬芥芯芸芣芰芾芷虎虱初表軋迎返近邵邸邱邶采金長門阜陀阿阻附"],["ab40","陂隹雨青非亟亭亮信侵侯便俠俑俏保促侶俘俟俊俗侮俐俄係俚俎俞侷兗冒冑冠剎剃削前剌剋則勇勉勃勁匍南卻厚叛咬哀咨哎哉咸咦咳哇哂咽咪品"],["aba1","哄哈咯咫咱咻咩咧咿囿垂型垠垣垢城垮垓奕契奏奎奐姜姘姿姣姨娃姥姪姚姦威姻孩宣宦室客宥封屎屏屍屋峙峒巷帝帥帟幽庠度建弈弭彥很待徊律徇後徉怒思怠急怎怨恍恰恨恢恆恃恬恫恪恤扁拜挖按拼拭持拮拽指拱拷"],["ac40","拯括拾拴挑挂政故斫施既春昭映昧是星昨昱昤曷柿染柱柔某柬架枯柵柩柯柄柑枴柚查枸柏柞柳枰柙柢柝柒歪殃殆段毒毗氟泉洋洲洪流津洌洱洞洗"],["aca1","活洽派洶洛泵洹洧洸洩洮洵洎洫炫為炳炬炯炭炸炮炤爰牲牯牴狩狠狡玷珊玻玲珍珀玳甚甭畏界畎畋疫疤疥疢疣癸皆皇皈盈盆盃盅省盹相眉看盾盼眇矜砂研砌砍祆祉祈祇禹禺科秒秋穿突竿竽籽紂紅紀紉紇約紆缸美羿耄"],["ad40","耐耍耑耶胖胥胚胃胄背胡胛胎胞胤胝致舢苧范茅苣苛苦茄若茂茉苒苗英茁苜苔苑苞苓苟苯茆虐虹虻虺衍衫要觔計訂訃貞負赴赳趴軍軌述迦迢迪迥"],["ada1","迭迫迤迨郊郎郁郃酋酊重閂限陋陌降面革韋韭音頁風飛食首香乘亳倌倍倣俯倦倥俸倩倖倆值借倚倒們俺倀倔倨俱倡個候倘俳修倭倪俾倫倉兼冤冥冢凍凌准凋剖剜剔剛剝匪卿原厝叟哨唐唁唷哼哥哲唆哺唔哩哭員唉哮哪"],["ae40","哦唧唇哽唏圃圄埂埔埋埃堉夏套奘奚娑娘娜娟娛娓姬娠娣娩娥娌娉孫屘宰害家宴宮宵容宸射屑展屐峭峽峻峪峨峰島崁峴差席師庫庭座弱徒徑徐恙"],["aea1","恣恥恐恕恭恩息悄悟悚悍悔悌悅悖扇拳挈拿捎挾振捕捂捆捏捉挺捐挽挪挫挨捍捌效敉料旁旅時晉晏晃晒晌晅晁書朔朕朗校核案框桓根桂桔栩梳栗桌桑栽柴桐桀格桃株桅栓栘桁殊殉殷氣氧氨氦氤泰浪涕消涇浦浸海浙涓"],["af40","浬涉浮浚浴浩涌涊浹涅浥涔烊烘烤烙烈烏爹特狼狹狽狸狷玆班琉珮珠珪珞畔畝畜畚留疾病症疲疳疽疼疹痂疸皋皰益盍盎眩真眠眨矩砰砧砸砝破砷"],["afa1","砥砭砠砟砲祕祐祠祟祖神祝祗祚秤秣秧租秦秩秘窄窈站笆笑粉紡紗紋紊素索純紐紕級紜納紙紛缺罟羔翅翁耆耘耕耙耗耽耿胱脂胰脅胭胴脆胸胳脈能脊胼胯臭臬舀舐航舫舨般芻茫荒荔荊茸荐草茵茴荏茲茹茶茗荀茱茨荃"],["b040","虔蚊蚪蚓蚤蚩蚌蚣蚜衰衷袁袂衽衹記訐討訌訕訊託訓訖訏訑豈豺豹財貢起躬軒軔軏辱送逆迷退迺迴逃追逅迸邕郡郝郢酒配酌釘針釗釜釙閃院陣陡"],["b0a1","陛陝除陘陞隻飢馬骨高鬥鬲鬼乾偺偽停假偃偌做偉健偶偎偕偵側偷偏倏偯偭兜冕凰剪副勒務勘動匐匏匙匿區匾參曼商啪啦啄啞啡啃啊唱啖問啕唯啤唸售啜唬啣唳啁啗圈國圉域堅堊堆埠埤基堂堵執培夠奢娶婁婉婦婪婀"],["b140","娼婢婚婆婊孰寇寅寄寂宿密尉專將屠屜屝崇崆崎崛崖崢崑崩崔崙崤崧崗巢常帶帳帷康庸庶庵庾張強彗彬彩彫得徙從徘御徠徜恿患悉悠您惋悴惦悽"],["b1a1","情悻悵惜悼惘惕惆惟悸惚惇戚戛扈掠控捲掖探接捷捧掘措捱掩掉掃掛捫推掄授掙採掬排掏掀捻捩捨捺敝敖救教敗啟敏敘敕敔斜斛斬族旋旌旎晝晚晤晨晦晞曹勗望梁梯梢梓梵桿桶梱梧梗械梃棄梭梆梅梔條梨梟梡梂欲殺"],["b240","毫毬氫涎涼淳淙液淡淌淤添淺清淇淋涯淑涮淞淹涸混淵淅淒渚涵淚淫淘淪深淮淨淆淄涪淬涿淦烹焉焊烽烯爽牽犁猜猛猖猓猙率琅琊球理現琍瓠瓶"],["b2a1","瓷甜產略畦畢異疏痔痕疵痊痍皎盔盒盛眷眾眼眶眸眺硫硃硎祥票祭移窒窕笠笨笛第符笙笞笮粒粗粕絆絃統紮紹紼絀細紳組累終紲紱缽羞羚翌翎習耜聊聆脯脖脣脫脩脰脤舂舵舷舶船莎莞莘荸莢莖莽莫莒莊莓莉莠荷荻荼"],["b340","莆莧處彪蛇蛀蚶蛄蚵蛆蛋蚱蚯蛉術袞袈被袒袖袍袋覓規訪訝訣訥許設訟訛訢豉豚販責貫貨貪貧赧赦趾趺軛軟這逍通逗連速逝逐逕逞造透逢逖逛途"],["b3a1","部郭都酗野釵釦釣釧釭釩閉陪陵陳陸陰陴陶陷陬雀雪雩章竟頂頃魚鳥鹵鹿麥麻傢傍傅備傑傀傖傘傚最凱割剴創剩勞勝勛博厥啻喀喧啼喊喝喘喂喜喪喔喇喋喃喳單喟唾喲喚喻喬喱啾喉喫喙圍堯堪場堤堰報堡堝堠壹壺奠"],["b440","婷媚婿媒媛媧孳孱寒富寓寐尊尋就嵌嵐崴嵇巽幅帽幀幃幾廊廁廂廄弼彭復循徨惑惡悲悶惠愜愣惺愕惰惻惴慨惱愎惶愉愀愒戟扉掣掌描揀揩揉揆揍"],["b4a1","插揣提握揖揭揮捶援揪換摒揚揹敞敦敢散斑斐斯普晰晴晶景暑智晾晷曾替期朝棺棕棠棘棗椅棟棵森棧棹棒棲棣棋棍植椒椎棉棚楮棻款欺欽殘殖殼毯氮氯氬港游湔渡渲湧湊渠渥渣減湛湘渤湖湮渭渦湯渴湍渺測湃渝渾滋"],["b540","溉渙湎湣湄湲湩湟焙焚焦焰無然煮焜牌犄犀猶猥猴猩琺琪琳琢琥琵琶琴琯琛琦琨甥甦畫番痢痛痣痙痘痞痠登發皖皓皴盜睏短硝硬硯稍稈程稅稀窘"],["b5a1","窗窖童竣等策筆筐筒答筍筋筏筑粟粥絞結絨絕紫絮絲絡給絢絰絳善翔翕耋聒肅腕腔腋腑腎脹腆脾腌腓腴舒舜菩萃菸萍菠菅萋菁華菱菴著萊菰萌菌菽菲菊萸萎萄菜萇菔菟虛蛟蛙蛭蛔蛛蛤蛐蛞街裁裂袱覃視註詠評詞証詁"],["b640","詔詛詐詆訴診訶詖象貂貯貼貳貽賁費賀貴買貶貿貸越超趁跎距跋跚跑跌跛跆軻軸軼辜逮逵週逸進逶鄂郵鄉郾酣酥量鈔鈕鈣鈉鈞鈍鈐鈇鈑閔閏開閑"],["b6a1","間閒閎隊階隋陽隅隆隍陲隄雁雅雄集雇雯雲韌項順須飧飪飯飩飲飭馮馭黃黍黑亂傭債傲傳僅傾催傷傻傯僇剿剷剽募勦勤勢勣匯嗟嗨嗓嗦嗎嗜嗇嗑嗣嗤嗯嗚嗡嗅嗆嗥嗉園圓塞塑塘塗塚塔填塌塭塊塢塒塋奧嫁嫉嫌媾媽媼"],["b740","媳嫂媲嵩嵯幌幹廉廈弒彙徬微愚意慈感想愛惹愁愈慎慌慄慍愾愴愧愍愆愷戡戢搓搾搞搪搭搽搬搏搜搔損搶搖搗搆敬斟新暗暉暇暈暖暄暘暍會榔業"],["b7a1","楚楷楠楔極椰概楊楨楫楞楓楹榆楝楣楛歇歲毀殿毓毽溢溯滓溶滂源溝滇滅溥溘溼溺溫滑準溜滄滔溪溧溴煎煙煩煤煉照煜煬煦煌煥煞煆煨煖爺牒猷獅猿猾瑯瑚瑕瑟瑞瑁琿瑙瑛瑜當畸瘀痰瘁痲痱痺痿痴痳盞盟睛睫睦睞督"],["b840","睹睪睬睜睥睨睢矮碎碰碗碘碌碉硼碑碓硿祺祿禁萬禽稜稚稠稔稟稞窟窠筷節筠筮筧粱粳粵經絹綑綁綏絛置罩罪署義羨群聖聘肆肄腱腰腸腥腮腳腫"],["b8a1","腹腺腦舅艇蒂葷落萱葵葦葫葉葬葛萼萵葡董葩葭葆虞虜號蛹蜓蜈蜇蜀蛾蛻蜂蜃蜆蜊衙裟裔裙補裘裝裡裊裕裒覜解詫該詳試詩詰誇詼詣誠話誅詭詢詮詬詹詻訾詨豢貊貉賊資賈賄貲賃賂賅跡跟跨路跳跺跪跤跦躲較載軾輊"],["b940","辟農運遊道遂達逼違遐遇遏過遍遑逾遁鄒鄗酬酪酩釉鈷鉗鈸鈽鉀鈾鉛鉋鉤鉑鈴鉉鉍鉅鈹鈿鉚閘隘隔隕雍雋雉雊雷電雹零靖靴靶預頑頓頊頒頌飼飴"],["b9a1","飽飾馳馱馴髡鳩麂鼎鼓鼠僧僮僥僖僭僚僕像僑僱僎僩兢凳劃劂匱厭嗾嘀嘛嘗嗽嘔嘆嘉嘍嘎嗷嘖嘟嘈嘐嗶團圖塵塾境墓墊塹墅塽壽夥夢夤奪奩嫡嫦嫩嫗嫖嫘嫣孵寞寧寡寥實寨寢寤察對屢嶄嶇幛幣幕幗幔廓廖弊彆彰徹慇"],["ba40","愿態慷慢慣慟慚慘慵截撇摘摔撤摸摟摺摑摧搴摭摻敲斡旗旖暢暨暝榜榨榕槁榮槓構榛榷榻榫榴槐槍榭槌榦槃榣歉歌氳漳演滾漓滴漩漾漠漬漏漂漢"],["baa1","滿滯漆漱漸漲漣漕漫漯澈漪滬漁滲滌滷熔熙煽熊熄熒爾犒犖獄獐瑤瑣瑪瑰瑭甄疑瘧瘍瘋瘉瘓盡監瞄睽睿睡磁碟碧碳碩碣禎福禍種稱窪窩竭端管箕箋筵算箝箔箏箸箇箄粹粽精綻綰綜綽綾綠緊綴網綱綺綢綿綵綸維緒緇綬"],["bb40","罰翠翡翟聞聚肇腐膀膏膈膊腿膂臧臺與舔舞艋蓉蒿蓆蓄蒙蒞蒲蒜蓋蒸蓀蓓蒐蒼蓑蓊蜿蜜蜻蜢蜥蜴蜘蝕蜷蜩裳褂裴裹裸製裨褚裯誦誌語誣認誡誓誤"],["bba1","說誥誨誘誑誚誧豪貍貌賓賑賒赫趙趕跼輔輒輕輓辣遠遘遜遣遙遞遢遝遛鄙鄘鄞酵酸酷酴鉸銀銅銘銖鉻銓銜銨鉼銑閡閨閩閣閥閤隙障際雌雒需靼鞅韶頗領颯颱餃餅餌餉駁骯骰髦魁魂鳴鳶鳳麼鼻齊億儀僻僵價儂儈儉儅凜"],["bc40","劇劈劉劍劊勰厲嘮嘻嘹嘲嘿嘴嘩噓噎噗噴嘶嘯嘰墀墟增墳墜墮墩墦奭嬉嫻嬋嫵嬌嬈寮寬審寫層履嶝嶔幢幟幡廢廚廟廝廣廠彈影德徵慶慧慮慝慕憂"],["bca1","慼慰慫慾憧憐憫憎憬憚憤憔憮戮摩摯摹撞撲撈撐撰撥撓撕撩撒撮播撫撚撬撙撢撳敵敷數暮暫暴暱樣樟槨樁樞標槽模樓樊槳樂樅槭樑歐歎殤毅毆漿潼澄潑潦潔澆潭潛潸潮澎潺潰潤澗潘滕潯潠潟熟熬熱熨牖犛獎獗瑩璋璃"],["bd40","瑾璀畿瘠瘩瘟瘤瘦瘡瘢皚皺盤瞎瞇瞌瞑瞋磋磅確磊碾磕碼磐稿稼穀稽稷稻窯窮箭箱範箴篆篇篁箠篌糊締練緯緻緘緬緝編緣線緞緩綞緙緲緹罵罷羯"],["bda1","翩耦膛膜膝膠膚膘蔗蔽蔚蓮蔬蔭蔓蔑蔣蔡蔔蓬蔥蓿蔆螂蝴蝶蝠蝦蝸蝨蝙蝗蝌蝓衛衝褐複褒褓褕褊誼諒談諄誕請諸課諉諂調誰論諍誶誹諛豌豎豬賠賞賦賤賬賭賢賣賜質賡赭趟趣踫踐踝踢踏踩踟踡踞躺輝輛輟輩輦輪輜輞"],["be40","輥適遮遨遭遷鄰鄭鄧鄱醇醉醋醃鋅銻銷鋪銬鋤鋁銳銼鋒鋇鋰銲閭閱霄霆震霉靠鞍鞋鞏頡頫頜颳養餓餒餘駝駐駟駛駑駕駒駙骷髮髯鬧魅魄魷魯鴆鴉"],["bea1","鴃麩麾黎墨齒儒儘儔儐儕冀冪凝劑劓勳噙噫噹噩噤噸噪器噥噱噯噬噢噶壁墾壇壅奮嬝嬴學寰導彊憲憑憩憊懍憶憾懊懈戰擅擁擋撻撼據擄擇擂操撿擒擔撾整曆曉暹曄曇暸樽樸樺橙橫橘樹橄橢橡橋橇樵機橈歙歷氅濂澱澡"],["bf40","濃澤濁澧澳激澹澶澦澠澴熾燉燐燒燈燕熹燎燙燜燃燄獨璜璣璘璟璞瓢甌甍瘴瘸瘺盧盥瞠瞞瞟瞥磨磚磬磧禦積穎穆穌穋窺篙簑築篤篛篡篩篦糕糖縊"],["bfa1","縑縈縛縣縞縝縉縐罹羲翰翱翮耨膳膩膨臻興艘艙蕊蕙蕈蕨蕩蕃蕉蕭蕪蕞螃螟螞螢融衡褪褲褥褫褡親覦諦諺諫諱謀諜諧諮諾謁謂諷諭諳諶諼豫豭貓賴蹄踱踴蹂踹踵輻輯輸輳辨辦遵遴選遲遼遺鄴醒錠錶鋸錳錯錢鋼錫錄錚"],["c040","錐錦錡錕錮錙閻隧隨險雕霎霑霖霍霓霏靛靜靦鞘頰頸頻頷頭頹頤餐館餞餛餡餚駭駢駱骸骼髻髭鬨鮑鴕鴣鴦鴨鴒鴛默黔龍龜優償儡儲勵嚎嚀嚐嚅嚇"],["c0a1","嚏壕壓壑壎嬰嬪嬤孺尷屨嶼嶺嶽嶸幫彌徽應懂懇懦懋戲戴擎擊擘擠擰擦擬擱擢擭斂斃曙曖檀檔檄檢檜櫛檣橾檗檐檠歜殮毚氈濘濱濟濠濛濤濫濯澀濬濡濩濕濮濰燧營燮燦燥燭燬燴燠爵牆獰獲璩環璦璨癆療癌盪瞳瞪瞰瞬"],["c140","瞧瞭矯磷磺磴磯礁禧禪穗窿簇簍篾篷簌篠糠糜糞糢糟糙糝縮績繆縷縲繃縫總縱繅繁縴縹繈縵縿縯罄翳翼聱聲聰聯聳臆臃膺臂臀膿膽臉膾臨舉艱薪"],["c1a1","薄蕾薜薑薔薯薛薇薨薊虧蟀蟑螳蟒蟆螫螻螺蟈蟋褻褶襄褸褽覬謎謗謙講謊謠謝謄謐豁谿豳賺賽購賸賻趨蹉蹋蹈蹊轄輾轂轅輿避遽還邁邂邀鄹醣醞醜鍍鎂錨鍵鍊鍥鍋錘鍾鍬鍛鍰鍚鍔闊闋闌闈闆隱隸雖霜霞鞠韓顆颶餵騁"],["c240","駿鮮鮫鮪鮭鴻鴿麋黏點黜黝黛鼾齋叢嚕嚮壙壘嬸彝懣戳擴擲擾攆擺擻擷斷曜朦檳檬櫃檻檸櫂檮檯歟歸殯瀉瀋濾瀆濺瀑瀏燻燼燾燸獷獵璧璿甕癖癘"],["c2a1","癒瞽瞿瞻瞼礎禮穡穢穠竄竅簫簧簪簞簣簡糧織繕繞繚繡繒繙罈翹翻職聶臍臏舊藏薩藍藐藉薰薺薹薦蟯蟬蟲蟠覆覲觴謨謹謬謫豐贅蹙蹣蹦蹤蹟蹕軀轉轍邇邃邈醫醬釐鎔鎊鎖鎢鎳鎮鎬鎰鎘鎚鎗闔闖闐闕離雜雙雛雞霤鞣鞦"],["c340","鞭韹額顏題顎顓颺餾餿餽餮馥騎髁鬃鬆魏魎魍鯊鯉鯽鯈鯀鵑鵝鵠黠鼕鼬儳嚥壞壟壢寵龐廬懲懷懶懵攀攏曠曝櫥櫝櫚櫓瀛瀟瀨瀚瀝瀕瀘爆爍牘犢獸"],["c3a1","獺璽瓊瓣疇疆癟癡矇礙禱穫穩簾簿簸簽簷籀繫繭繹繩繪羅繳羶羹羸臘藩藝藪藕藤藥藷蟻蠅蠍蟹蟾襠襟襖襞譁譜識證譚譎譏譆譙贈贊蹼蹲躇蹶蹬蹺蹴轔轎辭邊邋醱醮鏡鏑鏟鏃鏈鏜鏝鏖鏢鏍鏘鏤鏗鏨關隴難霪霧靡韜韻類"],["c440","願顛颼饅饉騖騙鬍鯨鯧鯖鯛鶉鵡鵲鵪鵬麒麗麓麴勸嚨嚷嚶嚴嚼壤孀孃孽寶巉懸懺攘攔攙曦朧櫬瀾瀰瀲爐獻瓏癢癥礦礪礬礫竇競籌籃籍糯糰辮繽繼"],["c4a1","纂罌耀臚艦藻藹蘑藺蘆蘋蘇蘊蠔蠕襤覺觸議譬警譯譟譫贏贍躉躁躅躂醴釋鐘鐃鏽闡霰飄饒饑馨騫騰騷騵鰓鰍鹹麵黨鼯齟齣齡儷儸囁囀囂夔屬巍懼懾攝攜斕曩櫻欄櫺殲灌爛犧瓖瓔癩矓籐纏續羼蘗蘭蘚蠣蠢蠡蠟襪襬覽譴"],["c540","護譽贓躊躍躋轟辯醺鐮鐳鐵鐺鐸鐲鐫闢霸霹露響顧顥饗驅驃驀騾髏魔魑鰭鰥鶯鶴鷂鶸麝黯鼙齜齦齧儼儻囈囊囉孿巔巒彎懿攤權歡灑灘玀瓤疊癮癬"],["c5a1","禳籠籟聾聽臟襲襯觼讀贖贗躑躓轡酈鑄鑑鑒霽霾韃韁顫饕驕驍髒鬚鱉鰱鰾鰻鷓鷗鼴齬齪龔囌巖戀攣攫攪曬欐瓚竊籤籣籥纓纖纔臢蘸蘿蠱變邐邏鑣鑠鑤靨顯饜驚驛驗髓體髑鱔鱗鱖鷥麟黴囑壩攬灞癱癲矗罐羈蠶蠹衢讓讒"],["c640","讖艷贛釀鑪靂靈靄韆顰驟鬢魘鱟鷹鷺鹼鹽鼇齷齲廳欖灣籬籮蠻觀躡釁鑲鑰顱饞髖鬣黌灤矚讚鑷韉驢驥纜讜躪釅鑽鑾鑼鱷鱸黷豔鑿鸚爨驪鬱鸛鸞籲"],["c940","乂乜凵匚厂万丌乇亍囗兀屮彳丏冇与丮亓仂仉仈冘勼卬厹圠夃夬尐巿旡殳毌气爿丱丼仨仜仩仡仝仚刌匜卌圢圣夗夯宁宄尒尻屴屳帄庀庂忉戉扐氕"],["c9a1","氶汃氿氻犮犰玊禸肊阞伎优伬仵伔仱伀价伈伝伂伅伢伓伄仴伒冱刓刉刐劦匢匟卍厊吇囡囟圮圪圴夼妀奼妅奻奾奷奿孖尕尥屼屺屻屾巟幵庄异弚彴忕忔忏扜扞扤扡扦扢扙扠扚扥旯旮朾朹朸朻机朿朼朳氘汆汒汜汏汊汔汋"],["ca40","汌灱牞犴犵玎甪癿穵网艸艼芀艽艿虍襾邙邗邘邛邔阢阤阠阣佖伻佢佉体佤伾佧佒佟佁佘伭伳伿佡冏冹刜刞刡劭劮匉卣卲厎厏吰吷吪呔呅吙吜吥吘"],["caa1","吽呏呁吨吤呇囮囧囥坁坅坌坉坋坒夆奀妦妘妠妗妎妢妐妏妧妡宎宒尨尪岍岏岈岋岉岒岊岆岓岕巠帊帎庋庉庌庈庍弅弝彸彶忒忑忐忭忨忮忳忡忤忣忺忯忷忻怀忴戺抃抌抎抏抔抇扱扻扺扰抁抈扷扽扲扴攷旰旴旳旲旵杅杇"],["cb40","杙杕杌杈杝杍杚杋毐氙氚汸汧汫沄沋沏汱汯汩沚汭沇沕沜汦汳汥汻沎灴灺牣犿犽狃狆狁犺狅玕玗玓玔玒町甹疔疕皁礽耴肕肙肐肒肜芐芏芅芎芑芓"],["cba1","芊芃芄豸迉辿邟邡邥邞邧邠阰阨阯阭丳侘佼侅佽侀侇佶佴侉侄佷佌侗佪侚佹侁佸侐侜侔侞侒侂侕佫佮冞冼冾刵刲刳剆刱劼匊匋匼厒厔咇呿咁咑咂咈呫呺呾呥呬呴呦咍呯呡呠咘呣呧呤囷囹坯坲坭坫坱坰坶垀坵坻坳坴坢"],["cc40","坨坽夌奅妵妺姏姎妲姌姁妶妼姃姖妱妽姀姈妴姇孢孥宓宕屄屇岮岤岠岵岯岨岬岟岣岭岢岪岧岝岥岶岰岦帗帔帙弨弢弣弤彔徂彾彽忞忥怭怦怙怲怋"],["cca1","怴怊怗怳怚怞怬怢怍怐怮怓怑怌怉怜戔戽抭抴拑抾抪抶拊抮抳抯抻抩抰抸攽斨斻昉旼昄昒昈旻昃昋昍昅旽昑昐曶朊枅杬枎枒杶杻枘枆构杴枍枌杺枟枑枙枃杽极杸杹枔欥殀歾毞氝沓泬泫泮泙沶泔沭泧沷泐泂沺泃泆泭泲"],["cd40","泒泝沴沊沝沀泞泀洰泍泇沰泹泏泩泑炔炘炅炓炆炄炑炖炂炚炃牪狖狋狘狉狜狒狔狚狌狑玤玡玭玦玢玠玬玝瓝瓨甿畀甾疌疘皯盳盱盰盵矸矼矹矻矺"],["cda1","矷祂礿秅穸穻竻籵糽耵肏肮肣肸肵肭舠芠苀芫芚芘芛芵芧芮芼芞芺芴芨芡芩苂芤苃芶芢虰虯虭虮豖迒迋迓迍迖迕迗邲邴邯邳邰阹阽阼阺陃俍俅俓侲俉俋俁俔俜俙侻侳俛俇俖侺俀侹俬剄剉勀勂匽卼厗厖厙厘咺咡咭咥哏"],["ce40","哃茍咷咮哖咶哅哆咠呰咼咢咾呲哞咰垵垞垟垤垌垗垝垛垔垘垏垙垥垚垕壴复奓姡姞姮娀姱姝姺姽姼姶姤姲姷姛姩姳姵姠姾姴姭宨屌峐峘峌峗峋峛"],["cea1","峞峚峉峇峊峖峓峔峏峈峆峎峟峸巹帡帢帣帠帤庰庤庢庛庣庥弇弮彖徆怷怹恔恲恞恅恓恇恉恛恌恀恂恟怤恄恘恦恮扂扃拏挍挋拵挎挃拫拹挏挌拸拶挀挓挔拺挕拻拰敁敃斪斿昶昡昲昵昜昦昢昳昫昺昝昴昹昮朏朐柁柲柈枺"],["cf40","柜枻柸柘柀枷柅柫柤柟枵柍枳柷柶柮柣柂枹柎柧柰枲柼柆柭柌枮柦柛柺柉柊柃柪柋欨殂殄殶毖毘毠氠氡洨洴洭洟洼洿洒洊泚洳洄洙洺洚洑洀洝浂"],["cfa1","洁洘洷洃洏浀洇洠洬洈洢洉洐炷炟炾炱炰炡炴炵炩牁牉牊牬牰牳牮狊狤狨狫狟狪狦狣玅珌珂珈珅玹玶玵玴珫玿珇玾珃珆玸珋瓬瓮甮畇畈疧疪癹盄眈眃眄眅眊盷盻盺矧矨砆砑砒砅砐砏砎砉砃砓祊祌祋祅祄秕种秏秖秎窀"],["d040","穾竑笀笁籺籸籹籿粀粁紃紈紁罘羑羍羾耇耎耏耔耷胘胇胠胑胈胂胐胅胣胙胜胊胕胉胏胗胦胍臿舡芔苙苾苹茇苨茀苕茺苫苖苴苬苡苲苵茌苻苶苰苪"],["d0a1","苤苠苺苳苭虷虴虼虳衁衎衧衪衩觓訄訇赲迣迡迮迠郱邽邿郕郅邾郇郋郈釔釓陔陏陑陓陊陎倞倅倇倓倢倰倛俵俴倳倷倬俶俷倗倜倠倧倵倯倱倎党冔冓凊凄凅凈凎剡剚剒剞剟剕剢勍匎厞唦哢唗唒哧哳哤唚哿唄唈哫唑唅哱"],["d140","唊哻哷哸哠唎唃唋圁圂埌堲埕埒垺埆垽垼垸垶垿埇埐垹埁夎奊娙娖娭娮娕娏娗娊娞娳孬宧宭宬尃屖屔峬峿峮峱峷崀峹帩帨庨庮庪庬弳弰彧恝恚恧"],["d1a1","恁悢悈悀悒悁悝悃悕悛悗悇悜悎戙扆拲挐捖挬捄捅挶捃揤挹捋捊挼挩捁挴捘捔捙挭捇挳捚捑挸捗捀捈敊敆旆旃旄旂晊晟晇晑朒朓栟栚桉栲栳栻桋桏栖栱栜栵栫栭栯桎桄栴栝栒栔栦栨栮桍栺栥栠欬欯欭欱欴歭肂殈毦毤"],["d240","毨毣毢毧氥浺浣浤浶洍浡涒浘浢浭浯涑涍淯浿涆浞浧浠涗浰浼浟涂涘洯浨涋浾涀涄洖涃浻浽浵涐烜烓烑烝烋缹烢烗烒烞烠烔烍烅烆烇烚烎烡牂牸"],["d2a1","牷牶猀狺狴狾狶狳狻猁珓珙珥珖玼珧珣珩珜珒珛珔珝珚珗珘珨瓞瓟瓴瓵甡畛畟疰痁疻痄痀疿疶疺皊盉眝眛眐眓眒眣眑眕眙眚眢眧砣砬砢砵砯砨砮砫砡砩砳砪砱祔祛祏祜祓祒祑秫秬秠秮秭秪秜秞秝窆窉窅窋窌窊窇竘笐"],["d340","笄笓笅笏笈笊笎笉笒粄粑粊粌粈粍粅紞紝紑紎紘紖紓紟紒紏紌罜罡罞罠罝罛羖羒翃翂翀耖耾耹胺胲胹胵脁胻脀舁舯舥茳茭荄茙荑茥荖茿荁茦茜茢"],["d3a1","荂荎茛茪茈茼荍茖茤茠茷茯茩荇荅荌荓茞茬荋茧荈虓虒蚢蚨蚖蚍蚑蚞蚇蚗蚆蚋蚚蚅蚥蚙蚡蚧蚕蚘蚎蚝蚐蚔衃衄衭衵衶衲袀衱衿衯袃衾衴衼訒豇豗豻貤貣赶赸趵趷趶軑軓迾迵适迿迻逄迼迶郖郠郙郚郣郟郥郘郛郗郜郤酐"],["d440","酎酏釕釢釚陜陟隼飣髟鬯乿偰偪偡偞偠偓偋偝偲偈偍偁偛偊偢倕偅偟偩偫偣偤偆偀偮偳偗偑凐剫剭剬剮勖勓匭厜啵啶唼啍啐唴唪啑啢唶唵唰啒啅"],["d4a1","唌唲啥啎唹啈唭唻啀啋圊圇埻堔埢埶埜埴堀埭埽堈埸堋埳埏堇埮埣埲埥埬埡堎埼堐埧堁堌埱埩埰堍堄奜婠婘婕婧婞娸娵婭婐婟婥婬婓婤婗婃婝婒婄婛婈媎娾婍娹婌婰婩婇婑婖婂婜孲孮寁寀屙崞崋崝崚崠崌崨崍崦崥崏"],["d540","崰崒崣崟崮帾帴庱庴庹庲庳弶弸徛徖徟悊悐悆悾悰悺惓惔惏惤惙惝惈悱惛悷惊悿惃惍惀挲捥掊掂捽掽掞掭掝掗掫掎捯掇掐据掯捵掜捭掮捼掤挻掟"],["d5a1","捸掅掁掑掍捰敓旍晥晡晛晙晜晢朘桹梇梐梜桭桮梮梫楖桯梣梬梩桵桴梲梏桷梒桼桫桲梪梀桱桾梛梖梋梠梉梤桸桻梑梌梊桽欶欳欷欸殑殏殍殎殌氪淀涫涴涳湴涬淩淢涷淶淔渀淈淠淟淖涾淥淜淝淛淴淊涽淭淰涺淕淂淏淉"],["d640","淐淲淓淽淗淍淣涻烺焍烷焗烴焌烰焄烳焐烼烿焆焓焀烸烶焋焂焎牾牻牼牿猝猗猇猑猘猊猈狿猏猞玈珶珸珵琄琁珽琇琀珺珼珿琌琋珴琈畤畣痎痒痏"],["d6a1","痋痌痑痐皏皉盓眹眯眭眱眲眴眳眽眥眻眵硈硒硉硍硊硌砦硅硐祤祧祩祪祣祫祡离秺秸秶秷窏窔窐笵筇笴笥笰笢笤笳笘笪笝笱笫笭笯笲笸笚笣粔粘粖粣紵紽紸紶紺絅紬紩絁絇紾紿絊紻紨罣羕羜羝羛翊翋翍翐翑翇翏翉耟"],["d740","耞耛聇聃聈脘脥脙脛脭脟脬脞脡脕脧脝脢舑舸舳舺舴舲艴莐莣莨莍荺荳莤荴莏莁莕莙荵莔莩荽莃莌莝莛莪莋荾莥莯莈莗莰荿莦莇莮荶莚虙虖蚿蚷"],["d7a1","蛂蛁蛅蚺蚰蛈蚹蚳蚸蛌蚴蚻蚼蛃蚽蚾衒袉袕袨袢袪袚袑袡袟袘袧袙袛袗袤袬袌袓袎覂觖觙觕訰訧訬訞谹谻豜豝豽貥赽赻赹趼跂趹趿跁軘軞軝軜軗軠軡逤逋逑逜逌逡郯郪郰郴郲郳郔郫郬郩酖酘酚酓酕釬釴釱釳釸釤釹釪"],["d840","釫釷釨釮镺閆閈陼陭陫陱陯隿靪頄飥馗傛傕傔傞傋傣傃傌傎傝偨傜傒傂傇兟凔匒匑厤厧喑喨喥喭啷噅喢喓喈喏喵喁喣喒喤啽喌喦啿喕喡喎圌堩堷"],["d8a1","堙堞堧堣堨埵塈堥堜堛堳堿堶堮堹堸堭堬堻奡媯媔媟婺媢媞婸媦婼媥媬媕媮娷媄媊媗媃媋媩婻婽媌媜媏媓媝寪寍寋寔寑寊寎尌尰崷嵃嵫嵁嵋崿崵嵑嵎嵕崳崺嵒崽崱嵙嵂崹嵉崸崼崲崶嵀嵅幄幁彘徦徥徫惉悹惌惢惎惄愔"],["d940","惲愊愖愅惵愓惸惼惾惁愃愘愝愐惿愄愋扊掔掱掰揎揥揨揯揃撝揳揊揠揶揕揲揵摡揟掾揝揜揄揘揓揂揇揌揋揈揰揗揙攲敧敪敤敜敨敥斌斝斞斮旐旒"],["d9a1","晼晬晻暀晱晹晪晲朁椌棓椄棜椪棬棪棱椏棖棷棫棤棶椓椐棳棡椇棌椈楰梴椑棯棆椔棸棐棽棼棨椋椊椗棎棈棝棞棦棴棑椆棔棩椕椥棇欹欻欿欼殔殗殙殕殽毰毲毳氰淼湆湇渟湉溈渼渽湅湢渫渿湁湝湳渜渳湋湀湑渻渃渮湞"],["da40","湨湜湡渱渨湠湱湫渹渢渰湓湥渧湸湤湷湕湹湒湦渵渶湚焠焞焯烻焮焱焣焥焢焲焟焨焺焛牋牚犈犉犆犅犋猒猋猰猢猱猳猧猲猭猦猣猵猌琮琬琰琫琖"],["daa1","琚琡琭琱琤琣琝琩琠琲瓻甯畯畬痧痚痡痦痝痟痤痗皕皒盚睆睇睄睍睅睊睎睋睌矞矬硠硤硥硜硭硱硪确硰硩硨硞硢祴祳祲祰稂稊稃稌稄窙竦竤筊笻筄筈筌筎筀筘筅粢粞粨粡絘絯絣絓絖絧絪絏絭絜絫絒絔絩絑絟絎缾缿罥"],["db40","罦羢羠羡翗聑聏聐胾胔腃腊腒腏腇脽腍脺臦臮臷臸臹舄舼舽舿艵茻菏菹萣菀菨萒菧菤菼菶萐菆菈菫菣莿萁菝菥菘菿菡菋菎菖菵菉萉萏菞萑萆菂菳"],["dba1","菕菺菇菑菪萓菃菬菮菄菻菗菢萛菛菾蛘蛢蛦蛓蛣蛚蛪蛝蛫蛜蛬蛩蛗蛨蛑衈衖衕袺裗袹袸裀袾袶袼袷袽袲褁裉覕覘覗觝觚觛詎詍訹詙詀詗詘詄詅詒詈詑詊詌詏豟貁貀貺貾貰貹貵趄趀趉跘跓跍跇跖跜跏跕跙跈跗跅軯軷軺"],["dc40","軹軦軮軥軵軧軨軶軫軱軬軴軩逭逴逯鄆鄬鄄郿郼鄈郹郻鄁鄀鄇鄅鄃酡酤酟酢酠鈁鈊鈥鈃鈚鈦鈏鈌鈀鈒釿釽鈆鈄鈧鈂鈜鈤鈙鈗鈅鈖镻閍閌閐隇陾隈"],["dca1","隉隃隀雂雈雃雱雰靬靰靮頇颩飫鳦黹亃亄亶傽傿僆傮僄僊傴僈僂傰僁傺傱僋僉傶傸凗剺剸剻剼嗃嗛嗌嗐嗋嗊嗝嗀嗔嗄嗩喿嗒喍嗏嗕嗢嗖嗈嗲嗍嗙嗂圔塓塨塤塏塍塉塯塕塎塝塙塥塛堽塣塱壼嫇嫄嫋媺媸媱媵媰媿嫈媻嫆"],["dd40","媷嫀嫊媴媶嫍媹媐寖寘寙尟尳嵱嵣嵊嵥嵲嵬嵞嵨嵧嵢巰幏幎幊幍幋廅廌廆廋廇彀徯徭惷慉慊愫慅愶愲愮慆愯慏愩慀戠酨戣戥戤揅揱揫搐搒搉搠搤"],["dda1","搳摃搟搕搘搹搷搢搣搌搦搰搨摁搵搯搊搚摀搥搧搋揧搛搮搡搎敯斒旓暆暌暕暐暋暊暙暔晸朠楦楟椸楎楢楱椿楅楪椹楂楗楙楺楈楉椵楬椳椽楥棰楸椴楩楀楯楄楶楘楁楴楌椻楋椷楜楏楑椲楒椯楻椼歆歅歃歂歈歁殛嗀毻毼"],["de40","毹毷毸溛滖滈溏滀溟溓溔溠溱溹滆滒溽滁溞滉溷溰滍溦滏溲溾滃滜滘溙溒溎溍溤溡溿溳滐滊溗溮溣煇煔煒煣煠煁煝煢煲煸煪煡煂煘煃煋煰煟煐煓"],["dea1","煄煍煚牏犍犌犑犐犎猼獂猻猺獀獊獉瑄瑊瑋瑒瑑瑗瑀瑏瑐瑎瑂瑆瑍瑔瓡瓿瓾瓽甝畹畷榃痯瘏瘃痷痾痼痹痸瘐痻痶痭痵痽皙皵盝睕睟睠睒睖睚睩睧睔睙睭矠碇碚碔碏碄碕碅碆碡碃硹碙碀碖硻祼禂祽祹稑稘稙稒稗稕稢稓"],["df40","稛稐窣窢窞竫筦筤筭筴筩筲筥筳筱筰筡筸筶筣粲粴粯綈綆綀綍絿綅絺綎絻綃絼綌綔綄絽綒罭罫罧罨罬羦羥羧翛翜耡腤腠腷腜腩腛腢腲朡腞腶腧腯"],["dfa1","腄腡舝艉艄艀艂艅蓱萿葖葶葹蒏蒍葥葑葀蒆葧萰葍葽葚葙葴葳葝蔇葞萷萺萴葺葃葸萲葅萩菙葋萯葂萭葟葰萹葎葌葒葯蓅蒎萻葇萶萳葨葾葄萫葠葔葮葐蜋蜄蛷蜌蛺蛖蛵蝍蛸蜎蜉蜁蛶蜍蜅裖裋裍裎裞裛裚裌裐覅覛觟觥觤"],["e040","觡觠觢觜触詶誆詿詡訿詷誂誄詵誃誁詴詺谼豋豊豥豤豦貆貄貅賌赨赩趑趌趎趏趍趓趔趐趒跰跠跬跱跮跐跩跣跢跧跲跫跴輆軿輁輀輅輇輈輂輋遒逿"],["e0a1","遄遉逽鄐鄍鄏鄑鄖鄔鄋鄎酮酯鉈鉒鈰鈺鉦鈳鉥鉞銃鈮鉊鉆鉭鉬鉏鉠鉧鉯鈶鉡鉰鈱鉔鉣鉐鉲鉎鉓鉌鉖鈲閟閜閞閛隒隓隑隗雎雺雽雸雵靳靷靸靲頏頍頎颬飶飹馯馲馰馵骭骫魛鳪鳭鳧麀黽僦僔僗僨僳僛僪僝僤僓僬僰僯僣僠"],["e140","凘劀劁勩勫匰厬嘧嘕嘌嘒嗼嘏嘜嘁嘓嘂嗺嘝嘄嗿嗹墉塼墐墘墆墁塿塴墋塺墇墑墎塶墂墈塻墔墏壾奫嫜嫮嫥嫕嫪嫚嫭嫫嫳嫢嫠嫛嫬嫞嫝嫙嫨嫟孷寠"],["e1a1","寣屣嶂嶀嵽嶆嵺嶁嵷嶊嶉嶈嵾嵼嶍嵹嵿幘幙幓廘廑廗廎廜廕廙廒廔彄彃彯徶愬愨慁慞慱慳慒慓慲慬憀慴慔慺慛慥愻慪慡慖戩戧戫搫摍摛摝摴摶摲摳摽摵摦撦摎撂摞摜摋摓摠摐摿搿摬摫摙摥摷敳斠暡暠暟朅朄朢榱榶槉"],["e240","榠槎榖榰榬榼榑榙榎榧榍榩榾榯榿槄榽榤槔榹槊榚槏榳榓榪榡榞槙榗榐槂榵榥槆歊歍歋殞殟殠毃毄毾滎滵滱漃漥滸漷滻漮漉潎漙漚漧漘漻漒滭漊"],["e2a1","漶潳滹滮漭潀漰漼漵滫漇漎潃漅滽滶漹漜滼漺漟漍漞漈漡熇熐熉熀熅熂熏煻熆熁熗牄牓犗犕犓獃獍獑獌瑢瑳瑱瑵瑲瑧瑮甀甂甃畽疐瘖瘈瘌瘕瘑瘊瘔皸瞁睼瞅瞂睮瞀睯睾瞃碲碪碴碭碨硾碫碞碥碠碬碢碤禘禊禋禖禕禔禓"],["e340","禗禈禒禐稫穊稰稯稨稦窨窫窬竮箈箜箊箑箐箖箍箌箛箎箅箘劄箙箤箂粻粿粼粺綧綷緂綣綪緁緀緅綝緎緄緆緋緌綯綹綖綼綟綦綮綩綡緉罳翢翣翥翞"],["e3a1","耤聝聜膉膆膃膇膍膌膋舕蒗蒤蒡蒟蒺蓎蓂蒬蒮蒫蒹蒴蓁蓍蒪蒚蒱蓐蒝蒧蒻蒢蒔蓇蓌蒛蒩蒯蒨蓖蒘蒶蓏蒠蓗蓔蓒蓛蒰蒑虡蜳蜣蜨蝫蝀蜮蜞蜡蜙蜛蝃蜬蝁蜾蝆蜠蜲蜪蜭蜼蜒蜺蜱蜵蝂蜦蜧蜸蜤蜚蜰蜑裷裧裱裲裺裾裮裼裶裻"],["e440","裰裬裫覝覡覟覞觩觫觨誫誙誋誒誏誖谽豨豩賕賏賗趖踉踂跿踍跽踊踃踇踆踅跾踀踄輐輑輎輍鄣鄜鄠鄢鄟鄝鄚鄤鄡鄛酺酲酹酳銥銤鉶銛鉺銠銔銪銍"],["e4a1","銦銚銫鉹銗鉿銣鋮銎銂銕銢鉽銈銡銊銆銌銙銧鉾銇銩銝銋鈭隞隡雿靘靽靺靾鞃鞀鞂靻鞄鞁靿韎韍頖颭颮餂餀餇馝馜駃馹馻馺駂馽駇骱髣髧鬾鬿魠魡魟鳱鳲鳵麧僿儃儰僸儆儇僶僾儋儌僽儊劋劌勱勯噈噂噌嘵噁噊噉噆噘"],["e540","噚噀嘳嘽嘬嘾嘸嘪嘺圚墫墝墱墠墣墯墬墥墡壿嫿嫴嫽嫷嫶嬃嫸嬂嫹嬁嬇嬅嬏屧嶙嶗嶟嶒嶢嶓嶕嶠嶜嶡嶚嶞幩幝幠幜緳廛廞廡彉徲憋憃慹憱憰憢憉"],["e5a1","憛憓憯憭憟憒憪憡憍慦憳戭摮摰撖撠撅撗撜撏撋撊撌撣撟摨撱撘敶敺敹敻斲斳暵暰暩暲暷暪暯樀樆樗槥槸樕槱槤樠槿槬槢樛樝槾樧槲槮樔槷槧橀樈槦槻樍槼槫樉樄樘樥樏槶樦樇槴樖歑殥殣殢殦氁氀毿氂潁漦潾澇濆澒"],["e640","澍澉澌潢潏澅潚澖潶潬澂潕潲潒潐潗澔澓潝漀潡潫潽潧澐潓澋潩潿澕潣潷潪潻熲熯熛熰熠熚熩熵熝熥熞熤熡熪熜熧熳犘犚獘獒獞獟獠獝獛獡獚獙"],["e6a1","獢璇璉璊璆璁瑽璅璈瑼瑹甈甇畾瘥瘞瘙瘝瘜瘣瘚瘨瘛皜皝皞皛瞍瞏瞉瞈磍碻磏磌磑磎磔磈磃磄磉禚禡禠禜禢禛歶稹窲窴窳箷篋箾箬篎箯箹篊箵糅糈糌糋緷緛緪緧緗緡縃緺緦緶緱緰緮緟罶羬羰羭翭翫翪翬翦翨聤聧膣膟"],["e740","膞膕膢膙膗舖艏艓艒艐艎艑蔤蔻蔏蔀蔩蔎蔉蔍蔟蔊蔧蔜蓻蔫蓺蔈蔌蓴蔪蓲蔕蓷蓫蓳蓼蔒蓪蓩蔖蓾蔨蔝蔮蔂蓽蔞蓶蔱蔦蓧蓨蓰蓯蓹蔘蔠蔰蔋蔙蔯虢"],["e7a1","蝖蝣蝤蝷蟡蝳蝘蝔蝛蝒蝡蝚蝑蝞蝭蝪蝐蝎蝟蝝蝯蝬蝺蝮蝜蝥蝏蝻蝵蝢蝧蝩衚褅褌褔褋褗褘褙褆褖褑褎褉覢覤覣觭觰觬諏諆誸諓諑諔諕誻諗誾諀諅諘諃誺誽諙谾豍貏賥賟賙賨賚賝賧趠趜趡趛踠踣踥踤踮踕踛踖踑踙踦踧"],["e840","踔踒踘踓踜踗踚輬輤輘輚輠輣輖輗遳遰遯遧遫鄯鄫鄩鄪鄲鄦鄮醅醆醊醁醂醄醀鋐鋃鋄鋀鋙銶鋏鋱鋟鋘鋩鋗鋝鋌鋯鋂鋨鋊鋈鋎鋦鋍鋕鋉鋠鋞鋧鋑鋓"],["e8a1","銵鋡鋆銴镼閬閫閮閰隤隢雓霅霈霂靚鞊鞎鞈韐韏頞頝頦頩頨頠頛頧颲餈飺餑餔餖餗餕駜駍駏駓駔駎駉駖駘駋駗駌骳髬髫髳髲髱魆魃魧魴魱魦魶魵魰魨魤魬鳼鳺鳽鳿鳷鴇鴀鳹鳻鴈鴅鴄麃黓鼏鼐儜儓儗儚儑凞匴叡噰噠噮"],["e940","噳噦噣噭噲噞噷圜圛壈墽壉墿墺壂墼壆嬗嬙嬛嬡嬔嬓嬐嬖嬨嬚嬠嬞寯嶬嶱嶩嶧嶵嶰嶮嶪嶨嶲嶭嶯嶴幧幨幦幯廩廧廦廨廥彋徼憝憨憖懅憴懆懁懌憺"],["e9a1","憿憸憌擗擖擐擏擉撽撉擃擛擳擙攳敿敼斢曈暾曀曊曋曏暽暻暺曌朣樴橦橉橧樲橨樾橝橭橶橛橑樨橚樻樿橁橪橤橐橏橔橯橩橠樼橞橖橕橍橎橆歕歔歖殧殪殫毈毇氄氃氆澭濋澣濇澼濎濈潞濄澽澞濊澨瀄澥澮澺澬澪濏澿澸"],["ea40","澢濉澫濍澯澲澰燅燂熿熸燖燀燁燋燔燊燇燏熽燘熼燆燚燛犝犞獩獦獧獬獥獫獪瑿璚璠璔璒璕璡甋疀瘯瘭瘱瘽瘳瘼瘵瘲瘰皻盦瞚瞝瞡瞜瞛瞢瞣瞕瞙"],["eaa1","瞗磝磩磥磪磞磣磛磡磢磭磟磠禤穄穈穇窶窸窵窱窷篞篣篧篝篕篥篚篨篹篔篪篢篜篫篘篟糒糔糗糐糑縒縡縗縌縟縠縓縎縜縕縚縢縋縏縖縍縔縥縤罃罻罼罺羱翯耪耩聬膱膦膮膹膵膫膰膬膴膲膷膧臲艕艖艗蕖蕅蕫蕍蕓蕡蕘"],["eb40","蕀蕆蕤蕁蕢蕄蕑蕇蕣蔾蕛蕱蕎蕮蕵蕕蕧蕠薌蕦蕝蕔蕥蕬虣虥虤螛螏螗螓螒螈螁螖螘蝹螇螣螅螐螑螝螄螔螜螚螉褞褦褰褭褮褧褱褢褩褣褯褬褟觱諠"],["eba1","諢諲諴諵諝謔諤諟諰諈諞諡諨諿諯諻貑貒貐賵賮賱賰賳赬赮趥趧踳踾踸蹀蹅踶踼踽蹁踰踿躽輶輮輵輲輹輷輴遶遹遻邆郺鄳鄵鄶醓醐醑醍醏錧錞錈錟錆錏鍺錸錼錛錣錒錁鍆錭錎錍鋋錝鋺錥錓鋹鋷錴錂錤鋿錩錹錵錪錔錌"],["ec40","錋鋾錉錀鋻錖閼闍閾閹閺閶閿閵閽隩雔霋霒霐鞙鞗鞔韰韸頵頯頲餤餟餧餩馞駮駬駥駤駰駣駪駩駧骹骿骴骻髶髺髹髷鬳鮀鮅鮇魼魾魻鮂鮓鮒鮐魺鮕"],["eca1","魽鮈鴥鴗鴠鴞鴔鴩鴝鴘鴢鴐鴙鴟麈麆麇麮麭黕黖黺鼒鼽儦儥儢儤儠儩勴嚓嚌嚍嚆嚄嚃噾嚂噿嚁壖壔壏壒嬭嬥嬲嬣嬬嬧嬦嬯嬮孻寱寲嶷幬幪徾徻懃憵憼懧懠懥懤懨懞擯擩擣擫擤擨斁斀斶旚曒檍檖檁檥檉檟檛檡檞檇檓檎"],["ed40","檕檃檨檤檑橿檦檚檅檌檒歛殭氉濌澩濴濔濣濜濭濧濦濞濲濝濢濨燡燱燨燲燤燰燢獳獮獯璗璲璫璐璪璭璱璥璯甐甑甒甏疄癃癈癉癇皤盩瞵瞫瞲瞷瞶"],["eda1","瞴瞱瞨矰磳磽礂磻磼磲礅磹磾礄禫禨穜穛穖穘穔穚窾竀竁簅簏篲簀篿篻簎篴簋篳簂簉簃簁篸篽簆篰篱簐簊糨縭縼繂縳顈縸縪繉繀繇縩繌縰縻縶繄縺罅罿罾罽翴翲耬膻臄臌臊臅臇膼臩艛艚艜薃薀薏薧薕薠薋薣蕻薤薚薞"],["ee40","蕷蕼薉薡蕺蕸蕗薎薖薆薍薙薝薁薢薂薈薅蕹蕶薘薐薟虨螾螪螭蟅螰螬螹螵螼螮蟉蟃蟂蟌螷螯蟄蟊螴螶螿螸螽蟞螲褵褳褼褾襁襒褷襂覭覯覮觲觳謞"],["eea1","謘謖謑謅謋謢謏謒謕謇謍謈謆謜謓謚豏豰豲豱豯貕貔賹赯蹎蹍蹓蹐蹌蹇轃轀邅遾鄸醚醢醛醙醟醡醝醠鎡鎃鎯鍤鍖鍇鍼鍘鍜鍶鍉鍐鍑鍠鍭鎏鍌鍪鍹鍗鍕鍒鍏鍱鍷鍻鍡鍞鍣鍧鎀鍎鍙闇闀闉闃闅閷隮隰隬霠霟霘霝霙鞚鞡鞜"],["ef40","鞞鞝韕韔韱顁顄顊顉顅顃餥餫餬餪餳餲餯餭餱餰馘馣馡騂駺駴駷駹駸駶駻駽駾駼騃骾髾髽鬁髼魈鮚鮨鮞鮛鮦鮡鮥鮤鮆鮢鮠鮯鴳鵁鵧鴶鴮鴯鴱鴸鴰"],["efa1","鵅鵂鵃鴾鴷鵀鴽翵鴭麊麉麍麰黈黚黻黿鼤鼣鼢齔龠儱儭儮嚘嚜嚗嚚嚝嚙奰嬼屩屪巀幭幮懘懟懭懮懱懪懰懫懖懩擿攄擽擸攁攃擼斔旛曚曛曘櫅檹檽櫡櫆檺檶檷櫇檴檭歞毉氋瀇瀌瀍瀁瀅瀔瀎濿瀀濻瀦濼濷瀊爁燿燹爃燽獶"],["f040","璸瓀璵瓁璾璶璻瓂甔甓癜癤癙癐癓癗癚皦皽盬矂瞺磿礌礓礔礉礐礒礑禭禬穟簜簩簙簠簟簭簝簦簨簢簥簰繜繐繖繣繘繢繟繑繠繗繓羵羳翷翸聵臑臒"],["f0a1","臐艟艞薴藆藀藃藂薳薵薽藇藄薿藋藎藈藅薱薶藒蘤薸薷薾虩蟧蟦蟢蟛蟫蟪蟥蟟蟳蟤蟔蟜蟓蟭蟘蟣螤蟗蟙蠁蟴蟨蟝襓襋襏襌襆襐襑襉謪謧謣謳謰謵譇謯謼謾謱謥謷謦謶謮謤謻謽謺豂豵貙貘貗賾贄贂贀蹜蹢蹠蹗蹖蹞蹥蹧"],["f140","蹛蹚蹡蹝蹩蹔轆轇轈轋鄨鄺鄻鄾醨醥醧醯醪鎵鎌鎒鎷鎛鎝鎉鎧鎎鎪鎞鎦鎕鎈鎙鎟鎍鎱鎑鎲鎤鎨鎴鎣鎥闒闓闑隳雗雚巂雟雘雝霣霢霥鞬鞮鞨鞫鞤鞪"],["f1a1","鞢鞥韗韙韖韘韺顐顑顒颸饁餼餺騏騋騉騍騄騑騊騅騇騆髀髜鬈鬄鬅鬩鬵魊魌魋鯇鯆鯃鮿鯁鮵鮸鯓鮶鯄鮹鮽鵜鵓鵏鵊鵛鵋鵙鵖鵌鵗鵒鵔鵟鵘鵚麎麌黟鼁鼀鼖鼥鼫鼪鼩鼨齌齕儴儵劖勷厴嚫嚭嚦嚧嚪嚬壚壝壛夒嬽嬾嬿巃幰"],["f240","徿懻攇攐攍攉攌攎斄旞旝曞櫧櫠櫌櫑櫙櫋櫟櫜櫐櫫櫏櫍櫞歠殰氌瀙瀧瀠瀖瀫瀡瀢瀣瀩瀗瀤瀜瀪爌爊爇爂爅犥犦犤犣犡瓋瓅璷瓃甖癠矉矊矄矱礝礛"],["f2a1","礡礜礗礞禰穧穨簳簼簹簬簻糬糪繶繵繸繰繷繯繺繲繴繨罋罊羃羆羷翽翾聸臗臕艤艡艣藫藱藭藙藡藨藚藗藬藲藸藘藟藣藜藑藰藦藯藞藢蠀蟺蠃蟶蟷蠉蠌蠋蠆蟼蠈蟿蠊蠂襢襚襛襗襡襜襘襝襙覈覷覶觶譐譈譊譀譓譖譔譋譕"],["f340","譑譂譒譗豃豷豶貚贆贇贉趬趪趭趫蹭蹸蹳蹪蹯蹻軂轒轑轏轐轓辴酀鄿醰醭鏞鏇鏏鏂鏚鏐鏹鏬鏌鏙鎩鏦鏊鏔鏮鏣鏕鏄鏎鏀鏒鏧镽闚闛雡霩霫霬霨霦"],["f3a1","鞳鞷鞶韝韞韟顜顙顝顗颿颽颻颾饈饇饃馦馧騚騕騥騝騤騛騢騠騧騣騞騜騔髂鬋鬊鬎鬌鬷鯪鯫鯠鯞鯤鯦鯢鯰鯔鯗鯬鯜鯙鯥鯕鯡鯚鵷鶁鶊鶄鶈鵱鶀鵸鶆鶋鶌鵽鵫鵴鵵鵰鵩鶅鵳鵻鶂鵯鵹鵿鶇鵨麔麑黀黼鼭齀齁齍齖齗齘匷嚲"],["f440","嚵嚳壣孅巆巇廮廯忀忁懹攗攖攕攓旟曨曣曤櫳櫰櫪櫨櫹櫱櫮櫯瀼瀵瀯瀷瀴瀱灂瀸瀿瀺瀹灀瀻瀳灁爓爔犨獽獼璺皫皪皾盭矌矎矏矍矲礥礣礧礨礤礩"],["f4a1","禲穮穬穭竷籉籈籊籇籅糮繻繾纁纀羺翿聹臛臙舋艨艩蘢藿蘁藾蘛蘀藶蘄蘉蘅蘌藽蠙蠐蠑蠗蠓蠖襣襦覹觷譠譪譝譨譣譥譧譭趮躆躈躄轙轖轗轕轘轚邍酃酁醷醵醲醳鐋鐓鏻鐠鐏鐔鏾鐕鐐鐨鐙鐍鏵鐀鏷鐇鐎鐖鐒鏺鐉鏸鐊鏿"],["f540","鏼鐌鏶鐑鐆闞闠闟霮霯鞹鞻韽韾顠顢顣顟飁飂饐饎饙饌饋饓騲騴騱騬騪騶騩騮騸騭髇髊髆鬐鬒鬑鰋鰈鯷鰅鰒鯸鱀鰇鰎鰆鰗鰔鰉鶟鶙鶤鶝鶒鶘鶐鶛"],["f5a1","鶠鶔鶜鶪鶗鶡鶚鶢鶨鶞鶣鶿鶩鶖鶦鶧麙麛麚黥黤黧黦鼰鼮齛齠齞齝齙龑儺儹劘劗囃嚽嚾孈孇巋巏廱懽攛欂櫼欃櫸欀灃灄灊灈灉灅灆爝爚爙獾甗癪矐礭礱礯籔籓糲纊纇纈纋纆纍罍羻耰臝蘘蘪蘦蘟蘣蘜蘙蘧蘮蘡蘠蘩蘞蘥"],["f640","蠩蠝蠛蠠蠤蠜蠫衊襭襩襮襫觺譹譸譅譺譻贐贔趯躎躌轞轛轝酆酄酅醹鐿鐻鐶鐩鐽鐼鐰鐹鐪鐷鐬鑀鐱闥闤闣霵霺鞿韡顤飉飆飀饘饖騹騽驆驄驂驁騺"],["f6a1","騿髍鬕鬗鬘鬖鬺魒鰫鰝鰜鰬鰣鰨鰩鰤鰡鶷鶶鶼鷁鷇鷊鷏鶾鷅鷃鶻鶵鷎鶹鶺鶬鷈鶱鶭鷌鶳鷍鶲鹺麜黫黮黭鼛鼘鼚鼱齎齥齤龒亹囆囅囋奱孋孌巕巑廲攡攠攦攢欋欈欉氍灕灖灗灒爞爟犩獿瓘瓕瓙瓗癭皭礵禴穰穱籗籜籙籛籚"],["f740","糴糱纑罏羇臞艫蘴蘵蘳蘬蘲蘶蠬蠨蠦蠪蠥襱覿覾觻譾讄讂讆讅譿贕躕躔躚躒躐躖躗轠轢酇鑌鑐鑊鑋鑏鑇鑅鑈鑉鑆霿韣顪顩飋饔饛驎驓驔驌驏驈驊"],["f7a1","驉驒驐髐鬙鬫鬻魖魕鱆鱈鰿鱄鰹鰳鱁鰼鰷鰴鰲鰽鰶鷛鷒鷞鷚鷋鷐鷜鷑鷟鷩鷙鷘鷖鷵鷕鷝麶黰鼵鼳鼲齂齫龕龢儽劙壨壧奲孍巘蠯彏戁戃戄攩攥斖曫欑欒欏毊灛灚爢玂玁玃癰矔籧籦纕艬蘺虀蘹蘼蘱蘻蘾蠰蠲蠮蠳襶襴襳觾"],["f840","讌讎讋讈豅贙躘轤轣醼鑢鑕鑝鑗鑞韄韅頀驖驙鬞鬟鬠鱒鱘鱐鱊鱍鱋鱕鱙鱌鱎鷻鷷鷯鷣鷫鷸鷤鷶鷡鷮鷦鷲鷰鷢鷬鷴鷳鷨鷭黂黐黲黳鼆鼜鼸鼷鼶齃齏"],["f8a1","齱齰齮齯囓囍孎屭攭曭曮欓灟灡灝灠爣瓛瓥矕礸禷禶籪纗羉艭虃蠸蠷蠵衋讔讕躞躟躠躝醾醽釂鑫鑨鑩雥靆靃靇韇韥驞髕魙鱣鱧鱦鱢鱞鱠鸂鷾鸇鸃鸆鸅鸀鸁鸉鷿鷽鸄麠鼞齆齴齵齶囔攮斸欘欙欗欚灢爦犪矘矙礹籩籫糶纚"],["f940","纘纛纙臠臡虆虇虈襹襺襼襻觿讘讙躥躤躣鑮鑭鑯鑱鑳靉顲饟鱨鱮鱭鸋鸍鸐鸏鸒鸑麡黵鼉齇齸齻齺齹圞灦籯蠼趲躦釃鑴鑸鑶鑵驠鱴鱳鱱鱵鸔鸓黶鼊"],["f9a1","龤灨灥糷虪蠾蠽蠿讞貜躩軉靋顳顴飌饡馫驤驦驧鬤鸕鸗齈戇欞爧虌躨钂钀钁驩驨鬮鸙爩虋讟钃鱹麷癵驫鱺鸝灩灪麤齾齉龘碁銹裏墻恒粧嫺╔╦╗╠╬╣╚╩╝╒╤╕╞╪╡╘╧╛╓╥╖╟╫╢╙╨╜║═╭╮╰╯▓"]]},function(n,t,r){(function(){var t,u,l;function e(t,e,n){this.type=t,this.length=e,this.lengthType=null!=n?n:"count"}u=r(48).Number,l=r(26),e.prototype.decode=function(t,e){var n,r,i,o,a,s;if(i=t.pos,o=[],n=e,null!=this.length&&(r=l.resolveLength(this.length,t,e)),this.length instanceof u&&(Object.defineProperties(o,{parent:{value:e},_startOffset:{value:i},_currentOffset:{value:0,writable:!0},_length:{value:r}}),n=o),null==r||"bytes"===this.lengthType)for(a=null!=r?t.pos+r:(null!=e?e._length:void 0)?e._startOffset+e._length:t.length;t.pos<a;)o.push(this.type.decode(t,n));else for(s=0;s<r;s+=1)o.push(this.type.decode(t,n));return o},e.prototype.size=function(t,e){var n,r,i,o;if(!t)return this.type.size(null,e)*l.resolveLength(this.length,null,e);for(r=0,this.length instanceof u&&(r+=this.length.size(),e={parent:e}),i=0,o=t.length;i<o;i++)n=t[i],r+=this.type.size(n,e);return r},e.prototype.encode=function(t,e,n){var r,i,o,a,s,l;for(r=n,this.length instanceof u&&((r={pointers:[],startOffset:t.pos,parent:n}).pointerOffset=t.pos+this.size(e,r),this.length.encode(t,e.length)),s=0,l=e.length;s<l;s++)o=e[s],this.type.encode(t,o,r);if(this.length instanceof u)for(i=0;i<r.pointers.length;)(a=r.pointers[i++]).type.encode(t,a.val)},t=e,n.exports=t}).call(this)},function(n,t,r){(function(){var t,a;function e(t){this.fields=null!=t?t:{}}a=r(26),e.prototype.decode=function(t,e,n){var r,i;return null==n&&(n=0),r=this._setup(t,e,n),this._parseFields(t,r,this.fields),null!=(i=this.process)&&i.call(r,t),r},e.prototype._setup=function(t,e,n){var r;return r={},Object.defineProperties(r,{parent:{value:e},_startOffset:{value:t.pos},_currentOffset:{value:0,writable:!0},_length:{value:n}}),r},e.prototype._parseFields=function(t,e,n){var r,i,o;for(r in n)void 0!==(o="function"==typeof(i=n[r])?i.call(e,e):i.decode(t,e))&&(o instanceof a.PropertyDescriptor?Object.defineProperty(e,r,o):e[r]=o),e._currentOffset=t.pos-e._startOffset},e.prototype.size=function(t,e,n){var r,i,o,a,s;for(i in null==t&&(t={}),null==n&&(n=!0),r={parent:e,val:t,pointerSize:0},o=0,s=this.fields)null!=(a=s[i]).size&&(o+=a.size(t[i],r));return n&&(o+=r.pointerSize),o},e.prototype.encode=function(t,e,n){var r,i,o,a,s,l,u;for(o in null!=(l=this.preEncode)&&l.call(e,t),(r={pointers:[],startOffset:t.pos,parent:n,val:e,pointerSize:0}).pointerOffset=t.pos+this.size(e,r,!1),u=this.fields)null!=(s=u[o]).encode&&s.encode(t,e[o],r);for(i=0;i<r.pointers.length;)(a=r.pointers[i++]).type.encode(t,a.val,a.parent)},t=e,n.exports=t}).call(this)},function(t,e,n){t.exports=!n(14)&&!n(37)(function(){return 7!=Object.defineProperty(n(165)("div"),"a",{get:function(){return 7}}).a})},function(t,e,n){var r=n(20),i=n(21).document,o=r(i)&&r(i.createElement);t.exports=function(t){return o?i.createElement(t):{}}},function(t,e){t.exports=function(t){if("function"!=typeof t)throw TypeError(t+" is not a function!");return t}},function(t,e){t.exports=function(t,e){return{value:e,done:!!t}}},function(t,e,n){t.exports=n(27)},function(t,e,n){var a=n(17),s=n(28),l=n(61);t.exports=n(14)?Object.defineProperties:function(t,e){s(t);for(var n,r=l(e),i=r.length,o=0;o<i;)a.f(t,n=r[o++],e[n]);return t}},function(t,e,n){var a=n(36),s=n(35),l=n(335)(!1),u=n(114)("IE_PROTO");t.exports=function(t,e){var n,r=s(t),i=0,o=[];for(n in r)n!=u&&a(r,n)&&o.push(n);for(;e.length>i;)a(r,n=e[i++])&&(~l(o,n)||o.push(n));return o}},function(t,e,n){var r=n(113),i=Math.max,o=Math.min;t.exports=function(t,e){return(t=r(t))<0?i(t+e,0):o(t,e)}},function(t,e,n){t.exports={default:n(343),__esModule:!0}},function(t,e,n){var r=n(106);t.exports=Array.isArray||function(t){return"Array"==r(t)}},function(t,e,n){var r=n(170),i=n(116).concat("length","prototype");e.f=Object.getOwnPropertyNames||function(t){return r(t,i)}},function(t,e,n){"use strict";e.__esModule=!0,e.default=function(t,e){if(!(t instanceof e))throw new TypeError("Cannot call a class as a function")}},function(t,e,n){"use strict";e.__esModule=!0;var r,i=n(124),o=(r=i)&&r.__esModule?r:{default:r};function a(t,e){for(var n=0;n<e.length;n++){var r=e[n];r.enumerable=r.enumerable||!1,r.configurable=!0,"value"in r&&(r.writable=!0),(0,o.default)(t,r.key,r)}}e.default=function(t,e,n){return e&&a(t.prototype,e),n&&a(t,n),t}},function(t,e,n){"use strict";function a(t,e){var n,r=p(e);if("F"!==r)return t._i[r];for(n=t._f;n;n=n.n)if(n.k==e)return n}var s=n(17).f,l=n(75),u=n(178),c=n(38),f=n(179),h=n(80),r=n(112),i=n(167),o=n(359),d=n(14),p=n(79).fastKey,g=n(125),v=d?"_s":"size";t.exports={getConstructor:function(t,o,n,r){var i=t(function(t,e){f(t,i,o,"_i"),t._t=o,t._i=l(null),t._f=void 0,t._l=void 0,t[v]=0,null!=e&&h(e,n,t[r],t)});return u(i.prototype,{clear:function(){for(var t=g(this,o),e=t._i,n=t._f;n;n=n.n)n.r=!0,n.p&&(n.p=n.p.n=void 0),delete e[n.i];t._f=t._l=void 0,t[v]=0},delete:function(t){var e=g(this,o),n=a(e,t);if(n){var r=n.n,i=n.p;delete e._i[n.i],n.r=!0,i&&(i.n=r),r&&(r.p=i),e._f==n&&(e._f=r),e._l==n&&(e._l=i),e[v]--}return!!n},forEach:function(t,e){g(this,o);for(var n,r=c(t,1<arguments.length?e:void 0,3);n=n?n.n:this._f;)for(r(n.v,n.k,this);n&&n.r;)n=n.p},has:function(t){return!!a(g(this,o),t)}}),d&&s(i.prototype,"size",{get:function(){return g(this,o)[v]}}),i},def:function(t,e,n){var r,i,o=a(t,e);return o?o.v=n:(t._l=o={i:i=p(e,!0),k:e,v:n,p:r=t._l,n:void 0,r:!1},t._f||(t._f=o),r&&(r.n=o),t[v]++,"F"!==i&&(t._i[i]=o)),t},getEntry:a,setStrong:function(t,n,e){r(t,n,function(t,e){this._t=g(t,n),this._k=e,this._l=void 0},function(){for(var t=this._k,e=this._l;e&&e.r;)e=e.p;return this._t&&(this._l=e=e?e.n:this._t._f)?i(0,"keys"==t?e.k:"values"==t?e.v:[e.k,e.v]):(this._t=void 0,i(1))},e?"entries":"values",!e,!0),o(n)}}},function(t,e,n){var i=n(27);t.exports=function(t,e,n){for(var r in e)n&&t[r]?t[r]=e[r]:i(t,r,e[r]);return t}},function(t,e){t.exports=function(t,e,n,r){if(!(t instanceof e)||void 0!==r&&r in t)throw TypeError(n+": incorrect invocation!");return t}},function(t,e,n){var o=n(28);t.exports=function(e,t,n,r){try{return r?t(o(n)[0],n[1]):t(n)}catch(t){var i=e.return;throw void 0!==i&&o(i.call(e)),t}}},function(t,e,n){var r=n(49),i=n(12)("iterator"),o=Array.prototype;t.exports=function(t){return void 0!==t&&(r.Array===t||o[i]===t)}},function(t,e,n){"use strict";var f=n(21),h=n(7),d=n(79),p=n(37),g=n(27),v=n(178),y=n(80),b=n(179),m=n(20),w=n(78),x=n(17).f,_=n(360)(0),k=n(14);t.exports=function(n,t,e,r,i,o){var a=f[n],s=a,l=i?"set":"add",u=s&&s.prototype,c={};return k&&"function"==typeof s&&(o||u.forEach&&!p(function(){(new s).entries().next()}))?(s=t(function(t,e){b(t,s,n,"_c"),t._c=new a,null!=e&&y(e,i,t[l],t)}),_("add,clear,delete,forEach,get,has,set,keys,values,entries,toJSON".split(","),function(r){var i="add"==r||"set"==r;r in u&&(!o||"clear"!=r)&&g(s.prototype,r,function(t,e){if(b(this,s,r),!i&&o&&!m(t))return"get"==r&&void 0;var n=this._c[r](0===t?0:t,e);return i?this:n})}),o||x(s.prototype,"size",{get:function(){return this._c.size}})):(s=r.getConstructor(t,n,i,l),v(s.prototype,e),d.NEED=!0),w(s,n),c[n]=s,h(h.G+h.W+h.F,c),o||r.setStrong(s,n,i),s}},function(t,e,n){var r=n(118),i=n(364);t.exports=function(t){return function(){if(r(this)!=t)throw TypeError(t+"#toJSON isn't generic");return i(this)}}},function(t,e,n){"use strict";var r=n(7);t.exports=function(t){r(r.S,t,{of:function(){for(var t=arguments.length,e=new Array(t);t--;)e[t]=arguments[t];return new this(e)}})}},function(t,e,n){"use strict";var r=n(7),l=n(166),u=n(38),c=n(80);t.exports=function(t){r(r.S,t,{from:function(t,e,n){var r,i,o,a,s=e;return l(this),(r=void 0!==s)&&l(s),null==t?new this:(i=[],r?(o=0,a=u(s,n,2),c(t,!1,function(t){i.push(a(t,o++))})):c(t,!1,i.push,i),new this(i))}})}},function(t,e){var l=0,o=-3;function n(){this.table=new Uint16Array(16),this.trans=new Uint16Array(288)}function a(t,e){this.source=t,this.sourceIndex=0,this.tag=0,this.bitcount=0,this.dest=e,this.destLen=0,this.ltree=new n,this.dtree=new n}var s=new n,u=new n,c=new Uint8Array(30),f=new Uint16Array(30),h=new Uint8Array(30),d=new Uint16Array(30),p=new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]),g=new n,v=new Uint8Array(320);function r(t,e,n,r){var i,o;for(i=0;i<n;++i)t[i]=0;for(i=0;i<30-n;++i)t[i+n]=i/n|0;for(o=r,i=0;i<30;++i)e[i]=o,o+=1<<t[i]}var y=new Uint16Array(16);function b(t,e,n,r){var i,o;for(i=0;i<16;++i)t.table[i]=0;for(i=0;i<r;++i)t.table[e[n+i]]++;for(i=o=t.table[0]=0;i<16;++i)y[i]=o,o+=t.table[i];for(i=0;i<r;++i)e[n+i]&&(t.trans[y[e[n+i]]++]=i)}function m(t){t.bitcount--||(t.tag=t.source[t.sourceIndex++],t.bitcount=7);var e=1&t.tag;return t.tag>>>=1,e}function w(t,e,n){if(!e)return n;for(;t.bitcount<24;)t.tag|=t.source[t.sourceIndex++]<<t.bitcount,t.bitcount+=8;var r=t.tag&65535>>>16-e;return t.tag>>>=e,t.bitcount-=e,r+n}function x(t,e){for(;t.bitcount<24;)t.tag|=t.source[t.sourceIndex++]<<t.bitcount,t.bitcount+=8;for(var n=0,r=0,i=0,o=t.tag;r=2*r+(1&o),o>>>=1,++i,n+=e.table[i],0<=(r-=e.table[i]););return t.tag=o,t.bitcount-=i,e.trans[n+r]}function _(t,e,n){var r,i,o,a,s,l;for(r=w(t,5,257),i=w(t,5,1),o=w(t,4,4),a=0;a<19;++a)v[a]=0;for(a=0;a<o;++a){var u=w(t,3,0);v[p[a]]=u}for(b(g,v,0,19),s=0;s<r+i;){var c=x(t,g);switch(c){case 16:var f=v[s-1];for(l=w(t,2,3);l;--l)v[s++]=f;break;case 17:for(l=w(t,3,3);l;--l)v[s++]=0;break;case 18:for(l=w(t,7,11);l;--l)v[s++]=0;break;default:v[s++]=c}}b(e,v,0,r),b(n,v,r,i)}function k(t,e,n){for(;;){var r,i,o,a,s=x(t,e);if(256===s)return l;if(s<256)t.dest[t.destLen++]=s;else for(r=w(t,c[s-=257],f[s]),i=x(t,n),a=o=t.destLen-w(t,h[i],d[i]);a<o+r;++a)t.dest[t.destLen++]=t.dest[a]}}function S(t){for(var e,n;8<t.bitcount;)t.sourceIndex--,t.bitcount-=8;if((e=256*(e=t.source[t.sourceIndex+1])+t.source[t.sourceIndex])!==(65535&~(256*t.source[t.sourceIndex+3]+t.source[t.sourceIndex+2])))return o;for(t.sourceIndex+=4,n=e;n;--n)t.dest[t.destLen++]=t.source[t.sourceIndex++];return t.bitcount=0,l}!function(t,e){var n;for(n=0;n<7;++n)t.table[n]=0;for(t.table[7]=24,t.table[8]=152,t.table[9]=112,n=0;n<24;++n)t.trans[n]=256+n;for(n=0;n<144;++n)t.trans[24+n]=n;for(n=0;n<8;++n)t.trans[168+n]=280+n;for(n=0;n<112;++n)t.trans[176+n]=144+n;for(n=0;n<5;++n)e.table[n]=0;for(e.table[5]=32,n=0;n<32;++n)e.trans[n]=n}(s,u),r(c,f,4,3),r(h,d,2,1),c[28]=0,f[28]=258,t.exports=function(t,e){var n,r,i=new a(t,e);do{switch(n=m(i),w(i,2,0)){case 0:r=S(i);break;case 1:r=k(i,s,u);break;case 2:_(i,i.ltree,i.dtree),r=k(i,i.ltree,i.dtree);break;default:r=o}if(r!==l)throw new Error("Data error")}while(!n);return i.destLen<i.dest.length?"function"==typeof i.dest.slice?i.dest.slice(0,i.destLen):i.dest.subarray(0,i.destLen):i.dest}},function(t,e,n){var o=n(188).BrotliInput,a=n(188).BrotliOutput,mt=n(416),wt=n(189),xt=n(190).HuffmanCode,m=n(190).BrotliBuildHuffmanTable,_t=n(420),kt=n(421),St=n(422),w=8,x=16,Ct=256,At=704,Pt=26,Et=6,Ot=2,_=8,i=255,Tt=1080,k=18,S=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),It=16,Bt=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),Lt=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),r=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function Rt(t){var e;return 0===t.readBits(1)?16:0<(e=t.readBits(3))?17+e:0<(e=t.readBits(3))?8+e:17}function Mt(t){if(t.readBits(1)){var e=t.readBits(3);return 0===e?1:t.readBits(e)+(1<<e)}return 0}function s(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function Ft(t){var e,n,r,i=new s;if(i.input_end=t.readBits(1),i.input_end&&t.readBits(1))return i;if(7===(e=t.readBits(2)+4)){if(i.is_metadata=!0,0!==t.readBits(1))throw new Error("Invalid reserved bit");if(0===(n=t.readBits(2)))return i;for(r=0;r<n;r++){var o=t.readBits(8);if(r+1===n&&1<n&&0===o)throw new Error("Invalid size byte");i.meta_block_length|=o<<8*r}}else for(r=0;r<e;++r){var a=t.readBits(4);if(r+1===e&&4<e&&0===a)throw new Error("Invalid size nibble");i.meta_block_length|=a<<4*r}return++i.meta_block_length,i.input_end||i.is_metadata||(i.is_uncompressed=t.readBits(1)),i}function Dt(t,e,n){var r;return n.fillBitWindow(),0<(r=t[e+=n.val_>>>n.bit_pos_&i].bits-_)&&(n.bit_pos_+=_,e+=t[e].value,e+=n.val_>>>n.bit_pos_&(1<<r)-1),n.bit_pos_+=t[e].bits,t[e].value}function zt(t,e,n,r){var i,o,a=new Uint8Array(t);if(r.readMoreInput(),1===(o=r.readBits(2))){for(var s=t-1,l=0,u=new Int32Array(4),c=r.readBits(2)+1;s;)s>>=1,++l;for(f=0;f<c;++f)u[f]=r.readBits(l)%t,a[u[f]]=2;switch(a[u[0]]=1,c){case 1:break;case 3:if(u[0]===u[1]||u[0]===u[2]||u[1]===u[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(u[0]===u[1])throw new Error("[ReadHuffmanCode] invalid symbols");a[u[1]]=1;break;case 4:if(u[0]===u[1]||u[0]===u[2]||u[0]===u[3]||u[1]===u[2]||u[1]===u[3]||u[2]===u[3])throw new Error("[ReadHuffmanCode] invalid symbols");r.readBits(1)?(a[u[2]]=3,a[u[3]]=3):a[u[0]]=2}}else{var f,h=new Uint8Array(k),d=32,p=0,g=[new xt(2,0),new xt(2,4),new xt(2,3),new xt(3,2),new xt(2,0),new xt(2,4),new xt(2,3),new xt(4,1),new xt(2,0),new xt(2,4),new xt(2,3),new xt(3,2),new xt(2,0),new xt(2,4),new xt(2,3),new xt(4,5)];for(f=o;f<k&&0<d;++f){var v,y=S[f],b=0;r.fillBitWindow(),b+=r.val_>>>r.bit_pos_&15,r.bit_pos_+=g[b].bits,v=g[b].value,0!==(h[y]=v)&&(d-=32>>v,++p)}if(1!==p&&0!==d)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function(t,e,n,r){for(var i=0,o=w,a=0,s=0,l=32768,u=[],c=0;c<32;c++)u.push(new xt(0,0));for(m(u,0,5,t,k);i<e&&0<l;){var f,h=0;if(r.readMoreInput(),r.fillBitWindow(),h+=r.val_>>>r.bit_pos_&31,r.bit_pos_+=u[h].bits,(f=255&u[h].value)<x)(a=0)!=(n[i++]=f)&&(l-=32768>>(o=f));else{var d,p,g=f-14,v=0;if(f===x&&(v=o),s!==v&&(a=0,s=v),0<(d=a)&&(a-=2,a<<=g),e<i+(p=(a+=r.readBits(g)+3)-d))throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var y=0;y<p;y++)n[i+y]=s;i+=p,0!==s&&(l-=p<<15-s)}}if(0!==l)throw new Error("[ReadHuffmanCodeLengths] space = "+l);for(;i<e;i++)n[i]=0}(h,t,a,r)}if(0===(i=m(e,n,_,a,t)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return i}function Nt(t,e,n){var r,i;return r=Dt(t,e,n),i=kt.kBlockLengthPrefixCode[r].nbits,kt.kBlockLengthPrefixCode[r].offset+n.readBits(i)}function c(t,e){for(var n=t[e],r=e;r;--r)t[r]=t[r-1];t[0]=n}function Ut(t,e){this.alphabet_size=t,this.num_htrees=e,this.codes=new Array(e+e*r[t+31>>>5]),this.htrees=new Uint32Array(e)}function Wt(t,e){var n,r,i={num_htrees:null,context_map:null},o=0;e.readMoreInput();var a=i.num_htrees=Mt(e)+1,s=i.context_map=new Uint8Array(t);if(a<=1)return i;for(e.readBits(1)&&(o=e.readBits(4)+1),n=[],r=0;r<Tt;r++)n[r]=new xt(0,0);for(zt(a+o,n,0,e),r=0;r<t;){var l;if(e.readMoreInput(),0===(l=Dt(n,0,e)))s[r]=0,++r;else if(l<=o)for(var u=1+(1<<l)+e.readBits(l);--u;){if(t<=r)throw new Error("[DecodeContextMap] i >= context_map_size");s[r]=0,++r}else s[r]=l-o,++r}return e.readBits(1)&&function(t,e){var n,r=new Uint8Array(256);for(n=0;n<256;++n)r[n]=n;for(n=0;n<e;++n){var i=t[n];t[n]=r[i],i&&c(r,i)}}(s,t),i}function jt(t,e,n,r,i,o,a){var s,l=2*n,u=n,c=Dt(e,n*Tt,a);t<=(s=0===c?i[l+(1&o[u])]:1===c?i[l+(o[u]-1&1)]+1:c-2)&&(s-=t),r[n]=s,i[l+(1&o[u])]=s,++o[u]}function Gt(t,e,n,r,i,o){var a,s=i+1,l=n&i,u=o.pos_&mt.IBUF_MASK;if(e<8||o.bit_pos_+(e<<3)<o.bit_end_pos_)for(;0<e--;)o.readMoreInput(),r[l++]=o.readBits(8),l===s&&(t.write(r,s),l=0);else{if(o.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;o.bit_pos_<32;)r[l]=o.val_>>>o.bit_pos_,o.bit_pos_+=8,++l,--e;if(u+(a=o.bit_end_pos_-o.bit_pos_>>3)>mt.IBUF_MASK){for(var c=mt.IBUF_MASK+1-u,f=0;f<c;f++)r[l+f]=o.buf_[u+f];a-=c,l+=c,e-=c,u=0}for(f=0;f<a;f++)r[l+f]=o.buf_[u+f];if(e-=a,s<=(l+=a)){t.write(r,s),l-=s;for(f=0;f<l;f++)r[f]=r[s+f]}for(;s<=l+e;){if(a=s-l,o.input_.read(r,l,a)<a)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");t.write(r,s),e-=a,l=0}if(o.input_.read(r,l,e)<e)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");o.reset()}}function l(t){var e=new o(t),n=new mt(e);return Rt(n),Ft(n).meta_block_length}function u(t,e){var n,r,i,o,a,s,l,u,c,f,h,d,p,g,v,y=0,b=0,m=0,w=[16,15,11,4],x=0,_=0,k=0,S=[new Ut(0,0),new Ut(0,0),new Ut(0,0)],C=128+mt.READ_SIZE;i=(1<<(r=Rt(f=new mt(t))))-16,a=(o=1<<r)-1,s=new Uint8Array(o+C+wt.maxDictionaryWordLength),l=o,u=[],c=[];for(var A=0;A<3*Tt;A++)u[A]=new xt(0,0),c[A]=new xt(0,0);for(;!b;){var P,E,O,T,I,B,L,R,M,F,D,z=0,N=[1<<28,1<<28,1<<28],U=[0],W=[1,1,1],j=[0,1,0,1,0,1],G=[0],H=null,V=null,q=0,Z=null,Y=0,X=0,K=0;for(n=0;n<3;++n)S[n].codes=null,S[n].htrees=null;f.readMoreInput();var J=Ft(f);if(y+(z=J.meta_block_length)>e.buffer.length){var Q=new Uint8Array(y+z);Q.set(e.buffer),e.buffer=Q}if(b=J.input_end,P=J.is_uncompressed,J.is_metadata)for(void 0,v=(g=f).bit_pos_+7&-8,g.readBits(v-g.bit_pos_);0<z;--z)f.readMoreInput(),f.readBits(8);else if(0!==z)if(P)f.bit_pos_=f.bit_pos_+7&-8,Gt(e,z,y,s,a,f),y+=z;else{for(n=0;n<3;++n)W[n]=Mt(f)+1,2<=W[n]&&(zt(W[n]+2,u,n*Tt,f),zt(Pt,c,n*Tt,f),N[n]=Nt(c,n*Tt,f),G[n]=1);for(f.readMoreInput(),T=(1<<(E=f.readBits(2)))-1,I=(O=It+(f.readBits(4)<<E))+(48<<E),H=new Uint8Array(W[0]),n=0;n<W[0];++n)f.readMoreInput(),H[n]=f.readBits(2)<<1;var $=Wt(W[0]<<Et,f);L=$.num_htrees,B=$.context_map;var tt=Wt(W[2]<<Ot,f);for(M=tt.num_htrees,R=tt.context_map,S[0]=new Ut(Ct,L),S[1]=new Ut(At,W[1]),S[2]=new Ut(I,M),n=0;n<3;++n)S[n].decode(f);for(F=H[U[Z=V=0]],X=_t.lookupOffsets[F],K=_t.lookupOffsets[F+1],D=S[1].htrees[0];0<z;){var et,nt,rt,it,ot,at,st,lt,ut,ct,ft,ht;for(f.readMoreInput(),0===N[1]&&(jt(W[1],u,1,U,j,G,f),N[1]=Nt(c,Tt,f),D=S[1].htrees[U[1]]),--N[1],st=2<=(nt=(et=Dt(S[1].codes,D,f))>>6)?(nt-=2,-1):0,rt=kt.kInsertRangeLut[nt]+(et>>3&7),it=kt.kCopyRangeLut[nt]+(7&et),ot=kt.kInsertLengthPrefixCode[rt].offset+f.readBits(kt.kInsertLengthPrefixCode[rt].nbits),at=kt.kCopyLengthPrefixCode[it].offset+f.readBits(kt.kCopyLengthPrefixCode[it].nbits),_=s[y-1&a],k=s[y-2&a],ut=0;ut<ot;++ut)f.readMoreInput(),0===N[0]&&(jt(W[0],u,0,U,j,G,f),N[0]=Nt(c,0,f),V=U[0]<<Et,F=H[U[0]],X=_t.lookupOffsets[F],K=_t.lookupOffsets[F+1]),q=B[V+(_t.lookup[X+_]|_t.lookup[K+k])],--N[0],k=_,_=Dt(S[0].codes,S[0].htrees[q],f),s[y&a]=_,(y&a)==a&&e.write(s,o),++y;if((z-=ot)<=0)break;if(st<0)if(f.readMoreInput(),0===N[2]&&(jt(W[2],u,2,U,j,G,f),N[2]=Nt(c,2*Tt,f),Z=U[2]<<Ot),--N[2],Y=R[Z+(255&(4<at?3:at-2))],O<=(st=Dt(S[2].codes,S[2].htrees[Y],f)))ht=(st-=O)&T,st=O+((dt=(2+(1&(st>>=E))<<(ft=1+(st>>1)))-4)+f.readBits(ft)<<E)+ht;if(d=w,p=x,(lt=(h=st)<It?(p+=Bt[h],d[p&=3]+Lt[h]):h-It+1)<0)throw new Error("[BrotliDecompress] invalid distance");if(ct=y&a,(m=y<i&&m!==i?y:i)<lt){if(!(at>=wt.minDictionaryWordLength&&at<=wt.maxDictionaryWordLength))throw new Error("Invalid backward reference. pos: "+y+" distance: "+lt+" len: "+at+" bytes left: "+z);var dt=wt.offsetsByLength[at],pt=lt-m-1,gt=wt.sizeBitsByLength[at],vt=pt>>gt;if(dt+=(pt&(1<<gt)-1)*at,!(vt<St.kNumTransforms))throw new Error("Invalid backward reference. pos: "+y+" distance: "+lt+" len: "+at+" bytes left: "+z);var yt=St.transformDictionaryWord(s,ct,dt,at,vt);if(y+=yt,z-=yt,l<=(ct+=yt)){e.write(s,o);for(var bt=0;bt<ct-l;bt++)s[bt]=s[l+bt]}}else{if(0<st&&(w[3&x]=lt,++x),z<at)throw new Error("Invalid backward reference. pos: "+y+" distance: "+lt+" len: "+at+" bytes left: "+z);for(ut=0;ut<at;++ut)s[y&a]=s[y-lt&a],(y&a)==a&&e.write(s,o),++y,--z}_=s[y-1&a],k=s[y-2&a]}y&=1073741823}}e.write(s,y&a)}Ut.prototype.decode=function(t){var e,n=0;for(e=0;e<this.num_htrees;++e)this.htrees[e]=n,n+=zt(this.alphabet_size,this.codes,n,t)},e.BrotliDecompressedSize=l,e.BrotliDecompressBuffer=function(t,e){var n=new o(t);null==e&&(e=l(t));var r=new Uint8Array(e),i=new a(r);return u(n,i),i.pos<i.buffer.length&&(i.buffer=i.buffer.subarray(0,i.pos)),i.buffer},e.BrotliDecompress=u,wt.init()},function(t,e){function n(t){this.buffer=t,this.pos=0}function r(t){this.buffer=t,this.pos=0}n.prototype.read=function(t,e,n){this.pos+n>this.buffer.length&&(n=this.buffer.length-this.pos);for(var r=0;r<n;r++)t[e+r]=this.buffer[this.pos+r];return this.pos+=n,n},e.BrotliInput=n,r.prototype.write=function(t,e){if(this.pos+e>this.buffer.length)throw new Error("Output buffer is not large enough");return this.buffer.set(t.subarray(0,e),this.pos),this.pos+=e,e},e.BrotliOutput=r},function(t,e,n){var r=n(417);e.init=function(){e.dictionary=r.init()},e.offsetsByLength=new Uint32Array([0,0,0,0,0,4096,9216,21504,35840,44032,53248,63488,74752,87040,93696,100864,104704,106752,108928,113536,115968,118528,119872,121280,122016]),e.sizeBitsByLength=new Uint8Array([0,0,0,0,10,10,11,11,10,10,10,10,10,9,9,8,7,7,8,7,7,6,6,5,5]),e.minDictionaryWordLength=4,e.maxDictionaryWordLength=24},function(t,e){function b(t,e){this.bits=t,this.value=e}e.HuffmanCode=b;var m=15;function w(t,e){for(var n=1<<e-1;t&n;)n>>=1;return(t&n-1)+n}function x(t,e,n,r,i){for(;t[e+(r-=n)]=new b(i.bits,i.value),0<r;);}function _(t,e,n){for(var r=1<<e-n;e<m&&!((r-=t[e])<=0);)++e,r<<=1;return e-n}e.BrotliBuildHuffmanTable=function(t,e,n,r,i){var o,a,s,l,u,c,f,h,d,p,g=e,v=new Int32Array(m+1),y=new Int32Array(m+1);for(p=new Int32Array(i),a=0;a<i;a++)v[r[a]]++;for(y[1]=0,o=1;o<m;o++)y[o+1]=y[o]+v[o];for(a=0;a<i;a++)0!==r[a]&&(p[y[r[a]]++]=a);if(d=h=1<<(f=n),1===y[m]){for(s=0;s<d;++s)t[e+s]=new b(0,65535&p[0]);return d}for(a=s=0,o=1,l=2;o<=n;++o,l<<=1)for(;0<v[o];--v[o])x(t,e+s,l,h,new b(255&o,65535&p[a++])),s=w(s,o);for(c=d-1,u=-1,o=n+1,l=2;o<=m;++o,l<<=1)for(;0<v[o];--v[o])(s&c)!==u&&(e+=h,d+=h=1<<(f=_(v,o,n)),t[g+(u=s&c)]=new b(f+n&255,e-g-u&65535)),x(t,e+(s>>n),l,h,new b(o-n&255,65535&p[a++])),s=w(s,o);return d}},function(I,t,B){(function(){var e,n,r,i,o,a,s,l,u,c,f,h,t,d,p,g,v,y,b,m,w,x,_,k,S,C,A,P,E,O;function T(t){this.string=t,this.pos=0,this.lastPos=0,this.curClass=null,this.nextClass=null}b=B(126),x=B(423),C=B(424),C.BK,u=C.CR,C.LF,C.NL,o=C.CB,r=C.BA,C.SP,m=C.WJ,y=C.SP,i=C.BK,h=C.LF,d=C.NL,e=C.AI,n=C.AL,g=C.SA,v=C.SG,w=C.XX,s=C.CJ,C.ID,p=C.NS,C.characterClasses,A=B(425),c=A.DI_BRK,f=A.IN_BRK,a=A.CI_BRK,l=A.CP_BRK,A.PR_BRK,S=A.pairTable,k=x.toByteArray("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"),_=new b(k),T.prototype.nextCodePoint=function(){var t,e;return t=this.string.charCodeAt(this.pos++),e=this.string.charCodeAt(this.pos),55296<=t&&t<=56319&&56320<=e&&e<=57343?(this.pos++,1024*(t-55296)+(e-56320)+65536):t},E=function(t){switch(t){case e:return n;case g:case v:case w:return n;case s:return p;default:return t}},O=function(t){switch(t){case h:case d:return i;case o:return r;case y:return m;default:return t}},T.prototype.nextCharClass=function(t){return null==t&&(t=!1),E(_.get(this.nextCodePoint()))},P=function(t,e){this.position=t,this.required=null!=e&&e},T.prototype.nextBreak=function(){var t,e,n;for(null==this.curClass&&(this.curClass=O(this.nextCharClass()));this.pos<this.string.length;){if(this.lastPos=this.pos,e=this.nextClass,this.nextClass=this.nextCharClass(),this.curClass===i||this.curClass===u&&this.nextClass!==h)return this.curClass=O(E(this.nextClass)),new P(this.lastPos,!0);if(null==(t=function(){switch(this.nextClass){case y:return this.curClass;case i:case h:case d:return i;case u:return u;case o:return r}}.call(this))){switch(n=!1,S[this.curClass][this.nextClass]){case c:n=!0;break;case f:n=e===y;break;case a:if(!(n=e===y))continue;break;case l:if(e!==y)continue}if(this.curClass=this.nextClass,n)return new P(this.lastPos)}else if(this.curClass=t,this.nextClass===o)return new P(this.lastPos)}if(this.pos>=this.string.length)return this.lastPos<this.string.length?(this.lastPos=this.string.length,new P(this.string.length)):null},t=T,I.exports=t}).call(this)},function(t,e,n){"use strict";function r(){this.events={}}r.prototype.startTracking=function(t,e){var n=this.events[t]||(this.events[t]=[]);n.indexOf(e)<0&&n.push(e)},r.prototype.stopTracking=function(t,e){var n=this.events[t];if(n){var r=n.indexOf(e);0<=r&&n.splice(r,1)}},r.prototype.emit=function(t){var e=Array.prototype.slice.call(arguments,1),n=this.events[t];n&&n.forEach(function(t){t.apply(this,e)})},r.prototype.auto=function(t,e,n){this.startTracking(t,e),n(),this.stopTracking(t,e)},t.exports=r},function(t,e,n){"use strict";var a=n(0).isString,s=n(0).isArray,l=n(0).isUndefined,u=n(0).isNull;function r(t,e){this.defaultStyle=e||{},this.styleDictionary=t,this.styleOverrides=[]}r.prototype.clone=function(){var e=new r(this.styleDictionary,this.defaultStyle);return this.styleOverrides.forEach(function(t){e.styleOverrides.push(t)}),e},r.prototype.push=function(t){this.styleOverrides.push(t)},r.prototype.pop=function(t){for(t=t||1;0<t--;)this.styleOverrides.pop()},r.prototype.autopush=function(e){if(a(e))return 0;var t=[];e.style&&(t=s(e.style)?e.style:[e.style]);for(var n=0,r=t.length;n<r;n++)this.push(t[n]);var i={},o=!1;return["font","fontSize","fontFeatures","bold","italics","alignment","color","columnGap","fillColor","decoration","decorationStyle","decorationColor","background","lineHeight","characterSpacing","noWrap","markerColor","leadingIndent"].forEach(function(t){l(e[t])||u(e[t])||(i[t]=e[t],o=!0)}),o&&this.push(i),t.length+(o?1:0)},r.prototype.auto=function(t,e){var n=this.autopush(t),r=e();return 0<n&&this.pop(n),r},r.prototype.getProperty=function(t){if(this.styleOverrides)for(var e=this.styleOverrides.length-1;0<=e;e--){var n=this.styleOverrides[e];if(a(n)){var r=this.styleDictionary[n];if(r&&!l(r[t])&&!u(r[t]))return r[t]}else if(!l(n[t])&&!u(n[t]))return n[t]}return this.defaultStyle&&this.defaultStyle[t]},t.exports=r},function(t,e,n){"use strict";var r=n(192),i=n(0).isString;function o(t,e){this.pages=[],this.pageMargins=e,this.x=e.left,this.availableWidth=t.width-e.left-e.right,this.availableHeight=0,this.page=-1,this.snapshots=[],this.endingCell=null,this.tracker=new r,this.backgroundLength=[],this.addPage(t)}o.prototype.beginColumnGroup=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,bottomMost:{x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page},endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth}),this.lastColumnWidth=0},o.prototype.beginColumn=function(t,e,n){var r=this.snapshots[this.snapshots.length-1];this.calculateBottomMost(r),this.endingCell=n,this.page=r.page,this.x=this.x+this.lastColumnWidth+(e||0),this.y=r.y,this.availableWidth=t,this.availableHeight=r.availableHeight,this.lastColumnWidth=t},o.prototype.calculateBottomMost=function(t){this.endingCell?(this.saveContextInEndingCell(this.endingCell),this.endingCell=null):t.bottomMost=function(t,e){var n;n=t.page>e.page?t:e.page>t.page?e:t.y>e.y?t:e;return{page:n.page,x:n.x,y:n.y,availableHeight:n.availableHeight,availableWidth:n.availableWidth}}(this,t.bottomMost)},o.prototype.markEnding=function(t){this.page=t._columnEndingContext.page,this.x=t._columnEndingContext.x,this.y=t._columnEndingContext.y,this.availableWidth=t._columnEndingContext.availableWidth,this.availableHeight=t._columnEndingContext.availableHeight,this.lastColumnWidth=t._columnEndingContext.lastColumnWidth},o.prototype.saveContextInEndingCell=function(t){t._columnEndingContext={page:this.page,x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,lastColumnWidth:this.lastColumnWidth}},o.prototype.completeColumnGroup=function(t){var e=this.snapshots.pop();this.calculateBottomMost(e),this.endingCell=null,this.x=e.x;var n=e.bottomMost.y;t&&(e.page===e.bottomMost.page?e.y+t>n&&(n=e.y+t):n+=t),this.y=n,this.page=e.bottomMost.page,this.availableWidth=e.availableWidth,this.availableHeight=e.bottomMost.availableHeight,t&&(this.availableHeight-=n-e.bottomMost.y),this.lastColumnWidth=e.lastColumnWidth},o.prototype.addMargin=function(t,e){this.x+=t,this.availableWidth-=t+(e||0)},o.prototype.moveDown=function(t){return this.y+=t,this.availableHeight-=t,0<this.availableHeight},o.prototype.initializePage=function(){this.y=this.pageMargins.top,this.availableHeight=this.getCurrentPage().pageSize.height-this.pageMargins.top-this.pageMargins.bottom,this.pageSnapshot().availableWidth=this.getCurrentPage().pageSize.width-this.pageMargins.left-this.pageMargins.right},o.prototype.pageSnapshot=function(){return this.snapshots[0]?this.snapshots[0]:this},o.prototype.moveTo=function(t,e){null!=t&&(this.x=t,this.availableWidth=this.getCurrentPage().pageSize.width-this.x-this.pageMargins.right),null!=e&&(this.y=e,this.availableHeight=this.getCurrentPage().pageSize.height-this.y-this.pageMargins.bottom)},o.prototype.moveToRelative=function(t,e){null!=t&&(this.x=this.x+t),null!=e&&(this.y=this.y+e)},o.prototype.beginDetachedBlock=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth})},o.prototype.endDetachedBlock=function(){var t=this.snapshots.pop();this.x=t.x,this.y=t.y,this.availableWidth=t.availableWidth,this.availableHeight=t.availableHeight,this.page=t.page,this.endingCell=t.endingCell,this.lastColumnWidth=t.lastColumnWidth};function l(t,e){return(e=function(t,e){return void 0===t?e:i(t)&&"landscape"===t.toLowerCase()?"landscape":"portrait"}(e,t.pageSize.orientation))!==t.pageSize.orientation?{orientation:e,width:t.pageSize.height,height:t.pageSize.width}:{orientation:t.pageSize.orientation,width:t.pageSize.width,height:t.pageSize.height}}o.prototype.moveToNextPage=function(t){var e=this.page+1,n=this.page,r=this.y,i=e>=this.pages.length;if(i){var o=this.availableWidth,a=this.getCurrentPage().pageSize.orientation,s=l(this.getCurrentPage(),t);this.addPage(s),a===s.orientation&&(this.availableWidth=o)}else this.page=e,this.initializePage();return{newPageCreated:i,prevPage:n,prevY:r,y:this.y}},o.prototype.addPage=function(t){var e={items:[],pageSize:t};return this.pages.push(e),this.backgroundLength.push(0),this.page=this.pages.length-1,this.initializePage(),this.tracker.emit("pageAdded"),e},o.prototype.getCurrentPage=function(){return this.page<0||this.page>=this.pages.length?null:this.pages[this.page]},o.prototype.getCurrentPosition=function(){var t=this.getCurrentPage().pageSize,e=t.height-this.pageMargins.top-this.pageMargins.bottom,n=t.width-this.pageMargins.left-this.pageMargins.right;return{pageNumber:this.page+1,pageOrientation:t.orientation,pageInnerHeight:e,pageInnerWidth:n,left:this.x,top:this.y,verticalRatio:(this.y-this.pageMargins.top)/e,horizontalRatio:(this.x-this.pageMargins.left)/n}},t.exports=o},function(t,e,n){"use strict";function r(t){this.maxWidth=t,this.leadingCut=0,this.trailingCut=0,this.inlineWidths=0,this.inlines=[]}r.prototype.getAscenderHeight=function(){var e=0;return this.inlines.forEach(function(t){e=Math.max(e,t.font.ascender/1e3*t.fontSize)}),e},r.prototype.hasEnoughSpaceForInline=function(t,e){if(e=e||[],0===this.inlines.length)return!0;if(this.newLineForced)return!1;var n=t.width,r=t.trailingCut||0;if(t.noNewLine)for(var i=0,o=e.length;i<o;i++){var a=e[i];if(n+=a.width,r+=a.trailingCut||0,!a.noNewLine)break}return this.inlineWidths+n-this.leadingCut-r<=this.maxWidth},r.prototype.addInline=function(t){0===this.inlines.length&&(this.leadingCut=t.leadingCut||0),this.trailingCut=t.trailingCut||0,t.x=this.inlineWidths-this.leadingCut,this.inlines.push(t),this.inlineWidths+=t.width,t.lineEnd&&(this.newLineForced=!0)},r.prototype.getWidth=function(){return this.inlineWidths-this.leadingCut-this.trailingCut},r.prototype.getAvailableWidth=function(){return this.maxWidth-this.getWidth()},r.prototype.getHeight=function(){var e=0;return this.inlines.forEach(function(t){e=Math.max(e,t.height||0)}),e},t.exports=r},function(e,t,n){(function(t){e.exports=t.pdfMake=n(197)}).call(this,n(25))},function(t,e,n){"use strict";(function(i,o){var r=n(0).isFunction,a=n(0).isUndefined,s=(n(0).isNull,n(200).saveAs),l={Roboto:{normal:"Roboto-Regular.ttf",bold:"Roboto-Medium.ttf",italics:"Roboto-Italic.ttf",bolditalics:"Roboto-MediumItalic.ttf"}};function u(t,e,n,r){this.docDefinition=t,this.tableLayouts=e||null,this.fonts=n||l,this.vfs=r}u.prototype._createDoc=function(t){t=t||{},this.tableLayouts&&(t.tableLayouts=this.tableLayouts);var e=new(n(201))(this.fonts);return n(52).bindFS(this.vfs),e.createPdfKitDocument(this.docDefinition,t)},u.prototype._flushDoc=function(e,t){var n,r=[];e.on("readable",function(){for(var t;null!==(t=e.read(9007199254740991));)r.push(t)}),e.on("end",function(){n=i.concat(r),t(n,e._pdfMakePages)}),e.end()},u.prototype._getPages=function(t,n){if(!n)throw"_getPages is an async method and needs a callback argument";var e=this._createDoc(t);this._flushDoc(e,function(t,e){n(e)})},u.prototype._bufferToBlob=function(e){var n;try{n=new Blob([e],{type:"application/pdf"})}catch(t){if("InvalidStateError"===t.name){var r=new Uint8Array(e);n=new Blob([r.buffer],{type:"application/pdf"})}}if(!n)throw"Could not generate blob";return n},u.prototype._openWindow=function(){var t=window.open("","_blank");if(null===t)throw"Open PDF in new window blocked by browser";return t},u.prototype._openPdf=function(t,n){n=n||this._openWindow();try{this.getBlob(function(t){var e=(window.URL||window.webkitURL).createObjectURL(t);n.location.href=e},t)}catch(t){throw n.close(),t}},u.prototype.open=function(t,e){(t=t||{}).autoPrint=!1,e=e||null,this._openPdf(t,e)},u.prototype.print=function(t,e){(t=t||{}).autoPrint=!0,e=e||null,this._openPdf(t,e)},u.prototype.download=function(e,n,t){r(e)&&(a(n)||(t=n),n=e,e=null),e=e||"file.pdf",this.getBlob(function(t){s(t,e),r(n)&&n()},t)},u.prototype.getBase64=function(e,t){if(!e)throw"getBase64 is an async method and needs a callback argument";this.getBuffer(function(t){e(t.toString("base64"))},t)},u.prototype.getDataUrl=function(e,t){if(!e)throw"getDataUrl is an async method and needs a callback argument";this.getBuffer(function(t){e("data:application/pdf;base64,"+t.toString("base64"))},t)},u.prototype.getBlob=function(n,t){if(!n)throw"getBlob is an async method and needs a callback argument";var r=this;this.getBuffer(function(t){var e=r._bufferToBlob(t);n(e)},t)},u.prototype.getBuffer=function(e,t){if(!e)throw"getBuffer is an async method and needs a callback argument";var n=this._createDoc(t);this._flushDoc(n,function(t){e(t)})},u.prototype.getStream=function(t){return this._createDoc(t)},t.exports={createPdf:function(t,e,n,r){if(!Object.keys||"undefined"==typeof Uint16Array)throw"Your browser does not provide the level of support needed";return new u(t,e||o.pdfMake.tableLayouts,n||o.pdfMake.fonts,r||o.pdfMake.vfs)}}}).call(this,n(4).Buffer,n(25))},function(t,e,n){"use strict";e.byteLength=function(t){var e=f(t),n=e[0],r=e[1];return 3*(n+r)/4-r},e.toByteArray=function(t){for(var e,n=f(t),r=n[0],i=n[1],o=new c(function(t,e,n){return 3*(e+n)/4-n}(0,r,i)),a=0,s=0<i?r-4:r,l=0;l<s;l+=4)e=u[t.charCodeAt(l)]<<18|u[t.charCodeAt(l+1)]<<12|u[t.charCodeAt(l+2)]<<6|u[t.charCodeAt(l+3)],o[a++]=e>>16&255,o[a++]=e>>8&255,o[a++]=255&e;2===i&&(e=u[t.charCodeAt(l)]<<2|u[t.charCodeAt(l+1)]>>4,o[a++]=255&e);1===i&&(e=u[t.charCodeAt(l)]<<10|u[t.charCodeAt(l+1)]<<4|u[t.charCodeAt(l+2)]>>2,o[a++]=e>>8&255,o[a++]=255&e);return o},e.fromByteArray=function(t){for(var e,n=t.length,r=n%3,i=[],o=0,a=n-r;o<a;o+=16383)i.push(l(t,o,a<o+16383?a:o+16383));1==r?(e=t[n-1],i.push(s[e>>2]+s[e<<4&63]+"==")):2==r&&(e=(t[n-2]<<8)+t[n-1],i.push(s[e>>10]+s[e>>4&63]+s[e<<2&63]+"="));return i.join("")};for(var s=[],u=[],c="undefined"!=typeof Uint8Array?Uint8Array:Array,r="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",i=0,o=r.length;i<o;++i)s[i]=r[i],u[r.charCodeAt(i)]=i;function f(t){var e=t.length;if(0<e%4)throw new Error("Invalid string. Length must be a multiple of 4");var n=t.indexOf("=");return-1===n&&(n=e),[n,n===e?0:4-n%4]}function l(t,e,n){for(var r,i,o=[],a=e;a<n;a+=3)r=(t[a]<<16&16711680)+(t[a+1]<<8&65280)+(255&t[a+2]),o.push(s[(i=r)>>18&63]+s[i>>12&63]+s[i>>6&63]+s[63&i]);return o.join("")}u["-".charCodeAt(0)]=62,u["_".charCodeAt(0)]=63},function(t,e){e.read=function(t,e,n,r,i){var o,a,s=8*i-r-1,l=(1<<s)-1,u=l>>1,c=-7,f=n?i-1:0,h=n?-1:1,d=t[e+f];for(f+=h,o=d&(1<<-c)-1,d>>=-c,c+=s;0<c;o=256*o+t[e+f],f+=h,c-=8);for(a=o&(1<<-c)-1,o>>=-c,c+=r;0<c;a=256*a+t[e+f],f+=h,c-=8);if(0===o)o=1-u;else{if(o===l)return a?NaN:1/0*(d?-1:1);a+=Math.pow(2,r),o-=u}return(d?-1:1)*a*Math.pow(2,o-r)},e.write=function(t,e,n,r,i,o){var a,s,l,u=8*o-i-1,c=(1<<u)-1,f=c>>1,h=23===i?Math.pow(2,-24)-Math.pow(2,-77):0,d=r?0:o-1,p=r?1:-1,g=e<0||0===e&&1/e<0?1:0;for(e=Math.abs(e),isNaN(e)||e===1/0?(s=isNaN(e)?1:0,a=c):(a=Math.floor(Math.log(e)/Math.LN2),e*(l=Math.pow(2,-a))<1&&(a--,l*=2),2<=(e+=1<=a+f?h/l:h*Math.pow(2,1-f))*l&&(a++,l/=2),c<=a+f?(s=0,a=c):1<=a+f?(s=(e*l-1)*Math.pow(2,i),a+=f):(s=e*Math.pow(2,f-1)*Math.pow(2,i),a=0));8<=i;t[n+d]=255&s,d+=p,s/=256,i-=8);for(a=a<<i|s,u+=i;0<u;t[n+d]=255&a,d+=p,a/=256,u-=8);t[n+d-p]|=128*g}},function(s,i,t){(function(t){var e,n,r;n=[],void 0===(r="function"==typeof(e=function(){"use strict";function c(t,e,n){var r=new XMLHttpRequest;r.open("GET",t),r.responseType="blob",r.onload=function(){i(r.response,e,n)},r.onerror=function(){},r.send()}function o(t){var e=new XMLHttpRequest;e.open("HEAD",t,!1);try{e.send()}catch(t){}return 200<=e.status&&e.status<=299}function a(e){try{e.dispatchEvent(new MouseEvent("click"))}catch(t){var n=document.createEvent("MouseEvents");n.initMouseEvent("click",!0,!0,window,0,0,0,80,20,!1,!1,!1,!1,0,null),e.dispatchEvent(n)}}var f="object"==typeof window&&window.window===window?window:"object"==typeof self&&self.self===self?self:"object"==typeof t&&t.global===t?t:void 0,i=f.saveAs||("object"!=typeof window||window!==f?function(){}:"undefined"!=typeof HTMLAnchorElement&&"download"in HTMLAnchorElement.prototype?function(t,e,n){var r=f.URL||f.webkitURL,i=document.createElement("a");e=e||t.name||"download",i.download=e,i.rel="noopener","string"==typeof t?(i.href=t,i.origin===location.origin?a(i):o(i.href)?c(t,e,n):a(i,i.target="_blank")):(i.href=r.createObjectURL(t),setTimeout(function(){r.revokeObjectURL(i.href)},4e4),setTimeout(function(){a(i)},0))}:"msSaveOrOpenBlob"in navigator?function(t,e,n){if(e=e||t.name||"download","string"!=typeof t)navigator.msSaveOrOpenBlob(function(t,e){return void 0===e?e={autoBom:!1}:"object"!=typeof e&&(e={autoBom:!e}),e.autoBom&&/^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(t.type)?new Blob(["\ufeff",t],{type:t.type}):t}(t,n),e);else if(o(t))c(t,e,n);else{var r=document.createElement("a");r.href=t,r.target="_blank",setTimeout(function(){a(r)})}}:function(t,e,n,r){if((r=r||open("","_blank"))&&(r.document.title=r.document.body.innerText="downloading..."),"string"==typeof t)return c(t,e,n);var i="application/octet-stream"===t.type,o=/constructor/i.test(f.HTMLElement)||f.safari,a=/CriOS\/[\d]+/.test(navigator.userAgent);if((a||i&&o)&&"object"==typeof FileReader){var s=new FileReader;s.onloadend=function(){var t=s.result;t=a?t:t.replace(/^data:[^;]*;/,"data:attachment/file;"),r?r.location.href=t:location=t,r=null},s.readAsDataURL(t)}else{var l=f.URL||f.webkitURL,u=l.createObjectURL(t);r?r.location=u:location.href=u,r=null,setTimeout(function(){l.revokeObjectURL(u)},4e4)}});f.saveAs=i.saveAs=i,s.exports=i})?e.apply(i,n):e)||(s.exports=r)}).call(this,t(25))},function(t,e,n){"use strict";var u=n(202),c=n(427),f=n(428),h=n(435),d=n(436),p=n(437),g=n(127),v=n(0).isFunction,y=n(0).isString,b=n(0).isNumber,m=n(0).isBoolean,r=n(0).isArray,w=n(0).isUndefined;function i(t){this.fontDescriptors=t}function x(t){if(!t)return null;if(b(t))t={left:t,right:t,top:t,bottom:t};else if(r(t))if(2===t.length)t={left:t[0],top:t[1],right:t[0],bottom:t[1]};else{if(4!==t.length)throw"Invalid pageMargins definition";t={left:t[0],top:t[1],right:t[2],bottom:t[3]}}return t}function _(t,e){var n=e.options.size[0]>e.options.size[1]?"landscape":"portrait";if(t.pageSize.orientation!==n){var r=e.options.size[0],i=e.options.size[1];e.options.size=[i,r]}}function k(t,e,n,r){function i(t,e){var n,r,i=new g(null);if(w(t.positions))throw"Page reference id not found";var o=t.positions[0].pageNumber.toString();switch(e.text=o,n=i.widthOfString(e.text,e.font,e.fontSize,e.characterSpacing,e.fontFeatures),r=e.width-n,e.width=n,e.alignment){case"right":e.x+=r;break;case"center":e.x+=r/2}}t._pageNodeRef&&i(t._pageNodeRef,t.inlines[0]),e=e||0,n=n||0;var o=t.getHeight(),a=o-t.getAscenderHeight();p.drawBackground(t,e,n,r);for(var s=0,l=t.inlines.length;s<l;s++){var u=t.inlines[s],c=o-u.font.ascender/1e3*u.fontSize-a;u._pageNodeRef&&i(u._pageNodeRef,u);var f={lineBreak:!1,textWidth:u.width,characterSpacing:u.characterSpacing,wordCount:1,link:u.link};if(u.linkToDestination&&(f.goTo=u.linkToDestination),t.id&&0===s&&(f.destination=t.id),u.fontFeatures&&(f.features=u.fontFeatures),r.opacity(u.opacity||1),r.fill(u.color||"black"),r._font=u.font,r.fontSize(u.fontSize),r.text(u.text,e+u.x,n+c,f),u.linkToPage){r.ref({Type:"Action",S:"GoTo",D:[u.linkToPage,0,0]}).end();r.annotate(e+u.x,n+c,u.width,u.height,{Subtype:"Link",Dest:[u.linkToPage-1,"XYZ",null,null,null]})}}p.drawDecorations(t,e,n,r)}function S(t,e){var n=t.watermark;e.fill(n.color),e.opacity(n.opacity),e.save();var r=-180*Math.atan2(e.page.height,e.page.width)/Math.PI;e.rotate(r,{origin:[e.page.width/2,e.page.height/2]});var i=e.page.width/2-n.size.size.width/2,o=e.page.height/2-n.size.size.height/4;e._font=n.font,e.fontSize(n.size.fontSize),e.text(n.text,i,o,{lineBreak:!1}),e.restore()}function C(t,e){e.lineWidth(t.lineWidth||1),t.dash?e.dash(t.dash.length,{space:t.dash.space||t.dash.length,phase:t.dash.phase||0}):e.undash(),e.lineJoin(t.lineJoin||"miter"),e.lineCap(t.lineCap||"butt");var n=null;switch(t.type){case"ellipse":e.ellipse(t.x,t.y,t.r1,t.r2),t.linearGradient&&(n=e.linearGradient(t.x-t.r1,t.y,t.x+t.r1,t.y));break;case"rect":t.r?e.roundedRect(t.x,t.y,t.w,t.h,t.r):e.rect(t.x,t.y,t.w,t.h),t.linearGradient&&(n=e.linearGradient(t.x,t.y,t.x+t.w,t.y));break;case"line":e.moveTo(t.x1,t.y1),e.lineTo(t.x2,t.y2);break;case"polyline":if(0===t.points.length)break;e.moveTo(t.points[0].x,t.points[0].y);for(var r=1,i=t.points.length;r<i;r++)e.lineTo(t.points[r].x,t.points[r].y);if(1<t.points.length){var o=t.points[0],a=t.points[t.points.length-1];(t.closePath||o.x===a.x&&o.y===a.y)&&e.closePath()}break;case"path":e.path(t.d)}if(t.linearGradient&&n){var s=1/(t.linearGradient.length-1);for(r=0;r<t.linearGradient.length;r++)n.stop(r*s,t.linearGradient[r]);t.color=n}t.color&&t.lineColor?(e.fillColor(t.color,t.fillOpacity||1),e.strokeColor(t.lineColor,t.strokeOpacity||1),e.fillAndStroke()):t.color?(e.fillColor(t.color,t.fillOpacity||1),e.fill()):(e.strokeColor(t.lineColor||"black",t.strokeOpacity||1),e.stroke())}function A(t,e,n,r){r.opacity(t.opacity||1),r.image(t.image,t.x,t.y,{width:t._width,height:t._height}),t.link&&r.link(t.x,t.y,t._width,t._height,t.link)}function P(t,e){e.save(),e.addContent(t.x+" "+t.y+" "+t.width+" "+t.height+" re"),e.clip()}i.prototype.createPdfKitDocument=function(t,e){e=e||{},t.version=t.version||"1.3",t.compress=!m(t.compress)||t.compress,t.images=t.images||{};var n=function(t,e){t&&"auto"===t.height&&(t.height=1/0);var n=function(t){if(y(t)){var e=h[t.toUpperCase()];if(!e)throw"Page size "+t+" not recognized";return{width:e[0],height:e[1]}}return t}(t||"A4");!function(t){return!!y(t)&&("portrait"===(t=t.toLowerCase())&&n.width>n.height||"landscape"===t&&n.width<n.height)}(e)||(n={width:n.height,height:n.width});return n.orientation=n.width>n.height?"landscape":"portrait",n}(t.pageSize,t.pageOrientation),r={size:[n.width,n.height],pdfVersion:t.version,compress:t.compress,userPassword:t.userPassword,ownerPassword:t.ownerPassword,permissions:t.permissions,fontLayoutCache:!m(e.fontLayoutCache)||e.fontLayoutCache,bufferPages:e.bufferPages||!1,autoFirstPage:!1,font:null};this.pdfKitDoc=u.createPdfDocument(r),function(t,e){if(e.info.Producer="pdfmake",e.info.Creator="pdfmake",t.info)for(var n in t.info){var r=t.info[n];r&&(void 0,o=(i=n).charAt(0).toUpperCase()+i.slice(1),n=-1!==["Title","Author","Subject","Keywords","Creator","Producer","CreationDate","ModDate","Trapped"].indexOf(o)?o:i.replace(/\s+/g,""),e.info[n]=r)}var i,o}(t,this.pdfKitDoc),this.fontProvider=new c(this.fontDescriptors,this.pdfKitDoc);var i=new f(n,x(t.pageMargins||40),new d(this.pdfKitDoc,t.images));!function(t){t.registerTableLayouts({noBorders:{hLineWidth:function(t){return 0},vLineWidth:function(t){return 0},paddingLeft:function(t){return t?4:0},paddingRight:function(t,e){return t<e.table.widths.length-1?4:0}},headerLineOnly:{hLineWidth:function(t,e){return 0===t||t===e.table.body.length?0:t===e.table.headerRows?2:0},vLineWidth:function(t){return 0},paddingLeft:function(t){return 0===t?0:8},paddingRight:function(t,e){return t===e.table.widths.length-1?0:8}},lightHorizontalLines:{hLineWidth:function(t,e){return 0===t||t===e.table.body.length?0:t===e.table.headerRows?2:1},vLineWidth:function(t){return 0},hLineColor:function(t){return 1===t?"black":"#aaa"},paddingLeft:function(t){return 0===t?0:8},paddingRight:function(t,e){return t===e.table.widths.length-1?0:8}}})}(i),e.tableLayouts&&i.registerTableLayouts(e.tableLayouts);var o=i.layoutDocument(t.content,this.fontProvider,t.styles||{},t.defaultStyle||{fontSize:12,font:"Roboto"},t.background,t.header,t.footer,t.images,t.watermark,t.pageBreakBefore),a=t.maxPagesNumber||-1;if(b(a)&&-1<a&&(o=o.slice(0,a)),n.height===1/0){var s=function(t,e){var n=x(e||40),r=n.top;return t.forEach(function(t){t.items.forEach(function(t){var e=function(t){var e=t.item.y,n=function(t){return v(t.item.getHeight)?t.item.getHeight():t.item._height?t.item._height:0}(t);return e+n}(t);r<e&&(r=e)})}),r+=n.bottom}(o,t.pageMargins);this.pdfKitDoc.options.size=[n.width,s]}if(function(t,e,n,r){n._pdfMakePages=t,n.addPage();var i=0;r&&t.forEach(function(t){i+=t.items.length});var o=0;r=r||function(){};for(var a=0;a<t.length;a++){0<a&&(_(t[a],n),n.addPage(n.options));for(var s=t[a],l=0,u=s.items.length;l<u;l++){var c=s.items[l];switch(c.type){case"vector":C(c.item,n);break;case"line":k(c.item,c.item.x,c.item.y,n);break;case"image":A(c.item,c.item.x,c.item.y,n);break;case"beginClip":P(c.item,n);break;case"endClip":n.restore()}r(++o/i)}s.watermark&&S(s,n)}}(o,this.fontProvider,this.pdfKitDoc,e.progressCallback),e.autoPrint){var l=this.pdfKitDoc.ref({Type:"Action",S:"Named",N:"Print"});(this.pdfKitDoc._root.data.OpenAction=l).end()}return this.pdfKitDoc},t.exports=i},function(t,e,n){"use strict";var r,i=(r=n(203))&&"object"==typeof r&&"default"in r?r.default:r;t.exports={getEngineInstance:function(){return i},createPdfDocument:function(t){return new i(t=t||{})}}},function(t,Jt,Qt){"use strict";(function(p,t){Object.defineProperty(Jt,"__esModule",{value:!0}),Jt.default=void 0,Qt(204),Qt(206),Qt(207),Qt(209),Qt(210),Qt(222),Qt(224),Qt(225),Qt(229),Qt(230),Qt(231),Qt(234),Qt(235),Qt(236),Qt(237),Qt(88),Qt(238),Qt(240),Qt(241),Qt(244),Qt(246),Qt(247),Qt(248),Qt(252),Qt(253);var e=i(Qt(95)),h=i(Qt(154)),d=i(Qt(274)),o=i(Qt(298)),n=Qt(68),g=i(Qt(191)),r=i(Qt(426));function i(t){return t&&t.__esModule?t:{default:t}}var s=Qt(52);function u(t,e){if(!(t instanceof e))throw new TypeError("Cannot call a class as a function")}function a(t,e){for(var n=0;n<e.length;n++){var r=e[n];r.enumerable=r.enumerable||!1,r.configurable=!0,"value"in r&&(r.writable=!0),Object.defineProperty(t,r.key,r)}}function f(t,e,n){return e&&a(t.prototype,e),n&&a(t,n),t}function c(t,e){if("function"!=typeof e&&null!==e)throw new TypeError("Super expression must either be null or a function");t.prototype=Object.create(e&&e.prototype,{constructor:{value:t,writable:!0,configurable:!0}}),e&&l(t,e)}function v(t){return(v=Object.setPrototypeOf?Object.getPrototypeOf:function(t){return t.__proto__||Object.getPrototypeOf(t)})(t)}function l(t,e){return(l=Object.setPrototypeOf||function(t,e){return t.__proto__=e,t})(t,e)}function y(t){if(void 0===t)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return t}function b(t,e){return!e||"object"!=typeof e&&"function"!=typeof e?y(t):e}function j(t,e){return function(t){if(Array.isArray(t))return t}(t)||function(t,e){var n=[],r=!0,i=!1,o=void 0;try{for(var a,s=t[Symbol.iterator]();!(r=(a=s.next()).done)&&(n.push(a.value),!e||n.length!==e);r=!0);}catch(t){i=!0,o=t}finally{try{r||null==s.return||s.return()}finally{if(i)throw o}}return n}(t,e)||function(){throw new TypeError("Invalid attempt to destructure non-iterable instance")}()}function G(t){return function(t){if(Array.isArray(t)){for(var e=0,n=new Array(t.length);e<t.length;e++)n[e]=t[e];return n}}(t)||function(t){if(Symbol.iterator in Object(t)||"[object Arguments]"===Object.prototype.toString.call(t))return Array.from(t)}(t)||function(){throw new TypeError("Invalid attempt to spread non-iterable instance")}()}function m(t,e){return(Array(e+1).join("0")+t).slice(-e)}var w=function(){function t(){u(this,t)}return f(t,[{key:"toString",value:function(){throw new Error("Must be implemented by subclasses")}}]),t}(),x=function(){function t(){u(this,t),this._items={}}return f(t,[{key:"add",value:function(t,e){return this._items[t]=e}},{key:"get",value:function(t){return this._items[t]}},{key:"toString",value:function(){var t=Object.keys(this._items).sort(function(t,e){return t.localeCompare(e)}),e=["<<"];if(1<t.length){var n=t[0],r=t[t.length-1];e.push("  /Limits ".concat(S.convert([new String(n),new String(r)])))}e.push("  /Names [");var i=!0,o=!1,a=void 0;try{for(var s,l=t[Symbol.iterator]();!(i=(s=l.next()).done);i=!0){var u=s.value;e.push("    ".concat(S.convert(new String(u))," ").concat(S.convert(this._items[u])))}}catch(t){o=!0,a=t}finally{try{i||null==l.return||l.return()}finally{if(o)throw a}}return e.push("]"),e.push(">>"),e.join("\n")}}]),t}(),_=/[\n\r\t\b\f\(\)\\]/g,k={"\n":"\\n","\r":"\\r","\t":"\\t","\b":"\\b","\f":"\\f","\\":"\\\\","(":"\\(",")":"\\)"},S=function(){function d(){u(this,d)}return f(d,null,[{key:"convert",value:function(t,e){var n=1<arguments.length&&void 0!==e?e:null;if("string"==typeof t)return"/".concat(t);if(t instanceof String){for(var r,i=t,o=!1,a=0,s=i.length;a<s;a++)if(127<i.charCodeAt(a)){o=!0;break}return r=o?function(t){var e=t.length;if(1&e)throw new Error("Buffer length must be even");for(var n=0,r=e-1;n<r;n+=2){var i=t[n];t[n]=t[n+1],t[n+1]=i}return t}(p.from("\ufeff".concat(i),"utf16le")):p.from(i.valueOf(),"ascii"),i=(i=n?n(r).toString("binary"):r.toString("binary")).replace(_,function(t){return k[t]}),"(".concat(i,")")}if(p.isBuffer(t))return"<".concat(t.toString("hex"),">");if(t instanceof w||t instanceof x)return t.toString();if(t instanceof Date){var l="D:".concat(m(t.getUTCFullYear(),4))+m(t.getUTCMonth()+1,2)+m(t.getUTCDate(),2)+m(t.getUTCHours(),2)+m(t.getUTCMinutes(),2)+m(t.getUTCSeconds(),2)+"Z";return n&&(l=(l=n(new p(l,"ascii")).toString("binary")).replace(_,function(t){return k[t]})),"(".concat(l,")")}if(Array.isArray(t)){var u=t.map(function(t){return d.convert(t,n)}).join(" ");return"[".concat(u,"]")}if("[object Object]"!=={}.toString.call(t))return"number"==typeof t?d.number(t):"".concat(t);var c=["<<"];for(var f in t){var h=t[f];c.push("/".concat(f," ").concat(d.convert(h,n)))}return c.push(">>"),c.join("\n")}},{key:"number",value:function(t){if(-1e21<t&&t<1e21)return Math.round(1e6*t)/1e6;throw new Error("unsupported number: ".concat(t))}}]),d}(),C=function(){function i(t,e){var n,r=2<arguments.length&&void 0!==arguments[2]?arguments[2]:{};return u(this,i),(n=b(this,v(i).call(this))).document=t,n.id=e,n.data=r,n.gen=0,n.compress=n.document.compress&&!n.data.Filter,n.uncompressedLength=0,n.buffer=[],n}return c(i,w),f(i,[{key:"write",value:function(t){if(p.isBuffer(t)||(t=new p(t+"\n","binary")),this.uncompressedLength+=t.length,null==this.data.Length&&(this.data.Length=0),this.buffer.push(t),this.data.Length+=t.length,this.compress)return this.data.Filter="FlateDecode"}},{key:"end",value:function(t){return t&&this.write(t),this.finalize()}},{key:"finalize",value:function(){this.offset=this.document._offset;var t=this.document._security?this.document._security.getEncryptFn(this.id,this.gen):null;this.buffer.length&&(this.buffer=p.concat(this.buffer),this.compress&&(this.buffer=h.default.deflateSync(this.buffer)),t&&(this.buffer=t(this.buffer)),this.data.Length=this.buffer.length),this.document._write("".concat(this.id," ").concat(this.gen," obj")),this.document._write(S.convert(this.data,t)),this.buffer.length&&(this.document._write("stream"),this.document._write(this.buffer),this.buffer=[],this.document._write("\nendstream")),this.document._write("endobj"),this.document._refEnd(this)}},{key:"toString",value:function(){return"".concat(this.id," ").concat(this.gen," R")}}]),i}(),A={top:72,left:72,bottom:72,right:72},P={"4A0":[4767.87,6740.79],"2A0":[3370.39,4767.87],A0:[2383.94,3370.39],A1:[1683.78,2383.94],A2:[1190.55,1683.78],A3:[841.89,1190.55],A4:[595.28,841.89],A5:[419.53,595.28],A6:[297.64,419.53],A7:[209.76,297.64],A8:[147.4,209.76],A9:[104.88,147.4],A10:[73.7,104.88],B0:[2834.65,4008.19],B1:[2004.09,2834.65],B2:[1417.32,2004.09],B3:[1000.63,1417.32],B4:[708.66,1000.63],B5:[498.9,708.66],B6:[354.33,498.9],B7:[249.45,354.33],B8:[175.75,249.45],B9:[124.72,175.75],B10:[87.87,124.72],C0:[2599.37,3676.54],C1:[1836.85,2599.37],C2:[1298.27,1836.85],C3:[918.43,1298.27],C4:[649.13,918.43],C5:[459.21,649.13],C6:[323.15,459.21],C7:[229.61,323.15],C8:[161.57,229.61],C9:[113.39,161.57],C10:[79.37,113.39],RA0:[2437.8,3458.27],RA1:[1729.13,2437.8],RA2:[1218.9,1729.13],RA3:[864.57,1218.9],RA4:[609.45,864.57],SRA0:[2551.18,3628.35],SRA1:[1814.17,2551.18],SRA2:[1275.59,1814.17],SRA3:[907.09,1275.59],SRA4:[637.8,907.09],EXECUTIVE:[521.86,756],FOLIO:[612,936],LEGAL:[612,1008],LETTER:[612,792],TABLOID:[792,1224]},E=function(){function r(t){var e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:{};u(this,r),this.document=t,this.size=e.size||"letter",this.layout=e.layout||"portrait","number"==typeof e.margin?this.margins={top:e.margin,left:e.margin,bottom:e.margin,right:e.margin}:this.margins=e.margins||A;var n=Array.isArray(this.size)?this.size:P[this.size.toUpperCase()];this.width=n["portrait"===this.layout?0:1],this.height=n["portrait"===this.layout?1:0],this.content=this.document.ref(),this.resources=this.document.ref({ProcSet:["PDF","Text","ImageB","ImageC","ImageI"]}),this.dictionary=this.document.ref({Type:"Page",Parent:this.document._root.data.Pages,MediaBox:[0,0,this.width,this.height],Contents:this.content,Resources:this.resources})}return f(r,[{key:"maxY",value:function(){return this.height-this.margins.bottom}},{key:"write",value:function(t){return this.content.write(t)}},{key:"end",value:function(){return this.dictionary.end(),this.resources.end(),this.content.end()}},{key:"fonts",get:function(){var t=this.resources.data;return null!=t.Font?t.Font:t.Font={}}},{key:"xobjects",get:function(){var t=this.resources.data;return null!=t.XObject?t.XObject:t.XObject={}}},{key:"ext_gstates",get:function(){var t=this.resources.data;return null!=t.ExtGState?t.ExtGState:t.ExtGState={}}},{key:"patterns",get:function(){var t=this.resources.data;return null!=t.Pattern?t.Pattern:t.Pattern={}}},{key:"annotations",get:function(){var t=this.dictionary.data;return null!=t.Annots?t.Annots:t.Annots=[]}}]),r}();function O(t,e){if(t<e[0])return!1;for(var n=0,r=e.length/2;n<=r;){var i=Math.floor((n+r)/2),o=2*i;if(t>=e[o]&&t<=e[1+o])return!0;t>e[1+o]?n=i+1:r=i-1}return!1}var T=[545,545,564,591,686,687,751,767,848,863,880,883,886,889,891,893,895,899,907,907,909,909,930,930,975,975,1015,1023,1159,1159,1231,1231,1270,1271,1274,1279,1296,1328,1367,1368,1376,1376,1416,1416,1419,1424,1442,1442,1466,1466,1477,1487,1515,1519,1525,1547,1549,1562,1564,1566,1568,1568,1595,1599,1622,1631,1774,1775,1791,1791,1806,1806,1837,1839,1867,1919,1970,2304,2308,2308,2362,2363,2382,2383,2389,2391,2417,2432,2436,2436,2445,2446,2449,2450,2473,2473,2481,2481,2483,2485,2490,2491,2493,2493,2501,2502,2505,2506,2510,2518,2520,2523,2526,2526,2532,2533,2555,2561,2563,2564,2571,2574,2577,2578,2601,2601,2609,2609,2612,2612,2615,2615,2618,2619,2621,2621,2627,2630,2633,2634,2638,2648,2653,2653,2655,2661,2677,2688,2692,2692,2700,2700,2702,2702,2706,2706,2729,2729,2737,2737,2740,2740,2746,2747,2758,2758,2762,2762,2766,2767,2769,2783,2785,2789,2800,2816,2820,2820,2829,2830,2833,2834,2857,2857,2865,2865,2868,2869,2874,2875,2884,2886,2889,2890,2894,2901,2904,2907,2910,2910,2914,2917,2929,2945,2948,2948,2955,2957,2961,2961,2966,2968,2971,2971,2973,2973,2976,2978,2981,2983,2987,2989,2998,2998,3002,3005,3011,3013,3017,3017,3022,3030,3032,3046,3059,3072,3076,3076,3085,3085,3089,3089,3113,3113,3124,3124,3130,3133,3141,3141,3145,3145,3150,3156,3159,3167,3170,3173,3184,3201,3204,3204,3213,3213,3217,3217,3241,3241,3252,3252,3258,3261,3269,3269,3273,3273,3278,3284,3287,3293,3295,3295,3298,3301,3312,3329,3332,3332,3341,3341,3345,3345,3369,3369,3386,3389,3396,3397,3401,3401,3406,3414,3416,3423,3426,3429,3440,3457,3460,3460,3479,3481,3506,3506,3516,3516,3518,3519,3527,3529,3531,3534,3541,3541,3543,3543,3552,3569,3573,3584,3643,3646,3676,3712,3715,3715,3717,3718,3721,3721,3723,3724,3726,3731,3736,3736,3744,3744,3748,3748,3750,3750,3752,3753,3756,3756,3770,3770,3774,3775,3781,3781,3783,3783,3790,3791,3802,3803,3806,3839,3912,3912,3947,3952,3980,3983,3992,3992,4029,4029,4045,4046,4048,4095,4130,4130,4136,4136,4139,4139,4147,4149,4154,4159,4186,4255,4294,4303,4345,4346,4348,4351,4442,4446,4515,4519,4602,4607,4615,4615,4679,4679,4681,4681,4686,4687,4695,4695,4697,4697,4702,4703,4743,4743,4745,4745,4750,4751,4783,4783,4785,4785,4790,4791,4799,4799,4801,4801,4806,4807,4815,4815,4823,4823,4847,4847,4879,4879,4881,4881,4886,4887,4895,4895,4935,4935,4955,4960,4989,5023,5109,5120,5751,5759,5789,5791,5873,5887,5901,5901,5909,5919,5943,5951,5972,5983,5997,5997,6001,6001,6004,6015,6109,6111,6122,6143,6159,6159,6170,6175,6264,6271,6314,7679,7836,7839,7930,7935,7958,7959,7966,7967,8006,8007,8014,8015,8024,8024,8026,8026,8028,8028,8030,8030,8062,8063,8117,8117,8133,8133,8148,8149,8156,8156,8176,8177,8181,8181,8191,8191,8275,8278,8280,8286,8292,8297,8306,8307,8335,8351,8370,8399,8427,8447,8507,8508,8524,8530,8580,8591,9167,9215,9255,9279,9291,9311,9471,9471,9748,9749,9752,9752,9854,9855,9866,9984,9989,9989,9994,9995,10024,10024,10060,10060,10062,10062,10067,10069,10071,10071,10079,10080,10133,10135,10160,10160,10175,10191,10220,10223,11008,11903,11930,11930,12020,12031,12246,12271,12284,12287,12352,12352,12439,12440,12544,12548,12589,12592,12687,12687,12728,12783,12829,12831,12868,12880,12924,12926,13004,13007,13055,13055,13175,13178,13278,13279,13311,13311,19894,19967,40870,40959,42125,42127,42183,44031,55204,55295,64046,64047,64107,64255,64263,64274,64280,64284,64311,64311,64317,64317,64319,64319,64322,64322,64325,64325,64434,64466,64832,64847,64912,64913,64968,64975,65021,65023,65040,65055,65060,65071,65095,65096,65107,65107,65127,65127,65132,65135,65141,65141,65277,65278,65280,65280,65471,65473,65480,65481,65488,65489,65496,65497,65501,65503,65511,65511,65519,65528,65536,66303,66335,66335,66340,66351,66379,66559,66598,66599,66638,118783,119030,119039,119079,119081,119262,119807,119893,119893,119965,119965,119968,119969,119971,119972,119975,119976,119981,119981,119994,119994,119996,119996,120001,120001,120004,120004,120070,120070,120075,120076,120085,120085,120093,120093,120122,120122,120127,120127,120133,120133,120135,120137,120145,120145,120484,120487,120778,120781,120832,131069,173783,194559,195102,196605,196608,262141,262144,327677,327680,393213,393216,458749,458752,524285,524288,589821,589824,655357,655360,720893,720896,786429,786432,851965,851968,917501,917504,917504,917506,917535,917632,983037],I=function(t){return O(t,T)},B=[173,173,847,847,6150,6150,6155,6155,6156,6156,6157,6157,8203,8203,8204,8204,8205,8205,8288,8288,65024,65024,65025,65025,65026,65026,65027,65027,65028,65028,65029,65029,65030,65030,65031,65031,65032,65032,65033,65033,65034,65034,65035,65035,65036,65036,65037,65037,65038,65038,65039,65039,65279,65279],L=[160,160,5760,5760,8192,8192,8193,8193,8194,8194,8195,8195,8196,8196,8197,8197,8198,8198,8199,8199,8200,8200,8201,8201,8202,8202,8203,8203,8239,8239,8287,8287,12288,12288],R=[128,159,1757,1757,1807,1807,6158,6158,8204,8204,8205,8205,8232,8232,8233,8233,8288,8288,8289,8289,8290,8290,8291,8291,8298,8303,65279,65279,65529,65532,119155,119162],M=[64976,65007,65534,65535,131070,131071,196606,196607,262142,262143,327678,327679,393214,393215,458750,458751,524286,524287,589822,589823,655358,655359,720894,720895,786430,786431,851966,851967,917502,917503,983038,983039,1114110,1114111],F=[0,31,127,127,832,832,833,833,8206,8206,8207,8207,8234,8234,8235,8235,8236,8236,8237,8237,8238,8238,8298,8298,8299,8299,8300,8300,8301,8301,8302,8302,8303,8303,12272,12283,55296,57343,57344,63743,65529,65529,65530,65530,65531,65531,65532,65532,65533,65533,917505,917505,917536,917631,983040,1048573,1048576,1114109],D=function(t){return O(t,L)||O(t,F)||O(t,R)||O(t,M)},z=[1470,1470,1472,1472,1475,1475,1488,1514,1520,1524,1563,1563,1567,1567,1569,1594,1600,1610,1645,1647,1649,1749,1757,1757,1765,1766,1786,1790,1792,1805,1808,1808,1810,1836,1920,1957,1969,1969,8207,8207,64285,64285,64287,64296,64298,64310,64312,64316,64318,64318,64320,64321,64323,64324,64326,64433,64467,64829,64848,64911,64914,64967,65008,65020,65136,65140,65142,65276],N=function(t){return O(t,z)},U=[65,90,97,122,170,170,181,181,186,186,192,214,216,246,248,544,546,563,592,685,688,696,699,705,720,721,736,740,750,750,890,890,902,902,904,906,908,908,910,929,931,974,976,1013,1024,1154,1162,1230,1232,1269,1272,1273,1280,1295,1329,1366,1369,1375,1377,1415,1417,1417,2307,2307,2309,2361,2365,2368,2377,2380,2384,2384,2392,2401,2404,2416,2434,2435,2437,2444,2447,2448,2451,2472,2474,2480,2482,2482,2486,2489,2494,2496,2503,2504,2507,2508,2519,2519,2524,2525,2527,2529,2534,2545,2548,2554,2565,2570,2575,2576,2579,2600,2602,2608,2610,2611,2613,2614,2616,2617,2622,2624,2649,2652,2654,2654,2662,2671,2674,2676,2691,2691,2693,2699,2701,2701,2703,2705,2707,2728,2730,2736,2738,2739,2741,2745,2749,2752,2761,2761,2763,2764,2768,2768,2784,2784,2790,2799,2818,2819,2821,2828,2831,2832,2835,2856,2858,2864,2866,2867,2870,2873,2877,2878,2880,2880,2887,2888,2891,2892,2903,2903,2908,2909,2911,2913,2918,2928,2947,2947,2949,2954,2958,2960,2962,2965,2969,2970,2972,2972,2974,2975,2979,2980,2984,2986,2990,2997,2999,3001,3006,3007,3009,3010,3014,3016,3018,3020,3031,3031,3047,3058,3073,3075,3077,3084,3086,3088,3090,3112,3114,3123,3125,3129,3137,3140,3168,3169,3174,3183,3202,3203,3205,3212,3214,3216,3218,3240,3242,3251,3253,3257,3262,3262,3264,3268,3271,3272,3274,3275,3285,3286,3294,3294,3296,3297,3302,3311,3330,3331,3333,3340,3342,3344,3346,3368,3370,3385,3390,3392,3398,3400,3402,3404,3415,3415,3424,3425,3430,3439,3458,3459,3461,3478,3482,3505,3507,3515,3517,3517,3520,3526,3535,3537,3544,3551,3570,3572,3585,3632,3634,3635,3648,3654,3663,3675,3713,3714,3716,3716,3719,3720,3722,3722,3725,3725,3732,3735,3737,3743,3745,3747,3749,3749,3751,3751,3754,3755,3757,3760,3762,3763,3773,3773,3776,3780,3782,3782,3792,3801,3804,3805,3840,3863,3866,3892,3894,3894,3896,3896,3902,3911,3913,3946,3967,3967,3973,3973,3976,3979,4030,4037,4039,4044,4047,4047,4096,4129,4131,4135,4137,4138,4140,4140,4145,4145,4152,4152,4160,4183,4256,4293,4304,4344,4347,4347,4352,4441,4447,4514,4520,4601,4608,4614,4616,4678,4680,4680,4682,4685,4688,4694,4696,4696,4698,4701,4704,4742,4744,4744,4746,4749,4752,4782,4784,4784,4786,4789,4792,4798,4800,4800,4802,4805,4808,4814,4816,4822,4824,4846,4848,4878,4880,4880,4882,4885,4888,4894,4896,4934,4936,4954,4961,4988,5024,5108,5121,5750,5761,5786,5792,5872,5888,5900,5902,5905,5920,5937,5941,5942,5952,5969,5984,5996,5998,6e3,6016,6070,6078,6085,6087,6088,6100,6106,6108,6108,6112,6121,6160,6169,6176,6263,6272,6312,7680,7835,7840,7929,7936,7957,7960,7965,7968,8005,8008,8013,8016,8023,8025,8025,8027,8027,8029,8029,8031,8061,8064,8116,8118,8124,8126,8126,8130,8132,8134,8140,8144,8147,8150,8155,8160,8172,8178,8180,8182,8188,8206,8206,8305,8305,8319,8319,8450,8450,8455,8455,8458,8467,8469,8469,8473,8477,8484,8484,8486,8486,8488,8488,8490,8493,8495,8497,8499,8505,8509,8511,8517,8521,8544,8579,9014,9082,9109,9109,9372,9449,12293,12295,12321,12329,12337,12341,12344,12348,12353,12438,12445,12447,12449,12538,12540,12543,12549,12588,12593,12686,12688,12727,12784,12828,12832,12867,12896,12923,12927,12976,12992,13003,13008,13054,13056,13174,13179,13277,13280,13310,13312,19893,19968,40869,40960,42124,44032,55203,55296,64045,64048,64106,64256,64262,64275,64279,65313,65338,65345,65370,65382,65470,65474,65479,65482,65487,65490,65495,65498,65500,66304,66334,66336,66339,66352,66378,66560,66597,66600,66637,118784,119029,119040,119078,119082,119142,119146,119154,119171,119172,119180,119209,119214,119261,119808,119892,119894,119964,119966,119967,119970,119970,119973,119974,119977,119980,119982,119993,119995,119995,119997,12e4,120002,120003,120005,120069,120071,120074,120077,120084,120086,120092,120094,120121,120123,120126,120128,120132,120134,120134,120138,120144,120146,120483,120488,120777,131072,173782,194560,195101,983040,1048573,1048576,1114109],W=function(t){return O(t,U)},H=function(t){return O(t,L)},V=function(t){return O(t,B)},q=function(t){return t.codePointAt(0)},Z=function(t){return t[0]},Y=function(t){return t[t.length-1]};function X(t){for(var e=[],n=t.length,r=0;r<n;r+=1){var i=t.charCodeAt(r);if(55296<=i&&i<=56319&&r+1<n){var o=t.charCodeAt(r+1);if(56320<=o&&o<=57343){e.push(1024*(i-55296)+o-56320+65536),r+=1;continue}}e.push(i)}return e}var K=function(){function c(t){var e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:{};if(u(this,c),!e.ownerPassword&&!e.userPassword)throw new Error("None of owner password and user password is defined.");this.document=t,this._setupEncryption(e)}return f(c,null,[{key:"generateFileID",value:function(t){var e=0<arguments.length&&void 0!==t?t:{},n="".concat(e.CreationDate.getTime(),"\n");for(var r in e)e.hasOwnProperty(r)&&(n+="".concat(r,": ").concat(e[r].toString(),"\n"));return et(d.default.MD5(n))}},{key:"generateRandomWordArray",value:function(t){return d.default.lib.WordArray.random(t)}},{key:"create",value:function(t,e){var n=1<arguments.length&&void 0!==e?e:{};return n.ownerPassword||n.userPassword?new c(t,n):null}}]),f(c,[{key:"_setupEncryption",value:function(t){switch(t.pdfVersion){case"1.4":case"1.5":this.version=2;break;case"1.6":case"1.7":this.version=4;break;case"1.7ext3":this.version=5;break;default:this.version=1}var e={Filter:"Standard"};switch(this.version){case 1:case 2:case 4:this._setupEncryptionV1V2V4(this.version,e,t);break;case 5:this._setupEncryptionV5(e,t)}this.dictionary=this.document.ref(e)}},{key:"_setupEncryptionV1V2V4",value:function(t,e,n){var r,i;switch(t){case 1:r=2,this.keyBits=40,i=function(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:{},e=-64;t.printing&&(e|=4);t.modifying&&(e|=8);t.copying&&(e|=16);t.annotating&&(e|=32);return e}(n.permissions);break;case 2:r=3,this.keyBits=128,i=J(n.permissions);break;case 4:r=4,this.keyBits=128,i=J(n.permissions)}var o,a=Q(n.userPassword),s=n.ownerPassword?Q(n.ownerPassword):a,l=function(t,e,n,r){for(var i=r,o=3<=t?51:1,a=0;a<o;a++)i=d.default.MD5(i);var s=i.clone();s.sigBytes=e/8;var l=n;o=3<=t?20:1;for(var u=0;u<o;u++){for(var c=Math.ceil(s.sigBytes/4),f=0;f<c;f++)s.words[f]=i.words[f]^(u|u<<8|u<<16|u<<24);l=d.default.RC4.encrypt(l,s).ciphertext}return l}(r,this.keyBits,a,s);this.encryptionKey=function(t,e,n,r,i,o){for(var a=r.clone().concat(i).concat(d.default.lib.WordArray.create([tt(o)],4)).concat(d.default.lib.WordArray.create(n)),s=3<=t?51:1,l=0;l<s;l++)(a=d.default.MD5(a)).sigBytes=e/8;return a}(r,this.keyBits,this.document._id,a,l,i),o=2===r?function(t){return d.default.RC4.encrypt(Q(),t).ciphertext}(this.encryptionKey):function(t,e){for(var n=e.clone(),r=d.default.MD5(Q().concat(d.default.lib.WordArray.create(t))),i=0;i<20;i++){for(var o=Math.ceil(n.sigBytes/4),a=0;a<o;a++)n.words[a]=e.words[a]^(i|i<<8|i<<16|i<<24);r=d.default.RC4.encrypt(r,n).ciphertext}return r.concat(d.default.lib.WordArray.create(null,16))}(this.document._id,this.encryptionKey),2<=(e.V=t)&&(e.Length=this.keyBits),4===t&&(e.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV2",Length:this.keyBits/8}},e.StmF="StdCF",e.StrF="StdCF"),e.R=r,e.O=et(l),e.U=et(o),e.P=i}},{key:"_setupEncryptionV5",value:function(t,e){this.keyBits=256;var n=J(e),r=$(e.userPassword),i=e.ownerPassword?$(e.ownerPassword):r;this.encryptionKey=function(t){return t(32)}(c.generateRandomWordArray);var o=function(t,e){var n=e(8),r=e(8);return d.default.SHA256(t.clone().concat(n)).concat(n).concat(r)}(r,c.generateRandomWordArray),a=function(t,e,n){var r=d.default.SHA256(t.clone().concat(e)),i={mode:d.default.mode.CBC,padding:d.default.pad.NoPadding,iv:d.default.lib.WordArray.create(null,16)};return d.default.AES.encrypt(n,r,i).ciphertext}(r,d.default.lib.WordArray.create(o.words.slice(10,12),8),this.encryptionKey),s=function(t,e,n){var r=n(8),i=n(8);return d.default.SHA256(t.clone().concat(r).concat(e)).concat(r).concat(i)}(i,o,c.generateRandomWordArray),l=function(t,e,n,r){var i=d.default.SHA256(t.clone().concat(e).concat(n)),o={mode:d.default.mode.CBC,padding:d.default.pad.NoPadding,iv:d.default.lib.WordArray.create(null,16)};return d.default.AES.encrypt(r,i,o).ciphertext}(i,d.default.lib.WordArray.create(s.words.slice(10,12),8),o,this.encryptionKey),u=function(t,e,n){var r=d.default.lib.WordArray.create([tt(t),4294967295,1415668834],12).concat(n(4)),i={mode:d.default.mode.ECB,padding:d.default.pad.NoPadding};return d.default.AES.encrypt(r,e,i).ciphertext}(n,this.encryptionKey,c.generateRandomWordArray);t.V=5,t.Length=this.keyBits,t.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV3",Length:this.keyBits/8}},t.StmF="StdCF",t.StrF="StdCF",t.R=5,t.O=et(s),t.OE=et(l),t.U=et(o),t.UE=et(a),t.P=n,t.Perms=et(u)}},{key:"getEncryptFn",value:function(t,e){var n,r;if(this.version<5&&(n=this.encryptionKey.clone().concat(d.default.lib.WordArray.create([(255&t)<<24|(65280&t)<<8|t>>8&65280|255&e,(65280&e)<<16],5))),1===this.version||2===this.version){var i=d.default.MD5(n);return i.sigBytes=Math.min(16,this.keyBits/8+5),function(t){return et(d.default.RC4.encrypt(d.default.lib.WordArray.create(t),i).ciphertext)}}r=4===this.version?d.default.MD5(n.concat(d.default.lib.WordArray.create([1933667412],4))):this.encryptionKey;var o=c.generateRandomWordArray(16),a={mode:d.default.mode.CBC,padding:d.default.pad.Pkcs7,iv:o};return function(t){return et(o.clone().concat(d.default.AES.encrypt(d.default.lib.WordArray.create(t),r,a).ciphertext))}}},{key:"end",value:function(){this.dictionary.end()}}]),c}();function J(t){var e=0<arguments.length&&void 0!==t?t:{},n=-3904;return"lowResolution"===e.printing&&(n|=4),"highResolution"===e.printing&&(n|=2052),e.modifying&&(n|=8),e.copying&&(n|=16),e.annotating&&(n|=32),e.fillingForms&&(n|=256),e.contentAccessibility&&(n|=512),e.documentAssembly&&(n|=1024),n}function Q(t){for(var e=0<arguments.length&&void 0!==t?t:"",n=new p(32),r=e.length,i=0;i<r&&i<32;){var o=e.charCodeAt(i);if(255<o)throw new Error("Password contains one or more invalid characters.");n[i]=o,i++}for(;i<32;)n[i]=lt[i-r],i++;return d.default.lib.WordArray.create(n)}function $(t){var e=0<arguments.length&&void 0!==t?t:"";e=unescape(encodeURIComponent(function(t,e){var n=1<arguments.length&&void 0!==e?e:{};if("string"!=typeof t)throw new TypeError("Expected string.");if(0===t.length)return"";var r=X(t).map(function(t){return H(t)?32:t}).filter(function(t){return!V(t)}),i=String.fromCodePoint.apply(null,r).normalize("NFKC"),o=X(i);if(o.some(D))throw new Error("Prohibited character, see https://tools.ietf.org/html/rfc4013#section-2.3");if(!0!==n.allowUnassigned&&o.some(I))throw new Error("Unassigned code point, see https://tools.ietf.org/html/rfc4013#section-2.5");var a=o.some(N),s=o.some(W);if(a&&s)throw new Error("String must not contain RandALCat and LCat at the same time, see https://tools.ietf.org/html/rfc3454#section-6");var l=N(q(Z(i))),u=N(q(Y(i)));if(a&&(!l||!u))throw new Error("Bidirectional RandALCat character must be the first and the last character of the string, see https://tools.ietf.org/html/rfc3454#section-6");return i}(e)));for(var n=Math.min(127,e.length),r=new p(n),i=0;i<n;i++)r[i]=e.charCodeAt(i);return d.default.lib.WordArray.create(r)}function tt(t){return(255&t)<<24|(65280&t)<<8|t>>8&65280|t>>24&255}function et(t){for(var e=[],n=0;n<t.sigBytes;n++)e.push(t.words[Math.floor(n/4)]>>8*(3-n%4)&255);return p.from(e)}var nt,rt,it,ot,at,st,lt=[40,191,78,94,78,117,138,65,100,0,78,86,255,250,1,8,46,46,0,182,208,104,62,128,47,12,169,254,100,83,105,122],ut=S.number,ct=function(){function e(t){u(this,e),this.doc=t,this.stops=[],this.embedded=!1,this.transform=[1,0,0,1,0,0]}return f(e,[{key:"stop",value:function(t,e,n){if(null==n&&(n=1),e=this.doc._normalizeColor(e),0===this.stops.length)if(3===e.length)this._colorSpace="DeviceRGB";else if(4===e.length)this._colorSpace="DeviceCMYK";else{if(1!==e.length)throw new Error("Unknown color space");this._colorSpace="DeviceGray"}else if("DeviceRGB"===this._colorSpace&&3!==e.length||"DeviceCMYK"===this._colorSpace&&4!==e.length||"DeviceGray"===this._colorSpace&&1!==e.length)throw new Error("All gradient stops must use the same color space");return n=Math.max(0,Math.min(1,n)),this.stops.push([t,e,n]),this}},{key:"setTransform",value:function(t,e,n,r,i,o){return this.transform=[t,e,n,r,i,o],this}},{key:"embed",value:function(t){var e;if(0!==this.stops.length){this.embedded=!0,this.matrix=t;var n=this.stops[this.stops.length-1];n[0]<1&&this.stops.push([1,n[1],n[2]]);for(var r=[],i=[],o=[],a=0,s=this.stops.length-1;a<s;a++)i.push(0,1),a+2!==s&&r.push(this.stops[a+1][0]),e=this.doc.ref({FunctionType:2,Domain:[0,1],C0:this.stops[a+0][1],C1:this.stops[a+1][1],N:1}),o.push(e),e.end();1===o.length?e=o[0]:(e=this.doc.ref({FunctionType:3,Domain:[0,1],Functions:o,Bounds:r,Encode:i})).end(),this.id="Sh".concat(++this.doc._gradCount);var l=this.shader(e);l.end();var u=this.doc.ref({Type:"Pattern",PatternType:2,Shading:l,Matrix:this.matrix.map(function(t){return ut(t)})});if(u.end(),this.stops.some(function(t){return t[2]<1})){var c=this.opacityGradient(),f=!0,h=!(c._colorSpace="DeviceGray"),d=void 0;try{for(var p,g=this.stops[Symbol.iterator]();!(f=(p=g.next()).done);f=!0){var v=p.value;c.stop(v[0],[v[2]])}}catch(t){h=!0,d=t}finally{try{f||null==g.return||g.return()}finally{if(h)throw d}}c=c.embed(this.matrix);var y=[0,0,this.doc.page.width,this.doc.page.height],b=this.doc.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:y,Group:{Type:"Group",S:"Transparency",CS:"DeviceGray"},Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:c}}});b.write("/Pattern cs /Sh1 scn"),b.end("".concat(y.join(" ")," re f"));var m=this.doc.ref({Type:"ExtGState",SMask:{Type:"Mask",S:"Luminosity",G:b}});m.end();var w=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:y,XStep:y[2],YStep:y[3],Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:u},ExtGState:{Gs1:m}}});w.write("/Gs1 gs /Pattern cs /Sh1 scn"),w.end("".concat(y.join(" ")," re f")),this.doc.page.patterns[this.id]=w}else this.doc.page.patterns[this.id]=u;return u}}},{key:"apply",value:function(t){var e=j(this.doc._ctm,6),n=e[0],r=e[1],i=e[2],o=e[3],a=e[4],s=e[5],l=j(this.transform,6),u=l[0],c=l[1],f=l[2],h=l[3],d=l[4],p=l[5],g=[n*u+i*c,r*u+o*c,n*f+i*h,r*f+o*h,n*d+i*p+a,r*d+o*p+s];return this.embedded&&g.join(" ")===this.matrix.join(" ")||this.embed(g),this.doc.addContent("/".concat(this.id," ").concat(t))}}]),e}(),ft={PDFGradient:ct,PDFLinearGradient:function(){function a(t,e,n,r,i){var o;return u(this,a),(o=b(this,v(a).call(this,t))).x1=e,o.y1=n,o.x2=r,o.y2=i,o}return c(a,ct),f(a,[{key:"shader",value:function(t){return this.doc.ref({ShadingType:2,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.x2,this.y2],Function:t,Extend:[!0,!0]})}},{key:"opacityGradient",value:function(){return new a(this.doc,this.x1,this.y1,this.x2,this.y2)}}]),a}(),PDFRadialGradient:function(){function l(t,e,n,r,i,o,a){var s;return u(this,l),(s=b(this,v(l).call(this,t))).doc=t,s.x1=e,s.y1=n,s.r1=r,s.x2=i,s.y2=o,s.r2=a,s}return c(l,ct),f(l,[{key:"shader",value:function(t){return this.doc.ref({ShadingType:3,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.r1,this.x2,this.y2,this.r2],Function:t,Extend:[!0,!0]})}},{key:"opacityGradient",value:function(){return new l(this.doc,this.x1,this.y1,this.r1,this.x2,this.y2,this.r2)}}]),l}()},ht=ft.PDFGradient,dt=ft.PDFLinearGradient,pt=ft.PDFRadialGradient,gt={initColor:function(){return this._opacityRegistry={},this._opacityCount=0,this._gradCount=0},_normalizeColor:function(t){if(t instanceof ht)return t;if("string"==typeof t)if("#"===t.charAt(0)){4===t.length&&(t=t.replace(/#([0-9A-F])([0-9A-F])([0-9A-F])/i,"#$1$1$2$2$3$3"));var e=parseInt(t.slice(1),16);t=[e>>16,e>>8&255,255&e]}else vt[t]&&(t=vt[t]);return Array.isArray(t)?(3===t.length?t=t.map(function(t){return t/255}):4===t.length&&(t=t.map(function(t){return t/100})),t):null},_setColor:function(t,e){if(!(t=this._normalizeColor(t)))return!1;var n=e?"SCN":"scn";if(t instanceof ht)this._setColorSpace("Pattern",e),t.apply(n);else{var r=4===t.length?"DeviceCMYK":"DeviceRGB";this._setColorSpace(r,e),t=t.join(" "),this.addContent("".concat(t," ").concat(n))}return!0},_setColorSpace:function(t,e){var n=e?"CS":"cs";return this.addContent("/".concat(t," ").concat(n))},fillColor:function(t,e){return this._setColor(t,!1)&&this.fillOpacity(e),this._fillColor=[t,e],this},strokeColor:function(t,e){return this._setColor(t,!0)&&this.strokeOpacity(e),this},opacity:function(t){return this._doOpacity(t,t),this},fillOpacity:function(t){return this._doOpacity(t,null),this},strokeOpacity:function(t){return this._doOpacity(null,t),this},_doOpacity:function(t,e){var n,r;if(null!=t||null!=e){null!=t&&(t=Math.max(0,Math.min(1,t))),null!=e&&(e=Math.max(0,Math.min(1,e)));var i="".concat(t,"_").concat(e);if(this._opacityRegistry[i]){var o=j(this._opacityRegistry[i],2);n=o[0],r=o[1]}else{n={Type:"ExtGState"},null!=t&&(n.ca=t),null!=e&&(n.CA=e),(n=this.ref(n)).end();var a=++this._opacityCount;r="Gs".concat(a),this._opacityRegistry[i]=[n,r]}return this.page.ext_gstates[r]=n,this.addContent("/".concat(r," gs"))}},linearGradient:function(t,e,n,r){return new dt(this,t,e,n,r)},radialGradient:function(t,e,n,r,i,o){return new pt(this,t,e,n,r,i,o)}},vt={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgreen:[0,100,0],darkgrey:[169,169,169],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0],yellowgreen:[154,205,50]};nt=rt=it=ot=at=st=0;function yt(t){return"0000".concat(t.toString(16)).slice(-4)}function bt(t){Object.assign(Xt.prototype,t)}var mt={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},wt={M:function(t,e){return nt=e[0],rt=e[1],it=ot=null,at=nt,st=rt,t.moveTo(nt,rt)},m:function(t,e){return nt+=e[0],rt+=e[1],it=ot=null,at=nt,st=rt,t.moveTo(nt,rt)},C:function(t,e){return nt=e[4],rt=e[5],it=e[2],ot=e[3],t.bezierCurveTo.apply(t,G(e||[]))},c:function(t,e){return t.bezierCurveTo(e[0]+nt,e[1]+rt,e[2]+nt,e[3]+rt,e[4]+nt,e[5]+rt),it=nt+e[2],ot=rt+e[3],nt+=e[4],rt+=e[5]},S:function(t,e){return null===it&&(it=nt,ot=rt),t.bezierCurveTo(nt-(it-nt),rt-(ot-rt),e[0],e[1],e[2],e[3]),it=e[0],ot=e[1],nt=e[2],rt=e[3]},s:function(t,e){return null===it&&(it=nt,ot=rt),t.bezierCurveTo(nt-(it-nt),rt-(ot-rt),nt+e[0],rt+e[1],nt+e[2],rt+e[3]),it=nt+e[0],ot=rt+e[1],nt+=e[2],rt+=e[3]},Q:function(t,e){return it=e[0],ot=e[1],nt=e[2],rt=e[3],t.quadraticCurveTo(e[0],e[1],nt,rt)},q:function(t,e){return t.quadraticCurveTo(e[0]+nt,e[1]+rt,e[2]+nt,e[3]+rt),it=nt+e[0],ot=rt+e[1],nt+=e[2],rt+=e[3]},T:function(t,e){return ot=null===it?(it=nt,rt):(it=nt-(it-nt),rt-(ot-rt)),t.quadraticCurveTo(it,ot,e[0],e[1]),it=nt-(it-nt),ot=rt-(ot-rt),nt=e[0],rt=e[1]},t:function(t,e){return ot=null===it?(it=nt,rt):(it=nt-(it-nt),rt-(ot-rt)),t.quadraticCurveTo(it,ot,nt+e[0],rt+e[1]),nt+=e[0],rt+=e[1]},A:function(t,e){return xt(t,nt,rt,e),nt=e[5],rt=e[6]},a:function(t,e){return e[5]+=nt,e[6]+=rt,xt(t,nt,rt,e),nt=e[5],rt=e[6]},L:function(t,e){return nt=e[0],rt=e[1],it=ot=null,t.lineTo(nt,rt)},l:function(t,e){return nt+=e[0],rt+=e[1],it=ot=null,t.lineTo(nt,rt)},H:function(t,e){return nt=e[0],it=ot=null,t.lineTo(nt,rt)},h:function(t,e){return nt+=e[0],it=ot=null,t.lineTo(nt,rt)},V:function(t,e){return rt=e[0],it=ot=null,t.lineTo(nt,rt)},v:function(t,e){return rt+=e[0],it=ot=null,t.lineTo(nt,rt)},Z:function(t){return t.closePath(),nt=at,rt=st},z:function(t){return t.closePath(),nt=at,rt=st}},xt=function(t,e,n,r){var i=j(r,7),o=i[0],a=i[1],s=i[2],l=i[3],u=i[4],c=i[5],f=i[6],h=_t(c,f,o,a,l,u,s,e,n),d=!0,p=!1,g=void 0;try{for(var v,y=h[Symbol.iterator]();!(d=(v=y.next()).done);d=!0){var b=v.value,m=kt.apply(void 0,G(b||[]));t.bezierCurveTo.apply(t,G(m||[]))}}catch(t){p=!0,g=t}finally{try{d||null==y.return||y.return()}finally{if(p)throw g}}},_t=function(t,e,n,r,i,o,a,s,l){var u=a*(Math.PI/180),c=Math.sin(u),f=Math.cos(u);n=Math.abs(n),r=Math.abs(r);var h=(it=f*(s-t)*.5+c*(l-e)*.5)*it/(n*n)+(ot=f*(l-e)*.5-c*(s-t)*.5)*ot/(r*r);1<h&&(n*=h=Math.sqrt(h),r*=h);var d=f/n,p=c/n,g=-c/r,v=f/r,y=d*s+p*l,b=g*s+v*l,m=d*t+p*e,w=g*t+v*e,x=1/((m-y)*(m-y)+(w-b)*(w-b))-.25;x<0&&(x=0);var _=Math.sqrt(x);o===i&&(_=-_);var k=.5*(y+m)-_*(w-b),S=.5*(b+w)+_*(m-y),C=Math.atan2(b-S,y-k),A=Math.atan2(w-S,m-k)-C;A<0&&1===o?A+=2*Math.PI:0<A&&0===o&&(A-=2*Math.PI);for(var P=Math.ceil(Math.abs(A/(.5*Math.PI+.001))),E=[],O=0;O<P;O++){var T=C+O*A/P,I=C+(O+1)*A/P;E[O]=[k,S,T,I,n,r,c,f]}return E},kt=function(t,e,n,r,i,o,a,s){var l=s*i,u=-a*o,c=a*i,f=s*o,h=.5*(r-n),d=8/3*Math.sin(.5*h)*Math.sin(.5*h)/Math.sin(h),p=t+Math.cos(n)-d*Math.sin(n),g=e+Math.sin(n)+d*Math.cos(n),v=t+Math.cos(r),y=e+Math.sin(r),b=v+d*Math.sin(r),m=y-d*Math.cos(r);return[l*p+u*g,c*p+f*g,l*b+u*m,c*b+f*m,l*v+u*y,c*v+f*y]},St=function(){function t(){u(this,t)}return f(t,null,[{key:"apply",value:function(t,e){!function(t,e){nt=rt=it=ot=at=st=0;for(var n=0;n<t.length;n++){var r=t[n];"function"==typeof wt[r.cmd]&&wt[r.cmd](e,r.args)}}(function(t){var e,n=[],r=[],i="",o=!1,a=0,s=!0,l=!1,u=void 0;try{for(var c,f=t[Symbol.iterator]();!(s=(c=f.next()).done);s=!0){var h=c.value;if(null!=mt[h])a=mt[h],e&&(0<i.length&&(r[r.length]=+i),n[n.length]={cmd:e,args:r},i="",o=!(r=[])),e=h;else if([" ",","].includes(h)||"-"===h&&0<i.length&&"e"!==i[i.length-1]||"."===h&&o){if(0===i.length)continue;r.length===a?(n[n.length]={cmd:e,args:r},r=[+i],"M"===e&&(e="L"),"m"===e&&(e="l")):r[r.length]=+i,o="."===h,i=["-","."].includes(h)?h:""}else i+=h,"."===h&&(o=!0)}}catch(t){l=!0,u=t}finally{try{s||null==f.return||f.return()}finally{if(l)throw u}}return 0<i.length&&(r.length===a?(n[n.length]={cmd:e,args:r},r=[+i],"M"===e&&(e="L"),"m"===e&&(e="l")):r[r.length]=+i),n[n.length]={cmd:e,args:r},n}(e),t)}}]),t}(),Ct=S.number,At=(Math.sqrt(2)-1)/3*4,Pt={initVector:function(){return this._ctm=[1,0,0,1,0,0],this._ctmStack=[]},save:function(){return this._ctmStack.push(this._ctm.slice()),this.addContent("q")},restore:function(){return this._ctm=this._ctmStack.pop()||[1,0,0,1,0,0],this.addContent("Q")},closePath:function(){return this.addContent("h")},lineWidth:function(t){return this.addContent("".concat(Ct(t)," w"))},_CAP_STYLES:{BUTT:0,ROUND:1,SQUARE:2},lineCap:function(t){return"string"==typeof t&&(t=this._CAP_STYLES[t.toUpperCase()]),this.addContent("".concat(t," J"))},_JOIN_STYLES:{MITER:0,ROUND:1,BEVEL:2},lineJoin:function(t){return"string"==typeof t&&(t=this._JOIN_STYLES[t.toUpperCase()]),this.addContent("".concat(t," j"))},miterLimit:function(t){return this.addContent("".concat(Ct(t)," M"))},dash:function(t,e){var n=1<arguments.length&&void 0!==e?e:{},r=t;if(Array.isArray(t)||(t=[t,n.space||t]),!t.every(function(t){return Number.isFinite(t)&&0<t}))throw new Error("dash(".concat(JSON.stringify(r),", ").concat(JSON.stringify(n),") invalid, lengths must be numeric and greater than zero"));return t=t.map(Ct).join(" "),this.addContent("[".concat(t,"] ").concat(Ct(n.phase||0)," d"))},undash:function(){return this.addContent("[] 0 d")},moveTo:function(t,e){return this.addContent("".concat(Ct(t)," ").concat(Ct(e)," m"))},lineTo:function(t,e){return this.addContent("".concat(Ct(t)," ").concat(Ct(e)," l"))},bezierCurveTo:function(t,e,n,r,i,o){return this.addContent("".concat(Ct(t)," ").concat(Ct(e)," ").concat(Ct(n)," ").concat(Ct(r)," ").concat(Ct(i)," ").concat(Ct(o)," c"))},quadraticCurveTo:function(t,e,n,r){return this.addContent("".concat(Ct(t)," ").concat(Ct(e)," ").concat(Ct(n)," ").concat(Ct(r)," v"))},rect:function(t,e,n,r){return this.addContent("".concat(Ct(t)," ").concat(Ct(e)," ").concat(Ct(n)," ").concat(Ct(r)," re"))},roundedRect:function(t,e,n,r,i){null==i&&(i=0);var o=(i=Math.min(i,.5*n,.5*r))*(1-At);return this.moveTo(t+i,e),this.lineTo(t+n-i,e),this.bezierCurveTo(t+n-o,e,t+n,e+o,t+n,e+i),this.lineTo(t+n,e+r-i),this.bezierCurveTo(t+n,e+r-o,t+n-o,e+r,t+n-i,e+r),this.lineTo(t+i,e+r),this.bezierCurveTo(t+o,e+r,t,e+r-o,t,e+r-i),this.lineTo(t,e+i),this.bezierCurveTo(t,e+o,t+o,e,t+i,e),this.closePath()},ellipse:function(t,e,n,r){null==r&&(r=n);var i=n*At,o=r*At,a=(t-=n)+2*n,s=(e-=r)+2*r,l=t+n,u=e+r;return this.moveTo(t,u),this.bezierCurveTo(t,u-o,l-i,e,l,e),this.bezierCurveTo(l+i,e,a,u-o,a,u),this.bezierCurveTo(a,u+o,l+i,s,l,s),this.bezierCurveTo(l-i,s,t,u+o,t,u),this.closePath()},circle:function(t,e,n){return this.ellipse(t,e,n)},arc:function(t,e,n,r,i,o){null==o&&(o=!1);var a=2*Math.PI,s=.5*Math.PI,l=i-r;if(Math.abs(l)>a)l=a;else if(0!==l&&o!==l<0){l=(o?-1:1)*a+l}var u=Math.ceil(Math.abs(l)/s),c=l/u,f=c/s*At*n,h=r,d=-Math.sin(h)*f,p=Math.cos(h)*f,g=t+Math.cos(h)*n,v=e+Math.sin(h)*n;this.moveTo(g,v);for(var y=0;y<u;y++){var b=g+d,m=v+p;h+=c,g=t+Math.cos(h)*n,v=e+Math.sin(h)*n;var w=g-(d=-Math.sin(h)*f),x=v-(p=Math.cos(h)*f);this.bezierCurveTo(b,m,w,x,g,v)}return this},polygon:function(){for(var t=arguments.length,e=new Array(t),n=0;n<t;n++)e[n]=arguments[n];this.moveTo.apply(this,G(e.shift()||[]));for(var r=0,i=e;r<i.length;r++){var o=i[r];this.lineTo.apply(this,G(o||[]))}return this.closePath()},path:function(t){return St.apply(this,t),this},_windingRule:function(t){return/even-?odd/.test(t)?"*":""},fill:function(t,e){return/(even-?odd)|(non-?zero)/.test(t)&&(e=t,t=null),t&&this.fillColor(t),this.addContent("f".concat(this._windingRule(e)))},stroke:function(t){return t&&this.strokeColor(t),this.addContent("S")},fillAndStroke:function(t,e,n){null==e&&(e=t);var r=/(even-?odd)|(non-?zero)/;return r.test(t)&&(n=t,t=null),r.test(e)&&(n=e,e=t),t&&(this.fillColor(t),this.strokeColor(e)),this.addContent("B".concat(this._windingRule(n)))},clip:function(t){return this.addContent("W".concat(this._windingRule(t)," n"))},transform:function(t,e,n,r,i,o){var a=this._ctm,s=j(a,6),l=s[0],u=s[1],c=s[2],f=s[3],h=s[4],d=s[5];a[0]=l*t+c*e,a[1]=u*t+f*e,a[2]=l*n+c*r,a[3]=u*n+f*r,a[4]=l*i+c*o+h,a[5]=u*i+f*o+d;var p=[t,e,n,r,i,o].map(function(t){return Ct(t)}).join(" ");return this.addContent("".concat(p," cm"))},translate:function(t,e){return this.transform(1,0,0,1,t,e)},rotate:function(t,e){var n,r=1<arguments.length&&void 0!==e?e:{},i=t*Math.PI/180,o=Math.cos(i),a=Math.sin(i),s=n=0;if(null!=r.origin){var l=j(r.origin,2),u=(s=l[0])*a+(n=l[1])*o;s-=s*o-n*a,n-=u}return this.transform(o,a,-a,o,s,n)},scale:function(t,e,n){var r,i=2<arguments.length&&void 0!==n?n:{};null==e&&(e=t),"object"==typeof e&&(i=e,e=t);var o=r=0;if(null!=i.origin){var a=j(i.origin,2);o=a[0],r=a[1],o-=t*o,r-=e*r}return this.transform(t,0,0,e,o,r)}},Et={402:131,8211:150,8212:151,8216:145,8217:146,8218:130,8220:147,8221:148,8222:132,8224:134,8225:135,8226:149,8230:133,8364:128,8240:137,8249:139,8250:155,710:136,8482:153,338:140,339:156,732:152,352:138,353:154,376:159,381:142,382:158},Ot=".notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n  \nspace         exclam         quotedbl       numbersign\ndollar        percent        ampersand      quotesingle\nparenleft     parenright     asterisk       plus\ncomma         hyphen         period         slash\nzero          one            two            three\nfour          five           six            seven\neight         nine           colon          semicolon\nless          equal          greater        question\n  \nat            A              B              C\nD             E              F              G\nH             I              J              K\nL             M              N              O\nP             Q              R              S\nT             U              V              W\nX             Y              Z              bracketleft\nbackslash     bracketright   asciicircum    underscore\n  \ngrave         a              b              c\nd             e              f              g\nh             i              j              k\nl             m              n              o\np             q              r              s\nt             u              v              w\nx             y              z              braceleft\nbar           braceright     asciitilde     .notdef\n  \nEuro          .notdef        quotesinglbase florin\nquotedblbase  ellipsis       dagger         daggerdbl\ncircumflex    perthousand    Scaron         guilsinglleft\nOE            .notdef        Zcaron         .notdef\n.notdef       quoteleft      quoteright     quotedblleft\nquotedblright bullet         endash         emdash\ntilde         trademark      scaron         guilsinglright\noe            .notdef        zcaron         ydieresis\n  \nspace         exclamdown     cent           sterling\ncurrency      yen            brokenbar      section\ndieresis      copyright      ordfeminine    guillemotleft\nlogicalnot    hyphen         registered     macron\ndegree        plusminus      twosuperior    threesuperior\nacute         mu             paragraph      periodcentered\ncedilla       onesuperior    ordmasculine   guillemotright\nonequarter    onehalf        threequarters  questiondown\n  \nAgrave        Aacute         Acircumflex    Atilde\nAdieresis     Aring          AE             Ccedilla\nEgrave        Eacute         Ecircumflex    Edieresis\nIgrave        Iacute         Icircumflex    Idieresis\nEth           Ntilde         Ograve         Oacute\nOcircumflex   Otilde         Odieresis      multiply\nOslash        Ugrave         Uacute         Ucircumflex\nUdieresis     Yacute         Thorn          germandbls\n  \nagrave        aacute         acircumflex    atilde\nadieresis     aring          ae             ccedilla\negrave        eacute         ecircumflex    edieresis\nigrave        iacute         icircumflex    idieresis\neth           ntilde         ograve         oacute\nocircumflex   otilde         odieresis      divide\noslash        ugrave         uacute         ucircumflex\nudieresis     yacute         thorn          ydieresis".split(/\s+/),Tt=function(){function n(t){u(this,n),this.contents=t,this.attributes={},this.glyphWidths={},this.boundingBoxes={},this.kernPairs={},this.parse(),this.charWidths=new Array(256);for(var e=0;e<=255;e++)this.charWidths[e]=this.glyphWidths[Ot[e]];this.bbox=this.attributes.FontBBox.split(/\s+/).map(function(t){return+t}),this.ascender=+(this.attributes.Ascender||0),this.descender=+(this.attributes.Descender||0),this.xHeight=+(this.attributes.XHeight||0),this.capHeight=+(this.attributes.CapHeight||0),this.lineGap=this.bbox[3]-this.bbox[1]-(this.ascender-this.descender)}return f(n,null,[{key:"open",value:function(t){return new n(s.readFileSync(t,"utf8"))}}]),f(n,[{key:"parse",value:function(){var t="",e=!0,n=!1,r=void 0;try{for(var i,o=this.contents.split("\n")[Symbol.iterator]();!(e=(i=o.next()).done);e=!0){var a,s,l=i.value;if(a=l.match(/^Start(\w+)/))t=a[1];else if(a=l.match(/^End(\w+)/))t="";else switch(t){case"FontMetrics":var u=(a=l.match(/(^\w+)\s+(.*)/))[1],c=a[2];(s=this.attributes[u])?(Array.isArray(s)||(s=this.attributes[u]=[s]),s.push(c)):this.attributes[u]=c;break;case"CharMetrics":if(!/^CH?\s/.test(l))continue;var f=l.match(/\bN\s+(\.?\w+)\s*;/)[1];this.glyphWidths[f]=+l.match(/\bWX\s+(\d+)\s*;/)[1];break;case"KernPairs":(a=l.match(/^KPX\s+(\.?\w+)\s+(\.?\w+)\s+(-?\d+)/))&&(this.kernPairs[a[1]+"\0"+a[2]]=parseInt(a[3]))}}}catch(t){n=!0,r=t}finally{try{e||null==o.return||o.return()}finally{if(n)throw r}}}},{key:"encodeText",value:function(t){for(var e=[],n=0,r=t.length;n<r;n++){var i=t.charCodeAt(n);i=Et[i]||i,e.push(i.toString(16))}return e}},{key:"glyphsForString",value:function(t){for(var e=[],n=0,r=t.length;n<r;n++){var i=t.charCodeAt(n);e.push(this.characterToGlyph(i))}return e}},{key:"characterToGlyph",value:function(t){return Ot[Et[t]||t]||".notdef"}},{key:"widthOfGlyph",value:function(t){return this.glyphWidths[t]||0}},{key:"getKernPair",value:function(t,e){return this.kernPairs[t+"\0"+e]||0}},{key:"advancesForGlyphs",value:function(t){for(var e=[],n=0;n<t.length;n++){var r=t[n],i=t[n+1];e.push(this.widthOfGlyph(r)+this.getKernPair(r,i))}return e}}]),n}(),It=function(){function t(){u(this,t)}return f(t,[{key:"encode",value:function(){throw new Error("Must be implemented by subclasses")}},{key:"widthOfString",value:function(){throw new Error("Must be implemented by subclasses")}},{key:"ref",value:function(){return null!=this.dictionary?this.dictionary:this.dictionary=this.document.ref()}},{key:"finalize",value:function(){if(!this.embedded&&null!=this.dictionary)return this.embed(),this.embedded=!0}},{key:"embed",value:function(){throw new Error("Must be implemented by subclasses")}},{key:"lineHeight",value:function(t,e){null==e&&(e=!1);var n=e?this.lineGap:0;return(this.ascender+n-this.descender)/1e3*t}}]),t}(),Bt={Courier:function(){return s.readFileSync(t+"/data/Courier.afm","utf8")},"Courier-Bold":function(){return s.readFileSync(t+"/data/Courier-Bold.afm","utf8")},"Courier-Oblique":function(){return s.readFileSync(t+"/data/Courier-Oblique.afm","utf8")},"Courier-BoldOblique":function(){return s.readFileSync(t+"/data/Courier-BoldOblique.afm","utf8")},Helvetica:function(){return s.readFileSync(t+"/data/Helvetica.afm","utf8")},"Helvetica-Bold":function(){return s.readFileSync(t+"/data/Helvetica-Bold.afm","utf8")},"Helvetica-Oblique":function(){return s.readFileSync(t+"/data/Helvetica-Oblique.afm","utf8")},"Helvetica-BoldOblique":function(){return s.readFileSync(t+"/data/Helvetica-BoldOblique.afm","utf8")},"Times-Roman":function(){return s.readFileSync(t+"/data/Times-Roman.afm","utf8")},"Times-Bold":function(){return s.readFileSync(t+"/data/Times-Bold.afm","utf8")},"Times-Italic":function(){return s.readFileSync(t+"/data/Times-Italic.afm","utf8")},"Times-BoldItalic":function(){return s.readFileSync(t+"/data/Times-BoldItalic.afm","utf8")},Symbol:function(){return s.readFileSync(t+"/data/Symbol.afm","utf8")},ZapfDingbats:function(){return s.readFileSync(t+"/data/ZapfDingbats.afm","utf8")}},Lt=function(){function o(t,e,n){var r;u(this,o),(r=b(this,v(o).call(this))).document=t,r.name=e,r.id=n,r.font=new Tt(Bt[r.name]());var i=r.font;return r.ascender=i.ascender,r.descender=i.descender,r.bbox=i.bbox,r.lineGap=i.lineGap,r.xHeight=i.xHeight,r.capHeight=i.capHeight,r}return c(o,It),f(o,[{key:"embed",value:function(){return this.dictionary.data={Type:"Font",BaseFont:this.name,Subtype:"Type1",Encoding:"WinAnsiEncoding"},this.dictionary.end()}},{key:"encode",value:function(t){for(var e=this.font.encodeText(t),n=this.font.glyphsForString("".concat(t)),r=this.font.advancesForGlyphs(n),i=[],o=0;o<n.length;o++){var a=n[o];i.push({xAdvance:r[o],yAdvance:0,xOffset:0,yOffset:0,advanceWidth:this.font.widthOfGlyph(a)})}return[e,i]}},{key:"widthOfString",value:function(t,e){var n=this.font.glyphsForString("".concat(t)),r=this.font.advancesForGlyphs(n),i=0,o=!0,a=!1,s=void 0;try{for(var l,u=r[Symbol.iterator]();!(o=(l=u.next()).done);o=!0){i+=l.value}}catch(t){a=!0,s=t}finally{try{o||null==u.return||u.return()}finally{if(a)throw s}}return i*(e/1e3)}}],[{key:"isStandardFont",value:function(t){return t in Bt}}]),o}(),Rt=function(){function i(t,e,n){var r;return u(this,i),(r=b(this,v(i).call(this))).document=t,r.font=e,r.id=n,r.subset=r.font.createSubset(),r.unicode=[[0]],r.widths=[r.font.getGlyph(0).advanceWidth],r.name=r.font.postscriptName,r.scale=1e3/r.font.unitsPerEm,r.ascender=r.font.ascent*r.scale,r.descender=r.font.descent*r.scale,r.xHeight=r.font.xHeight*r.scale,r.capHeight=r.font.capHeight*r.scale,r.lineGap=r.font.lineGap*r.scale,r.bbox=r.font.bbox,!1!==t.options.fontLayoutCache&&(r.layoutCache=Object.create(null)),r}return c(i,It),f(i,[{key:"layoutRun",value:function(t,e){for(var n=this.font.layout(t,e),r=0;r<n.positions.length;r++){var i=n.positions[r];for(var o in i)i[o]*=this.scale;i.advanceWidth=n.glyphs[r].advanceWidth*this.scale}return n}},{key:"layoutCached",value:function(t){if(!this.layoutCache)return this.layoutRun(t);var e;if(e=this.layoutCache[t])return e;var n=this.layoutRun(t);return this.layoutCache[t]=n}},{key:"layout",value:function(t,e,n){if(e)return this.layoutRun(t,e);for(var r=n?null:[],i=n?null:[],o=0,a=0,s=0;s<=t.length;){var l;if(s===t.length&&a<s||(l=t.charAt(s),[" ","\t"].includes(l))){var u=this.layoutCached(t.slice(a,++s));n||(r=r.concat(u.glyphs),i=i.concat(u.positions)),o+=u.advanceWidth,a=s}else s++}return{glyphs:r,positions:i,advanceWidth:o}}},{key:"encode",value:function(t,e){for(var n=this.layout(t,e),r=n.glyphs,i=n.positions,o=[],a=0;a<r.length;a++){var s=r[a],l=this.subset.includeGlyph(s.id);o.push("0000".concat(l.toString(16)).slice(-4)),null==this.widths[l]&&(this.widths[l]=s.advanceWidth*this.scale),null==this.unicode[l]&&(this.unicode[l]=s.codePoints)}return[o,i]}},{key:"widthOfString",value:function(t,e,n){return this.layout(t,n,!0).advanceWidth*(e/1e3)}},{key:"embed",value:function(){var e=this,t=null!=this.subset.cff,n=this.document.ref();t&&(n.data.Subtype="CIDFontType0C"),this.subset.encodeStream().on("data",function(t){return n.write(t)}).on("end",function(){return n.end()});var r=((null!=this.font["OS/2"]?this.font["OS/2"].sFamilyClass:void 0)||0)>>8,i=0;this.font.post.isFixedPitch&&(i|=1),1<=r&&r<=7&&(i|=2),i|=4,10==r&&(i|=8),this.font.head.macStyle.italic&&(i|=64);var o=[1,2,3,4,5,6].map(function(t){return String.fromCharCode((e.id.charCodeAt(t)||74)+16)}).join("")+"+"+this.font.postscriptName,a=this.font.bbox,s=this.document.ref({Type:"FontDescriptor",FontName:o,Flags:i,FontBBox:[a.minX*this.scale,a.minY*this.scale,a.maxX*this.scale,a.maxY*this.scale],ItalicAngle:this.font.italicAngle,Ascent:this.ascender,Descent:this.descender,CapHeight:(this.font.capHeight||this.font.ascent)*this.scale,XHeight:(this.font.xHeight||0)*this.scale,StemV:0});t?s.data.FontFile3=n:s.data.FontFile2=n,s.end();var l=this.document.ref({Type:"Font",Subtype:t?"CIDFontType0":"CIDFontType2",BaseFont:o,CIDSystemInfo:{Registry:new String("Adobe"),Ordering:new String("Identity"),Supplement:0},FontDescriptor:s,W:[0,this.widths]});return l.end(),this.dictionary.data={Type:"Font",Subtype:"Type0",BaseFont:o,Encoding:"Identity-H",DescendantFonts:[l],ToUnicode:this.toUnicodeCmap()},this.dictionary.end()}},{key:"toUnicodeCmap",value:function(){var t=this.document.ref(),e=[],n=!0,r=!1,i=void 0;try{for(var o,a=this.unicode[Symbol.iterator]();!(n=(o=a.next()).done);n=!0){var s=o.value,l=[],u=!0,c=!1,f=void 0;try{for(var h,d=s[Symbol.iterator]();!(u=(h=d.next()).done);u=!0){var p=h.value;65535<p&&(p-=65536,l.push(yt(p>>>10&1023|55296)),p=56320|1023&p),l.push(yt(p))}}catch(t){c=!0,f=t}finally{try{u||null==d.return||d.return()}finally{if(c)throw f}}e.push("<".concat(l.join(" "),">"))}}catch(t){r=!0,i=t}finally{try{n||null==a.return||a.return()}finally{if(r)throw i}}return t.end("/CIDInit /ProcSet findresource begin\n12 dict begin\nbegincmap\n/CIDSystemInfo <<\n  /Registry (Adobe)\n  /Ordering (UCS)\n  /Supplement 0\n>> def\n/CMapName /Adobe-Identity-UCS def\n/CMapType 2 def\n1 begincodespacerange\n<0000><ffff>\nendcodespacerange\n1 beginbfrange\n<0000> <".concat(yt(e.length-1),"> [").concat(e.join(" "),"]\nendbfrange\nendcmap\nCMapName currentdict /CMap defineresource pop\nend\nend")),t}}]),i}(),Mt=function(){function t(){u(this,t)}return f(t,null,[{key:"open",value:function(t,e,n,r){var i;if("string"==typeof e){if(Lt.isStandardFont(e))return new Lt(t,e,r);e=s.readFileSync(e)}if(p.isBuffer(e)?i=o.default.create(e,n):e instanceof Uint8Array?i=o.default.create(new p(e),n):e instanceof ArrayBuffer&&(i=o.default.create(new p(new Uint8Array(e)),n)),null==i)throw new Error("Not a supported font format or standard PDF font.");return new Rt(t,i,r)}}]),t}(),Ft={initFonts:function(t){var e=0<arguments.length&&void 0!==t?t:"Helvetica";this._fontFamilies={},this._fontCount=0,this._fontSize=12,this._font=null,this._registeredFonts={},e&&this.font(e)},font:function(t,e,n){var r,i;if("number"==typeof e&&(n=e,e=null),"string"==typeof t&&this._registeredFonts[t]){r=t;var o=this._registeredFonts[t];t=o.src,e=o.family}else"string"!=typeof(r=e||t)&&(r=null);if(null!=n&&this.fontSize(n),i=this._fontFamilies[r])return this._font=i,this;var a="F".concat(++this._fontCount);return this._font=Mt.open(this,t,e,a),(i=this._fontFamilies[this._font.name])?this._font=i:(r&&(this._fontFamilies[r]=this._font),this._font.name&&(this._fontFamilies[this._font.name]=this._font)),this},fontSize:function(t){return this._fontSize=t,this},currentLineHeight:function(t){return null==t&&(t=!1),this._font.lineHeight(this._fontSize,t)},registerFont:function(t,e,n){return this._registeredFonts[t]={src:e,family:n},this}},Dt=function(){function r(t,e){var n;return u(this,r),(n=b(this,v(r).call(this))).document=t,n.indent=e.indent||0,n.characterSpacing=e.characterSpacing||0,n.wordSpacing=0===e.wordSpacing,n.columns=e.columns||1,n.columnGap=null!=e.columnGap?e.columnGap:18,n.lineWidth=(e.width-n.columnGap*(n.columns-1))/n.columns,n.spaceLeft=n.lineWidth,n.startX=n.document.x,n.startY=n.document.y,n.column=1,n.ellipsis=e.ellipsis,n.continuedX=0,n.features=e.features,null!=e.height?(n.height=e.height,n.maxY=n.startY+e.height):n.maxY=n.document.page.maxY(),n.on("firstLine",function(t){var e=n.continuedX||n.indent;return n.document.x+=e,n.lineWidth-=e,n.once("line",function(){if(n.document.x-=e,n.lineWidth+=e,t.continued&&!n.continuedX&&(n.continuedX=n.indent),!t.continued)return n.continuedX=0})}),n.on("lastLine",function(t){var e=t.align;return"justify"===e&&(t.align="left"),n.lastLine=!0,n.once("line",function(){return n.document.y+=t.paragraphGap||0,t.align=e,n.lastLine=!1})}),n}return c(r,n.EventEmitter),f(r,[{key:"wordWidth",value:function(t){return this.document.widthOfString(t,this)+this.characterSpacing+this.wordSpacing}},{key:"eachWord",value:function(t,e){for(var n,r=new g.default(t),i=null,o=Object.create(null);n=r.nextBreak();){var a,s=t.slice((null!=i?i.position:void 0)||0,n.position),l=null!=o[s]?o[s]:o[s]=this.wordWidth(s);if(l>this.lineWidth+this.continuedX)for(var u=i,c={};s.length;){var f,h;l>this.spaceLeft?(f=Math.ceil(this.spaceLeft/(l/s.length)),h=(l=this.wordWidth(s.slice(0,f)))<=this.spaceLeft&&f<s.length):f=s.length;for(var d=l>this.spaceLeft&&0<f;d||h;)d?d=(l=this.wordWidth(s.slice(0,--f)))>this.spaceLeft&&0<f:(d=(l=this.wordWidth(s.slice(0,++f)))>this.spaceLeft&&0<f,h=l<=this.spaceLeft&&f<s.length);if(c.required=n.required||f<s.length,a=e(s.slice(0,f),l,c,u),u={required:!1},s=s.slice(f),l=this.wordWidth(s),!1===a)break}else a=e(s,l,n,i);if(!1===a)break;i=n}}},{key:"wrap",value:function(t,o){var a=this;null!=o.indent&&(this.indent=o.indent),null!=o.characterSpacing&&(this.characterSpacing=o.characterSpacing),null!=o.wordSpacing&&(this.wordSpacing=o.wordSpacing),null!=o.ellipsis&&(this.ellipsis=o.ellipsis);var e=this.document.y+this.document.currentLineHeight(!0);(this.document.y>this.maxY||e>this.maxY)&&this.nextSection();function s(){return o.textWidth=u+a.wordSpacing*(c-1),o.wordCount=c,o.lineWidth=a.lineWidth,r=a.document.y,a.emit("line",l,o,a),n++}var l="",u=0,c=0,n=0,r=this.document.y;return this.emit("sectionStart",o,this),this.eachWord(t,function(t,e,n,r){if(null!=r&&!r.required||(a.emit("firstLine",o,a),a.spaceLeft=a.lineWidth),e<=a.spaceLeft&&(l+=t,u+=e,c++),n.required||e>a.spaceLeft){var i=a.document.currentLineHeight(!0);if(null!=a.height&&a.ellipsis&&a.document.y+2*i>a.maxY&&a.column>=a.columns){for(!0===a.ellipsis&&(a.ellipsis="…"),l=l.replace(/\s+$/,""),u=a.wordWidth(l+a.ellipsis);l&&u>a.lineWidth;)l=l.slice(0,-1).replace(/\s+$/,""),u=a.wordWidth(l+a.ellipsis);u<=a.lineWidth&&(l+=a.ellipsis),u=a.wordWidth(l)}if(n.required&&(e>a.spaceLeft&&(s(),l=t,u=e,c=1),a.emit("lastLine",o,a)),s(),a.document.y+i>a.maxY)if(!a.nextSection())return c=0,l="",!1;return c=n.required?(a.spaceLeft=a.lineWidth,l="",u=0):(a.spaceLeft=a.lineWidth-e,l=t,u=e,1)}return a.spaceLeft-=e}),0<c&&(this.emit("lastLine",o,this),s()),this.emit("sectionEnd",o,this),!0===o.continued?(1<n&&(this.continuedX=0),this.continuedX+=o.textWidth||0,this.document.y=r):this.document.x=this.startX}},{key:"nextSection",value:function(t){if(this.emit("sectionEnd",t,this),++this.column>this.columns){if(null!=this.height)return!1;var e;if(this.document.addPage(),this.column=1,this.startY=this.document.page.margins.top,this.maxY=this.document.page.maxY(),this.document.x=this.startX,this.document._fillColor)(e=this.document).fillColor.apply(e,G(this.document._fillColor||[]));this.emit("pageBreak",t,this)}else this.document.x+=this.lineWidth+this.columnGap,this.document.y=this.startY,this.emit("columnBreak",t,this);return this.emit("sectionStart",t,this),!0}}]),r}(),zt=S.number,Nt={initText:function(){return this._line=this._line.bind(this),this.x=0,this.y=0,this._lineGap=0},lineGap:function(t){return this._lineGap=t,this},moveDown:function(t){return null==t&&(t=1),this.y+=this.currentLineHeight(!0)*t+this._lineGap,this},moveUp:function(t){return null==t&&(t=1),this.y-=this.currentLineHeight(!0)*t+this._lineGap,this},_text:function(t,e,n,r,i){if(r=this._initOptions(e,n,r),t=null==t?"":"".concat(t),r.wordSpacing&&(t=t.replace(/\s{2,}/g," ")),r.width){var o=this._wrapper;o||(o=new Dt(this,r)).on("line",i),this._wrapper=r.continued?o:null,this._textOptions=r.continued?r:null,o.wrap(t,r)}else{var a=!0,s=!1,l=void 0;try{for(var u,c=t.split("\n")[Symbol.iterator]();!(a=(u=c.next()).done);a=!0){i(u.value,r)}}catch(t){s=!0,l=t}finally{try{a||null==c.return||c.return()}finally{if(s)throw l}}}return this},text:function(t,e,n,r){return this._text(t,e,n,r,this._line)},widthOfString:function(t,e){var n=1<arguments.length&&void 0!==e?e:{};return this._font.widthOfString(t,this._fontSize,n.features)+(n.characterSpacing||0)*(t.length-1)},heightOfString:function(t,e){var n=this,r=this.x,i=this.y;(e=this._initOptions(e)).height=1/0;var o=e.lineGap||this._lineGap||0;this._text(t,this.x,this.y,e,function(){return n.y+=n.currentLineHeight(!0)+o});var a=this.y-i;return this.x=r,this.y=i,a},list:function(t,e,n,r,i){var o=this,a=(r=this._initOptions(e,n,r)).listType||"bullet",s=Math.round(this._font.ascender/1e3*this._fontSize),l=s/2,u=r.bulletRadius||s/3,c=r.textIndent||("bullet"===a?5*u:2*s),f=r.bulletIndent||("bullet"===a?8*u:2*s),h=1,d=[],p=[],g=[];!function t(e){for(var n=1,r=0;r<e.length;r++){var i=e[r];Array.isArray(i)?(h++,t(i),h--):(d.push(i),p.push(h),"bullet"!==a&&g.push(n++))}}(t);(i=new Dt(this,r)).on("line",this._line),h=1;var v=0;return i.on("firstLine",function(){var t;if((t=p[v++])!==h){var e=f*(t-h);o.x+=e,i.lineWidth-=e,h=t}switch(a){case"bullet":return o.circle(o.x-c+u,o.y+l,u),o.fill();case"numbered":case"lettered":var n=function(t){switch(a){case"numbered":return"".concat(t,".");case"lettered":var e=String.fromCharCode((t-1)%26+65),n=Math.floor((t-1)/26+1),r=Array(n+1).join(e);return"".concat(r,".")}}(g[v-1]);return o._fragment(n,o.x-c,o.y,r)}}),i.on("sectionStart",function(){var t=c+f*(h-1);return o.x+=t,i.lineWidth-=t}),i.on("sectionEnd",function(){var t=c+f*(h-1);return o.x-=t,i.lineWidth+=t}),i.wrap(d.join("\n"),r),this},_initOptions:function(t,e,n){var r=0<arguments.length&&void 0!==t?t:{},i=1<arguments.length?e:void 0,o=2<arguments.length&&void 0!==n?n:{};"object"==typeof r&&(o=r,r=null);var a=Object.assign({},o);if(this._textOptions)for(var s in this._textOptions){var l=this._textOptions[s];"continued"!==s&&null==a[s]&&(a[s]=l)}return null!=r&&(this.x=r),null!=i&&(this.y=i),!1!==a.lineBreak&&null==a.width&&(a.width=this.page.width-this.x-this.page.margins.right),a.columns||(a.columns=0),null==a.columnGap&&(a.columnGap=18),a},_line:function(t,e,n){var r=1<arguments.length&&void 0!==e?e:{},i=2<arguments.length?n:void 0;this._fragment(t,this.x,this.y,r);var o=r.lineGap||this._lineGap||0;return i?this.y+=this.currentLineHeight(!0)+o:this.x+=this.widthOfString(t)},_fragment:function(t,e,n,r){var i,o,a,s,l,u,c=this;if(0!==(t="".concat(t).replace(/\n/g,"")).length){var f=r.align||"left",h=r.wordSpacing||0,d=r.characterSpacing||0;if(r.width)switch(f){case"right":l=this.widthOfString(t.replace(/\s+$/,""),r),e+=r.lineWidth-l;break;case"center":e+=r.lineWidth/2-r.textWidth/2;break;case"justify":u=t.trim().split(/\s+/),l=this.widthOfString(t.replace(/\s+/g,""),r);var p=this.widthOfString(" ")+d;h=Math.max(0,(r.lineWidth-l)/Math.max(1,u.length-1)-p)}if("number"==typeof r.baseline)i=-r.baseline;else{switch(r.baseline){case"svg-middle":i=.5*this._font.xHeight;break;case"middle":case"svg-central":i=.5*(this._font.descender+this._font.ascender);break;case"bottom":case"ideographic":i=this._font.descender;break;case"alphabetic":i=0;break;case"mathematical":i=.5*this._font.ascender;break;case"hanging":i=.8*this._font.ascender;break;case"top":i=this._font.ascender;break;default:i=this._font.ascender}i=i/1e3*this._fontSize}var g,v=r.textWidth+h*(r.wordCount-1)+d*(t.length-1);if(null!=r.link&&this.link(e,n,v,this.currentLineHeight(),r.link),null!=r.goTo&&this.goTo(e,n,v,this.currentLineHeight(),r.goTo),null!=r.destination&&this.addNamedDestination(r.destination,"XYZ",e,n,null),r.underline||r.strike){this.save(),r.stroke||this.strokeColor.apply(this,G(this._fillColor||[]));var y=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(y);var b=r.underline?1:2,m=n+this.currentLineHeight()/b;r.underline&&(m-=y),this.moveTo(e,m),this.lineTo(e+v,m),this.stroke(),this.restore()}if(this.save(),r.oblique)g="number"==typeof r.oblique?-Math.tan(r.oblique*Math.PI/180):-.25,this.transform(1,0,0,1,e,n),this.transform(1,0,g,1,-g*i,0),this.transform(1,0,0,1,-e,-n);this.transform(1,0,0,-1,0,this.page.height),n=this.page.height-n-i,null==this.page.fonts[this._font.id]&&(this.page.fonts[this._font.id]=this._font.ref()),this.addContent("BT"),this.addContent("1 0 0 1 ".concat(zt(e)," ").concat(zt(n)," Tm")),this.addContent("/".concat(this._font.id," ").concat(zt(this._fontSize)," Tf"));var w=r.fill&&r.stroke?2:r.stroke?1:0;if(w&&this.addContent("".concat(w," Tr")),d&&this.addContent("".concat(zt(d)," Tc")),h){u=t.trim().split(/\s+/),h+=this.widthOfString(" ")+d,h*=1e3/this._fontSize,o=[];var x=!0,_=!(s=[]),k=void 0;try{for(var S,C=u[Symbol.iterator]();!(x=(S=C.next()).done);x=!0){var A=S.value,P=j(this._font.encode(A,r.features),2),E=P[0],O=P[1];o=o.concat(E),s=s.concat(O);var T={},I=s[s.length-1];for(var B in I){var L=I[B];T[B]=L}T.xAdvance+=h,s[s.length-1]=T}}catch(t){_=!0,k=t}finally{try{x||null==C.return||C.return()}finally{if(_)throw k}}}else{var R=j(this._font.encode(t,r.features),2);o=R[0],s=R[1]}var M=this._fontSize/1e3,F=[],D=0,z=!1,N=function(t){if(D<t){var e=o.slice(D,t).join(""),n=s[t-1].xAdvance-s[t-1].advanceWidth;F.push("<".concat(e,"> ").concat(zt(-n)))}return D=t},U=function(t){if(N(t),0<F.length)return c.addContent("[".concat(F.join(" "),"] TJ")),F.length=0};for(a=0;a<s.length;a++){var W=s[a];W.xOffset||W.yOffset?(U(a),this.addContent("1 0 0 1 ".concat(zt(e+W.xOffset*M)," ").concat(zt(n+W.yOffset*M)," Tm")),U(a+1),z=!0):(z&&(this.addContent("1 0 0 1 ".concat(zt(e)," ").concat(zt(n)," Tm")),z=!1),W.xAdvance-W.advanceWidth!=0&&N(a+1)),e+=W.xAdvance*M}return U(a),this.addContent("ET"),this.restore()}}},Ut=[65472,65473,65474,65475,65477,65478,65479,65480,65481,65482,65483,65484,65485,65486,65487],Wt={1:"DeviceGray",3:"DeviceRGB",4:"DeviceCMYK"},jt=function(){function o(t,e){var n;if(u(this,o),this.data=t,this.label=e,65496!==this.data.readUInt16BE(0))throw"SOI not found in JPEG";for(var r=2;r<this.data.length&&(n=this.data.readUInt16BE(r),r+=2,!Ut.includes(n));)r+=this.data.readUInt16BE(r);if(!Ut.includes(n))throw"Invalid JPEG.";r+=2,this.bits=this.data[r++],this.height=this.data.readUInt16BE(r),r+=2,this.width=this.data.readUInt16BE(r),r+=2;var i=this.data[r++];this.colorSpace=Wt[i],this.obj=null}return f(o,[{key:"embed",value:function(t){if(!this.obj)return this.obj=t.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:this.bits,Width:this.width,Height:this.height,ColorSpace:this.colorSpace,Filter:"DCTDecode"}),"DeviceCMYK"===this.colorSpace&&(this.obj.data.Decode=[1,0,1,0,1,0,1,0]),this.obj.end(this.data),this.data=null}}]),o}(),Gt=function(){function n(t,e){u(this,n),this.label=e,this.image=new r.default(t),this.width=this.image.width,this.height=this.image.height,this.imgData=this.image.imgData,this.obj=null}return f(n,[{key:"embed",value:function(t){if(this.document=t,!this.obj){var e=this.image.hasAlphaChannel;if(this.obj=this.document.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:e?8:this.image.bits,Width:this.width,Height:this.height,Filter:"FlateDecode"}),!e){var n=this.document.ref({Predictor:15,Colors:this.image.colors,BitsPerComponent:this.image.bits,Columns:this.width});(this.obj.data.DecodeParms=n).end()}if(0===this.image.palette.length)this.obj.data.ColorSpace=this.image.colorSpace;else{var r=this.document.ref();r.end(new p(this.image.palette)),this.obj.data.ColorSpace=["Indexed","DeviceRGB",this.image.palette.length/3-1,r]}if(null!=this.image.transparency.grayscale){var i=this.image.transparency.grayscale;this.obj.data.Mask=[i,i]}else if(this.image.transparency.rgb){var o=this.image.transparency.rgb,a=[],s=!0,l=!1,u=void 0;try{for(var c,f=o[Symbol.iterator]();!(s=(c=f.next()).done);s=!0){var h=c.value;a.push(h,h)}}catch(t){l=!0,u=t}finally{try{s||null==f.return||f.return()}finally{if(l)throw u}}this.obj.data.Mask=a}else{if(this.image.transparency.indexed)return this.loadIndexedAlphaChannel();if(e)return this.splitAlphaChannel()}this.finalize()}}},{key:"finalize",value:function(){if(this.alphaChannel){var t=this.document.ref({Type:"XObject",Subtype:"Image",Height:this.height,Width:this.width,BitsPerComponent:8,Filter:"FlateDecode",ColorSpace:"DeviceGray",Decode:[0,1]});t.end(this.alphaChannel),this.obj.data.SMask=t}return this.obj.end(this.imgData),this.image=null,this.imgData=null}},{key:"splitAlphaChannel",value:function(){var f=this;return this.image.decodePixels(function(t){for(var e,n,r=f.image.colors,i=f.width*f.height,o=new p(i*r),a=new p(i),s=n=e=0,l=t.length,u=16===f.image.bits?1:0;s<l;){for(var c=0;c<r;c++)o[n++]=t[s++],s+=u;a[e++]=t[s++],s+=u}return f.imgData=h.default.deflateSync(o),f.alphaChannel=h.default.deflateSync(a),f.finalize()})}},{key:"loadIndexedAlphaChannel",value:function(){var o=this,a=this.image.transparency.indexed;return this.image.decodePixels(function(t){for(var e=new p(o.width*o.height),n=0,r=0,i=t.length;r<i;r++)e[n++]=a[t[r]];return o.alphaChannel=h.default.deflateSync(e),o.finalize()})}}]),n}(),Ht=function(){function t(){u(this,t)}return f(t,null,[{key:"open",value:function(t,e){var n;if(p.isBuffer(t))n=t;else if(t instanceof ArrayBuffer)n=new p(new Uint8Array(t));else{var r;if(r=/^data:.+;base64,(.*)$/.exec(t))n=new p(r[1],"base64");else if(!(n=s.readFileSync(t)))return}if(255===n[0]&&216===n[1])return new jt(n,e);if(137===n[0]&&"PNG"===n.toString("ascii",1,4))return new Gt(n,e);throw new Error("Unknown image format.")}}]),t}(),Vt={initImages:function(){return this._imageRegistry={},this._imageCount=0},image:function(t,e,n){var r,i,o,a,s,l,u=3<arguments.length&&void 0!==arguments[3]?arguments[3]:{};"object"==typeof e&&(u=e,e=null),e=null!=(s=null!=e?e:u.x)?s:this.x,n=null!=(l=null!=n?n:u.y)?l:this.y,"string"==typeof t&&(o=this._imageRegistry[t]),o=o||(t.width&&t.height?t:this.openImage(t)),o.obj||o.embed(this),null==this.page.xobjects[o.label]&&(this.page.xobjects[o.label]=o.obj);var c=u.width||o.width,f=u.height||o.height;if(u.width&&!u.height){var h=c/o.width;c=o.width*h,f=o.height*h}else if(u.height&&!u.width){var d=f/o.height;c=o.width*d,f=o.height*d}else if(u.scale)c=o.width*u.scale,f=o.height*u.scale;else if(u.fit){var p=j(u.fit,2);(i=p[0])/(r=p[1])<(a=o.width/o.height)?f=(c=i)/a:c=(f=r)*a}else if(u.cover){var g=j(u.cover,2);(i=g[0])/(r=g[1])<(a=o.width/o.height)?c=(f=r)*a:f=(c=i)/a}return(u.fit||u.cover)&&("center"===u.align?e=e+i/2-c/2:"right"===u.align&&(e=e+i-c),"center"===u.valign?n=n+r/2-f/2:"bottom"===u.valign&&(n=n+r-f)),null!=u.link&&this.link(e,n,c,f,u.link),null!=u.goTo&&this.goTo(e,n,c,f,u.goTo),null!=u.destination&&this.addNamedDestination(u.destination,"XYZ",e,n,null),this.y===n&&(this.y+=f),this.save(),this.transform(c,0,0,-f,e,n+f),this.addContent("/".concat(o.label," Do")),this.restore(),this},openImage:function(t){var e;return"string"==typeof t&&(e=this._imageRegistry[t]),e||(e=Ht.open(t,"I".concat(++this._imageCount)),"string"==typeof t&&(this._imageRegistry[t]=e)),e}},qt={annotate:function(t,e,n,r,i){for(var o in i.Type="Annot",i.Rect=this._convertRect(t,e,n,r),i.Border=[0,0,0],"Link"!==i.Subtype&&null==i.C&&(i.C=this._normalizeColor(i.color||[0,0,0])),delete i.color,"string"==typeof i.Dest&&(i.Dest=new String(i.Dest)),i){var a=i[o];i[o[0].toUpperCase()+o.slice(1)]=a}var s=this.ref(i);return this.page.annotations.push(s),s.end(),this},note:function(t,e,n,r,i,o){var a=5<arguments.length&&void 0!==o?o:{};return a.Subtype="Text",a.Contents=new String(i),a.Name="Comment",null==a.color&&(a.color=[243,223,92]),this.annotate(t,e,n,r,a)},goTo:function(t,e,n,r,i,o){var a=5<arguments.length&&void 0!==o?o:{};return a.Subtype="Link",a.A=this.ref({S:"GoTo",D:new String(i)}),a.A.end(),this.annotate(t,e,n,r,a)},link:function(t,e,n,r,i,o){var a=5<arguments.length&&void 0!==o?o:{};if(a.Subtype="Link","number"==typeof i){var s=this._root.data.Pages.data;if(!(0<=i&&i<s.Kids.length))throw new Error("The document has no page ".concat(i));a.A=this.ref({S:"GoTo",D:[s.Kids[i],"XYZ",null,null,null]}),a.A.end()}else a.A=this.ref({S:"URI",URI:new String(i)}),a.A.end();return this.annotate(t,e,n,r,a)},_markup:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{},a=j(this._convertRect(t,e,n,r),4),s=a[0],l=a[1],u=a[2],c=a[3];return o.QuadPoints=[s,c,u,c,s,l,u,l],o.Contents=new String,this.annotate(t,e,n,r,o)},highlight:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="Highlight",null==o.color&&(o.color=[241,238,148]),this._markup(t,e,n,r,o)},underline:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="Underline",this._markup(t,e,n,r,o)},strike:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="StrikeOut",this._markup(t,e,n,r,o)},lineAnnotation:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="Line",o.Contents=new String,o.L=[t,this.page.height-e,n,this.page.height-r],this.annotate(t,e,n,r,o)},rectAnnotation:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="Square",o.Contents=new String,this.annotate(t,e,n,r,o)},ellipseAnnotation:function(t,e,n,r,i){var o=4<arguments.length&&void 0!==i?i:{};return o.Subtype="Circle",o.Contents=new String,this.annotate(t,e,n,r,o)},textAnnotation:function(t,e,n,r,i,o){var a=5<arguments.length&&void 0!==o?o:{};return a.Subtype="FreeText",a.Contents=new String(i),a.DA=new String,this.annotate(t,e,n,r,a)},_convertRect:function(t,e,n,r){var i=e;e+=r;var o=t+n,a=j(this._ctm,6),s=a[0],l=a[1],u=a[2],c=a[3],f=a[4],h=a[5];return[t=s*t+u*e+f,e=l*t+c*e+h,o=s*o+u*i+f,i=l*o+c*i+h]}},Zt=function(){function o(t,e,n,r){var i=4<arguments.length&&void 0!==arguments[4]?arguments[4]:{expanded:!1};u(this,o),this.document=t,this.options=i,this.outlineData={},null!==r&&(this.outlineData.Dest=[r.dictionary,"Fit"]),null!==e&&(this.outlineData.Parent=e),null!==n&&(this.outlineData.Title=new String(n)),this.dictionary=this.document.ref(this.outlineData),this.children=[]}return f(o,[{key:"addItem",value:function(t,e){var n=1<arguments.length&&void 0!==e?e:{expanded:!1},r=new o(this.document,this.dictionary,t,this.document.page,n);return this.children.push(r),r}},{key:"endOutline",value:function(){if(0<this.children.length){this.options.expanded&&(this.outlineData.Count=this.children.length);var t=this.children[0],e=this.children[this.children.length-1];this.outlineData.First=t.dictionary,this.outlineData.Last=e.dictionary;for(var n=0,r=this.children.length;n<r;n++){var i=this.children[n];0<n&&(i.outlineData.Prev=this.children[n-1].dictionary),n<this.children.length-1&&(i.outlineData.Next=this.children[n+1].dictionary),i.endOutline()}}return this.dictionary.end()}}]),o}(),Yt={initOutline:function(){return this.outline=new Zt(this,null,null,null)},endOutline:function(){if(this.outline.endOutline(),0<this.outline.children.length)return this._root.data.Outlines=this.outline.dictionary,this._root.data.PageMode="UseOutlines"}},Xt=function(){function a(){var t,e=0<arguments.length&&void 0!==arguments[0]?arguments[0]:{};switch(u(this,a),((t=b(this,v(a).call(this,e))).options=e).pdfVersion){case"1.4":t.version=1.4;break;case"1.5":t.version=1.5;break;case"1.6":t.version=1.6;break;case"1.7":case"1.7ext3":t.version=1.7;break;default:t.version=1.3}t.compress=null==t.options.compress||t.options.compress,t._pageBuffer=[],t._pageBufferStart=0,t._offsets=[],t._waiting=0,t._ended=!1,t._offset=0;var n=t.ref({Type:"Pages",Count:0,Kids:[]}),r=t.ref({Dests:new x});if(t._root=t.ref({Type:"Catalog",Pages:n,Names:r}),t.page=null,t.initColor(),t.initVector(),t.initFonts(e.font),t.initText(),t.initImages(),t.initOutline(),t.info={Producer:"PDFKit",Creator:"PDFKit",CreationDate:new Date},t.options.info)for(var i in t.options.info){var o=t.options.info[i];t.info[i]=o}return t._id=K.generateFileID(t.info),t._security=K.create(y(t),e),t._write("%PDF-".concat(t.version)),t._write("%ÿÿÿÿ"),!1!==t.options.autoFirstPage&&t.addPage(),t}return c(a,e.default.Readable),f(a,[{key:"addPage",value:function(t){null==t&&(t=this.options),this.options.bufferPages||this.flushPages(),this.page=new E(this,t),this._pageBuffer.push(this.page);var e=this._root.data.Pages.data;return e.Kids.push(this.page.dictionary),e.Count++,this.x=this.page.margins.left,this.y=this.page.margins.top,this._ctm=[1,0,0,1,0,0],this.transform(1,0,0,-1,0,this.page.height),this.emit("pageAdded"),this}},{key:"bufferedPageRange",value:function(){return{start:this._pageBufferStart,count:this._pageBuffer.length}}},{key:"switchToPage",value:function(t){var e;if(!(e=this._pageBuffer[t-this._pageBufferStart]))throw new Error("switchToPage(".concat(t,") out of bounds, current buffer covers pages ").concat(this._pageBufferStart," to ").concat(this._pageBufferStart+this._pageBuffer.length-1));return this.page=e}},{key:"flushPages",value:function(){var t=this._pageBuffer;this._pageBuffer=[],this._pageBufferStart+=t.length;var e=!0,n=!1,r=void 0;try{for(var i,o=t[Symbol.iterator]();!(e=(i=o.next()).done);e=!0){i.value.end()}}catch(t){n=!0,r=t}finally{try{e||null==o.return||o.return()}finally{if(n)throw r}}}},{key:"addNamedDestination",value:function(t){for(var e=arguments.length,n=new Array(1<e?e-1:0),r=1;r<e;r++)n[r-1]=arguments[r];0===n.length&&(n=["XYZ",null,null,null]),"XYZ"===n[0]&&null!==n[2]&&(n[2]=this.page.height-n[2]),n.unshift(this.page.dictionary),this._root.data.Names.data.Dests.add(t,n)}},{key:"ref",value:function(t){var e=new C(this,this._offsets.length+1,t);return this._offsets.push(null),this._waiting++,e}},{key:"_read",value:function(){}},{key:"_write",value:function(t){return p.isBuffer(t)||(t=new p(t+"\n","binary")),this.push(t),this._offset+=t.length}},{key:"addContent",value:function(t){return this.page.write(t),this}},{key:"_refEnd",value:function(t){if(this._offsets[t.id-1]=t.offset,0==--this._waiting&&this._ended)return this._finalize(),this._ended=!1}},{key:"write",value:function(t,e){new Error("PDFDocument#write is deprecated, and will be removed in a future version of PDFKit. Please pipe the document into a Node stream.");return this.pipe(s.createWriteStream(t)),this.end(),this.once("end",e)}},{key:"end",value:function(){for(var t in this.flushPages(),this._info=this.ref(),this.info){var e=this.info[t];"string"==typeof e&&(e=new String(e));var n=this.ref(e);n.end(),this._info.data[t]=n}for(var r in this._info.end(),this._fontFamilies){this._fontFamilies[r].finalize()}return this.endOutline(),this._root.end(),this._root.data.Pages.end(),this._root.data.Names.end(),this._security&&this._security.end(),0===this._waiting?this._finalize():this._ended=!0}},{key:"_finalize",value:function(t){var e=this._offset;this._write("xref"),this._write("0 ".concat(this._offsets.length+1)),this._write("0000000000 65535 f ");var n=!0,r=!1,i=void 0;try{for(var o,a=this._offsets[Symbol.iterator]();!(n=(o=a.next()).done);n=!0){var s=o.value;s="0000000000".concat(s).slice(-10),this._write(s+" 00000 n ")}}catch(t){r=!0,i=t}finally{try{n||null==a.return||a.return()}finally{if(r)throw i}}var l={Size:this._offsets.length+1,Root:this._root,Info:this._info,ID:[this._id,this._id]};return this._security&&(l.Encrypt=this._security.dictionary),this._write("trailer"),this._write(S.convert(l)),this._write("startxref"),this._write("".concat(e)),this._write("%%EOF"),this.push(null)}},{key:"toString",value:function(){return"[object PDFDocument]"}}]),a}();bt(gt),bt(Pt),bt(Ft),bt(Nt),bt(Vt),bt(qt),bt(Yt);var Kt=Xt;Jt.default=Kt}).call(this,Qt(4).Buffer,"/")},function(t,e,n){"use strict";n(130)("strike",function(t){return function(){return t(this,"strike","","")}})},function(t,e,n){t.exports=n(62)("native-function-to-string",Function.toString)},function(t,e,n){"use strict";n(130)("link",function(e){return function(t){return e(this,"a","href",t)}})},function(t,e,n){var r=n(6);r(r.S+r.F,"Object",{assign:n(208)})},function(t,e,n){"use strict";var h=n(9),d=n(42),p=n(85),g=n(63),v=n(19),y=n(81),i=Object.assign;t.exports=!i||n(10)(function(){var t={},e={},n=Symbol(),r="abcdefghijklmnopqrst";return t[n]=7,r.split("").forEach(function(t){e[t]=t}),7!=i({},t)[n]||Object.keys(i({},e)).join("")!=r})?function(t,e){for(var n=v(t),r=arguments.length,i=1,o=p.f,a=g.f;i<r;)for(var s,l=y(arguments[i++]),u=o?d(l).concat(o(l)):d(l),c=u.length,f=0;f<c;)s=u[f++],h&&!a.call(l,s)||(n[s]=l[s]);return n}:i},function(t,e,n){var r=n(6);r(r.P,"Array",{fill:n(86)}),n(87)("fill")},function(t,e,n){n(211)("Uint8",1,function(r){return function(t,e,n){return r(this,t,e,n)}})},function(t,e,n){"use strict";if(n(9)){var y=n(41),b=n(8),m=n(10),w=n(6),x=n(135),r=n(212),p=n(54),_=n(137),i=n(40),k=n(15),o=n(136),a=n(31),S=n(16),C=n(138),s=n(56),l=n(53),u=n(23),A=n(65),P=n(18),g=n(19),v=n(139),E=n(66),O=n(140),T=n(57).f,I=n(141),c=n(29),f=n(3),h=n(215),d=n(82),B=n(143),L=n(88),R=n(44),M=n(145),F=n(220),D=n(86),z=n(221),N=n(11),U=n(67),W=N.f,j=U.f,G=b.RangeError,H=b.TypeError,V=b.Uint8Array,q="ArrayBuffer",Z="Shared"+q,Y="BYTES_PER_ELEMENT",X="prototype",K=Array[X],J=r.ArrayBuffer,Q=r.DataView,$=h(0),tt=h(2),et=h(3),nt=h(4),rt=h(5),it=h(6),ot=d(!0),at=d(!1),st=L.values,lt=L.keys,ut=L.entries,ct=K.lastIndexOf,ft=K.reduce,ht=K.reduceRight,dt=K.join,pt=K.sort,gt=K.slice,vt=K.toString,yt=K.toLocaleString,bt=f("iterator"),mt=f("toStringTag"),wt=c("typed_constructor"),xt=c("def_constructor"),_t=x.CONSTR,kt=x.TYPED,St=x.VIEW,Ct="Wrong length!",At=h(1,function(t,e){return It(B(t,t[xt]),e)}),Pt=m(function(){return 1===new V(new Uint16Array([1]).buffer)[0]}),Et=!!V&&!!V[X].set&&m(function(){new V(1).set({})}),Ot=function(t,e){var n=a(t);if(n<0||n%e)throw G("Wrong offset!");return n},Tt=function(t){if(P(t)&&kt in t)return t;throw H(t+" is not a typed array!")},It=function(t,e){if(!(P(t)&&wt in t))throw H("It is not a typed array constructor!");return new t(e)},Bt=function(t,e){return Lt(B(t,t[xt]),e)},Lt=function(t,e){for(var n=0,r=e.length,i=It(t,r);n<r;)i[n]=e[n++];return i},Rt=function(t,e,n){W(t,e,{get:function(){return this._d[n]}})},Mt=function(t,e,n){var r,i,o,a,s,l,u=g(t),c=arguments.length,f=1<c?e:void 0,h=void 0!==f,d=I(u);if(null!=d&&!v(d)){for(l=d.call(u),o=[],r=0;!(s=l.next()).done;r++)o.push(s.value);u=o}for(h&&2<c&&(f=p(f,n,2)),r=0,i=S(u.length),a=It(this,i);r<i;r++)a[r]=h?f(u[r],r):u[r];return a},Ft=function(){for(var t=0,e=arguments.length,n=It(this,e);t<e;)n[t]=arguments[t++];return n},Dt=!!V&&m(function(){yt.call(new V(1))}),zt=function(){return yt.apply(Dt?gt.call(Tt(this)):Tt(this),arguments)},Nt={copyWithin:function(t,e,n){return z.call(Tt(this),t,e,2<arguments.length?n:void 0)},every:function(t,e){return nt(Tt(this),t,1<arguments.length?e:void 0)},fill:function(t){return D.apply(Tt(this),arguments)},filter:function(t,e){return Bt(this,tt(Tt(this),t,1<arguments.length?e:void 0))},find:function(t,e){return rt(Tt(this),t,1<arguments.length?e:void 0)},findIndex:function(t,e){return it(Tt(this),t,1<arguments.length?e:void 0)},forEach:function(t,e){$(Tt(this),t,1<arguments.length?e:void 0)},indexOf:function(t,e){return at(Tt(this),t,1<arguments.length?e:void 0)},includes:function(t,e){return ot(Tt(this),t,1<arguments.length?e:void 0)},join:function(t){return dt.apply(Tt(this),arguments)},lastIndexOf:function(t){return ct.apply(Tt(this),arguments)},map:function(t,e){return At(Tt(this),t,1<arguments.length?e:void 0)},reduce:function(t){return ft.apply(Tt(this),arguments)},reduceRight:function(t){return ht.apply(Tt(this),arguments)},reverse:function(){for(var t,e=Tt(this).length,n=Math.floor(e/2),r=0;r<n;)t=this[r],this[r++]=this[--e],this[e]=t;return this},some:function(t,e){return et(Tt(this),t,1<arguments.length?e:void 0)},sort:function(t){return pt.call(Tt(this),t)},subarray:function(t,e){var n=Tt(this),r=n.length,i=s(t,r);return new(B(n,n[xt]))(n.buffer,n.byteOffset+i*n.BYTES_PER_ELEMENT,S((void 0===e?r:s(e,r))-i))}},Ut=function(t,e){return Bt(this,gt.call(Tt(this),t,e))},Wt=function(t,e){Tt(this);var n=Ot(e,1),r=this.length,i=g(t),o=S(i.length),a=0;if(r<o+n)throw G(Ct);for(;a<o;)this[n+a]=i[a++]},jt={entries:function(){return ut.call(Tt(this))},keys:function(){return lt.call(Tt(this))},values:function(){return st.call(Tt(this))}},Gt=function(t,e){return P(t)&&t[kt]&&"symbol"!=typeof e&&e in t&&String(+e)==String(e)},Ht=function(t,e){return Gt(t,e=l(e,!0))?i(2,t[e]):j(t,e)},Vt=function(t,e,n){return!(Gt(t,e=l(e,!0))&&P(n)&&u(n,"value"))||u(n,"get")||u(n,"set")||n.configurable||u(n,"writable")&&!n.writable||u(n,"enumerable")&&!n.enumerable?W(t,e,n):(t[e]=n.value,t)};_t||(U.f=Ht,N.f=Vt),w(w.S+w.F*!_t,"Object",{getOwnPropertyDescriptor:Ht,defineProperty:Vt}),m(function(){vt.call({})})&&(vt=yt=function(){return dt.call(this)});var qt=o({},Nt);o(qt,jt),k(qt,bt,jt.values),o(qt,{slice:Ut,set:Wt,constructor:function(){},toString:vt,toLocaleString:zt}),Rt(qt,"buffer","b"),Rt(qt,"byteOffset","o"),Rt(qt,"byteLength","l"),Rt(qt,"length","e"),W(qt,mt,{get:function(){return this[kt]}}),t.exports=function(t,f,e,i){function h(t,e){W(t,e,{get:function(){return function(t,e){var n=t._d;return n.v[r](e*f+n.o,Pt)}(this,e)},set:function(t){return function(t,e,n){var r=t._d;i&&(n=(n=Math.round(n))<0?0:255<n?255:255&n),r.v[o](e*f+r.o,n,Pt)}(this,e,t)},enumerable:!0})}var d=t+((i=!!i)?"Clamped":"")+"Array",r="get"+t,o="set"+t,p=b[d],a=p||{},n=p&&O(p),s=!p||!x.ABV,l={},u=p&&p[X];s?(p=e(function(t,e,n,r){_(t,p,d,"_d");var i,o,a,s,l=0,u=0;if(P(e)){if(!(e instanceof J||(s=A(e))==q||s==Z))return kt in e?Lt(p,e):Mt.call(p,e);i=e,u=Ot(n,f);var c=e.byteLength;if(void 0===r){if(c%f)throw G(Ct);if((o=c-u)<0)throw G(Ct)}else if(c<(o=S(r)*f)+u)throw G(Ct);a=o/f}else a=C(e),i=new J(o=a*f);for(k(t,"_d",{b:i,o:u,l:o,e:a,v:new Q(i)});l<a;)h(t,l++)}),u=p[X]=E(qt),k(u,"constructor",p)):m(function(){p(1)})&&m(function(){new p(-1)})&&M(function(t){new p,new p(null),new p(1.5),new p(t)},!0)||(p=e(function(t,e,n,r){var i;return _(t,p,d),P(e)?e instanceof J||(i=A(e))==q||i==Z?void 0!==r?new a(e,Ot(n,f),r):void 0!==n?new a(e,Ot(n,f)):new a(e):kt in e?Lt(p,e):Mt.call(p,e):new a(C(e))}),$(n!==Function.prototype?T(a).concat(T(n)):T(a),function(t){t in p||k(p,t,a[t])}),p[X]=u,y||(u.constructor=p));var c=u[bt],g=!!c&&("values"==c.name||null==c.name),v=jt.values;k(p,wt,!0),k(u,kt,d),k(u,St,!0),k(u,xt,p),(i?new p(1)[mt]==d:mt in u)||W(u,mt,{get:function(){return d}}),l[d]=p,w(w.G+w.W+w.F*(p!=a),l),w(w.S,d,{BYTES_PER_ELEMENT:f}),w(w.S+w.F*m(function(){a.of.call(p,1)}),d,{from:Mt,of:Ft}),Y in u||k(u,Y,f),w(w.P,d,Nt),F(d),w(w.P+w.F*Et,d,{set:Wt}),w(w.P+w.F*!g,d,jt),y||u.toString==vt||(u.toString=vt),w(w.P+w.F*m(function(){new p(1).slice()}),d,{slice:Ut}),w(w.P+w.F*(m(function(){return[1,2].toLocaleString()!=new p([1,2]).toLocaleString()})||!m(function(){u.toLocaleString.call([1,2])})),d,{toLocaleString:zt}),R[d]=g?c:v,y||g||k(u,bt,v)}}else t.exports=function(){}},function(t,e,n){"use strict";var r=n(8),i=n(9),o=n(41),a=n(135),s=n(15),l=n(136),u=n(10),c=n(137),f=n(31),h=n(16),d=n(138),p=n(57).f,g=n(11).f,v=n(86),y=n(64),b="ArrayBuffer",m="DataView",w="prototype",x="Wrong index!",_=r[b],k=r[m],S=r.Math,C=r.RangeError,A=r.Infinity,P=_,E=S.abs,O=S.pow,T=S.floor,I=S.log,B=S.LN2,L="byteLength",R="byteOffset",M=i?"_b":"buffer",F=i?"_l":L,D=i?"_o":R;function z(t,e,n){var r,i,o,a=new Array(n),s=8*n-e-1,l=(1<<s)-1,u=l>>1,c=23===e?O(2,-24)-O(2,-77):0,f=0,h=t<0||0===t&&1/t<0?1:0;for((t=E(t))!=t||t===A?(i=t!=t?1:0,r=l):(r=T(I(t)/B),t*(o=O(2,-r))<1&&(r--,o*=2),2<=(t+=1<=r+u?c/o:c*O(2,1-u))*o&&(r++,o/=2),l<=r+u?(i=0,r=l):1<=r+u?(i=(t*o-1)*O(2,e),r+=u):(i=t*O(2,u-1)*O(2,e),r=0));8<=e;a[f++]=255&i,i/=256,e-=8);for(r=r<<e|i,s+=e;0<s;a[f++]=255&r,r/=256,s-=8);return a[--f]|=128*h,a}function N(t,e,n){var r,i=8*n-e-1,o=(1<<i)-1,a=o>>1,s=i-7,l=n-1,u=t[l--],c=127&u;for(u>>=7;0<s;c=256*c+t[l],l--,s-=8);for(r=c&(1<<-s)-1,c>>=-s,s+=e;0<s;r=256*r+t[l],l--,s-=8);if(0===c)c=1-a;else{if(c===o)return r?NaN:u?-A:A;r+=O(2,e),c-=a}return(u?-1:1)*r*O(2,c-e)}function U(t){return t[3]<<24|t[2]<<16|t[1]<<8|t[0]}function W(t){return[255&t]}function j(t){return[255&t,t>>8&255]}function G(t){return[255&t,t>>8&255,t>>16&255,t>>24&255]}function H(t){return z(t,52,8)}function V(t){return z(t,23,4)}function q(t,e,n){g(t[w],e,{get:function(){return this[n]}})}function Z(t,e,n,r){var i=d(+n);if(i+e>t[F])throw C(x);var o=t[M]._b,a=i+t[D],s=o.slice(a,a+e);return r?s:s.reverse()}function Y(t,e,n,r,i,o){var a=d(+n);if(a+e>t[F])throw C(x);for(var s=t[M]._b,l=a+t[D],u=r(+i),c=0;c<e;c++)s[l+c]=u[o?c:e-c-1]}if(a.ABV){if(!u(function(){_(1)})||!u(function(){new _(-1)})||u(function(){return new _,new _(1.5),new _(NaN),_.name!=b})){for(var X,K=(_=function(t){return c(this,_),new P(d(t))})[w]=P[w],J=p(P),Q=0;J.length>Q;)(X=J[Q++])in _||s(_,X,P[X]);o||(K.constructor=_)}var $=new k(new _(2)),tt=k[w].setInt8;$.setInt8(0,2147483648),$.setInt8(1,2147483649),!$.getInt8(0)&&$.getInt8(1)||l(k[w],{setInt8:function(t,e){tt.call(this,t,e<<24>>24)},setUint8:function(t,e){tt.call(this,t,e<<24>>24)}},!0)}else _=function(t){c(this,_,b);var e=d(t);this._b=v.call(new Array(e),0),this[F]=e},k=function(t,e,n){c(this,k,m),c(t,_,m);var r=t[F],i=f(e);if(i<0||r<i)throw C("Wrong offset!");if(r<i+(n=void 0===n?r-i:h(n)))throw C("Wrong length!");this[M]=t,this[D]=i,this[F]=n},i&&(q(_,L,"_l"),q(k,"buffer","_b"),q(k,L,"_l"),q(k,R,"_o")),l(k[w],{getInt8:function(t){return Z(this,1,t)[0]<<24>>24},getUint8:function(t){return Z(this,1,t)[0]},getInt16:function(t,e){var n=Z(this,2,t,e);return(n[1]<<8|n[0])<<16>>16},getUint16:function(t,e){var n=Z(this,2,t,e);return n[1]<<8|n[0]},getInt32:function(t,e){return U(Z(this,4,t,e))},getUint32:function(t,e){return U(Z(this,4,t,e))>>>0},getFloat32:function(t,e){return N(Z(this,4,t,e),23,4)},getFloat64:function(t,e){return N(Z(this,8,t,e),52,8)},setInt8:function(t,e){Y(this,1,t,W,e)},setUint8:function(t,e){Y(this,1,t,W,e)},setInt16:function(t,e,n){Y(this,2,t,j,e,n)},setUint16:function(t,e,n){Y(this,2,t,j,e,n)},setInt32:function(t,e,n){Y(this,4,t,G,e,n)},setUint32:function(t,e,n){Y(this,4,t,G,e,n)},setFloat32:function(t,e,n){Y(this,4,t,V,e,n)},setFloat64:function(t,e,n){Y(this,8,t,H,e,n)}});y(_,b),y(k,m),s(k[w],a.VIEW,!0),e[b]=_,e[m]=k},function(t,e,n){var a=n(11),s=n(13),l=n(42);t.exports=n(9)?Object.defineProperties:function(t,e){s(t);for(var n,r=l(e),i=r.length,o=0;o<i;)a.f(t,n=r[o++],e[n]);return t}},function(t,e,n){var r=n(8).document;t.exports=r&&r.documentElement},function(t,e,n){var m=n(54),w=n(81),x=n(19),_=n(16),r=n(216);t.exports=function(f,t){var h=1==f,d=2==f,p=3==f,g=4==f,v=6==f,y=5==f||v,b=t||r;return function(t,e,n){for(var r,i,o=x(t),a=w(o),s=m(e,n,3),l=_(a.length),u=0,c=h?b(t,l):d?b(t,0):void 0;u<l;u++)if((y||u in a)&&(i=s(r=a[u],u,o),f))if(h)c[u]=i;else if(i)switch(f){case 3:return!0;case 5:return r;case 6:return u;case 2:c.push(r)}else if(g)return!1;return v?-1:p||g?g:c}}},function(t,e,n){var r=n(217);t.exports=function(t,e){return new(r(t))(e)}},function(t,e,n){var r=n(18),i=n(142),o=n(3)("species");t.exports=function(t){var e;return i(t)&&("function"!=typeof(e=t.constructor)||e!==Array&&!i(e.prototype)||(e=void 0),r(e)&&null===(e=e[o])&&(e=void 0)),void 0===e?Array:e}},function(t,e){t.exports=function(t,e){return{value:e,done:!!t}}},function(t,e,n){"use strict";var r=n(66),i=n(40),o=n(64),a={};n(15)(a,n(3)("iterator"),function(){return this}),t.exports=function(t,e,n){t.prototype=r(a,{next:i(1,n)}),o(t,e+" Iterator")}},function(t,e,n){"use strict";var r=n(8),i=n(11),o=n(9),a=n(3)("species");t.exports=function(t){var e=r[t];o&&e&&!e[a]&&i.f(e,a,{configurable:!0,get:function(){return this}})}},function(t,e,n){"use strict";var c=n(19),f=n(56),h=n(16);t.exports=[].copyWithin||function(t,e,n){var r=c(this),i=h(r.length),o=f(t,i),a=f(e,i),s=2<arguments.length?n:void 0,l=Math.min((void 0===s?i:f(s,i))-a,i-o),u=1;for(a<o&&o<a+l&&(u=-1,a+=l-1,o+=l-1);0<l--;)a in r?r[o]=r[a]:delete r[o],o+=u,a+=u;return r}},function(t,e,n){"use strict";var f=n(13),h=n(16),d=n(89),p=n(91);n(92)("match",1,function(r,i,u,c){return[function(t){var e=r(this),n=null==t?void 0:t[i];return void 0!==n?n.call(t,e):new RegExp(t)[i](String(e))},function(t){var e=c(u,t,this);if(e.done)return e.value;var n=f(t),r=String(this);if(!n.global)return p(n,r);for(var i,o=n.unicode,a=[],s=n.lastIndex=0;null!==(i=p(n,r));){var l=String(i[0]);""===(a[s]=l)&&(n.lastIndex=d(r,h(n.lastIndex),o)),s++}return 0===s?null:a}]})},function(t,e,n){"use strict";var r=n(93);n(6)({target:"RegExp",proto:!0,forced:r!==/./.exec},{exec:r})},function(t,e,n){"use strict";var f=n(146),m=n(13),w=n(143),x=n(89),_=n(16),k=n(91),h=n(93),r=n(10),S=Math.min,d=[].push,a="split",p="length",g="lastIndex",C=4294967295,A=!r(function(){RegExp(C,"y")});n(92)("split",2,function(i,o,v,y){var b;return b="c"=="abbc"[a](/(b)*/)[1]||4!="test"[a](/(?:)/,-1)[p]||2!="ab"[a](/(?:ab)*/)[p]||4!="."[a](/(.?)(.?)/)[p]||1<"."[a](/()()/)[p]||""[a](/.?/)[p]?function(t,e){var n=String(this);if(void 0===t&&0===e)return[];if(!f(t))return v.call(n,t,e);for(var r,i,o,a=[],s=(t.ignoreCase?"i":"")+(t.multiline?"m":"")+(t.unicode?"u":"")+(t.sticky?"y":""),l=0,u=void 0===e?C:e>>>0,c=new RegExp(t.source,s+"g");(r=h.call(c,n))&&!(l<(i=c[g])&&(a.push(n.slice(l,r.index)),1<r[p]&&r.index<n[p]&&d.apply(a,r.slice(1)),o=r[0][p],l=i,a[p]>=u));)c[g]===r.index&&c[g]++;return l===n[p]?!o&&c.test("")||a.push(""):a.push(n.slice(l)),a[p]>u?a.slice(0,u):a}:"0"[a](void 0,0)[p]?function(t,e){return void 0===t&&0===e?[]:v.call(this,t,e)}:v,[function(t,e){var n=i(this),r=null==t?void 0:t[o];return void 0!==r?r.call(t,n,e):b.call(String(n),t,e)},function(t,e){var n=y(b,t,this,e,b!==v);if(n.done)return n.value;var r=m(t),i=String(this),o=w(r,RegExp),a=r.unicode,s=(r.ignoreCase?"i":"")+(r.multiline?"m":"")+(r.unicode?"u":"")+(A?"y":"g"),l=new o(A?r:"^(?:"+r.source+")",s),u=void 0===e?C:e>>>0;if(0==u)return[];if(0===i.length)return null===k(l,i)?[i]:[];for(var c=0,f=0,h=[];f<i.length;){l.lastIndex=A?f:0;var d,p=k(l,A?i:i.slice(f));if(null===p||(d=S(_(l.lastIndex+(A?0:f)),i.length))===c)f=x(i,f,a);else{if(h.push(i.slice(c,f)),h.length===u)return h;for(var g=1;g<=p.length-1;g++)if(h.push(p[g]),h.length===u)return h;f=c=d}}return h.push(i.slice(c)),h}]})},function(t,e,n){"use strict";function r(t){var e=c(t,!1);if("string"==typeof e&&2<e.length){var n,r,i,o=(e=m?e.trim():d(e,3)).charCodeAt(0);if(43===o||45===o){if(88===(n=e.charCodeAt(2))||120===n)return NaN}else if(48===o){switch(e.charCodeAt(1)){case 66:case 98:r=2,i=49;break;case 79:case 111:r=8,i=55;break;default:return+e}for(var a,s=e.slice(2),l=0,u=s.length;l<u;l++)if((a=s.charCodeAt(l))<48||i<a)return NaN;return parseInt(s,r)}}return+e}var i=n(8),o=n(23),a=n(55),s=n(226),c=n(53),l=n(10),u=n(57).f,f=n(67).f,h=n(11).f,d=n(227).trim,p="Number",g=i[p],v=g,y=g.prototype,b=a(n(66)(y))==p,m="trim"in String.prototype;if(!g(" 0o1")||!g("0b1")||g("+0x1")){g=function(t){var e=arguments.length<1?0:t,n=this;return n instanceof g&&(b?l(function(){y.valueOf.call(n)}):a(n)!=p)?s(new v(r(e)),n,g):r(e)};for(var w,x=n(9)?u(v):"MAX_VALUE,MIN_VALUE,NaN,NEGATIVE_INFINITY,POSITIVE_INFINITY,EPSILON,isFinite,isInteger,isNaN,isSafeInteger,MAX_SAFE_INTEGER,MIN_SAFE_INTEGER,parseFloat,parseInt,isInteger".split(","),_=0;x.length>_;_++)o(v,w=x[_])&&!o(g,w)&&h(g,w,f(v,w));(g.prototype=y).constructor=g,n(22)(i,p,g)}},function(t,e,n){var o=n(18),a=n(147).set;t.exports=function(t,e,n){var r,i=e.constructor;return i!==n&&"function"==typeof i&&(r=i.prototype)!==n.prototype&&o(r)&&a&&a(t,r),t}},function(t,e,n){function r(t,e,n){var r={},i=s(function(){return!!l[t]()||"​"!="​"[t]()}),o=r[t]=i?e(f):l[t];n&&(r[n]=o),a(a.P+a.F*i,"String",r)}var a=n(6),i=n(30),s=n(10),l=n(228),o="["+l+"]",u=RegExp("^"+o+o+"*"),c=RegExp(o+o+"*$"),f=r.trim=function(t,e){return t=String(i(t)),1&e&&(t=t.replace(u,"")),2&e&&(t=t.replace(c,"")),t};t.exports=r},function(t,e){t.exports="\t\n\v\f\r   ᠎             　\u2028\u2029\ufeff"},function(t,e,n){var r=n(6),i=n(8).isFinite;r(r.S,"Number",{isFinite:function(t){return"number"==typeof t&&i(t)}})},function(t,e,n){"use strict";var r=n(6),i=n(82)(!0);r(r.P,"Array",{includes:function(t,e){return i(this,t,1<arguments.length?e:void 0)}}),n(87)("includes")},function(t,e,n){"use strict";var r=n(6),i=n(232);r(r.P+r.F*n(233)("includes"),"String",{includes:function(t,e){return!!~i(this,t,"includes").indexOf(t,1<arguments.length?e:void 0)}})},function(t,e,n){var r=n(146),i=n(30);t.exports=function(t,e,n){if(r(e))throw TypeError("String#"+n+" doesn't accept regex!");return String(i(t))}},function(t,e,n){var r=n(3)("match");t.exports=function(e){var n=/./;try{"/./"[e](n)}catch(t){try{return n[r]=!1,!"/./"[e](n)}catch(t){}}return!0}},function(t,e,n){var r=n(11).f,i=Function.prototype,o=/^\s*function ([^ (]*)/;"name"in i||n(9)&&r(i,"name",{configurable:!0,get:function(){try{return(""+this).match(o)[1]}catch(t){return""}}})},function(t,e,n){var r=n(6),o=n(56),a=String.fromCharCode,i=String.fromCodePoint;r(r.S+r.F*(!!i&&1!=i.length),"String",{fromCodePoint:function(t){for(var e,n=[],r=arguments.length,i=0;i<r;){if(e=+arguments[i++],o(e,1114111)!==e)throw RangeError(e+" is not a valid code point");n.push(e<65536?a(e):a(55296+((e-=65536)>>10),e%1024+56320))}return n.join("")}})},function(t,e,n){"use strict";var r=n(6),i=n(90)(!1);r(r.P,"String",{codePointAt:function(t){return i(this,t)}})},function(t,e,n){"use strict";var S=n(13),r=n(19),C=n(16),A=n(31),P=n(89),E=n(91),O=Math.max,T=Math.min,h=Math.floor,d=/\$([$&`']|\d\d?|<[^>]*>)/g,p=/\$([$&`']|\d\d?)/g;n(92)("replace",2,function(i,o,x,_){return[function(t,e){var n=i(this),r=null==t?void 0:t[o];return void 0!==r?r.call(t,n,e):x.call(String(n),t,e)},function(t,e){var n=_(x,t,this,e);if(n.done)return n.value;var r=S(t),i=String(this),o="function"==typeof e;o||(e=String(e));var a=r.global;if(a){var s=r.unicode;r.lastIndex=0}for(var l=[];;){var u=E(r,i);if(null===u)break;if(l.push(u),!a)break;""===String(u[0])&&(r.lastIndex=P(i,C(r.lastIndex),s))}for(var c,f="",h=0,d=0;d<l.length;d++){u=l[d];for(var p=String(u[0]),g=O(T(A(u.index),i.length),0),v=[],y=1;y<u.length;y++)v.push(void 0===(c=u[y])?c:String(c));var b=u.groups;if(o){var m=[p].concat(v,g,i);void 0!==b&&m.push(b);var w=String(e.apply(void 0,m))}else w=k(p,i,g,v,b,e);h<=g&&(f+=i.slice(h,g)+w,h=g+p.length)}return f+i.slice(h)}];function k(o,a,s,l,u,t){var c=s+o.length,f=l.length,e=p;return void 0!==u&&(u=r(u),e=d),x.call(t,e,function(t,e){var n;switch(e.charAt(0)){case"$":return"$";case"&":return o;case"`":return a.slice(0,s);case"'":return a.slice(c);case"<":n=u[e.slice(1,-1)];break;default:var r=+e;if(0==r)return t;if(f<r){var i=h(r/10);return 0===i?t:i<=f?void 0===l[i-1]?e.charAt(1):l[i-1]+e.charAt(1):t}n=l[r-1]}return void 0===n?"":n})}})},function(t,e,n){var r=n(19),i=n(42);n(239)("keys",function(){return function(t){return i(r(t))}})},function(t,e,n){var i=n(6),o=n(39),a=n(10);t.exports=function(t,e){var n=(o.Object||{})[t]||Object[t],r={};r[t]=e(n),i(i.S+i.F*a(function(){n(1)}),"Object",r)}},function(t,e,n){"use strict";var r=n(90)(!0);n(144)(String,"String",function(t){this._t=String(t),this._i=0},function(){var t,e=this._t,n=this._i;return n>=e.length?{value:void 0,done:!0}:(t=r(e,n),this._i+=t.length,{value:t,done:!1})})},function(t,e,n){"use strict";var p=n(54),r=n(6),g=n(19),v=n(242),y=n(139),b=n(16),m=n(243),w=n(141);r(r.S+r.F*!n(145)(function(t){Array.from(t)}),"Array",{from:function(t,e,n){var r,i,o,a,s=g(t),l="function"==typeof this?this:Array,u=arguments.length,c=1<u?e:void 0,f=void 0!==c,h=0,d=w(s);if(f&&(c=p(c,2<u?n:void 0,2)),null==d||l==Array&&y(d))for(i=new l(r=b(s.length));h<r;h++)m(i,h,f?c(s[h],h):s[h]);else for(a=d.call(s),i=new l;!(o=a.next()).done;h++)m(i,h,f?v(a,c,[o.value,h],!0):o.value);return i.length=h,i}})},function(t,e,n){var o=n(13);t.exports=function(e,t,n,r){try{return r?t(o(n)[0],n[1]):t(n)}catch(t){var i=e.return;throw void 0!==i&&o(i.call(e)),t}}},function(t,e,n){"use strict";var r=n(11),i=n(40);t.exports=function(t,e,n){e in t?r.f(t,e,i(0,n)):t[e]=n}},function(t,e,n){"use strict";n(245);function r(t){n(22)(RegExp.prototype,s,t,!0)}var i=n(13),o=n(94),a=n(9),s="toString",l=/./[s];n(10)(function(){return"/a/b"!=l.call({source:"a",flags:"b"})})?r(function(){var t=i(this);return"/".concat(t.source,"/","flags"in t?t.flags:!a&&t instanceof RegExp?o.call(t):void 0)}):l.name!=s&&r(function(){return l.call(this)})},function(t,e,n){n(9)&&"g"!=/./g.flags&&n(11).f(RegExp.prototype,"flags",{configurable:!0,get:n(94)})},function(t,e,n){"use strict";var r=n(65),i={};i[n(3)("toStringTag")]="z",i+""!="[object z]"&&n(22)(Object.prototype,"toString",function(){return"[object "+r(this)+"]"},!0)},function(t,e,n){n(148)("asyncIterator")},function(t,e,n){"use strict";function r(t){var e=Z[t]=T(N[j]);return e._k=t,e}function i(t,e){S(t);for(var n,r=_(e=P(e)),i=0,o=r.length;i<o;)et(t,n=r[i++],e[n]);return t}function o(t){var e=V.call(this,t=E(t,!0));return!(this===X&&c(Z,t)&&!c(Y,t))&&(!(e||!c(this,t)||!c(Z,t)||c(this,G)&&this[G][t])||e)}function a(t,e){if(t=P(t),e=E(e,!0),t!==X||!c(Z,e)||c(Y,e)){var n=F(t,e);return!n||!c(Z,e)||c(t,G)&&t[G][e]||(n.enumerable=!0),n}}function s(t){for(var e,n=z(P(t)),r=[],i=0;n.length>i;)c(Z,e=n[i++])||e==G||e==p||r.push(e);return r}function l(t){for(var e,n=t===X,r=z(n?Y:P(t)),i=[],o=0;r.length>o;)!c(Z,e=r[o++])||n&&!c(X,e)||i.push(Z[e]);return i}var u=n(8),c=n(23),f=n(9),h=n(6),d=n(22),p=n(249).KEY,g=n(10),v=n(62),y=n(64),b=n(29),m=n(3),w=n(149),x=n(148),_=n(250),k=n(142),S=n(13),C=n(18),A=n(19),P=n(43),E=n(53),O=n(40),T=n(66),I=n(251),B=n(67),L=n(85),R=n(11),M=n(42),F=B.f,D=R.f,z=I.f,N=u.Symbol,U=u.JSON,W=U&&U.stringify,j="prototype",G=m("_hidden"),H=m("toPrimitive"),V={}.propertyIsEnumerable,q=v("symbol-registry"),Z=v("symbols"),Y=v("op-symbols"),X=Object[j],K="function"==typeof N&&!!L.f,J=u.QObject,Q=!J||!J[j]||!J[j].findChild,$=f&&g(function(){return 7!=T(D({},"a",{get:function(){return D(this,"a",{value:7}).a}})).a})?function(t,e,n){var r=F(X,e);r&&delete X[e],D(t,e,n),r&&t!==X&&D(X,e,r)}:D,tt=K&&"symbol"==typeof N.iterator?function(t){return"symbol"==typeof t}:function(t){return t instanceof N},et=function(t,e,n){return t===X&&et(Y,e,n),S(t),e=E(e,!0),S(n),c(Z,e)?(n.enumerable?(c(t,G)&&t[G][e]&&(t[G][e]=!1),n=T(n,{enumerable:O(0,!1)})):(c(t,G)||D(t,G,O(1,{})),t[G][e]=!0),$(t,e,n)):D(t,e,n)};K||(d((N=function(t){if(this instanceof N)throw TypeError("Symbol is not a constructor!");var e=b(0<arguments.length?t:void 0),n=function(t){this===X&&n.call(Y,t),c(this,G)&&c(this[G],e)&&(this[G][e]=!1),$(this,e,O(1,t))};return f&&Q&&$(X,e,{configurable:!0,set:n}),r(e)})[j],"toString",function(){return this._k}),B.f=a,R.f=et,n(57).f=I.f=s,n(63).f=o,L.f=l,f&&!n(41)&&d(X,"propertyIsEnumerable",o,!0),w.f=function(t){return r(m(t))}),h(h.G+h.W+h.F*!K,{Symbol:N});for(var nt="hasInstance,isConcatSpreadable,iterator,match,replace,search,species,split,toPrimitive,toStringTag,unscopables".split(","),rt=0;nt.length>rt;)m(nt[rt++]);for(var it=M(m.store),ot=0;it.length>ot;)x(it[ot++]);h(h.S+h.F*!K,"Symbol",{for:function(t){return c(q,t+="")?q[t]:q[t]=N(t)},keyFor:function(t){if(!tt(t))throw TypeError(t+" is not a symbol!");for(var e in q)if(q[e]===t)return e},useSetter:function(){Q=!0},useSimple:function(){Q=!1}}),h(h.S+h.F*!K,"Object",{create:function(t,e){return void 0===e?T(t):i(T(t),e)},defineProperty:et,defineProperties:i,getOwnPropertyDescriptor:a,getOwnPropertyNames:s,getOwnPropertySymbols:l});var at=g(function(){L.f(1)});h(h.S+h.F*at,"Object",{getOwnPropertySymbols:function(t){return L.f(A(t))}}),U&&h(h.S+h.F*(!K||g(function(){var t=N();return"[null]"!=W([t])||"{}"!=W({a:t})||"{}"!=W(Object(t))})),"JSON",{stringify:function(t){for(var e,n,r=[t],i=1;i<arguments.length;)r.push(arguments[i++]);if(n=e=r[1],(C(e)||void 0!==t)&&!tt(t))return k(e)||(e=function(t,e){if("function"==typeof n&&(e=n.call(this,t,e)),!tt(e))return e}),r[1]=e,W.apply(U,r)}}),N[j][H]||n(15)(N[j],H,N[j].valueOf),y(N,"Symbol"),y(Math,"Math",!0),y(u.JSON,"JSON",!0)},function(t,e,n){function r(t){s(t,i,{value:{i:"O"+ ++l,w:{}}})}var i=n(29)("meta"),o=n(18),a=n(23),s=n(11).f,l=0,u=Object.isExtensible||function(){return!0},c=!n(10)(function(){return u(Object.preventExtensions({}))}),f=t.exports={KEY:i,NEED:!1,fastKey:function(t,e){if(!o(t))return"symbol"==typeof t?t:("string"==typeof t?"S":"P")+t;if(!a(t,i)){if(!u(t))return"F";if(!e)return"E";r(t)}return t[i].i},getWeak:function(t,e){if(!a(t,i)){if(!u(t))return!0;if(!e)return!1;r(t)}return t[i].w},onFreeze:function(t){return c&&f.NEED&&u(t)&&!a(t,i)&&r(t),t}}},function(t,e,n){var s=n(42),l=n(85),u=n(63);t.exports=function(t){var e=s(t),n=l.f;if(n)for(var r,i=n(t),o=u.f,a=0;i.length>a;)o.call(t,r=i[a++])&&e.push(r);return e}},function(t,e,n){var r=n(43),i=n(57).f,o={}.toString,a="object"==typeof window&&window&&Object.getOwnPropertyNames?Object.getOwnPropertyNames(window):[];t.exports.f=function(t){return a&&"[object Window]"==o.call(t)?function(t){try{return i(t)}catch(t){return a.slice()}}(t):i(r(t))}},function(t,e,n){for(var r=n(88),i=n(42),o=n(22),a=n(8),s=n(15),l=n(44),u=n(3),c=u("iterator"),f=u("toStringTag"),h=l.Array,d={CSSRuleList:!0,CSSStyleDeclaration:!1,CSSValueList:!1,ClientRectList:!1,DOMRectList:!1,DOMStringList:!1,DOMTokenList:!0,DataTransferItemList:!1,FileList:!1,HTMLAllCollection:!1,HTMLCollection:!1,HTMLFormElement:!1,HTMLSelectElement:!1,MediaList:!0,MimeTypeArray:!1,NamedNodeMap:!1,NodeList:!0,PaintRequestList:!1,Plugin:!1,PluginArray:!1,SVGLengthList:!1,SVGNumberList:!1,SVGPathSegList:!1,SVGPointList:!1,SVGStringList:!1,SVGTransformList:!1,SourceBufferList:!1,StyleSheetList:!0,TextTrackCueList:!1,TextTrackList:!1,TouchList:!1},p=i(d),g=0;g<p.length;g++){var v,y=p[g],b=d[y],m=a[y],w=m&&m.prototype;if(w&&(w[c]||s(w,c,h),w[f]||s(w,f,y),l[y]=h,b))for(v in r)w[v]||o(w,v,r[v],!0)}},function(t,e,n){var r=n(6);r(r.S,"Object",{setPrototypeOf:n(147).set})},function(t,e){},function(t,e,n){"use strict";var s=n(70).Buffer,r=n(256);function i(){!function(t,e){if(!(t instanceof e))throw new TypeError("Cannot call a class as a function")}(this,i),this.head=null,this.tail=null,this.length=0}t.exports=(i.prototype.push=function(t){var e={data:t,next:null};0<this.length?this.tail.next=e:this.head=e,this.tail=e,++this.length},i.prototype.unshift=function(t){var e={data:t,next:this.head};0===this.length&&(this.tail=e),this.head=e,++this.length},i.prototype.shift=function(){if(0!==this.length){var t=this.head.data;return 1===this.length?this.head=this.tail=null:this.head=this.head.next,--this.length,t}},i.prototype.clear=function(){this.head=this.tail=null,this.length=0},i.prototype.join=function(t){if(0===this.length)return"";for(var e=this.head,n=""+e.data;e=e.next;)n+=t+e.data;return n},i.prototype.concat=function(t){if(0===this.length)return s.alloc(0);if(1===this.length)return this.head.data;for(var e,n,r,i=s.allocUnsafe(t>>>0),o=this.head,a=0;o;)e=o.data,n=i,r=a,e.copy(n,r),a+=o.data.length,o=o.next;return i},i),r&&r.inspect&&r.inspect.custom&&(t.exports.prototype[r.inspect.custom]=function(){var t=r.inspect({length:this.length});return this.constructor.name+" "+t})},function(t,e){},function(t,e,n){(function(n){function r(t){try{if(!n.localStorage)return!1}catch(t){return!1}var e=n.localStorage[t];return null!=e&&"true"===String(e).toLowerCase()}t.exports=function(t,e){if(r("noDeprecation"))return t;var n=!1;return function(){if(!n){if(r("throwDeprecation"))throw new Error(e);r("traceDeprecation"),n=!0}return t.apply(this,arguments)}}}).call(this,n(25))},function(t,e,n){"use strict";t.exports=o;var r=n(153),i=n(58);function o(t){if(!(this instanceof o))return new o(t);r.call(this,t)}i.inherits=n(32),i.inherits(o,r),o.prototype._transform=function(t,e,n){n(null,t)}},function(t,e,n){t.exports=n(97)},function(t,e,n){t.exports=n(33)},function(t,e,n){t.exports=n(96).Transform},function(t,e,n){t.exports=n(96).PassThrough},function(t,h,r){"use strict";(function(u,c){var f=r(155),o=r(266),a=r(267),s=r(270),t=r(273);for(var e in t)h[e]=t[e];h.NONE=0,h.DEFLATE=1,h.INFLATE=2,h.GZIP=3,h.GUNZIP=4,h.DEFLATERAW=5,h.INFLATERAW=6,h.UNZIP=7;function n(t){if("number"!=typeof t||t<h.DEFLATE||t>h.UNZIP)throw new TypeError("Bad argument");this.dictionary=null,this.err=0,this.flush=0,this.init_done=!1,this.level=0,this.memLevel=0,this.mode=t,this.strategy=0,this.windowBits=0,this.write_in_progress=!1,this.pending_close=!1,this.gzip_id_bytes_read=0}n.prototype.close=function(){this.write_in_progress?this.pending_close=!0:(this.pending_close=!1,f(this.init_done,"close before init"),f(this.mode<=h.UNZIP),this.mode===h.DEFLATE||this.mode===h.GZIP||this.mode===h.DEFLATERAW?a.deflateEnd(this.strm):this.mode!==h.INFLATE&&this.mode!==h.GUNZIP&&this.mode!==h.INFLATERAW&&this.mode!==h.UNZIP||s.inflateEnd(this.strm),this.mode=h.NONE,this.dictionary=null)},n.prototype.write=function(t,e,n,r,i,o,a){return this._write(!0,t,e,n,r,i,o,a)},n.prototype.writeSync=function(t,e,n,r,i,o,a){return this._write(!1,t,e,n,r,i,o,a)},n.prototype._write=function(t,e,n,r,i,o,a,s){if(f.equal(arguments.length,8),f(this.init_done,"write before init"),f(this.mode!==h.NONE,"already finalized"),f.equal(!1,this.write_in_progress,"write already in progress"),f.equal(!1,this.pending_close,"close is pending"),this.write_in_progress=!0,f.equal(!1,void 0===e,"must provide flush value"),this.write_in_progress=!0,e!==h.Z_NO_FLUSH&&e!==h.Z_PARTIAL_FLUSH&&e!==h.Z_SYNC_FLUSH&&e!==h.Z_FULL_FLUSH&&e!==h.Z_FINISH&&e!==h.Z_BLOCK)throw new Error("Invalid flush value");if(null==n&&(n=u.alloc(0),r=i=0),this.strm.avail_in=i,this.strm.input=n,this.strm.next_in=r,this.strm.avail_out=s,this.strm.output=o,this.strm.next_out=a,this.flush=e,!t)return this._process(),this._checkError()?this._afterSync():void 0;var l=this;return c.nextTick(function(){l._process(),l._after()}),this},n.prototype._afterSync=function(){var t=this.strm.avail_out,e=this.strm.avail_in;return this.write_in_progress=!1,[e,t]},n.prototype._process=function(){var t=null;switch(this.mode){case h.DEFLATE:case h.GZIP:case h.DEFLATERAW:this.err=a.deflate(this.strm,this.flush);break;case h.UNZIP:switch(0<this.strm.avail_in&&(t=this.strm.next_in),this.gzip_id_bytes_read){case 0:if(null===t)break;if(31!==this.strm.input[t]){this.mode=h.INFLATE;break}if(t++,(this.gzip_id_bytes_read=1)===this.strm.avail_in)break;case 1:if(null===t)break;139===this.strm.input[t]?(this.gzip_id_bytes_read=2,this.mode=h.GUNZIP):this.mode=h.INFLATE;break;default:throw new Error("invalid number of gzip magic number bytes read")}case h.INFLATE:case h.GUNZIP:case h.INFLATERAW:for(this.err=s.inflate(this.strm,this.flush),this.err===h.Z_NEED_DICT&&this.dictionary&&(this.err=s.inflateSetDictionary(this.strm,this.dictionary),this.err===h.Z_OK?this.err=s.inflate(this.strm,this.flush):this.err===h.Z_DATA_ERROR&&(this.err=h.Z_NEED_DICT));0<this.strm.avail_in&&this.mode===h.GUNZIP&&this.err===h.Z_STREAM_END&&0!==this.strm.next_in[0];)this.reset(),this.err=s.inflate(this.strm,this.flush);break;default:throw new Error("Unknown mode "+this.mode)}},n.prototype._checkError=function(){switch(this.err){case h.Z_OK:case h.Z_BUF_ERROR:if(0!==this.strm.avail_out&&this.flush===h.Z_FINISH)return this._error("unexpected end of file"),!1;break;case h.Z_STREAM_END:break;case h.Z_NEED_DICT:return null==this.dictionary?this._error("Missing dictionary"):this._error("Bad dictionary"),!1;default:return this._error("Zlib error"),!1}return!0},n.prototype._after=function(){if(this._checkError()){var t=this.strm.avail_out,e=this.strm.avail_in;this.write_in_progress=!1,this.callback(e,t),this.pending_close&&this.close()}},n.prototype._error=function(t){this.strm.msg&&(t=this.strm.msg),this.onerror(t,this.err),this.write_in_progress=!1,this.pending_close&&this.close()},n.prototype.init=function(t,e,n,r,i){f(4===arguments.length||5===arguments.length,"init(windowBits, level, memLevel, strategy, [dictionary])"),f(8<=t&&t<=15,"invalid windowBits"),f(-1<=e&&e<=9,"invalid compression level"),f(1<=n&&n<=9,"invalid memlevel"),f(r===h.Z_FILTERED||r===h.Z_HUFFMAN_ONLY||r===h.Z_RLE||r===h.Z_FIXED||r===h.Z_DEFAULT_STRATEGY,"invalid strategy"),this._init(e,t,n,r,i),this._setDictionary()},n.prototype.params=function(){throw new Error("deflateParams Not supported")},n.prototype.reset=function(){this._reset(),this._setDictionary()},n.prototype._init=function(t,e,n,r,i){switch(this.level=t,this.windowBits=e,this.memLevel=n,this.strategy=r,this.flush=h.Z_NO_FLUSH,this.err=h.Z_OK,this.mode!==h.GZIP&&this.mode!==h.GUNZIP||(this.windowBits+=16),this.mode===h.UNZIP&&(this.windowBits+=32),this.mode!==h.DEFLATERAW&&this.mode!==h.INFLATERAW||(this.windowBits=-1*this.windowBits),this.strm=new o,this.mode){case h.DEFLATE:case h.GZIP:case h.DEFLATERAW:this.err=a.deflateInit2(this.strm,this.level,h.Z_DEFLATED,this.windowBits,this.memLevel,this.strategy);break;case h.INFLATE:case h.GUNZIP:case h.INFLATERAW:case h.UNZIP:this.err=s.inflateInit2(this.strm,this.windowBits);break;default:throw new Error("Unknown mode "+this.mode)}this.err!==h.Z_OK&&this._error("Init error"),this.dictionary=i,this.write_in_progress=!1,this.init_done=!0},n.prototype._setDictionary=function(){if(null!=this.dictionary){switch(this.err=h.Z_OK,this.mode){case h.DEFLATE:case h.DEFLATERAW:this.err=a.deflateSetDictionary(this.strm,this.dictionary)}this.err!==h.Z_OK&&this._error("Failed to set dictionary")}},n.prototype._reset=function(){switch(this.err=h.Z_OK,this.mode){case h.DEFLATE:case h.DEFLATERAW:case h.GZIP:this.err=a.deflateReset(this.strm);break;case h.INFLATE:case h.INFLATERAW:case h.GUNZIP:this.err=s.inflateReset(this.strm)}this.err!==h.Z_OK&&this._error("Failed to reset stream")},h.Zlib=n}).call(this,r(4).Buffer,r(24))},function(t,e,n){"use strict";
/*
object-assign
(c) Sindre Sorhus
@license MIT
*/var l=Object.getOwnPropertySymbols,u=Object.prototype.hasOwnProperty,c=Object.prototype.propertyIsEnumerable;t.exports=function(){try{if(!Object.assign)return!1;var t=new String("abc");if(t[5]="de","5"===Object.getOwnPropertyNames(t)[0])return!1;for(var e={},n=0;n<10;n++)e["_"+String.fromCharCode(n)]=n;if("0123456789"!==Object.getOwnPropertyNames(e).map(function(t){return e[t]}).join(""))return!1;var r={};return"abcdefghijklmnopqrst".split("").forEach(function(t){r[t]=t}),"abcdefghijklmnopqrst"===Object.keys(Object.assign({},r)).join("")}catch(t){return!1}}()?Object.assign:function(t,e){for(var n,r,i=function(t){if(null==t)throw new TypeError("Object.assign cannot be called with null or undefined");return Object(t)}(t),o=1;o<arguments.length;o++){for(var a in n=Object(arguments[o]))u.call(n,a)&&(i[a]=n[a]);if(l){r=l(n);for(var s=0;s<r.length;s++)c.call(n,r[s])&&(i[r[s]]=n[r[s]])}}return i}},function(t,e){t.exports=function(t){return t&&"object"==typeof t&&"function"==typeof t.copy&&"function"==typeof t.fill&&"function"==typeof t.readUInt8}},function(t,e,n){"use strict";t.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},function(t,e,n){"use strict";var l,h=n(71),u=n(268),d=n(156),p=n(157),r=n(269),c=0,f=4,g=0,v=-2,y=-1,b=4,i=2,m=8,w=9,o=286,a=30,s=19,x=2*o+1,_=15,k=3,S=258,C=S+k+1,A=42,P=113,E=1,O=2,T=3,I=4;function B(t,e){return t.msg=r[e],e}function L(t){return(t<<1)-(4<t?9:0)}function R(t){for(var e=t.length;0<=--e;)t[e]=0}function M(t){var e=t.state,n=e.pending;n>t.avail_out&&(n=t.avail_out),0!==n&&(h.arraySet(t.output,e.pending_buf,e.pending_out,n,t.next_out),t.next_out+=n,e.pending_out+=n,t.total_out+=n,t.avail_out-=n,e.pending-=n,0===e.pending&&(e.pending_out=0))}function F(t,e){u._tr_flush_block(t,0<=t.block_start?t.block_start:-1,t.strstart-t.block_start,e),t.block_start=t.strstart,M(t.strm)}function D(t,e){t.pending_buf[t.pending++]=e}function z(t,e){t.pending_buf[t.pending++]=e>>>8&255,t.pending_buf[t.pending++]=255&e}function N(t,e){var n,r,i=t.max_chain_length,o=t.strstart,a=t.prev_length,s=t.nice_match,l=t.strstart>t.w_size-C?t.strstart-(t.w_size-C):0,u=t.window,c=t.w_mask,f=t.prev,h=t.strstart+S,d=u[o+a-1],p=u[o+a];t.prev_length>=t.good_match&&(i>>=2),s>t.lookahead&&(s=t.lookahead);do{if(u[(n=e)+a]===p&&u[n+a-1]===d&&u[n]===u[o]&&u[++n]===u[o+1]){o+=2,n++;do{}while(u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&u[++o]===u[++n]&&o<h);if(r=S-(h-o),o=h-S,a<r){if(t.match_start=e,s<=(a=r))break;d=u[o+a-1],p=u[o+a]}}}while((e=f[e&c])>l&&0!=--i);return a<=t.lookahead?a:t.lookahead}function U(t){var e,n,r,i,o,a,s,l,u,c,f=t.w_size;do{if(i=t.window_size-t.lookahead-t.strstart,t.strstart>=f+(f-C)){for(h.arraySet(t.window,t.window,f,f,0),t.match_start-=f,t.strstart-=f,t.block_start-=f,e=n=t.hash_size;r=t.head[--e],t.head[e]=f<=r?r-f:0,--n;);for(e=n=f;r=t.prev[--e],t.prev[e]=f<=r?r-f:0,--n;);i+=f}if(0===t.strm.avail_in)break;if(a=t.strm,s=t.window,l=t.strstart+t.lookahead,u=i,c=void 0,c=a.avail_in,u<c&&(c=u),n=0===c?0:(a.avail_in-=c,h.arraySet(s,a.input,a.next_in,c,l),1===a.state.wrap?a.adler=d(a.adler,s,c,l):2===a.state.wrap&&(a.adler=p(a.adler,s,c,l)),a.next_in+=c,a.total_in+=c,c),t.lookahead+=n,t.lookahead+t.insert>=k)for(o=t.strstart-t.insert,t.ins_h=t.window[o],t.ins_h=(t.ins_h<<t.hash_shift^t.window[o+1])&t.hash_mask;t.insert&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[o+k-1])&t.hash_mask,t.prev[o&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=o,o++,t.insert--,!(t.lookahead+t.insert<k)););}while(t.lookahead<C&&0!==t.strm.avail_in)}function W(t,e){for(var n,r;;){if(t.lookahead<C){if(U(t),t.lookahead<C&&e===c)return E;if(0===t.lookahead)break}if(n=0,t.lookahead>=k&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+k-1])&t.hash_mask,n=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!==n&&t.strstart-n<=t.w_size-C&&(t.match_length=N(t,n)),t.match_length>=k)if(r=u._tr_tally(t,t.strstart-t.match_start,t.match_length-k),t.lookahead-=t.match_length,t.match_length<=t.max_lazy_match&&t.lookahead>=k){for(t.match_length--;t.strstart++,t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+k-1])&t.hash_mask,n=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart,0!=--t.match_length;);t.strstart++}else t.strstart+=t.match_length,t.match_length=0,t.ins_h=t.window[t.strstart],t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+1])&t.hash_mask;else r=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++;if(r&&(F(t,!1),0===t.strm.avail_out))return E}return t.insert=t.strstart<k-1?t.strstart:k-1,e===f?(F(t,!0),0===t.strm.avail_out?T:I):t.last_lit&&(F(t,!1),0===t.strm.avail_out)?E:O}function j(t,e){for(var n,r,i;;){if(t.lookahead<C){if(U(t),t.lookahead<C&&e===c)return E;if(0===t.lookahead)break}if(n=0,t.lookahead>=k&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+k-1])&t.hash_mask,n=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),t.prev_length=t.match_length,t.prev_match=t.match_start,t.match_length=k-1,0!==n&&t.prev_length<t.max_lazy_match&&t.strstart-n<=t.w_size-C&&(t.match_length=N(t,n),t.match_length<=5&&(1===t.strategy||t.match_length===k&&4096<t.strstart-t.match_start)&&(t.match_length=k-1)),t.prev_length>=k&&t.match_length<=t.prev_length){for(i=t.strstart+t.lookahead-k,r=u._tr_tally(t,t.strstart-1-t.prev_match,t.prev_length-k),t.lookahead-=t.prev_length-1,t.prev_length-=2;++t.strstart<=i&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+k-1])&t.hash_mask,n=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!=--t.prev_length;);if(t.match_available=0,t.match_length=k-1,t.strstart++,r&&(F(t,!1),0===t.strm.avail_out))return E}else if(t.match_available){if((r=u._tr_tally(t,0,t.window[t.strstart-1]))&&F(t,!1),t.strstart++,t.lookahead--,0===t.strm.avail_out)return E}else t.match_available=1,t.strstart++,t.lookahead--}return t.match_available&&(r=u._tr_tally(t,0,t.window[t.strstart-1]),t.match_available=0),t.insert=t.strstart<k-1?t.strstart:k-1,e===f?(F(t,!0),0===t.strm.avail_out?T:I):t.last_lit&&(F(t,!1),0===t.strm.avail_out)?E:O}function G(t,e,n,r,i){this.good_length=t,this.max_lazy=e,this.nice_length=n,this.max_chain=r,this.func=i}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=m,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new h.Buf16(2*x),this.dyn_dtree=new h.Buf16(2*(2*a+1)),this.bl_tree=new h.Buf16(2*(2*s+1)),R(this.dyn_ltree),R(this.dyn_dtree),R(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new h.Buf16(_+1),this.heap=new h.Buf16(2*o+1),R(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new h.Buf16(2*o+1),R(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function V(t){var e;return t&&t.state?(t.total_in=t.total_out=0,t.data_type=i,(e=t.state).pending=0,e.pending_out=0,e.wrap<0&&(e.wrap=-e.wrap),e.status=e.wrap?A:P,t.adler=2===e.wrap?0:1,e.last_flush=c,u._tr_init(e),g):B(t,v)}function q(t){var e=V(t);return e===g&&function(t){t.window_size=2*t.w_size,R(t.head),t.max_lazy_match=l[t.level].max_lazy,t.good_match=l[t.level].good_length,t.nice_match=l[t.level].nice_length,t.max_chain_length=l[t.level].max_chain,t.strstart=0,t.block_start=0,t.lookahead=0,t.insert=0,t.match_length=t.prev_length=k-1,t.match_available=0,t.ins_h=0}(t.state),e}function Z(t,e,n,r,i,o){if(!t)return v;var a=1;if(e===y&&(e=6),r<0?(a=0,r=-r):15<r&&(a=2,r-=16),i<1||w<i||n!==m||r<8||15<r||e<0||9<e||o<0||b<o)return B(t,v);8===r&&(r=9);var s=new H;return(t.state=s).strm=t,s.wrap=a,s.gzhead=null,s.w_bits=r,s.w_size=1<<s.w_bits,s.w_mask=s.w_size-1,s.hash_bits=i+7,s.hash_size=1<<s.hash_bits,s.hash_mask=s.hash_size-1,s.hash_shift=~~((s.hash_bits+k-1)/k),s.window=new h.Buf8(2*s.w_size),s.head=new h.Buf16(s.hash_size),s.prev=new h.Buf16(s.w_size),s.lit_bufsize=1<<i+6,s.pending_buf_size=4*s.lit_bufsize,s.pending_buf=new h.Buf8(s.pending_buf_size),s.d_buf=1*s.lit_bufsize,s.l_buf=3*s.lit_bufsize,s.level=e,s.strategy=o,s.method=n,q(t)}l=[new G(0,0,0,0,function(t,e){var n=65535;for(n>t.pending_buf_size-5&&(n=t.pending_buf_size-5);;){if(t.lookahead<=1){if(U(t),0===t.lookahead&&e===c)return E;if(0===t.lookahead)break}t.strstart+=t.lookahead,t.lookahead=0;var r=t.block_start+n;if((0===t.strstart||t.strstart>=r)&&(t.lookahead=t.strstart-r,t.strstart=r,F(t,!1),0===t.strm.avail_out))return E;if(t.strstart-t.block_start>=t.w_size-C&&(F(t,!1),0===t.strm.avail_out))return E}return t.insert=0,e===f?(F(t,!0),0===t.strm.avail_out?T:I):(t.strstart>t.block_start&&(F(t,!1),t.strm.avail_out),E)}),new G(4,4,8,4,W),new G(4,5,16,8,W),new G(4,6,32,32,W),new G(4,4,16,16,j),new G(8,16,32,32,j),new G(8,16,128,128,j),new G(8,32,128,256,j),new G(32,128,258,1024,j),new G(32,258,258,4096,j)],e.deflateInit=function(t,e){return Z(t,e,m,15,8,0)},e.deflateInit2=Z,e.deflateReset=q,e.deflateResetKeep=V,e.deflateSetHeader=function(t,e){return t&&t.state?2!==t.state.wrap?v:(t.state.gzhead=e,g):v},e.deflate=function(t,e){var n,r,i,o;if(!t||!t.state||5<e||e<0)return t?B(t,v):v;if(r=t.state,!t.output||!t.input&&0!==t.avail_in||666===r.status&&e!==f)return B(t,0===t.avail_out?-5:v);if(r.strm=t,n=r.last_flush,r.last_flush=e,r.status===A)if(2===r.wrap)t.adler=0,D(r,31),D(r,139),D(r,8),r.gzhead?(D(r,(r.gzhead.text?1:0)+(r.gzhead.hcrc?2:0)+(r.gzhead.extra?4:0)+(r.gzhead.name?8:0)+(r.gzhead.comment?16:0)),D(r,255&r.gzhead.time),D(r,r.gzhead.time>>8&255),D(r,r.gzhead.time>>16&255),D(r,r.gzhead.time>>24&255),D(r,9===r.level?2:2<=r.strategy||r.level<2?4:0),D(r,255&r.gzhead.os),r.gzhead.extra&&r.gzhead.extra.length&&(D(r,255&r.gzhead.extra.length),D(r,r.gzhead.extra.length>>8&255)),r.gzhead.hcrc&&(t.adler=p(t.adler,r.pending_buf,r.pending,0)),r.gzindex=0,r.status=69):(D(r,0),D(r,0),D(r,0),D(r,0),D(r,0),D(r,9===r.level?2:2<=r.strategy||r.level<2?4:0),D(r,3),r.status=P);else{var a=m+(r.w_bits-8<<4)<<8;a|=(2<=r.strategy||r.level<2?0:r.level<6?1:6===r.level?2:3)<<6,0!==r.strstart&&(a|=32),a+=31-a%31,r.status=P,z(r,a),0!==r.strstart&&(z(r,t.adler>>>16),z(r,65535&t.adler)),t.adler=1}if(69===r.status)if(r.gzhead.extra){for(i=r.pending;r.gzindex<(65535&r.gzhead.extra.length)&&(r.pending!==r.pending_buf_size||(r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),M(t),i=r.pending,r.pending!==r.pending_buf_size));)D(r,255&r.gzhead.extra[r.gzindex]),r.gzindex++;r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),r.gzindex===r.gzhead.extra.length&&(r.gzindex=0,r.status=73)}else r.status=73;if(73===r.status)if(r.gzhead.name){i=r.pending;do{if(r.pending===r.pending_buf_size&&(r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),M(t),i=r.pending,r.pending===r.pending_buf_size)){o=1;break}o=r.gzindex<r.gzhead.name.length?255&r.gzhead.name.charCodeAt(r.gzindex++):0,D(r,o)}while(0!==o);r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),0===o&&(r.gzindex=0,r.status=91)}else r.status=91;if(91===r.status)if(r.gzhead.comment){i=r.pending;do{if(r.pending===r.pending_buf_size&&(r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),M(t),i=r.pending,r.pending===r.pending_buf_size)){o=1;break}o=r.gzindex<r.gzhead.comment.length?255&r.gzhead.comment.charCodeAt(r.gzindex++):0,D(r,o)}while(0!==o);r.gzhead.hcrc&&r.pending>i&&(t.adler=p(t.adler,r.pending_buf,r.pending-i,i)),0===o&&(r.status=103)}else r.status=103;if(103===r.status&&(r.gzhead.hcrc?(r.pending+2>r.pending_buf_size&&M(t),r.pending+2<=r.pending_buf_size&&(D(r,255&t.adler),D(r,t.adler>>8&255),t.adler=0,r.status=P)):r.status=P),0!==r.pending){if(M(t),0===t.avail_out)return r.last_flush=-1,g}else if(0===t.avail_in&&L(e)<=L(n)&&e!==f)return B(t,-5);if(666===r.status&&0!==t.avail_in)return B(t,-5);if(0!==t.avail_in||0!==r.lookahead||e!==c&&666!==r.status){var s=2===r.strategy?function(t,e){for(var n;;){if(0===t.lookahead&&(U(t),0===t.lookahead)){if(e===c)return E;break}if(t.match_length=0,n=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++,n&&(F(t,!1),0===t.strm.avail_out))return E}return t.insert=0,e===f?(F(t,!0),0===t.strm.avail_out?T:I):t.last_lit&&(F(t,!1),0===t.strm.avail_out)?E:O}(r,e):3===r.strategy?function(t,e){for(var n,r,i,o,a=t.window;;){if(t.lookahead<=S){if(U(t),t.lookahead<=S&&e===c)return E;if(0===t.lookahead)break}if(t.match_length=0,t.lookahead>=k&&0<t.strstart&&(r=a[i=t.strstart-1])===a[++i]&&r===a[++i]&&r===a[++i]){o=t.strstart+S;do{}while(r===a[++i]&&r===a[++i]&&r===a[++i]&&r===a[++i]&&r===a[++i]&&r===a[++i]&&r===a[++i]&&r===a[++i]&&i<o);t.match_length=S-(o-i),t.match_length>t.lookahead&&(t.match_length=t.lookahead)}if(t.match_length>=k?(n=u._tr_tally(t,1,t.match_length-k),t.lookahead-=t.match_length,t.strstart+=t.match_length,t.match_length=0):(n=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++),n&&(F(t,!1),0===t.strm.avail_out))return E}return t.insert=0,e===f?(F(t,!0),0===t.strm.avail_out?T:I):t.last_lit&&(F(t,!1),0===t.strm.avail_out)?E:O}(r,e):l[r.level].func(r,e);if(s!==T&&s!==I||(r.status=666),s===E||s===T)return 0===t.avail_out&&(r.last_flush=-1),g;if(s===O&&(1===e?u._tr_align(r):5!==e&&(u._tr_stored_block(r,0,0,!1),3===e&&(R(r.head),0===r.lookahead&&(r.strstart=0,r.block_start=0,r.insert=0))),M(t),0===t.avail_out))return r.last_flush=-1,g}return e!==f?g:r.wrap<=0?1:(2===r.wrap?(D(r,255&t.adler),D(r,t.adler>>8&255),D(r,t.adler>>16&255),D(r,t.adler>>24&255),D(r,255&t.total_in),D(r,t.total_in>>8&255),D(r,t.total_in>>16&255),D(r,t.total_in>>24&255)):(z(r,t.adler>>>16),z(r,65535&t.adler)),M(t),0<r.wrap&&(r.wrap=-r.wrap),0!==r.pending?g:1)},e.deflateEnd=function(t){var e;return t&&t.state?(e=t.state.status)!==A&&69!==e&&73!==e&&91!==e&&103!==e&&e!==P&&666!==e?B(t,v):(t.state=null,e===P?B(t,-3):g):v},e.deflateSetDictionary=function(t,e){var n,r,i,o,a,s,l,u,c=e.length;if(!t||!t.state)return v;if(2===(o=(n=t.state).wrap)||1===o&&n.status!==A||n.lookahead)return v;for(1===o&&(t.adler=d(t.adler,e,c,0)),n.wrap=0,c>=n.w_size&&(0===o&&(R(n.head),n.strstart=0,n.block_start=0,n.insert=0),u=new h.Buf8(n.w_size),h.arraySet(u,e,c-n.w_size,n.w_size,0),e=u,c=n.w_size),a=t.avail_in,s=t.next_in,l=t.input,t.avail_in=c,t.next_in=0,t.input=e,U(n);n.lookahead>=k;){for(r=n.strstart,i=n.lookahead-(k-1);n.ins_h=(n.ins_h<<n.hash_shift^n.window[r+k-1])&n.hash_mask,n.prev[r&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=r,r++,--i;);n.strstart=r,n.lookahead=k-1,U(n)}return n.strstart+=n.lookahead,n.block_start=n.strstart,n.insert=n.lookahead,n.lookahead=0,n.match_length=n.prev_length=k-1,n.match_available=0,t.next_in=s,t.input=l,t.avail_in=a,n.wrap=o,g},e.deflateInfo="pako deflate (from Nodeca project)"},function(t,e,n){"use strict";var i=n(71),s=0,l=1;function r(t){for(var e=t.length;0<=--e;)t[e]=0}var o=0,a=29,u=256,c=u+1+a,f=30,h=19,v=2*c+1,y=15,d=16,p=7,g=256,b=16,m=17,w=18,x=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],_=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],k=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],C=new Array(2*(c+2));r(C);var A=new Array(2*f);r(A);var P=new Array(512);r(P);var E=new Array(256);r(E);var O=new Array(a);r(O);var T,I,B,L=new Array(f);function R(t,e,n,r,i){this.static_tree=t,this.extra_bits=e,this.extra_base=n,this.elems=r,this.max_length=i,this.has_stree=t&&t.length}function M(t,e){this.dyn_tree=t,this.max_code=0,this.stat_desc=e}function F(t){return t<256?P[t]:P[256+(t>>>7)]}function D(t,e){t.pending_buf[t.pending++]=255&e,t.pending_buf[t.pending++]=e>>>8&255}function z(t,e,n){t.bi_valid>d-n?(t.bi_buf|=e<<t.bi_valid&65535,D(t,t.bi_buf),t.bi_buf=e>>d-t.bi_valid,t.bi_valid+=n-d):(t.bi_buf|=e<<t.bi_valid&65535,t.bi_valid+=n)}function N(t,e,n){z(t,n[2*e],n[2*e+1])}function U(t,e){for(var n=0;n|=1&t,t>>>=1,n<<=1,0<--e;);return n>>>1}function W(t,e,n){var r,i,o=new Array(y+1),a=0;for(r=1;r<=y;r++)o[r]=a=a+n[r-1]<<1;for(i=0;i<=e;i++){var s=t[2*i+1];0!==s&&(t[2*i]=U(o[s]++,s))}}function j(t){var e;for(e=0;e<c;e++)t.dyn_ltree[2*e]=0;for(e=0;e<f;e++)t.dyn_dtree[2*e]=0;for(e=0;e<h;e++)t.bl_tree[2*e]=0;t.dyn_ltree[2*g]=1,t.opt_len=t.static_len=0,t.last_lit=t.matches=0}function G(t){8<t.bi_valid?D(t,t.bi_buf):0<t.bi_valid&&(t.pending_buf[t.pending++]=t.bi_buf),t.bi_buf=0,t.bi_valid=0}function H(t,e,n,r){var i=2*e,o=2*n;return t[i]<t[o]||t[i]===t[o]&&r[e]<=r[n]}function V(t,e,n){for(var r=t.heap[n],i=n<<1;i<=t.heap_len&&(i<t.heap_len&&H(e,t.heap[i+1],t.heap[i],t.depth)&&i++,!H(e,r,t.heap[i],t.depth));)t.heap[n]=t.heap[i],n=i,i<<=1;t.heap[n]=r}function q(t,e,n){var r,i,o,a,s=0;if(0!==t.last_lit)for(;r=t.pending_buf[t.d_buf+2*s]<<8|t.pending_buf[t.d_buf+2*s+1],i=t.pending_buf[t.l_buf+s],s++,0===r?N(t,i,e):(N(t,(o=E[i])+u+1,e),0!==(a=x[o])&&z(t,i-=O[o],a),N(t,o=F(--r),n),0!==(a=_[o])&&z(t,r-=L[o],a)),s<t.last_lit;);N(t,g,e)}function Z(t,e){var n,r,i,o=e.dyn_tree,a=e.stat_desc.static_tree,s=e.stat_desc.has_stree,l=e.stat_desc.elems,u=-1;for(t.heap_len=0,t.heap_max=v,n=0;n<l;n++)0!==o[2*n]?(t.heap[++t.heap_len]=u=n,t.depth[n]=0):o[2*n+1]=0;for(;t.heap_len<2;)o[2*(i=t.heap[++t.heap_len]=u<2?++u:0)]=1,t.depth[i]=0,t.opt_len--,s&&(t.static_len-=a[2*i+1]);for(e.max_code=u,n=t.heap_len>>1;1<=n;n--)V(t,o,n);for(i=l;n=t.heap[1],t.heap[1]=t.heap[t.heap_len--],V(t,o,1),r=t.heap[1],t.heap[--t.heap_max]=n,t.heap[--t.heap_max]=r,o[2*i]=o[2*n]+o[2*r],t.depth[i]=(t.depth[n]>=t.depth[r]?t.depth[n]:t.depth[r])+1,o[2*n+1]=o[2*r+1]=i,t.heap[1]=i++,V(t,o,1),2<=t.heap_len;);t.heap[--t.heap_max]=t.heap[1],function(t,e){var n,r,i,o,a,s,l=e.dyn_tree,u=e.max_code,c=e.stat_desc.static_tree,f=e.stat_desc.has_stree,h=e.stat_desc.extra_bits,d=e.stat_desc.extra_base,p=e.stat_desc.max_length,g=0;for(o=0;o<=y;o++)t.bl_count[o]=0;for(l[2*t.heap[t.heap_max]+1]=0,n=t.heap_max+1;n<v;n++)p<(o=l[2*l[2*(r=t.heap[n])+1]+1]+1)&&(o=p,g++),l[2*r+1]=o,u<r||(t.bl_count[o]++,a=0,d<=r&&(a=h[r-d]),s=l[2*r],t.opt_len+=s*(o+a),f&&(t.static_len+=s*(c[2*r+1]+a)));if(0!==g){do{for(o=p-1;0===t.bl_count[o];)o--;t.bl_count[o]--,t.bl_count[o+1]+=2,t.bl_count[p]--,g-=2}while(0<g);for(o=p;0!==o;o--)for(r=t.bl_count[o];0!==r;)u<(i=t.heap[--n])||(l[2*i+1]!==o&&(t.opt_len+=(o-l[2*i+1])*l[2*i],l[2*i+1]=o),r--)}}(t,e),W(o,u,t.bl_count)}function Y(t,e,n){var r,i,o=-1,a=e[1],s=0,l=7,u=4;for(0===a&&(l=138,u=3),e[2*(n+1)+1]=65535,r=0;r<=n;r++)i=a,a=e[2*(r+1)+1],++s<l&&i===a||(s<u?t.bl_tree[2*i]+=s:0!==i?(i!==o&&t.bl_tree[2*i]++,t.bl_tree[2*b]++):s<=10?t.bl_tree[2*m]++:t.bl_tree[2*w]++,o=i,u=(s=0)===a?(l=138,3):i===a?(l=6,3):(l=7,4))}function X(t,e,n){var r,i,o=-1,a=e[1],s=0,l=7,u=4;for(0===a&&(l=138,u=3),r=0;r<=n;r++)if(i=a,a=e[2*(r+1)+1],!(++s<l&&i===a)){if(s<u)for(;N(t,i,t.bl_tree),0!=--s;);else 0!==i?(i!==o&&(N(t,i,t.bl_tree),s--),N(t,b,t.bl_tree),z(t,s-3,2)):s<=10?(N(t,m,t.bl_tree),z(t,s-3,3)):(N(t,w,t.bl_tree),z(t,s-11,7));o=i,u=(s=0)===a?(l=138,3):i===a?(l=6,3):(l=7,4)}}r(L);var K=!1;function J(t,e,n,r){z(t,(o<<1)+(r?1:0),3),function(t,e,n,r){G(t),r&&(D(t,n),D(t,~n)),i.arraySet(t.pending_buf,t.window,e,n,t.pending),t.pending+=n}(t,e,n,!0)}e._tr_init=function(t){K||(function(){var t,e,n,r,i,o=new Array(y+1);for(r=n=0;r<a-1;r++)for(O[r]=n,t=0;t<1<<x[r];t++)E[n++]=r;for(E[n-1]=r,r=i=0;r<16;r++)for(L[r]=i,t=0;t<1<<_[r];t++)P[i++]=r;for(i>>=7;r<f;r++)for(L[r]=i<<7,t=0;t<1<<_[r]-7;t++)P[256+i++]=r;for(e=0;e<=y;e++)o[e]=0;for(t=0;t<=143;)C[2*t+1]=8,t++,o[8]++;for(;t<=255;)C[2*t+1]=9,t++,o[9]++;for(;t<=279;)C[2*t+1]=7,t++,o[7]++;for(;t<=287;)C[2*t+1]=8,t++,o[8]++;for(W(C,c+1,o),t=0;t<f;t++)A[2*t+1]=5,A[2*t]=U(t,5);T=new R(C,x,u+1,c,y),I=new R(A,_,0,f,y),B=new R(new Array(0),k,0,h,p)}(),K=!0),t.l_desc=new M(t.dyn_ltree,T),t.d_desc=new M(t.dyn_dtree,I),t.bl_desc=new M(t.bl_tree,B),t.bi_buf=0,t.bi_valid=0,j(t)},e._tr_stored_block=J,e._tr_flush_block=function(t,e,n,r){var i,o,a=0;0<t.level?(2===t.strm.data_type&&(t.strm.data_type=function(t){var e,n=4093624447;for(e=0;e<=31;e++,n>>>=1)if(1&n&&0!==t.dyn_ltree[2*e])return s;if(0!==t.dyn_ltree[18]||0!==t.dyn_ltree[20]||0!==t.dyn_ltree[26])return l;for(e=32;e<u;e++)if(0!==t.dyn_ltree[2*e])return l;return s}(t)),Z(t,t.l_desc),Z(t,t.d_desc),a=function(t){var e;for(Y(t,t.dyn_ltree,t.l_desc.max_code),Y(t,t.dyn_dtree,t.d_desc.max_code),Z(t,t.bl_desc),e=h-1;3<=e&&0===t.bl_tree[2*S[e]+1];e--);return t.opt_len+=3*(e+1)+5+5+4,e}(t),i=t.opt_len+3+7>>>3,(o=t.static_len+3+7>>>3)<=i&&(i=o)):i=o=n+5,n+4<=i&&-1!==e?J(t,e,n,r):4===t.strategy||o===i?(z(t,2+(r?1:0),3),q(t,C,A)):(z(t,4+(r?1:0),3),function(t,e,n,r){var i;for(z(t,e-257,5),z(t,n-1,5),z(t,r-4,4),i=0;i<r;i++)z(t,t.bl_tree[2*S[i]+1],3);X(t,t.dyn_ltree,e-1),X(t,t.dyn_dtree,n-1)}(t,t.l_desc.max_code+1,t.d_desc.max_code+1,a+1),q(t,t.dyn_ltree,t.dyn_dtree)),j(t),r&&G(t)},e._tr_tally=function(t,e,n){return t.pending_buf[t.d_buf+2*t.last_lit]=e>>>8&255,t.pending_buf[t.d_buf+2*t.last_lit+1]=255&e,t.pending_buf[t.l_buf+t.last_lit]=255&n,t.last_lit++,0===e?t.dyn_ltree[2*n]++:(t.matches++,e--,t.dyn_ltree[2*(E[n]+u+1)]++,t.dyn_dtree[2*F(e)]++),t.last_lit===t.lit_bufsize-1},e._tr_align=function(t){z(t,2,3),N(t,g,C),function(t){16===t.bi_valid?(D(t,t.bi_buf),t.bi_buf=0,t.bi_valid=0):8<=t.bi_valid&&(t.pending_buf[t.pending++]=255&t.bi_buf,t.bi_buf>>=8,t.bi_valid-=8)}(t)}},function(t,e,n){"use strict";t.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},function(t,e,n){"use strict";var O=n(71),T=n(156),I=n(157),B=n(271),L=n(272),R=1,M=2,F=0,D=-2,z=1,r=852,i=592;function N(t){return(t>>>24&255)+(t>>>8&65280)+((65280&t)<<8)+((255&t)<<24)}function o(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new O.Buf16(320),this.work=new O.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(t){var e;return t&&t.state?(e=t.state,t.total_in=t.total_out=e.total=0,t.msg="",e.wrap&&(t.adler=1&e.wrap),e.mode=z,e.last=0,e.havedict=0,e.dmax=32768,e.head=null,e.hold=0,e.bits=0,e.lencode=e.lendyn=new O.Buf32(r),e.distcode=e.distdyn=new O.Buf32(i),e.sane=1,e.back=-1,F):D}function s(t){var e;return t&&t.state?((e=t.state).wsize=0,e.whave=0,e.wnext=0,a(t)):D}function l(t,e){var n,r;return t&&t.state?(r=t.state,e<0?(n=0,e=-e):(n=1+(e>>4),e<48&&(e&=15)),e&&(e<8||15<e)?D:(null!==r.window&&r.wbits!==e&&(r.window=null),r.wrap=n,r.wbits=e,s(t))):D}function u(t,e){var n,r;return t?(r=new o,(t.state=r).window=null,(n=l(t,e))!==F&&(t.state=null),n):D}var c,f,h=!0;function U(t){if(h){var e;for(c=new O.Buf32(512),f=new O.Buf32(32),e=0;e<144;)t.lens[e++]=8;for(;e<256;)t.lens[e++]=9;for(;e<280;)t.lens[e++]=7;for(;e<288;)t.lens[e++]=8;for(L(R,t.lens,0,288,c,0,t.work,{bits:9}),e=0;e<32;)t.lens[e++]=5;L(M,t.lens,0,32,f,0,t.work,{bits:5}),h=!1}t.lencode=c,t.lenbits=9,t.distcode=f,t.distbits=5}function W(t,e,n,r){var i,o=t.state;return null===o.window&&(o.wsize=1<<o.wbits,o.wnext=0,o.whave=0,o.window=new O.Buf8(o.wsize)),r>=o.wsize?(O.arraySet(o.window,e,n-o.wsize,o.wsize,0),o.wnext=0,o.whave=o.wsize):(r<(i=o.wsize-o.wnext)&&(i=r),O.arraySet(o.window,e,n-r,i,o.wnext),(r-=i)?(O.arraySet(o.window,e,n-r,r,0),o.wnext=r,o.whave=o.wsize):(o.wnext+=i,o.wnext===o.wsize&&(o.wnext=0),o.whave<o.wsize&&(o.whave+=i))),0}e.inflateReset=s,e.inflateReset2=l,e.inflateResetKeep=a,e.inflateInit=function(t){return u(t,15)},e.inflateInit2=u,e.inflate=function(t,e){var n,r,i,o,a,s,l,u,c,f,h,d,p,g,v,y,b,m,w,x,_,k,S,C,A=0,P=new O.Buf8(4),E=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!t||!t.state||!t.output||!t.input&&0!==t.avail_in)return D;12===(n=t.state).mode&&(n.mode=13),a=t.next_out,i=t.output,l=t.avail_out,o=t.next_in,r=t.input,s=t.avail_in,u=n.hold,c=n.bits,f=s,h=l,k=F;t:for(;;)switch(n.mode){case z:if(0===n.wrap){n.mode=13;break}for(;c<16;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(2&n.wrap&&35615===u){P[n.check=0]=255&u,P[1]=u>>>8&255,n.check=I(n.check,P,2,0),c=u=0,n.mode=2;break}if(n.flags=0,n.head&&(n.head.done=!1),!(1&n.wrap)||(((255&u)<<8)+(u>>8))%31){t.msg="incorrect header check",n.mode=30;break}if(8!=(15&u)){t.msg="unknown compression method",n.mode=30;break}if(c-=4,_=8+(15&(u>>>=4)),0===n.wbits)n.wbits=_;else if(_>n.wbits){t.msg="invalid window size",n.mode=30;break}n.dmax=1<<_,t.adler=n.check=1,n.mode=512&u?10:12,c=u=0;break;case 2:for(;c<16;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(n.flags=u,8!=(255&n.flags)){t.msg="unknown compression method",n.mode=30;break}if(57344&n.flags){t.msg="unknown header flags set",n.mode=30;break}n.head&&(n.head.text=u>>8&1),512&n.flags&&(P[0]=255&u,P[1]=u>>>8&255,n.check=I(n.check,P,2,0)),c=u=0,n.mode=3;case 3:for(;c<32;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.head&&(n.head.time=u),512&n.flags&&(P[0]=255&u,P[1]=u>>>8&255,P[2]=u>>>16&255,P[3]=u>>>24&255,n.check=I(n.check,P,4,0)),c=u=0,n.mode=4;case 4:for(;c<16;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.head&&(n.head.xflags=255&u,n.head.os=u>>8),512&n.flags&&(P[0]=255&u,P[1]=u>>>8&255,n.check=I(n.check,P,2,0)),c=u=0,n.mode=5;case 5:if(1024&n.flags){for(;c<16;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.length=u,n.head&&(n.head.extra_len=u),512&n.flags&&(P[0]=255&u,P[1]=u>>>8&255,n.check=I(n.check,P,2,0)),c=u=0}else n.head&&(n.head.extra=null);n.mode=6;case 6:if(1024&n.flags&&(s<(d=n.length)&&(d=s),d&&(n.head&&(_=n.head.extra_len-n.length,n.head.extra||(n.head.extra=new Array(n.head.extra_len)),O.arraySet(n.head.extra,r,o,d,_)),512&n.flags&&(n.check=I(n.check,r,d,o)),s-=d,o+=d,n.length-=d),n.length))break t;n.length=0,n.mode=7;case 7:if(2048&n.flags){if(0===s)break t;for(d=0;_=r[o+d++],n.head&&_&&n.length<65536&&(n.head.name+=String.fromCharCode(_)),_&&d<s;);if(512&n.flags&&(n.check=I(n.check,r,d,o)),s-=d,o+=d,_)break t}else n.head&&(n.head.name=null);n.length=0,n.mode=8;case 8:if(4096&n.flags){if(0===s)break t;for(d=0;_=r[o+d++],n.head&&_&&n.length<65536&&(n.head.comment+=String.fromCharCode(_)),_&&d<s;);if(512&n.flags&&(n.check=I(n.check,r,d,o)),s-=d,o+=d,_)break t}else n.head&&(n.head.comment=null);n.mode=9;case 9:if(512&n.flags){for(;c<16;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(u!==(65535&n.check)){t.msg="header crc mismatch",n.mode=30;break}c=u=0}n.head&&(n.head.hcrc=n.flags>>9&1,n.head.done=!0),t.adler=n.check=0,n.mode=12;break;case 10:for(;c<32;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}t.adler=n.check=N(u),c=u=0,n.mode=11;case 11:if(0===n.havedict)return t.next_out=a,t.avail_out=l,t.next_in=o,t.avail_in=s,n.hold=u,n.bits=c,2;t.adler=n.check=1,n.mode=12;case 12:if(5===e||6===e)break t;case 13:if(n.last){u>>>=7&c,c-=7&c,n.mode=27;break}for(;c<3;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}switch(n.last=1&u,c-=1,3&(u>>>=1)){case 0:n.mode=14;break;case 1:if(U(n),n.mode=20,6!==e)break;u>>>=2,c-=2;break t;case 2:n.mode=17;break;case 3:t.msg="invalid block type",n.mode=30}u>>>=2,c-=2;break;case 14:for(u>>>=7&c,c-=7&c;c<32;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if((65535&u)!=(u>>>16^65535)){t.msg="invalid stored block lengths",n.mode=30;break}if(n.length=65535&u,c=u=0,n.mode=15,6===e)break t;case 15:n.mode=16;case 16:if(d=n.length){if(s<d&&(d=s),l<d&&(d=l),0===d)break t;O.arraySet(i,r,o,d,a),s-=d,o+=d,l-=d,a+=d,n.length-=d;break}n.mode=12;break;case 17:for(;c<14;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(n.nlen=257+(31&u),u>>>=5,c-=5,n.ndist=1+(31&u),u>>>=5,c-=5,n.ncode=4+(15&u),u>>>=4,c-=4,286<n.nlen||30<n.ndist){t.msg="too many length or distance symbols",n.mode=30;break}n.have=0,n.mode=18;case 18:for(;n.have<n.ncode;){for(;c<3;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.lens[E[n.have++]]=7&u,u>>>=3,c-=3}for(;n.have<19;)n.lens[E[n.have++]]=0;if(n.lencode=n.lendyn,n.lenbits=7,S={bits:n.lenbits},k=L(0,n.lens,0,19,n.lencode,0,n.work,S),n.lenbits=S.bits,k){t.msg="invalid code lengths set",n.mode=30;break}n.have=0,n.mode=19;case 19:for(;n.have<n.nlen+n.ndist;){for(;y=(A=n.lencode[u&(1<<n.lenbits)-1])>>>16&255,b=65535&A,!((v=A>>>24)<=c);){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(b<16)u>>>=v,c-=v,n.lens[n.have++]=b;else{if(16===b){for(C=v+2;c<C;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(u>>>=v,c-=v,0===n.have){t.msg="invalid bit length repeat",n.mode=30;break}_=n.lens[n.have-1],d=3+(3&u),u>>>=2,c-=2}else if(17===b){for(C=v+3;c<C;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}c-=v,_=0,d=3+(7&(u>>>=v)),u>>>=3,c-=3}else{for(C=v+7;c<C;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}c-=v,_=0,d=11+(127&(u>>>=v)),u>>>=7,c-=7}if(n.have+d>n.nlen+n.ndist){t.msg="invalid bit length repeat",n.mode=30;break}for(;d--;)n.lens[n.have++]=_}}if(30===n.mode)break;if(0===n.lens[256]){t.msg="invalid code -- missing end-of-block",n.mode=30;break}if(n.lenbits=9,S={bits:n.lenbits},k=L(R,n.lens,0,n.nlen,n.lencode,0,n.work,S),n.lenbits=S.bits,k){t.msg="invalid literal/lengths set",n.mode=30;break}if(n.distbits=6,n.distcode=n.distdyn,S={bits:n.distbits},k=L(M,n.lens,n.nlen,n.ndist,n.distcode,0,n.work,S),n.distbits=S.bits,k){t.msg="invalid distances set",n.mode=30;break}if(n.mode=20,6===e)break t;case 20:n.mode=21;case 21:if(6<=s&&258<=l){t.next_out=a,t.avail_out=l,t.next_in=o,t.avail_in=s,n.hold=u,n.bits=c,B(t,h),a=t.next_out,i=t.output,l=t.avail_out,o=t.next_in,r=t.input,s=t.avail_in,u=n.hold,c=n.bits,12===n.mode&&(n.back=-1);break}for(n.back=0;y=(A=n.lencode[u&(1<<n.lenbits)-1])>>>16&255,b=65535&A,!((v=A>>>24)<=c);){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(y&&0==(240&y)){for(m=v,w=y,x=b;y=(A=n.lencode[x+((u&(1<<m+w)-1)>>m)])>>>16&255,b=65535&A,!(m+(v=A>>>24)<=c);){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}u>>>=m,c-=m,n.back+=m}if(u>>>=v,c-=v,n.back+=v,n.length=b,0===y){n.mode=26;break}if(32&y){n.back=-1,n.mode=12;break}if(64&y){t.msg="invalid literal/length code",n.mode=30;break}n.extra=15&y,n.mode=22;case 22:if(n.extra){for(C=n.extra;c<C;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.length+=u&(1<<n.extra)-1,u>>>=n.extra,c-=n.extra,n.back+=n.extra}n.was=n.length,n.mode=23;case 23:for(;y=(A=n.distcode[u&(1<<n.distbits)-1])>>>16&255,b=65535&A,!((v=A>>>24)<=c);){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(0==(240&y)){for(m=v,w=y,x=b;y=(A=n.distcode[x+((u&(1<<m+w)-1)>>m)])>>>16&255,b=65535&A,!(m+(v=A>>>24)<=c);){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}u>>>=m,c-=m,n.back+=m}if(u>>>=v,c-=v,n.back+=v,64&y){t.msg="invalid distance code",n.mode=30;break}n.offset=b,n.extra=15&y,n.mode=24;case 24:if(n.extra){for(C=n.extra;c<C;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}n.offset+=u&(1<<n.extra)-1,u>>>=n.extra,c-=n.extra,n.back+=n.extra}if(n.offset>n.dmax){t.msg="invalid distance too far back",n.mode=30;break}n.mode=25;case 25:if(0===l)break t;if(d=h-l,n.offset>d){if((d=n.offset-d)>n.whave&&n.sane){t.msg="invalid distance too far back",n.mode=30;break}p=d>n.wnext?(d-=n.wnext,n.wsize-d):n.wnext-d,d>n.length&&(d=n.length),g=n.window}else g=i,p=a-n.offset,d=n.length;for(l<d&&(d=l),l-=d,n.length-=d;i[a++]=g[p++],--d;);0===n.length&&(n.mode=21);break;case 26:if(0===l)break t;i[a++]=n.length,l--,n.mode=21;break;case 27:if(n.wrap){for(;c<32;){if(0===s)break t;s--,u|=r[o++]<<c,c+=8}if(h-=l,t.total_out+=h,n.total+=h,h&&(t.adler=n.check=n.flags?I(n.check,i,h,a-h):T(n.check,i,h,a-h)),h=l,(n.flags?u:N(u))!==n.check){t.msg="incorrect data check",n.mode=30;break}c=u=0}n.mode=28;case 28:if(n.wrap&&n.flags){for(;c<32;){if(0===s)break t;s--,u+=r[o++]<<c,c+=8}if(u!==(4294967295&n.total)){t.msg="incorrect length check",n.mode=30;break}c=u=0}n.mode=29;case 29:k=1;break t;case 30:k=-3;break t;case 31:return-4;case 32:default:return D}return t.next_out=a,t.avail_out=l,t.next_in=o,t.avail_in=s,n.hold=u,n.bits=c,(n.wsize||h!==t.avail_out&&n.mode<30&&(n.mode<27||4!==e))&&W(t,t.output,t.next_out,h-t.avail_out)?(n.mode=31,-4):(f-=t.avail_in,h-=t.avail_out,t.total_in+=f,t.total_out+=h,n.total+=h,n.wrap&&h&&(t.adler=n.check=n.flags?I(n.check,i,h,t.next_out-h):T(n.check,i,h,t.next_out-h)),t.data_type=n.bits+(n.last?64:0)+(12===n.mode?128:0)+(20===n.mode||15===n.mode?256:0),(0==f&&0===h||4===e)&&k===F&&(k=-5),k)},e.inflateEnd=function(t){if(!t||!t.state)return D;var e=t.state;return e.window&&(e.window=null),t.state=null,F},e.inflateGetHeader=function(t,e){var n;return t&&t.state?0==(2&(n=t.state).wrap)?D:((n.head=e).done=!1,F):D},e.inflateSetDictionary=function(t,e){var n,r=e.length;return t&&t.state?0!==(n=t.state).wrap&&11!==n.mode?D:11===n.mode&&T(1,e,r,0)!==n.check?-3:W(t,e,r,r)?(n.mode=31,-4):(n.havedict=1,F):D},e.inflateInfo="pako inflate (from Nodeca project)"},function(t,e,n){"use strict";t.exports=function(t,e){var n,r,i,o,a,s,l,u,c,f,h,d,p,g,v,y,b,m,w,x,_,k,S,C,A;n=t.state,r=t.next_in,C=t.input,i=r+(t.avail_in-5),o=t.next_out,A=t.output,a=o-(e-t.avail_out),s=o+(t.avail_out-257),l=n.dmax,u=n.wsize,c=n.whave,f=n.wnext,h=n.window,d=n.hold,p=n.bits,g=n.lencode,v=n.distcode,y=(1<<n.lenbits)-1,b=(1<<n.distbits)-1;t:do{p<15&&(d+=C[r++]<<p,p+=8,d+=C[r++]<<p,p+=8),m=g[d&y];e:for(;;){if(d>>>=w=m>>>24,p-=w,0===(w=m>>>16&255))A[o++]=65535&m;else{if(!(16&w)){if(0==(64&w)){m=g[(65535&m)+(d&(1<<w)-1)];continue e}if(32&w){n.mode=12;break t}t.msg="invalid literal/length code",n.mode=30;break t}x=65535&m,(w&=15)&&(p<w&&(d+=C[r++]<<p,p+=8),x+=d&(1<<w)-1,d>>>=w,p-=w),p<15&&(d+=C[r++]<<p,p+=8,d+=C[r++]<<p,p+=8),m=v[d&b];n:for(;;){if(d>>>=w=m>>>24,p-=w,!(16&(w=m>>>16&255))){if(0==(64&w)){m=v[(65535&m)+(d&(1<<w)-1)];continue n}t.msg="invalid distance code",n.mode=30;break t}if(_=65535&m,p<(w&=15)&&(d+=C[r++]<<p,(p+=8)<w&&(d+=C[r++]<<p,p+=8)),l<(_+=d&(1<<w)-1)){t.msg="invalid distance too far back",n.mode=30;break t}if(d>>>=w,p-=w,(w=o-a)<_){if(c<(w=_-w)&&n.sane){t.msg="invalid distance too far back",n.mode=30;break t}if(S=h,(k=0)===f){if(k+=u-w,w<x){for(x-=w;A[o++]=h[k++],--w;);k=o-_,S=A}}else if(f<w){if(k+=u+f-w,(w-=f)<x){for(x-=w;A[o++]=h[k++],--w;);if(k=0,f<x){for(x-=w=f;A[o++]=h[k++],--w;);k=o-_,S=A}}}else if(k+=f-w,w<x){for(x-=w;A[o++]=h[k++],--w;);k=o-_,S=A}for(;2<x;)A[o++]=S[k++],A[o++]=S[k++],A[o++]=S[k++],x-=3;x&&(A[o++]=S[k++],1<x&&(A[o++]=S[k++]))}else{for(k=o-_;A[o++]=A[k++],A[o++]=A[k++],A[o++]=A[k++],2<(x-=3););x&&(A[o++]=A[k++],1<x&&(A[o++]=A[k++]))}break}}break}}while(r<i&&o<s);r-=x=p>>3,d&=(1<<(p-=x<<3))-1,t.next_in=r,t.next_out=o,t.avail_in=r<i?i-r+5:5-(r-i),t.avail_out=o<s?s-o+257:257-(o-s),n.hold=d,n.bits=p}},function(t,e,n){"use strict";var R=n(71),M=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],F=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],D=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],z=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];t.exports=function(t,e,n,r,i,o,a,s){var l,u,c,f,h,d,p,g,v,y=s.bits,b=0,m=0,w=0,x=0,_=0,k=0,S=0,C=0,A=0,P=0,E=null,O=0,T=new R.Buf16(16),I=new R.Buf16(16),B=null,L=0;for(b=0;b<=15;b++)T[b]=0;for(m=0;m<r;m++)T[e[n+m]]++;for(_=y,x=15;1<=x&&0===T[x];x--);if(x<_&&(_=x),0===x)return i[o++]=20971520,i[o++]=20971520,s.bits=1,0;for(w=1;w<x&&0===T[w];w++);for(_<w&&(_=w),b=C=1;b<=15;b++)if(C<<=1,(C-=T[b])<0)return-1;if(0<C&&(0===t||1!==x))return-1;for(I[1]=0,b=1;b<15;b++)I[b+1]=I[b]+T[b];for(m=0;m<r;m++)0!==e[n+m]&&(a[I[e[n+m]]++]=m);if(d=0===t?(E=B=a,19):1===t?(E=M,O-=257,B=F,L-=257,256):(E=D,B=z,-1),b=w,h=o,S=m=P=0,c=-1,f=(A=1<<(k=_))-1,1===t&&852<A||2===t&&592<A)return 1;for(;;){for(p=b-S,v=a[m]<d?(g=0,a[m]):a[m]>d?(g=B[L+a[m]],E[O+a[m]]):(g=96,0),l=1<<b-S,w=u=1<<k;i[h+(P>>S)+(u-=l)]=p<<24|g<<16|v|0,0!==u;);for(l=1<<b-1;P&l;)l>>=1;if(0!==l?(P&=l-1,P+=l):P=0,m++,0==--T[b]){if(b===x)break;b=e[n+a[m]]}if(_<b&&(P&f)!==c){for(0===S&&(S=_),h+=w,C=1<<(k=b-S);k+S<x&&!((C-=T[k+S])<=0);)k++,C<<=1;if(A+=1<<k,1===t&&852<A||2===t&&592<A)return 1;i[c=P&f]=_<<24|k<<16|h-o|0}}return 0!==P&&(i[h+P]=b-S<<24|64<<16|0),s.bits=_,0}},function(t,e,n){"use strict";t.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},function(t,e,n){var r;t.exports=(r=n(1),n(72),n(275),n(276),n(45),n(46),n(100),n(158),n(277),n(159),n(278),n(279),n(280),n(101),n(281),n(34),n(5),n(282),n(283),n(284),n(285),n(286),n(287),n(288),n(289),n(290),n(291),n(292),n(293),n(294),n(295),n(296),n(297),r)},function(t,e,n){var r;t.exports=(r=n(1),function(){if("function"==typeof ArrayBuffer){var t=r.lib.WordArray,i=t.init;(t.init=function(t){if(t instanceof ArrayBuffer&&(t=new Uint8Array(t)),(t instanceof Int8Array||"undefined"!=typeof Uint8ClampedArray&&t instanceof Uint8ClampedArray||t instanceof Int16Array||t instanceof Uint16Array||t instanceof Int32Array||t instanceof Uint32Array||t instanceof Float32Array||t instanceof Float64Array)&&(t=new Uint8Array(t.buffer,t.byteOffset,t.byteLength)),t instanceof Uint8Array){for(var e=t.byteLength,n=[],r=0;r<e;r++)n[r>>>2]|=t[r]<<24-r%4*8;i.call(this,n,e)}else i.apply(this,arguments)}).prototype=t}}(),r.lib.WordArray)},function(t,e,n){var r;t.exports=(r=n(1),function(){var t=r,i=t.lib.WordArray,e=t.enc;function a(t){return t<<8&4278255360|t>>>8&16711935}e.Utf16=e.Utf16BE={stringify:function(t){for(var e=t.words,n=t.sigBytes,r=[],i=0;i<n;i+=2){var o=e[i>>>2]>>>16-i%4*8&65535;r.push(String.fromCharCode(o))}return r.join("")},parse:function(t){for(var e=t.length,n=[],r=0;r<e;r++)n[r>>>1]|=t.charCodeAt(r)<<16-r%2*16;return i.create(n,2*e)}},e.Utf16LE={stringify:function(t){for(var e=t.words,n=t.sigBytes,r=[],i=0;i<n;i+=2){var o=a(e[i>>>2]>>>16-i%4*8&65535);r.push(String.fromCharCode(o))}return r.join("")},parse:function(t){for(var e=t.length,n=[],r=0;r<e;r++)n[r>>>1]|=a(t.charCodeAt(r)<<16-r%2*16);return i.create(n,2*e)}}}(),r.enc.Utf16)},function(t,e,n){var o;t.exports=(o=n(1),n(158),function(){var t=o,e=t.lib.WordArray,n=t.algo,r=n.SHA256,i=n.SHA224=r.extend({_doReset:function(){this._hash=new e.init([3238371032,914150663,812702999,4144912697,4290775857,1750603025,1694076839,3204075428])},_doFinalize:function(){var t=r._doFinalize.call(this);return t.sigBytes-=4,t}});t.SHA224=r._createHelper(i),t.HmacSHA224=r._createHmacHelper(i)}(),o.SHA224)},function(t,e,n){var s;t.exports=(s=n(1),n(72),n(159),function(){var t=s,e=t.x64,n=e.Word,r=e.WordArray,i=t.algo,o=i.SHA512,a=i.SHA384=o.extend({_doReset:function(){this._hash=new r.init([new n.init(3418070365,3238371032),new n.init(1654270250,914150663),new n.init(2438529370,812702999),new n.init(355462360,4144912697),new n.init(1731405415,4290775857),new n.init(2394180231,1750603025),new n.init(3675008525,1694076839),new n.init(1203062813,3204075428)])},_doFinalize:function(){var t=o._doFinalize.call(this);return t.sigBytes-=16,t}});t.SHA384=o._createHelper(a),t.HmacSHA384=o._createHmacHelper(a)}(),s.SHA384)},function(t,e,n){var o;t.exports=(o=n(1),n(72),function(h){var t=o,e=t.lib,d=e.WordArray,r=e.Hasher,c=t.x64.Word,n=t.algo,O=[],T=[],I=[];!function(){for(var t=1,e=0,n=0;n<24;n++){O[t+5*e]=(n+1)*(n+2)/2%64;var r=(2*t+3*e)%5;t=e%5,e=r}for(t=0;t<5;t++)for(e=0;e<5;e++)T[t+5*e]=e+(2*t+3*e)%5*5;for(var i=1,o=0;o<24;o++){for(var a=0,s=0,l=0;l<7;l++){if(1&i){var u=(1<<l)-1;u<32?s^=1<<u:a^=1<<u-32}128&i?i=i<<1^113:i<<=1}I[o]=c.create(a,s)}}();var B=[];!function(){for(var t=0;t<25;t++)B[t]=c.create()}();var i=n.SHA3=r.extend({cfg:r.cfg.extend({outputLength:512}),_doReset:function(){for(var t=this._state=[],e=0;e<25;e++)t[e]=new c.init;this.blockSize=(1600-2*this.cfg.outputLength)/32},_doProcessBlock:function(t,e){for(var n=this._state,r=this.blockSize/2,i=0;i<r;i++){var o=t[e+2*i],a=t[e+2*i+1];o=16711935&(o<<8|o>>>24)|4278255360&(o<<24|o>>>8),a=16711935&(a<<8|a>>>24)|4278255360&(a<<24|a>>>8),(S=n[i]).high^=a,S.low^=o}for(var s=0;s<24;s++){for(var l=0;l<5;l++){for(var u=0,c=0,f=0;f<5;f++)u^=(S=n[l+5*f]).high,c^=S.low;var h=B[l];h.high=u,h.low=c}for(l=0;l<5;l++){var d=B[(l+4)%5],p=B[(l+1)%5],g=p.high,v=p.low;for(u=d.high^(g<<1|v>>>31),c=d.low^(v<<1|g>>>31),f=0;f<5;f++)(S=n[l+5*f]).high^=u,S.low^=c}for(var y=1;y<25;y++){var b=(S=n[y]).high,m=S.low,w=O[y];c=w<32?(u=b<<w|m>>>32-w,m<<w|b>>>32-w):(u=m<<w-32|b>>>64-w,b<<w-32|m>>>64-w);var x=B[T[y]];x.high=u,x.low=c}var _=B[0],k=n[0];for(_.high=k.high,_.low=k.low,l=0;l<5;l++)for(f=0;f<5;f++){var S=n[y=l+5*f],C=B[y],A=B[(l+1)%5+5*f],P=B[(l+2)%5+5*f];S.high=C.high^~A.high&P.high,S.low=C.low^~A.low&P.low}S=n[0];var E=I[s];S.high^=E.high,S.low^=E.low}},_doFinalize:function(){var t=this._data,e=t.words,n=(this._nDataBytes,8*t.sigBytes),r=32*this.blockSize;e[n>>>5]|=1<<24-n%32,e[(h.ceil((1+n)/r)*r>>>5)-1]|=128,t.sigBytes=4*e.length,this._process();for(var i=this._state,o=this.cfg.outputLength/8,a=o/8,s=[],l=0;l<a;l++){var u=i[l],c=u.high,f=u.low;c=16711935&(c<<8|c>>>24)|4278255360&(c<<24|c>>>8),f=16711935&(f<<8|f>>>24)|4278255360&(f<<24|f>>>8),s.push(f),s.push(c)}return new d.init(s,o)},clone:function(){for(var t=r.clone.call(this),e=t._state=this._state.slice(0),n=0;n<25;n++)e[n]=e[n].clone();return t}});t.SHA3=r._createHelper(i),t.HmacSHA3=r._createHmacHelper(i)}(Math),o.SHA3)},function(t,e,n){var a;t.exports=(a=n(1),
/** @preserve
	(c) 2012 by Cédric Mesnil. All rights reserved.

	Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

	    - Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
	    - Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	*/
function(){var t=a,e=t.lib,n=e.WordArray,r=e.Hasher,i=t.algo,k=n.create([0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,7,4,13,1,10,6,15,3,12,0,9,5,2,14,11,8,3,10,14,4,9,15,8,1,2,7,0,6,13,11,5,12,1,9,11,10,0,8,12,4,13,3,7,15,14,5,6,2,4,0,5,9,7,12,2,10,14,1,3,8,11,6,15,13]),S=n.create([5,14,7,0,9,2,11,4,13,6,15,8,1,10,3,12,6,11,3,7,0,13,5,10,14,15,8,12,4,9,1,2,15,5,1,3,7,14,6,9,11,8,12,2,10,0,4,13,8,6,4,1,3,11,15,0,5,12,2,13,9,7,10,14,12,15,10,4,1,5,8,7,6,2,13,14,0,3,9,11]),C=n.create([11,14,15,12,5,8,7,9,11,13,14,15,6,7,9,8,7,6,8,13,11,9,7,15,7,12,15,9,11,7,13,12,11,13,6,7,14,9,13,15,14,8,13,6,5,12,7,5,11,12,14,15,14,15,9,8,9,14,5,6,8,6,5,12,9,15,5,11,6,8,13,12,5,12,13,14,11,8,5,6]),A=n.create([8,9,9,11,13,15,15,5,7,7,8,11,14,14,12,6,9,13,15,7,12,8,9,11,7,7,12,7,6,15,13,11,9,7,15,11,8,6,6,14,12,13,5,14,13,13,7,5,15,5,8,11,14,14,6,14,6,9,12,9,12,5,15,8,8,5,12,9,12,5,14,6,8,13,6,5,15,13,11,11]),P=n.create([0,1518500249,1859775393,2400959708,2840853838]),E=n.create([1352829926,1548603684,1836072691,2053994217,0]),o=i.RIPEMD160=r.extend({_doReset:function(){this._hash=n.create([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function(t,e){for(var n=0;n<16;n++){var r=e+n,i=t[r];t[r]=16711935&(i<<8|i>>>24)|4278255360&(i<<24|i>>>8)}var o,a,s,l,u,c,f,h,d,p,g,v=this._hash.words,y=P.words,b=E.words,m=k.words,w=S.words,x=C.words,_=A.words;for(c=o=v[0],f=a=v[1],h=s=v[2],d=l=v[3],p=u=v[4],n=0;n<80;n+=1)g=o+t[e+m[n]]|0,g+=n<16?O(a,s,l)+y[0]:n<32?T(a,s,l)+y[1]:n<48?I(a,s,l)+y[2]:n<64?B(a,s,l)+y[3]:L(a,s,l)+y[4],g=(g=R(g|=0,x[n]))+u|0,o=u,u=l,l=R(s,10),s=a,a=g,g=c+t[e+w[n]]|0,g+=n<16?L(f,h,d)+b[0]:n<32?B(f,h,d)+b[1]:n<48?I(f,h,d)+b[2]:n<64?T(f,h,d)+b[3]:O(f,h,d)+b[4],g=(g=R(g|=0,_[n]))+p|0,c=p,p=d,d=R(h,10),h=f,f=g;g=v[1]+s+d|0,v[1]=v[2]+l+p|0,v[2]=v[3]+u+c|0,v[3]=v[4]+o+f|0,v[4]=v[0]+a+h|0,v[0]=g},_doFinalize:function(){var t=this._data,e=t.words,n=8*this._nDataBytes,r=8*t.sigBytes;e[r>>>5]|=128<<24-r%32,e[14+(64+r>>>9<<4)]=16711935&(n<<8|n>>>24)|4278255360&(n<<24|n>>>8),t.sigBytes=4*(e.length+1),this._process();for(var i=this._hash,o=i.words,a=0;a<5;a++){var s=o[a];o[a]=16711935&(s<<8|s>>>24)|4278255360&(s<<24|s>>>8)}return i},clone:function(){var t=r.clone.call(this);return t._hash=this._hash.clone(),t}});function O(t,e,n){return t^e^n}function T(t,e,n){return t&e|~t&n}function I(t,e,n){return(t|~e)^n}function B(t,e,n){return t&n|e&~n}function L(t,e,n){return t^(e|~n)}function R(t,e){return t<<e|t>>>32-e}t.RIPEMD160=r._createHelper(o),t.HmacRIPEMD160=r._createHmacHelper(o)}(Math),a.RIPEMD160)},function(t,e,n){var a;t.exports=(a=n(1),n(100),n(101),function(){var t=a,e=t.lib,n=e.Base,y=e.WordArray,r=t.algo,i=r.SHA1,b=r.HMAC,o=r.PBKDF2=n.extend({cfg:n.extend({keySize:4,hasher:i,iterations:1}),init:function(t){this.cfg=this.cfg.extend(t)},compute:function(t,e){for(var n=this.cfg,r=b.create(n.hasher,t),i=y.create(),o=y.create([1]),a=i.words,s=o.words,l=n.keySize,u=n.iterations;a.length<l;){var c=r.update(e).finalize(o);r.reset();for(var f=c.words,h=f.length,d=c,p=1;p<u;p++){d=r.finalize(d),r.reset();for(var g=d.words,v=0;v<h;v++)f[v]^=g[v]}i.concat(c),s[0]++}return i.sigBytes=4*l,i}});t.PBKDF2=function(t,e,n){return o.create(n).compute(t,e)}}(),a.PBKDF2)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.mode.CFB=function(){var t=r.lib.BlockCipherMode.extend();function o(t,e,n,r){var i=this._iv;if(i){var o=i.slice(0);this._iv=void 0}else o=this._prevBlock;r.encryptBlock(o,0);for(var a=0;a<n;a++)t[e+a]^=o[a]}return t.Encryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize;o.call(this,t,e,r,n),this._prevBlock=t.slice(e,e+r)}}),t.Decryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize,i=t.slice(e,e+r);o.call(this,t,e,r,n),this._prevBlock=i}}),t}(),r.mode.CFB)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.mode.CTR=function(){var t=r.lib.BlockCipherMode.extend(),e=t.Encryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize,i=this._iv,o=this._counter;i&&(o=this._counter=i.slice(0),this._iv=void 0);var a=o.slice(0);n.encryptBlock(a,0),o[r-1]=o[r-1]+1|0;for(var s=0;s<r;s++)t[e+s]^=a[s]}});return t.Decryptor=e,t}(),r.mode.CTR)},function(t,e,n){var r;t.exports=(r=n(1),n(5),
/** @preserve
	 * Counter block mode compatible with  Dr Brian Gladman fileenc.c
	 * derived from CryptoJS.mode.CTR
	 * Jan Hruby jhruby.web@gmail.com
	 */
r.mode.CTRGladman=function(){var t=r.lib.BlockCipherMode.extend();function l(t){if(255==(t>>24&255)){var e=t>>16&255,n=t>>8&255,r=255&t;255===e?(e=0,255===n?(n=0,255===r?r=0:++r):++n):++e,t=0,t+=e<<16,t+=n<<8,t+=r}else t+=1<<24;return t}var e=t.Encryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize,i=this._iv,o=this._counter;i&&(o=this._counter=i.slice(0),this._iv=void 0),function(t){0===(t[0]=l(t[0]))&&(t[1]=l(t[1]))}(o);var a=o.slice(0);n.encryptBlock(a,0);for(var s=0;s<r;s++)t[e+s]^=a[s]}});return t.Decryptor=e,t}(),r.mode.CTRGladman)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.mode.OFB=function(){var t=r.lib.BlockCipherMode.extend(),e=t.Encryptor=t.extend({processBlock:function(t,e){var n=this._cipher,r=n.blockSize,i=this._iv,o=this._keystream;i&&(o=this._keystream=i.slice(0),this._iv=void 0),n.encryptBlock(o,0);for(var a=0;a<r;a++)t[e+a]^=o[a]}});return t.Decryptor=e,t}(),r.mode.OFB)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.mode.ECB=function(){var t=r.lib.BlockCipherMode.extend();return t.Encryptor=t.extend({processBlock:function(t,e){this._cipher.encryptBlock(t,e)}}),t.Decryptor=t.extend({processBlock:function(t,e){this._cipher.decryptBlock(t,e)}}),t}(),r.mode.ECB)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.pad.AnsiX923={pad:function(t,e){var n=t.sigBytes,r=4*e,i=r-n%r,o=n+i-1;t.clamp(),t.words[o>>>2]|=i<<24-o%4*8,t.sigBytes+=i},unpad:function(t){var e=255&t.words[t.sigBytes-1>>>2];t.sigBytes-=e}},r.pad.Ansix923)},function(t,e,n){var i;t.exports=(i=n(1),n(5),i.pad.Iso10126={pad:function(t,e){var n=4*e,r=n-t.sigBytes%n;t.concat(i.lib.WordArray.random(r-1)).concat(i.lib.WordArray.create([r<<24],1))},unpad:function(t){var e=255&t.words[t.sigBytes-1>>>2];t.sigBytes-=e}},i.pad.Iso10126)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.pad.Iso97971={pad:function(t,e){t.concat(r.lib.WordArray.create([2147483648],1)),r.pad.ZeroPadding.pad(t,e)},unpad:function(t){r.pad.ZeroPadding.unpad(t),t.sigBytes--}},r.pad.Iso97971)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.pad.ZeroPadding={pad:function(t,e){var n=4*e;t.clamp(),t.sigBytes+=n-(t.sigBytes%n||n)},unpad:function(t){for(var e=t.words,n=t.sigBytes-1;!(e[n>>>2]>>>24-n%4*8&255);)n--;t.sigBytes=n+1}},r.pad.ZeroPadding)},function(t,e,n){var r;t.exports=(r=n(1),n(5),r.pad.NoPadding={pad:function(){},unpad:function(){}},r.pad.NoPadding)},function(t,e,n){var i;t.exports=(i=n(1),n(5),function(){var t=i,n=t.lib.CipherParams,r=t.enc.Hex;t.format.Hex={stringify:function(t){return t.ciphertext.toString(r)},parse:function(t){var e=r.parse(t);return n.create({ciphertext:e})}}}(),i.format.Hex)},function(t,e,n){var i;t.exports=(i=n(1),n(45),n(46),n(34),n(5),function(){var t=i,e=t.lib.BlockCipher,n=t.algo,u=[],c=[],f=[],h=[],d=[],p=[],g=[],v=[],y=[],b=[];!function(){for(var t=[],e=0;e<256;e++)t[e]=e<128?e<<1:e<<1^283;var n=0,r=0;for(e=0;e<256;e++){var i=r^r<<1^r<<2^r<<3^r<<4;i=i>>>8^255&i^99,u[n]=i;var o=t[c[i]=n],a=t[o],s=t[a],l=257*t[i]^16843008*i;f[n]=l<<24|l>>>8,h[n]=l<<16|l>>>16,d[n]=l<<8|l>>>24,p[n]=l,l=16843009*s^65537*a^257*o^16843008*n,g[i]=l<<24|l>>>8,v[i]=l<<16|l>>>16,y[i]=l<<8|l>>>24,b[i]=l,n?(n=o^t[t[t[s^o]]],r^=t[t[r]]):n=r=1}}();var m=[0,1,2,4,8,16,32,64,128,27,54],r=n.AES=e.extend({_doReset:function(){if(!this._nRounds||this._keyPriorReset!==this._key){for(var t=this._keyPriorReset=this._key,e=t.words,n=t.sigBytes/4,r=4*(1+(this._nRounds=6+n)),i=this._keySchedule=[],o=0;o<r;o++)if(o<n)i[o]=e[o];else{var a=i[o-1];o%n?6<n&&o%n==4&&(a=u[a>>>24]<<24|u[a>>>16&255]<<16|u[a>>>8&255]<<8|u[255&a]):(a=u[(a=a<<8|a>>>24)>>>24]<<24|u[a>>>16&255]<<16|u[a>>>8&255]<<8|u[255&a],a^=m[o/n|0]<<24),i[o]=i[o-n]^a}for(var s=this._invKeySchedule=[],l=0;l<r;l++)o=r-l,a=l%4?i[o]:i[o-4],s[l]=l<4||o<=4?a:g[u[a>>>24]]^v[u[a>>>16&255]]^y[u[a>>>8&255]]^b[u[255&a]]}},encryptBlock:function(t,e){this._doCryptBlock(t,e,this._keySchedule,f,h,d,p,u)},decryptBlock:function(t,e){var n=t[e+1];t[e+1]=t[e+3],t[e+3]=n,this._doCryptBlock(t,e,this._invKeySchedule,g,v,y,b,c),n=t[e+1],t[e+1]=t[e+3],t[e+3]=n},_doCryptBlock:function(t,e,n,r,i,o,a,s){for(var l=this._nRounds,u=t[e]^n[0],c=t[e+1]^n[1],f=t[e+2]^n[2],h=t[e+3]^n[3],d=4,p=1;p<l;p++){var g=r[u>>>24]^i[c>>>16&255]^o[f>>>8&255]^a[255&h]^n[d++],v=r[c>>>24]^i[f>>>16&255]^o[h>>>8&255]^a[255&u]^n[d++],y=r[f>>>24]^i[h>>>16&255]^o[u>>>8&255]^a[255&c]^n[d++],b=r[h>>>24]^i[u>>>16&255]^o[c>>>8&255]^a[255&f]^n[d++];u=g,c=v,f=y,h=b}g=(s[u>>>24]<<24|s[c>>>16&255]<<16|s[f>>>8&255]<<8|s[255&h])^n[d++],v=(s[c>>>24]<<24|s[f>>>16&255]<<16|s[h>>>8&255]<<8|s[255&u])^n[d++],y=(s[f>>>24]<<24|s[h>>>16&255]<<16|s[u>>>8&255]<<8|s[255&c])^n[d++],b=(s[h>>>24]<<24|s[u>>>16&255]<<16|s[c>>>8&255]<<8|s[255&f])^n[d++],t[e]=g,t[e+1]=v,t[e+2]=y,t[e+3]=b},keySize:8});t.AES=e._createHelper(r)}(),i.AES)},function(t,e,n){var s;t.exports=(s=n(1),n(45),n(46),n(34),n(5),function(){var t=s,e=t.lib,n=e.WordArray,r=e.BlockCipher,i=t.algo,u=[57,49,41,33,25,17,9,1,58,50,42,34,26,18,10,2,59,51,43,35,27,19,11,3,60,52,44,36,63,55,47,39,31,23,15,7,62,54,46,38,30,22,14,6,61,53,45,37,29,21,13,5,28,20,12,4],c=[14,17,11,24,1,5,3,28,15,6,21,10,23,19,12,4,26,8,16,7,27,20,13,2,41,52,31,37,47,55,30,40,51,45,33,48,44,49,39,56,34,53,46,42,50,36,29,32],f=[1,2,4,6,8,10,12,14,15,17,19,21,23,25,27,28],h=[{0:8421888,268435456:32768,536870912:8421378,805306368:2,1073741824:512,1342177280:8421890,1610612736:8389122,1879048192:8388608,2147483648:514,2415919104:8389120,2684354560:33280,2952790016:8421376,3221225472:32770,3489660928:8388610,3758096384:0,4026531840:33282,134217728:0,402653184:8421890,671088640:33282,939524096:32768,1207959552:8421888,1476395008:512,1744830464:8421378,2013265920:2,2281701376:8389120,2550136832:33280,2818572288:8421376,3087007744:8389122,3355443200:8388610,3623878656:32770,3892314112:514,4160749568:8388608,1:32768,268435457:2,536870913:8421888,805306369:8388608,1073741825:8421378,1342177281:33280,1610612737:512,1879048193:8389122,2147483649:8421890,2415919105:8421376,2684354561:8388610,2952790017:33282,3221225473:514,3489660929:8389120,3758096385:32770,4026531841:0,134217729:8421890,402653185:8421376,671088641:8388608,939524097:512,1207959553:32768,1476395009:8388610,1744830465:2,2013265921:33282,2281701377:32770,2550136833:8389122,2818572289:514,3087007745:8421888,3355443201:8389120,3623878657:0,3892314113:33280,4160749569:8421378},{0:1074282512,16777216:16384,33554432:524288,50331648:1074266128,67108864:1073741840,83886080:1074282496,100663296:1073758208,117440512:16,134217728:540672,150994944:1073758224,167772160:1073741824,184549376:540688,201326592:524304,218103808:0,234881024:16400,251658240:1074266112,8388608:1073758208,25165824:540688,41943040:16,58720256:1073758224,75497472:1074282512,92274688:1073741824,109051904:524288,125829120:1074266128,142606336:524304,159383552:0,176160768:16384,192937984:1074266112,209715200:1073741840,226492416:540672,243269632:1074282496,260046848:16400,268435456:0,285212672:1074266128,301989888:1073758224,318767104:1074282496,335544320:1074266112,352321536:16,369098752:540688,385875968:16384,402653184:16400,419430400:524288,436207616:524304,452984832:1073741840,469762048:540672,486539264:1073758208,503316480:1073741824,520093696:1074282512,276824064:540688,293601280:524288,310378496:1074266112,327155712:16384,343932928:1073758208,360710144:1074282512,377487360:16,394264576:1073741824,411041792:1074282496,427819008:1073741840,444596224:1073758224,461373440:524304,478150656:0,494927872:16400,511705088:1074266128,528482304:540672},{0:260,1048576:0,2097152:67109120,3145728:65796,4194304:65540,5242880:67108868,6291456:67174660,7340032:67174400,8388608:67108864,9437184:67174656,10485760:65792,11534336:67174404,12582912:67109124,13631488:65536,14680064:4,15728640:256,524288:67174656,1572864:67174404,2621440:0,3670016:67109120,4718592:67108868,5767168:65536,6815744:65540,7864320:260,8912896:4,9961472:256,11010048:67174400,12058624:65796,13107200:65792,14155776:67109124,15204352:67174660,16252928:67108864,16777216:67174656,17825792:65540,18874368:65536,19922944:67109120,20971520:256,22020096:67174660,23068672:67108868,24117248:0,25165824:67109124,26214400:67108864,27262976:4,28311552:65792,29360128:67174400,30408704:260,31457280:65796,32505856:67174404,17301504:67108864,18350080:260,19398656:67174656,20447232:0,21495808:65540,22544384:67109120,23592960:256,24641536:67174404,25690112:65536,26738688:67174660,27787264:65796,28835840:67108868,29884416:67109124,30932992:67174400,31981568:4,33030144:65792},{0:2151682048,65536:2147487808,131072:4198464,196608:2151677952,262144:0,327680:4198400,393216:2147483712,458752:4194368,524288:2147483648,589824:4194304,655360:64,720896:2147487744,786432:2151678016,851968:4160,917504:4096,983040:2151682112,32768:2147487808,98304:64,163840:2151678016,229376:2147487744,294912:4198400,360448:2151682112,425984:0,491520:2151677952,557056:4096,622592:2151682048,688128:4194304,753664:4160,819200:2147483648,884736:4194368,950272:4198464,1015808:2147483712,1048576:4194368,1114112:4198400,1179648:2147483712,1245184:0,1310720:4160,1376256:2151678016,1441792:2151682048,1507328:2147487808,1572864:2151682112,1638400:2147483648,1703936:2151677952,1769472:4198464,1835008:2147487744,1900544:4194304,1966080:64,2031616:4096,1081344:2151677952,1146880:2151682112,1212416:0,1277952:4198400,1343488:4194368,1409024:2147483648,1474560:2147487808,1540096:64,1605632:2147483712,1671168:4096,1736704:2147487744,1802240:2151678016,1867776:4160,1933312:2151682048,1998848:4194304,2064384:4198464},{0:128,4096:17039360,8192:262144,12288:536870912,16384:537133184,20480:16777344,24576:553648256,28672:262272,32768:16777216,36864:537133056,40960:536871040,45056:553910400,49152:553910272,53248:0,57344:17039488,61440:553648128,2048:17039488,6144:553648256,10240:128,14336:17039360,18432:262144,22528:537133184,26624:553910272,30720:536870912,34816:537133056,38912:0,43008:553910400,47104:16777344,51200:536871040,55296:553648128,59392:16777216,63488:262272,65536:262144,69632:128,73728:536870912,77824:553648256,81920:16777344,86016:553910272,90112:537133184,94208:16777216,98304:553910400,102400:553648128,106496:17039360,110592:537133056,114688:262272,118784:536871040,122880:0,126976:17039488,67584:553648256,71680:16777216,75776:17039360,79872:537133184,83968:536870912,88064:17039488,92160:128,96256:553910272,100352:262272,104448:553910400,108544:0,112640:553648128,116736:16777344,120832:262144,124928:537133056,129024:536871040},{0:268435464,256:8192,512:270532608,768:270540808,1024:268443648,1280:2097152,1536:2097160,1792:268435456,2048:0,2304:268443656,2560:2105344,2816:8,3072:270532616,3328:2105352,3584:8200,3840:270540800,128:270532608,384:270540808,640:8,896:2097152,1152:2105352,1408:268435464,1664:268443648,1920:8200,2176:2097160,2432:8192,2688:268443656,2944:270532616,3200:0,3456:270540800,3712:2105344,3968:268435456,4096:268443648,4352:270532616,4608:270540808,4864:8200,5120:2097152,5376:268435456,5632:268435464,5888:2105344,6144:2105352,6400:0,6656:8,6912:270532608,7168:8192,7424:268443656,7680:270540800,7936:2097160,4224:8,4480:2105344,4736:2097152,4992:268435464,5248:268443648,5504:8200,5760:270540808,6016:270532608,6272:270540800,6528:270532616,6784:8192,7040:2105352,7296:2097160,7552:0,7808:268435456,8064:268443656},{0:1048576,16:33555457,32:1024,48:1049601,64:34604033,80:0,96:1,112:34603009,128:33555456,144:1048577,160:33554433,176:34604032,192:34603008,208:1025,224:1049600,240:33554432,8:34603009,24:0,40:33555457,56:34604032,72:1048576,88:33554433,104:33554432,120:1025,136:1049601,152:33555456,168:34603008,184:1048577,200:1024,216:34604033,232:1,248:1049600,256:33554432,272:1048576,288:33555457,304:34603009,320:1048577,336:33555456,352:34604032,368:1049601,384:1025,400:34604033,416:1049600,432:1,448:0,464:34603008,480:33554433,496:1024,264:1049600,280:33555457,296:34603009,312:1,328:33554432,344:1048576,360:1025,376:34604032,392:33554433,408:34603008,424:0,440:34604033,456:1049601,472:1024,488:33555456,504:1048577},{0:134219808,1:131072,2:134217728,3:32,4:131104,5:134350880,6:134350848,7:2048,8:134348800,9:134219776,10:133120,11:134348832,12:2080,13:0,14:134217760,15:133152,2147483648:2048,2147483649:134350880,2147483650:134219808,2147483651:134217728,2147483652:134348800,2147483653:133120,2147483654:133152,2147483655:32,2147483656:134217760,2147483657:2080,2147483658:131104,2147483659:134350848,2147483660:0,2147483661:134348832,2147483662:134219776,2147483663:131072,16:133152,17:134350848,18:32,19:2048,20:134219776,21:134217760,22:134348832,23:131072,24:0,25:131104,26:134348800,27:134219808,28:134350880,29:133120,30:2080,31:134217728,2147483664:131072,2147483665:2048,2147483666:134348832,2147483667:133152,2147483668:32,2147483669:134348800,2147483670:134217728,2147483671:134219808,2147483672:134350880,2147483673:134217760,2147483674:134219776,2147483675:0,2147483676:133120,2147483677:2080,2147483678:131104,2147483679:134350848}],d=[4160749569,528482304,33030144,2064384,129024,8064,504,2147483679],o=i.DES=r.extend({_doReset:function(){for(var t=this._key.words,e=[],n=0;n<56;n++){var r=u[n]-1;e[n]=t[r>>>5]>>>31-r%32&1}for(var i=this._subKeys=[],o=0;o<16;o++){var a=i[o]=[],s=f[o];for(n=0;n<24;n++)a[n/6|0]|=e[(c[n]-1+s)%28]<<31-n%6,a[4+(n/6|0)]|=e[28+(c[n+24]-1+s)%28]<<31-n%6;for(a[0]=a[0]<<1|a[0]>>>31,n=1;n<7;n++)a[n]=a[n]>>>4*(n-1)+3;a[7]=a[7]<<5|a[7]>>>27}var l=this._invSubKeys=[];for(n=0;n<16;n++)l[n]=i[15-n]},encryptBlock:function(t,e){this._doCryptBlock(t,e,this._subKeys)},decryptBlock:function(t,e){this._doCryptBlock(t,e,this._invSubKeys)},_doCryptBlock:function(t,e,n){this._lBlock=t[e],this._rBlock=t[e+1],p.call(this,4,252645135),p.call(this,16,65535),g.call(this,2,858993459),g.call(this,8,16711935),p.call(this,1,1431655765);for(var r=0;r<16;r++){for(var i=n[r],o=this._lBlock,a=this._rBlock,s=0,l=0;l<8;l++)s|=h[l][((a^i[l])&d[l])>>>0];this._lBlock=a,this._rBlock=o^s}var u=this._lBlock;this._lBlock=this._rBlock,this._rBlock=u,p.call(this,1,1431655765),g.call(this,8,16711935),g.call(this,2,858993459),p.call(this,16,65535),p.call(this,4,252645135),t[e]=this._lBlock,t[e+1]=this._rBlock},keySize:2,ivSize:2,blockSize:2});function p(t,e){var n=(this._lBlock>>>t^this._rBlock)&e;this._rBlock^=n,this._lBlock^=n<<t}function g(t,e){var n=(this._rBlock>>>t^this._lBlock)&e;this._lBlock^=n,this._rBlock^=n<<t}t.DES=r._createHelper(o);var a=i.TripleDES=r.extend({_doReset:function(){var t=this._key.words;this._des1=o.createEncryptor(n.create(t.slice(0,2))),this._des2=o.createEncryptor(n.create(t.slice(2,4))),this._des3=o.createEncryptor(n.create(t.slice(4,6)))},encryptBlock:function(t,e){this._des1.encryptBlock(t,e),this._des2.decryptBlock(t,e),this._des3.encryptBlock(t,e)},decryptBlock:function(t,e){this._des3.decryptBlock(t,e),this._des2.encryptBlock(t,e),this._des1.decryptBlock(t,e)},keySize:6,ivSize:2,blockSize:2});t.TripleDES=r._createHelper(a)}(),s.TripleDES)},function(t,e,n){var a;t.exports=(a=n(1),n(45),n(46),n(34),n(5),function(){var t=a,e=t.lib.StreamCipher,n=t.algo,r=n.RC4=e.extend({_doReset:function(){for(var t=this._key,e=t.words,n=t.sigBytes,r=this._S=[],i=0;i<256;i++)r[i]=i;for(var o=i=0;i<256;i++){var a=i%n,s=e[a>>>2]>>>24-a%4*8&255;o=(o+r[i]+s)%256;var l=r[i];r[i]=r[o],r[o]=l}this._i=this._j=0},_doProcessBlock:function(t,e){t[e]^=i.call(this)},keySize:8,ivSize:0});function i(){for(var t=this._S,e=this._i,n=this._j,r=0,i=0;i<4;i++){n=(n+t[e=(e+1)%256])%256;var o=t[e];t[e]=t[n],t[n]=o,r|=t[(t[e]+t[n])%256]<<24-8*i}return this._i=e,this._j=n,r}t.RC4=e._createHelper(r);var o=n.RC4Drop=r.extend({cfg:r.cfg.extend({drop:192}),_doReset:function(){r._doReset.call(this);for(var t=this.cfg.drop;0<t;t--)i.call(this)}});t.RC4Drop=e._createHelper(o)}(),a.RC4)},function(t,e,n){var o;t.exports=(o=n(1),n(45),n(46),n(34),n(5),function(){var t=o,e=t.lib.StreamCipher,n=t.algo,i=[],l=[],u=[],r=n.Rabbit=e.extend({_doReset:function(){for(var t=this._key.words,e=this.cfg.iv,n=0;n<4;n++)t[n]=16711935&(t[n]<<8|t[n]>>>24)|4278255360&(t[n]<<24|t[n]>>>8);var r=this._X=[t[0],t[3]<<16|t[2]>>>16,t[1],t[0]<<16|t[3]>>>16,t[2],t[1]<<16|t[0]>>>16,t[3],t[2]<<16|t[1]>>>16],i=this._C=[t[2]<<16|t[2]>>>16,4294901760&t[0]|65535&t[1],t[3]<<16|t[3]>>>16,4294901760&t[1]|65535&t[2],t[0]<<16|t[0]>>>16,4294901760&t[2]|65535&t[3],t[1]<<16|t[1]>>>16,4294901760&t[3]|65535&t[0]];for(n=this._b=0;n<4;n++)h.call(this);for(n=0;n<8;n++)i[n]^=r[n+4&7];if(e){var o=e.words,a=o[0],s=o[1],l=16711935&(a<<8|a>>>24)|4278255360&(a<<24|a>>>8),u=16711935&(s<<8|s>>>24)|4278255360&(s<<24|s>>>8),c=l>>>16|4294901760&u,f=u<<16|65535&l;for(i[0]^=l,i[1]^=c,i[2]^=u,i[3]^=f,i[4]^=l,i[5]^=c,i[6]^=u,i[7]^=f,n=0;n<4;n++)h.call(this)}},_doProcessBlock:function(t,e){var n=this._X;h.call(this),i[0]=n[0]^n[5]>>>16^n[3]<<16,i[1]=n[2]^n[7]>>>16^n[5]<<16,i[2]=n[4]^n[1]>>>16^n[7]<<16,i[3]=n[6]^n[3]>>>16^n[1]<<16;for(var r=0;r<4;r++)i[r]=16711935&(i[r]<<8|i[r]>>>24)|4278255360&(i[r]<<24|i[r]>>>8),t[e+r]^=i[r]},blockSize:4,ivSize:2});function h(){for(var t=this._X,e=this._C,n=0;n<8;n++)l[n]=e[n];for(e[0]=e[0]+1295307597+this._b|0,e[1]=e[1]+3545052371+(e[0]>>>0<l[0]>>>0?1:0)|0,e[2]=e[2]+886263092+(e[1]>>>0<l[1]>>>0?1:0)|0,e[3]=e[3]+1295307597+(e[2]>>>0<l[2]>>>0?1:0)|0,e[4]=e[4]+3545052371+(e[3]>>>0<l[3]>>>0?1:0)|0,e[5]=e[5]+886263092+(e[4]>>>0<l[4]>>>0?1:0)|0,e[6]=e[6]+1295307597+(e[5]>>>0<l[5]>>>0?1:0)|0,e[7]=e[7]+3545052371+(e[6]>>>0<l[6]>>>0?1:0)|0,this._b=e[7]>>>0<l[7]>>>0?1:0,n=0;n<8;n++){var r=t[n]+e[n],i=65535&r,o=r>>>16,a=((i*i>>>17)+i*o>>>15)+o*o,s=((4294901760&r)*r|0)+((65535&r)*r|0);u[n]=a^s}t[0]=u[0]+(u[7]<<16|u[7]>>>16)+(u[6]<<16|u[6]>>>16)|0,t[1]=u[1]+(u[0]<<8|u[0]>>>24)+u[7]|0,t[2]=u[2]+(u[1]<<16|u[1]>>>16)+(u[0]<<16|u[0]>>>16)|0,t[3]=u[3]+(u[2]<<8|u[2]>>>24)+u[1]|0,t[4]=u[4]+(u[3]<<16|u[3]>>>16)+(u[2]<<16|u[2]>>>16)|0,t[5]=u[5]+(u[4]<<8|u[4]>>>24)+u[3]|0,t[6]=u[6]+(u[5]<<16|u[5]>>>16)+(u[4]<<16|u[4]>>>16)|0,t[7]=u[7]+(u[6]<<8|u[6]>>>24)+u[5]|0}t.Rabbit=e._createHelper(r)}(),o.Rabbit)},function(t,e,n){var o;t.exports=(o=n(1),n(45),n(46),n(34),n(5),function(){var t=o,e=t.lib.StreamCipher,n=t.algo,i=[],l=[],u=[],r=n.RabbitLegacy=e.extend({_doReset:function(){for(var t=this._key.words,e=this.cfg.iv,n=this._X=[t[0],t[3]<<16|t[2]>>>16,t[1],t[0]<<16|t[3]>>>16,t[2],t[1]<<16|t[0]>>>16,t[3],t[2]<<16|t[1]>>>16],r=this._C=[t[2]<<16|t[2]>>>16,4294901760&t[0]|65535&t[1],t[3]<<16|t[3]>>>16,4294901760&t[1]|65535&t[2],t[0]<<16|t[0]>>>16,4294901760&t[2]|65535&t[3],t[1]<<16|t[1]>>>16,4294901760&t[3]|65535&t[0]],i=this._b=0;i<4;i++)h.call(this);for(i=0;i<8;i++)r[i]^=n[i+4&7];if(e){var o=e.words,a=o[0],s=o[1],l=16711935&(a<<8|a>>>24)|4278255360&(a<<24|a>>>8),u=16711935&(s<<8|s>>>24)|4278255360&(s<<24|s>>>8),c=l>>>16|4294901760&u,f=u<<16|65535&l;for(r[0]^=l,r[1]^=c,r[2]^=u,r[3]^=f,r[4]^=l,r[5]^=c,r[6]^=u,r[7]^=f,i=0;i<4;i++)h.call(this)}},_doProcessBlock:function(t,e){var n=this._X;h.call(this),i[0]=n[0]^n[5]>>>16^n[3]<<16,i[1]=n[2]^n[7]>>>16^n[5]<<16,i[2]=n[4]^n[1]>>>16^n[7]<<16,i[3]=n[6]^n[3]>>>16^n[1]<<16;for(var r=0;r<4;r++)i[r]=16711935&(i[r]<<8|i[r]>>>24)|4278255360&(i[r]<<24|i[r]>>>8),t[e+r]^=i[r]},blockSize:4,ivSize:2});function h(){for(var t=this._X,e=this._C,n=0;n<8;n++)l[n]=e[n];for(e[0]=e[0]+1295307597+this._b|0,e[1]=e[1]+3545052371+(e[0]>>>0<l[0]>>>0?1:0)|0,e[2]=e[2]+886263092+(e[1]>>>0<l[1]>>>0?1:0)|0,e[3]=e[3]+1295307597+(e[2]>>>0<l[2]>>>0?1:0)|0,e[4]=e[4]+3545052371+(e[3]>>>0<l[3]>>>0?1:0)|0,e[5]=e[5]+886263092+(e[4]>>>0<l[4]>>>0?1:0)|0,e[6]=e[6]+1295307597+(e[5]>>>0<l[5]>>>0?1:0)|0,e[7]=e[7]+3545052371+(e[6]>>>0<l[6]>>>0?1:0)|0,this._b=e[7]>>>0<l[7]>>>0?1:0,n=0;n<8;n++){var r=t[n]+e[n],i=65535&r,o=r>>>16,a=((i*i>>>17)+i*o>>>15)+o*o,s=((4294901760&r)*r|0)+((65535&r)*r|0);u[n]=a^s}t[0]=u[0]+(u[7]<<16|u[7]>>>16)+(u[6]<<16|u[6]>>>16)|0,t[1]=u[1]+(u[0]<<8|u[0]>>>24)+u[7]|0,t[2]=u[2]+(u[1]<<16|u[1]>>>16)+(u[0]<<16|u[0]>>>16)|0,t[3]=u[3]+(u[2]<<8|u[2]>>>24)+u[1]|0,t[4]=u[4]+(u[3]<<16|u[3]>>>16)+(u[2]<<16|u[2]>>>16)|0,t[5]=u[5]+(u[4]<<8|u[4]>>>24)+u[3]|0,t[6]=u[6]+(u[5]<<16|u[5]>>>16)+(u[4]<<16|u[4]>>>16)|0,t[7]=u[7]+(u[6]<<8|u[6]>>>24)+u[5]|0}t.RabbitLegacy=e._createHelper(r)}(),o.RabbitLegacy)},function(Eu,t,Ou){"use strict";(function(h,n){function t(t){return t&&"object"==typeof t&&"default"in t?t.default:t}var A=t(Ou(299)),e=t(Ou(328)),H=t(Ou(111)),L=t(Ou(340)),c=t(Ou(119)),f=t(Ou(351)),d=t(Ou(124)),p=t(Ou(175)),r=t(Ou(176)),i=t(Ou(356)),o=t(Ou(367)),a=t(Ou(368)),g=Ou(26),v=t(Ou(376)),_=t(Ou(379)),y=t(Ou(382)),b=t(Ou(386)),m=t(Ou(389)),w=t(Ou(394)),P=t(Ou(400)),s=t(Ou(126)),l=t(Ou(402)),x=t(Ou(411)),u=t(Ou(414)),k=t(Ou(186)),S=t(Ou(415)),C={logErrors:!1},E=[];function O(t,e,n){if(n.get){var r=n.get;n.get=function(){var t=r.call(this);return d(this,e,{value:t}),t}}else if("function"==typeof n.value){var a=n.value;return{get:function(){var o=new i;function t(){for(var t=arguments.length,e=Array(t),n=0;n<t;n++)e[n]=arguments[n];var r=0<e.length?e[0]:"value";if(o.has(r))return o.get(r);var i=a.apply(this,e);return o.set(r,i),i}return d(this,e,{value:t}),t}}}}C.registerFormat=function(t){E.push(t)},C.openSync=function(t,e){var n=Ou(52).readFileSync(t);return C.create(n,e)},C.open=function(t,r,i){"function"==typeof r&&(i=r,r=null),Ou(52).readFile(t,function(t,e){if(t)return i(t);try{var n=C.create(e,r)}catch(t){return i(t)}return i(null,n)})},C.create=function(t,e){for(var n=0;n<E.length;n++){var r=E[n];if(r.probe(t)){var i=new r(new A.DecodeStream(t));return e?i.getFont(e):i}}throw new Error("Unknown font format")},C.defaultLanguage="en",C.setDefaultLanguage=function(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:"en";C.defaultLanguage=t};var T=new A.Struct({firstCode:A.uint16,entryCount:A.uint16,idDelta:A.int16,idRangeOffset:A.uint16}),I=new A.Struct({startCharCode:A.uint32,endCharCode:A.uint32,glyphID:A.uint32}),B=new A.Struct({startUnicodeValue:A.uint24,additionalCount:A.uint8}),R=new A.Struct({unicodeValue:A.uint24,glyphID:A.uint16}),M=new A.Array(B,A.uint32),F=new A.Array(R,A.uint32),D=new A.Struct({varSelector:A.uint24,defaultUVS:new A.Pointer(A.uint32,M,{type:"parent"}),nonDefaultUVS:new A.Pointer(A.uint32,F,{type:"parent"})}),z=new A.VersionedStruct(A.uint16,{0:{length:A.uint16,language:A.uint16,codeMap:new A.LazyArray(A.uint8,256)},2:{length:A.uint16,language:A.uint16,subHeaderKeys:new A.Array(A.uint16,256),subHeaderCount:function(t){return Math.max.apply(Math,t.subHeaderKeys)},subHeaders:new A.LazyArray(T,"subHeaderCount"),glyphIndexArray:new A.LazyArray(A.uint16,"subHeaderCount")},4:{length:A.uint16,language:A.uint16,segCountX2:A.uint16,segCount:function(t){return t.segCountX2>>1},searchRange:A.uint16,entrySelector:A.uint16,rangeShift:A.uint16,endCode:new A.LazyArray(A.uint16,"segCount"),reservedPad:new A.Reserved(A.uint16),startCode:new A.LazyArray(A.uint16,"segCount"),idDelta:new A.LazyArray(A.int16,"segCount"),idRangeOffset:new A.LazyArray(A.uint16,"segCount"),glyphIndexArray:new A.LazyArray(A.uint16,function(t){return(t.length-t._currentOffset)/2})},6:{length:A.uint16,language:A.uint16,firstCode:A.uint16,entryCount:A.uint16,glyphIndices:new A.LazyArray(A.uint16,"entryCount")},8:{reserved:new A.Reserved(A.uint16),length:A.uint32,language:A.uint16,is32:new A.LazyArray(A.uint8,8192),nGroups:A.uint32,groups:new A.LazyArray(I,"nGroups")},10:{reserved:new A.Reserved(A.uint16),length:A.uint32,language:A.uint32,firstCode:A.uint32,entryCount:A.uint32,glyphIndices:new A.LazyArray(A.uint16,"numChars")},12:{reserved:new A.Reserved(A.uint16),length:A.uint32,language:A.uint32,nGroups:A.uint32,groups:new A.LazyArray(I,"nGroups")},13:{reserved:new A.Reserved(A.uint16),length:A.uint32,language:A.uint32,nGroups:A.uint32,groups:new A.LazyArray(I,"nGroups")},14:{length:A.uint32,numRecords:A.uint32,varSelectors:new A.LazyArray(D,"numRecords")}}),N=new A.Struct({platformID:A.uint16,encodingID:A.uint16,table:new A.Pointer(A.uint32,z,{type:"parent",lazy:!0})}),U=new A.Struct({version:A.uint16,numSubtables:A.uint16,tables:new A.Array(N,"numSubtables")}),W=new A.Struct({version:A.int32,revision:A.int32,checkSumAdjustment:A.uint32,magicNumber:A.uint32,flags:A.uint16,unitsPerEm:A.uint16,created:new A.Array(A.int32,2),modified:new A.Array(A.int32,2),xMin:A.int16,yMin:A.int16,xMax:A.int16,yMax:A.int16,macStyle:new A.Bitfield(A.uint16,["bold","italic","underline","outline","shadow","condensed","extended"]),lowestRecPPEM:A.uint16,fontDirectionHint:A.int16,indexToLocFormat:A.int16,glyphDataFormat:A.int16}),j=new A.Struct({version:A.int32,ascent:A.int16,descent:A.int16,lineGap:A.int16,advanceWidthMax:A.uint16,minLeftSideBearing:A.int16,minRightSideBearing:A.int16,xMaxExtent:A.int16,caretSlopeRise:A.int16,caretSlopeRun:A.int16,caretOffset:A.int16,reserved:new A.Reserved(A.int16,4),metricDataFormat:A.int16,numberOfMetrics:A.uint16}),G=new A.Struct({advance:A.uint16,bearing:A.int16}),V=new A.Struct({metrics:new A.LazyArray(G,function(t){return t.parent.hhea.numberOfMetrics}),bearings:new A.LazyArray(A.int16,function(t){return t.parent.maxp.numGlyphs-t.parent.hhea.numberOfMetrics})}),q=new A.Struct({version:A.int32,numGlyphs:A.uint16,maxPoints:A.uint16,maxContours:A.uint16,maxComponentPoints:A.uint16,maxComponentContours:A.uint16,maxZones:A.uint16,maxTwilightPoints:A.uint16,maxStorage:A.uint16,maxFunctionDefs:A.uint16,maxInstructionDefs:A.uint16,maxStackElements:A.uint16,maxSizeOfInstructions:A.uint16,maxComponentElements:A.uint16,maxComponentDepth:A.uint16});function Z(t,e,n){var r=2<arguments.length&&void 0!==n?n:0;return 1===t&&X[r]?X[r]:Y[t][e]}var Y=[["utf16be","utf16be","utf16be","utf16be","utf16be","utf16be"],["macroman","shift-jis","big5","euc-kr","iso-8859-6","iso-8859-8","macgreek","maccyrillic","symbol","Devanagari","Gurmukhi","Gujarati","Oriya","Bengali","Tamil","Telugu","Kannada","Malayalam","Sinhalese","Burmese","Khmer","macthai","Laotian","Georgian","Armenian","gb-2312-80","Tibetan","Mongolian","Geez","maccenteuro","Vietnamese","Sindhi"],["ascii"],["symbol","utf16be","shift-jis","gb18030","big5","wansung","johab",null,null,null,"utf16be"]],X={15:"maciceland",17:"macturkish",18:"maccroatian",24:"maccenteuro",25:"maccenteuro",26:"maccenteuro",27:"maccenteuro",28:"maccenteuro",30:"maciceland",37:"macromania",38:"maccenteuro",39:"maccenteuro",40:"maccenteuro",143:"macinuit",146:"macgaelic"},K=[[],{0:"en",30:"fo",60:"ks",90:"rw",1:"fr",31:"fa",61:"ku",91:"rn",2:"de",32:"ru",62:"sd",92:"ny",3:"it",33:"zh",63:"bo",93:"mg",4:"nl",34:"nl-BE",64:"ne",94:"eo",5:"sv",35:"ga",65:"sa",128:"cy",6:"es",36:"sq",66:"mr",129:"eu",7:"da",37:"ro",67:"bn",130:"ca",8:"pt",38:"cz",68:"as",131:"la",9:"no",39:"sk",69:"gu",132:"qu",10:"he",40:"si",70:"pa",133:"gn",11:"ja",41:"yi",71:"or",134:"ay",12:"ar",42:"sr",72:"ml",135:"tt",13:"fi",43:"mk",73:"kn",136:"ug",14:"el",44:"bg",74:"ta",137:"dz",15:"is",45:"uk",75:"te",138:"jv",16:"mt",46:"be",76:"si",139:"su",17:"tr",47:"uz",77:"my",140:"gl",18:"hr",48:"kk",78:"km",141:"af",19:"zh-Hant",49:"az-Cyrl",79:"lo",142:"br",20:"ur",50:"az-Arab",80:"vi",143:"iu",21:"hi",51:"hy",81:"id",144:"gd",22:"th",52:"ka",82:"tl",145:"gv",23:"ko",53:"mo",83:"ms",146:"ga",24:"lt",54:"ky",84:"ms-Arab",147:"to",25:"pl",55:"tg",85:"am",148:"el-polyton",26:"hu",56:"tk",86:"ti",149:"kl",27:"es",57:"mn-CN",87:"om",150:"az",28:"lv",58:"mn",88:"so",151:"nn",29:"se",59:"ps",89:"sw"},[],{1078:"af",16393:"en-IN",1159:"rw",1074:"tn",1052:"sq",6153:"en-IE",1089:"sw",1115:"si",1156:"gsw",8201:"en-JM",1111:"kok",1051:"sk",1118:"am",17417:"en-MY",1042:"ko",1060:"sl",5121:"ar-DZ",5129:"en-NZ",1088:"ky",11274:"es-AR",15361:"ar-BH",13321:"en-PH",1108:"lo",16394:"es-BO",3073:"ar",18441:"en-SG",1062:"lv",13322:"es-CL",2049:"ar-IQ",7177:"en-ZA",1063:"lt",9226:"es-CO",11265:"ar-JO",11273:"en-TT",2094:"dsb",5130:"es-CR",13313:"ar-KW",2057:"en-GB",1134:"lb",7178:"es-DO",12289:"ar-LB",1033:"en",1071:"mk",12298:"es-EC",4097:"ar-LY",12297:"en-ZW",2110:"ms-BN",17418:"es-SV",6145:"ary",1061:"et",1086:"ms",4106:"es-GT",8193:"ar-OM",1080:"fo",1100:"ml",18442:"es-HN",16385:"ar-QA",1124:"fil",1082:"mt",2058:"es-MX",1025:"ar-SA",1035:"fi",1153:"mi",19466:"es-NI",10241:"ar-SY",2060:"fr-BE",1146:"arn",6154:"es-PA",7169:"aeb",3084:"fr-CA",1102:"mr",15370:"es-PY",14337:"ar-AE",1036:"fr",1148:"moh",10250:"es-PE",9217:"ar-YE",5132:"fr-LU",1104:"mn",20490:"es-PR",1067:"hy",6156:"fr-MC",2128:"mn-CN",3082:"es",1101:"as",4108:"fr-CH",1121:"ne",1034:"es",2092:"az-Cyrl",1122:"fy",1044:"nb",21514:"es-US",1068:"az",1110:"gl",2068:"nn",14346:"es-UY",1133:"ba",1079:"ka",1154:"oc",8202:"es-VE",1069:"eu",3079:"de-AT",1096:"or",2077:"sv-FI",1059:"be",1031:"de",1123:"ps",1053:"sv",2117:"bn",5127:"de-LI",1045:"pl",1114:"syr",1093:"bn-IN",4103:"de-LU",1046:"pt",1064:"tg",8218:"bs-Cyrl",2055:"de-CH",2070:"pt-PT",2143:"tzm",5146:"bs",1032:"el",1094:"pa",1097:"ta",1150:"br",1135:"kl",1131:"qu-BO",1092:"tt",1026:"bg",1095:"gu",2155:"qu-EC",1098:"te",1027:"ca",1128:"ha",3179:"qu",1054:"th",3076:"zh-HK",1037:"he",1048:"ro",1105:"bo",5124:"zh-MO",1081:"hi",1047:"rm",1055:"tr",2052:"zh",1038:"hu",1049:"ru",1090:"tk",4100:"zh-SG",1039:"is",9275:"smn",1152:"ug",1028:"zh-TW",1136:"ig",4155:"smj-NO",1058:"uk",1155:"co",1057:"id",5179:"smj",1070:"hsb",1050:"hr",1117:"iu",3131:"se-FI",1056:"ur",4122:"hr-BA",2141:"iu-Latn",1083:"se",2115:"uz-Cyrl",1029:"cs",2108:"ga",2107:"se-SE",1091:"uz",1030:"da",1076:"xh",8251:"sms",1066:"vi",1164:"prs",1077:"zu",6203:"sma-NO",1106:"cy",1125:"dv",1040:"it",7227:"sms",1160:"wo",2067:"nl-BE",2064:"it-CH",1103:"sa",1157:"sah",1043:"nl",1041:"ja",7194:"sr-Cyrl-BA",1144:"ii",3081:"en-AU",1099:"kn",3098:"sr",1130:"yo",10249:"en-BZ",1087:"kk",6170:"sr-Latn-BA",4105:"en-CA",1107:"km",2074:"sr-Latn",9225:"en-029",1158:"quc",1132:"nso"}],J=new A.Struct({platformID:A.uint16,encodingID:A.uint16,languageID:A.uint16,nameID:A.uint16,length:A.uint16,string:new A.Pointer(A.uint16,new A.String("length",function(t){return Z(t.platformID,t.encodingID,t.languageID)}),{type:"parent",relativeTo:"parent.stringOffset",allowNull:!1})}),Q=new A.Struct({length:A.uint16,tag:new A.Pointer(A.uint16,new A.String("length","utf16be"),{type:"parent",relativeTo:"stringOffset"})}),NameTable=new A.VersionedStruct(A.uint16,{0:{count:A.uint16,stringOffset:A.uint16,records:new A.Array(J,"count")},1:{count:A.uint16,stringOffset:A.uint16,records:new A.Array(J,"count"),langTagCount:A.uint16,langTags:new A.Array(Q,"langTagCount")}}),$=["copyright","fontFamily","fontSubfamily","uniqueSubfamily","fullName","version","postscriptName","trademark","manufacturer","designer","description","vendorURL","designerURL","license","licenseURL",null,"preferredFamily","preferredSubfamily","compatibleFull","sampleText","postscriptCIDFontName","wwsFamilyName","wwsSubfamilyName"];NameTable.process=function(t){var e={},n=this.records,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o,s=K[a.platformID][a.languageID];null==s&&null!=this.langTags&&32768<=a.languageID&&(s=this.langTags[a.languageID-32768].tag),null==s&&(s=a.platformID+"-"+a.languageID);var l=256<=a.nameID?"fontFeatures":$[a.nameID]||a.nameID;null==e[l]&&(e[l]={});var u=e[l];256<=a.nameID&&(u=u[a.nameID]||(u[a.nameID]={})),"string"!=typeof a.string&&"string"==typeof u[s]||(u[s]=a.string)}this.records=e},NameTable.preEncode=function(){if(!Array.isArray(this.records)){this.version=0;var t=[];for(var e in this.records){var n=this.records[e];"fontFeatures"!==e&&(t.push({platformID:3,encodingID:1,languageID:1033,nameID:$.indexOf(e),length:h.byteLength(n.en,"utf16le"),string:n.en}),"postscriptName"===e&&t.push({platformID:1,encodingID:0,languageID:0,nameID:$.indexOf(e),length:n.en.length,string:n.en}))}this.records=t,this.count=t.length,this.stringOffset=NameTable.size(this,null,!1)}};var tt=new A.VersionedStruct(A.uint16,{header:{xAvgCharWidth:A.int16,usWeightClass:A.uint16,usWidthClass:A.uint16,fsType:new A.Bitfield(A.uint16,[null,"noEmbedding","viewOnly","editable",null,null,null,null,"noSubsetting","bitmapOnly"]),ySubscriptXSize:A.int16,ySubscriptYSize:A.int16,ySubscriptXOffset:A.int16,ySubscriptYOffset:A.int16,ySuperscriptXSize:A.int16,ySuperscriptYSize:A.int16,ySuperscriptXOffset:A.int16,ySuperscriptYOffset:A.int16,yStrikeoutSize:A.int16,yStrikeoutPosition:A.int16,sFamilyClass:A.int16,panose:new A.Array(A.uint8,10),ulCharRange:new A.Array(A.uint32,4),vendorID:new A.String(4),fsSelection:new A.Bitfield(A.uint16,["italic","underscore","negative","outlined","strikeout","bold","regular","useTypoMetrics","wws","oblique"]),usFirstCharIndex:A.uint16,usLastCharIndex:A.uint16},0:{},1:{typoAscender:A.int16,typoDescender:A.int16,typoLineGap:A.int16,winAscent:A.uint16,winDescent:A.uint16,codePageRange:new A.Array(A.uint32,2)},2:{typoAscender:A.int16,typoDescender:A.int16,typoLineGap:A.int16,winAscent:A.uint16,winDescent:A.uint16,codePageRange:new A.Array(A.uint32,2),xHeight:A.int16,capHeight:A.int16,defaultChar:A.uint16,breakChar:A.uint16,maxContent:A.uint16},5:{typoAscender:A.int16,typoDescender:A.int16,typoLineGap:A.int16,winAscent:A.uint16,winDescent:A.uint16,codePageRange:new A.Array(A.uint32,2),xHeight:A.int16,capHeight:A.int16,defaultChar:A.uint16,breakChar:A.uint16,maxContent:A.uint16,usLowerOpticalPointSize:A.uint16,usUpperOpticalPointSize:A.uint16}}),et=tt.versions;et[3]=et[4]=et[2];var nt=new A.VersionedStruct(A.fixed32,{header:{italicAngle:A.fixed32,underlinePosition:A.int16,underlineThickness:A.int16,isFixedPitch:A.uint32,minMemType42:A.uint32,maxMemType42:A.uint32,minMemType1:A.uint32,maxMemType1:A.uint32},1:{},2:{numberOfGlyphs:A.uint16,glyphNameIndex:new A.Array(A.uint16,"numberOfGlyphs"),names:new A.Array(new A.String(A.uint8))},2.5:{numberOfGlyphs:A.uint16,offsets:new A.Array(A.uint8,"numberOfGlyphs")},3:{},4:{map:new A.Array(A.uint32,function(t){return t.parent.maxp.numGlyphs})}}),rt=new A.Struct({controlValues:new A.Array(A.int16)}),it=new A.Struct({instructions:new A.Array(A.uint8)}),ot=new A.VersionedStruct("head.indexToLocFormat",{0:{offsets:new A.Array(A.uint16)},1:{offsets:new A.Array(A.uint32)}});ot.process=function(){if(0===this.version)for(var t=0;t<this.offsets.length;t++)this.offsets[t]<<=1},ot.preEncode=function(){if(0===this.version)for(var t=0;t<this.offsets.length;t++)this.offsets[t]>>>=1};var at=new A.Struct({controlValueProgram:new A.Array(A.uint8)}),st=new A.Array(new A.Buffer),lt=(ut.prototype.getCFFVersion=function(t){for(;t&&!t.hdrSize;)t=t.parent;return t?t.version:-1},ut.prototype.decode=function(t,e){var n=2<=this.getCFFVersion(e)?t.readUInt32BE():t.readUInt16BE();if(0===n)return[];var r=t.readUInt8(),i=void 0;if(1===r)i=A.uint8;else if(2===r)i=A.uint16;else if(3===r)i=A.uint24;else{if(4!==r)throw new Error("Bad offset size in CFFIndex: "+r+" "+t.pos);i=A.uint32}for(var o=[],a=t.pos+(n+1)*r-1,s=i.decode(t),l=0;l<n;l++){var u=i.decode(t);if(null!=this.type){var c=t.pos;t.pos=a+s,e.length=u-s,o.push(this.type.decode(t,e)),t.pos=c}else o.push({offset:a+s,length:u-s});s=u}return t.pos=a+s,o},ut.prototype.size=function(t,e){var n=2;if(0===t.length)return n;for(var r=this.type||new A.Buffer,i=1,o=0;o<t.length;o++){var a=t[o];i+=r.size(a,e)}var s=void 0;if(i<=255)s=A.uint8;else if(i<=65535)s=A.uint16;else if(i<=16777215)s=A.uint24;else{if(!(i<=4294967295))throw new Error("Bad offset in CFFIndex");s=A.uint32}return n+=1+s.size()*(t.length+1),n+=i-1,n},ut.prototype.encode=function(t,e,n){if(t.writeUInt16BE(e.length),0!==e.length){var r=this.type||new A.Buffer,i=[],o=1,a=e,s=Array.isArray(a),l=0;for(a=s?a:H(a);;){var u;if(s){if(l>=a.length)break;u=a[l++]}else{if((l=a.next()).done)break;u=l.value}var c=u,f=r.size(c,n);i.push(f),o+=f}var h=void 0;if(o<=255)h=A.uint8;else if(o<=65535)h=A.uint16;else if(o<=16777215)h=A.uint24;else{if(!(o<=4294967295))throw new Error("Bad offset in CFFIndex");h=A.uint32}t.writeUInt8(h.size()),o=1,h.encode(t,o);var d=i,p=Array.isArray(d),g=0;for(d=p?d:H(d);;){var v;if(p){if(g>=d.length)break;v=d[g++]}else{if((g=d.next()).done)break;v=g.value}o+=v,h.encode(t,o)}var y=e,b=Array.isArray(y),m=0;for(y=b?y:H(y);;){var w;if(b){if(m>=y.length)break;w=y[m++]}else{if((m=y.next()).done)break;w=m.value}var x=w;r.encode(t,x,n)}}},ut);function ut(t){p(this,ut),this.type=t}var ct=["0","1","2","3","4","5","6","7","8","9",".","E","E-",null,"-"],ft={".":10,E:11,"E-":12,"-":14},ht=(dt.decode=function(t,e){if(32<=e&&e<=246)return e-139;if(247<=e&&e<=250)return 256*(e-247)+t.readUInt8()+108;if(251<=e&&e<=254)return 256*-(e-251)-t.readUInt8()-108;if(28===e)return t.readInt16BE();if(29===e)return t.readInt32BE();if(30!==e)return null;for(var n="";;){var r=t.readUInt8(),i=r>>4;if(15==i)break;n+=ct[i];var o=15&r;if(15==o)break;n+=ct[o]}return parseFloat(n)},dt.size=function(t){if(t.forceLarge&&(t=32768),(0|t)===t)return-107<=t&&t<=107?1:108<=t&&t<=1131||-1131<=t&&t<=-108?2:-32768<=t&&t<=32767?3:5;var e=""+t;return 1+Math.ceil((e.length+1)/2)},dt.encode=function(t,e){var n=Number(e);if(e.forceLarge)return t.writeUInt8(29),t.writeInt32BE(n);if((0|n)===n)return-107<=n&&n<=107?t.writeUInt8(n+139):108<=n&&n<=1131?(n-=108,t.writeUInt8(247+(n>>8)),t.writeUInt8(255&n)):-1131<=n&&n<=-108?(n=-n-108,t.writeUInt8(251+(n>>8)),t.writeUInt8(255&n)):-32768<=n&&n<=32767?(t.writeUInt8(28),t.writeInt16BE(n)):(t.writeUInt8(29),t.writeInt32BE(n));t.writeUInt8(30);for(var r=""+n,i=0;i<r.length;i+=2){var o=r[i],a=ft[o]||+o;if(i===r.length-1)var s=15;else{var l=r[i+1];s=ft[l]||+l}t.writeUInt8(a<<4|15&s)}return 15!==s?t.writeUInt8(240):void 0},dt);function dt(){p(this,dt)}var pt=(gt.prototype.decodeOperands=function(n,r,i,t){var o=this;if(Array.isArray(n))return t.map(function(t,e){return o.decodeOperands(n[e],r,i,[t])});if(null!=n.decode)return n.decode(r,i,t);switch(n){case"number":case"offset":case"sid":return t[0];case"boolean":return!!t[0];default:return t}},gt.prototype.encodeOperands=function(n,r,i,t){var o=this;return Array.isArray(n)?t.map(function(t,e){return o.encodeOperands(n[e],r,i,t)[0]}):null!=n.encode?n.encode(r,t,i):"number"==typeof t?[t]:"boolean"==typeof t?[+t]:Array.isArray(t)?t:[t]},gt.prototype.decode=function(t,e){var n=t.pos+e.length,r={},i=[];for(var o in v(r,{parent:{value:e},_startOffset:{value:t.pos}}),this.fields){var a=this.fields[o];r[a[1]]=a[3]}for(;t.pos<n;){var s=t.readUInt8();if(s<28){12===s&&(s=s<<8|t.readUInt8());var l=this.fields[s];if(!l)throw new Error("Unknown operator "+s);var u=this.decodeOperands(l[2],t,r,i);null!=u&&(u instanceof g.PropertyDescriptor?d(r,l[1],u):r[l[1]]=u),i=[]}else i.push(ht.decode(t,s))}return r},gt.prototype.size=function(t,e,n){var r=!(2<arguments.length&&void 0!==n)||n,i={parent:e,val:t,pointerSize:0,startOffset:e.startOffset||0},o=0;for(var a in this.fields){var s=this.fields[a],l=t[s[1]];if(null!=l&&!_(l,s[3])){var u=this.encodeOperands(s[2],null,i,l),c=Array.isArray(u),f=0;for(u=c?u:H(u);;){var h;if(c){if(f>=u.length)break;h=u[f++]}else{if((f=u.next()).done)break;h=f.value}var d=h;o+=ht.size(d)}o+=(Array.isArray(s[0])?s[0]:[s[0]]).length}}return r&&(o+=i.pointerSize),o},gt.prototype.encode=function(t,e,n){var r={pointers:[],startOffset:t.pos,parent:n,val:e,pointerSize:0};r.pointerOffset=t.pos+this.size(e,r,!1);var i=this.ops,o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=s,u=e[l[1]];if(null!=u&&!_(u,l[3])){var c=this.encodeOperands(l[2],t,r,u),f=Array.isArray(c),h=0;for(c=f?c:H(c);;){var d;if(f){if(h>=c.length)break;d=c[h++]}else{if((h=c.next()).done)break;d=h.value}var p=d;ht.encode(t,p)}var g=Array.isArray(l[0])?l[0]:[l[0]],v=Array.isArray(g),y=0;for(g=v?g:H(g);;){var b;if(v){if(y>=g.length)break;b=g[y++]}else{if((y=g.next()).done)break;b=y.value}var m=b;t.writeUInt8(m)}}}for(var w=0;w<r.pointers.length;){var x=r.pointers[w++];x.type.encode(t,x.val,x.parent)}},gt);function gt(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:[];p(this,gt),this.ops=t,this.fields={};var e=t,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=Array.isArray(o[0])?o[0][0]<<8|o[0][1]:o[0];this.fields[a]=o}}var vt,yt=(vt=A.Pointer,a(bt,vt),bt.prototype.decode=function(t,e,n){return this.offsetType={decode:function(){return n[0]}},vt.prototype.decode.call(this,t,e,n)},bt.prototype.encode=function(t,e,n){if(!t)return this.offsetType={size:function(){return 0}},this.size(e,n),[new mt(0)];var r=null;return this.offsetType={encode:function(t,e){return r=e}},vt.prototype.encode.call(this,t,e,n),[new mt(r)]},bt);function bt(t){var e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:{};return p(this,bt),null==e.type&&(e.type="global"),o(this,vt.call(this,null,t,e))}var mt=(wt.prototype.valueOf=function(){return this.val},wt);function wt(t){p(this,wt),this.val=t,this.forceLarge=!0}function xt(){p(this,xt)}var _t=new pt([[6,"BlueValues","delta",null],[7,"OtherBlues","delta",null],[8,"FamilyBlues","delta",null],[9,"FamilyOtherBlues","delta",null],[[12,9],"BlueScale","number",.039625],[[12,10],"BlueShift","number",7],[[12,11],"BlueFuzz","number",1],[10,"StdHW","number",null],[11,"StdVW","number",null],[[12,12],"StemSnapH","delta",null],[[12,13],"StemSnapV","delta",null],[[12,14],"ForceBold","boolean",!1],[[12,17],"LanguageGroup","number",0],[[12,18],"ExpansionFactor","number",.06],[[12,19],"initialRandomSeed","number",0],[20,"defaultWidthX","number",0],[21,"nominalWidthX","number",0],[22,"vsindex","number",0],[23,"blend",(xt.decode=function(t,e,n){for(var r=n.pop();n.length>r;)n.pop()},xt),null],[19,"Subrs",new yt(new lt,{type:"local"}),null]]),kt=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall","001.000","001.001","001.002","001.003","Black","Bold","Book","Light","Medium","Regular","Roman","Semibold"],St=["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","","endash","dagger","daggerdbl","periodcentered","","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","","questiondown","","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","","ring","cedilla","","hungarumlaut","ogonek","caron","emdash","","","","","","","","","","","","","","","","","AE","","ordfeminine","","","","","Lslash","Oslash","OE","ordmasculine","","","","","","ae","","","","dotlessi","","","lslash","oslash","oe","germandbls"],Ct=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron"],At=new A.Struct({reserved:new A.Reserved(A.uint16),reqFeatureIndex:A.uint16,featureCount:A.uint16,featureIndexes:new A.Array(A.uint16,"featureCount")}),Pt=new A.Struct({tag:new A.String(4),langSys:new A.Pointer(A.uint16,At,{type:"parent"})}),Et=new A.Struct({defaultLangSys:new A.Pointer(A.uint16,At),count:A.uint16,langSysRecords:new A.Array(Pt,"count")}),Ot=new A.Struct({tag:new A.String(4),script:new A.Pointer(A.uint16,Et,{type:"parent"})}),Tt=new A.Array(Ot,A.uint16),It=new A.Struct({featureParams:A.uint16,lookupCount:A.uint16,lookupListIndexes:new A.Array(A.uint16,"lookupCount")}),Bt=new A.Struct({tag:new A.String(4),feature:new A.Pointer(A.uint16,It,{type:"parent"})}),Lt=new A.Array(Bt,A.uint16),Rt=new A.Struct({markAttachmentType:A.uint8,flags:new A.Bitfield(A.uint8,["rightToLeft","ignoreBaseGlyphs","ignoreLigatures","ignoreMarks","useMarkFilteringSet"])});function Mt(t){var e=new A.Struct({lookupType:A.uint16,flags:Rt,subTableCount:A.uint16,subTables:new A.Array(new A.Pointer(A.uint16,t),"subTableCount"),markFilteringSet:new A.Optional(A.uint16,function(t){return t.flags.flags.useMarkFilteringSet})});return new A.LazyArray(new A.Pointer(A.uint16,e),A.uint16)}var Ft,Dt=new A.Struct({start:A.uint16,end:A.uint16,startCoverageIndex:A.uint16}),zt=new A.VersionedStruct(A.uint16,{1:{glyphCount:A.uint16,glyphs:new A.Array(A.uint16,"glyphCount")},2:{rangeCount:A.uint16,rangeRecords:new A.Array(Dt,"rangeCount")}}),Nt=new A.Struct({start:A.uint16,end:A.uint16,class:A.uint16}),Ut=new A.VersionedStruct(A.uint16,{1:{startGlyph:A.uint16,glyphCount:A.uint16,classValueArray:new A.Array(A.uint16,"glyphCount")},2:{classRangeCount:A.uint16,classRangeRecord:new A.Array(Nt,"classRangeCount")}}),Wt=new A.Struct({a:A.uint16,b:A.uint16,deltaFormat:A.uint16}),jt=new A.Struct({sequenceIndex:A.uint16,lookupListIndex:A.uint16}),Gt=new A.Struct({glyphCount:A.uint16,lookupCount:A.uint16,input:new A.Array(A.uint16,function(t){return t.glyphCount-1}),lookupRecords:new A.Array(jt,"lookupCount")}),Ht=new A.Array(new A.Pointer(A.uint16,Gt),A.uint16),Vt=new A.Struct({glyphCount:A.uint16,lookupCount:A.uint16,classes:new A.Array(A.uint16,function(t){return t.glyphCount-1}),lookupRecords:new A.Array(jt,"lookupCount")}),qt=new A.Array(new A.Pointer(A.uint16,Vt),A.uint16),Zt=new A.VersionedStruct(A.uint16,{1:{coverage:new A.Pointer(A.uint16,zt),ruleSetCount:A.uint16,ruleSets:new A.Array(new A.Pointer(A.uint16,Ht),"ruleSetCount")},2:{coverage:new A.Pointer(A.uint16,zt),classDef:new A.Pointer(A.uint16,Ut),classSetCnt:A.uint16,classSet:new A.Array(new A.Pointer(A.uint16,qt),"classSetCnt")},3:{glyphCount:A.uint16,lookupCount:A.uint16,coverages:new A.Array(new A.Pointer(A.uint16,zt),"glyphCount"),lookupRecords:new A.Array(jt,"lookupCount")}}),Yt=new A.Struct({backtrackGlyphCount:A.uint16,backtrack:new A.Array(A.uint16,"backtrackGlyphCount"),inputGlyphCount:A.uint16,input:new A.Array(A.uint16,function(t){return t.inputGlyphCount-1}),lookaheadGlyphCount:A.uint16,lookahead:new A.Array(A.uint16,"lookaheadGlyphCount"),lookupCount:A.uint16,lookupRecords:new A.Array(jt,"lookupCount")}),Xt=new A.Array(new A.Pointer(A.uint16,Yt),A.uint16),Kt=new A.VersionedStruct(A.uint16,{1:{coverage:new A.Pointer(A.uint16,zt),chainCount:A.uint16,chainRuleSets:new A.Array(new A.Pointer(A.uint16,Xt),"chainCount")},2:{coverage:new A.Pointer(A.uint16,zt),backtrackClassDef:new A.Pointer(A.uint16,Ut),inputClassDef:new A.Pointer(A.uint16,Ut),lookaheadClassDef:new A.Pointer(A.uint16,Ut),chainCount:A.uint16,chainClassSet:new A.Array(new A.Pointer(A.uint16,Xt),"chainCount")},3:{backtrackGlyphCount:A.uint16,backtrackCoverage:new A.Array(new A.Pointer(A.uint16,zt),"backtrackGlyphCount"),inputGlyphCount:A.uint16,inputCoverage:new A.Array(new A.Pointer(A.uint16,zt),"inputGlyphCount"),lookaheadGlyphCount:A.uint16,lookaheadCoverage:new A.Array(new A.Pointer(A.uint16,zt),"lookaheadGlyphCount"),lookupCount:A.uint16,lookupRecords:new A.Array(jt,"lookupCount")}}),Jt=new A.Fixed(16,"BE",14),Qt=new A.Struct({startCoord:Jt,peakCoord:Jt,endCoord:Jt}),$t=new A.Struct({axisCount:A.uint16,regionCount:A.uint16,variationRegions:new A.Array(new A.Array(Qt,"axisCount"),"regionCount")}),te=new A.Struct({shortDeltas:new A.Array(A.int16,function(t){return t.parent.shortDeltaCount}),regionDeltas:new A.Array(A.int8,function(t){return t.parent.regionIndexCount-t.parent.shortDeltaCount}),deltas:function(t){return t.shortDeltas.concat(t.regionDeltas)}}),ee=new A.Struct({itemCount:A.uint16,shortDeltaCount:A.uint16,regionIndexCount:A.uint16,regionIndexes:new A.Array(A.uint16,"regionIndexCount"),deltaSets:new A.Array(te,"itemCount")}),ne=new A.Struct({format:A.uint16,variationRegionList:new A.Pointer(A.uint32,$t),variationDataCount:A.uint16,itemVariationData:new A.Array(new A.Pointer(A.uint32,ee),"variationDataCount")}),re=new A.VersionedStruct(A.uint16,{1:(Ft={axisIndex:A.uint16},Ft.axisIndex=A.uint16,Ft.filterRangeMinValue=Jt,Ft.filterRangeMaxValue=Jt,Ft)}),ie=new A.Struct({conditionCount:A.uint16,conditionTable:new A.Array(new A.Pointer(A.uint32,re),"conditionCount")}),oe=new A.Struct({featureIndex:A.uint16,alternateFeatureTable:new A.Pointer(A.uint32,It,{type:"parent"})}),ae=new A.Struct({version:A.fixed32,substitutionCount:A.uint16,substitutions:new A.Array(oe,"substitutionCount")}),se=new A.Struct({conditionSet:new A.Pointer(A.uint32,ie,{type:"parent"}),featureTableSubstitution:new A.Pointer(A.uint32,ae,{type:"parent"})}),le=new A.Struct({majorVersion:A.uint16,minorVersion:A.uint16,featureVariationRecordCount:A.uint32,featureVariationRecords:new A.Array(se,"featureVariationRecordCount")}),ue=(ce.prototype.decode=function(t,e,n){return this.predefinedOps[n[0]]?this.predefinedOps[n[0]]:this.type.decode(t,e,n)},ce.prototype.size=function(t,e){return this.type.size(t,e)},ce.prototype.encode=function(t,e,n){var r=this.predefinedOps.indexOf(e);return-1!==r?r:this.type.encode(t,e,n)},ce);function ce(t,e){p(this,ce),this.predefinedOps=t,this.type=e}var fe,he=(fe=A.Number,a(de,fe),de.prototype.decode=function(t){return 127&A.uint8.decode(t)},de);function de(){return p(this,de),o(this,fe.call(this,"UInt8"))}var pe,ge=new A.Struct({first:A.uint16,nLeft:A.uint8}),ve=new A.Struct({first:A.uint16,nLeft:A.uint16}),ye=new ue([St,["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclamsmall","Hungarumlautsmall","","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","","asuperior","bsuperior","centsuperior","dsuperior","esuperior","","","isuperior","","","lsuperior","msuperior","nsuperior","osuperior","","","rsuperior","ssuperior","tsuperior","","ff","fi","fl","ffi","ffl","parenleftinferior","","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdownsmall","centoldstyle","Lslashsmall","","","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","","Dotaccentsmall","","","Macronsmall","","","figuredash","hypheninferior","","","Ogoneksmall","Ringsmall","Cedillasmall","","","","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","","","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"]],new yt(new A.VersionedStruct(new he,{0:{nCodes:A.uint8,codes:new A.Array(A.uint8,"nCodes")},1:{nRanges:A.uint8,ranges:new A.Array(ge,"nRanges")}}),{lazy:!0})),be=(pe=A.Array,a(me,pe),me.prototype.decode=function(t,e){for(var n=g.resolveLength(this.length,t,e),r=0,i=[];r<n;){var o=this.type.decode(t,e);o.offset=r,r+=o.nLeft+1,i.push(o)}return i},me);function me(){return p(this,me),o(this,pe.apply(this,arguments))}var we=new ue([Ct,[".notdef","space","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"],[".notdef","space","dollaroldstyle","dollarsuperior","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","hyphensuperior","colonmonetary","onefitted","rupiah","centoldstyle","figuredash","hypheninferior","onequarter","onehalf","threequarters","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior"]],new yt(new A.VersionedStruct(A.uint8,{0:{glyphs:new A.Array(A.uint16,function(t){return t.parent.CharStrings.length-1})},1:{ranges:new be(ge,function(t){return t.parent.CharStrings.length-1})},2:{ranges:new be(ve,function(t){return t.parent.CharStrings.length-1})}}),{lazy:!0})),xe=new A.Struct({first:A.uint16,fd:A.uint8}),_e=new A.Struct({first:A.uint32,fd:A.uint16}),ke=new A.VersionedStruct(A.uint8,{0:{fds:new A.Array(A.uint8,function(t){return t.parent.CharStrings.length})},3:{nRanges:A.uint16,ranges:new A.Array(xe,"nRanges"),sentinel:A.uint16},4:{nRanges:A.uint32,ranges:new A.Array(_e,"nRanges"),sentinel:A.uint32}}),Se=new yt(_t),Ce=(Ae.prototype.decode=function(t,e,n){return e.length=n[0],Se.decode(t,e,[n[1]])},Ae.prototype.size=function(t,e){return[_t.size(t,e,!1),Se.size(t,e)[0]]},Ae.prototype.encode=function(t,e,n){return[_t.size(e,n,!1),Se.encode(t,e,n)[0]]},Ae);function Ae(){p(this,Ae)}var Pe=new pt([[18,"Private",new Ce,null],[[12,38],"FontName","sid",null]]),Ee=new pt([[[12,30],"ROS",["sid","sid","number"],null],[0,"version","sid",null],[1,"Notice","sid",null],[[12,0],"Copyright","sid",null],[2,"FullName","sid",null],[3,"FamilyName","sid",null],[4,"Weight","sid",null],[[12,1],"isFixedPitch","boolean",!1],[[12,2],"ItalicAngle","number",0],[[12,3],"UnderlinePosition","number",-100],[[12,4],"UnderlineThickness","number",50],[[12,5],"PaintType","number",0],[[12,6],"CharstringType","number",2],[[12,7],"FontMatrix","array",[.001,0,0,.001,0,0]],[13,"UniqueID","number",null],[5,"FontBBox","array",[0,0,0,0]],[[12,8],"StrokeWidth","number",0],[14,"XUID","array",null],[15,"charset",we,Ct],[16,"Encoding",ye,St],[17,"CharStrings",new yt(new lt),null],[18,"Private",new Ce,null],[[12,20],"SyntheticBase","number",null],[[12,21],"PostScript","sid",null],[[12,22],"BaseFontName","sid",null],[[12,23],"BaseFontBlend","delta",null],[[12,31],"CIDFontVersion","number",0],[[12,32],"CIDFontRevision","number",0],[[12,33],"CIDFontType","number",0],[[12,34],"CIDCount","number",8720],[[12,35],"UIDBase","number",null],[[12,37],"FDSelect",new yt(ke),null],[[12,36],"FDArray",new yt(new lt(Pe)),null],[[12,38],"FontName","sid",null]]),Oe=new A.Struct({length:A.uint16,itemVariationStore:ne}),Te=new pt([[[12,7],"FontMatrix","array",[.001,0,0,.001,0,0]],[17,"CharStrings",new yt(new lt),null],[[12,37],"FDSelect",new yt(ke),null],[[12,36],"FDArray",new yt(new lt(Pe)),null],[24,"vstore",new yt(Oe),null],[25,"maxstack","number",193]]),Ie=new A.VersionedStruct(A.fixed16,{1:{hdrSize:A.uint8,offSize:A.uint8,nameIndex:new lt(new A.String("length")),topDictIndex:new lt(Ee),stringIndex:new lt(new A.String("length")),globalSubrIndex:new lt},2:{hdrSize:A.uint8,length:A.uint16,topDict:Te,globalSubrIndex:new lt}}),Be=(Le.decode=function(t){return new Le(t)},Le.prototype.decode=function(){this.stream.pos;var t=Ie.decode(this.stream);for(var e in t){var n=t[e];this[e]=n}if(this.version<2){if(1!==this.topDictIndex.length)throw new Error("Only a single font is allowed in CFF");this.topDict=this.topDictIndex[0]}return this.isCIDFont=null!=this.topDict.ROS,this},Le.prototype.string=function(t){return 2<=this.version?null:t<kt.length?kt[t]:this.stringIndex[t-kt.length]},Le.prototype.getCharString=function(t){return this.stream.pos=this.topDict.CharStrings[t].offset,this.stream.readBuffer(this.topDict.CharStrings[t].length)},Le.prototype.getGlyphName=function(t){if(2<=this.version)return null;if(this.isCIDFont)return null;var e=this.topDict.charset;if(Array.isArray(e))return e[t];if(0===t)return".notdef";switch(t-=1,e.version){case 0:return this.string(e.glyphs[t]);case 1:case 2:for(var n=0;n<e.ranges.length;n++){var r=e.ranges[n];if(r.offset<=t&&t<=r.offset+r.nLeft)return this.string(r.first+(t-r.offset))}}return null},Le.prototype.fdForGlyph=function(t){if(!this.topDict.FDSelect)return null;switch(this.topDict.FDSelect.version){case 0:return this.topDict.FDSelect.fds[t];case 3:case 4:for(var e=this.topDict.FDSelect.ranges,n=0,r=e.length-1;n<=r;){var i=n+r>>1;if(t<e[i].first)r=i-1;else{if(!(i<r&&t>e[1+i].first))return e[i].fd;n=1+i}}default:throw new Error("Unknown FDSelect version: "+this.topDict.FDSelect.version)}},Le.prototype.privateDictForGlyph=function(t){if(this.topDict.FDSelect){var e=this.fdForGlyph(t);return this.topDict.FDArray[e]?this.topDict.FDArray[e].Private:null}return this.version<2?this.topDict.Private:this.topDict.FDArray[0].Private},r(Le,[{key:"postscriptName",get:function(){return this.version<2?this.nameIndex[0]:null}},{key:"fullName",get:function(){return this.string(this.topDict.FullName)}},{key:"familyName",get:function(){return this.string(this.topDict.FamilyName)}}]),Le);function Le(t){p(this,Le),this.stream=t,this.decode()}function Re(){p(this,Re)}function Me(){p(this,Me)}var Fe=new A.Struct({glyphIndex:A.uint16,vertOriginY:A.int16}),De=new A.Struct({majorVersion:A.uint16,minorVersion:A.uint16,defaultVertOriginY:A.int16,numVertOriginYMetrics:A.uint16,metrics:new A.Array(Fe,"numVertOriginYMetrics")}),ze=new A.Struct({height:A.uint8,width:A.uint8,horiBearingX:A.int8,horiBearingY:A.int8,horiAdvance:A.uint8,vertBearingX:A.int8,vertBearingY:A.int8,vertAdvance:A.uint8}),Ne=new A.Struct({height:A.uint8,width:A.uint8,bearingX:A.int8,bearingY:A.int8,advance:A.uint8}),Ue=new A.Struct({glyph:A.uint16,xOffset:A.int8,yOffset:A.int8}),We=(new A.VersionedStruct("version",{1:{metrics:Ne,data:Re},2:{metrics:Ne,data:Me},5:{data:Me},6:{metrics:ze,data:Re},7:{metrics:ze,data:Me},8:{metrics:Ne,pad:new A.Reserved(A.uint8),numComponents:A.uint16,components:new A.Array(Ue,"numComponents")},9:{metrics:ze,pad:new A.Reserved(A.uint8),numComponents:A.uint16,components:new A.Array(Ue,"numComponents")},17:{metrics:Ne,dataLen:A.uint32,data:new A.Buffer("dataLen")},18:{metrics:ze,dataLen:A.uint32,data:new A.Buffer("dataLen")},19:{dataLen:A.uint32,data:new A.Buffer("dataLen")}}),new A.Struct({ascender:A.int8,descender:A.int8,widthMax:A.uint8,caretSlopeNumerator:A.int8,caretSlopeDenominator:A.int8,caretOffset:A.int8,minOriginSB:A.int8,minAdvanceSB:A.int8,maxBeforeBL:A.int8,minAfterBL:A.int8,pad:new A.Reserved(A.int8,2)})),je=new A.Struct({glyphCode:A.uint16,offset:A.uint16}),Ge=new A.VersionedStruct(A.uint16,{header:{imageFormat:A.uint16,imageDataOffset:A.uint32},1:{offsetArray:new A.Array(A.uint32,function(t){return t.parent.lastGlyphIndex-t.parent.firstGlyphIndex+1})},2:{imageSize:A.uint32,bigMetrics:ze},3:{offsetArray:new A.Array(A.uint16,function(t){return t.parent.lastGlyphIndex-t.parent.firstGlyphIndex+1})},4:{numGlyphs:A.uint32,glyphArray:new A.Array(je,function(t){return t.numGlyphs+1})},5:{imageSize:A.uint32,bigMetrics:ze,numGlyphs:A.uint32,glyphCodeArray:new A.Array(A.uint16,"numGlyphs")}}),He=new A.Struct({firstGlyphIndex:A.uint16,lastGlyphIndex:A.uint16,subtable:new A.Pointer(A.uint32,Ge)}),Ve=new A.Struct({indexSubTableArray:new A.Pointer(A.uint32,new A.Array(He,1),{type:"parent"}),indexTablesSize:A.uint32,numberOfIndexSubTables:A.uint32,colorRef:A.uint32,hori:We,vert:We,startGlyphIndex:A.uint16,endGlyphIndex:A.uint16,ppemX:A.uint8,ppemY:A.uint8,bitDepth:A.uint8,flags:new A.Bitfield(A.uint8,["horizontal","vertical"])}),qe=new A.Struct({version:A.uint32,numSizes:A.uint32,sizes:new A.Array(Ve,"numSizes")}),Ze=new A.Struct({ppem:A.uint16,resolution:A.uint16,imageOffsets:new A.Array(new A.Pointer(A.uint32,"void"),function(t){return t.parent.parent.maxp.numGlyphs+1})}),Ye=new A.Struct({version:A.uint16,flags:new A.Bitfield(A.uint16,["renderOutlines"]),numImgTables:A.uint32,imageTables:new A.Array(new A.Pointer(A.uint32,Ze),"numImgTables")}),Xe=new A.Struct({gid:A.uint16,paletteIndex:A.uint16}),Ke=new A.Struct({gid:A.uint16,firstLayerIndex:A.uint16,numLayers:A.uint16}),Je=new A.Struct({version:A.uint16,numBaseGlyphRecords:A.uint16,baseGlyphRecord:new A.Pointer(A.uint32,new A.Array(Ke,"numBaseGlyphRecords")),layerRecords:new A.Pointer(A.uint32,new A.Array(Xe,"numLayerRecords"),{lazy:!0}),numLayerRecords:A.uint16}),Qe=new A.Struct({blue:A.uint8,green:A.uint8,red:A.uint8,alpha:A.uint8}),$e=new A.VersionedStruct(A.uint16,{header:{numPaletteEntries:A.uint16,numPalettes:A.uint16,numColorRecords:A.uint16,colorRecords:new A.Pointer(A.uint32,new A.Array(Qe,"numColorRecords")),colorRecordIndices:new A.Array(A.uint16,"numPalettes")},0:{},1:{offsetPaletteTypeArray:new A.Pointer(A.uint32,new A.Array(A.uint32,"numPalettes")),offsetPaletteLabelArray:new A.Pointer(A.uint32,new A.Array(A.uint16,"numPalettes")),offsetPaletteEntryLabelArray:new A.Pointer(A.uint32,new A.Array(A.uint16,"numPaletteEntries"))}}),tn=new A.VersionedStruct(A.uint16,{1:{coordinate:A.int16},2:{coordinate:A.int16,referenceGlyph:A.uint16,baseCoordPoint:A.uint16},3:{coordinate:A.int16,deviceTable:new A.Pointer(A.uint16,Wt)}}),en=new A.Struct({defaultIndex:A.uint16,baseCoordCount:A.uint16,baseCoords:new A.Array(new A.Pointer(A.uint16,tn),"baseCoordCount")}),nn=new A.Struct({tag:new A.String(4),minCoord:new A.Pointer(A.uint16,tn,{type:"parent"}),maxCoord:new A.Pointer(A.uint16,tn,{type:"parent"})}),rn=new A.Struct({minCoord:new A.Pointer(A.uint16,tn),maxCoord:new A.Pointer(A.uint16,tn),featMinMaxCount:A.uint16,featMinMaxRecords:new A.Array(nn,"featMinMaxCount")}),on=new A.Struct({tag:new A.String(4),minMax:new A.Pointer(A.uint16,rn,{type:"parent"})}),an=new A.Struct({baseValues:new A.Pointer(A.uint16,en),defaultMinMax:new A.Pointer(A.uint16,rn),baseLangSysCount:A.uint16,baseLangSysRecords:new A.Array(on,"baseLangSysCount")}),sn=new A.Struct({tag:new A.String(4),script:new A.Pointer(A.uint16,an,{type:"parent"})}),ln=new A.Array(sn,A.uint16),un=new A.Array(new A.String(4),A.uint16),cn=new A.Struct({baseTagList:new A.Pointer(A.uint16,un),baseScriptList:new A.Pointer(A.uint16,ln)}),fn=new A.VersionedStruct(A.uint32,{header:{horizAxis:new A.Pointer(A.uint16,cn),vertAxis:new A.Pointer(A.uint16,cn)},65536:{},65537:{itemVariationStore:new A.Pointer(A.uint32,ne)}}),hn=new A.Array(A.uint16,A.uint16),dn=new A.Struct({coverage:new A.Pointer(A.uint16,zt),glyphCount:A.uint16,attachPoints:new A.Array(new A.Pointer(A.uint16,hn),"glyphCount")}),pn=new A.VersionedStruct(A.uint16,{1:{coordinate:A.int16},2:{caretValuePoint:A.uint16},3:{coordinate:A.int16,deviceTable:new A.Pointer(A.uint16,Wt)}}),gn=new A.Array(new A.Pointer(A.uint16,pn),A.uint16),vn=new A.Struct({coverage:new A.Pointer(A.uint16,zt),ligGlyphCount:A.uint16,ligGlyphs:new A.Array(new A.Pointer(A.uint16,gn),"ligGlyphCount")}),yn=new A.Struct({markSetTableFormat:A.uint16,markSetCount:A.uint16,coverage:new A.Array(new A.Pointer(A.uint32,zt),"markSetCount")}),bn=new A.VersionedStruct(A.uint32,{header:{glyphClassDef:new A.Pointer(A.uint16,Ut),attachList:new A.Pointer(A.uint16,dn),ligCaretList:new A.Pointer(A.uint16,vn),markAttachClassDef:new A.Pointer(A.uint16,Ut)},65536:{},65538:{markGlyphSetsDef:new A.Pointer(A.uint16,yn)},65539:{markGlyphSetsDef:new A.Pointer(A.uint16,yn),itemVariationStore:new A.Pointer(A.uint32,ne)}}),mn=new A.Bitfield(A.uint16,["xPlacement","yPlacement","xAdvance","yAdvance","xPlaDevice","yPlaDevice","xAdvDevice","yAdvDevice"]),wn={xPlacement:A.int16,yPlacement:A.int16,xAdvance:A.int16,yAdvance:A.int16,xPlaDevice:new A.Pointer(A.uint16,Wt,{type:"global",relativeTo:"rel"}),yPlaDevice:new A.Pointer(A.uint16,Wt,{type:"global",relativeTo:"rel"}),xAdvDevice:new A.Pointer(A.uint16,Wt,{type:"global",relativeTo:"rel"}),yAdvDevice:new A.Pointer(A.uint16,Wt,{type:"global",relativeTo:"rel"})},xn=(_n.prototype.buildStruct=function(t){for(var e=t;!e[this.key]&&e.parent;)e=e.parent;if(e[this.key]){var n={rel:function(){return e._startOffset}},r=e[this.key];for(var i in r)r[i]&&(n[i]=wn[i]);return new A.Struct(n)}},_n.prototype.size=function(t,e){return this.buildStruct(e).size(t,e)},_n.prototype.decode=function(t,e){var n=this.buildStruct(e).decode(t,e);return delete n.rel,n},_n);function _n(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:"valueFormat";p(this,_n),this.key=t}var kn=new A.Struct({secondGlyph:A.uint16,value1:new xn("valueFormat1"),value2:new xn("valueFormat2")}),Sn=new A.Array(kn,A.uint16),Cn=new A.Struct({value1:new xn("valueFormat1"),value2:new xn("valueFormat2")}),An=new A.VersionedStruct(A.uint16,{1:{xCoordinate:A.int16,yCoordinate:A.int16},2:{xCoordinate:A.int16,yCoordinate:A.int16,anchorPoint:A.uint16},3:{xCoordinate:A.int16,yCoordinate:A.int16,xDeviceTable:new A.Pointer(A.uint16,Wt),yDeviceTable:new A.Pointer(A.uint16,Wt)}}),Pn=new A.Struct({entryAnchor:new A.Pointer(A.uint16,An,{type:"parent"}),exitAnchor:new A.Pointer(A.uint16,An,{type:"parent"})}),En=new A.Struct({class:A.uint16,markAnchor:new A.Pointer(A.uint16,An,{type:"parent"})}),On=new A.Array(En,A.uint16),Tn=new A.Array(new A.Pointer(A.uint16,An),function(t){return t.parent.classCount}),In=new A.Array(Tn,A.uint16),Bn=new A.Array(new A.Pointer(A.uint16,An),function(t){return t.parent.parent.classCount}),Ln=new A.Array(Bn,A.uint16),Rn=new A.Array(new A.Pointer(A.uint16,Ln),A.uint16),Mn=new A.VersionedStruct("lookupType",{1:new A.VersionedStruct(A.uint16,{1:{coverage:new A.Pointer(A.uint16,zt),valueFormat:mn,value:new xn},2:{coverage:new A.Pointer(A.uint16,zt),valueFormat:mn,valueCount:A.uint16,values:new A.LazyArray(new xn,"valueCount")}}),2:new A.VersionedStruct(A.uint16,{1:{coverage:new A.Pointer(A.uint16,zt),valueFormat1:mn,valueFormat2:mn,pairSetCount:A.uint16,pairSets:new A.LazyArray(new A.Pointer(A.uint16,Sn),"pairSetCount")},2:{coverage:new A.Pointer(A.uint16,zt),valueFormat1:mn,valueFormat2:mn,classDef1:new A.Pointer(A.uint16,Ut),classDef2:new A.Pointer(A.uint16,Ut),class1Count:A.uint16,class2Count:A.uint16,classRecords:new A.LazyArray(new A.LazyArray(Cn,"class2Count"),"class1Count")}}),3:{format:A.uint16,coverage:new A.Pointer(A.uint16,zt),entryExitCount:A.uint16,entryExitRecords:new A.Array(Pn,"entryExitCount")},4:{format:A.uint16,markCoverage:new A.Pointer(A.uint16,zt),baseCoverage:new A.Pointer(A.uint16,zt),classCount:A.uint16,markArray:new A.Pointer(A.uint16,On),baseArray:new A.Pointer(A.uint16,In)},5:{format:A.uint16,markCoverage:new A.Pointer(A.uint16,zt),ligatureCoverage:new A.Pointer(A.uint16,zt),classCount:A.uint16,markArray:new A.Pointer(A.uint16,On),ligatureArray:new A.Pointer(A.uint16,Rn)},6:{format:A.uint16,mark1Coverage:new A.Pointer(A.uint16,zt),mark2Coverage:new A.Pointer(A.uint16,zt),classCount:A.uint16,mark1Array:new A.Pointer(A.uint16,On),mark2Array:new A.Pointer(A.uint16,In)},7:Zt,8:Kt,9:{posFormat:A.uint16,lookupType:A.uint16,extension:new A.Pointer(A.uint32,Mn)}});Mn.versions[9].extension.type=Mn;var Fn=new A.VersionedStruct(A.uint32,{header:{scriptList:new A.Pointer(A.uint16,Tt),featureList:new A.Pointer(A.uint16,Lt),lookupList:new A.Pointer(A.uint16,new Mt(Mn))},65536:{},65537:{featureVariations:new A.Pointer(A.uint32,le)}}),Dn=new A.Array(A.uint16,A.uint16),zn=Dn,Nn=new A.Struct({glyph:A.uint16,compCount:A.uint16,components:new A.Array(A.uint16,function(t){return t.compCount-1})}),Un=new A.Array(new A.Pointer(A.uint16,Nn),A.uint16),Wn=new A.VersionedStruct("lookupType",{1:new A.VersionedStruct(A.uint16,{1:{coverage:new A.Pointer(A.uint16,zt),deltaGlyphID:A.int16},2:{coverage:new A.Pointer(A.uint16,zt),glyphCount:A.uint16,substitute:new A.LazyArray(A.uint16,"glyphCount")}}),2:{substFormat:A.uint16,coverage:new A.Pointer(A.uint16,zt),count:A.uint16,sequences:new A.LazyArray(new A.Pointer(A.uint16,Dn),"count")},3:{substFormat:A.uint16,coverage:new A.Pointer(A.uint16,zt),count:A.uint16,alternateSet:new A.LazyArray(new A.Pointer(A.uint16,zn),"count")},4:{substFormat:A.uint16,coverage:new A.Pointer(A.uint16,zt),count:A.uint16,ligatureSets:new A.LazyArray(new A.Pointer(A.uint16,Un),"count")},5:Zt,6:Kt,7:{substFormat:A.uint16,lookupType:A.uint16,extension:new A.Pointer(A.uint32,Wn)},8:{substFormat:A.uint16,coverage:new A.Pointer(A.uint16,zt),backtrackCoverage:new A.Array(new A.Pointer(A.uint16,zt),"backtrackGlyphCount"),lookaheadGlyphCount:A.uint16,lookaheadCoverage:new A.Array(new A.Pointer(A.uint16,zt),"lookaheadGlyphCount"),glyphCount:A.uint16,substitutes:new A.Array(A.uint16,"glyphCount")}});Wn.versions[7].extension.type=Wn;var jn=new A.VersionedStruct(A.uint32,{header:{scriptList:new A.Pointer(A.uint16,Tt),featureList:new A.Pointer(A.uint16,Lt),lookupList:new A.Pointer(A.uint16,new Mt(Wn))},65536:{},65537:{featureVariations:new A.Pointer(A.uint32,le)}}),Gn=new A.Array(A.uint16,A.uint16),Hn=new A.Struct({shrinkageEnableGSUB:new A.Pointer(A.uint16,Gn),shrinkageDisableGSUB:new A.Pointer(A.uint16,Gn),shrinkageEnableGPOS:new A.Pointer(A.uint16,Gn),shrinkageDisableGPOS:new A.Pointer(A.uint16,Gn),shrinkageJstfMax:new A.Pointer(A.uint16,new Mt(Mn)),extensionEnableGSUB:new A.Pointer(A.uint16,Gn),extensionDisableGSUB:new A.Pointer(A.uint16,Gn),extensionEnableGPOS:new A.Pointer(A.uint16,Gn),extensionDisableGPOS:new A.Pointer(A.uint16,Gn),extensionJstfMax:new A.Pointer(A.uint16,new Mt(Mn))}),Vn=new A.Array(new A.Pointer(A.uint16,Hn),A.uint16),qn=new A.Struct({tag:new A.String(4),jstfLangSys:new A.Pointer(A.uint16,Vn)}),Zn=new A.Struct({extenderGlyphs:new A.Pointer(A.uint16,new A.Array(A.uint16,A.uint16)),defaultLangSys:new A.Pointer(A.uint16,Vn),langSysCount:A.uint16,langSysRecords:new A.Array(qn,"langSysCount")}),Yn=new A.Struct({tag:new A.String(4),script:new A.Pointer(A.uint16,Zn,{type:"parent"})}),Xn=new A.Struct({version:A.uint32,scriptCount:A.uint16,scriptList:new A.Array(Yn,"scriptCount")}),Kn=(Jn.prototype.decode=function(t,e){switch(this.size(0,e)){case 1:return t.readUInt8();case 2:return t.readUInt16BE();case 3:return t.readUInt24BE();case 4:return t.readUInt32BE()}},Jn.prototype.size=function(t,e){return g.resolveLength(this._size,null,e)},Jn);function Jn(t){p(this,Jn),this._size=t}var Qn=new A.Struct({entry:new Kn(function(t){return 1+((48&t.parent.entryFormat)>>4)}),outerIndex:function(t){return t.entry>>1+(15&t.parent.entryFormat)},innerIndex:function(t){return t.entry&(1<<1+(15&t.parent.entryFormat))-1}}),$n=new A.Struct({entryFormat:A.uint16,mapCount:A.uint16,mapData:new A.Array(Qn,"mapCount")}),tr=new A.Struct({majorVersion:A.uint16,minorVersion:A.uint16,itemVariationStore:new A.Pointer(A.uint32,ne),advanceWidthMapping:new A.Pointer(A.uint32,$n),LSBMapping:new A.Pointer(A.uint32,$n),RSBMapping:new A.Pointer(A.uint32,$n)}),er=new A.Struct({format:A.uint32,length:A.uint32,offset:A.uint32}),nr=new A.Struct({reserved:new A.Reserved(A.uint16,2),cbSignature:A.uint32,signature:new A.Buffer("cbSignature")}),rr=new A.Struct({ulVersion:A.uint32,usNumSigs:A.uint16,usFlag:A.uint16,signatures:new A.Array(er,"usNumSigs"),signatureBlocks:new A.Array(nr,"usNumSigs")}),ir=new A.Struct({rangeMaxPPEM:A.uint16,rangeGaspBehavior:new A.Bitfield(A.uint16,["grayscale","gridfit","symmetricSmoothing","symmetricGridfit"])}),or=new A.Struct({version:A.uint16,numRanges:A.uint16,gaspRanges:new A.Array(ir,"numRanges")}),ar=new A.Struct({pixelSize:A.uint8,maximumWidth:A.uint8,widths:new A.Array(A.uint8,function(t){return t.parent.parent.maxp.numGlyphs})}),sr=new A.Struct({version:A.uint16,numRecords:A.int16,sizeDeviceRecord:A.int32,records:new A.Array(ar,"numRecords")}),lr=new A.Struct({left:A.uint16,right:A.uint16,value:A.int16}),ur=new A.Struct({firstGlyph:A.uint16,nGlyphs:A.uint16,offsets:new A.Array(A.uint16,"nGlyphs"),max:function(t){return t.offsets.length&&Math.max.apply(Math,t.offsets)}}),cr=new A.Struct({off:function(t){return t._startOffset-t.parent.parent._startOffset},len:function(t){return((t.parent.leftTable.max-t.off)/t.parent.rowWidth+1)*(t.parent.rowWidth/2)},values:new A.LazyArray(A.int16,"len")}),fr=new A.VersionedStruct("format",{0:{nPairs:A.uint16,searchRange:A.uint16,entrySelector:A.uint16,rangeShift:A.uint16,pairs:new A.Array(lr,"nPairs")},2:{rowWidth:A.uint16,leftTable:new A.Pointer(A.uint16,ur,{type:"parent"}),rightTable:new A.Pointer(A.uint16,ur,{type:"parent"}),array:new A.Pointer(A.uint16,cr,{type:"parent"})},3:{glyphCount:A.uint16,kernValueCount:A.uint8,leftClassCount:A.uint8,rightClassCount:A.uint8,flags:A.uint8,kernValue:new A.Array(A.int16,"kernValueCount"),leftClass:new A.Array(A.uint8,"glyphCount"),rightClass:new A.Array(A.uint8,"glyphCount"),kernIndex:new A.Array(A.uint8,function(t){return t.leftClassCount*t.rightClassCount})}}),hr=new A.VersionedStruct("version",{0:{subVersion:A.uint16,length:A.uint16,format:A.uint8,coverage:new A.Bitfield(A.uint8,["horizontal","minimum","crossStream","override"]),subtable:fr,padding:new A.Reserved(A.uint8,function(t){return t.length-t._currentOffset})},1:{length:A.uint32,coverage:new A.Bitfield(A.uint8,[null,null,null,null,null,"variation","crossStream","vertical"]),format:A.uint8,tupleIndex:A.uint16,subtable:fr,padding:new A.Reserved(A.uint8,function(t){return t.length-t._currentOffset})}}),dr=new A.VersionedStruct(A.uint16,{0:{nTables:A.uint16,tables:new A.Array(hr,"nTables")},1:{reserved:new A.Reserved(A.uint16),nTables:A.uint32,tables:new A.Array(hr,"nTables")}}),pr=new A.Struct({version:A.uint16,numGlyphs:A.uint16,yPels:new A.Array(A.uint8,"numGlyphs")}),gr=new A.Struct({version:A.uint16,fontNumber:A.uint32,pitch:A.uint16,xHeight:A.uint16,style:A.uint16,typeFamily:A.uint16,capHeight:A.uint16,symbolSet:A.uint16,typeface:new A.String(16),characterComplement:new A.String(8),fileName:new A.String(6),strokeWeight:new A.String(1),widthType:new A.String(1),serifStyle:A.uint8,reserved:new A.Reserved(A.uint8)}),vr=new A.Struct({bCharSet:A.uint8,xRatio:A.uint8,yStartRatio:A.uint8,yEndRatio:A.uint8}),yr=new A.Struct({yPelHeight:A.uint16,yMax:A.int16,yMin:A.int16}),br=new A.Struct({recs:A.uint16,startsz:A.uint8,endsz:A.uint8,entries:new A.Array(yr,"recs")}),mr=new A.Struct({version:A.uint16,numRecs:A.uint16,numRatios:A.uint16,ratioRanges:new A.Array(vr,"numRatios"),offsets:new A.Array(A.uint16,"numRatios"),groups:new A.Array(br,"numRecs")}),wr=new A.Struct({version:A.uint16,ascent:A.int16,descent:A.int16,lineGap:A.int16,advanceHeightMax:A.int16,minTopSideBearing:A.int16,minBottomSideBearing:A.int16,yMaxExtent:A.int16,caretSlopeRise:A.int16,caretSlopeRun:A.int16,caretOffset:A.int16,reserved:new A.Reserved(A.int16,4),metricDataFormat:A.int16,numberOfMetrics:A.uint16}),xr=new A.Struct({advance:A.uint16,bearing:A.int16}),_r=new A.Struct({metrics:new A.LazyArray(xr,function(t){return t.parent.vhea.numberOfMetrics}),bearings:new A.LazyArray(A.int16,function(t){return t.parent.maxp.numGlyphs-t.parent.vhea.numberOfMetrics})}),kr=new A.Fixed(16,"BE",14),Sr=new A.Struct({fromCoord:kr,toCoord:kr}),Cr=new A.Struct({pairCount:A.uint16,correspondence:new A.Array(Sr,"pairCount")}),Ar=new A.Struct({version:A.fixed32,axisCount:A.uint32,segment:new A.Array(Cr,"axisCount")}),Pr=(Er.prototype.getItem=function(t){if(null==this._items[t]){var e=this.stream.pos;this.stream.pos=this.base+this.type.size(null,this.parent)*t,this._items[t]=this.type.decode(this.stream,this.parent),this.stream.pos=e}return this._items[t]},Er.prototype.inspect=function(){return"[UnboundedArray "+this.type.constructor.name+"]"},Er);function Er(t,e,n){p(this,Er),this.type=t,this.stream=e,this.parent=n,this.base=this.stream.pos,this._items=[]}var Or,Tr=(Or=A.Array,a(Ir,Or),Ir.prototype.decode=function(t,e){return new Pr(this.type,t,e)},Ir);function Ir(t){return p(this,Ir),o(this,Or.call(this,t,0))}function Br(t){var e=0<arguments.length&&void 0!==t?t:A.uint16;function n(t){p(this,n),this.type=t}e=new(n.prototype.decode=function(t,e){return e=e.parent.parent,this.type.decode(t,e)},n.prototype.size=function(t,e){return e=e.parent.parent,this.type.size(t,e)},n.prototype.encode=function(t,e,n){return n=n.parent.parent,this.type.encode(t,e,n)},n)(e);var r=new A.Struct({unitSize:A.uint16,nUnits:A.uint16,searchRange:A.uint16,entrySelector:A.uint16,rangeShift:A.uint16}),i=new A.Struct({lastGlyph:A.uint16,firstGlyph:A.uint16,value:e}),o=new A.Struct({lastGlyph:A.uint16,firstGlyph:A.uint16,values:new A.Pointer(A.uint16,new A.Array(e,function(t){return t.lastGlyph-t.firstGlyph+1}),{type:"parent"})}),a=new A.Struct({glyph:A.uint16,value:e});return new A.VersionedStruct(A.uint16,{0:{values:new Tr(e)},2:{binarySearchHeader:r,segments:new A.Array(i,function(t){return t.binarySearchHeader.nUnits})},4:{binarySearchHeader:r,segments:new A.Array(o,function(t){return t.binarySearchHeader.nUnits})},6:{binarySearchHeader:r,segments:new A.Array(a,function(t){return t.binarySearchHeader.nUnits})},8:{firstGlyph:A.uint16,count:A.uint16,values:new A.Array(e,"count")}})}function Lr(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:{},e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:A.uint16,n=y({newState:A.uint16,flags:A.uint16},t),r=new A.Struct(n),i=new Tr(new A.Array(A.uint16,function(t){return t.nClasses}));return new A.Struct({nClasses:A.uint32,classTable:new A.Pointer(A.uint32,new Br(e)),stateArray:new A.Pointer(A.uint32,i),entryTable:new A.Pointer(A.uint32,new Tr(r))})}var Rr=new A.VersionedStruct("format",{0:{deltas:new A.Array(A.int16,32)},1:{deltas:new A.Array(A.int16,32),mappingData:new Br(A.uint16)},2:{standardGlyph:A.uint16,controlPoints:new A.Array(A.uint16,32)},3:{standardGlyph:A.uint16,controlPoints:new A.Array(A.uint16,32),mappingData:new Br(A.uint16)}}),Mr=new A.Struct({version:A.fixed32,format:A.uint16,defaultBaseline:A.uint16,subtable:Rr}),Fr=new A.Struct({setting:A.uint16,nameIndex:A.int16,name:function(t){return t.parent.parent.parent.name.records.fontFeatures[t.nameIndex]}}),Dr=new A.Struct({feature:A.uint16,nSettings:A.uint16,settingTable:new A.Pointer(A.uint32,new A.Array(Fr,"nSettings"),{type:"parent"}),featureFlags:new A.Bitfield(A.uint8,[null,null,null,null,null,null,"hasDefault","exclusive"]),defaultSetting:A.uint8,nameIndex:A.int16,name:function(t){return t.parent.parent.name.records.fontFeatures[t.nameIndex]}}),zr=new A.Struct({version:A.fixed32,featureNameCount:A.uint16,reserved1:new A.Reserved(A.uint16),reserved2:new A.Reserved(A.uint32),featureNames:new A.Array(Dr,"featureNameCount")}),Nr=new A.Struct({axisTag:new A.String(4),minValue:A.fixed32,defaultValue:A.fixed32,maxValue:A.fixed32,flags:A.uint16,nameID:A.uint16,name:function(t){return t.parent.parent.name.records.fontFeatures[t.nameID]}}),Ur=new A.Struct({nameID:A.uint16,name:function(t){return t.parent.parent.name.records.fontFeatures[t.nameID]},flags:A.uint16,coord:new A.Array(A.fixed32,function(t){return t.parent.axisCount}),postscriptNameID:new A.Optional(A.uint16,function(t){return 0<t.parent.instanceSize-t._currentOffset})}),Wr=new A.Struct({version:A.fixed32,offsetToData:A.uint16,countSizePairs:A.uint16,axisCount:A.uint16,axisSize:A.uint16,instanceCount:A.uint16,instanceSize:A.uint16,axis:new A.Array(Nr,"axisCount"),instance:new A.Array(Ur,"instanceCount")}),jr=new A.Fixed(16,"BE",14),Gr=(Hr.decode=function(t,e){return e.flags?t.readUInt32BE():2*t.readUInt16BE()},Hr);function Hr(){p(this,Hr)}var Vr=new A.Struct({version:A.uint16,reserved:new A.Reserved(A.uint16),axisCount:A.uint16,globalCoordCount:A.uint16,globalCoords:new A.Pointer(A.uint32,new A.Array(new A.Array(jr,"axisCount"),"globalCoordCount")),glyphCount:A.uint16,flags:A.uint16,offsetToData:A.uint32,offsets:new A.Array(new A.Pointer(Gr,"void",{relativeTo:"offsetToData",allowNull:!1}),function(t){return t.glyphCount+1})}),qr=new A.Struct({length:A.uint16,coverage:A.uint16,subFeatureFlags:A.uint32,stateTable:new function(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:{},e=(1<arguments.length&&void 0!==arguments[1]||A.uint16,new A.Struct({version:function(){return 8},firstGlyph:A.uint16,values:new A.Array(A.uint8,A.uint16)})),n=y({newStateOffset:A.uint16,newState:function(t){return(t.newStateOffset-(t.parent.stateArray.base-t.parent._startOffset))/t.parent.nClasses},flags:A.uint16},t),r=new A.Struct(n),i=new Tr(new A.Array(A.uint8,function(t){return t.nClasses}));return new A.Struct({nClasses:A.uint16,classTable:new A.Pointer(A.uint16,e),stateArray:new A.Pointer(A.uint16,i),entryTable:new A.Pointer(A.uint16,new Tr(r))})}}),Zr=new A.Struct({justClass:A.uint32,beforeGrowLimit:A.fixed32,beforeShrinkLimit:A.fixed32,afterGrowLimit:A.fixed32,afterShrinkLimit:A.fixed32,growFlags:A.uint16,shrinkFlags:A.uint16}),Yr=new A.Array(Zr,A.uint32),Xr=new A.VersionedStruct("actionType",{0:{lowerLimit:A.fixed32,upperLimit:A.fixed32,order:A.uint16,glyphs:new A.Array(A.uint16,A.uint16)},1:{addGlyph:A.uint16},2:{substThreshold:A.fixed32,addGlyph:A.uint16,substGlyph:A.uint16},3:{},4:{variationAxis:A.uint32,minimumLimit:A.fixed32,noStretchValue:A.fixed32,maximumLimit:A.fixed32},5:{flags:A.uint16,glyph:A.uint16}}),Kr=new A.Struct({actionClass:A.uint16,actionType:A.uint16,actionLength:A.uint32,actionData:Xr,padding:new A.Reserved(A.uint8,function(t){return t.actionLength-t._currentOffset})}),Jr=new A.Array(Kr,A.uint32),Qr=new A.Struct({lookupTable:new Br(new A.Pointer(A.uint16,Jr))}),$r=new A.Struct({classTable:new A.Pointer(A.uint16,qr,{type:"parent"}),wdcOffset:A.uint16,postCompensationTable:new A.Pointer(A.uint16,Qr,{type:"parent"}),widthDeltaClusters:new Br(new A.Pointer(A.uint16,Yr,{type:"parent",relativeTo:"wdcOffset"}))}),ti=new A.Struct({version:A.uint32,format:A.uint16,horizontal:new A.Pointer(A.uint16,$r),vertical:new A.Pointer(A.uint16,$r)}),ei={action:A.uint16},ni={markIndex:A.uint16,currentIndex:A.uint16},ri={currentInsertIndex:A.uint16,markedInsertIndex:A.uint16},ii=new A.Struct({items:new Tr(new A.Pointer(A.uint32,new Br))}),oi=new A.VersionedStruct("type",{0:{stateTable:new Lr},1:{stateTable:new Lr(ni),substitutionTable:new A.Pointer(A.uint32,ii)},2:{stateTable:new Lr(ei),ligatureActions:new A.Pointer(A.uint32,new Tr(A.uint32)),components:new A.Pointer(A.uint32,new Tr(A.uint16)),ligatureList:new A.Pointer(A.uint32,new Tr(A.uint16))},4:{lookupTable:new Br},5:{stateTable:new Lr(ri),insertionActions:new A.Pointer(A.uint32,new Tr(A.uint16))}}),ai=new A.Struct({length:A.uint32,coverage:A.uint24,type:A.uint8,subFeatureFlags:A.uint32,table:oi,padding:new A.Reserved(A.uint8,function(t){return t.length-t._currentOffset})}),si=new A.Struct({featureType:A.uint16,featureSetting:A.uint16,enableFlags:A.uint32,disableFlags:A.uint32}),li=new A.Struct({defaultFlags:A.uint32,chainLength:A.uint32,nFeatureEntries:A.uint32,nSubtables:A.uint32,features:new A.Array(si,"nFeatureEntries"),subtables:new A.Array(ai,"nSubtables")}),ui=new A.Struct({version:A.uint16,unused:new A.Reserved(A.uint16),nChains:A.uint32,chains:new A.Array(li,"nChains")}),ci=new A.Struct({left:A.int16,top:A.int16,right:A.int16,bottom:A.int16}),fi=new A.Struct({version:A.fixed32,format:A.uint16,lookupTable:new Br(ci)}),hi={};hi.cmap=U,hi.head=W,hi.hhea=j,hi.hmtx=V,hi.maxp=q,hi.name=NameTable,hi["OS/2"]=tt,hi.post=nt,hi.fpgm=it,hi.loca=ot,hi.prep=at,hi["cvt "]=rt,hi.glyf=st,hi["CFF "]=Be,hi.CFF2=Be,hi.VORG=De,hi.EBLC=qe,hi.CBLC=hi.EBLC,hi.sbix=Ye,hi.COLR=Je,hi.CPAL=$e,hi.BASE=fn,hi.GDEF=bn,hi.GPOS=Fn,hi.GSUB=jn,hi.JSTF=Xn,hi.HVAR=tr,hi.DSIG=rr,hi.gasp=or,hi.hdmx=sr,hi.kern=dr,hi.LTSH=pr,hi.PCLT=gr,hi.VDMX=mr,hi.vhea=wr,hi.vmtx=_r,hi.avar=Ar,hi.bsln=Mr,hi.feat=zr,hi.fvar=Wr,hi.gvar=Vr,hi.just=ti,hi.morx=ui,hi.opbd=fi;var di,pi=new A.Struct({tag:new A.String(4),checkSum:A.uint32,offset:new A.Pointer(A.uint32,"void",{type:"global"}),length:A.uint32}),gi=new A.Struct({tag:new A.String(4),numTables:A.uint16,searchRange:A.uint16,entrySelector:A.uint16,rangeShift:A.uint16,tables:new A.Array(pi,"numTables")});function vi(t,e){for(var n=0,r=t.length-1;n<=r;){var i=n+r>>1,o=e(t[i]);if(o<0)r=i-1;else{if(!(0<o))return i;n=1+i}}return-1}function yi(t,e){for(var n=[];t<e;)n.push(t++);return n}function bi(n,r,t,e,i){var o={};return Object.keys(e).forEach(function(t){o[t]=e[t]}),o.enumerable=!!o.enumerable,o.configurable=!!o.configurable,("value"in o||o.initializer)&&(o.writable=!0),o=t.slice().reverse().reduce(function(t,e){return e(n,r,t)||t},o),i&&void 0!==o.initializer&&(o.value=o.initializer?o.initializer.call(i):void 0,o.initializer=void 0),void 0===o.initializer&&(Object.defineProperty(n,r,o),o=null),o}gi.process=function(){var t={},e=this.tables,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;t[o.tag]=o}this.tables=t},gi.preEncode=function(t){var e=[];for(var n in this.tables){var r=this.tables[n];r&&e.push({tag:n,checkSum:0,offset:new A.VoidPointer(hi[n],r),length:hi[n].size(r)})}this.tag="true",this.numTables=e.length,this.tables=e;var i=Math.floor(Math.log(this.numTables)/Math.LN2),o=Math.pow(2,i);this.searchRange=16*o,this.entrySelector=Math.log(o)/Math.LN2,this.rangeShift=16*this.numTables-this.searchRange};try{var mi=Ou(103)}catch(t){}var wi=(xi.prototype.findSubtable=function(t,e){var n=e,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o,s=a[0],l=a[1],u=t.tables,c=Array.isArray(u),f=0;for(u=c?u:H(u);;){var h;if(c){if(f>=u.length)break;h=u[f++]}else{if((f=u.next()).done)break;h=f.value}var d=h;if(d.platformID===s&&d.encodingID===l)return d.table}}return null},xi.prototype.lookup=function(t,e){if(this.encoding)for(var n=mi.encode(b(t),this.encoding),r=t=0;r<n.length;r++)t=t<<8|n[r];else if(e){var i=this.getVariationSelector(t,e);if(i)return i}var o=this.cmap;switch(o.version){case 0:return o.codeMap.get(t)||0;case 4:for(var a=0,s=o.segCount-1;a<=s;){var l=a+s>>1;if(t<o.startCode.get(l))s=l-1;else{if(!(t>o.endCode.get(l))){var u=o.idRangeOffset.get(l),c=void 0;if(0===u)c=t+o.idDelta.get(l);else{var f=u/2+(t-o.startCode.get(l))-(o.segCount-l);0!==(c=o.glyphIndexArray.get(f)||0)&&(c+=o.idDelta.get(l))}return 65535&c}a=1+l}}return 0;case 8:throw new Error("TODO: cmap format 8");case 6:case 10:return o.glyphIndices.get(t-o.firstCode)||0;case 12:case 13:for(var h=0,d=o.nGroups-1;h<=d;){var p=h+d>>1,g=o.groups.get(p);if(t<g.startCharCode)d=p-1;else{if(!(t>g.endCharCode))return 12===o.version?g.glyphID+(t-g.startCharCode):g.glyphID;h=1+p}}return 0;case 14:throw new Error("TODO: cmap format 14");default:throw new Error("Unknown cmap format "+o.version)}},xi.prototype.getVariationSelector=function(e,n){if(!this.uvs)return 0;var t=this.uvs.varSelectors.toArray(),r=vi(t,function(t){return n-t.varSelector}),i=t[r];return-1!==r&&i.defaultUVS&&(r=vi(i.defaultUVS,function(t){return e<t.startUnicodeValue?-1:e>t.startUnicodeValue+t.additionalCount?1:0})),-1!==r&&i.nonDefaultUVS&&-1!==(r=vi(i.nonDefaultUVS,function(t){return e-t.unicodeValue}))?i.nonDefaultUVS[r].glyphID:0},xi.prototype.getCharacterSet=function(){var t=this.cmap;switch(t.version){case 0:return yi(0,t.codeMap.length);case 4:for(var e=[],n=t.endCode.toArray(),r=0;r<n.length;r++){var i=n[r]+1,o=t.startCode.get(r);e.push.apply(e,yi(o,i))}return e;case 8:throw new Error("TODO: cmap format 8");case 6:case 10:return yi(t.firstCode,t.firstCode+t.glyphIndices.length);case 12:case 13:var a=[],s=t.groups.toArray(),l=Array.isArray(s),u=0;for(s=l?s:H(s);;){var c;if(l){if(u>=s.length)break;c=s[u++]}else{if((u=s.next()).done)break;c=u.value}var f=c;a.push.apply(a,yi(f.startCharCode,f.endCharCode+1))}return a;case 14:throw new Error("TODO: cmap format 14");default:throw new Error("Unknown cmap format "+t.version)}},xi.prototype.codePointsForGlyph=function(t){var e=this.cmap;switch(e.version){case 0:for(var n=[],r=0;r<256;r++)e.codeMap.get(r)===t&&n.push(r);return n;case 4:for(var i=[],o=0;o<e.segCount;o++)for(var a=e.endCode.get(o),s=e.startCode.get(o),l=e.idRangeOffset.get(o),u=e.idDelta.get(o),c=s;c<=a;c++){var f=0;if(0===l)f=c+u;else{var h=l/2+(c-s)-(e.segCount-o);0!==(f=e.glyphIndexArray.get(h)||0)&&(f+=u)}f===t&&i.push(c)}return i;case 12:var d=[],p=e.groups.toArray(),g=Array.isArray(p),v=0;for(p=g?p:H(p);;){var y;if(g){if(v>=p.length)break;y=p[v++]}else{if((v=p.next()).done)break;y=v.value}var b=y;t>=b.glyphID&&t<=b.glyphID+(b.endCharCode-b.startCharCode)&&d.push(b.startCharCode+(t-b.glyphID))}return d;case 13:var m=[],w=e.groups.toArray(),x=Array.isArray(w),_=0;for(w=x?w:H(w);;){var k;if(x){if(_>=w.length)break;k=w[_++]}else{if((_=w.next()).done)break;k=_.value}var S=k;t===S.glyphID&&m.push.apply(m,yi(S.startCharCode,S.endCharCode+1))}return m;default:throw new Error("Unknown cmap format "+e.version)}},bi((di=xi).prototype,"getCharacterSet",[O],e(di.prototype,"getCharacterSet"),di.prototype),bi(di.prototype,"codePointsForGlyph",[O],e(di.prototype,"codePointsForGlyph"),di.prototype),di);function xi(t){if(p(this,xi),this.encoding=null,this.cmap=this.findSubtable(t,[[3,10],[0,6],[0,4],[3,1],[0,3],[0,2],[0,1],[0,0]]),!this.cmap&&mi){var e=t.tables,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=Z(o.platformID,o.encodingID,o.table.language-1);mi.encodingExists(a)&&(this.cmap=o.table,this.encoding=a)}}if(!this.cmap)throw new Error("Could not find a supported cmap table");this.uvs=this.findSubtable(t,[[0,5]]),this.uvs&&14!==this.uvs.version&&(this.uvs=null)}var _i=(ki.prototype.process=function(t,e){for(var n=0;n<t.length-1;n++){var r=t[n].id,i=t[n+1].id;e[n].xAdvance+=this.getKerning(r,i)}},ki.prototype.getKerning=function(e,n){var t=0,r=this.kern.tables,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;if(!s.coverage.crossStream){switch(s.version){case 0:if(!s.coverage.horizontal)continue;break;case 1:if(s.coverage.vertical||s.coverage.variation)continue;break;default:throw new Error("Unsupported kerning table version "+s.version)}var l=0,u=s.subtable;switch(s.format){case 0:var c=vi(u.pairs,function(t){return e-t.left||n-t.right});0<=c&&(l=u.pairs[c].value);break;case 2:var f=0,h=0;f=e>=u.leftTable.firstGlyph&&e<u.leftTable.firstGlyph+u.leftTable.nGlyphs?u.leftTable.offsets[e-u.leftTable.firstGlyph]:u.array.off,n>=u.rightTable.firstGlyph&&n<u.rightTable.firstGlyph+u.rightTable.nGlyphs&&(h=u.rightTable.offsets[n-u.rightTable.firstGlyph]);var d=(f+h-u.array.off)/2;l=u.array.values.get(d);break;case 3:if(e>=u.glyphCount||n>=u.glyphCount)return 0;l=u.kernValue[u.kernIndex[u.leftClass[e]*u.rightClassCount+u.rightClass[n]]];break;default:throw new Error("Unsupported kerning sub-table format "+s.format)}s.coverage.override?t=l:t+=l}}return t},ki);function ki(t){p(this,ki),this.kern=t.kern}var Si=(Ci.prototype.positionGlyphs=function(t,e){for(var n=0,r=0,i=0;i<t.length;i++)t[i].isMark?r=i:(n!==r&&this.positionCluster(t,e,n,r),n=r=i);return n!==r&&this.positionCluster(t,e,n,r),e},Ci.prototype.positionCluster=function(t,e,n,r){var i=t[n],o=i.cbox.copy();1<i.codePoints.length&&(o.minX+=(i.codePoints.length-1)*o.width/i.codePoints.length);for(var a=-e[n].xAdvance,s=0,l=this.font.unitsPerEm/16,u=n+1;u<=r;u++){var c=t[u],f=c.cbox,h=e[u],d=this.getCombiningClass(c.codePoints[0]);if("Not_Reordered"!==d){switch(h.xOffset=h.yOffset=0,d){case"Double_Above":case"Double_Below":h.xOffset+=o.minX-f.width/2-f.minX;break;case"Attached_Below_Left":case"Below_Left":case"Above_Left":h.xOffset+=o.minX-f.minX;break;case"Attached_Above_Right":case"Below_Right":case"Above_Right":h.xOffset+=o.maxX-f.width-f.minX;break;default:h.xOffset+=o.minX+(o.width-f.width)/2-f.minX}switch(d){case"Double_Below":case"Below_Left":case"Below":case"Below_Right":case"Attached_Below_Left":case"Attached_Below":"Attached_Below_Left"!==d&&"Attached_Below"!==d||(o.minY+=l),h.yOffset=-o.minY-f.maxY,o.minY+=f.height;break;case"Double_Above":case"Above_Left":case"Above":case"Above_Right":case"Attached_Above":case"Attached_Above_Right":"Attached_Above"!==d&&"Attached_Above_Right"!==d||(o.maxY+=l),h.yOffset=o.maxY-f.minY,o.maxY+=f.height}h.xAdvance=h.yAdvance=0,h.xOffset+=a,h.yOffset+=s}else a-=h.xAdvance,s-=h.yAdvance}},Ci.prototype.getCombiningClass=function(t){var e=P.getCombiningClass(t);if(3584==(-256&t))if("Not_Reordered"===e)switch(t){case 3633:case 3636:case 3637:case 3638:case 3639:case 3655:case 3660:case 3645:case 3662:return"Above_Right";case 3761:case 3764:case 3765:case 3766:case 3767:case 3771:case 3788:case 3789:return"Above";case 3772:return"Below"}else if(3642===t)return"Below_Right";switch(e){case"CCC10":case"CCC11":case"CCC12":case"CCC13":case"CCC14":case"CCC15":case"CCC16":case"CCC17":case"CCC18":case"CCC20":case"CCC22":return"Below";case"CCC23":return"Attached_Above";case"CCC24":return"Above_Right";case"CCC25":case"CCC19":return"Above_Left";case"CCC26":return"Above";case"CCC21":break;case"CCC27":case"CCC28":case"CCC30":case"CCC31":case"CCC33":case"CCC34":case"CCC35":case"CCC36":return"Above";case"CCC29":case"CCC32":return"Below";case"CCC103":return"Below_Right";case"CCC107":return"Above_Right";case"CCC118":return"Below";case"CCC122":return"Above";case"CCC129":case"CCC132":return"Below";case"CCC130":return"Above"}return e},Ci);function Ci(t){p(this,Ci),this.font=t}var Ai=(Pi.prototype.addPoint=function(t,e){Math.abs(t)!==1/0&&(t<this.minX&&(this.minX=t),t>this.maxX&&(this.maxX=t)),Math.abs(e)!==1/0&&(e<this.minY&&(this.minY=e),e>this.maxY&&(this.maxY=e))},Pi.prototype.copy=function(){return new Pi(this.minX,this.minY,this.maxX,this.maxY)},r(Pi,[{key:"width",get:function(){return this.maxX-this.minX}},{key:"height",get:function(){return this.maxY-this.minY}}]),Pi);function Pi(){var t=0<arguments.length&&void 0!==arguments[0]?arguments[0]:1/0,e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:1/0,n=2<arguments.length&&void 0!==arguments[2]?arguments[2]:-1/0,r=3<arguments.length&&void 0!==arguments[3]?arguments[3]:-1/0;p(this,Pi),this.minX=t,this.minY=e,this.maxX=n,this.maxY=r}var Ei={Caucasian_Albanian:"aghb",Arabic:"arab",Imperial_Aramaic:"armi",Armenian:"armn",Avestan:"avst",Balinese:"bali",Bamum:"bamu",Bassa_Vah:"bass",Batak:"batk",Bengali:["bng2","beng"],Bopomofo:"bopo",Brahmi:"brah",Braille:"brai",Buginese:"bugi",Buhid:"buhd",Chakma:"cakm",Canadian_Aboriginal:"cans",Carian:"cari",Cham:"cham",Cherokee:"cher",Coptic:"copt",Cypriot:"cprt",Cyrillic:"cyrl",Devanagari:["dev2","deva"],Deseret:"dsrt",Duployan:"dupl",Egyptian_Hieroglyphs:"egyp",Elbasan:"elba",Ethiopic:"ethi",Georgian:"geor",Glagolitic:"glag",Gothic:"goth",Grantha:"gran",Greek:"grek",Gujarati:["gjr2","gujr"],Gurmukhi:["gur2","guru"],Hangul:"hang",Han:"hani",Hanunoo:"hano",Hebrew:"hebr",Hiragana:"hira",Pahawh_Hmong:"hmng",Katakana_Or_Hiragana:"hrkt",Old_Italic:"ital",Javanese:"java",Kayah_Li:"kali",Katakana:"kana",Kharoshthi:"khar",Khmer:"khmr",Khojki:"khoj",Kannada:["knd2","knda"],Kaithi:"kthi",Tai_Tham:"lana",Lao:"lao ",Latin:"latn",Lepcha:"lepc",Limbu:"limb",Linear_A:"lina",Linear_B:"linb",Lisu:"lisu",Lycian:"lyci",Lydian:"lydi",Mahajani:"mahj",Mandaic:"mand",Manichaean:"mani",Mende_Kikakui:"mend",Meroitic_Cursive:"merc",Meroitic_Hieroglyphs:"mero",Malayalam:["mlm2","mlym"],Modi:"modi",Mongolian:"mong",Mro:"mroo",Meetei_Mayek:"mtei",Myanmar:["mym2","mymr"],Old_North_Arabian:"narb",Nabataean:"nbat",Nko:"nko ",Ogham:"ogam",Ol_Chiki:"olck",Old_Turkic:"orkh",Oriya:["ory2","orya"],Osmanya:"osma",Palmyrene:"palm",Pau_Cin_Hau:"pauc",Old_Permic:"perm",Phags_Pa:"phag",Inscriptional_Pahlavi:"phli",Psalter_Pahlavi:"phlp",Phoenician:"phnx",Miao:"plrd",Inscriptional_Parthian:"prti",Rejang:"rjng",Runic:"runr",Samaritan:"samr",Old_South_Arabian:"sarb",Saurashtra:"saur",Shavian:"shaw",Sharada:"shrd",Siddham:"sidd",Khudawadi:"sind",Sinhala:"sinh",Sora_Sompeng:"sora",Sundanese:"sund",Syloti_Nagri:"sylo",Syriac:"syrc",Tagbanwa:"tagb",Takri:"takr",Tai_Le:"tale",New_Tai_Lue:"talu",Tamil:["tml2","taml"],Tai_Viet:"tavt",Telugu:["tel2","telu"],Tifinagh:"tfng",Tagalog:"tglg",Thaana:"thaa",Thai:"thai",Tibetan:"tibt",Tirhuta:"tirh",Ugaritic:"ugar",Vai:"vai ",Warang_Citi:"wara",Old_Persian:"xpeo",Cuneiform:"xsux",Yi:"yi  ",Inherited:"zinh",Common:"zyyy",Unknown:"zzzz"},Oi={};for(var Ti in Ei){var Ii=Ei[Ti];if(Array.isArray(Ii)){var Bi=Ii,Li=Array.isArray(Bi),Ri=0;for(Bi=Li?Bi:H(Bi);;){var Mi;if(Li){if(Ri>=Bi.length)break;Mi=Bi[Ri++]}else{if((Ri=Bi.next()).done)break;Mi=Ri.value}Oi[Mi]=Ti}}else Oi[Ii]=Ti}var Fi={arab:!0,hebr:!0,syrc:!0,thaa:!0,cprt:!0,khar:!0,phnx:!0,"nko ":!0,lydi:!0,avst:!0,armi:!0,phli:!0,prti:!0,sarb:!0,orkh:!0,samr:!0,mand:!0,merc:!0,mero:!0,mani:!0,mend:!0,nbat:!0,narb:!0,palm:!0,phlp:!0};function Di(t){return Fi[t]?"rtl":"ltr"}var zi=(r(Ni,[{key:"advanceWidth",get:function(){var t=0,e=this.positions,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}t+=i.xAdvance}return t}},{key:"advanceHeight",get:function(){var t=0,e=this.positions,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}t+=i.yAdvance}return t}},{key:"bbox",get:function(){for(var t=new Ai,e=0,n=0,r=0;r<this.glyphs.length;r++){var i=this.glyphs[r],o=this.positions[r],a=i.bbox;t.addPoint(a.minX+e+o.xOffset,a.minY+n+o.yOffset),t.addPoint(a.maxX+e+o.xOffset,a.maxY+n+o.yOffset),e+=o.xAdvance,n+=o.yAdvance}return t}}]),Ni);function Ni(t,e,n,r,i){if(p(this,Ni),this.glyphs=t,this.positions=null,this.script=n,this.language=r||null,this.direction=i||Di(n),this.features={},Array.isArray(e)){var o=e,a=Array.isArray(o),s=0;for(o=a?o:H(o);;){var l;if(a){if(s>=o.length)break;l=o[s++]}else{if((s=o.next()).done)break;l=s.value}var u=l;this.features[u]=!0}}else"object"===(void 0===e?"undefined":c(e))&&(this.features=e)}for(var Ui=function t(){var e=0<arguments.length&&void 0!==arguments[0]?arguments[0]:0,n=1<arguments.length&&void 0!==arguments[1]?arguments[1]:0,r=2<arguments.length&&void 0!==arguments[2]?arguments[2]:0,i=3<arguments.length&&void 0!==arguments[3]?arguments[3]:0;p(this,t),this.xAdvance=e,this.yAdvance=n,this.xOffset=r,this.yOffset=i},Wi={allTypographicFeatures:{code:0,exclusive:!1,allTypeFeatures:0},ligatures:{code:1,exclusive:!1,requiredLigatures:0,commonLigatures:2,rareLigatures:4,rebusPictures:8,diphthongLigatures:10,squaredLigatures:12,abbrevSquaredLigatures:14,symbolLigatures:16,contextualLigatures:18,historicalLigatures:20},cursiveConnection:{code:2,exclusive:!0,unconnected:0,partiallyConnected:1,cursive:2},letterCase:{code:3,exclusive:!0},verticalSubstitution:{code:4,exclusive:!1,substituteVerticalForms:0},linguisticRearrangement:{code:5,exclusive:!1,linguisticRearrangement:0},numberSpacing:{code:6,exclusive:!0,monospacedNumbers:0,proportionalNumbers:1,thirdWidthNumbers:2,quarterWidthNumbers:3},smartSwash:{code:8,exclusive:!1,wordInitialSwashes:0,wordFinalSwashes:2,nonFinalSwashes:8},diacritics:{code:9,exclusive:!0,showDiacritics:0,hideDiacritics:1,decomposeDiacritics:2},verticalPosition:{code:10,exclusive:!0,normalPosition:0,superiors:1,inferiors:2,ordinals:3,scientificInferiors:4},fractions:{code:11,exclusive:!0,noFractions:0,verticalFractions:1,diagonalFractions:2},overlappingCharacters:{code:13,exclusive:!1,preventOverlap:0},typographicExtras:{code:14,exclusive:!1,slashedZero:4},mathematicalExtras:{code:15,exclusive:!1,mathematicalGreek:10},ornamentSets:{code:16,exclusive:!0,noOrnaments:0,dingbats:1,piCharacters:2,fleurons:3,decorativeBorders:4,internationalSymbols:5,mathSymbols:6},characterAlternatives:{code:17,exclusive:!0,noAlternates:0},designComplexity:{code:18,exclusive:!0,designLevel1:0,designLevel2:1,designLevel3:2,designLevel4:3,designLevel5:4},styleOptions:{code:19,exclusive:!0,noStyleOptions:0,displayText:1,engravedText:2,illuminatedCaps:3,titlingCaps:4,tallCaps:5},characterShape:{code:20,exclusive:!0,traditionalCharacters:0,simplifiedCharacters:1,JIS1978Characters:2,JIS1983Characters:3,JIS1990Characters:4,traditionalAltOne:5,traditionalAltTwo:6,traditionalAltThree:7,traditionalAltFour:8,traditionalAltFive:9,expertCharacters:10,JIS2004Characters:11,hojoCharacters:12,NLCCharacters:13,traditionalNamesCharacters:14},numberCase:{code:21,exclusive:!0,lowerCaseNumbers:0,upperCaseNumbers:1},textSpacing:{code:22,exclusive:!0,proportionalText:0,monospacedText:1,halfWidthText:2,thirdWidthText:3,quarterWidthText:4,altProportionalText:5,altHalfWidthText:6},transliteration:{code:23,exclusive:!0,noTransliteration:0},annotation:{code:24,exclusive:!0,noAnnotation:0,boxAnnotation:1,roundedBoxAnnotation:2,circleAnnotation:3,invertedCircleAnnotation:4,parenthesisAnnotation:5,periodAnnotation:6,romanNumeralAnnotation:7,diamondAnnotation:8,invertedBoxAnnotation:9,invertedRoundedBoxAnnotation:10},kanaSpacing:{code:25,exclusive:!0,fullWidthKana:0,proportionalKana:1},ideographicSpacing:{code:26,exclusive:!0,fullWidthIdeographs:0,proportionalIdeographs:1,halfWidthIdeographs:2},unicodeDecomposition:{code:27,exclusive:!1,canonicalComposition:0,compatibilityComposition:2,transcodingComposition:4},rubyKana:{code:28,exclusive:!1,rubyKana:2},CJKSymbolAlternatives:{code:29,exclusive:!0,noCJKSymbolAlternatives:0,CJKSymbolAltOne:1,CJKSymbolAltTwo:2,CJKSymbolAltThree:3,CJKSymbolAltFour:4,CJKSymbolAltFive:5},ideographicAlternatives:{code:30,exclusive:!0,noIdeographicAlternatives:0,ideographicAltOne:1,ideographicAltTwo:2,ideographicAltThree:3,ideographicAltFour:4,ideographicAltFive:5},CJKVerticalRomanPlacement:{code:31,exclusive:!0,CJKVerticalRomanCentered:0,CJKVerticalRomanHBaseline:1},italicCJKRoman:{code:32,exclusive:!1,CJKItalicRoman:2},caseSensitiveLayout:{code:33,exclusive:!1,caseSensitiveLayout:0,caseSensitiveSpacing:2},alternateKana:{code:34,exclusive:!1,alternateHorizKana:0,alternateVertKana:2},stylisticAlternatives:{code:35,exclusive:!1,noStylisticAlternates:0,stylisticAltOne:2,stylisticAltTwo:4,stylisticAltThree:6,stylisticAltFour:8,stylisticAltFive:10,stylisticAltSix:12,stylisticAltSeven:14,stylisticAltEight:16,stylisticAltNine:18,stylisticAltTen:20,stylisticAltEleven:22,stylisticAltTwelve:24,stylisticAltThirteen:26,stylisticAltFourteen:28,stylisticAltFifteen:30,stylisticAltSixteen:32,stylisticAltSeventeen:34,stylisticAltEighteen:36,stylisticAltNineteen:38,stylisticAltTwenty:40},contextualAlternates:{code:36,exclusive:!1,contextualAlternates:0,swashAlternates:2,contextualSwashAlternates:4},lowerCase:{code:37,exclusive:!0,defaultLowerCase:0,lowerCaseSmallCaps:1,lowerCasePetiteCaps:2},upperCase:{code:38,exclusive:!0,defaultUpperCase:0,upperCaseSmallCaps:1,upperCasePetiteCaps:2},languageTag:{code:39,exclusive:!0},CJKRomanSpacing:{code:103,exclusive:!0,halfWidthCJKRoman:0,proportionalCJKRoman:1,defaultCJKRoman:2,fullWidthCJKRoman:3}},ji=function(t,e){return[Wi[t].code,Wi[t][e]]},Gi={rlig:ji("ligatures","requiredLigatures"),clig:ji("ligatures","contextualLigatures"),dlig:ji("ligatures","rareLigatures"),hlig:ji("ligatures","historicalLigatures"),liga:ji("ligatures","commonLigatures"),hist:ji("ligatures","historicalLigatures"),smcp:ji("lowerCase","lowerCaseSmallCaps"),pcap:ji("lowerCase","lowerCasePetiteCaps"),frac:ji("fractions","diagonalFractions"),dnom:ji("fractions","diagonalFractions"),numr:ji("fractions","diagonalFractions"),afrc:ji("fractions","verticalFractions"),case:ji("caseSensitiveLayout","caseSensitiveLayout"),ccmp:ji("unicodeDecomposition","canonicalComposition"),cpct:ji("CJKVerticalRomanPlacement","CJKVerticalRomanCentered"),valt:ji("CJKVerticalRomanPlacement","CJKVerticalRomanCentered"),swsh:ji("contextualAlternates","swashAlternates"),cswh:ji("contextualAlternates","contextualSwashAlternates"),curs:ji("cursiveConnection","cursive"),c2pc:ji("upperCase","upperCasePetiteCaps"),c2sc:ji("upperCase","upperCaseSmallCaps"),init:ji("smartSwash","wordInitialSwashes"),fin2:ji("smartSwash","wordFinalSwashes"),medi:ji("smartSwash","nonFinalSwashes"),med2:ji("smartSwash","nonFinalSwashes"),fin3:ji("smartSwash","wordFinalSwashes"),fina:ji("smartSwash","wordFinalSwashes"),pkna:ji("kanaSpacing","proportionalKana"),half:ji("textSpacing","halfWidthText"),halt:ji("textSpacing","altHalfWidthText"),hkna:ji("alternateKana","alternateHorizKana"),vkna:ji("alternateKana","alternateVertKana"),ital:ji("italicCJKRoman","CJKItalicRoman"),lnum:ji("numberCase","upperCaseNumbers"),onum:ji("numberCase","lowerCaseNumbers"),mgrk:ji("mathematicalExtras","mathematicalGreek"),calt:ji("contextualAlternates","contextualAlternates"),vrt2:ji("verticalSubstitution","substituteVerticalForms"),vert:ji("verticalSubstitution","substituteVerticalForms"),tnum:ji("numberSpacing","monospacedNumbers"),pnum:ji("numberSpacing","proportionalNumbers"),sups:ji("verticalPosition","superiors"),subs:ji("verticalPosition","inferiors"),ordn:ji("verticalPosition","ordinals"),pwid:ji("textSpacing","proportionalText"),hwid:ji("textSpacing","halfWidthText"),qwid:ji("textSpacing","quarterWidthText"),twid:ji("textSpacing","thirdWidthText"),fwid:ji("textSpacing","proportionalText"),palt:ji("textSpacing","altProportionalText"),trad:ji("characterShape","traditionalCharacters"),smpl:ji("characterShape","simplifiedCharacters"),jp78:ji("characterShape","JIS1978Characters"),jp83:ji("characterShape","JIS1983Characters"),jp90:ji("characterShape","JIS1990Characters"),jp04:ji("characterShape","JIS2004Characters"),expt:ji("characterShape","expertCharacters"),hojo:ji("characterShape","hojoCharacters"),nlck:ji("characterShape","NLCCharacters"),tnam:ji("characterShape","traditionalNamesCharacters"),ruby:ji("rubyKana","rubyKana"),titl:ji("styleOptions","titlingCaps"),zero:ji("typographicExtras","slashedZero"),ss01:ji("stylisticAlternatives","stylisticAltOne"),ss02:ji("stylisticAlternatives","stylisticAltTwo"),ss03:ji("stylisticAlternatives","stylisticAltThree"),ss04:ji("stylisticAlternatives","stylisticAltFour"),ss05:ji("stylisticAlternatives","stylisticAltFive"),ss06:ji("stylisticAlternatives","stylisticAltSix"),ss07:ji("stylisticAlternatives","stylisticAltSeven"),ss08:ji("stylisticAlternatives","stylisticAltEight"),ss09:ji("stylisticAlternatives","stylisticAltNine"),ss10:ji("stylisticAlternatives","stylisticAltTen"),ss11:ji("stylisticAlternatives","stylisticAltEleven"),ss12:ji("stylisticAlternatives","stylisticAltTwelve"),ss13:ji("stylisticAlternatives","stylisticAltThirteen"),ss14:ji("stylisticAlternatives","stylisticAltFourteen"),ss15:ji("stylisticAlternatives","stylisticAltFifteen"),ss16:ji("stylisticAlternatives","stylisticAltSixteen"),ss17:ji("stylisticAlternatives","stylisticAltSeventeen"),ss18:ji("stylisticAlternatives","stylisticAltEighteen"),ss19:ji("stylisticAlternatives","stylisticAltNineteen"),ss20:ji("stylisticAlternatives","stylisticAltTwenty")},Hi=1;Hi<=99;Hi++)Gi["cv"+("00"+Hi).slice(-2)]=[Wi.characterAlternatives.code,Hi];var Vi,qi={};for(var Zi in Gi){var Yi=Gi[Zi];null==qi[Yi[0]]&&(qi[Yi[0]]={}),qi[Yi[0]][Yi[1]]=Zi}function Xi(t){var e=t[0],n=t[1];if(isNaN(e))var r=Wi[e]&&Wi[e].code;else r=e;if(isNaN(n))var i=Wi[e]&&Wi[e][n];else i=n;return[r,i]}var Ki,Ji,Qi,$i,to,eo,no=(ro.prototype.lookup=function(t){switch(this.table.version){case 0:return this.table.values.getItem(t);case 2:case 4:for(var e=0,n=this.table.binarySearchHeader.nUnits-1;e<=n;){var r=e+n>>1;if(65535===(a=this.table.segments[r]).firstGlyph)return null;if(t<a.firstGlyph)n=r-1;else{if(!(t>a.lastGlyph))return 2===this.table.version?a.value:a.values[t-a.firstGlyph];e=r+1}}return null;case 6:for(var i=0,o=this.table.binarySearchHeader.nUnits-1;i<=o;){var a;if(r=i+o>>1,65535===(a=this.table.segments[r]).glyph)return null;if(t<a.glyph)o=r-1;else{if(!(t>a.glyph))return a.value;i=r+1}}return null;case 8:return this.table.values[t-this.table.firstGlyph];default:throw new Error("Unknown lookup table format: "+this.table.version)}},ro.prototype.glyphsForValue=function(t){var e=[];switch(this.table.version){case 2:case 4:var n=this.table.segments,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;if(2===this.table.version&&a.value===t)e.push.apply(e,yi(a.firstGlyph,a.lastGlyph+1));else for(var s=0;s<a.values.length;s++)a.values[s]===t&&e.push(a.firstGlyph+s)}break;case 6:var l=this.table.segments,u=Array.isArray(l),c=0;for(l=u?l:H(l);;){var f;if(u){if(c>=l.length)break;f=l[c++]}else{if((c=l.next()).done)break;f=c.value}var h=f;h.value===t&&e.push(h.glyph)}break;case 8:for(var d=0;d<this.table.values.length;d++)this.table.values[d]===t&&e.push(this.table.firstGlyph+d);break;default:throw new Error("Unknown lookup table format: "+this.table.version)}return e},Qi=[O],$i=e(Ki=(Vi=ro).prototype,Ji="glyphsForValue"),to=Vi.prototype,eo={},Object.keys($i).forEach(function(t){eo[t]=$i[t]}),eo.enumerable=!!eo.enumerable,eo.configurable=!!eo.configurable,("value"in eo||eo.initializer)&&(eo.writable=!0),eo=Qi.slice().reverse().reduce(function(t,e){return e(Ki,Ji,t)||t},eo),to&&void 0!==eo.initializer&&(eo.value=eo.initializer?eo.initializer.call(to):void 0,eo.initializer=void 0),void 0===eo.initializer&&(Object.defineProperty(Ki,Ji,eo),eo=null),Vi);function ro(t){p(this,ro),this.table=t}var io,oo=(ao.prototype.process=function(t,e,n){for(var r=0,i=e?t.length-1:0,o=e?-1:1;1==o&&i<=t.length||-1==o&&-1<=i;){var a=null,s=1,l=!0;i===t.length||-1===i?s=0:65535===(a=t[i]).id?s=2:null==(s=this.lookupTable.lookup(a.id))&&(s=1);var u=this.stateTable.stateArray.getItem(r)[s],c=this.stateTable.entryTable.getItem(u);0!==s&&2!==s&&(n(a,c,i),l=!(16384&c.flags)),r=c.newState,l&&(i+=o)}return t},ao.prototype.traverse=function(t,e,n){var r=1<arguments.length&&void 0!==e?e:0,i=2<arguments.length&&void 0!==n?n:new w;if(!i.has(r)){i.add(r);for(var o=this.stateTable,a=o.nClasses,s=o.stateArray,l=o.entryTable,u=s.getItem(r),c=4;c<a;c++){var f=u[c],h=l.getItem(f),d=this.lookupTable.glyphsForValue(c),p=Array.isArray(d),g=0;for(d=p?d:H(d);;){var v;if(p){if(g>=d.length)break;v=d[g++]}else{if((g=d.next()).done)break;v=g.value}var y=v;t.enter&&t.enter(y,h),0!==h.newState&&this.traverse(t,h.newState,i),t.exit&&t.exit(y,h)}}}},ao);function ao(t){p(this,ao),this.stateTable=t,this.lookupTable=new no(t.classTable)}var so,lo,uo,co,fo,ho,po=(go.prototype.process=function(t,e){var n=1<arguments.length&&void 0!==e?e:{},r=this.morx.chains,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a,l=s.defaultFlags,u=s.features,c=Array.isArray(u),f=0;for(u=c?u:H(u);;){var h;if(c){if(f>=u.length)break;h=u[f++]}else{if((f=u.next()).done)break;h=f.value}var d,p=h;(d=n[p.featureType])&&d[p.featureSetting]&&(l&=p.disableFlags,l|=p.enableFlags)}var g=s.subtables,v=Array.isArray(g),y=0;for(g=v?g:H(g);;){var b;if(v){if(y>=g.length)break;b=g[y++]}else{if((y=g.next()).done)break;b=y.value}var m=b;m.subFeatureFlags&l&&this.processSubtable(m,t)}}for(var w=t.length-1;0<=w;)65535===t[w].id&&t.splice(w,1),w--;return t},go.prototype.processSubtable=function(t,e){if(this.subtable=t,this.glyphs=e,4!==this.subtable.type){this.ligatureStack=[],this.markedGlyph=null,this.firstGlyph=null,this.lastGlyph=null,this.markedIndex=null;var n=this.getStateMachine(t),r=this.getProcessor(),i=!!(4194304&this.subtable.coverage);return n.process(this.glyphs,i,r)}this.processNoncontextualSubstitutions(this.subtable,this.glyphs)},go.prototype.getStateMachine=function(t){return new oo(t.table.stateTable)},go.prototype.getProcessor=function(){switch(this.subtable.type){case 0:return this.processIndicRearragement;case 1:return this.processContextualSubstitution;case 2:return this.processLigature;case 4:return this.processNoncontextualSubstitutions;case 5:return this.processGlyphInsertion;default:throw new Error("Invalid morx subtable type: "+this.subtable.type)}},go.prototype.processIndicRearragement=function(t,e,n){32768&e.flags&&(this.firstGlyph=n),8192&e.flags&&(this.lastGlyph=n),function(t,e,n,r){switch(e){case 0:return;case 1:return vo(t,[n,1],[r,0]);case 2:return vo(t,[n,0],[r,1]);case 3:return vo(t,[n,1],[r,1]);case 4:return vo(t,[n,2],[r,0]);case 5:return vo(t,[n,2],[r,0],!0,!1);case 6:return vo(t,[n,0],[r,2]);case 7:return vo(t,[n,0],[r,2],!1,!0);case 8:return vo(t,[n,1],[r,2]);case 9:return vo(t,[n,1],[r,2],!1,!0);case 10:return vo(t,[n,2],[r,1]);case 11:return vo(t,[n,2],[r,1],!0,!1);case 12:return vo(t,[n,2],[r,2]);case 13:return vo(t,[n,2],[r,2],!0,!1);case 14:return vo(t,[n,2],[r,2],!1,!0);case 15:return vo(t,[n,2],[r,2],!0,!0);default:throw new Error("Unknown verb: "+e)}}(this.glyphs,15&e.flags,this.firstGlyph,this.lastGlyph)},go.prototype.processContextualSubstitution=function(t,e,n){var r=this.subtable.table.substitutionTable.items;if(65535!==e.markIndex){var i=r.getItem(e.markIndex),o=new no(i);t=this.glyphs[this.markedGlyph],(a=o.lookup(t.id))&&(this.glyphs[this.markedGlyph]=this.font.getGlyph(a,t.codePoints))}if(65535!==e.currentIndex){var a,s=r.getItem(e.currentIndex),l=new no(s);t=this.glyphs[n],(a=l.lookup(t.id))&&(this.glyphs[n]=this.font.getGlyph(a,t.codePoints))}32768&e.flags&&(this.markedGlyph=n)},go.prototype.processLigature=function(t,e,n){if(32768&e.flags&&this.ligatureStack.push(n),8192&e.flags){for(var r,i=this.subtable.table.ligatureActions,o=this.subtable.table.components,a=this.subtable.table.ligatureList,s=e.action,l=!1,u=0,c=[],f=[];!l;){var h,d=this.ligatureStack.pop();(h=c).unshift.apply(h,this.glyphs[d].codePoints);var p=i.getItem(s++);l=!!(2147483648&p);var g=!!(1073741824&p),v=(1073741823&p)<<2>>2;if(v+=this.glyphs[d].id,u+=o.getItem(v),l||g){var y=a.getItem(u);this.glyphs[d]=this.font.getGlyph(y,c),f.push(d),u=0,c=[]}else this.glyphs[d]=this.font.getGlyph(65535)}(r=this.ligatureStack).push.apply(r,f)}},go.prototype.processNoncontextualSubstitutions=function(t,e,n){var r=new no(t.table.lookupTable);for(n=0;n<e.length;n++){var i=e[n];if(65535!==i.id){var o=r.lookup(i.id);o&&(e[n]=this.font.getGlyph(o,i.codePoints))}}},go.prototype._insertGlyphs=function(t,e,n,r){for(var i,o=[];n--;){var a=this.subtable.table.insertionActions.getItem(e++);o.push(this.font.getGlyph(a))}r||t++,(i=this.glyphs).splice.apply(i,[t,0].concat(o))},go.prototype.processGlyphInsertion=function(t,e,n){if(32768&e.flags&&(this.markedIndex=n),65535!==e.markedInsertIndex){var r=(31&e.flags)>>>5,i=!!(1024&e.flags);this._insertGlyphs(this.markedIndex,e.markedInsertIndex,r,i)}if(65535!==e.currentInsertIndex){var o=(992&e.flags)>>>5,a=!!(2048&e.flags);this._insertGlyphs(n,e.currentInsertIndex,o,a)}},go.prototype.getSupportedFeatures=function(){var t=[],e=this.morx.chains,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i.features,a=Array.isArray(o),s=0;for(o=a?o:H(o);;){var l;if(a){if(s>=o.length)break;l=o[s++]}else{if((s=o.next()).done)break;l=s.value}var u=l;t.push([u.featureType,u.featureSetting])}}return t},go.prototype.generateInputs=function(t){return this.inputCache||this.generateInputCache(),this.inputCache[t]||[]},go.prototype.generateInputCache=function(){this.inputCache={};var t=this.morx.chains,e=Array.isArray(t),n=0;for(t=e?t:H(t);;){var r;if(e){if(n>=t.length)break;r=t[n++]}else{if((n=t.next()).done)break;r=n.value}var i=r,o=i.defaultFlags,a=i.subtables,s=Array.isArray(a),l=0;for(a=s?a:H(a);;){var u;if(s){if(l>=a.length)break;u=a[l++]}else{if((l=a.next()).done)break;u=l.value}var c=u;c.subFeatureFlags&o&&this.generateInputsForSubtable(c)}}},go.prototype.generateInputsForSubtable=function(t){var u=this;if(2===t.type){if(4194304&t.coverage)throw new Error("Reverse subtable, not supported.");this.subtable=t,this.ligatureStack=[];var e=this.getStateMachine(t),c=this.getProcessor(),f=[],h=[];this.glyphs=[],e.traverse({enter:function(t,e){var n=u.glyphs;h.push({glyphs:n.slice(),ligatureStack:u.ligatureStack.slice()});var r=u.font.getGlyph(t);f.push(r),n.push(f[f.length-1]),c(n[n.length-1],e,n.length-1);for(var i=0,o=0,a=0;a<n.length&&i<=1;a++)65535!==n[a].id&&(i++,o=n[a].id);if(1===i){var s=f.map(function(t){return t.id}),l=u.inputCache[o];l?l.push(s):u.inputCache[o]=[s]}},exit:function(){var t=h.pop();u.glyphs=t.glyphs,u.ligatureStack=t.ligatureStack,f.pop()}})}},uo=[O],co=e(so=(io=go).prototype,lo="getStateMachine"),fo=io.prototype,ho={},Object.keys(co).forEach(function(t){ho[t]=co[t]}),ho.enumerable=!!ho.enumerable,ho.configurable=!!ho.configurable,("value"in ho||ho.initializer)&&(ho.writable=!0),ho=uo.slice().reverse().reduce(function(t,e){return e(so,lo,t)||t},ho),fo&&void 0!==ho.initializer&&(ho.value=ho.initializer?ho.initializer.call(fo):void 0,ho.initializer=void 0),void 0===ho.initializer&&(Object.defineProperty(so,lo,ho),ho=null),io);function go(t){p(this,go),this.processIndicRearragement=this.processIndicRearragement.bind(this),this.processContextualSubstitution=this.processContextualSubstitution.bind(this),this.processLigature=this.processLigature.bind(this),this.processNoncontextualSubstitutions=this.processNoncontextualSubstitutions.bind(this),this.processGlyphInsertion=this.processGlyphInsertion.bind(this),this.font=t,this.morx=t.morx,this.inputCache=null}function vo(t,e,n,r,i){var o=3<arguments.length&&void 0!==r&&r,a=4<arguments.length&&void 0!==i&&i,s=t.splice(n[0]-(n[1]-1),n[1]);a&&s.reverse();var l=t.splice.apply(t,[e[0],e[1]].concat(s));return o&&l.reverse(),t.splice.apply(t,[n[0]-(e[1]-1),0].concat(l)),t}var yo=(bo.prototype.substitute=function(t){"rtl"===t.direction&&t.glyphs.reverse(),this.morxProcessor.process(t.glyphs,function(t){var e={};for(var n in t){var r;(r=Gi[n])&&(null==e[r[0]]&&(e[r[0]]={}),e[r[0]][r[1]]=t[n])}return e}(t.features))},bo.prototype.getAvailableFeatures=function(t,e){return function(t){var e={};if(Array.isArray(t))for(var n=0;n<t.length;n++){var r,i=Xi(t[n]);(r=qi[i[0]]&&qi[i[0]][i[1]])&&(e[r]=!0)}else if("object"===(void 0===t?"undefined":c(t)))for(var o in t){var a=t[o];for(var s in a){var l=void 0,u=Xi([o,s]);a[s]&&(l=qi[u[0]]&&qi[u[0]][u[1]])&&(e[l]=!0)}}return f(e)}(this.morxProcessor.getSupportedFeatures())},bo.prototype.stringsForGlyph=function(t){var e=this.morxProcessor.generateInputs(t),n=new w,r=e,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;this._addStrings(s,0,n,"")}return n},bo.prototype._addStrings=function(t,e,n,r){var i=this.font._cmapProcessor.codePointsForGlyph(t[e]),o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=r+b(s);e<t.length-1?this._addStrings(t,e+1,n,l):n.add(l)}},bo);function bo(t){p(this,bo),this.font=t,this.morxProcessor=new po(t),this.fallbackPosition=!1}var mo,wo,xo=(_o.prototype._addFeatures=function(t,e){var n=this.stages.length-1,r=this.stages[n],i=t,o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=s;null==this.allFeatures[l]&&(r.push(l),this.allFeatures[l]=n,e&&(this.globalFeatures[l]=!0))}},_o.prototype.add=function(t,e){var n=!(1<arguments.length&&void 0!==e)||e;if(0===this.stages.length&&this.stages.push([]),"string"==typeof t&&(t=[t]),Array.isArray(t))this._addFeatures(t,n);else{if("object"!==(void 0===t?"undefined":c(t)))throw new Error("Unsupported argument to ShapingPlan#add");this._addFeatures(t.global||[],!0),this._addFeatures(t.local||[],!1)}},_o.prototype.addStage=function(t,e){"function"==typeof t?this.stages.push(t,[]):(this.stages.push([]),this.add(t,e))},_o.prototype.setFeatureOverrides=function(t){if(Array.isArray(t))this.add(t);else if("object"===(void 0===t?"undefined":c(t)))for(var e in t)if(t[e])this.add(e);else if(null!=this.allFeatures[e]){var n=this.stages[this.allFeatures[e]];n.splice(n.indexOf(e),1),delete this.allFeatures[e],delete this.globalFeatures[e]}},_o.prototype.assignGlobalFeatures=function(t){var e=t,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;for(var a in this.globalFeatures)o.features[a]=!0}},_o.prototype.process=function(t,e,n){var r=this.stages,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;"function"==typeof s?n||s(this.font,e,this):0<s.length&&t.applyFeatures(s,e,n)}},_o);function _o(t,e,n){p(this,_o),this.font=t,this.script=e,this.direction=n,this.stages=[],this.globalFeatures={},this.allFeatures={}}var ko=["rvrn"],So=["ccmp","locl","rlig","mark","mkmk"],Co=["frac","numr","dnom"],Ao=["calt","clig","liga","rclt","curs","kern"],Po={ltr:["ltra","ltrm"],rtl:["rtla","rtlm"]},Eo=(Oo.plan=function(t,e,n){this.planPreprocessing(t),this.planFeatures(t),this.planPostprocessing(t,n),t.assignGlobalFeatures(e),this.assignFeatures(t,e)},Oo.planPreprocessing=function(t){t.add({global:[].concat(ko,Po[t.direction]),local:Co})},Oo.planFeatures=function(t){},Oo.planPostprocessing=function(t,e){t.add([].concat(So,Ao)),t.setFeatureOverrides(e)},Oo.assignFeatures=function(t,e){for(var n=0;n<e.length;n++){var r=e[n];if(8260===r.codePoints[0]){for(var i=n,o=n+1;0<i&&P.isDigit(e[i-1].codePoints[0]);)e[i-1].features.numr=!0,e[i-1].features.frac=!0,i--;for(;o<e.length&&P.isDigit(e[o].codePoints[0]);)e[o].features.dnom=!0,e[o].features.frac=!0,o++;r.features.frac=!0,n=o-1}}},wo=mo=Oo,mo.zeroMarkWidths="AFTER_GPOS",wo);function Oo(){p(this,Oo)}var To,Io=new s(h("AAEQAAAAAAAAADGgAZUBav7t2CtPA0EUBeDZB00pin9AJZIEgyUEj0QhweDAgQOJxCBRBElQSBwSicLgkOAwnNKZ5GaY2c7uzj4o5yZfZrrbefbuIx2nSq3CGmzAWH/+K+UO7MIe7MMhHMMpnMMFXMIVXIt2t3CnP088iPqjqNN8e4Ij7Rle4LUH82rLm6i/92A+RERERERERERNmfz/89GDeRARERERzbN8ceps2Iwt9H0C9/AJ6yOlDkbTczcot5VSm8Pm1vcFWfb7+BKOLTuOd2UlTX4wGP85Eg953lWPFbnuN7PkjtLmalOWbNenkHOSa7T3KmR9MVTZ2zZkVj1kHa68MueVKH0R4zqQ44WEXLM8VjcWHP0PtKLfPzQnMtGn3W4QYf6qxFxceVI394r2xnV+1rih0fV1Vzf3fO1n3evL5J78ruvZ5ptX2Rwy92Tfb1wlEqut3U+sZ3HXOeJ7/zDrbyuP6+Zz0fqa6Nv3vhY7Yu1xWnGevmsvsUpTT/RYIe8waUH/rvHMWKFzLfN8L+rTfp645mfX7ftlnfDtYxN59w0=","base64")),Bo=["isol","fina","fin2","fin3","medi","med2","init"],Lo={Non_Joining:0,Left_Joining:1,Right_Joining:2,Dual_Joining:3,Join_Causing:3,ALAPH:4,"DALATH RISH":5,Transparent:6},Ro="isol",Mo="fina",Fo="medi",Do="med2",zo="init",No=null,Uo=[[[No,No,0],[No,Ro,2],[No,Ro,1],[No,Ro,2],[No,Ro,1],[No,Ro,6]],[[No,No,0],[No,Ro,2],[No,Ro,1],[No,Ro,2],[No,"fin2",5],[No,Ro,6]],[[No,No,0],[No,Ro,2],[zo,Mo,1],[zo,Mo,3],[zo,Mo,4],[zo,Mo,6]],[[No,No,0],[No,Ro,2],[Fo,Mo,1],[Fo,Mo,3],[Fo,Mo,4],[Fo,Mo,6]],[[No,No,0],[No,Ro,2],[Do,Ro,1],[Do,Ro,2],[Do,"fin2",5],[Do,Ro,6]],[[No,No,0],[No,Ro,2],[Ro,Ro,1],[Ro,Ro,2],[Ro,"fin2",5],[Ro,Ro,6]],[[No,No,0],[No,Ro,2],[No,Ro,1],[No,Ro,2],[No,"fin3",5],[No,Ro,6]]],Wo=(a(jo,To=Eo),jo.planFeatures=function(t){t.add(["ccmp","locl"]);for(var e=0;e<Bo.length;e++){var n=Bo[e];t.addStage(n,!1)}t.addStage("mset")},jo.assignFeatures=function(t,e){To.assignFeatures.call(this,t,e);for(var n=-1,r=0,i=[],o=0;o<e.length;o++){var a,s,l=Go((h=e[o]).codePoints[0]);if(l!==Lo.Transparent){var u=Uo[r][l];s=u[0],a=u[1],r=u[2],s!==No&&-1!==n&&(i[n]=s),i[o]=a,n=o}else i[o]=No}for(var c=0;c<e.length;c++){var f,h=e[c];(f=i[c])&&(h.features[f]=!0)}},jo);function jo(){return p(this,jo),o(this,To.apply(this,arguments))}function Go(t){var e=Io.get(t);if(e)return e-1;var n=P.getCategory(t);return"Mn"===n||"Me"===n||"Cf"===n?Lo.Transparent:Lo.Non_Joining}var Ho=(Vo.prototype.reset=function(t,e){var n=0<arguments.length&&void 0!==t?t:{},r=1<arguments.length&&void 0!==e?e:0;this.options=n,this.flags=n.flags||{},this.markAttachmentType=n.markAttachmentType||0,this.index=r},Vo.prototype.shouldIgnore=function(t){return this.flags.ignoreMarks&&t.isMark||this.flags.ignoreBaseGlyphs&&t.isBase||this.flags.ignoreLigatures&&t.isLigature||this.markAttachmentType&&t.isMark&&t.markAttachmentType!==this.markAttachmentType},Vo.prototype.move=function(t){for(this.index+=t;0<=this.index&&this.index<this.glyphs.length&&this.shouldIgnore(this.glyphs[this.index]);)this.index+=t;return this.index<0||this.index>=this.glyphs.length?null:this.glyphs[this.index]},Vo.prototype.next=function(){return this.move(1)},Vo.prototype.prev=function(){return this.move(-1)},Vo.prototype.peek=function(t){var e=0<arguments.length&&void 0!==t?t:1,n=this.index,r=this.increment(e);return this.index=n,r},Vo.prototype.peekIndex=function(t){var e=0<arguments.length&&void 0!==t?t:1,n=this.index;this.increment(e);var r=this.index;return this.index=n,r},Vo.prototype.increment=function(t){var e=0<arguments.length&&void 0!==t?t:1,n=e<0?-1:1;for(e=Math.abs(e);e--;)this.move(n);return this.glyphs[this.index]},r(Vo,[{key:"cur",get:function(){return this.glyphs[this.index]||null}}]),Vo);function Vo(t,e){p(this,Vo),this.glyphs=t,this.reset(e)}var qo=["DFLT","dflt","latn"],Zo=(Yo.prototype.findScript=function(t){if(null==this.table.scriptList)return null;Array.isArray(t)||(t=[t]);var e=t,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=this.table.scriptList,s=Array.isArray(a),l=0;for(a=s?a:H(a);;){var u;if(s){if(l>=a.length)break;u=a[l++]}else{if((l=a.next()).done)break;u=l.value}var c=u;if(c.tag===o)return c}}return null},Yo.prototype.selectScript=function(t,e,n){var r=!1,i=void 0;if(!this.script||t!==this.scriptTag){if(!(i=(i=this.findScript(t))||this.findScript(qo)))return this.scriptTag;this.scriptTag=i.tag,this.script=i.script,this.language=null,r=!(this.languageTag=null)}if(n&&n===this.direction||(this.direction=n||Di(t)),e&&e.length<4&&(e+=" ".repeat(4-e.length)),!e||e!==this.languageTag){this.language=null;var o=this.script.langSysRecords,a=Array.isArray(o),s=0;for(o=a?o:H(o);;){var l;if(a){if(s>=o.length)break;l=o[s++]}else{if((s=o.next()).done)break;l=s.value}var u=l;if(u.tag===e){this.language=u.langSys,this.languageTag=u.tag;break}}this.language||(this.language=this.script.defaultLangSys,this.languageTag=null),r=!0}if(r&&(this.features={},this.language)){var c=this.language.featureIndexes,f=Array.isArray(c),h=0;for(c=f?c:H(c);;){var d;if(f){if(h>=c.length)break;d=c[h++]}else{if((h=c.next()).done)break;d=h.value}var p=d,g=this.table.featureList[p],v=this.substituteFeatureForVariations(p);this.features[g.tag]=v||g.feature}}return this.scriptTag},Yo.prototype.lookupsForFeatures=function(t,e){var n=e,r=[],i=0<arguments.length&&void 0!==t?t:[],o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=s,u=this.features[l];if(u){var c=u.lookupListIndexes,f=Array.isArray(c),h=0;for(c=f?c:H(c);;){var d;if(f){if(h>=c.length)break;d=c[h++]}else{if((h=c.next()).done)break;d=h.value}var p=d;n&&-1!==n.indexOf(p)||r.push({feature:l,index:p,lookup:this.table.lookupList.get(p)})}}}return r.sort(function(t,e){return t.index-e.index}),r},Yo.prototype.substituteFeatureForVariations=function(t){if(-1===this.variationsIndex)return null;var e=this.table.featureVariations.featureVariationRecords[this.variationsIndex].featureTableSubstitution.substitutions,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;if(o.featureIndex===t)return o.alternateFeatureTable}return null},Yo.prototype.findVariationsIndex=function(t){var e=this.table.featureVariations;if(!e)return-1;for(var n=e.featureVariationRecords,r=0;r<n.length;r++){var i=n[r].conditionSet.conditionTable;if(this.variationConditionsMatch(i,t))return r}return-1},Yo.prototype.variationConditionsMatch=function(t,n){return t.every(function(t){var e=t.axisIndex<n.length?n[t.axisIndex]:0;return t.filterRangeMinValue<=e&&e<=t.filterRangeMaxValue})},Yo.prototype.applyFeatures=function(t,e,n){var r=this.lookupsForFeatures(t);this.applyLookups(r,e,n)},Yo.prototype.applyLookups=function(t,e,n){this.glyphs=e,this.positions=n,this.glyphIterator=new Ho(e);var r=t,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a,l=s.feature,u=s.lookup;for(this.currentFeature=l,this.glyphIterator.reset(u.flags);this.glyphIterator.index<e.length;)if(l in this.glyphIterator.cur.features){var c=u.subTables,f=Array.isArray(c),h=0;for(c=f?c:H(c);;){var d;if(f){if(h>=c.length)break;d=c[h++]}else{if((h=c.next()).done)break;d=h.value}var p=d;if(this.applyLookup(u.lookupType,p))break}this.glyphIterator.next()}else this.glyphIterator.next()}},Yo.prototype.applyLookup=function(t,e){throw new Error("applyLookup must be implemented by subclasses")},Yo.prototype.applyLookupList=function(t){var e=this.glyphIterator.options,n=this.glyphIterator.index,r=t,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;this.glyphIterator.reset(e,n),this.glyphIterator.increment(s.sequenceIndex);var l=this.table.lookupList.get(s.lookupListIndex);this.glyphIterator.reset(l.flags,this.glyphIterator.index);var u=l.subTables,c=Array.isArray(u),f=0;for(u=c?u:H(u);;){var h;if(c){if(f>=u.length)break;h=u[f++]}else{if((f=u.next()).done)break;h=f.value}var d=h;if(this.applyLookup(l.lookupType,d))break}}return this.glyphIterator.reset(e,n),!0},Yo.prototype.coverageIndex=function(t,e){switch(null==e&&(e=this.glyphIterator.cur.id),t.version){case 1:return t.glyphs.indexOf(e);case 2:var n=t.rangeRecords,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;if(a.start<=e&&e<=a.end)return a.startCoverageIndex+e-a.start}}return-1},Yo.prototype.match=function(t,e,n,r){for(var i=this.glyphIterator.index,o=this.glyphIterator.increment(t),a=0;a<e.length&&o&&n(e[a],o);)r&&r.push(this.glyphIterator.index),a++,o=this.glyphIterator.next();return this.glyphIterator.index=i,!(a<e.length)&&(r||!0)},Yo.prototype.sequenceMatches=function(t,e){return this.match(t,e,function(t,e){return t===e.id})},Yo.prototype.sequenceMatchIndices=function(t,e){var n=this;return this.match(t,e,function(t,e){return n.currentFeature in e.features&&t===e.id},[])},Yo.prototype.coverageSequenceMatches=function(t,e){var n=this;return this.match(t,e,function(t,e){return 0<=n.coverageIndex(t,e.id)})},Yo.prototype.getClassID=function(t,e){switch(e.version){case 1:var n=t-e.startGlyph;if(0<=n&&n<e.classValueArray.length)return e.classValueArray[n];break;case 2:var r=e.classRangeRecord,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;if(s.start<=t&&t<=s.end)return s.class}}return 0},Yo.prototype.classSequenceMatches=function(t,e,n){var r=this;return this.match(t,e,function(t,e){return t===r.getClassID(e.id,n)})},Yo.prototype.applyContext=function(t){switch(t.version){case 1:var e=this.coverageIndex(t.coverage);if(-1===e)return!1;var n=t.ruleSets[e],r=n,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a;if(this.sequenceMatches(1,s.input))return this.applyLookupList(s.lookupRecords)}break;case 2:if(-1===this.coverageIndex(t.coverage))return!1;if(-1===(e=this.getClassID(this.glyphIterator.cur.id,t.classDef)))return!1;var l=n=t.classSet[e],u=Array.isArray(l),c=0;for(l=u?l:H(l);;){var f;if(u){if(c>=l.length)break;f=l[c++]}else{if((c=l.next()).done)break;f=c.value}var h=f;if(this.classSequenceMatches(1,h.classes,t.classDef))return this.applyLookupList(h.lookupRecords)}break;case 3:if(this.coverageSequenceMatches(0,t.coverages))return this.applyLookupList(t.lookupRecords)}return!1},Yo.prototype.applyChainingContext=function(t){switch(t.version){case 1:var e=this.coverageIndex(t.coverage);if(-1===e)return!1;var n=t.chainRuleSets[e],r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;if(this.sequenceMatches(-a.backtrack.length,a.backtrack)&&this.sequenceMatches(1,a.input)&&this.sequenceMatches(1+a.input.length,a.lookahead))return this.applyLookupList(a.lookupRecords)}break;case 2:if(-1===this.coverageIndex(t.coverage))return!1;e=this.getClassID(this.glyphIterator.cur.id,t.inputClassDef);var s=t.chainClassSet[e];if(!s)return!1;var l=s,u=Array.isArray(l),c=0;for(l=u?l:H(l);;){var f;if(u){if(c>=l.length)break;f=l[c++]}else{if((c=l.next()).done)break;f=c.value}var h=f;if(this.classSequenceMatches(-h.backtrack.length,h.backtrack,t.backtrackClassDef)&&this.classSequenceMatches(1,h.input,t.inputClassDef)&&this.classSequenceMatches(1+h.input.length,h.lookahead,t.lookaheadClassDef))return this.applyLookupList(h.lookupRecords)}break;case 3:if(this.coverageSequenceMatches(-t.backtrackGlyphCount,t.backtrackCoverage)&&this.coverageSequenceMatches(0,t.inputCoverage)&&this.coverageSequenceMatches(t.inputGlyphCount,t.lookaheadCoverage))return this.applyLookupList(t.lookupRecords)}return!1},Yo);function Yo(t,e){p(this,Yo),this.font=t,this.table=e,this.script=null,this.scriptTag=null,this.language=null,this.languageTag=null,this.features={},this.lookups={},this.variationsIndex=t._variationProcessor?this.findVariationsIndex(t._variationProcessor.normalizedCoords):-1,this.selectScript(),this.glyphs=[],this.positions=[],this.ligatureID=1,this.currentFeature=null}var Xo,Ko,Jo=(Qo.prototype.copy=function(){return new Qo(this._font,this.id,this.codePoints,this.features)},r(Qo,[{key:"id",get:function(){return this._id},set:function(t){this._id=t,this.substituted=!0;var e=this._font.GDEF;if(e&&e.glyphClassDef){var n=Zo.prototype.getClassID(t,e.glyphClassDef);this.isBase=1===n,this.isLigature=2===n,this.isMark=3===n,this.markAttachmentType=e.markAttachClassDef?Zo.prototype.getClassID(t,e.markAttachClassDef):0}else this.isMark=0<this.codePoints.length&&this.codePoints.every(P.isMark),this.isBase=!this.isMark,this.isLigature=1<this.codePoints.length,this.markAttachmentType=0}}]),Qo);function Qo(t,e){var n=2<arguments.length&&void 0!==arguments[2]?arguments[2]:[],r=arguments[3];if(p(this,Qo),this._font=t,this.codePoints=n,this.id=e,this.features={},Array.isArray(r))for(var i=0;i<r.length;i++){var o=r[i];this.features[o]=!0}else"object"===(void 0===r?"undefined":c(r))&&y(this.features,r);this.ligatureID=null,this.ligatureComponent=null,this.isLigated=!1,this.cursiveAttachment=null,this.markAttachment=null,this.shaperInfo=null,this.substituted=!1,this.isMultiplied=!1}var $o,ta=(a(ea,$o=Eo),ea.planFeatures=function(t){t.add(["ljmo","vjmo","tjmo"],!1)},ea.assignFeatures=function(t,e){for(var n=0,r=0;r<e.length;){var i,o=e[r].codePoints[0],a=Oa(o),s=Ra[n][a];switch(i=s[0],n=s[1],i){case Ta:t.font.hasGlyphForCodePoint(o)||(r=Fa(e,r,t.font));break;case Ia:r=Da(e,r,t.font);break;case Ba:za(e,r,t.font);break;case La:r=Na(e,r,t.font)}r++}},Ko=Xo=ea,Xo.zeroMarkWidths="NONE",Ko);function ea(){return p(this,ea),o(this,$o.apply(this,arguments))}var na=44032,ra=55204-na+1,ia=4352,oa=4449,aa=4519,sa=21,la=28,ua=ia+19-1,ca=oa+sa-1,fa=aa+la-1,ha=9676,da=function(t){return 4352<=t&&t<=4447||43360<=t&&t<=43388},pa=function(t){return 4448<=t&&t<=4519||55216<=t&&t<=55238},ga=function(t){return 4520<=t&&t<=4607||55243<=t&&t<=55291},va=function(t){return 12334<=t&&t<=12335},ya=function(t){return na<=t&&t<=55204},ba=function(t){return t-na<ra&&(t-na)%la==0},ma=function(t){return ia<=t&&t<=ua},wa=function(t){return oa<=t&&t<=ca},xa=function(t){return aa+1&&1<=t&&t<=fa},_a=0,ka=1,Sa=2,Ca=3,Aa=4,Pa=5,Ea=6;function Oa(t){return da(t)?ka:pa(t)?Sa:ga(t)?Ca:ba(t)?Aa:ya(t)?Pa:va(t)?Ea:_a}var Ta=1,Ia=2,Ba=4,La=5,Ra=[[[0,0],[0,1],[0,0],[0,0],[Ta,2],[Ta,3],[La,0]],[[0,0],[0,1],[Ia,2],[0,0],[Ta,2],[Ta,3],[La,0]],[[0,0],[0,1],[0,0],[Ia,3],[Ta,2],[Ta,3],[Ba,0]],[[0,0],[0,1],[0,0],[0,0],[Ta,2],[Ta,3],[Ba,0]]];function Ma(t,e,n){return new Jo(t,t.glyphForCodePoint(e).id,[e],n)}function Fa(t,e,n){var r=t[e],i=r.codePoints[0]-na,o=aa+i%la,a=ia+(i=i/la|0)/sa|0,s=oa+i%sa;if(!n.hasGlyphForCodePoint(a)||!n.hasGlyphForCodePoint(s)||o!==aa&&!n.hasGlyphForCodePoint(o))return e;var l=Ma(n,a,r.features);l.features.ljmo=!0;var u=Ma(n,s,r.features);u.features.vjmo=!0;var c=[l,u];if(aa<o){var f=Ma(n,o,r.features);f.features.tjmo=!0,c.push(f)}return t.splice.apply(t,[e,1].concat(c)),e+c.length-1}function Da(t,e,n){var r=t[e],i=Oa(t[e].codePoints[0]),o=t[e-1].codePoints[0],a=Oa(o),s=void 0,l=void 0,u=void 0,c=void 0;if(a===Aa&&i===Ca)s=o,c=r;else{i===Sa?(l=t[e-1],u=r):(l=t[e-2],u=t[e-1],c=r);var f=l.codePoints[0],h=u.codePoints[0];ma(f)&&wa(h)&&(s=na+((f-ia)*sa+(h-oa))*la)}var d=c&&c.codePoints[0]||aa;if(null!=s&&(d===aa||xa(d))){var p=s+(d-aa);if(n.hasGlyphForCodePoint(p)){var g=a===Sa?3:2;return t.splice(e-g+1,g,Ma(n,p,r.features)),e-g+1}}return l&&(l.features.ljmo=!0),u&&(u.features.vjmo=!0),c&&(c.features.tjmo=!0),a===Aa?(Fa(t,e-1,n),e+1):e}function za(t,e,n){var r=t[e],i=t[e].codePoints[0];if(0!==n.glyphForCodePoint(i).advanceWidth){var o=function(t){switch(Oa(t)){case Aa:case Pa:return 1;case Sa:return 2;case Ca:return 3}}(t[e-1].codePoints[0]);return t.splice(e,1),t.splice(e-o,0,r)}}function Na(t,e,n){var r=t[e],i=t[e].codePoints[0];if(n.hasGlyphForCodePoint(ha)){var o=Ma(n,ha,r.features),a=0===n.glyphForCodePoint(i).advanceWidth?e:e+1;t.splice(a,0,o),e++}return e}var Ua,Wa,ja,Ga={stateTable:[[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,2,3,4,5,6,7,8,9,0,10,11,11,12,13,14,15,16,17],[0,0,0,18,19,20,21,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,28,29,30,31,32,33,0,34,0,0,35,36,0,0,37,0],[0,0,0,38,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,39,0,0,0,40,41,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,43,44,44,8,9,0,0,0,0,12,43,0,0,0,0],[0,0,0,0,43,44,44,8,9,0,0,0,0,0,43,0,0,0,0],[0,0,0,45,46,47,48,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,50,0,0,51,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,52,0,0,0,0,0,0,0,0],[0,0,0,53,54,55,56,57,58,0,59,0,0,60,61,0,0,62,0],[0,0,0,4,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,63,64,0,0,40,41,0,9,0,10,0,0,0,42,0,63,0,0],[0,2,3,4,5,6,7,8,9,0,10,11,11,12,13,0,2,16,0],[0,0,0,18,65,20,21,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,0,66,67,67,8,9,0,10,0,0,0,68,0,0,0,0],[0,0,0,69,0,70,70,0,71,0,72,0,0,0,0,0,0,0,0],[0,0,0,73,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,75,0,0,0,76,77,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,79,80,80,22,23,0,0,0,0,25,79,0,0,0,0],[0,0,0,18,19,20,74,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,81,82,83,84,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,86,0,0,87,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,88,0,0,0,0,0,0,0,0],[0,0,0,18,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,89,90,0,0,76,77,0,23,0,24,0,0,0,78,0,89,0,0],[0,0,0,0,91,92,92,22,23,0,24,0,0,0,93,0,0,0,0],[0,0,0,94,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,96,0,0,0,97,98,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,35,100,0,0,0,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,0,100,0,0,0,0],[0,0,0,102,103,104,105,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,107,0,0,108,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,109,0,0,0,0,0,0,0,0],[0,0,0,28,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,110,111,0,0,97,98,0,33,0,34,0,0,0,99,0,110,0,0],[0,0,0,0,112,113,113,32,33,0,34,0,0,0,114,0,0,0,0],[0,0,0,0,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,0,0,115,116,117,118,8,9,0,10,0,0,119,120,0,0,16,0],[0,0,0,0,0,121,121,0,9,0,10,0,0,0,42,0,0,0,0],[0,39,0,122,0,123,123,8,9,0,10,0,0,0,42,0,39,0,0],[0,124,64,0,0,0,0,0,0,0,0,0,0,0,0,0,124,0,0],[0,39,0,0,0,121,125,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,126,126,8,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,46,47,48,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,47,47,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,127,127,49,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,128,127,127,49,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,129,130,131,132,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,50,0,0,0,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,134,0,0,0,0,0,0,0,0],[0,0,0,135,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,136,0,0,0,137,138,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,140,141,141,57,58,0,0,0,0,60,140,0,0,0,0],[0,0,0,0,140,141,141,57,58,0,0,0,0,0,140,0,0,0,0],[0,0,0,142,143,144,145,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,147,0,0,148,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,149,0,0,0,0,0,0,0,0],[0,0,0,53,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,150,151,0,0,137,138,0,58,0,59,0,0,0,139,0,150,0,0],[0,0,0,0,152,153,153,57,58,0,59,0,0,0,154,0,0,0,0],[0,0,0,155,116,156,157,8,9,0,10,0,0,158,120,0,0,16,0],[0,0,0,0,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,75,3,4,5,159,160,8,161,0,162,0,11,12,163,0,75,16,0],[0,0,0,0,0,40,164,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,0,165,0,0,0,0],[0,124,64,0,0,40,164,0,9,0,10,0,0,0,42,0,124,0,0],[0,0,0,0,0,70,70,0,71,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,71,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,166,0,0,167,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,168,0,0,0,0,0,0,0,0],[0,0,0,0,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,0,0,0,79,80,80,22,23,0,0,0,0,0,79,0,0,0,0],[0,0,0,169,170,171,172,22,23,0,24,0,0,173,174,0,0,27,0],[0,0,0,0,0,175,175,0,23,0,24,0,0,0,78,0,0,0,0],[0,75,0,176,0,177,177,22,23,0,24,0,0,0,78,0,75,0,0],[0,178,90,0,0,0,0,0,0,0,0,0,0,0,0,0,178,0,0],[0,75,0,0,0,175,179,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,180,180,22,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,82,83,84,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,83,83,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,181,181,85,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,182,181,181,85,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,183,184,185,186,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,86,0,0,0,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,188,0,0,0,0,0,0,0,0],[0,0,0,189,170,190,191,22,23,0,24,0,0,192,174,0,0,27,0],[0,0,0,0,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,76,193,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,0,194,0,0,0,0],[0,178,90,0,0,76,193,0,23,0,24,0,0,0,78,0,178,0,0],[0,0,0,0,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,0,100,0,0,0,0],[0,0,0,195,196,197,198,32,33,0,34,0,0,199,200,0,0,37,0],[0,0,0,0,0,201,201,0,33,0,34,0,0,0,99,0,0,0,0],[0,96,0,202,0,203,203,32,33,0,34,0,0,0,99,0,96,0,0],[0,204,111,0,0,0,0,0,0,0,0,0,0,0,0,0,204,0,0],[0,96,0,0,0,201,205,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,206,206,32,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,103,104,105,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,104,104,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,207,207,106,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,208,207,207,106,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,209,210,211,212,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,107,0,0,0,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,214,0,0,0,0,0,0,0,0],[0,0,0,215,196,216,217,32,33,0,34,0,0,218,200,0,0,37,0],[0,0,0,0,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,97,219,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,0,220,0,0,0,0],[0,204,111,0,0,97,219,0,33,0,34,0,0,0,99,0,204,0,0],[0,0,0,221,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,223,0,0,0,40,224,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,119,225,0,0,0,0],[0,0,0,115,116,117,222,8,9,0,10,0,0,119,120,0,0,16,0],[0,0,0,115,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,226,64,0,0,40,224,0,9,0,10,0,0,0,42,0,226,0,0],[0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0],[0,39,0,0,0,121,121,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,44,44,8,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,227,0,228,229,0,9,0,10,0,0,230,0,0,0,0,0],[0,39,0,122,0,121,121,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,231,231,49,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,232,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,130,131,132,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,131,131,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,233,233,133,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,234,233,233,133,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,235,236,237,238,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,0,0,240,241,242,243,57,58,0,59,0,0,244,245,0,0,62,0],[0,0,0,0,0,246,246,0,58,0,59,0,0,0,139,0,0,0,0],[0,136,0,247,0,248,248,57,58,0,59,0,0,0,139,0,136,0,0],[0,249,151,0,0,0,0,0,0,0,0,0,0,0,0,0,249,0,0],[0,136,0,0,0,246,250,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,251,251,57,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,143,144,145,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,144,144,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,252,252,146,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,253,252,252,146,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,254,255,256,257,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,147,0,0,0,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,259,0,0,0,0,0,0,0,0],[0,0,0,260,241,261,262,57,58,0,59,0,0,263,245,0,0,62,0],[0,0,0,0,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,137,264,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,0,265,0,0,0,0],[0,249,151,0,0,137,264,0,58,0,59,0,0,0,139,0,249,0,0],[0,0,0,221,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,158,225,0,0,0,0],[0,0,0,155,116,156,222,8,9,0,10,0,0,158,120,0,0,16,0],[0,0,0,155,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,43,266,266,8,161,0,24,0,0,12,267,0,0,0,0],[0,75,0,176,43,268,268,269,161,0,24,0,0,0,267,0,75,0,0],[0,0,0,0,0,270,0,0,271,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,272,0,0,0,0,0,0,0,0],[0,273,274,0,0,40,41,0,9,0,10,0,0,0,42,0,273,0,0],[0,0,0,40,0,123,123,8,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,121,275,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,166,0,0,0,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,276,0,0,0,0,0,0,0,0],[0,0,0,277,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,279,0,0,0,76,280,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,173,281,0,0,0,0],[0,0,0,169,170,171,278,22,23,0,24,0,0,173,174,0,0,27,0],[0,0,0,169,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,282,90,0,0,76,280,0,23,0,24,0,0,0,78,0,282,0,0],[0,0,0,0,0,0,0,0,23,0,0,0,0,0,0,0,0,0,0],[0,75,0,0,0,175,175,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,80,80,22,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,283,0,284,285,0,23,0,24,0,0,286,0,0,0,0,0],[0,75,0,176,0,175,175,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,0,0,22,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,287,287,85,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,288,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,184,185,186,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,185,185,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,289,289,187,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,290,289,289,187,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,291,292,293,294,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,277,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,192,281,0,0,0,0],[0,0,0,189,170,190,278,22,23,0,24,0,0,192,174,0,0,27,0],[0,0,0,189,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,76,0,177,177,22,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,175,296,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,297,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,299,0,0,0,97,300,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,199,301,0,0,0,0],[0,0,0,195,196,197,298,32,33,0,34,0,0,199,200,0,0,37,0],[0,0,0,195,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,302,111,0,0,97,300,0,33,0,34,0,0,0,99,0,302,0,0],[0,0,0,0,0,0,0,0,33,0,0,0,0,0,0,0,0,0,0],[0,96,0,0,0,201,201,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,101,101,32,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,303,0,304,305,0,33,0,34,0,0,306,0,0,0,0,0],[0,96,0,202,0,201,201,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,307,307,106,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,308,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,210,211,212,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,211,211,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,309,309,213,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,310,309,309,213,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,311,312,313,314,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,297,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,218,301,0,0,0,0],[0,0,0,215,196,216,298,32,33,0,34,0,0,218,200,0,0,37,0],[0,0,0,215,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,97,0,203,203,32,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,201,316,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,0,225,0,0,0,0],[0,0,0,317,318,319,320,8,9,0,10,0,0,321,322,0,0,16,0],[0,223,0,323,0,123,123,8,9,0,10,0,0,0,42,0,223,0,0],[0,223,0,0,0,121,324,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,325,318,326,327,8,9,0,10,0,0,328,322,0,0,16,0],[0,0,0,64,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,9,0,0,0,0,230,0,0,0,0,0],[0,0,0,227,0,228,121,0,9,0,10,0,0,230,0,0,0,0,0],[0,0,0,227,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,49,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,46,0,0],[0,0,0,0,0,329,329,133,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,330,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,236,237,238,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,237,237,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,331,331,239,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,332,331,331,239,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,333,40,121,334,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,335,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,337,0,0,0,137,338,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,244,339,0,0,0,0],[0,0,0,240,241,242,336,57,58,0,59,0,0,244,245,0,0,62,0],[0,0,0,240,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,340,151,0,0,137,338,0,58,0,59,0,0,0,139,0,340,0,0],[0,0,0,0,0,0,0,0,58,0,0,0,0,0,0,0,0,0,0],[0,136,0,0,0,246,246,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,141,141,57,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,341,0,342,343,0,58,0,59,0,0,344,0,0,0,0,0],[0,136,0,247,0,246,246,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,0,0,57,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,345,345,146,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,346,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,255,256,257,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,256,256,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,347,347,258,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,348,347,347,258,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,349,350,351,352,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,335,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,263,339,0,0,0,0],[0,0,0,260,241,261,336,57,58,0,59,0,0,263,245,0,0,62,0],[0,0,0,260,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,137,0,248,248,57,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,246,354,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,126,126,8,23,0,0,0,0,0,0,0,0,0,0],[0,355,90,0,0,121,125,0,9,0,10,0,0,0,42,0,355,0,0],[0,0,0,0,0,356,356,269,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,357,358,359,360,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,270,0,0,0,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,363,0,0,0,0,0,0,0,0],[0,0,0,364,116,365,366,8,161,0,162,0,0,367,120,0,0,16,0],[0,0,0,0,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,40,0,121,121,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,0,281,0,0,0,0],[0,0,0,369,370,371,372,22,23,0,24,0,0,373,374,0,0,27,0],[0,279,0,375,0,177,177,22,23,0,24,0,0,0,78,0,279,0,0],[0,279,0,0,0,175,376,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,377,370,378,379,22,23,0,24,0,0,380,374,0,0,27,0],[0,0,0,90,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,23,0,0,0,0,286,0,0,0,0,0],[0,0,0,283,0,284,175,0,23,0,24,0,0,286,0,0,0,0,0],[0,0,0,283,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,85,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,82,0,0],[0,0,0,0,0,381,381,187,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,382,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,292,293,294,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,293,293,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,383,383,295,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,384,383,383,295,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,385,76,175,386,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,76,0,175,175,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,0,301,0,0,0,0],[0,0,0,387,388,389,390,32,33,0,34,0,0,391,392,0,0,37,0],[0,299,0,393,0,203,203,32,33,0,34,0,0,0,99,0,299,0,0],[0,299,0,0,0,201,394,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,395,388,396,397,32,33,0,34,0,0,398,392,0,0,37,0],[0,0,0,111,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,33,0,0,0,0,306,0,0,0,0,0],[0,0,0,303,0,304,201,0,33,0,34,0,0,306,0,0,0,0,0],[0,0,0,303,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,106,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,103,0,0],[0,0,0,0,0,399,399,213,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,400,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,312,313,314,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,313,313,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,401,401,315,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,402,401,401,315,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,403,97,201,404,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,97,0,201,201,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,405,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,407,0,0,0,40,408,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,321,409,0,0,0,0],[0,0,0,317,318,319,406,8,9,0,10,0,0,321,322,0,0,16,0],[0,0,0,317,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,410,64,0,0,40,408,0,9,0,10,0,0,0,42,0,410,0,0],[0,223,0,0,0,121,121,0,9,0,10,0,0,0,42,0,223,0,0],[0,223,0,323,0,121,121,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,405,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,328,409,0,0,0,0],[0,0,0,325,318,326,406,8,9,0,10,0,0,328,322,0,0,16,0],[0,0,0,325,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,0,0,0,133,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,130,0,0],[0,0,0,0,0,411,411,239,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,412,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,40,121,334,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,413,0,0,0,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,0,339,0,0,0,0],[0,0,0,414,415,416,417,57,58,0,59,0,0,418,419,0,0,62,0],[0,337,0,420,0,248,248,57,58,0,59,0,0,0,139,0,337,0,0],[0,337,0,0,0,246,421,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,422,415,423,424,57,58,0,59,0,0,425,419,0,0,62,0],[0,0,0,151,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,58,0,0,0,0,344,0,0,0,0,0],[0,0,0,341,0,342,246,0,58,0,59,0,0,344,0,0,0,0,0],[0,0,0,341,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,146,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,143,0,0],[0,0,0,0,0,426,426,258,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,427,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,350,351,352,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,351,351,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,428,428,353,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,429,428,428,353,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,430,137,246,431,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,137,0,246,246,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,432,116,433,434,8,161,0,162,0,0,435,120,0,0,16,0],[0,0,0,0,0,180,180,269,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,358,359,360,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,359,359,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,436,436,361,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,437,436,436,361,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,438,439,440,441,442,161,0,162,0,0,0,362,0,0,0,0],[0,443,274,0,0,0,0,0,0,0,0,0,0,0,0,0,443,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,444,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,367,225,0,0,0,0],[0,0,0,364,116,365,445,8,161,0,162,0,0,367,120,0,0,16,0],[0,0,0,364,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,0,0,0,0,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,446,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,448,0,0,0,76,449,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,373,450,0,0,0,0],[0,0,0,369,370,371,447,22,23,0,24,0,0,373,374,0,0,27,0],[0,0,0,369,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,451,90,0,0,76,449,0,23,0,24,0,0,0,78,0,451,0,0],[0,279,0,0,0,175,175,0,23,0,24,0,0,0,78,0,279,0,0],[0,279,0,375,0,175,175,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,446,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,380,450,0,0,0,0],[0,0,0,377,370,378,447,22,23,0,24,0,0,380,374,0,0,27,0],[0,0,0,377,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,0,0,0,187,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,0,0],[0,0,0,0,0,452,452,295,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,453,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,76,175,386,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,454,0,0,0,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,455,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,457,0,0,0,97,458,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,391,459,0,0,0,0],[0,0,0,387,388,389,456,32,33,0,34,0,0,391,392,0,0,37,0],[0,0,0,387,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,460,111,0,0,97,458,0,33,0,34,0,0,0,99,0,460,0,0],[0,299,0,0,0,201,201,0,33,0,34,0,0,0,99,0,299,0,0],[0,299,0,393,0,201,201,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,455,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,398,459,0,0,0,0],[0,0,0,395,388,396,456,32,33,0,34,0,0,398,392,0,0,37,0],[0,0,0,395,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,0,0,0,213,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,210,0,0],[0,0,0,0,0,461,461,315,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,462,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,97,201,404,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,463,0,0,0,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,0,409,0,0,0,0],[0,0,0,464,465,466,467,8,9,0,10,0,0,468,469,0,0,16,0],[0,407,0,470,0,123,123,8,9,0,10,0,0,0,42,0,407,0,0],[0,407,0,0,0,121,471,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,472,465,473,474,8,9,0,10,0,0,475,469,0,0,16,0],[0,0,0,0,0,0,0,239,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,236,0,0],[0,0,0,0,0,0,476,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,477,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,479,0,0,0,137,480,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,418,481,0,0,0,0],[0,0,0,414,415,416,478,57,58,0,59,0,0,418,419,0,0,62,0],[0,0,0,414,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,482,151,0,0,137,480,0,58,0,59,0,0,0,139,0,482,0,0],[0,337,0,0,0,246,246,0,58,0,59,0,0,0,139,0,337,0,0],[0,337,0,420,0,246,246,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,477,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,425,481,0,0,0,0],[0,0,0,422,415,423,478,57,58,0,59,0,0,425,419,0,0,62,0],[0,0,0,422,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,0,0,0,258,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0],[0,0,0,0,0,483,483,353,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,484,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,137,246,431,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,485,0,0,0,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,444,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,435,225,0,0,0,0],[0,0,0,432,116,433,445,8,161,0,162,0,0,435,120,0,0,16,0],[0,0,0,432,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,0,486,486,361,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,487,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,439,440,441,442,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,440,440,442,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,488,488,442,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,489,488,488,442,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,490,491,492,493,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,495,0,496,497,0,161,0,162,0,0,498,0,0,0,0,0],[0,0,0,0,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,0,225,0,0,0,0],[0,0,0,0,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,0,450,0,0,0,0],[0,0,0,499,500,501,502,22,23,0,24,0,0,503,504,0,0,27,0],[0,448,0,505,0,177,177,22,23,0,24,0,0,0,78,0,448,0,0],[0,448,0,0,0,175,506,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,507,500,508,509,22,23,0,24,0,0,510,504,0,0,27,0],[0,0,0,0,0,0,0,295,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,292,0,0],[0,0,0,0,0,0,511,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,0,459,0,0,0,0],[0,0,0,512,513,514,515,32,33,0,34,0,0,516,517,0,0,37,0],[0,457,0,518,0,203,203,32,33,0,34,0,0,0,99,0,457,0,0],[0,457,0,0,0,201,519,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,520,513,521,522,32,33,0,34,0,0,523,517,0,0,37,0],[0,0,0,0,0,0,0,315,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,312,0,0],[0,0,0,0,0,0,524,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,525,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,527,0,0,0,40,528,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,468,529,0,0,0,0],[0,0,0,464,465,466,526,8,9,0,10,0,0,468,469,0,0,16,0],[0,0,0,464,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,530,64,0,0,40,528,0,9,0,10,0,0,0,42,0,530,0,0],[0,407,0,0,0,121,121,0,9,0,10,0,0,0,42,0,407,0,0],[0,407,0,470,0,121,121,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,525,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,475,529,0,0,0,0],[0,0,0,472,465,473,526,8,9,0,10,0,0,475,469,0,0,16,0],[0,0,0,472,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,40,0,0],[0,0,0,0,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,0,481,0,0,0,0],[0,0,0,531,532,533,534,57,58,0,59,0,0,535,536,0,0,62,0],[0,479,0,537,0,248,248,57,58,0,59,0,0,0,139,0,479,0,0],[0,479,0,0,0,246,538,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,539,532,540,541,57,58,0,59,0,0,542,536,0,0,62,0],[0,0,0,0,0,0,0,353,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,350,0,0],[0,0,0,0,0,0,543,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,361,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,358,0,0],[0,0,0,0,0,544,544,442,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,545,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,491,492,493,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,492,492,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,546,546,494,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,547,546,546,494,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,548,549,368,550,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,274,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,161,0,0,0,0,498,0,0,0,0,0],[0,0,0,495,0,496,368,0,161,0,162,0,0,498,0,0,0,0,0],[0,0,0,495,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,551,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,553,0,0,0,76,554,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,503,555,0,0,0,0],[0,0,0,499,500,501,552,22,23,0,24,0,0,503,504,0,0,27,0],[0,0,0,499,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,556,90,0,0,76,554,0,23,0,24,0,0,0,78,0,556,0,0],[0,448,0,0,0,175,175,0,23,0,24,0,0,0,78,0,448,0,0],[0,448,0,505,0,175,175,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,551,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,510,555,0,0,0,0],[0,0,0,507,500,508,552,22,23,0,24,0,0,510,504,0,0,27,0],[0,0,0,507,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,76,0,0],[0,0,0,557,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,559,0,0,0,97,560,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,516,561,0,0,0,0],[0,0,0,512,513,514,558,32,33,0,34,0,0,516,517,0,0,37,0],[0,0,0,512,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,562,111,0,0,97,560,0,33,0,34,0,0,0,99,0,562,0,0],[0,457,0,0,0,201,201,0,33,0,34,0,0,0,99,0,457,0,0],[0,457,0,518,0,201,201,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,557,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,523,561,0,0,0,0],[0,0,0,520,513,521,558,32,33,0,34,0,0,523,517,0,0,37,0],[0,0,0,520,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,97,0,0],[0,0,0,0,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,0,529,0,0,0,0],[0,0,0,563,66,564,565,8,9,0,10,0,0,566,68,0,0,16,0],[0,527,0,567,0,123,123,8,9,0,10,0,0,0,42,0,527,0,0],[0,527,0,0,0,121,568,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,569,66,570,571,8,9,0,10,0,0,572,68,0,0,16,0],[0,0,0,573,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,575,0,0,0,137,576,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,535,577,0,0,0,0],[0,0,0,531,532,533,574,57,58,0,59,0,0,535,536,0,0,62,0],[0,0,0,531,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,578,151,0,0,137,576,0,58,0,59,0,0,0,139,0,578,0,0],[0,479,0,0,0,246,246,0,58,0,59,0,0,0,139,0,479,0,0],[0,479,0,537,0,246,246,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,573,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,542,577,0,0,0,0],[0,0,0,539,532,540,574,57,58,0,59,0,0,542,536,0,0,62,0],[0,0,0,539,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,137,0,0],[0,0,0,0,0,0,0,442,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,439,0,0],[0,0,0,0,0,579,579,494,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,580,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,549,368,550,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,368,368,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,581,0,0,0,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,0,555,0,0,0,0],[0,0,0,582,91,583,584,22,23,0,24,0,0,585,93,0,0,27,0],[0,553,0,586,0,177,177,22,23,0,24,0,0,0,78,0,553,0,0],[0,553,0,0,0,175,587,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,588,91,589,590,22,23,0,24,0,0,591,93,0,0,27,0],[0,0,0,0,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,0,561,0,0,0,0],[0,0,0,592,112,593,594,32,33,0,34,0,0,595,114,0,0,37,0],[0,559,0,596,0,203,203,32,33,0,34,0,0,0,99,0,559,0,0],[0,559,0,0,0,201,597,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,598,112,599,600,32,33,0,34,0,0,601,114,0,0,37,0],[0,0,0,602,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,566,165,0,0,0,0],[0,0,0,563,66,564,67,8,9,0,10,0,0,566,68,0,0,16,0],[0,0,0,563,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,527,0,0,0,121,121,0,9,0,10,0,0,0,42,0,527,0,0],[0,527,0,567,0,121,121,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,602,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,572,165,0,0,0,0],[0,0,0,569,66,570,67,8,9,0,10,0,0,572,68,0,0,16,0],[0,0,0,569,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,0,577,0,0,0,0],[0,0,0,603,152,604,605,57,58,0,59,0,0,606,154,0,0,62,0],[0,575,0,607,0,248,248,57,58,0,59,0,0,0,139,0,575,0,0],[0,575,0,0,0,246,608,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,609,152,610,611,57,58,0,59,0,0,612,154,0,0,62,0],[0,0,0,0,0,0,0,494,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,491,0,0],[0,0,0,0,0,0,613,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,614,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,585,194,0,0,0,0],[0,0,0,582,91,583,92,22,23,0,24,0,0,585,93,0,0,27,0],[0,0,0,582,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,553,0,0,0,175,175,0,23,0,24,0,0,0,78,0,553,0,0],[0,553,0,586,0,175,175,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,614,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,591,194,0,0,0,0],[0,0,0,588,91,589,92,22,23,0,24,0,0,591,93,0,0,27,0],[0,0,0,588,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,615,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,595,220,0,0,0,0],[0,0,0,592,112,593,113,32,33,0,34,0,0,595,114,0,0,37,0],[0,0,0,592,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,559,0,0,0,201,201,0,33,0,34,0,0,0,99,0,559,0,0],[0,559,0,596,0,201,201,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,615,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,601,220,0,0,0,0],[0,0,0,598,112,599,113,32,33,0,34,0,0,601,114,0,0,37,0],[0,0,0,598,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,616,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,606,265,0,0,0,0],[0,0,0,603,152,604,153,57,58,0,59,0,0,606,154,0,0,62,0],[0,0,0,603,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,575,0,0,0,246,246,0,58,0,59,0,0,0,139,0,575,0,0],[0,575,0,607,0,246,246,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,616,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,612,265,0,0,0,0],[0,0,0,609,152,610,153,57,58,0,59,0,0,612,154,0,0,62,0],[0,0,0,609,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,549,0,0],[0,0,0,0,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0]],accepting:[!1,!0,!0,!0,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!1,!1,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0],tags:[[],["broken_cluster"],["consonant_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["consonant_syllable"],["broken_cluster"],["symbol_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["broken_cluster"],["broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],[],["broken_cluster"],["symbol_cluster"],[],["symbol_cluster"],["symbol_cluster"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["symbol_cluster"],["symbol_cluster"],["symbol_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],["symbol_cluster"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],[],["broken_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],[],["consonant_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],[],["vowel_syllable"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],[],[],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],["standalone_cluster"],[],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],[],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],[],[],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],[],[],[],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],["consonant_syllable"],["vowel_syllable"],["standalone_cluster"]]},Ha={categories:["O","IND","S","GB","B","FM","CGJ","VMAbv","VMPst","VAbv","VPst","CMBlw","VPre","VBlw","H","VMBlw","CMAbv","MBlw","CS","R","SUB","MPst","MPre","FAbv","FPst","FBlw","SMAbv","SMBlw","VMPre","ZWNJ","ZWJ","WJ","VS","N","HN","MAbv"],decompositions:{2507:[2503,2494],2508:[2503,2519],2888:[2887,2902],2891:[2887,2878],2892:[2887,2903],3018:[3014,3006],3019:[3015,3006],3020:[3014,3031],3144:[3142,3158],3264:[3263,3285],3271:[3270,3285],3272:[3270,3286],3274:[3270,3266],3275:[3270,3266,3285],3402:[3398,3390],3403:[3399,3390],3404:[3398,3415],3546:[3545,3530],3548:[3545,3535],3549:[3545,3535,3530],3550:[3545,3551],3635:[3661,3634],3763:[3789,3762],3955:[3953,3954],3957:[3953,3956],3958:[4018,3968],3959:[4018,3953,3968],3960:[4019,3968],3961:[4019,3953,3968],3969:[3953,3968],6971:[6970,6965],6973:[6972,6965],6976:[6974,6965],6977:[6975,6965],6979:[6978,6965],69934:[69937,69927],69935:[69938,69927],70475:[70471,70462],70476:[70471,70487],70843:[70841,70842],70844:[70841,70832],70846:[70841,70845],71098:[71096,71087],71099:[71097,71087]},stateTable:[[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[2,2,3,4,4,5,0,6,7,8,9,10,11,12,13,14,15,16,0,17,18,11,19,20,21,22,0,0,23,0,0,2,0,24,0,25],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,26,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,28,0,0,0,0,27,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,39,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,39,0,0,47],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,0,0,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,9,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,10,11,12,13,14,0,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,9,0,0,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,10,11,12,13,14,15,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,0,0,0,0,11,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,4,4,5,0,6,7,8,9,10,11,12,13,14,15,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,48,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,49,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,16,0,0,0,11,0,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,22,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,50,0,51,0],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,16,0,0,0,11,0,20,21,22,0,0,23,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,28,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,28,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,0,0,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,33,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,0,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,33,0,0,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,41,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,39,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,0,0,0,0,35,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,52,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,53,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,40,0,0,0,35,0,43,44,45,0,0,46,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,44,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,44,45,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,40,0,0,0,35,0,43,44,45,0,0,46,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,0,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,48,16,0,0,18,11,19,20,21,22,0,0,23,0,0,0,0,0,0,25],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,51,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,54,0,0],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,0,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,52,40,0,0,41,35,42,43,44,45,0,0,46,0,0,0,0,0,0,47],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,50,0,51,0]],accepting:[!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0],tags:[[],["broken_cluster"],["independent_cluster"],["symbol_cluster"],["standard_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["numeral_cluster"],["broken_cluster"],["independent_cluster"],["symbol_cluster"],["symbol_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["virama_terminated_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["broken_cluster"],["broken_cluster"],["numeral_cluster"],["number_joiner_terminated_cluster"],["standard_cluster"],["standard_cluster"],["numeral_cluster"]]},Va=1,qa=8,Za=16,Ya=32,Xa=64,Ka=128,Ja=8192,Qa=32768,$a=65536,ts=1<<17,es={Start:1,Ra_To_Become_Reph:2,Pre_M:4,Pre_C:8,Base_C:16,After_Main:32,Above_C:64,Before_Sub:128,Below_C:256,After_Sub:512,Before_Post:1024,Post_C:2048,After_Post:4096,Final_C:8192,SMVD:16384,End:32768},ns=2|$a|ts|4|2048|4096,rs=Xa|Ya,is=Za|16384,os={Default:{hasOldSpec:!1,virama:0,basePos:"Last",rephPos:es.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Devanagari:{hasOldSpec:!0,virama:2381,basePos:"Last",rephPos:es.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Bengali:{hasOldSpec:!0,virama:2509,basePos:"Last",rephPos:es.After_Sub,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Gurmukhi:{hasOldSpec:!0,virama:2637,basePos:"Last",rephPos:es.Before_Sub,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Gujarati:{hasOldSpec:!0,virama:2765,basePos:"Last",rephPos:es.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Oriya:{hasOldSpec:!0,virama:2893,basePos:"Last",rephPos:es.After_Main,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Tamil:{hasOldSpec:!0,virama:3021,basePos:"Last",rephPos:es.After_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Telugu:{hasOldSpec:!0,virama:3149,basePos:"Last",rephPos:es.After_Post,rephMode:"Explicit",blwfMode:"Post_Only"},Kannada:{hasOldSpec:!0,virama:3277,basePos:"Last",rephPos:es.After_Post,rephMode:"Implicit",blwfMode:"Post_Only"},Malayalam:{hasOldSpec:!0,virama:3405,basePos:"Last",rephPos:es.After_Main,rephMode:"Log_Repha",blwfMode:"Pre_And_Post"},Khmer:{hasOldSpec:!1,virama:6098,basePos:"First",rephPos:es.Ra_To_Become_Reph,rephMode:"Vis_Repha",blwfMode:"Pre_And_Post"}},as={6078:[6081,6078],6079:[6081,6079],6080:[6081,6080],6084:[6081,6084],6085:[6081,6085]},ss=Ha.decompositions,ls=new s(h("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","base64")),us=new l(Ga),cs=(a(fs,ja=Eo),fs.planFeatures=function(t){t.addStage(ys),t.addStage(["locl","ccmp"]),t.addStage(_s),t.addStage("nukt"),t.addStage("akhn"),t.addStage("rphf",!1),t.addStage("rkrf"),t.addStage("pref",!1),t.addStage("blwf",!1),t.addStage("abvf",!1),t.addStage("half",!1),t.addStage("pstf",!1),t.addStage("vatu"),t.addStage("cjct"),t.addStage("cfar",!1),t.addStage(ks),t.addStage({local:["init"],global:["pres","abvs","blws","psts","haln","dist","abvm","blwm","calt","clig"]}),t.unicodeScript=function(t){return Oi[t]}(t.script),t.indicConfig=os[t.unicodeScript]||os.Default,t.isOldSpec=t.indicConfig.hasOldSpec&&"2"!==t.script[t.script.length-1]},fs.assignFeatures=function(i,o){for(var t=function(n){var t=o[n].codePoints[0],e=as[t]||ss[t];if(e){var r=e.map(function(t){var e=i.font.glyphForCodePoint(t);return new Jo(i.font,e.id,[t],o[n].features)});o.splice.apply(o,[n,1].concat(r))}},e=o.length-1;0<=e;e--)t(e)},Wa=Ua=fs,Ua.zeroMarkWidths="NONE",Wa);function fs(){return p(this,fs),o(this,ja.apply(this,arguments))}function hs(t){return ls.get(t.codePoints[0])>>8}function ds(t){return 1<<(255&ls.get(t.codePoints[0]))}var ps,gs,vs=function t(e,n,r,i){p(this,t),this.category=e,this.position=n,this.syllableType=r,this.syllable=i};function ys(t,e){var n=0,r=0,i=us.match(e.map(hs)),o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=s,u=l[0],c=l[1],f=l[2];if(r<u){++n;for(var h=r;h<u;h++)e[h].shaperInfo=new vs(Va,es.End,"non_indic_cluster",n)}++n;for(var d=u;d<=c;d++)e[d].shaperInfo=new vs(1<<hs(e[d]),ds(e[d]),f[0],n);r=c+1}if(r<e.length){++n;for(var p=r;p<e.length;p++)e[p].shaperInfo=new vs(Va,es.End,"non_indic_cluster",n)}}function bs(t){return t.shaperInfo.category&ns}function ms(t){return t.shaperInfo.category&rs}function ws(t){return t.shaperInfo.category&is}function xs(t,e){var n=t,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o,a;if(r){if(i>=n.length)break;a=n[i++]}else{if((i=n.next()).done)break;a=i.value}a.features=((o={})[e]=!0,o)}return t[0]._font._layoutEngine.engine.GSUBProcessor.applyFeatures([e],t),1===t.length}function _s(t,e,n){var r,i,o,a=n.indicConfig,s=t._layoutEngine.engine.GSUBProcessor.features,l=t.glyphForCodePoint(9676).id,u=t.glyphForCodePoint(a.virama).id;if(u)for(var c=new Jo(t,u,[a.virama]),f=0;f<e.length;f++)e[f].shaperInfo.position===es.Base_C&&(e[f].shaperInfo.position=(r=e[f].copy(),o=void 0,xs((o=[i=c,r,i]).slice(0,2),"blwf")||xs(o.slice(1,3),"blwf")?es.Below_C:xs(o.slice(0,2),"pstf")||xs(o.slice(1,3),"pstf")?es.Post_C:xs(o.slice(0,2),"pref")||xs(o.slice(1,3),"pref")?es.Post_C:es.Base_C));for(var h=0,d=Ss(e,0);h<e.length;d=Ss(e,h=d)){var p=e[h].shaperInfo,g=(p.category,p.syllableType);if("symbol_cluster"!==g&&"non_indic_cluster"!==g){if("broken_cluster"===g&&l){var v=new Jo(t,l,[9676]);v.shaperInfo=new vs(1<<hs(v),ds(v),e[h].shaperInfo.syllableType,e[h].shaperInfo.syllable);for(var y=h;y<d&&e[y].shaperInfo.category===Qa;)y++;e.splice(y++,0,v),d++}var b=d,m=h,w=!1;if(a.rephPos!==es.Ra_To_Become_Reph&&s.rphf&&h+3<=d&&("Implicit"===a.rephMode&&!ms(e[h+2])||"Explicit"===a.rephMode&&e[h+2].shaperInfo.category===Xa)){var x=[e[h].copy(),e[h+1].copy(),e[h+2].copy()];if(xs(x.slice(0,2),"rphf")||"Explicit"===a.rephMode&&xs(x,"rphf")){for(m+=2;m<d&&ms(e[m]);)m++;b=h,w=!0}}else if("Log_Repha"===a.rephMode&&e[h].shaperInfo.category===Qa){for(m++;m<d&&ms(e[m]);)m++;b=h,w=!0}switch(a.basePos){case"Last":var _=d,k=!1;do{var S=e[--_].shaperInfo;if(bs(e[_])){if(S.position!==es.Below_C&&(S.position!==es.Post_C||k)){b=_;break}S.position===es.Below_C&&(k=!0),b=_}else if(h<_&&S.category===Xa&&e[_-1].shaperInfo.category===Za)break}while(m<_);break;case"First":for(var C=(b=h)+1;C<d;C++)bs(e[C])&&(e[C].shaperInfo.position=es.Below_C)}w&&b===h&&m-b<=2&&(w=!1);for(var A=h;A<b;A++){var P=e[A].shaperInfo;P.position=Math.min(es.Pre_C,P.position)}b<d&&(e[b].shaperInfo.position=es.Base_C);for(var E=b+1;E<d;E++)if(e[E].shaperInfo.category===Ka){for(var O=E+1;O<d;O++)if(bs(e[O])){e[O].shaperInfo.position=es.Final_C;break}break}if(w&&(e[h].shaperInfo.position=es.Ra_To_Become_Reph),n.isOldSpec)for(var T="Malayalam"!==n.unicodeScript,I=b+1;I<d;I++)if(e[I].shaperInfo.category===Za){var B=void 0;for(B=d-1;I<B&&!(bs(e[B])||T&&e[B].shaperInfo.category===Za);B--);if(e[B].shaperInfo.category!==Za&&I<B){var L=e[I];e.splice.apply(e,[I,0].concat(e.splice(I+1,B-I))),e[B]=L}break}for(var R=es.Start,M=h;M<d;M++){var F=e[M].shaperInfo;if(F.category&(rs|qa|Ja|ts|is&F.category)){if(F.position=R,F.category===Za&&F.position===es.Pre_M)for(var D=M;h<D;D--)if(e[D-1].shaperInfo.position!==es.Pre_M){F.position=e[D-1].shaperInfo.position;break}}else F.position!==es.SMVD&&(R=F.position)}for(var z=b,N=b+1;N<d;N++)if(bs(e[N])){for(var U=z+1;U<N;U++)e[U].shaperInfo.position<es.SMVD&&(e[U].shaperInfo.position=e[N].shaperInfo.position);z=N}else e[N].shaperInfo.category===Ka&&(z=N);var W=e.slice(h,d);W.sort(function(t,e){return t.shaperInfo.position-e.shaperInfo.position}),e.splice.apply(e,[h,W.length].concat(W));for(var j=h;j<d;j++)if(e[j].shaperInfo.position===es.Base_C){b=j;break}for(var G=h;G<d&&e[G].shaperInfo.position===es.Ra_To_Become_Reph;G++)e[G].features.rphf=!0;for(var H=!n.isOldSpec&&"Pre_And_Post"===a.blwfMode,V=h;V<b;V++)e[V].features.half=!0,H&&(e[V].features.blwf=!0);for(var q=b+1;q<d;q++)e[q].features.abvf=!0,e[q].features.pstf=!0,e[q].features.blwf=!0;if(n.isOldSpec&&"Devanagari"===n.unicodeScript)for(var Z=h;Z+1<b;Z++)e[Z].shaperInfo.category!==$a||e[Z+1].shaperInfo.category!==Za||Z+1!==b&&e[Z+2].shaperInfo.category!==Xa||(e[Z].features.blwf=!0,e[Z+1].features.blwf=!0);if(s.pref&&b+2<d)for(var Y=b+1;Y+2-1<d;Y++){if(xs([e[Y].copy(),e[Y+1].copy()],"pref")){for(var X=0;X<2;X++)e[Y++].features.pref=!0;if(s.cfar)for(;Y<d;Y++)e[Y].features.cfar=!0;break}}for(var K=h+1;K<d;K++)if(ms(e[K]))for(var J=e[K].shaperInfo.category===Ya,Q=K;Q--,J&&delete e[Q].features.half,h<Q&&!bs(e[Q]););}}}function ks(t,e,n){for(var r=n.indicConfig,i=t._layoutEngine.engine.GSUBProcessor.features,o=0,a=Ss(e,0);o<e.length;a=Ss(e,o=a)){for(var s=!!i.pref,l=o;l<a;l++)if(e[l].shaperInfo.position>=es.Base_C){if(s&&l+1<a)for(var u=l+1;u<a;u++)if(e[u].features.pref){if(!e[u].substituted||!e[u].isLigated||e[u].isMultiplied){for(l=u;l<a&&ws(e[l]);)l++;e[l].shaperInfo.position=es.BASE_C,s=!1}break}if("Malayalam"===n.unicodeScript)for(var c=l+1;c<a;c++){for(;c<a&&ms(e[c]);)c++;if(c===a||!ws(e[c]))break;for(c++;c<a&&ms(e[c]);)c++;c<a&&bs(e[c])&&e[c].shaperInfo.position===es.Below_C&&(e[l=c].shaperInfo.position=es.Base_C)}o<l&&e[l].shaperInfo.position>es.Base_C&&l--;break}if(l===a&&o<l&&e[l-1].shaperInfo.category===Xa&&l--,l<a)for(;o<l&&e[l].shaperInfo.category&(qa|is);)l--;if(o+1<a&&o<l){var f=l===a?l-2:l-1;if("Malayalam"!==n.unicodeScript&&"Tamil"!==n.unicodeScript){for(;o<f&&!(e[f].shaperInfo.category&(Ka|is));)f--;ws(e[f])&&e[f].shaperInfo.position!==es.Pre_M?f+1<a&&ms(e[f+1])&&f++:f=o}if(o<f&&e[f].shaperInfo.position!==es.Pre_M)for(var h=f;o<h;h--)if(e[h-1].shaperInfo.position===es.Pre_M){var d=h-1;d<l&&l<=f&&l--;var p=e[d];e.splice.apply(e,[d,0].concat(e.splice(d+1,f-d))),e[f]=p,f--}}if(o+1<a&&e[o].shaperInfo.position===es.Ra_To_Become_Reph&&e[o].shaperInfo.category===Qa!==(e[o].isLigated&&!e[o].isMultiplied)){var g=void 0,v=r.rephPos,y=!1;if(v!==es.After_Post){for(g=o+1;g<l&&!ws(e[g]);)g++;if(g<l&&ws(e[g])&&(g+1<l&&ms(e[g+1])&&g++,y=!0),!y&&v===es.After_Main){for(g=l;g+1<a&&e[g+1].shaperInfo.position<=es.After_Main;)g++;y=g<a}if(!y&&v===es.After_Sub){for(g=l;g+1<a&&!(e[g+1].shaperInfo.position&(es.Post_C|es.After_Post|es.SMVD));)g++;y=g<a}}if(!y){for(g=o+1;g<l&&!ws(e[g]);)g++;g<l&&ws(e[g])&&(g+1<l&&ms(e[g+1])&&g++,y=!0)}if(!y){for(g=a-1;o<g&&e[g].shaperInfo.position===es.SMVD;)g--;if(ws(e[g]))for(var b=l+1;b<g;b++)e[b].shaperInfo.category===Ka&&g--}var m=e[o];e.splice.apply(e,[o,0].concat(e.splice(o+1,g-o))),e[g]=m,o<l&&l<=g&&l--}if(s&&l+1<a)for(var w=l+1;w<a;w++)if(e[w].features.pref){if(e[w].isLigated&&!e[w].isMultiplied){var x=l;if("Malayalam"!==n.unicodeScript&&"Tamil"!==n.unicodeScript){for(;o<x&&!(e[x-1].shaperInfo.category&(Ka|is));)x--;if(o<x&&e[x-1].shaperInfo.category===Ka)for(var _=w,k=l+1;k<_;k++)if(e[k].shaperInfo.category===Ka){x--;break}}o<x&&ws(e[x-1])&&x<a&&ms(e[x])&&x++;var S=w,C=e[S];e.splice.apply(e,[x+1,0].concat(e.splice(x,S-x))),e[x]=C,x<=l&&l<S&&l++}break}e[o].shaperInfo.position!==es.Pre_M||o&&/Cf|Mn/.test(P.getCategory(e[o-1].codePoints[0]))||(e[o].features.init=!0)}}function Ss(t,e){if(e>=t.length)return e;for(var n=t[e].shaperInfo.syllable;++e<t.length&&t[e].shaperInfo.syllable===n;);return e}var Cs,As=Ha.categories,Ps=Ha.decompositions,Es=new s(h("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","base64")),Os=new l(Ha),Ts=(a(Is,Cs=Eo),Is.planFeatures=function(t){t.addStage(Rs),t.addStage(["locl","ccmp","nukt","akhn"]),t.addStage(Ms),t.addStage(["rphf"],!1),t.addStage(Fs),t.addStage(Ms),t.addStage(["pref"]),t.addStage(Ds),t.addStage(["rkrf","abvf","blwf","half","pstf","vatu","cjct"]),t.addStage(zs),t.addStage(["abvs","blws","pres","psts","dist","abvm","blwm"])},Is.assignFeatures=function(r,i){for(var t=function(n){var t=i[n].codePoints[0];if(Ps[t]){var e=Ps[t].map(function(t){var e=r.font.glyphForCodePoint(t);return new Jo(r.font,e.id,[t],i[n].features)});i.splice.apply(i,[n,1].concat(e))}},e=i.length-1;0<=e;e--)t(e)},gs=ps=Is,ps.zeroMarkWidths="BEFORE_GPOS",gs);function Is(){return p(this,Is),o(this,Cs.apply(this,arguments))}function Bs(t){return Es.get(t.codePoints[0])}var Ls=function t(e,n,r){p(this,t),this.category=e,this.syllableType=n,this.syllable=r};function Rs(t,e){var n=0,r=Os.match(e.map(Bs)),i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a,l=s[0],u=s[1],c=s[2];++n;for(var f=l;f<=u;f++)e[f].shaperInfo=new Ls(As[Bs(e[f])],c[0],n);for(var h="R"===e[l].shaperInfo.category?1:Math.min(3,u-l),d=l;d<l+h;d++)e[d].features.rphf=!0}}function Ms(t,e){var n=e,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}o.substituted=!1}}function Fs(t,e){var n=e,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;a.substituted&&a.features.rphf&&(a.shaperInfo.category="R")}}function Ds(t,e){var n=e,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;a.substituted&&(a.shaperInfo.category="VPre")}}function zs(t,e){for(var n=t.glyphForCodePoint(9676).id,r=0,i=Ns(e,0);r<e.length;i=Ns(e,r=i)){var o=void 0,a=void 0,s=e[r].shaperInfo,l=s.syllableType;if("virama_terminated_cluster"===l||"standard_cluster"===l||"broken_cluster"===l){if("broken_cluster"===l&&n){var u=new Jo(t,n,[9676]);for(u.shaperInfo=s,o=r;o<i&&"R"===e[o].shaperInfo.category;o++);e.splice(++o,0,u),i++}if("R"===s.category&&1<i-r)for(o=r+1;o<i;o++)if(Ws(s=e[o].shaperInfo)||Us(e[o])){Us(e[o])&&o--,e.splice.apply(e,[r,0].concat(e.splice(r+1,o-r),[e[o]]));break}for(o=r,a=i;o<i;o++)Ws(s=e[o].shaperInfo)||Us(e[o])?a=Us(e[o])?o+1:o:("VPre"===s.category||"VMPre"===s.category)&&a<o&&e.splice.apply(e,[a,1,e[o]].concat(e.splice(a,o-a)))}}}function Ns(t,e){if(e>=t.length)return e;for(var n=t[e].shaperInfo.syllable;++e<t.length&&t[e].shaperInfo.syllable===n;);return e}function Us(t){return"H"===t.shaperInfo.category&&!t.isLigated}function Ws(t){return"B"===t.category||"GB"===t.category}var js={arab:Wo,mong:Wo,syrc:Wo,"nko ":Wo,phag:Wo,mand:Wo,mani:Wo,phlp:Wo,hang:ta,bng2:cs,beng:cs,dev2:cs,deva:cs,gjr2:cs,gujr:cs,guru:cs,gur2:cs,knda:cs,knd2:cs,mlm2:cs,mlym:cs,ory2:cs,orya:cs,taml:cs,tml2:cs,telu:cs,tel2:cs,khmr:cs,bali:Ts,batk:Ts,brah:Ts,bugi:Ts,buhd:Ts,cakm:Ts,cham:Ts,dupl:Ts,egyp:Ts,gran:Ts,hano:Ts,java:Ts,kthi:Ts,kali:Ts,khar:Ts,khoj:Ts,sind:Ts,lepc:Ts,limb:Ts,mahj:Ts,mtei:Ts,modi:Ts,hmng:Ts,rjng:Ts,saur:Ts,shrd:Ts,sidd:Ts,sinh:Ts,sund:Ts,sylo:Ts,tglg:Ts,tagb:Ts,tale:Ts,lana:Ts,tavt:Ts,takr:Ts,tibt:Ts,tfng:Ts,tirh:Ts,latn:Eo,DFLT:Eo};var Gs,Hs=(a(Vs,Gs=Zo),Vs.prototype.applyLookup=function(t,e){var r=this;switch(t){case 1:var n=this.coverageIndex(e.coverage);if(-1===n)return!1;var i=this.glyphIterator.cur;switch(e.version){case 1:i.id=i.id+e.deltaGlyphID&65535;break;case 2:i.id=e.substitute.get(n)}return!0;case 2:var o=this.coverageIndex(e.coverage);if(-1===o)return!1;var a,s=e.sequences.get(o);if(0===s.length)return this.glyphs.splice(this.glyphIterator.index,1),!0;this.glyphIterator.cur.id=s[0],this.glyphIterator.cur.ligatureComponent=0;var l=this.glyphIterator.cur.features,u=this.glyphIterator.cur,c=s.slice(1).map(function(t,e){var n=new Jo(r.font,t,void 0,l);return n.shaperInfo=u.shaperInfo,n.isLigated=u.isLigated,n.ligatureComponent=e+1,n.substituted=!0,n.isMultiplied=!0,n});return(a=this.glyphs).splice.apply(a,[this.glyphIterator.index+1,0].concat(c)),!0;case 3:var f=this.coverageIndex(e.coverage);return-1!==f&&(this.glyphIterator.cur.id=e.alternateSet.get(f)[0],!0);case 4:var h=this.coverageIndex(e.coverage);if(-1===h)return!1;var d=e.ligatureSets.get(h),p=Array.isArray(d),g=0;for(d=p?d:H(d);;){var v;if(p){if(g>=d.length)break;v=d[g++]}else{if((g=d.next()).done)break;v=g.value}var y=v,b=this.sequenceMatchIndices(1,y.components);if(b){var m=this.glyphIterator.cur,w=m.codePoints.slice(),x=b,_=Array.isArray(x),k=0;for(x=_?x:H(x);;){var S;if(_){if(k>=x.length)break;S=x[k++]}else{if((k=x.next()).done)break;S=k.value}var C=S;w.push.apply(w,this.glyphs[C].codePoints)}var A=new Jo(this.font,y.glyph,w,m.features);A.shaperInfo=m.shaperInfo,A.isLigated=!0,A.substituted=!0;for(var P=m.isMark,E=0;E<b.length&&P;E++)P=this.glyphs[b[E]].isMark;A.ligatureID=P?null:this.ligatureID++;var O=m.ligatureID,T=m.codePoints.length,I=T,B=this.glyphIterator.index+1,L=b,R=Array.isArray(L),M=0;for(L=R?L:H(L);;){var F;if(R){if(M>=L.length)break;F=L[M++]}else{if((M=L.next()).done)break;F=M.value}var D=F;if(P)B=D;else for(;B<D;){var z=I-T+Math.min(this.glyphs[B].ligatureComponent||1,T);this.glyphs[B].ligatureID=A.ligatureID,this.glyphs[B].ligatureComponent=z,B++}O=this.glyphs[B].ligatureID,I+=T=this.glyphs[B].codePoints.length,B++}if(O&&!P)for(var N=B;N<this.glyphs.length&&this.glyphs[N].ligatureID===O;N++)z=I-T+Math.min(this.glyphs[N].ligatureComponent||1,T),this.glyphs[N].ligatureComponent=z;for(var U=b.length-1;0<=U;U--)this.glyphs.splice(b[U],1);return this.glyphs[this.glyphIterator.index]=A,!0}}return!1;case 5:return this.applyContext(e);case 6:return this.applyChainingContext(e);case 7:return this.applyLookup(e.lookupType,e.extension);default:throw new Error("GSUB lookupType "+t+" is not supported")}},Vs);function Vs(){return p(this,Vs),o(this,Gs.apply(this,arguments))}var qs,Zs=(a(Ys,qs=Zo),Ys.prototype.applyPositionValue=function(t,e){var n=this.positions[this.glyphIterator.peekIndex(t)];null!=e.xAdvance&&(n.xAdvance+=e.xAdvance),null!=e.yAdvance&&(n.yAdvance+=e.yAdvance),null!=e.xPlacement&&(n.xOffset+=e.xPlacement),null!=e.yPlacement&&(n.yOffset+=e.yPlacement);var r=this.font._variationProcessor,i=this.font.GDEF&&this.font.GDEF.itemVariationStore;r&&i&&(e.xPlaDevice&&(n.xOffset+=r.getDelta(i,e.xPlaDevice.a,e.xPlaDevice.b)),e.yPlaDevice&&(n.yOffset+=r.getDelta(i,e.yPlaDevice.a,e.yPlaDevice.b)),e.xAdvDevice&&(n.xAdvance+=r.getDelta(i,e.xAdvDevice.a,e.xAdvDevice.b)),e.yAdvDevice&&(n.yAdvance+=r.getDelta(i,e.yAdvDevice.a,e.yAdvDevice.b)))},Ys.prototype.applyLookup=function(t,e){switch(t){case 1:var n=this.coverageIndex(e.coverage);if(-1===n)return!1;switch(e.version){case 1:this.applyPositionValue(0,e.value);break;case 2:this.applyPositionValue(0,e.values.get(n))}return!0;case 2:var r=this.glyphIterator.peek();if(!r)return!1;var i=this.coverageIndex(e.coverage);if(-1===i)return!1;switch(e.version){case 1:var o=e.pairSets.get(i),a=Array.isArray(o),s=0;for(o=a?o:H(o);;){var l;if(a){if(s>=o.length)break;l=o[s++]}else{if((s=o.next()).done)break;l=s.value}var u=l;if(u.secondGlyph===r.id)return this.applyPositionValue(0,u.value1),this.applyPositionValue(1,u.value2),!0}return!1;case 2:var c=this.getClassID(this.glyphIterator.cur.id,e.classDef1),f=this.getClassID(r.id,e.classDef2);if(-1===c||-1===f)return!1;var h=e.classRecords.get(c).get(f);return this.applyPositionValue(0,h.value1),this.applyPositionValue(1,h.value2),!0}case 3:var d=this.glyphIterator.peekIndex(),p=this.glyphs[d];if(!p)return!1;var g=e.entryExitRecords[this.coverageIndex(e.coverage)];if(!g||!g.exitAnchor)return!1;var v=e.entryExitRecords[this.coverageIndex(e.coverage,p.id)];if(!v||!v.entryAnchor)return!1;var y=this.getAnchor(v.entryAnchor),b=this.getAnchor(g.exitAnchor),m=this.positions[this.glyphIterator.index],w=this.positions[d];switch(this.direction){case"ltr":m.xAdvance=b.x+m.xOffset;var x=y.x+w.xOffset;w.xAdvance-=x,w.xOffset-=x;break;case"rtl":x=b.x+m.xOffset,m.xAdvance-=x,m.xOffset-=x,w.xAdvance=y.x+w.xOffset}return this.glyphIterator.flags.rightToLeft?(this.glyphIterator.cur.cursiveAttachment=d,m.yOffset=y.y-b.y):(p.cursiveAttachment=this.glyphIterator.index,m.yOffset=b.y-y.y),!0;case 4:var _=this.coverageIndex(e.markCoverage);if(-1===_)return!1;for(var k=this.glyphIterator.index;0<=--k&&(this.glyphs[k].isMark||0<this.glyphs[k].ligatureComponent););if(k<0)return!1;var S=this.coverageIndex(e.baseCoverage,this.glyphs[k].id);if(-1===S)return!1;var C=e.markArray[_],A=e.baseArray[S][C.class];return this.applyAnchor(C,A,k),!0;case 5:var P=this.coverageIndex(e.markCoverage);if(-1===P)return!1;for(var E=this.glyphIterator.index;0<=--E&&this.glyphs[E].isMark;);if(E<0)return!1;var O=this.coverageIndex(e.ligatureCoverage,this.glyphs[E].id);if(-1===O)return!1;var T=e.ligatureArray[O],I=this.glyphIterator.cur,B=this.glyphs[E],L=B.ligatureID&&B.ligatureID===I.ligatureID&&0<I.ligatureComponent?Math.min(I.ligatureComponent,B.codePoints.length)-1:B.codePoints.length-1,R=e.markArray[P],M=T[L][R.class];return this.applyAnchor(R,M,E),!0;case 6:var F=this.coverageIndex(e.mark1Coverage);if(-1===F)return!1;var D=this.glyphIterator.peekIndex(-1),z=this.glyphs[D];if(!z||!z.isMark)return!1;var N=this.glyphIterator.cur,U=!1;if(N.ligatureID===z.ligatureID?N.ligatureID?N.ligatureComponent===z.ligatureComponent&&(U=!0):U=!0:(N.ligatureID&&!N.ligatureComponent||z.ligatureID&&!z.ligatureComponent)&&(U=!0),!U)return!1;var W=this.coverageIndex(e.mark2Coverage,z.id);if(-1===W)return!1;var j=e.mark1Array[F],G=e.mark2Array[W][j.class];return this.applyAnchor(j,G,D),!0;case 7:return this.applyContext(e);case 8:return this.applyChainingContext(e);case 9:return this.applyLookup(e.lookupType,e.extension);default:throw new Error("Unsupported GPOS table: "+t)}},Ys.prototype.applyAnchor=function(t,e,n){var r=this.getAnchor(e),i=this.getAnchor(t.markAnchor),o=(this.positions[n],this.positions[this.glyphIterator.index]);o.xOffset=r.x-i.x,o.yOffset=r.y-i.y,this.glyphIterator.cur.markAttachment=n},Ys.prototype.getAnchor=function(t){var e=t.xCoordinate,n=t.yCoordinate,r=this.font._variationProcessor,i=this.font.GDEF&&this.font.GDEF.itemVariationStore;return r&&i&&(t.xDeviceTable&&(e+=r.getDelta(i,t.xDeviceTable.a,t.xDeviceTable.b)),t.yDeviceTable&&(n+=r.getDelta(i,t.yDeviceTable.a,t.yDeviceTable.b))),{x:e,y:n}},Ys.prototype.applyFeatures=function(t,e,n){qs.prototype.applyFeatures.call(this,t,e,n);for(var r=0;r<this.glyphs.length;r++)this.fixCursiveAttachment(r);this.fixMarkAttachment()},Ys.prototype.fixCursiveAttachment=function(t){var e=this.glyphs[t];if(null!=e.cursiveAttachment){var n=e.cursiveAttachment;e.cursiveAttachment=null,this.fixCursiveAttachment(n),this.positions[t].yOffset+=this.positions[n].yOffset}},Ys.prototype.fixMarkAttachment=function(){for(var t=0;t<this.glyphs.length;t++){var e=this.glyphs[t];if(null!=e.markAttachment){var n=e.markAttachment;if(this.positions[t].xOffset+=this.positions[n].xOffset,this.positions[t].yOffset+=this.positions[n].yOffset,"ltr"===this.direction)for(var r=n;r<t;r++)this.positions[t].xOffset-=this.positions[r].xAdvance,this.positions[t].yOffset-=this.positions[r].yAdvance;else for(var i=n+1;i<t+1;i++)this.positions[t].xOffset+=this.positions[i].xAdvance,this.positions[t].yOffset+=this.positions[i].yAdvance}}},Ys);function Ys(){return p(this,Ys),o(this,qs.apply(this,arguments))}var Xs=(Ks.prototype.setup=function(t){var e=this;this.glyphInfos=t.glyphs.map(function(t){return new Jo(e.font,t.id,[].concat(t.codePoints))});var n=null;for(var r in this.GPOSProcessor&&(n=this.GPOSProcessor.selectScript(t.script,t.language,t.direction)),this.GSUBProcessor&&(n=this.GSUBProcessor.selectScript(t.script,t.language,t.direction)),this.shaper=function(t){Array.isArray(t)||(t=[t]);var e=t,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=js[i];if(o)return o}return Eo}(n),this.plan=new xo(this.font,n,t.direction),this.shaper.plan(this.plan,this.glyphInfos,t.features),this.plan.allFeatures)t.features[r]=!0},Ks.prototype.substitute=function(t){var e=this;this.GSUBProcessor&&(this.plan.process(this.GSUBProcessor,this.glyphInfos),t.glyphs=this.glyphInfos.map(function(t){return e.font.getGlyph(t.id,t.codePoints)}))},Ks.prototype.position=function(t){return"BEFORE_GPOS"===this.shaper.zeroMarkWidths&&this.zeroMarkAdvances(t.positions),this.GPOSProcessor&&this.plan.process(this.GPOSProcessor,this.glyphInfos,t.positions),"AFTER_GPOS"===this.shaper.zeroMarkWidths&&this.zeroMarkAdvances(t.positions),"rtl"===t.direction&&(t.glyphs.reverse(),t.positions.reverse()),this.GPOSProcessor&&this.GPOSProcessor.features},Ks.prototype.zeroMarkAdvances=function(t){for(var e=0;e<this.glyphInfos.length;e++)this.glyphInfos[e].isMark&&(t[e].xAdvance=0,t[e].yAdvance=0)},Ks.prototype.cleanup=function(){this.glyphInfos=null,this.plan=null,this.shaper=null},Ks.prototype.getAvailableFeatures=function(t,e){var n=[];return this.GSUBProcessor&&(this.GSUBProcessor.selectScript(t,e),n.push.apply(n,f(this.GSUBProcessor.features))),this.GPOSProcessor&&(this.GPOSProcessor.selectScript(t,e),n.push.apply(n,f(this.GPOSProcessor.features))),n},Ks);function Ks(t){p(this,Ks),this.font=t,this.glyphInfos=null,this.plan=null,this.GSUBProcessor=null,this.GPOSProcessor=null,this.fallbackPosition=!0,t.GSUB&&(this.GSUBProcessor=new Hs(t,t.GSUB)),t.GPOS&&(this.GPOSProcessor=new Zs(t,t.GPOS))}var Js=(Qs.prototype.layout=function(t,e,n,r,i){if("string"==typeof e&&(i=r,r=n,n=e,e=[]),"string"==typeof t){null==n&&(n=function(t){for(var e=t.length,n=0;n<e;){var r=t.charCodeAt(n++);if(55296<=r&&r<=56319&&n<e){var i=t.charCodeAt(n);56320<=i&&i<=57343&&(n++,r=((1023&r)<<10)+(1023&i)+65536)}var o=P.getScript(r);if("Common"!==o&&"Inherited"!==o&&"Unknown"!==o)return Ei[o]}return Ei.Unknown}(t));var o=this.font.glyphsForString(t)}else{if(null==n){var a=[],s=t,l=Array.isArray(s),u=0;for(s=l?s:H(s);;){var c;if(l){if(u>=s.length)break;c=s[u++]}else{if((u=s.next()).done)break;c=u.value}var f=c;a.push.apply(a,f.codePoints)}n=function(t){for(var e=0;e<t.length;e++){var n=t[e],r=P.getScript(n);if("Common"!==r&&"Inherited"!==r&&"Unknown"!==r)return Ei[r]}return Ei.Unknown}(a)}o=t}var h=new zi(o,e,n,r,i);return 0===o.length?h.positions=[]:(this.engine&&this.engine.setup&&this.engine.setup(h),this.substitute(h),this.position(h),this.hideDefaultIgnorables(h.glyphs,h.positions),this.engine&&this.engine.cleanup&&this.engine.cleanup()),h},Qs.prototype.substitute=function(t){this.engine&&this.engine.substitute&&this.engine.substitute(t)},Qs.prototype.position=function(t){t.positions=t.glyphs.map(function(t){return new Ui(t.advanceWidth)});var e=null;this.engine&&this.engine.position&&(e=this.engine.position(t)),e||this.engine&&!this.engine.fallbackPosition||(this.unicodeLayoutEngine||(this.unicodeLayoutEngine=new Si(this.font)),this.unicodeLayoutEngine.positionGlyphs(t.glyphs,t.positions)),e&&e.kern||!1===t.features.kern||!this.font.kern||(this.kernProcessor||(this.kernProcessor=new _i(this.font)),this.kernProcessor.process(t.glyphs,t.positions),t.features.kern=!0)},Qs.prototype.hideDefaultIgnorables=function(t,e){for(var n=this.font.glyphForCodePoint(32),r=0;r<t.length;r++)this.isDefaultIgnorable(t[r].codePoints[0])&&(t[r]=n,e[r].xAdvance=0,e[r].yAdvance=0)},Qs.prototype.isDefaultIgnorable=function(t){var e=t>>16;if(0==e)switch(t>>8){case 0:return 173===t;case 3:return 847===t;case 6:return 1564===t;case 23:return 6068<=t&&t<=6069;case 24:return 6155<=t&&t<=6158;case 32:return 8203<=t&&t<=8207||8234<=t&&t<=8238||8288<=t&&t<=8303;case 254:return 65024<=t&&t<=65039||65279===t;case 255:return 65520<=t&&t<=65528;default:return!1}else switch(e){case 1:return 113824<=t&&t<=113827||119155<=t&&t<=119162;case 14:return 917504<=t&&t<=921599;default:return!1}},Qs.prototype.getAvailableFeatures=function(t,e){var n=[];return this.engine&&n.push.apply(n,this.engine.getAvailableFeatures(t,e)),this.font.kern&&-1===n.indexOf("kern")&&n.push("kern"),n},Qs.prototype.stringsForGlyph=function(t){var e=new w,n=this.font._cmapProcessor.codePointsForGlyph(t),r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o;e.add(b(a))}if(this.engine&&this.engine.stringsForGlyph){var s=this.engine.stringsForGlyph(t),l=Array.isArray(s),u=0;for(s=l?s:H(s);;){var c;if(l){if(u>=s.length)break;c=s[u++]}else{if((u=s.next()).done)break;c=u.value}var f=c;e.add(f)}}return m(e)},Qs);function Qs(t){p(this,Qs),this.font=t,this.unicodeLayoutEngine=null,this.kernProcessor=null,this.font.morx?this.engine=new yo(this.font):(this.font.GSUB||this.font.GPOS)&&(this.engine=new Xs(this.font))}var $s={moveTo:"M",lineTo:"L",quadraticCurveTo:"Q",bezierCurveTo:"C",closePath:"Z"},tl=(el.prototype.toFunction=function(){var t=this.commands.map(function(t){return"  ctx."+t.command+"("+t.args.join(", ")+");"});return new Function("ctx",t.join("\n"))},el.prototype.toSVG=function(){return this.commands.map(function(t){var e=t.args.map(function(t){return Math.round(100*t)/100});return""+$s[t.command]+e.join(" ")}).join("")},el.prototype.mapPoints=function(t){var e=new el,n=this.commands,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}for(var a=o,s=[],l=0;l<a.args.length;l+=2){var u=t(a.args[l],a.args[l+1]),c=u[0],f=u[1];s.push(c,f)}e[a.command].apply(e,s)}return e},el.prototype.transform=function(n,r,i,o,a,s){return this.mapPoints(function(t,e){return[t=n*t+i*e+a,e=r*t+o*e+s]})},el.prototype.translate=function(t,e){return this.transform(1,0,0,1,t,e)},el.prototype.rotate=function(t){var e=Math.cos(t),n=Math.sin(t);return this.transform(e,n,-n,e,0,0)},el.prototype.scale=function(t,e){var n=1<arguments.length&&void 0!==e?e:t;return this.transform(t,0,0,n,0,0)},r(el,[{key:"cbox",get:function(){if(!this._cbox){var t=new Ai,e=this.commands,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}for(var o=i,a=0;a<o.args.length;a+=2)t.addPoint(o.args[a],o.args[a+1])}this._cbox=L(t)}return this._cbox}},{key:"bbox",get:function(){if(this._bbox)return this._bbox;var t=new Ai,e=0,n=0,r=function(t){return Math.pow(1-t,3)*_[A]+3*Math.pow(1-t,2)*t*k[A]+3*(1-t)*Math.pow(t,2)*S[A]+Math.pow(t,3)*C[A]},i=this.commands,o=Array.isArray(i),a=0;for(i=o?i:H(i);;){var s;if(o){if(a>=i.length)break;s=i[a++]}else{if((a=i.next()).done)break;s=a.value}var l=s;switch(l.command){case"moveTo":case"lineTo":var u=l.args,c=u[0],f=u[1];t.addPoint(c,f),e=c,n=f;break;case"quadraticCurveTo":case"bezierCurveTo":if("quadraticCurveTo"===l.command)var h=l.args,d=h[0],p=h[1],g=e+2/3*(d-e),v=n+2/3*(p-n),y=(w=h[2])+2/3*(d-w),b=(x=h[3])+2/3*(p-x);else var m=l.args,w=(g=m[0],v=m[1],y=m[2],b=m[3],m[4]),x=m[5];t.addPoint(w,x);for(var _=[e,n],k=[g,v],S=[y,b],C=[w,x],A=0;A<=1;A++){var P=6*_[A]-12*k[A]+6*S[A],E=-3*_[A]+9*k[A]-9*S[A]+3*C[A];if(l=3*k[A]-3*_[A],0!=E){var O=Math.pow(P,2)-4*l*E;if(!(O<0)){var T=(-P+Math.sqrt(O))/(2*E);0<T&&T<1&&(0===A?t.addPoint(r(T),t.maxY):1===A&&t.addPoint(t.maxX,r(T)));var I=(-P-Math.sqrt(O))/(2*E);0<I&&I<1&&(0===A?t.addPoint(r(I),t.maxY):1===A&&t.addPoint(t.maxX,r(I)))}}else{if(0==P)continue;var B=-l/P;0<B&&B<1&&(0===A?t.addPoint(r(B),t.maxY):1===A&&t.addPoint(t.maxX,r(B)))}}e=w,n=x}}return this._bbox=L(t)}}]),el);function el(){p(this,el),this.commands=[],this._bbox=null,this._cbox=null}for(var nl=["moveTo","lineTo","quadraticCurveTo","bezierCurveTo","closePath"],rl=function(){var r=nl[il];tl.prototype[r]=function(){for(var t=arguments.length,e=Array(t),n=0;n<t;n++)e[n]=arguments[n];return this._bbox=this._cbox=null,this.commands.push({command:r,args:e}),this}},il=0;il<nl.length;il++)rl();var ol,al=[".notdef",".null","nonmarkingreturn","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quotesingle","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","grave","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","Adieresis","Aring","Ccedilla","Eacute","Ntilde","Odieresis","Udieresis","aacute","agrave","acircumflex","adieresis","atilde","aring","ccedilla","eacute","egrave","ecircumflex","edieresis","iacute","igrave","icircumflex","idieresis","ntilde","oacute","ograve","ocircumflex","odieresis","otilde","uacute","ugrave","ucircumflex","udieresis","dagger","degree","cent","sterling","section","bullet","paragraph","germandbls","registered","copyright","trademark","acute","dieresis","notequal","AE","Oslash","infinity","plusminus","lessequal","greaterequal","yen","mu","partialdiff","summation","product","pi","integral","ordfeminine","ordmasculine","Omega","ae","oslash","questiondown","exclamdown","logicalnot","radical","florin","approxequal","Delta","guillemotleft","guillemotright","ellipsis","nonbreakingspace","Agrave","Atilde","Otilde","OE","oe","endash","emdash","quotedblleft","quotedblright","quoteleft","quoteright","divide","lozenge","ydieresis","Ydieresis","fraction","currency","guilsinglleft","guilsinglright","fi","fl","daggerdbl","periodcentered","quotesinglbase","quotedblbase","perthousand","Acircumflex","Ecircumflex","Aacute","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Oacute","Ocircumflex","apple","Ograve","Uacute","Ucircumflex","Ugrave","dotlessi","circumflex","tilde","macron","breve","dotaccent","ring","cedilla","hungarumlaut","ogonek","caron","Lslash","lslash","Scaron","scaron","Zcaron","zcaron","brokenbar","Eth","eth","Yacute","yacute","Thorn","thorn","minus","multiply","onesuperior","twosuperior","threesuperior","onehalf","onequarter","threequarters","franc","Gbreve","gbreve","Idotaccent","Scedilla","scedilla","Cacute","cacute","Ccaron","ccaron","dcroat"];function sl(n,r,t,e,i){var o={};return Object.keys(e).forEach(function(t){o[t]=e[t]}),o.enumerable=!!o.enumerable,o.configurable=!!o.configurable,("value"in o||o.initializer)&&(o.writable=!0),o=t.slice().reverse().reduce(function(t,e){return e(n,r,t)||t},o),i&&void 0!==o.initializer&&(o.value=o.initializer?o.initializer.call(i):void 0,o.initializer=void 0),void 0===o.initializer&&(Object.defineProperty(n,r,o),o=null),o}var ll=(ul.prototype._getPath=function(){return new tl},ul.prototype._getCBox=function(){return this.path.cbox},ul.prototype._getBBox=function(){return this.path.bbox},ul.prototype._getTableMetrics=function(t){if(this.id<t.metrics.length)return t.metrics.get(this.id);var e=t.metrics.get(t.metrics.length-1);return{advance:e?e.advance:0,bearing:t.bearings.get(this.id-t.metrics.length)||0}},ul.prototype._getMetrics=function(t){if(this._metrics)return this._metrics;var e,n=this._getTableMetrics(this._font.hmtx),r=n.advance,i=n.bearing;if(this._font.vmtx)var o=this._getTableMetrics(this._font.vmtx),a=o.advance,s=o.bearing;else if(null==t&&(t=this.cbox),(e=this._font["OS/2"])&&0<e.version)a=Math.abs(e.typoAscender-e.typoDescender),s=e.typoAscender-t.maxY;else{var l=this._font.hhea;a=Math.abs(l.ascent-l.descent),s=l.ascent-t.maxY}return this._font._variationProcessor&&this._font.HVAR&&(r+=this._font._variationProcessor.getAdvanceAdjustment(this.id,this._font.HVAR)),this._metrics={advanceWidth:r,advanceHeight:a,leftBearing:i,topBearing:s}},ul.prototype.getScaledPath=function(t){var e=1/this._font.unitsPerEm*t;return this.path.scale(e)},ul.prototype._getName=function(){var t=this._font.post;if(!t)return null;switch(t.version){case 1:return al[this.id];case 2:var e=t.glyphNameIndex[this.id];return e<al.length?al[e]:t.names[e-al.length];case 2.5:return al[this.id+t.offsets[this.id]];case 4:return String.fromCharCode(t.map[this.id])}},ul.prototype.render=function(t,e){t.save();var n=1/this._font.head.unitsPerEm*e;t.scale(n,n),this.path.toFunction()(t),t.fill(),t.restore()},r(ul,[{key:"cbox",get:function(){return this._getCBox()}},{key:"bbox",get:function(){return this._getBBox()}},{key:"path",get:function(){return this._getPath()}},{key:"advanceWidth",get:function(){return this._getMetrics().advanceWidth}},{key:"advanceHeight",get:function(){return this._getMetrics().advanceHeight}},{key:"ligatureCaretPositions",get:function(){}},{key:"name",get:function(){return this._getName()}}]),sl((ol=ul).prototype,"cbox",[O],e(ol.prototype,"cbox"),ol.prototype),sl(ol.prototype,"bbox",[O],e(ol.prototype,"bbox"),ol.prototype),sl(ol.prototype,"path",[O],e(ol.prototype,"path"),ol.prototype),sl(ol.prototype,"advanceWidth",[O],e(ol.prototype,"advanceWidth"),ol.prototype),sl(ol.prototype,"advanceHeight",[O],e(ol.prototype,"advanceHeight"),ol.prototype),sl(ol.prototype,"name",[O],e(ol.prototype,"name"),ol.prototype),ol);function ul(t,e,n){p(this,ul),this.id=t,this.codePoints=e,this._font=n,this.isMark=0<this.codePoints.length&&this.codePoints.every(P.isMark),this.isLigature=1<this.codePoints.length}var cl=new A.Struct({numberOfContours:A.int16,xMin:A.int16,yMin:A.int16,xMax:A.int16,yMax:A.int16}),fl=(hl.prototype.copy=function(){return new hl(this.onCurve,this.endContour,this.x,this.y)},hl);function hl(t,e){var n=2<arguments.length&&void 0!==arguments[2]?arguments[2]:0,r=3<arguments.length&&void 0!==arguments[3]?arguments[3]:0;p(this,hl),this.onCurve=t,this.endContour=e,this.x=n,this.y=r}function dl(t,e,n){p(this,dl),this.glyphID=t,this.dx=e,this.dy=n,this.pos=0,this.scaleX=this.scaleY=1,this.scale01=this.scale10=0}var pl,gl=(a(vl,pl=ll),vl.prototype._getCBox=function(t){if(this._font._variationProcessor&&!t)return this.path.cbox;var e=this._font._getTableStream("glyf");e.pos+=this._font.loca.offsets[this.id];var n=cl.decode(e),r=new Ai(n.xMin,n.yMin,n.xMax,n.yMax);return L(r)},vl.prototype._parseGlyphCoord=function(t,e,n,r){if(n){var i=t.readUInt8();r||(i=-i),i+=e}else i=r?e:e+t.readInt16BE();return i},vl.prototype._decode=function(){var t=this._font.loca.offsets[this.id];if(t===this._font.loca.offsets[this.id+1])return null;var e=this._font._getTableStream("glyf");e.pos+=t;var n=e.pos,r=cl.decode(e);return 0<r.numberOfContours?this._decodeSimple(r,e):r.numberOfContours<0&&this._decodeComposite(r,e,n),r},vl.prototype._decodeSimple=function(t,e){t.points=[];var n=new A.Array(A.uint16,t.numberOfContours).decode(e);t.instructions=new A.Array(A.uint8,A.uint16).decode(e);for(var r=[],i=n[n.length-1]+1;r.length<i;){var o=e.readUInt8();if(r.push(o),8&o)for(var a=e.readUInt8(),s=0;s<a;s++)r.push(o)}for(var l=0;l<r.length;l++){o=r[l];var u=new fl(!!(1&o),0<=n.indexOf(l),0,0);t.points.push(u)}var c=0;for(l=0;l<r.length;l++)o=r[l],t.points[l].x=c=this._parseGlyphCoord(e,c,2&o,16&o);var f=0;for(l=0;l<r.length;l++)o=r[l],t.points[l].y=f=this._parseGlyphCoord(e,f,4&o,32&o);if(this._font._variationProcessor){var h=t.points.slice();h.push.apply(h,this._getPhantomPoints(t)),this._font._variationProcessor.transformPoints(this.id,h),t.phantomPoints=h.slice(-4)}},vl.prototype._decodeComposite=function(t,e,n){for(var r=2<arguments.length&&void 0!==n?n:0,i=!(t.components=[]),o=32;32&o;){o=e.readUInt16BE();var a=e.pos-r,s=e.readUInt16BE();if(i=i||0!=(256&o),1&o)var l=e.readInt16BE(),u=e.readInt16BE();else l=e.readInt8(),u=e.readInt8();(h=new dl(s,l,u)).pos=a,8&o?h.scaleX=h.scaleY=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824:64&o?(h.scaleX=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824,h.scaleY=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824):128&o&&(h.scaleX=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824,h.scale01=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824,h.scale10=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824,h.scaleY=(e.readUInt8()<<24|e.readUInt8()<<16)/1073741824),t.components.push(h)}if(this._font._variationProcessor){for(var c=[],f=0;f<t.components.length;f++){var h=t.components[f];c.push(new fl(!0,!0,h.dx,h.dy))}c.push.apply(c,this._getPhantomPoints(t)),this._font._variationProcessor.transformPoints(this.id,c),t.phantomPoints=c.splice(-4,4);for(var d=0;d<c.length;d++){var p=c[d];t.components[d].dx=p.x,t.components[d].dy=p.y}}return i},vl.prototype._getPhantomPoints=function(t){var e=this._getCBox(!0);null==this._metrics&&(this._metrics=ll.prototype._getMetrics.call(this,e));var n=this._metrics,r=n.advanceWidth,i=n.advanceHeight,o=n.leftBearing,a=n.topBearing;return[new fl(!1,!0,t.xMin-o,0),new fl(!1,!0,t.xMin-o+r,0),new fl(!1,!0,0,t.yMax+a),new fl(!1,!0,0,t.yMax+a+i)]},vl.prototype._getContours=function(){var t=this._decode();if(!t)return[];var e=[];if(t.numberOfContours<0){var n=t.components,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}for(var a=o,s=this._font.getGlyph(a.glyphID)._getContours(),l=0;l<s.length;l++)for(var u=s[l],c=0;c<u.length;c++){var f=u[c],h=f.x*a.scaleX+f.y*a.scale01+a.dx,d=f.y*a.scaleY+f.x*a.scale10+a.dy;e.push(new fl(f.onCurve,f.endContour,h,d))}}}else e=t.points||[];t.phantomPoints&&!this._font.directory.tables.HVAR&&(this._metrics.advanceWidth=t.phantomPoints[1].x-t.phantomPoints[0].x,this._metrics.advanceHeight=t.phantomPoints[3].y-t.phantomPoints[2].y,this._metrics.leftBearing=t.xMin-t.phantomPoints[0].x,this._metrics.topBearing=t.phantomPoints[2].y-t.yMax);for(var p=[],g=[],v=0;v<e.length;v++){var y=e[v];g.push(y),y.endContour&&(p.push(g),g=[])}return p},vl.prototype._getMetrics=function(){if(this._metrics)return this._metrics;var t=this._getCBox(!0);return pl.prototype._getMetrics.call(this,t),this._font._variationProcessor&&!this._font.HVAR&&this.path,this._metrics},vl.prototype._getPath=function(){for(var t=this._getContours(),e=new tl,n=0;n<t.length;n++){var r=t[n],i=r[0],o=r[r.length-1],a=0;if(i.onCurve){var s=null;a=1}else s=i=o.onCurve?o:new fl(!1,!1,(i.x+o.x)/2,(i.y+o.y)/2);e.moveTo(i.x,i.y);for(var l=a;l<r.length;l++){var u=r[l],c=0===l?i:r[l-1];if(c.onCurve&&u.onCurve)e.lineTo(u.x,u.y);else if(c.onCurve&&!u.onCurve)s=u;else if(c.onCurve||u.onCurve){if(c.onCurve||!u.onCurve)throw new Error("Unknown TTF path state");e.quadraticCurveTo(s.x,s.y,u.x,u.y),s=null}else{var f=(c.x+u.x)/2,h=(c.y+u.y)/2;e.quadraticCurveTo(c.x,c.y,f,h);var s=u}}s&&e.quadraticCurveTo(s.x,s.y,i.x,i.y),e.closePath()}return e},vl);function vl(){return p(this,vl),o(this,pl.apply(this,arguments))}var yl,bl=(a(ml,yl=ll),ml.prototype._getName=function(){return this._font.CFF2?yl.prototype._getName.call(this):this._font["CFF "].getGlyphName(this.id)},ml.prototype.bias=function(t){return t.length<1240?107:t.length<33900?1131:32768},ml.prototype._getPath=function(){var q=this._font.stream,Z=(q.pos,this._font.CFF2||this._font["CFF "]),t=Z.topDict.CharStrings[this.id],Y=t.offset+t.length;q.pos=t.offset;var X=new tl,K=[],J=[],e=null,Q=0,$=0,tt=0,et=void 0,nt=void 0,rt=!1;this._usedGsubrs=et={},this._usedSubrs=nt={};var it=Z.globalSubrIndex||[],ot=this.bias(it),n=Z.privateDictForGlyph(this.id)||{},at=n.Subrs||[],st=this.bias(at),lt=Z.topDict.vstore&&Z.topDict.vstore.itemVariationStore,ut=n.vsindex,ct=this._font._variationProcessor;function ft(){null==e&&(e=K.shift()+n.nominalWidthX)}function ht(){return K.length%2!=0&&ft(),Q+=K.length>>1,K.length=0}function dt(t,e){rt&&X.closePath(),X.moveTo(t,e),rt=!0}return function t(){for(;q.pos<Y;){var e=q.readUInt8();if(e<32)switch(e){case 1:case 3:case 18:case 23:ht();break;case 4:1<K.length&&ft(),tt+=K.shift(),dt($,tt);break;case 5:for(;2<=K.length;)$+=K.shift(),tt+=K.shift(),X.lineTo($,tt);break;case 6:case 7:for(var n=6===e;1<=K.length;)n?$+=K.shift():tt+=K.shift(),X.lineTo($,tt),n=!n;break;case 8:for(;0<K.length;){var r=$+K.shift(),i=tt+K.shift(),o=r+K.shift(),a=i+K.shift();$=o+K.shift(),tt=a+K.shift(),X.bezierCurveTo(r,i,o,a,$,tt)}break;case 10:var s=K.pop()+st,l=at[s];if(l){nt[s]=!0;var u=q.pos,c=Y;q.pos=l.offset,Y=l.offset+l.length,t(),q.pos=u,Y=c}break;case 11:if(2<=Z.version)break;return;case 14:if(2<=Z.version)break;0<K.length&&ft(),rt&&(X.closePath(),rt=!1);break;case 15:if(Z.version<2)throw new Error("vsindex operator not supported in CFF v1");ut=K.pop();break;case 16:if(Z.version<2)throw new Error("blend operator not supported in CFF v1");if(!ct)throw new Error("blend operator in non-variation font");for(var f=ct.getBlendVector(lt,ut),h=K.pop(),d=h*f.length,p=K.length-d,g=p-h,v=0;v<h;v++){for(var y=K[g+v],b=0;b<f.length;b++)y+=f[b]*K[p++];K[g+v]=y}for(;d--;)K.pop();break;case 19:case 20:ht(),q.pos+=Q+7>>3;break;case 21:2<K.length&&ft(),$+=K.shift(),tt+=K.shift(),dt($,tt);break;case 22:1<K.length&&ft(),dt($+=K.shift(),tt);break;case 24:for(;8<=K.length;)r=$+K.shift(),i=tt+K.shift(),o=r+K.shift(),a=i+K.shift(),$=o+K.shift(),tt=a+K.shift(),X.bezierCurveTo(r,i,o,a,$,tt);$+=K.shift(),tt+=K.shift(),X.lineTo($,tt);break;case 25:for(;8<=K.length;)$+=K.shift(),tt+=K.shift(),X.lineTo($,tt);r=$+K.shift(),i=tt+K.shift(),o=r+K.shift(),a=i+K.shift(),$=o+K.shift(),tt=a+K.shift(),X.bezierCurveTo(r,i,o,a,$,tt);break;case 26:for(K.length%2&&($+=K.shift());4<=K.length;)r=$,i=tt+K.shift(),o=r+K.shift(),a=i+K.shift(),$=o,tt=a+K.shift(),X.bezierCurveTo(r,i,o,a,$,tt);break;case 27:for(K.length%2&&(tt+=K.shift());4<=K.length;)r=$+K.shift(),i=tt,o=r+K.shift(),a=i+K.shift(),$=o+K.shift(),tt=a,X.bezierCurveTo(r,i,o,a,$,tt);break;case 28:K.push(q.readInt16BE());break;case 29:s=K.pop()+ot,(l=it[s])&&(et[s]=!0,u=q.pos,c=Y,q.pos=l.offset,Y=l.offset+l.length,t(),q.pos=u,Y=c);break;case 30:case 31:for(n=31===e;4<=K.length;)n?(r=$+K.shift(),i=tt,o=r+K.shift(),a=i+K.shift(),tt=a+K.shift(),$=o+(1===K.length?K.shift():0)):(r=$,i=tt+K.shift(),o=r+K.shift(),a=i+K.shift(),$=o+K.shift(),tt=a+(1===K.length?K.shift():0)),X.bezierCurveTo(r,i,o,a,$,tt),n=!n;break;case 12:switch(e=q.readUInt8()){case 3:var m=K.pop(),w=K.pop();K.push(m&&w?1:0);break;case 4:m=K.pop(),w=K.pop(),K.push(m||w?1:0);break;case 5:m=K.pop(),K.push(m?0:1);break;case 9:m=K.pop(),K.push(Math.abs(m));break;case 10:m=K.pop(),w=K.pop(),K.push(m+w);break;case 11:m=K.pop(),w=K.pop(),K.push(m-w);break;case 12:m=K.pop(),w=K.pop(),K.push(m/w);break;case 14:m=K.pop(),K.push(-m);break;case 15:m=K.pop(),w=K.pop(),K.push(m===w?1:0);break;case 18:K.pop();break;case 20:var x=K.pop(),_=K.pop();J[_]=x;break;case 21:_=K.pop(),K.push(J[_]||0);break;case 22:var k=K.pop(),S=K.pop(),C=K.pop(),A=K.pop();K.push(C<=A?k:S);break;case 23:K.push(Math.random());break;case 24:m=K.pop(),w=K.pop(),K.push(m*w);break;case 26:m=K.pop(),K.push(Math.sqrt(m));break;case 27:m=K.pop(),K.push(m,m);break;case 28:m=K.pop(),w=K.pop(),K.push(w,m);break;case 29:(_=K.pop())<0?_=0:_>K.length-1&&(_=K.length-1),K.push(K[_]);break;case 30:var P=K.pop(),E=K.pop();if(0<=E)for(;0<E;){for(var O=K[P-1],T=P-2;0<=T;T--)K[T+1]=K[T];K[0]=O,E--}else for(;E<0;){O=K[0];for(var I=0;I<=P;I++)K[I]=K[I+1];K[P-1]=O,E++}break;case 34:r=$+K.shift(),i=tt,o=r+K.shift(),a=i+K.shift();var B=o+K.shift(),L=a,R=B+K.shift(),M=L,F=R+K.shift(),D=M,z=F+K.shift(),N=D;$=z,tt=N,X.bezierCurveTo(r,i,o,a,B,L),X.bezierCurveTo(R,M,F,D,z,N);break;case 35:for(var U=[],W=0;W<=5;W++)$+=K.shift(),tt+=K.shift(),U.push($,tt);X.bezierCurveTo.apply(X,U.slice(0,6)),X.bezierCurveTo.apply(X,U.slice(6)),K.shift();break;case 36:r=$+K.shift(),i=tt+K.shift(),o=r+K.shift(),M=L=a=i+K.shift(),F=(R=(B=o+K.shift())+K.shift())+K.shift(),D=M+K.shift(),z=F+K.shift(),$=z,tt=N=D,X.bezierCurveTo(r,i,o,a,B,L),X.bezierCurveTo(R,M,F,D,z,N);break;case 37:var j=$,G=tt;U=[];for(var H=0;H<=4;H++)$+=K.shift(),tt+=K.shift(),U.push($,tt);Math.abs($-j)>Math.abs(tt-G)?($+=K.shift(),tt=G):($=j,tt+=K.shift()),U.push($,tt),X.bezierCurveTo.apply(X,U.slice(0,6)),X.bezierCurveTo.apply(X,U.slice(6));break;default:throw new Error("Unknown op: 12 "+e)}break;default:throw new Error("Unknown op: "+e)}else if(e<247)K.push(e-139);else if(e<251){var V=q.readUInt8();K.push(256*(e-247)+V+108)}else e<255?(V=q.readUInt8(),K.push(256*-(e-251)-V-108)):K.push(q.readInt32BE()/65536)}}(),rt&&X.closePath(),X},ml);function ml(){return p(this,ml),o(this,yl.apply(this,arguments))}var wl,xl=new A.Struct({originX:A.uint16,originY:A.uint16,type:new A.String(4),data:new A.Buffer(function(t){return t.parent.buflen-t._currentOffset})}),_l=(a(kl,wl=gl),kl.prototype.getImageForSize=function(t){for(var e=0;e<this._font.sbix.imageTables.length;e++){var n=this._font.sbix.imageTables[e];if(n.ppem>=t)break}var r=n.imageOffsets,i=r[this.id],o=r[this.id+1];return i===o?null:(this._font.stream.pos=i,xl.decode(this._font.stream,{buflen:o-i}))},kl.prototype.render=function(t,e){var n=this.getImageForSize(e);if(null!=n){var r=e/this._font.unitsPerEm;t.image(n.data,{height:e,x:n.originX,y:(this.bbox.minY-n.originY)*r})}this._font.sbix.flags.renderOutlines&&wl.prototype.render.call(this,t,e)},kl);function kl(){return p(this,kl),o(this,wl.apply(this,arguments))}function Sl(t,e){p(this,Sl),this.glyph=t,this.color=e}var Cl,Al=(a(Pl,Cl=ll),Pl.prototype._getBBox=function(){for(var t=new Ai,e=0;e<this.layers.length;e++){var n=this.layers[e].glyph.bbox;t.addPoint(n.minX,n.minY),t.addPoint(n.maxX,n.maxY)}return t},Pl.prototype.render=function(t,e){var n=this.layers,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o,s=a.glyph,l=a.color;t.fillColor([l.red,l.green,l.blue],l.alpha/255*100),s.render(t,e)}},r(Pl,[{key:"layers",get:function(){for(var t=this._font.CPAL,e=this._font.COLR,n=0,r=e.baseGlyphRecord.length-1;n<=r;){var i=n+r>>1,o=e.baseGlyphRecord[i];if(this.id<o.gid)r=i-1;else{if(!(this.id>o.gid)){var a=o;break}n=1+i}}if(null==a){var s=this._font._getBaseGlyph(this.id);return[new Sl(s,c={red:0,green:0,blue:0,alpha:255})]}for(var l=[],u=a.firstLayerIndex;u<a.firstLayerIndex+a.numLayers;u++){o=e.layerRecords[u];var c=t.colorRecords[o.paletteIndex];s=this._font._getBaseGlyph(o.gid),l.push(new Sl(s,c))}return l}}]),Pl);function Pl(){return p(this,Pl),o(this,Cl.apply(this,arguments))}var El=(Ol.prototype.normalizeCoords=function(t){for(var e=[],n=0;n<this.font.fvar.axis.length;n++){var r=this.font.fvar.axis[n];t[n]<r.defaultValue?e.push((t[n]-r.defaultValue+x)/(r.defaultValue-r.minValue+x)):e.push((t[n]-r.defaultValue+x)/(r.maxValue-r.defaultValue+x))}if(this.font.avar)for(n=0;n<this.font.avar.segment.length;n++)for(var i=this.font.avar.segment[n],o=0;o<i.correspondence.length;o++){var a=i.correspondence[o];if(1<=o&&e[n]<a.fromCoord){var s=i.correspondence[o-1];e[n]=((e[n]-s.fromCoord)*(a.toCoord-s.toCoord)+x)/(a.fromCoord-s.fromCoord+x)+s.toCoord;break}}return e},Ol.prototype.transformPoints=function(t,e){if(this.font.fvar&&this.font.gvar){var n=this.font.gvar;if(!(t>=n.glyphCount)){var r=n.offsets[t];if(r!==n.offsets[t+1]){var i=this.font.stream;if(i.pos=r,!(i.pos>=i.length)){var o=i.readUInt16BE(),a=r+i.readUInt16BE();if(32768&o){var s=i.pos;i.pos=a;var l=this.decodePoints();a=i.pos,i.pos=s}var u=e.map(function(t){return t.copy()});o&=4095;for(var c=0;c<o;c++){var f=i.readUInt16BE(),h=i.readUInt16BE();if(32768&h)for(var d=[],p=0;p<n.axisCount;p++)d.push(i.readInt16BE()/16384);else{if((4095&h)>=n.globalCoordCount)throw new Error("Invalid gvar table");d=n.globalCoords[4095&h]}if(16384&h){for(var g=[],v=0;v<n.axisCount;v++)g.push(i.readInt16BE()/16384);for(var y=[],b=0;b<n.axisCount;b++)y.push(i.readInt16BE()/16384)}var m=this.tupleFactor(h,d,g,y);if(0!==m){if(s=i.pos,i.pos=a,8192&h)var w=this.decodePoints();else w=l;var x=0===w.length?e.length:w.length,_=this.decodeDeltas(x),k=this.decodeDeltas(x);if(0===w.length)for(var S=0;S<e.length;S++){var C=e[S];C.x+=Math.round(_[S]*m),C.y+=Math.round(k[S]*m)}else{for(var A=u.map(function(t){return t.copy()}),P=e.map(function(){return!1}),E=0;E<w.length;E++){var O=w[E];if(O<e.length){var T=A[O];P[O]=!0,T.x+=Math.round(_[E]*m),T.y+=Math.round(k[E]*m)}}this.interpolateMissingDeltas(A,u,P);for(var I=0;I<e.length;I++){var B=A[I].x-u[I].x,L=A[I].y-u[I].y;e[I].x+=B,e[I].y+=L}}a+=f,i.pos=s}else a+=f}}}}}},Ol.prototype.decodePoints=function(){var t=this.font.stream,e=t.readUInt8();128&e&&(e=(127&e)<<8|t.readUInt8());for(var n=new Uint16Array(e),r=0,i=0;r<e;)for(var o=t.readUInt8(),a=1+(127&o),s=128&o?t.readUInt16:t.readUInt8,l=0;l<a&&r<e;l++)i+=s.call(t),n[r++]=i;return n},Ol.prototype.decodeDeltas=function(t){for(var e=this.font.stream,n=0,r=new Int16Array(t);n<t;){var i=e.readUInt8(),o=1+(63&i);if(128&i)n+=o;else for(var a=64&i?e.readInt16BE:e.readInt8,s=0;s<o&&n<t;s++)r[n++]=a.call(e)}return r},Ol.prototype.tupleFactor=function(t,e,n,r){for(var i=this.normalizedCoords,o=this.font.gvar,a=1,s=0;s<o.axisCount;s++)if(0!==e[s]){if(0===i[s])return 0;if(0==(16384&t)){if(i[s]<Math.min(0,e[s])||i[s]>Math.max(0,e[s]))return 0;a=(a*i[s]+x)/(e[s]+x)}else{if(i[s]<n[s]||i[s]>r[s])return 0;a=i[s]<e[s]?a*(i[s]-n[s]+x)/(e[s]-n[s]+x):a*(r[s]-i[s]+x)/(r[s]-e[s]+x)}}return a},Ol.prototype.interpolateMissingDeltas=function(t,e,n){if(0!==t.length)for(var r=0;r<t.length;){for(var i=r,o=r,a=t[o];!a.endContour;)a=t[++o];for(;r<=o&&!n[r];)r++;if(!(o<r)){var s=r,l=r;for(r++;r<=o;)n[r]&&(this.deltaInterpolate(l+1,r-1,l,r,e,t),l=r),r++;l===s?this.deltaShift(i,o,l,e,t):(this.deltaInterpolate(l+1,o,l,s,e,t),0<s&&this.deltaInterpolate(i,s-1,l,s,e,t)),r=o+1}}},Ol.prototype.deltaInterpolate=function(t,e,n,r,i,o){if(!(e<t))for(var a=["x","y"],s=0;s<a.length;s++){var l=a[s];if(i[n][l]>i[r][l]){var u=n;n=r,r=u}var c=i[n][l],f=i[r][l],h=o[n][l],d=o[r][l];if(c!==f||h===d)for(var p=c===f?0:(d-h)/(f-c),g=t;g<=e;g++){var v=i[g][l];v<=c?v+=h-c:f<=v?v+=d-f:v=h+(v-c)*p,o[g][l]=v}}},Ol.prototype.deltaShift=function(t,e,n,r,i){var o=i[n].x-r[n].x,a=i[n].y-r[n].y;if(0!=o||0!=a)for(var s=t;s<=e;s++)s!==n&&(i[s].x+=o,i[s].y+=a)},Ol.prototype.getAdvanceAdjustment=function(t,e){var n=void 0,r=void 0;if(e.advanceWidthMapping){var i=t;i>=e.advanceWidthMapping.mapCount&&(i=e.advanceWidthMapping.mapCount-1),e.advanceWidthMapping.entryFormat;var o=e.advanceWidthMapping.mapData[i];n=o.outerIndex,r=o.innerIndex}else n=0,r=t;return this.getDelta(e.itemVariationStore,n,r)},Ol.prototype.getDelta=function(t,e,n){if(e>=t.itemVariationData.length)return 0;var r=t.itemVariationData[e];if(n>=r.deltaSets.length)return 0;for(var i=r.deltaSets[n],o=this.getBlendVector(t,e),a=0,s=0;s<r.regionIndexCount;s++)a+=i.deltas[s]*o[s];return a},Ol.prototype.getBlendVector=function(t,e){var n=t.itemVariationData[e];if(this.blendVectors.has(n))return this.blendVectors.get(n);for(var r=this.normalizedCoords,i=[],o=0;o<n.regionIndexCount;o++){for(var a=1,s=n.regionIndexes[o],l=t.variationRegionList.variationRegions[s],u=0;u<l.length;u++){var c=l[u];a*=c.startCoord>c.peakCoord||c.peakCoord>c.endCoord?1:c.startCoord<0&&0<c.endCoord&&0!==c.peakCoord?1:0===c.peakCoord?1:r[u]<c.startCoord||r[u]>c.endCoord?0:r[u]===c.peakCoord?1:r[u]<c.peakCoord?(r[u]-c.startCoord+x)/(c.peakCoord-c.startCoord+x):(c.endCoord-r[u]+x)/(c.endCoord-c.peakCoord+x)}i[o]=a}return this.blendVectors.set(n,i),i},Ol);function Ol(t,e){p(this,Ol),this.font=t,this.normalizedCoords=this.normalizeCoords(e),this.blendVectors=new i}var Tl=(Il.prototype.includeGlyph=function(t){return"object"===(void 0===t?"undefined":c(t))&&(t=t.id),null==this.mapping[t]&&(this.glyphs.push(t),this.mapping[t]=this.glyphs.length-1),this.mapping[t]},Il.prototype.encodeStream=function(){var t=this,e=new A.EncodeStream;return n.nextTick(function(){return t.encode(e),e.end()}),e},Il);function Il(t){p(this,Il),this.font=t,this.glyphs=[],this.mapping={},this.includeGlyph(0)}var Bl=(Ll.size=function(t){return 0<=t&&t<=255?1:2},Ll.encode=function(t,e){0<=e&&e<=255?t.writeUInt8(e):t.writeInt16BE(e)},Ll);function Ll(){p(this,Ll)}var Rl=new A.Struct({numberOfContours:A.int16,xMin:A.int16,yMin:A.int16,xMax:A.int16,yMax:A.int16,endPtsOfContours:new A.Array(A.uint16,"numberOfContours"),instructions:new A.Array(A.uint8,A.uint16),flags:new A.Array(A.uint8,0),xPoints:new A.Array(Bl,0),yPoints:new A.Array(Bl,0)}),Ml=(Fl.prototype.encodeSimple=function(t,e){for(var n=1<arguments.length&&void 0!==e?e:[],r=[],i=[],o=[],a=[],s=0,l=0,u=0,c=0,f=0,h=0;h<t.commands.length;h++){for(var d=t.commands[h],p=0;p<d.args.length;p+=2){var g=d.args[p],v=d.args[p+1],y=0;if("quadraticCurveTo"===d.command&&2===p){var b=t.commands[h+1];if(b&&"quadraticCurveTo"===b.command){var m=(l+b.args[0])/2,w=(u+b.args[1])/2;if(g===m&&v===w)continue}}"quadraticCurveTo"===d.command&&0===p||(y|=1),y=this._encodePoint(g,l,i,y,2,16),(y=this._encodePoint(v,u,o,y,4,32))===c&&s<255?(a[a.length-1]|=8,s++):(0<s&&(a.push(s),s=0),a.push(y),c=y),l=g,u=v,f++}"closePath"===d.command&&r.push(f-1)}1<t.commands.length&&"closePath"!==t.commands[t.commands.length-1].command&&r.push(f-1);var x=t.bbox,_={numberOfContours:r.length,xMin:x.minX,yMin:x.minY,xMax:x.maxX,yMax:x.maxY,endPtsOfContours:r,instructions:n,flags:a,xPoints:i,yPoints:o},k=Rl.size(_),S=4-k%4,C=new A.EncodeStream(k+S);return Rl.encode(C,_),0!=S&&C.fill(0,S),C.buffer},Fl.prototype._encodePoint=function(t,e,n,r,i,o){var a=t-e;return t===e?r|=o:(-255<=a&&a<=255&&(r|=i,a<0?a=-a:r|=o),n.push(a)),r},Fl);function Fl(){p(this,Fl)}var Dl,zl=(a(Nl,Dl=Tl),Nl.prototype._addGlyph=function(t){var e=this.font.getGlyph(t),n=e._decode(),r=this.font.loca.offsets[t],i=this.font.loca.offsets[t+1],o=this.font._getTableStream("glyf");o.pos+=r;var a=o.readBuffer(i-r);if(n&&n.numberOfContours<0){a=new h(a);var s=n.components,l=Array.isArray(s),u=0;for(s=l?s:H(s);;){var c;if(l){if(u>=s.length)break;c=s[u++]}else{if((u=s.next()).done)break;c=u.value}var f=c;t=this.includeGlyph(f.glyphID),a.writeUInt16BE(t,f.pos)}}else n&&this.font._variationProcessor&&(a=this.glyphEncoder.encodeSimple(e.path,n.instructions));return this.glyf.push(a),this.loca.offsets.push(this.offset),this.hmtx.metrics.push({advance:e.advanceWidth,bearing:e._getMetrics().leftBearing}),this.offset+=a.length,this.glyf.length-1},Nl.prototype.encode=function(t){this.glyf=[],this.offset=0,this.loca={offsets:[],version:this.font.loca.version},this.hmtx={metrics:[],bearings:[]};for(var e=0;e<this.glyphs.length;)this._addGlyph(this.glyphs[e++]);var n=u(this.font.maxp);n.numGlyphs=this.glyf.length,this.loca.offsets.push(this.offset);var r=u(this.font.head);r.indexToLocFormat=this.loca.version;var i=u(this.font.hhea);i.numberOfMetrics=this.hmtx.metrics.length,gi.encode(t,{tables:{head:r,hhea:i,loca:this.loca,maxp:n,"cvt ":this.font["cvt "],prep:this.font.prep,glyf:this.glyf,hmtx:this.hmtx,fpgm:this.font.fpgm}})},Nl);function Nl(t){p(this,Nl);var e=o(this,Dl.call(this,t));return e.glyphEncoder=new Ml,e}var Ul,Wl,jl=(a(Gl,Ul=Tl),Gl.prototype.subsetCharstrings=function(){this.charstrings=[];var t={},e=this.glyphs,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;this.charstrings.push(this.cff.getCharString(o));var a=this.font.getGlyph(o);for(var s in a.path,a._usedGsubrs)t[s]=!0}this.gsubrs=this.subsetSubrs(this.cff.globalSubrIndex,t)},Gl.prototype.subsetSubrs=function(t,e){for(var n=[],r=0;r<t.length;r++){var i=t[r];e[r]?(this.cff.stream.pos=i.offset,n.push(this.cff.stream.readBuffer(i.length))):n.push(new h([11]))}return n},Gl.prototype.subsetFontdict=function(t){t.FDArray=[],t.FDSelect={version:0,fds:[]};var e={},n=[],r=this.glyphs,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}var s=a,l=this.cff.fdForGlyph(s);if(null!=l){e[l]||(t.FDArray.push(y({},this.cff.topDict.FDArray[l])),n.push({})),e[l]=!0,t.FDSelect.fds.push(t.FDArray.length-1);var u=this.font.getGlyph(s);for(var c in u.path,u._usedSubrs)n[n.length-1][c]=!0}}for(var f=0;f<t.FDArray.length;f++){var h=t.FDArray[f];delete h.FontName,h.Private&&h.Private.Subrs&&(h.Private=y({},h.Private),h.Private.Subrs=this.subsetSubrs(h.Private.Subrs,n[f]))}},Gl.prototype.createCIDFontdict=function(t){var e={},n=this.glyphs,r=Array.isArray(n),i=0;for(n=r?n:H(n);;){var o;if(r){if(i>=n.length)break;o=n[i++]}else{if((i=n.next()).done)break;o=i.value}var a=o,s=this.font.getGlyph(a);for(var l in s.path,s._usedSubrs)e[l]=!0}var u=y({},this.cff.topDict.Private);return this.cff.topDict.Private&&this.cff.topDict.Private.Subrs&&(u.Subrs=this.subsetSubrs(this.cff.topDict.Private.Subrs,e)),t.FDArray=[{Private:u}],t.FDSelect={version:3,nRanges:1,ranges:[{first:0,fd:0}],sentinel:this.charstrings.length}},Gl.prototype.addString=function(t){return t?(this.strings||(this.strings=[]),this.strings.push(t),kt.length+this.strings.length-1):null},Gl.prototype.encode=function(t){this.subsetCharstrings();var e={version:255<this.charstrings.length?2:1,ranges:[{first:1,nLeft:this.charstrings.length-2}]},n=y({},this.cff.topDict);n.Private=null,n.charset=e,n.Encoding=null,n.CharStrings=this.charstrings;for(var r=["version","Notice","Copyright","FullName","FamilyName","Weight","PostScript","BaseFontName","FontName"],i=0;i<r.length;i++){var o=r[i];n[o]=this.addString(this.cff.string(n[o]))}n.ROS=[this.addString("Adobe"),this.addString("Identity"),0],n.CIDCount=this.charstrings.length,this.cff.isCIDFont?this.subsetFontdict(n):this.createCIDFontdict(n);var a={version:1,hdrSize:this.cff.hdrSize,offSize:4,header:this.cff.header,nameIndex:[this.cff.postscriptName],topDictIndex:[n],stringIndex:this.strings,globalSubrIndex:this.gsubrs};Ie.encode(t,a)},Gl);function Gl(t){p(this,Gl);var e=o(this,Ul.call(this,t));if(e.cff=e.font["CFF "],!e.cff)throw new Error("Not a CFF Font");return e}function Hl(n,r,t,e,i){var o={};return Object.keys(e).forEach(function(t){o[t]=e[t]}),o.enumerable=!!o.enumerable,o.configurable=!!o.configurable,("value"in o||o.initializer)&&(o.writable=!0),o=t.slice().reverse().reduce(function(t,e){return e(n,r,t)||t},o),i&&void 0!==o.initializer&&(o.value=o.initializer?o.initializer.call(i):void 0,o.initializer=void 0),void 0===o.initializer&&(Object.defineProperty(n,r,o),o=null),o}var Vl=(ql.probe=function(t){var e=t.toString("ascii",0,4);return"true"===e||"OTTO"===e||e===String.fromCharCode(0,1,0,0)},ql.prototype.setDefaultLanguage=function(t){var e=0<arguments.length&&void 0!==t?t:null;this.defaultLanguage=e},ql.prototype._getTable=function(t){if(!(t.tag in this._tables))try{this._tables[t.tag]=this._decodeTable(t)}catch(t){C.logErrors}return this._tables[t.tag]},ql.prototype._getTableStream=function(t){var e=this.directory.tables[t];return e?(this.stream.pos=e.offset,this.stream):null},ql.prototype._decodeDirectory=function(){return this.directory=gi.decode(this.stream,{_startOffset:0})},ql.prototype._decodeTable=function(t){var e=this.stream.pos,n=this._getTableStream(t.tag),r=hi[t.tag].decode(n,this,t.length);return this.stream.pos=e,r},ql.prototype.getName=function(t,e){var n=1<arguments.length&&void 0!==e?e:this.defaultLanguage||C.defaultLanguage,r=this.name&&this.name.records[t];return r&&(r[n]||r[this.defaultLanguage]||r[C.defaultLanguage]||r.en||r[f(r)[0]])||null},ql.prototype.hasGlyphForCodePoint=function(t){return!!this._cmapProcessor.lookup(t)},ql.prototype.glyphForCodePoint=function(t){return this.getGlyph(this._cmapProcessor.lookup(t),[t])},ql.prototype.glyphsForString=function(t){for(var e=[],n=t.length,r=0,i=-1,o=-1;r<=n;){var a=0,s=0;if(r<n){if(55296<=(a=t.charCodeAt(r++))&&a<=56319&&r<n){var l=t.charCodeAt(r);56320<=l&&l<=57343&&(r++,a=((1023&a)<<10)+(1023&l)+65536)}s=65024<=a&&a<=65039||917760<=a&&a<=917999?1:0}else r++;0===o&&1===s?e.push(this.getGlyph(this._cmapProcessor.lookup(i,a),[i,a])):0===o&&0===s&&e.push(this.glyphForCodePoint(i)),i=a,o=s}return e},ql.prototype.layout=function(t,e,n,r,i){return this._layoutEngine.layout(t,e,n,r,i)},ql.prototype.stringsForGlyph=function(t){return this._layoutEngine.stringsForGlyph(t)},ql.prototype.getAvailableFeatures=function(t,e){return this._layoutEngine.getAvailableFeatures(t,e)},ql.prototype._getBaseGlyph=function(t,e){var n=1<arguments.length&&void 0!==e?e:[];return this._glyphs[t]||(this.directory.tables.glyf?this._glyphs[t]=new gl(t,n,this):(this.directory.tables["CFF "]||this.directory.tables.CFF2)&&(this._glyphs[t]=new bl(t,n,this))),this._glyphs[t]||null},ql.prototype.getGlyph=function(t,e){var n=1<arguments.length&&void 0!==e?e:[];return this._glyphs[t]||(this.directory.tables.sbix?this._glyphs[t]=new _l(t,n,this):this.directory.tables.COLR&&this.directory.tables.CPAL?this._glyphs[t]=new Al(t,n,this):this._getBaseGlyph(t,n)),this._glyphs[t]||null},ql.prototype.createSubset=function(){return this.directory.tables["CFF "]?new jl(this):new zl(this)},ql.prototype.getVariation=function(r){if(!this.directory.tables.fvar||!(this.directory.tables.gvar&&this.directory.tables.glyf||this.directory.tables.CFF2))throw new Error("Variations require a font with the fvar, gvar and glyf, or CFF2 tables.");if("string"==typeof r&&(r=this.namedVariations[r]),"object"!==(void 0===r?"undefined":c(r)))throw new Error("Variation settings must be either a variation name or settings object.");var t=this.fvar.axis.map(function(t,e){var n=t.axisTag.trim();return n in r?Math.max(t.minValue,Math.min(t.maxValue,r[n])):t.defaultValue}),e=new A.DecodeStream(this.stream.buffer);e.pos=this._directoryPos;var n=new ql(e,t);return n._tables=this._tables,n},ql.prototype.getFont=function(t){return this.getVariation(t)},r(ql,[{key:"postscriptName",get:function(){return this.getName("postscriptName")}},{key:"fullName",get:function(){return this.getName("fullName")}},{key:"familyName",get:function(){return this.getName("fontFamily")}},{key:"subfamilyName",get:function(){return this.getName("fontSubfamily")}},{key:"copyright",get:function(){return this.getName("copyright")}},{key:"version",get:function(){return this.getName("version")}},{key:"ascent",get:function(){return this.hhea.ascent}},{key:"descent",get:function(){return this.hhea.descent}},{key:"lineGap",get:function(){return this.hhea.lineGap}},{key:"underlinePosition",get:function(){return this.post.underlinePosition}},{key:"underlineThickness",get:function(){return this.post.underlineThickness}},{key:"italicAngle",get:function(){return this.post.italicAngle}},{key:"capHeight",get:function(){var t=this["OS/2"];return t?t.capHeight:this.ascent}},{key:"xHeight",get:function(){var t=this["OS/2"];return t?t.xHeight:0}},{key:"numGlyphs",get:function(){return this.maxp.numGlyphs}},{key:"unitsPerEm",get:function(){return this.head.unitsPerEm}},{key:"bbox",get:function(){return L(new Ai(this.head.xMin,this.head.yMin,this.head.xMax,this.head.yMax))}},{key:"_cmapProcessor",get:function(){return new wi(this.cmap)}},{key:"characterSet",get:function(){return this._cmapProcessor.getCharacterSet()}},{key:"_layoutEngine",get:function(){return new Js(this)}},{key:"availableFeatures",get:function(){return this._layoutEngine.getAvailableFeatures()}},{key:"variationAxes",get:function(){var t={};if(!this.fvar)return t;var e=this.fvar.axis,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;t[o.axisTag.trim()]={name:o.name.en,min:o.minValue,default:o.defaultValue,max:o.maxValue}}return t}},{key:"namedVariations",get:function(){var t={};if(!this.fvar)return t;var e=this.fvar.instance,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}for(var o=i,a={},s=0;s<this.fvar.axis.length;s++)a[this.fvar.axis[s].axisTag.trim()]=o.coord[s];t[o.name.en]=a}return t}},{key:"_variationProcessor",get:function(){if(!this.fvar)return null;var t=this.variationCoords;return t||this.CFF2?(t=t||this.fvar.axis.map(function(t){return t.defaultValue}),new El(this,t)):null}}]),Hl((Wl=ql).prototype,"bbox",[O],e(Wl.prototype,"bbox"),Wl.prototype),Hl(Wl.prototype,"_cmapProcessor",[O],e(Wl.prototype,"_cmapProcessor"),Wl.prototype),Hl(Wl.prototype,"characterSet",[O],e(Wl.prototype,"characterSet"),Wl.prototype),Hl(Wl.prototype,"_layoutEngine",[O],e(Wl.prototype,"_layoutEngine"),Wl.prototype),Hl(Wl.prototype,"variationAxes",[O],e(Wl.prototype,"variationAxes"),Wl.prototype),Hl(Wl.prototype,"namedVariations",[O],e(Wl.prototype,"namedVariations"),Wl.prototype),Hl(Wl.prototype,"_variationProcessor",[O],e(Wl.prototype,"_variationProcessor"),Wl.prototype),Wl);function ql(t){var e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:null;for(var n in p(this,ql),this.defaultLanguage=null,this.stream=t,this.variationCoords=e,this._directoryPos=this.stream.pos,this._tables={},this._glyphs={},this._decodeDirectory(),this.directory.tables){var r=this.directory.tables[n];hi[n]&&0<r.length&&d(this,n,{get:this._getTable.bind(this,r)})}}var Zl=new A.Struct({tag:new A.String(4),offset:new A.Pointer(A.uint32,"void",{type:"global"}),compLength:A.uint32,length:A.uint32,origChecksum:A.uint32}),Yl=new A.Struct({tag:new A.String(4),flavor:A.uint32,length:A.uint32,numTables:A.uint16,reserved:new A.Reserved(A.uint16),totalSfntSize:A.uint32,majorVersion:A.uint16,minorVersion:A.uint16,metaOffset:A.uint32,metaLength:A.uint32,metaOrigLength:A.uint32,privOffset:A.uint32,privLength:A.uint32,tables:new A.Array(Zl,"numTables")});Yl.process=function(){var t={},e=this.tables,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i;t[o.tag]=o}this.tables=t};var Xl,Kl=(a(Jl,Xl=Vl),Jl.probe=function(t){return"wOFF"===t.toString("ascii",0,4)},Jl.prototype._decodeDirectory=function(){this.directory=Yl.decode(this.stream,{_startOffset:0})},Jl.prototype._getTableStream=function(t){var e=this.directory.tables[t];if(e){if(this.stream.pos=e.offset,e.compLength<e.length){this.stream.pos+=2;var n=new h(e.length),r=k(this.stream.readBuffer(e.compLength-2),n);return new A.DecodeStream(r)}return this.stream}return null},Jl);function Jl(){return p(this,Jl),o(this,Xl.apply(this,arguments))}var Ql,$l=(a(tu,Ql=gl),tu.prototype._decode=function(){return this._font._transformedGlyphs[this.id]},tu.prototype._getCBox=function(){return this.path.bbox},tu);function tu(){return p(this,tu),o(this,Ql.apply(this,arguments))}var eu={decode:function(t){for(var e=0,n=[0,1,2,3,4],r=0;r<n.length;r++){var i=t.readUInt8();if(3758096384&e)throw new Error("Overflow");if(e=e<<7|127&i,0==(128&i))return e}throw new Error("Bad base 128 number")}},nu=["cmap","head","hhea","hmtx","maxp","name","OS/2","post","cvt ","fpgm","glyf","loca","prep","CFF ","VORG","EBDT","EBLC","gasp","hdmx","kern","LTSH","PCLT","VDMX","vhea","vmtx","BASE","GDEF","GPOS","GSUB","EBSC","JSTF","MATH","CBDT","CBLC","COLR","CPAL","SVG ","sbix","acnt","avar","bdat","bloc","bsln","cvar","fdsc","feat","fmtx","fvar","gvar","hsty","just","lcar","mort","morx","opbd","prop","trak","Zapf","Silf","Glat","Gloc","Feat","Sill"],ru=new A.Struct({flags:A.uint8,customTag:new A.Optional(new A.String(4),function(t){return 63==(63&t.flags)}),tag:function(t){return t.customTag||nu[63&t.flags]},length:eu,transformVersion:function(t){return t.flags>>>6&3},transformed:function(t){return"glyf"===t.tag||"loca"===t.tag?0===t.transformVersion:0!==t.transformVersion},transformLength:new A.Optional(eu,function(t){return t.transformed})}),iu=new A.Struct({tag:new A.String(4),flavor:A.uint32,length:A.uint32,numTables:A.uint16,reserved:new A.Reserved(A.uint16),totalSfntSize:A.uint32,totalCompressedSize:A.uint32,majorVersion:A.uint16,minorVersion:A.uint16,metaOffset:A.uint32,metaLength:A.uint32,metaOrigLength:A.uint32,privOffset:A.uint32,privLength:A.uint32,tables:new A.Array(ru,"numTables")});iu.process=function(){for(var t={},e=0;e<this.tables.length;e++){var n=this.tables[e];t[n.tag]=n}return this.tables=t};var ou,au=(a(su,ou=Vl),su.probe=function(t){return"wOF2"===t.toString("ascii",0,4)},su.prototype._decodeDirectory=function(){this.directory=iu.decode(this.stream),this._dataPos=this.stream.pos},su.prototype._decompress=function(){if(!this._decompressed){this.stream.pos=this._dataPos;var t=this.stream.readBuffer(this.directory.totalCompressedSize),e=0;for(var n in this.directory.tables){var r=this.directory.tables[n];r.offset=e,e+=null!=r.transformLength?r.transformLength:r.length}var i=S(t,e);if(!i)throw new Error("Error decoding compressed data in WOFF2");this.stream=new A.DecodeStream(new h(i)),this._decompressed=!0}},su.prototype._decodeTable=function(t){return this._decompress(),ou.prototype._decodeTable.call(this,t)},su.prototype._getBaseGlyph=function(t,e){var n=1<arguments.length&&void 0!==e?e:[];if(!this._glyphs[t])return this.directory.tables.glyf&&this.directory.tables.glyf.transformed?(this._transformedGlyphs||this._transformGlyfTable(),this._glyphs[t]=new $l(t,n,this)):ou.prototype._getBaseGlyph.call(this,t,n)},su.prototype._transformGlyfTable=function(){this._decompress(),this.stream.pos=this.directory.tables.glyf.offset;for(var t=GlyfTable.decode(this.stream),e=[],n=0;n<t.numGlyphs;n++){var r={},i=t.nContours.readInt16BE();if(0<(r.numberOfContours=i)){for(var o=[],a=0,s=0;s<i;s++)a+=pu(t.nPoints),o.push(a);r.points=vu(t.flags,t.glyphs,a);for(var l=0;l<i;l++)r.points[o[l]-1].endContour=!0;pu(t.glyphs)}else i<0&&gl.prototype._decodeComposite.call({_font:this},r,t.composites)&&pu(t.glyphs);e.push(r)}this._transformedGlyphs=e},su);function su(){return p(this,su),o(this,ou.apply(this,arguments))}var lu=(uu.prototype.decode=function(t,e){return new A.DecodeStream(this._buf.decode(t,e))},uu);function uu(t){p(this,uu),this.length=t,this._buf=new A.Buffer(t)}var GlyfTable=new A.Struct({version:A.uint32,numGlyphs:A.uint16,indexFormat:A.uint16,nContourStreamSize:A.uint32,nPointsStreamSize:A.uint32,flagStreamSize:A.uint32,glyphStreamSize:A.uint32,compositeStreamSize:A.uint32,bboxStreamSize:A.uint32,instructionStreamSize:A.uint32,nContours:new lu("nContourStreamSize"),nPoints:new lu("nPointsStreamSize"),flags:new lu("flagStreamSize"),glyphs:new lu("glyphStreamSize"),composites:new lu("compositeStreamSize"),bboxes:new lu("bboxStreamSize"),instructions:new lu("instructionStreamSize")}),cu=253,fu=254,hu=255,du=253;function pu(t){var e=t.readUInt8();return e===cu?t.readUInt16BE():e===hu?t.readUInt8()+du:e===fu?t.readUInt8()+2*du:e}function gu(t,e){return 1&t?e:-e}function vu(t,e,n){for(var r=void 0,i=r=0,o=[],a=0;a<n;a++){var s=0,l=0,u=t.readUInt8(),c=!(u>>7);if((u&=127)<10)s=0,l=gu(u,((14&u)<<7)+e.readUInt8());else if(u<20)s=gu(u,((u-10&14)<<7)+e.readUInt8()),l=0;else if(u<84){s=gu(u,1+(48&(f=u-20))+((h=e.readUInt8())>>4)),l=gu(u>>1,1+((12&f)<<2)+(15&h))}else if(u<120){var f;s=gu(u,1+((f=u-84)/12<<8)+e.readUInt8()),l=gu(u>>1,1+(f%12>>2<<8)+e.readUInt8())}else if(u<124){var h=e.readUInt8(),d=e.readUInt8();s=gu(u,(h<<4)+(d>>4)),l=gu(u>>1,((15&d)<<8)+e.readUInt8())}else s=gu(u,e.readUInt16BE()),l=gu(u>>1,e.readUInt16BE());i+=s,r+=l,o.push(new fl(c,!1,i,r))}return o}var yu=new A.VersionedStruct(A.uint32,{65536:{numFonts:A.uint32,offsets:new A.Array(A.uint32,"numFonts")},131072:{numFonts:A.uint32,offsets:new A.Array(A.uint32,"numFonts"),dsigTag:A.uint32,dsigLength:A.uint32,dsigOffset:A.uint32}}),bu=(mu.probe=function(t){return"ttcf"===t.toString("ascii",0,4)},mu.prototype.getFont=function(t){var e=this.header.offsets,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=new A.DecodeStream(this.stream.buffer);a.pos=o;var s=new Vl(a);if(s.postscriptName===t)return s}return null},r(mu,[{key:"fonts",get:function(){var t=[],e=this.header.offsets,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=new A.DecodeStream(this.stream.buffer);a.pos=o,t.push(new Vl(a))}return t}}]),mu);function mu(t){if(p(this,mu),"ttcf"!==(this.stream=t).readString(4))throw new Error("Not a TrueType collection");this.header=yu.decode(t)}var wu=new A.String(A.uint8),xu=(new A.Struct({len:A.uint32,buf:new A.Buffer("len")}),new A.Struct({id:A.uint16,nameOffset:A.int16,attr:A.uint8,dataOffset:A.uint24,handle:A.uint32})),_u=new A.Struct({name:new A.String(4),maxTypeIndex:A.uint16,refList:new A.Pointer(A.uint16,new A.Array(xu,function(t){return t.maxTypeIndex+1}),{type:"parent"})}),ku=new A.Struct({length:A.uint16,types:new A.Array(_u,function(t){return t.length+1})}),Su=new A.Struct({reserved:new A.Reserved(A.uint8,24),typeList:new A.Pointer(A.uint16,ku),nameListOffset:new A.Pointer(A.uint16,"void")}),Cu=new A.Struct({dataOffset:A.uint32,map:new A.Pointer(A.uint32,Su),dataLength:A.uint32,mapLength:A.uint32}),Au=(Pu.probe=function(t){var e=new A.DecodeStream(t);try{var n=Cu.decode(e)}catch(t){return!1}var r=n.map.typeList.types,i=Array.isArray(r),o=0;for(r=i?r:H(r);;){var a;if(i){if(o>=r.length)break;a=r[o++]}else{if((o=r.next()).done)break;a=o.value}if("sfnt"===a.name)return!0}return!1},Pu.prototype.getFont=function(t){if(!this.sfnt)return null;var e=this.sfnt.refList,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=this.header.dataOffset+o.dataOffset+4,s=new A.DecodeStream(this.stream.buffer.slice(a)),l=new Vl(s);if(l.postscriptName===t)return l}return null},r(Pu,[{key:"fonts",get:function(){var t=[],e=this.sfnt.refList,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=this.header.dataOffset+o.dataOffset+4,s=new A.DecodeStream(this.stream.buffer.slice(a));t.push(new Vl(s))}return t}}]),Pu);function Pu(t){p(this,Pu),this.stream=t,this.header=Cu.decode(this.stream);var e=this.header.map.typeList.types,n=Array.isArray(e),r=0;for(e=n?e:H(e);;){var i;if(n){if(r>=e.length)break;i=e[r++]}else{if((r=e.next()).done)break;i=r.value}var o=i,a=o.refList,s=Array.isArray(a),l=0;for(a=s?a:H(a);;){var u;if(s){if(l>=a.length)break;u=a[l++]}else{if((l=a.next()).done)break;u=l.value}var c=u;0<=c.nameOffset?(this.stream.pos=c.nameOffset+this.header.map.nameListOffset,c.name=wu.decode(this.stream)):c.name=null}"sfnt"===o.name&&(this.sfnt=o)}}C.registerFormat(Vl),C.registerFormat(Kl),C.registerFormat(au),C.registerFormat(bu),C.registerFormat(Au),Eu.exports=C}).call(this,Ou(4).Buffer,Ou(24))},function(t,i,o){(function(){var t,e,n,r;for(t in i.EncodeStream=o(300),i.DecodeStream=o(102),i.Array=o(162),i.LazyArray=o(318),i.Bitfield=o(319),i.Boolean=o(320),i.Buffer=o(321),i.Enum=o(322),i.Optional=o(323),i.Reserved=o(324),i.String=o(325),i.Struct=o(163),i.VersionedStruct=o(326),n=o(48))e=n[t],i[t]=e;for(t in r=o(327))e=r[t],i[t]=e}).call(this)},function(n,t,r){(function(l){(function(){var i,t,s,e,o={}.hasOwnProperty;e=r(95),i=r(102);try{s=r(103)}catch(t){}t=function(t){var e;function r(t){null==t&&(t=65536),r.__super_